/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.Iterator;

import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.sharedtypes.STVerticalAlignRun;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.CTBooleanProperty;
import org.xlsx4j.sml.CTBorder;
import org.xlsx4j.sml.CTBorderPr;
import org.xlsx4j.sml.CTCellAlignment;
import org.xlsx4j.sml.CTCellProtection;
import org.xlsx4j.sml.CTColor;
import org.xlsx4j.sml.CTFill;
import org.xlsx4j.sml.CTFont;
import org.xlsx4j.sml.CTFontName;
import org.xlsx4j.sml.CTFontSize;
import org.xlsx4j.sml.CTNumFmt;
import org.xlsx4j.sml.CTPatternFill;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTUnderlineProperty;
import org.xlsx4j.sml.CTVerticalAlignFontProperty;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.ObjectFactory;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STBorderStyle;
import org.xlsx4j.sml.STHorizontalAlignment;
import org.xlsx4j.sml.STPatternType;
import org.xlsx4j.sml.STUnderlineValues;
import org.xlsx4j.sml.STVerticalAlignment;
import org.xlsx4j.sml.SheetData;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.OperationDocument;


public class CellUtils {

	private final static Logger log = LoggerFactory.getLogger(CellUtils.class);

    public static void applyCellPropertiesToCell(OperationDocument operationDocument, WorksheetPart worksheetPart, SheetData sheetData, JSONObject attrs, Row row, Cell cell)
        throws FilterException, JSONException {

        final long oldAutoStyle = cell.getStyle();
        final long newAutoStyle = applyAutoStyle(oldAutoStyle, attrs);
    	if(oldAutoStyle!=newAutoStyle) {
    	    cell.setS(newAutoStyle);
            sheetData.applyCachedCellData(cell);
    	}
    }

    public static long applyAutoStyle(long oldAutoStyle, JSONObject attrs)
        throws FilterException, JSONException {

        final String newAStyle = attrs.optString("styleId", null);
    	if(newAStyle!=null&&newAStyle.startsWith("a")) {
    		return Long.valueOf(newAStyle.substring(1));
    	}
    	return oldAutoStyle;
    }

    public static JSONObject createCellProperties(Cell cell, CTStylesheet stylesheet)
        throws JSONException {

        JSONObject jsonCellProperties = null;
        if(cell!=null) {
            jsonCellProperties = new JSONObject();
            jsonCellProperties.put("styleId", "a" + Long.toString(cell.getStyle()));
        }
        return jsonCellProperties!=null&&jsonCellProperties.length()>0 ? jsonCellProperties : null;
    }

    static public Long getAttributeIndex(Long index, Boolean isApply, boolean createStyle) {
        if(createStyle) {
            if((isApply==null||isApply.booleanValue())&&index!=null) {
                return index;
            }
            return Long.valueOf(0);
        }
        return index!=null?index:Long.valueOf(0);
    }

    static private CTFill createDetachedFill(CTStylesheet stylesheet, CTXf xf) {

        Long fillId = xf.getFillId();
        if(fillId==null||fillId.longValue()==0) {
            return Context.getsmlObjectFactory().createCTFill();
        }
        CTFill ctFill = stylesheet.getFillByIndex(fillId);
        if(ctFill!=null) {
            return ctFill.clone();
        }
        log.warn("xlsx export: could not get CTFill for Id");
        return Context.getsmlObjectFactory().createCTFill();
    }

    static private CTBorder createDetachedBorder(CTStylesheet stylesheet, CTXf xf) {

        Long borderId = xf.getBorderId();
        if(borderId==null||borderId.longValue()==0) {
            return Context.getsmlObjectFactory().createCTBorder();
        }
        CTBorder ctBorder = stylesheet.getBorderByIndex(borderId);
        if(ctBorder!=null) {
            return ctBorder.clone();
        }
        log.warn("xlsx export: could not get CTBorder for Id");
        return Context.getsmlObjectFactory().createCTBorder();
    }

    private static CTBorderPr applyBorder(OperationDocument operationDocument, JSONObject borderProperties, CTStylesheet stylesheet, CTBorderPr _borderPr)
        throws FilterException, JSONException {

        CTBorderPr borderPr = _borderPr!=null?_borderPr:Context.getsmlObjectFactory().createCTBorderPr();

        JSONObject jsonColor = borderProperties.optJSONObject("color");
        if(jsonColor!=null) {
            CTColor color = Context.getsmlObjectFactory().createCTColor();
            Utils.applyColor(operationDocument, color, jsonColor);
            borderPr.setColor(color);
        }

        int width = borderProperties.optInt("width", Utils.THIN_WIDTH_HMM);
        double pixelWidth = Utils.hmmToPixel(width);

        String style = borderProperties.optString("style", null);
        if(style!=null) {

            STBorderStyle borderStyle = STBorderStyle.THIN;
            if(style.equals("none")) {
                borderStyle = STBorderStyle.NONE;
            }
            else if(style.equals("single")) {
                if (pixelWidth >= 2.5) {
                    borderStyle = STBorderStyle.THICK;
                } else if (pixelWidth >= 1.5) {
                    borderStyle = STBorderStyle.MEDIUM;
                } else if (pixelWidth >= 0.75) {
                    borderStyle = STBorderStyle.THIN;
                } else {
                    borderStyle = STBorderStyle.HAIR;
                }
            }
            else if(style.equals("double")) {
                borderStyle = STBorderStyle.DOUBLE;
            }
            else if(style.equals("dotted")) {
                if (pixelWidth >= 1.5) {
                    borderStyle = STBorderStyle.MEDIUM_DASH_DOT_DOT;
                } else {
                	borderStyle = STBorderStyle.DOTTED;
                }
            }
            else if(style.equals("dashed")) {
                if (pixelWidth >= 1.5) {
                    borderStyle = STBorderStyle.MEDIUM_DASHED;
                } else {
                    borderStyle = STBorderStyle.DASHED;
                }
            }
            else if(style.equals("dashDot")) {
                if (pixelWidth >= 1.5) {
                    borderStyle = STBorderStyle.MEDIUM_DASH_DOT;
                } else {
                	borderStyle = STBorderStyle.DASH_DOT;
                }
            }
            else if(style.equals("dashDotDot")) {
                if (pixelWidth >= 1.5) {
                    borderStyle = STBorderStyle.MEDIUM_DASH_DOT_DOT;
                } else {
                	borderStyle = STBorderStyle.DASH_DOT_DOT;
                }
            }
            borderPr.setStyle(borderStyle);
        }
        return borderPr;
    }

    private static Long getNumberFormatByIdAndFormatCode(CTStylesheet stylesheet, long numberFormatId, String numberFormatCode) {

        if(numberFormatCode.isEmpty()) {
            // numberFormatId without numberFormatCode is only valid if this Id is already existing or it is a predefined Id (<=49)
            CTNumFmt numFmt = stylesheet.getNumberFormatById(numberFormatId);
            if(numFmt!=null) {
                return numberFormatId;
            }
            else if(numberFormatId<=49) {
                if(!numberFormatCode.isEmpty()) {
                    numFmt = Context.getsmlObjectFactory().createCTNumFmt();
                    numFmt.setNumFmtId(numberFormatId);
                    numFmt.setFormatCode(numberFormatCode);
                    stylesheet.applyNumberFormat(numFmt);
                }
                return numberFormatId;
            }
            else {
                log.warn("xlsx export: trying to apply a non predefined numberFormat without format code");
                return null;
            }
        }
        else if (numberFormatCode.equals("General")) {
            return new Long(0L);
        }
        return stylesheet.getOrApplyNumberFormatByFormatCode(numberFormatCode).getNumFmtId();
    }

    private static Long getNumberFormatByFormatCode(CTStylesheet stylesheet, String numberFormatCode) {
        return stylesheet.getOrApplyNumberFormatByFormatCode(numberFormatCode.isEmpty() ? "General" : numberFormatCode).getNumFmtId();
    }

    public static void applyCellProperties(OperationDocument operationDocument, JSONObject cellProperties, CTXf xf, CTStylesheet stylesheet, boolean autostyle)
        throws FilterException, JSONException {

        final ObjectFactory objectFactory = Context.getsmlObjectFactory();

        // detached property sets are only created if needed
        CTFill          detachedFill = null;
        CTBorder        detachedBorder = null;

        Iterator<String> keys = cellProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = cellProperties.get(attr);

            // CTNumFmt
            if(attr.equals("numberFormat")) {
            	if(value instanceof JSONObject) {
                    Long destNumFmtId = null;
                    final long numberFormatId = ((JSONObject)value).optLong("id", -1);
                    String numberFormatCode = ((JSONObject)value).optString("code", "");
                    if(numberFormatCode.equals("BOOLEAN")) {
                    	numberFormatCode = "\"TRUE\";\"TRUE\";\"FALSE\"";
                    }
                    if(numberFormatId>=0) {
                        destNumFmtId = getNumberFormatByIdAndFormatCode(stylesheet, numberFormatId, numberFormatCode);
                    }
                    else {
                        destNumFmtId = getNumberFormatByFormatCode(stylesheet, numberFormatCode);
                    }
                    xf.setNumFmtId(destNumFmtId);
                    xf.setApplyNumberFormat(autostyle?true:null);
                }
                else {
                    xf.setNumFmtId(null);
                }
            }
            // CTFill
            else if(attr.equals("fillColor")) {
            	if(value instanceof JSONObject) {
                    detachedFill = createDetachedFill(stylesheet, xf);
                    detachedFill.setGradientFill(null);
                    CTPatternFill patternFill = objectFactory.createCTPatternFill();
                    detachedFill.setPatternFill(patternFill);
                    patternFill.setBgColor(null);
                    CTColor color = objectFactory.createCTColor();
                    Utils.applyColor(operationDocument, color, (JSONObject)value);
                    patternFill.setFgColor(color);

                    //COLOR.AUTO works fine for fillColor
                    if(color.isAuto() != null && color.isAuto()){
                        patternFill.setPatternType(STPatternType.NONE);
                    } else {
                        patternFill.setPatternType(STPatternType.SOLID);
                    }
                    if(autostyle) {
                        xf.setApplyFill(true);
                    }
                }
                else {
                    xf.setFillId(null);
                }
            }
            // CTBorder
            else if(attr.equals("borderLeft")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setLeft(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getLeft()));
                    if(autostyle) {
                        xf.setApplyBorder(true);
                    }
                }
                else {
                    detachedBorder.setLeft(null);
                }
            }
            else if(attr.equals("borderRight")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setRight(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getRight()));
                    if(autostyle) {
                        xf.setApplyBorder(true);
                    }
                }
                else {
                    detachedBorder.setRight(null);
                }
            }
            else if(attr.equals("borderTop")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setTop(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getTop()));
                    if(autostyle) {
                        xf.setApplyBorder(true);
                    }
                }
                else {
                    detachedBorder.setTop(null);
                }
            }
            else if(attr.equals("borderBottom")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setBottom(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getBottom()));
                    if(autostyle) {
                        xf.setApplyBorder(true);
                    }
                }
                else {
                    detachedBorder.setBottom(null);
                }
            }
            else if(attr.equals("borderInsideHor")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setHorizontal(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getHorizontal()));
                    if(autostyle) {
                        xf.setApplyBorder(true);
                    }
                }
                else {
                    detachedBorder.setHorizontal(null);
                }
            }
            else if(attr.equals("borderInsideVert")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setVertical(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getVertical()));
                    if(autostyle) {
                        xf.setApplyBorder(true);
                    }
                }
                else {
                    detachedBorder.setVertical(null);
                }
            }
            // CTCellAlignment
            else if(attr.equals("alignHor")) {
                CTCellAlignment cellAlignment = xf.getAlignment();
                if(cellAlignment==null) {
                    cellAlignment = objectFactory.createCTCellAlignment();
                    xf.setAlignment(cellAlignment);
                }
                if(value instanceof String) {
                    String alignment = (String)value;
                    if(alignment.equals("left")) {
                        cellAlignment.setHorizontal(STHorizontalAlignment.LEFT);
                    }
                    else if(alignment.equals("center")) {
                        cellAlignment.setHorizontal(STHorizontalAlignment.CENTER);
                    }
                    else if(alignment.equals("right")) {
                        cellAlignment.setHorizontal(STHorizontalAlignment.RIGHT);
                    }
                    else if(alignment.equals("justify")) {
                        cellAlignment.setHorizontal(STHorizontalAlignment.JUSTIFY);
                    }
                    else {
                        cellAlignment.setHorizontal(STHorizontalAlignment.GENERAL);
                    }
                    if(autostyle) {
                        xf.setApplyAlignment(true);
                    }
                }
                else {
                    cellAlignment.setHorizontal(STHorizontalAlignment.GENERAL);
                }
            }
            else if(attr.equals("alignVert")) {
                CTCellAlignment cellAlignment = xf.getAlignment();
                if(cellAlignment==null) {
                    cellAlignment = objectFactory.createCTCellAlignment();
                    xf.setAlignment(cellAlignment);
                }
                if(value instanceof String) {
                    String alignment = (String)value;
                    if(alignment.equals("top")) {
                        cellAlignment.setVertical(STVerticalAlignment.TOP);
                    }
                    else if(alignment.equals("middle")) {
                        cellAlignment.setVertical(STVerticalAlignment.CENTER);
                    }
                    else if(alignment.equals("justify")) {
                        cellAlignment.setVertical(STVerticalAlignment.JUSTIFY);
                    }
                    else {
                        cellAlignment.setVertical(STVerticalAlignment.BOTTOM);
                    }
                    if(autostyle) {
                        xf.setApplyAlignment(true);
                    }
                }
                else {
                    cellAlignment.setVertical(STVerticalAlignment.BOTTOM);
                }
            }
            else if(attr.equals("wrapText")) {
                CTCellAlignment cellAlignment = xf.getAlignment();
                if(cellAlignment==null) {
                    cellAlignment = objectFactory.createCTCellAlignment();
                    xf.setAlignment(cellAlignment);
                }
                if(value instanceof Boolean) {
                    cellAlignment.setWrapText((Boolean)value);
                    if(autostyle) {
                        xf.setApplyAlignment(true);
                    }
                }
                else {
                    cellAlignment.setWrapText(null);
                }
            }
            //CTCellProtection
            else if(attr.equals("unlocked")) {
                CTCellProtection cellProtection = xf.getProtection();
                if(cellProtection==null) {
                    cellProtection = objectFactory.createCTCellProtection();
                    xf.setProtection(cellProtection);
                }
                if(value instanceof Boolean) {
                    cellProtection.setLocked(!(Boolean)value);
                    if(autostyle) {
                        xf.setApplyProtection(true);
                    }
                }
                else {
                    cellProtection.setLocked(null);
                }
            }
            else if (attr.equals("hidden")) {
                CTCellProtection cellProtection = xf.getProtection();
                if(cellProtection==null) {
                    cellProtection = objectFactory.createCTCellProtection();
                    xf.setProtection(cellProtection);
                }
                if(value instanceof Boolean) {
                    cellProtection.setHidden((Boolean)value);
                    if(autostyle) {
                        xf.setApplyProtection(true);
                    }
                }
                else {
                    cellProtection.setHidden(null);
                }
            }
        }

        // CTFill
        if(detachedFill!=null) {
            xf.setFillId(stylesheet.getOrApplyFill(detachedFill));
        }
        if(detachedBorder!=null) {
            xf.setBorderId(stylesheet.getOrApplyBorder(detachedBorder));
        }
    }

    public static JSONObject createCellProperties(CTXf xf, CTStylesheet stylesheet, boolean createStyle)
        throws JSONException {

        JSONObject jsonCellProperties = new JSONObject();
        // CTNumFmt
        Long _numberFormatId = getAttributeIndex(xf.getNumFmtId(), xf.isApplyNumberFormat(), createStyle);
        long numberFormatId = _numberFormatId==null?0:_numberFormatId.longValue();
        Utils.createNumberFormat(jsonCellProperties, numberFormatId, stylesheet.getNumberFormatById(numberFormatId));

        // CTFILL
        Utils.createFill(jsonCellProperties, stylesheet, stylesheet.getFillByIndex(getAttributeIndex(xf.getFillId(), xf.isApplyFill(), createStyle)));

        // CTBorders
        Utils.createBorders(jsonCellProperties, stylesheet, stylesheet.getBorderByIndex(getAttributeIndex(xf.getBorderId(), xf.isApplyBorder(), createStyle)));

        // CTCellAlignment
        if(createStyle) {
	        if(xf.isApplyAlignment()!=null&&xf.isApplyAlignment().booleanValue()) {
	            Utils.createAlignment(jsonCellProperties, xf.getAlignment());
	        }
	        // CTCellProtection
	        if(xf.isApplyProtection()!=null&&xf.isApplyProtection().booleanValue()) {
	            Utils.createProtection(jsonCellProperties, xf.getProtection());
	        }
        }
        else {
        	Utils.createAlignment(jsonCellProperties, xf.getAlignment());
        	Utils.createProtection(jsonCellProperties, xf.getProtection());
        }
        return jsonCellProperties.length()>0?jsonCellProperties:null;
    }

    public static void applyCharacterProperties(OperationDocument operationDocument, JSONObject characterProperties, CTXf xf, CTStylesheet stylesheet, boolean autostyle)
        throws FilterException, JSONException {

        final ObjectFactory objectFactory = Context.getsmlObjectFactory();

        CTFont font = stylesheet.getFontByIndex(xf.getFontId());
        if(font==null) {
            log.warn("xlsx export: could not get font for Id");
            font = objectFactory.createCTFont();
        }
        final CTFont detachedFont = font.clone();

        Iterator<String> keys = characterProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = characterProperties.get(attr);
            if(attr.equals("fontName")) {
                if(value instanceof String) {
                    CTFontName fontName = detachedFont.getName();
                    if(fontName==null) {
                    	fontName = objectFactory.createCTFontName();
                    	detachedFont.setName(fontName);
                    }
                    fontName.setVal((String)value);
                    // the scheme property has higher priority, so we have to remove it, otherwise
                    // the font name would not be used
                    detachedFont.setScheme(null);
                }
                else {
                	detachedFont.setName(null);
                }
            }
            else if (attr.equals("fontSize")) {
                if(value instanceof Number) {
                    CTFontSize fontSize = detachedFont.getSz();
                    if(fontSize==null) {
                    	fontSize = objectFactory.createCTFontSize();
                    	detachedFont.setSz(fontSize);
                    }
                    fontSize.setVal(((Number)value).doubleValue());
                }
                else {
                	detachedFont.setSz(null);
                }
            }
            else if(attr.equals("bold")) {
                if(value instanceof Boolean) {
                    CTBooleanProperty b = detachedFont.getB();
                    if(b==null) {
                    	b = objectFactory.createCTBooleanProperty();
                    	detachedFont.setB(b);
                    }
                    b.setVal((Boolean)value);
                }
                else {
                	detachedFont.setB(null);
                }
            }
            else if(attr.equals("italic")) {
                if(value instanceof Boolean) {
                    CTBooleanProperty i = detachedFont.getI();
                    if(i==null) {
                    	i = objectFactory.createCTBooleanProperty();
                    	detachedFont.setI(i);
                    }
                    i.setVal((Boolean)value);
                }
                else {
                	detachedFont.setI(null);
                }
            }
            else if(attr.equals("strike")) {
                if(value instanceof String) {
                    CTBooleanProperty strike = detachedFont.getStrike();
                    if(strike==null) {
                    	strike = objectFactory.createCTBooleanProperty();
                    	detachedFont.setStrike(strike);
                    }
                    strike.setVal(((String)value).equals("none")==false);
                }
                else {
                	detachedFont.setStrike(null);
                }

            }
            else if(attr.equals("color")) {
                if(value instanceof JSONObject) {
                    CTColor color = detachedFont.getColor();
                    if(color==null) {
                    	color = objectFactory.createCTColor();
                    	detachedFont.setColor(color);
                    }
                    Utils.applyColor(operationDocument, color, (JSONObject)value);
                }
                else {
                	detachedFont.setColor(null);
                }
            }
            else if(attr.equals("vertAlign")) {
                if(value instanceof String) {
                    CTVerticalAlignFontProperty vertAlign = detachedFont.getVertAlign();
                    if(vertAlign==null) {
                    	vertAlign = objectFactory.createCTVerticalAlignFontProperty();
                    	detachedFont.setVertAlign(vertAlign);
                    }
                    STVerticalAlignRun align = STVerticalAlignRun.BASELINE;
                    if(((String)value).equals("sup")) {
                        align = STVerticalAlignRun.SUPERSCRIPT;
                    }
                    else if(((String)value).equals("sub")) {
                        align = STVerticalAlignRun.SUBSCRIPT;
                    }
                    vertAlign.setVal(align);
                }
                else {
                	detachedFont.setVertAlign(null);
                }
            }
            else if(attr.equals("underline")) {
                if(value instanceof Boolean) {
                    CTUnderlineProperty u = detachedFont.getU();
                    if(u==null) {
                    	u = objectFactory.createCTUnderlineProperty();
                    	detachedFont.setU(u);
                    }
                    u.setVal(((Boolean)value).booleanValue() ? STUnderlineValues.SINGLE : STUnderlineValues.NONE);
                }
                else {
                	detachedFont.setU(null);
                }
            }
        }
        long fontIndex = stylesheet.getOrApplyFont(detachedFont);
        xf.setFontId(fontIndex);
        if(autostyle) {
            xf.setApplyFont(true);
        }
    }

    public static JSONObject createCharacterProperties(JSONObject jsonCharacterProperties, CTStylesheet stylesheet, CTFont font)
    	throws JSONException {

    	if(font!=null) {
        	final CTFontName fontName = font.getName();
            if(fontName!=null) {
                Commons.jsonPut(jsonCharacterProperties, "fontName", fontName.getVal());
            }
            final CTFontSize fontSize = font.getSz();
            if(fontSize!=null) {
                Commons.jsonPut(jsonCharacterProperties, "fontSize", fontSize.getVal());
            }
            final CTBooleanProperty b = font.getB();
            Commons.jsonPut(jsonCharacterProperties, "bold", b!=null ? b.isVal() : false);

            final CTBooleanProperty i = font.getI();
            Commons.jsonPut(jsonCharacterProperties, "italic", i!=null ? i.isVal() : false);

            final CTBooleanProperty strike = font.getStrike();
            Commons.jsonPut(jsonCharacterProperties, "strike", strike != null ? strike.isVal() ? "single" : "none" : "none");

            final CTColor color = font.getColor();
            Commons.jsonPut(jsonCharacterProperties, "color", color!=null ? Utils.createColor(stylesheet, color) : Utils.createDefaultColor());

            final CTVerticalAlignFontProperty vertAlign = font.getVertAlign();
            String align = "baseline";
            if(vertAlign!=null) {
                switch(vertAlign.getVal()) {
                    case BASELINE :    break;
                    case SUPERSCRIPT : align = "sup"; break;
                    case SUBSCRIPT :   align = "sub"; break;
                }
            }
            Commons.jsonPut(jsonCharacterProperties, "vertAlign", align);

            final CTUnderlineProperty u = font.getU();
            if(u!=null) {
                final STUnderlineValues underline = u.getVal();
                if(underline!=null) {
                    Commons.jsonPut(jsonCharacterProperties, "underline", underline!=STUnderlineValues.NONE);
                }
            }
            else {
            	Commons.jsonPut(jsonCharacterProperties, "underline", false);
            }
        }
        return jsonCharacterProperties.length()>0?jsonCharacterProperties:null;
    }

    private static CTXf createDetachedDefaultXf() {
        final CTXf detachedDefaultXf = Context.getsmlObjectFactory().createCTXf();
        detachedDefaultXf.setXfId(Long.valueOf(0));
        detachedDefaultXf.setFillId(Long.valueOf(0));
        detachedDefaultXf.setFontId(Long.valueOf(0));
        detachedDefaultXf.setNumFmtId(Long.valueOf(0));
        detachedDefaultXf.setBorderId(Long.valueOf(0));
        return detachedDefaultXf;
    }
    public static CTXf getDetachedXfFromStyle(CTStylesheet stylesheet, Long styleIndex) {

        CTXf detachedXf = null;

        // we assume that the first style is an empty default style and is containing only
        // zero indexes, for this case we don't clone and are using the objectFactory
        if(styleIndex.longValue()>0) {
            final CTXf cellStyleXfSource = stylesheet.getCellStyleXfsByIndex(styleIndex);
            if(cellStyleXfSource!=null) {
                detachedXf = cellStyleXfSource.clone(false);
                detachedXf.setXfId(styleIndex);
            }
        }
        if(detachedXf==null) {
            detachedXf = createDetachedDefaultXf();
        }
        return detachedXf;
    }

    public static CTXf getDetachedXfFromAutoStyle(CTStylesheet stylesheet, long cellXfIndex, Long newStyleIndex) {

        CTXf detachedXf;

        final CTXf xfSource = stylesheet.getCellXfByIndex(cellXfIndex);         // current auto-attributes
        if(newStyleIndex==null) {
            detachedXf = xfSource==null?createDetachedDefaultXf():xfSource.clone(true);
        }
        else {
            final CTXf cellStyleXfSource = stylesheet.getCellStyleXfsByIndex(newStyleIndex);
            if(cellStyleXfSource==null) {
                detachedXf = createDetachedDefaultXf();
            }
            else {
                detachedXf = cellStyleXfSource.clone(true);
                detachedXf.setXfId(newStyleIndex);
            }
            if(detachedXf.isApplyBorder()!=null&&!detachedXf.isApplyBorder().booleanValue()) {
                detachedXf.setApplyBorder(true);
                detachedXf.setBorderId(xfSource.getBorderId());
            }
            if(detachedXf.isApplyAlignment()!=null&&!detachedXf.isApplyAlignment().booleanValue()) {
                detachedXf.setApplyAlignment(true);
                final CTCellAlignment cellAlignmentSource = xfSource.getAlignment();
                detachedXf.setAlignment(cellAlignmentSource!=null?cellAlignmentSource.clone():null);
            }
            if(detachedXf.isApplyFill()!=null&&!detachedXf.isApplyFill().booleanValue()) {
                detachedXf.setApplyFill(true);
                detachedXf.setFillId(xfSource.getFillId());
            }
            if(detachedXf.isApplyFont()!=null&&!detachedXf.isApplyFont().booleanValue()) {
                detachedXf.setApplyFont(true);
                detachedXf.setFontId(xfSource.getFontId());
            }
            if(detachedXf.isApplyNumberFormat()!=null&&!detachedXf.isApplyNumberFormat().booleanValue()) {
                detachedXf.setApplyNumberFormat(true);
                detachedXf.setNumFmtId(xfSource.getNumFmtId());
            }
            if(detachedXf.isApplyProtection()!=null&&!detachedXf.isApplyProtection().booleanValue()) {
                detachedXf.setApplyProtection(true);
                final CTCellProtection cellProtectionSource = xfSource.getProtection();
                detachedXf.setProtection(cellProtectionSource!=null?cellProtectionSource.clone():null);
            }
            detachedXf.setPivotButton(xfSource.isPivotButton());
            detachedXf.setQuotePrefix(xfSource.isQuotePrefix());
        }
        return detachedXf;
    }
}
