/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.List;
import java.util.TreeSet;

import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.attribute.table.TableDefaultCellStyleNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableIdAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableNumberRowsRepeatedAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableStyleNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableVisibilityAttribute;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnGroupElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.type.Length;
import org.odftoolkit.odfdom.type.Length.Unit;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import com.openexchange.office.ods.dom.SmlUtils.CellRef;

public class Row implements Comparable<Row>, IAttrs<Row> {

	private final TreeSet<Cell> cells = new TreeSet<Cell>();
	private String rowStyle;
	private String defaultCellStyle = null;

	private int r;
	private int repeated = 1;
	private int groupLevel = 0;
	private Visibility visibility = null;
	private String id = null;
	
	Row(String rowStyle, int rowNumber) {
		this.rowStyle = rowStyle;
		r = rowNumber;
	}

	Row(String rowStyle, int rowNumber, int grpLvl, boolean groupStarted) {
		this.rowStyle = rowStyle;
		r = rowNumber;
		setGroupLevel(grpLvl);
		setStartingGroup(groupStarted);
	}

	/* returns a live list to the columns */
	public TreeSet<Cell> getCells() {
		return cells;
	}

    public Cell getCell(int c, boolean forceCreate, boolean cutFloor, boolean cutCeiling) {
    	if(cells.isEmpty()) {
    		if(!forceCreate) {
    			return null;
    		}
    		// creating new cells up to c
    		final Cell newCell = new Cell(0);
    		newCell.setRepeated(c+1);
    		cells.add(newCell);
    	}
    	Cell retValue = cells.floor(new Cell(c));
    	if((retValue==null)||(retValue.getColumn()+retValue.getRepeated()<=c)) {
    		if(!forceCreate) {
    			return null;
    		}
    		// we have to create new cells from end of floorValue up to c
        	final Cell newCell = new Cell(retValue!=null?retValue.getColumn()+retValue.getRepeated():0);
        	newCell.setRepeated((c-newCell.getColumn())+1);
        	cells.add(newCell);
        	retValue = newCell;
    	}
    	// Cells up to c are available
    	if(cutFloor&&retValue.getColumn()!=c) {
    		final Cell newCell = retValue.clone(true);
    		newCell.setColumn(c);
			newCell.setRepeated((retValue.getColumn()+retValue.getRepeated())-c);
			cells.add(newCell);
	    	retValue.setRepeated(c-retValue.getColumn());
	    	retValue = newCell;
    	}
    	if(cutCeiling&&((retValue.getColumn()+retValue.getRepeated())-1)>c) {
    		final Cell newCell = retValue.clone(true);
    		newCell.setColumn(c+1);
			newCell.setRepeated(((retValue.getColumn()+retValue.getRepeated())-1)-c);
	    	cells.add(newCell);
	    	retValue.setRepeated(retValue.getRepeated()-newCell.getRepeated());
    	}
    	return retValue;
    }

	public int getRow() {
		return r;
	}

	public void setRow(int row) {
		r = row;
	}

	public int getRepeated() {
    	return repeated;
    }

    public void setRepeated(int val) {
    	repeated = val;
    }

    public String getRowStyle() {
    	return rowStyle;
    }

    private void setRowStyle(String rowStyle) {
    	this.rowStyle = rowStyle;
    }

    public void replaceStyle(OfficeStyles officeStyles, String newStyle) {
    	if(getRowStyle()!=null) {
    		officeStyles.decRowStyleUsage(getRowStyle());
    	}
    	if(newStyle!=null) {
    		officeStyles.incRowStyleUsage(newStyle);
    	}
    	setRowStyle(newStyle);
    }

    public String getDefaultCellStyle() {
    	return defaultCellStyle;
    }

    public void setDefaultCellStyle(String val) {
    	defaultCellStyle = val;
    }

    public Visibility getVisibility() {
    	return visibility;
    }

    public void setVisibility(Visibility value) {
    	visibility = value;
    }

    public String getId() {
    	return id;
    }

    public void setId(String val) {
    	id = val;
    }

    public void insertCells(int start, int count) {

    	// assuring that the first cell is available
    	getCell(start, true, true, false);

    	final Object[] rs = cells.tailSet(new Cell(start), true).toArray();
    	for(int i=rs.length-1;i>=0;i--) {
    		final Cell ce = (Cell)rs[i];
    		cells.remove(ce);
    		int newCellNumber = ce.getColumn()+count;
    		if(newCellNumber<Sheet.getMaxRowCount()) {
	    		ce.setColumn(ce.getColumn()+count);
	    		cells.add(ce);
	    		if(newCellNumber+ce.getRepeated()>Sheet.getMaxRowCount()) {
	    			ce.setRepeated(Sheet.getMaxRowCount()-newCellNumber);
	    		}
    		}
    	}
    	Cell previousCell = null;
    	if(start>0) {
    		previousCell = getCell(start-1, false, false, false);
    	}
    	final Cell newRow = new Cell(start);
    	if(previousCell!=null) {
    		newRow.setCellStyle(previousCell.getCellStyle());
    	}
    	newRow.setRepeated(count);
    	cells.add(newRow);
    }

    public void deleteCells(int start, int count) {

    	getCell(start, true, true, true);
    	if(count>1) {
    		getCell((start+count)-1, true, false, true);
    	}
    	final Object[] rs = cells.tailSet(new Cell(start), true).toArray();
    	for(Object obj:rs) {
			final Cell ce = (Cell)obj;
	   		cells.remove(ce);
    		if(ce.getColumn()>=start+count) {
    			ce.setColumn(ce.getColumn()-count);
    			cells.add(ce);
    		}
    	}
    }

    public void writeObject(SerializationHandler output, Sheet sheet)
    	throws SAXException {

    	SaxContextHandler.startElement(output, TableTableRowElement.ELEMENT_NAME);
		if(getRepeated()>1) {
			SaxContextHandler.addOdfAttribute(output, TableNumberRowsRepeatedAttribute.ATTRIBUTE_NAME, null, Integer.toString(getRepeated()));
		}
		if(getRowStyle()!=null) {
			SaxContextHandler.addOdfAttribute(output, TableStyleNameAttribute.ATTRIBUTE_NAME, null, getRowStyle());
		}
		if(getDefaultCellStyle()!=null) {
			SaxContextHandler.addOdfAttribute(output, TableDefaultCellStyleNameAttribute.ATTRIBUTE_NAME, null, getDefaultCellStyle());
		}
		if(getVisibility()!=null) {
			SaxContextHandler.addOdfAttribute(output, TableVisibilityAttribute.ATTRIBUTE_NAME, null, getVisibility().toString());
		}
		if(getId()!=null) {
			SaxContextHandler.addOdfAttribute(output, TableIdAttribute.ATTRIBUTE_NAME, null, getId());
		}

		MergeCell merge = null;
		final List<MergeCell> mergeCells = sheet.getMergeCells();
		for(Cell cell:cells) {
			if(merge!=null) {
				// check if the cell fits into the old merge
				if(merge.getCellRefRange().getEnd().getColumn()<cell.getColumn()) {
					merge = null;
				}
			}
			if(merge==null) {
				// check if we can find a new merge for our new cell
				for(MergeCell mergeCell:mergeCells) {
					final CellRef mergeCellStart = mergeCell.getCellRefRange().getStart();
					if(mergeCellStart.getColumn()==cell.getColumn()&&mergeCellStart.getRow()<=getRow()) {
						final CellRef mergeCellEnd = mergeCell.getCellRefRange().getEnd();
						if(mergeCellEnd.getRow()>=getRow()) {
							merge = mergeCell;
						}
					}
				}
			}
			cell.writeObject(output, sheet, this, merge);
		}
		SaxContextHandler.endElement(output, TableTableRowElement.ELEMENT_NAME);
	}

    @Override
    public void createAttributes(JSONObject attrs, OdfSpreadsheetDocument doc)
    	throws JSONException, SAXException {

        final JSONObject rowProperties = new JSONObject(2);
        if(getVisibility()!=null) {
        	if(getVisibility()==Visibility.COLLAPSE) {
        		rowProperties.put("visible", false);
        	}
        	else if(getVisibility()==Visibility.FILTER) {
        		rowProperties.put("filtered",  true);
        	}
        	else {
        		rowProperties.put("visible",  true);
        		rowProperties.put("filtered", false);
        	}
        }
        else {
        	rowProperties.put("visible", true);
        	rowProperties.put("filtered", false);
        }
        if(rowStyle!=null&&!rowStyle.isEmpty()) {
        	final OdfOfficeAutomaticStyles autoStyles = ((Content)doc.getContentDom()).getAutomaticStyles();
        	final OdfStyle odfStyle = autoStyles.getStyle(rowStyle, OdfStyleFamily.TableRow);
        	if(odfStyle!=null) {
        		final Element tableRowProperties = odfStyle.getChildElement(OdfDocumentNamespace.STYLE.getUri(), "table-row-properties");
        		if(tableRowProperties!=null) {
	        		final String width = tableRowProperties.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "row-height");
	        		if(width!=null&&!width.isEmpty()) {
	        			final Double dest = Length.getLength(width, Unit.MILLIMETER);
	        			if(dest!=null) {
	        				rowProperties.put("height", dest*100);
	        				rowProperties.put("customHeight", true);
	        			}
	        		}
        		}
        	}
        }
        if(!rowProperties.isEmpty()) {
        	attrs.put("row", rowProperties);
        }
    }

    public boolean isStartingGroup() {
    	return (groupLevel&1)!=0;
    }

    public void setStartingGroup(boolean value) {
		groupLevel|=1;
		if(!value) {
			groupLevel^=1;
    	}
    }

    public int getGroupLevel() {
    	return groupLevel>>1;
    }

    public void setGroupLevel(int value) {
    	groupLevel |= value<<1;
    }

	public static int writeGroupDifference(SerializationHandler output, int currentGroupLevel, int newGroupLevel, boolean isStartingGroup)
			throws SAXException {

		int difference = newGroupLevel - currentGroupLevel;
		if(difference>0) {
			SaxContextHandler.startElement(output, TableTableColumnGroupElement.ELEMENT_NAME);
		}
		else if(difference<0) {
			SaxContextHandler.endElement(output, TableTableColumnGroupElement.ELEMENT_NAME);
		}
		else if(isStartingGroup&&currentGroupLevel>0) {
			SaxContextHandler.endElement(output, TableTableColumnGroupElement.ELEMENT_NAME);
			SaxContextHandler.startElement(output, TableTableColumnGroupElement.ELEMENT_NAME);
		}
		return difference;
	}

	public static class RowAttrHash extends AttrsHash<Row> {

		public RowAttrHash(Row val) {
			super(val, val.getRepeated());
		}

		// hash and equals includes: defaultCellStyle, rowStyle and visibility

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime
					* result
					+ ((getObject().getDefaultCellStyle() == null) ? 0 : getObject().getDefaultCellStyle().hashCode());
			result = prime * result
					+ ((getObject().getRowStyle() == null) ? 0 : getObject().getRowStyle().hashCode());
			result = prime * result
					+ ((getObject().getVisibility() == null) ? 0 : getObject().getVisibility().hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			RowAttrHash other = (RowAttrHash) obj;
			if (getObject().getDefaultCellStyle() == null) {
				if (other.getObject().getDefaultCellStyle() != null)
					return false;
			} else if (!getObject().getDefaultCellStyle().equals(other.getObject().getDefaultCellStyle()))
				return false;
			if (getObject().getRowStyle() == null) {
				if (other.getObject().getRowStyle() != null)
					return false;
			} else if (!getObject().getRowStyle().equals(other.getObject().getRowStyle()))
				return false;
			if (getObject().getVisibility() != other.getObject().getVisibility())
				return false;
			return true;
		}
	}
	
	public RowAttrHash createAttrHash() {
		return new RowAttrHash(this);
	}

	@Override
	public int compareTo(Row o) {
		return r - o.getRow();
	}
}
