/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.attribute.table.TableDefaultCellStyleNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableIdAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableNumberColumnsRepeatedAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableStyleNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableVisibilityAttribute;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnGroupElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.type.Length;
import org.odftoolkit.odfdom.type.Length.Unit;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

public class Column implements Comparable<Column>, IAttrs<Column> {

	private String columnStyle;
	private String defaultCellStyle = null;

	private int min;
	private int max;
	private int groupLevel;
	private Visibility visibility = null;
	private String id = null;

	Column(String columnStyle, int valMin) {
		min = valMin;
		max = valMin;
		groupLevel = 0;
		this.columnStyle = columnStyle;
	}

	Column(String columnStyle, int valMin, int valMax) {
		min = valMin;
		max = valMax;
		groupLevel = 0;
		this.columnStyle = columnStyle;
	}

	Column(String columnStyle, int valMin, int valMax, int grpLvl, boolean groupStarted) {
		min = valMin;
		max = valMax;
		setGroupLevel(grpLvl);
		setStartingGroup(groupStarted);
		this.columnStyle = columnStyle;
	}

    public int getMin() {
        return min;
    }

    public void setMin(int value) {
        this.min = value;
    }

    public int getMax() {
        return max;
    }

    public void setMax(int value) {
        this.max = value;
    }

    public String getColumnStyle() {
    	return columnStyle;
    }

    private void setColumnStyle(String columnStyle) {
    	this.columnStyle = columnStyle;
    }

    public void replaceStyle(OfficeStyles officeStyles, String newStyle) {
    	if(getColumnStyle()!=null) {
    		officeStyles.decColumnStyleUsage(getColumnStyle());
    	}
    	if(newStyle!=null) {
    		officeStyles.incColumnStyleUsage(newStyle);
    	}
    	setColumnStyle(newStyle);
    }

    public String getDefaultCellStyle() {
    	return defaultCellStyle;
    }

    public void setDefaultCellStyle(String val) {
    	defaultCellStyle = val;
    }

    public Visibility getVisibility() {
    	return visibility;
    }

    public void setVisibility(Visibility value) {
    	visibility = value;
    }

    public String getId() {
    	return id;
    }

    public void setId(String val) {
    	id = val;
    }

    public void writeObject(SerializationHandler output) {
		SaxContextHandler.startElement(output, TableTableColumnElement.ELEMENT_NAME);
		if(max-min>0) {
			SaxContextHandler.addOdfAttribute(output, TableNumberColumnsRepeatedAttribute.ATTRIBUTE_NAME, null, Integer.toString(((max-min)+1)));
		}
		if(getColumnStyle()!=null) {
			SaxContextHandler.addOdfAttribute(output, TableStyleNameAttribute.ATTRIBUTE_NAME, null, getColumnStyle());
		}
		if(getDefaultCellStyle()!=null) {
			SaxContextHandler.addOdfAttribute(output, TableDefaultCellStyleNameAttribute.ATTRIBUTE_NAME, null, getDefaultCellStyle());
		}
		if(getVisibility()!=null) {
			SaxContextHandler.addOdfAttribute(output, TableVisibilityAttribute.ATTRIBUTE_NAME, null, getVisibility().toString());
		}
		if(getId()!=null) {
			SaxContextHandler.addOdfAttribute(output, TableIdAttribute.ATTRIBUTE_NAME, null, getId());
		}
		SaxContextHandler.endElement(output, TableTableColumnElement.ELEMENT_NAME);
	}

    public void createAttributes(JSONObject attrs, OdfSpreadsheetDocument doc)
    	throws JSONException, SAXException {

        final JSONObject columnProperties = new JSONObject(2);
        if(getVisibility()!=null&&getVisibility()==Visibility.COLLAPSE) {
            columnProperties.put("visible", false);
        }
        if(columnStyle!=null&&!columnStyle.isEmpty()) {
        	final OdfOfficeAutomaticStyles autoStyles = ((Content)doc.getContentDom()).getAutomaticStyles();
        	final OdfStyle odfStyle = autoStyles.getStyle(columnStyle, OdfStyleFamily.TableColumn);
        	if(odfStyle!=null) {
        		final Element tableColumnProperties = odfStyle.getChildElement(OdfDocumentNamespace.STYLE.getUri(), "table-column-properties");
        		if(tableColumnProperties!=null) {
	        		final String width = tableColumnProperties.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "column-width");
	        		if(width!=null&&!width.isEmpty()) {
	        			final Double dest = Length.getLength(width, Unit.MILLIMETER);
	        			if(dest!=null) {
	        				columnProperties.put("width", dest*100);
	        			}
	        		}
        		}
        	}
        }
        if(!columnProperties.isEmpty()) {
        	attrs.put("column", columnProperties);
        }
    }

    public boolean isStartingGroup() {
    	return (groupLevel&1)!=0;
    }

    public void setStartingGroup(boolean value) {
		groupLevel|=1;
		if(!value) {
			groupLevel^=1;
    	}
    }

    public int getGroupLevel() {
    	return groupLevel>>1;
    }

    public void setGroupLevel(int value) {
    	groupLevel |= value<<1;
    }

	public static int writeGroupDifference(SerializationHandler output, int currentGroupLevel, int newGroupLevel, boolean isStartingGroup)
			throws SAXException {

		int difference = newGroupLevel - currentGroupLevel;
		if(difference>0) {
			SaxContextHandler.startElement(output, TableTableColumnGroupElement.ELEMENT_NAME);
		}
		else if(difference<0) {
			SaxContextHandler.endElement(output, TableTableColumnGroupElement.ELEMENT_NAME);
		}
		else if(isStartingGroup&&currentGroupLevel>0) {
			SaxContextHandler.endElement(output, TableTableColumnGroupElement.ELEMENT_NAME);
			SaxContextHandler.startElement(output, TableTableColumnGroupElement.ELEMENT_NAME);
		}
		return difference;
	}

	class ColumnAttrHash extends AttrsHash<Column> {

		public ColumnAttrHash(Column val) {
			super(val, (val.getMax()-val.getMin())+1);
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result
					+ ((columnStyle == null) ? 0 : columnStyle.hashCode());
			result = prime
					* result
					+ ((defaultCellStyle == null) ? 0 : defaultCellStyle.hashCode());
			result = prime * result + ((id == null) ? 0 : id.hashCode());
			result = prime * result
					+ ((visibility == null) ? 0 : visibility.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			ColumnAttrHash other = (ColumnAttrHash) obj;
			if (columnStyle == null) {
				if (other.getObject().getColumnStyle() != null)
					return false;
			} else if (!columnStyle.equals(other.getObject().getColumnStyle()))
				return false;
			if (defaultCellStyle == null) {
				if (other.getObject().getDefaultCellStyle() != null)
					return false;
			} else if (!defaultCellStyle.equals(other.getObject().getDefaultCellStyle()))
				return false;
			if (id == null) {
				if (other.getObject().getId() != null)
					return false;
			} else if (!id.equals(other.getObject().getId()))
				return false;
			if (visibility != other.getObject().getVisibility())
				return false;
			return true;
		}
	}

	public ColumnAttrHash createAttrHash() {
		return new ColumnAttrHash(this);
	}

	@Override
	public int compareTo(Column o) {
		return min - o.getMin();
	}
}
