/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.hazelcast.doc;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.openexchange.exception.OXException;
import com.openexchange.management.ManagementAware;
import com.openexchange.management.ManagementObject;
import com.openexchange.office.hazelcast.access.HazelcastAccess;
import com.openexchange.office.hazelcast.management.HazelcastDocumentDirectoryMBean;
import com.openexchange.office.hazelcast.management.HazelcastDocumentDirectoryManagement;
import com.openexchange.office.hazelcast.serialization.document.PortableDocumentState;
import com.openexchange.office.hazelcast.serialization.document.PortableID;
import com.openexchange.office.tools.directory.DocResourceID;
import com.openexchange.office.tools.directory.DocumentState;
import com.openexchange.osgi.ExceptionUtils;



public class HazelcastDocumentDirectory implements DocumentDirectory, ManagementAware<HazelcastDocumentDirectoryMBean> {

    private final String mapIdentifier;
    private final HazelcastDocumentDirectoryManagement managementObject;

    public HazelcastDocumentDirectory(final String mapIdentifier) {
        this.mapIdentifier = mapIdentifier;
        this.managementObject = new HazelcastDocumentDirectoryManagement(this);
    }

    @Override
    public DocumentState get(DocResourceID id) throws OXException {
        DocumentState foundDoc = null;
        PortableID currentPortableId = new PortableID(id);

        IMap<PortableID, PortableDocumentState> allDocStates = getDocStateMapping();
        PortableDocumentState portableDoc = allDocStates.get(currentPortableId);
        if (portableDoc != null) {
            foundDoc = PortableDocumentState.createFrom(portableDoc);
        }

        return foundDoc;
    }

    @Override
    public Map<DocResourceID, DocumentState> get(Collection<DocResourceID> ids) throws OXException {
        final Map<DocResourceID, DocumentState> foundDocs = new HashMap<DocResourceID, DocumentState>();
        final Set<PortableID> docResIds = new HashSet<PortableID>();

        for (DocResourceID id : ids) {
            docResIds.add(new PortableID(id));
        }

        if (!docResIds.isEmpty()) {
            IMap<PortableID, PortableDocumentState> allResources = getDocStateMapping();
            Map<PortableID, PortableDocumentState> matchingPortableDocs = allResources.getAll(docResIds);
            if (matchingPortableDocs != null) {
                for (Entry<PortableID, PortableDocumentState> entry : matchingPortableDocs.entrySet()) {
                    final PortableID foundID = entry.getKey();
                    final PortableDocumentState foundDoc = entry.getValue();
                    final DocResourceID docResId = DocResourceID.createDocResourceID(foundID.toString());
                    foundDocs.put(docResId, PortableDocumentState.createFrom(foundDoc));
                }
            }
        }

        return foundDocs;
    }

    @Override
    public DocumentState set(final DocResourceID id, final DocumentState docState) throws OXException {
        final PortableDocumentState currentPortableDoc = new PortableDocumentState(docState);
        final PortableID currentPortableID = new PortableID(id);

        PortableDocumentState previousPortableDocState = null;
        DocumentState prevDocState = null;
        try {
            final IMap<PortableID, PortableDocumentState> allDocStates = getDocStateMapping();
            previousPortableDocState = allDocStates.get(currentPortableID);

            if (previousPortableDocState != null) {
                // but the previous resource provides a presence
                allDocStates.put(currentPortableID, currentPortableDoc);
            } else {
                previousPortableDocState = allDocStates.putIfAbsent(currentPortableID, currentPortableDoc);
            }
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            throw new OXException(t);
        }

        if (null != previousPortableDocState) {
            prevDocState = PortableDocumentState.createFrom(previousPortableDocState);
        }

        return prevDocState;
    }

    @Override
    public void setIfAbsent(DocResourceID id, DocumentState docState) throws OXException {
        final PortableDocumentState currentPortableDoc = new PortableDocumentState(docState);
        final PortableID currentPortableID = new PortableID(id);

        PortableDocumentState previousPortableDocState = null;
        try {
            final IMap<PortableID, PortableDocumentState> allDocStates = getDocStateMapping();
            previousPortableDocState = allDocStates.get(currentPortableID);

            if (previousPortableDocState == null) {
                allDocStates.put(currentPortableID, currentPortableDoc);
            }
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            throw new OXException(t);
        }
    }

    @Override
    public DocumentState remove(DocResourceID id) throws OXException {
        final PortableID currentPortableID = new PortableID(id);

        PortableDocumentState previousPortableDocState = null;
        DocumentState prevDocState = null;
        try {
            final IMap<PortableID, PortableDocumentState> allDocStates = getDocStateMapping();
            previousPortableDocState = allDocStates.get(currentPortableID);

            if (previousPortableDocState == null) {
                allDocStates.remove(currentPortableID);
            }
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            throw new OXException(t);
        }

        if (null != previousPortableDocState) {
            prevDocState = PortableDocumentState.createFrom(previousPortableDocState);
        }

        return prevDocState;
    }

    @Override
    public Map<DocResourceID, DocumentState> remove(Collection<DocResourceID> ids) throws OXException {
        final Map<DocResourceID, DocumentState> removed = new HashMap<DocResourceID, DocumentState>();
        final Set<PortableID> docResIds = new HashSet<PortableID>();

        for (DocResourceID id : ids) {
            docResIds.add(new PortableID(id));
        }

        if (!docResIds.isEmpty()) {
            IMap<PortableID, PortableDocumentState> allResources = getDocStateMapping();
            Map<PortableID, PortableDocumentState> matchingPortableDocs = allResources.getAll(docResIds);

            if (matchingPortableDocs != null) {
                for (Entry<PortableID, PortableDocumentState> entry : matchingPortableDocs.entrySet()) {
                    final PortableID foundID = entry.getKey();
                    final PortableDocumentState foundDoc = entry.getValue();
                    final DocResourceID docResId = DocResourceID.createDocResourceID(foundID.toString());
                    removed.put(docResId, PortableDocumentState.createFrom(foundDoc));

                    allResources.remove(docResId);
                }
            }
        }

        return removed;
    }

    @Override
    public ManagementObject<HazelcastDocumentDirectoryMBean> getManagementObject() {
        return managementObject;
    }

    /**
     * Get the mapping of full IDs to the Resource e.g. ox://marc.arens@premium/random <-> Resource. The resource includes the
     * {@link RoutingInfo} needed to address clients identified by the {@link ID}
     *
     * @return the map used for mapping full IDs to ResourceMaps.
     * @throws OXException if the map couldn't be fetched from hazelcast
     */
    public IMap<PortableID, PortableDocumentState> getDocStateMapping() throws OXException {
        HazelcastInstance hazelcast = HazelcastAccess.getHazelcastInstance();
        return hazelcast.getMap(mapIdentifier);
    }
}
