/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.lang.mutable.MutableInt;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;

import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openxchange.office_communication.tools.common.SerializationUtils;

//=============================================================================
public class CalcEngineRecordedFile
{
	//-------------------------------------------------------------------------
	public static final String EXTENSION = ".ops";

	//-------------------------------------------------------------------------
	public static final String ENCODING = "utf-8";
	
	//-------------------------------------------------------------------------
	public static final boolean APPEND = true;

	//-------------------------------------------------------------------------
	public static final String LINE_ENDING = "\n";

	//-------------------------------------------------------------------------
	public static final String KEY_OP       = "operation";
	public static final String KEY_REQUEST  = "request"  ;
	public static final String KEY_RESPONSE = "response" ;
	public static final String KEY_ERROR    = "error"    ;

	//-------------------------------------------------------------------------
    public static final String OP_EXECUTE = "op_execute";
    public static final String OP_COPY    = "op_copy"   ;
    public static final String OP_PASTE   = "op_paste"  ;

	//-------------------------------------------------------------------------
	public static final String CLIPBOARDEVENT_SEPARATOR = "%|%";

	//-------------------------------------------------------------------------
	public CalcEngineRecordedFile ()
	    throws Exception
	{}

	//-------------------------------------------------------------------------
	public void setFile (final File aFile)
		throws Exception
	{
		m_aFile = aFile;
	}
	
	//-------------------------------------------------------------------------
	public void setFile (final InputStream aStream)
		throws Exception
	{
		m_aStream = aStream;
	}

	//-------------------------------------------------------------------------
	public void create ()
		throws Exception
	{
		m_aFile.createNewFile();
	}

	//-------------------------------------------------------------------------
	public void close ()
		throws Exception
	{
		// ...
	}

	//-------------------------------------------------------------------------
	public < T extends Serializable > void recordRequest (final String sOp  ,
							   	  						  final T      aData)
		throws Exception
	{
		final String sData = SerializationUtils.mapObject2String(aData);

		final List< String > lLines = new ArrayList< String > ();

		lLines.add (KEY_OP     );
		lLines.add (sOp        );
		lLines.add (KEY_REQUEST);
		lLines.add (sData      );

		FileUtils.writeLines(m_aFile, ENCODING, lLines, LINE_ENDING, APPEND);
	}

	//-------------------------------------------------------------------------
	public < T extends Serializable > void recordResponse (final String sOp  ,
							   	  						   final T      aData)
		throws Exception
	{
		final String sData = SerializationUtils.mapObject2String(aData);

		final List< String > lLines = new ArrayList< String > ();

		lLines.add (KEY_RESPONSE);
		lLines.add (sData       );

		FileUtils.writeLines(m_aFile, ENCODING, lLines, LINE_ENDING, APPEND);
	}

	//-------------------------------------------------------------------------
	public void recordError (final String    sOp   ,
							 final Throwable aError)
		throws Exception
	{
		final String sData = SerializationUtils.mapObject2String(aError);

		final List< String > lLines = new ArrayList< String > ();

		lLines.add (KEY_ERROR);
		lLines.add (sData    );

		FileUtils.writeLines(m_aFile, ENCODING, lLines, LINE_ENDING, APPEND);
	}

	//-------------------------------------------------------------------------
	public Triple< String, Object, Object > getFirst ()
		throws Exception
	{
		m_nActLine.setValue(0);
		return impl_readItem (m_nActLine);
	}
	
	//-------------------------------------------------------------------------
	public Triple< String, Object, Object > getNext ()
		throws Exception
	{
		return impl_readItem (m_nActLine);
	}

	//-------------------------------------------------------------------------
	private Triple< String, Object, Object > impl_readItem (final MutableInt nIndex)
			throws Exception
	{
		final List< String > lLines = mem_FileContent ();
		
		if (nIndex.intValue() >= lLines.size())
			return null;
		
		final String sOpKey       = lLines.get(nIndex.intValue());
		nIndex.increment();
		final String sOp          = lLines.get(nIndex.intValue());
		nIndex.increment();

		final String sRequestKey  = lLines.get(nIndex.intValue());
		nIndex.increment();
		final String sRequest     = lLines.get(nIndex.intValue());
		nIndex.increment();
		
		final String sResponseKey = lLines.get(nIndex.intValue());
		nIndex.increment();
		final String sResponse    = lLines.get(nIndex.intValue());
		nIndex.increment();
		
		Validate.isTrue (StringUtils.equals(sOpKey      , KEY_OP      ), "KEY_OP not valid."      );
		Validate.isTrue (StringUtils.equals(sRequestKey , KEY_REQUEST ), "KEY_REQUEST not valid." );
		Validate.isTrue (StringUtils.equals(sResponseKey, KEY_RESPONSE)
				      || StringUtils.equals(sResponseKey, KEY_ERROR   ), "KEY_RESPONSE not valid.");

		Validate.notEmpty(sOp      , "Miss operation.");
		Validate.notEmpty(sRequest , "Miss request."  );
	  //Validate.notEmpty(sResponse, "Miss response." );
		
		final Object aRequest  = SerializationUtils.mapString2Object(sRequest );
		final Object aResponse = SerializationUtils.mapString2Object(sResponse);
		
		final Triple< String, Object, Object > aItem = Triple.of(sOp, aRequest, aResponse);
		return aItem;
	}
	
	//-------------------------------------------------------------------------
	public static Pair< String, String > parsePasteRequest (final String sRequest)
	    throws Exception
	{
		final String                 sHandle    = StringUtils.substringBefore(sRequest, CalcEngineRecordedFile.CLIPBOARDEVENT_SEPARATOR);
		final String                 sSelection = StringUtils.substringAfter (sRequest, CalcEngineRecordedFile.CLIPBOARDEVENT_SEPARATOR);
		final Pair< String, String > lParts     = Pair.of(sHandle, sSelection);
		return lParts;
	}

	//-------------------------------------------------------------------------
	public static String constructPasteRequest (final CalcEngineClipBoardEvent aEvent)
	    throws Exception
	{
		final StringBuffer sRequest = new StringBuffer (256);
		sRequest.append (aEvent.sClipboardHandle    );
		sRequest.append (CalcEngineRecordedFile.CLIPBOARDEVENT_SEPARATOR   );
		sRequest.append (aEvent.sTargetSelectionJSON);
		return sRequest.toString ();
	}

	//-------------------------------------------------------------------------
	public static Pair< String, String > parseCopyRequest (final String sRequest)
	    throws Exception
	{
		final String                 sHandle    = StringUtils.substringBefore(sRequest, CalcEngineRecordedFile.CLIPBOARDEVENT_SEPARATOR);
		final String                 sSelection = StringUtils.substringAfter (sRequest, CalcEngineRecordedFile.CLIPBOARDEVENT_SEPARATOR);
		final Pair< String, String > lParts     = Pair.of(sHandle, sSelection);
		return lParts;
	}

	//-------------------------------------------------------------------------
	public static String constructCopyRequest (final CalcEngineClipBoardEvent aEvent)
	    throws Exception
	{
		final StringBuffer sRequest = new StringBuffer (256);
		sRequest.append (aEvent.sClipboardHandle    );
		sRequest.append (CalcEngineRecordedFile.CLIPBOARDEVENT_SEPARATOR   );
		sRequest.append (aEvent.sSourceSelectionJSON);
		return sRequest.toString ();
	}

	//-------------------------------------------------------------------------
	public static void verifyFiles (final File aFolder)
		throws Exception
	{
		final File[] lFiles = aFolder.listFiles();
		for (final File aFile : lFiles)
			verifyFile (aFile);
	}

	//-------------------------------------------------------------------------
	public static void verifyFile (final File aFile)
		throws Exception
	{
		System.out.println ("##### verify file '"+aFile.getAbsolutePath()+"' =======================");
		
		final CalcEngineRecordedFile aVerify = new CalcEngineRecordedFile ();
		aVerify.setFile(aFile);
		
		Triple< String, Object, Object > aItem = aVerify.getFirst();
		System.out.println (toHumanReadableString(aItem));
		
		while (true)
		{
			aItem = aVerify.getNext();
			if (aItem == null)
				break;
			System.out.println (toHumanReadableString(aItem));
		}
	}
	
	//-------------------------------------------------------------------------
	public static String toHumanReadableString (final Triple< String, Object, Object > aItem)
	    throws Exception
	{
		if (aItem == null)
			return "<null>";

		final String sOp       =          aItem.getLeft  ();
		final String sRequest  = (String) aItem.getMiddle();
		final String sResponse = (String) aItem.getRight ();
		
		final StringBuffer sString = new StringBuffer (256);

		sString.append (KEY_OP           );
		sString.append (" = "            );
		sString.append (sOp              );
		sString.append ("\n"             );
		
		if (StringUtils.equals(sOp, OP_EXECUTE))
		{
			sString.append (KEY_REQUEST      );
			sString.append (" = "            );
			sString.append (aItem.getMiddle());
			sString.append ("\n"             );
		}
		else
		if (StringUtils.equals(sOp, OP_COPY))
		{
			final Pair< String, String > lParts = parseCopyRequest (sRequest);
			sString.append ("clipboard-handle = ");
			sString.append (lParts.getLeft()     );
			sString.append ("\n"                 );
			sString.append ("source-selection = ");
			sString.append (lParts.getRight()    );
			sString.append ("\n"                 );
		}
		else
		if (StringUtils.equals(sOp, OP_PASTE))
		{
			final Pair< String, String > lParts = parsePasteRequest (sRequest);
			sString.append ("clipboard-handle = ");
			sString.append (lParts.getLeft()     );
			sString.append ("\n"                 );
			sString.append ("target-selection = ");
			sString.append (lParts.getRight()    );
			sString.append ("\n"                 );
		}
		else
			throw new UnsupportedOperationException ("Support for op '"+sOp+"' not implemented.");
		
		sString.append (KEY_RESPONSE);
		sString.append (" = "       );
		sString.append (sResponse   );
		sString.append ("\n"        );

		return sString.toString ();
	}
	
	//-------------------------------------------------------------------------
	private List< String > mem_FileContent ()
	    throws Exception
	{
		if (m_aFileContent != null)
			return m_aFileContent;

		if (m_aFile != null)
			m_aFileContent = FileUtils.readLines(m_aFile, ENCODING);
		else
		if (m_aStream != null)
			m_aFileContent = IOUtils.readLines(m_aStream, ENCODING);
		else
			throw new IOException ("No file nor any stream content defined.");

		return m_aFileContent;
	}
	
	//-------------------------------------------------------------------------
	public static void main (final String[] lArgs)
		throws Exception
	{
		final File aFolder = new File("/tmp/ce-recording");
		verifyFiles (aFolder);
	}
	
	//-------------------------------------------------------------------------
	private File m_aFile = null;

	//-------------------------------------------------------------------------
	private InputStream m_aStream = null;

	//-------------------------------------------------------------------------
	private List< String > m_aFileContent = null;

	//-------------------------------------------------------------------------
	private MutableInt m_nActLine = new MutableInt (0);
}
