/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;

import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
/** Can be used to load balance requests to sticky-ness or sticky-less target channels.
 * 
 *  This load balancer does not know the real channel, the real connection,
 *  the real queue ... it knows IDs of those target channel only.
 *  It's up to you to provide such target channel ID and to know how it works.
 *
 *  We implement the container for counting requests and implement different
 *  balance algorithm on top of that.
 *  
 *  Furthermore this container provide information which target channel needs
 *  a restart and shouldn't be used any longer. But it's up to you to handle that information right.
 */
public class CalcEngineLoadBalancer
{
	//-------------------------------------------------------------------------
	private static final Log LOG = LogFactory.getJclLog(CalcEngineLoadBalancer.class);

	//-------------------------------------------------------------------------
	public static boolean DEFAULT_STATE_4_ACTIVE = true;
	
    //-------------------------------------------------------------------------
	private CalcEngineLoadBalancer ()
	{}

	//-------------------------------------------------------------------------
	public static synchronized CalcEngineLoadBalancer get (final String sBalancerID)
	    throws Exception
	{
		Validate.notEmpty(sBalancerID, "Invalid argument 'balancerId'.");
		
		final Map< String, CalcEngineLoadBalancer > aRegistry = mem_BalancerRegistry ();
		      CalcEngineLoadBalancer                aBalancer = aRegistry.get (sBalancerID);
		
		if (aBalancer == null)
		{
			aBalancer = new CalcEngineLoadBalancer ();
			aBalancer.m_sID = sBalancerID;
			aRegistry.put(sBalancerID, aBalancer);
		}
		
		return aBalancer;
	}

	//-------------------------------------------------------------------------
	public synchronized void setRequestThreshold4ChannelExchange (final long nTreshold)
	    throws Exception
	{
		Validate.isTrue (nTreshold > 0, "Invalid argument 'treshold'.");
		m_nThreshold4ChannelExchange_RequestCount = nTreshold;
	}
	
	//-------------------------------------------------------------------------
	public synchronized void setDocThreshold4ChannelExchange (final long nTreshold)
	    throws Exception
	{
		Validate.isTrue (nTreshold > 0, "Invalid argument 'treshold'.");
		m_nThreshold4ChannelExchange_DocCount = nTreshold;
	}

	//-------------------------------------------------------------------------
	public synchronized void registerTargetChannel (final String sChannelID)
	    throws Exception
	{
		registerTargetChannel (sChannelID, DEFAULT_STATE_4_ACTIVE);
	}

	//-------------------------------------------------------------------------
	public synchronized void registerTargetChannel (final String  sChannelID,
												    final boolean bActive   )
	    throws Exception
	{
		Validate.notEmpty(sChannelID, "Invalid argument 'channelId'.");
		
		final Map< String, ChannelDescriptor > lInfos = mem_Channels ();
		      ChannelDescriptor                aInfo  = lInfos.get(sChannelID);

		if (aInfo == null)
		{
			aInfo               = new ChannelDescriptor ();
			aInfo.sChannelID    = sChannelID;
			aInfo.bActive       = bActive;
			aInfo.nRefCount     = 0;
			aInfo.nRequestCount = 0;
			lInfos.put(sChannelID, aInfo);
		}
	}

	//-------------------------------------------------------------------------
	/** reserve those mentioned channels by deactivating them.
	 *  <p>
	 *  Later on you can call {@link exchangeChannel} to deactivate one
	 *  channel and activate another one from the list of reserved channels.
	 *  </p>
	 * 
	 *  @attention THIS METHOD HAS TO BE CALLED AFTER {@link registerTargetChannel} !
	 *  
	 *  @param	lChannels [IN]
	 *  		the list of channels to be deactivates and handled as reserve.
	 */
	public synchronized void reserveChannelsForExchange (final String... lChannels)
	    throws Exception
	{
		Validate.notEmpty (lChannels, "Invalid argument 'channels'.");
		
		for (String sChannel : lChannels)
		{
			final ChannelDescriptor aChannel = impl_accessChannel (sChannel);
			aChannel.bActive = false;
		}
	}

	//-------------------------------------------------------------------------
	public synchronized String getBalancedChannel ()
	    throws Exception
	{
		final Map< String, ChannelDescriptor > lChannels     = mem_Channels ();
		final int                              nChannelCount = lChannels.size ();

		if (nChannelCount < 1)
			throw new RuntimeException ("No channels configured. Did you call registerTargetChannel () ?");
		
		final int                           nMaxChannelIndex    = nChannelCount - 1;
		final Iterator< ChannelDescriptor > pChannels           = lChannels.values ().iterator();
			  long                          nLowestRequestCount = Long.MAX_VALUE;
			  ChannelDescriptor             rFirstChannel       = null;
			  ChannelDescriptor             rMinUsedChannel     = null;
			  ChannelDescriptor             rRoundRobinChannel  = null;
		
		if (m_nRoundRobinIndex >= nMaxChannelIndex)
			m_nRoundRobinIndex = 0;
		else
			m_nRoundRobinIndex++;
		
		int nChannelIndex = 0;
		while (pChannels.hasNext())
		{
			final ChannelDescriptor aChannel = pChannels.next();
			
			if ( ! aChannel.bActive)
				continue;
			
			if (rFirstChannel == null)
				rFirstChannel = aChannel;
			
			if (aChannel.nRequestCount < nLowestRequestCount)
			{
				rMinUsedChannel     = aChannel;
				nLowestRequestCount = aChannel.nRequestCount;
			}
			
			if (nChannelIndex == m_nRoundRobinIndex)
				rRoundRobinChannel = aChannel;
			nChannelIndex++;
		}

		if (rRoundRobinChannel != null)
			return rRoundRobinChannel.sChannelID;

		if (rMinUsedChannel != null)
			return rMinUsedChannel.sChannelID;
		
		if (rFirstChannel != null)
			return rFirstChannel.sChannelID;
		
		// ???
		throw new RuntimeException ("Shouldnt be reached ... but it was .-)");
	}

	//-------------------------------------------------------------------------
	public synchronized void bindItemToChannel (final String sChannelID,
										        final String sItem     )
	    throws Exception
	{
		Validate.notEmpty(sChannelID, "Invalid argument 'channelId'.");
		Validate.notEmpty(sItem     , "Invalid argument 'item'."     );
		
		final Map< String, String > lItems = mem_Item2ChannelMap ();
		lItems.put(sItem, sChannelID);

        final ChannelDescriptor aChannel = impl_accessChannel (sChannelID);
		aChannel.nRefCount++;
		
		impl_dumpState ();
	}

	//-------------------------------------------------------------------------
	public synchronized void unbindItemFromChannel (final String sItem)
	    throws Exception
	{
		Validate.notEmpty(sItem, "Invalid argument 'item'.");
		
		final Map< String, String > lItems     = mem_Item2ChannelMap ();
		final String                sChannelID = lItems.get(sItem);
		lItems.remove(sItem);
		
		if (StringUtils.isEmpty(sChannelID))
		{
			LOG.warn("No channel for item '"+sItem+"' ?!");
		}
		else
		{
	        final ChannelDescriptor aChannel = impl_accessChannel (sChannelID);
			aChannel.nRefCount--;
		}

		impl_dumpState ();
	}

	//-------------------------------------------------------------------------
	public synchronized String getChannelForItem (final String sItem)
	    throws Exception
	{
		Validate.notEmpty(sItem, "Invalid argument 'item'.");
		
		final Map< String, String > aMap     = mem_Item2ChannelMap ();
		final String                sChannel = aMap.get (sItem);
		
		return sChannel;
	}

	//-------------------------------------------------------------------------
	public synchronized long countRequest (final String sChannelID,
										   final long   nRequests )
	    throws Exception
	{
		final ChannelDescriptor aChannel = impl_accessChannel (sChannelID);

	    if (aChannel.nRequestCount >= Long.MAX_VALUE)
	    	throw new RuntimeException ("Congratulation : your request count exceeded the number of "+Long.MAX_VALUE+". Game is over now.");

	    aChannel.nRequestCount    += nRequests;
	    aChannel.nRequestCountSum += nRequests;
		impl_dumpState ();
		
		return aChannel.nRequestCount;
	}

	//-------------------------------------------------------------------------
	public synchronized void resetRequestCountForChannel (final String sChannelID)
	    throws Exception
	{
		final ChannelDescriptor aChannel = impl_accessChannel (sChannelID);
	    aChannel.nRequestCount = 0;
		impl_dumpState ();
	}

	//-------------------------------------------------------------------------
	public synchronized long countDocuments (final String sChannelID,
										     final long   nDocs     )
	    throws Exception
	{
		final ChannelDescriptor aChannel = impl_accessChannel (sChannelID);

	    if (aChannel.nDocCount >= Long.MAX_VALUE)
	    	throw new RuntimeException ("Congratulation : your doc count exceeded the number of "+Long.MAX_VALUE+". Game is over now.");

	    aChannel.nDocCount += nDocs;
		impl_dumpState ();
		
		return aChannel.nDocCount;
	}

	//-------------------------------------------------------------------------
	public synchronized void resetDocCountForChannel (final String sChannelID)
	    throws Exception
	{
		final ChannelDescriptor aChannel = impl_accessChannel (sChannelID);
	    aChannel.nDocCount = 0;
		impl_dumpState ();
	}

	//-------------------------------------------------------------------------
	/** return true if channel needs to be recreated/restarted.
	 * 
	 *  Some trigger are used internally ...
	 *  e.g. : request count, document count.
	 *  If those counts exceed a configured threshold
	 *  this method return TRUE.
	 *  
	 *  @note	A threshold is active only if it's value is > 0 !
	 * 
	 * 	@param	sChannelID [IN]
	 * 			the channel where thresholds has to be checked.
	 *
	 * 	@return TRUE if this channel needs to be restarted; FALSE otherwise.
	 */
	public synchronized boolean needsChannelExchange (final String sChannelID)
	    throws Exception
	{
		final ChannelDescriptor aChannel       = impl_accessChannel (sChannelID);
		      boolean           bNeedsExchange = false;
		
		if ( ! bNeedsExchange && m_nThreshold4ChannelExchange_RequestCount > 0)
			bNeedsExchange = (aChannel.nRequestCount > m_nThreshold4ChannelExchange_RequestCount);
		
		if ( ! bNeedsExchange && m_nThreshold4ChannelExchange_DocCount > 0)
			bNeedsExchange = (aChannel.nDocCount > m_nThreshold4ChannelExchange_DocCount);
		
		return bNeedsExchange;
	}
	
	//-------------------------------------------------------------------------
	public synchronized void activateChannel (final String sChannelID)
	    throws Exception
	{
		final ChannelDescriptor aChannel = impl_accessChannel (sChannelID);
		aChannel.bActive = true;
		impl_dumpState ();
	}

	//-------------------------------------------------------------------------
	public synchronized void deactivateChannel (final String sChannelID)
	    throws Exception
	{
		final ChannelDescriptor aChannel = impl_accessChannel (sChannelID);
		aChannel.bActive = false;
		impl_dumpState ();
	}
	
	//-------------------------------------------------------------------------
	public synchronized boolean exchangeChannel (final String                  sChannel ,
												 final INotify4OutboundChannel iListener)
	    throws Exception
	{
		final ChannelDescriptor aChannelFrom = impl_accessChannel   (sChannel);
		final ChannelDescriptor aChannelTo   = impl_findBestReservedChannel ();
		
		if (aChannelTo == null)
			return false;
		
		aChannelFrom.bActive = false;
		aChannelTo  .bActive = true ;
		
		// this do not block ... it run asynchronous !
		impl_asyncWaitForOutboundChannelAndNotify (aChannelFrom, iListener);
		
		return true;
	}

	//-------------------------------------------------------------------------
	private ChannelDescriptor impl_accessChannel (final String sChannelID)
	    throws Exception
	{
		Validate.notEmpty(sChannelID, "Invalid argument 'targetChannelId'.");

		final Map< String, ChannelDescriptor > lChannels = mem_Channels ();
	    final ChannelDescriptor                aChannel  = lChannels.get(sChannelID);
	    
	    if (aChannel == null)
	    	throw new RuntimeException ("Channel '"+sChannelID+"' is unknown.");
	    
	    return aChannel;
	}
	
	//-------------------------------------------------------------------------
	private ChannelDescriptor impl_findBestReservedChannel ()
	    throws Exception
	{
		final Iterator< ChannelDescriptor > lChannels        = mem_Channels ().values ().iterator ();
		      long                          nMinRequestCount = Long.MAX_VALUE;
		      ChannelDescriptor             aFirstChannel    = null;
		      ChannelDescriptor             aMinUsedChannel  = null;

		while (lChannels.hasNext())
		{
			final ChannelDescriptor aChannel = lChannels.next();
			
			// a) active=false means it's a reserved channel
			//    ignore non reserved channels
			if (aChannel.bActive)
				continue;
			
		    // b) ref-count > 0 is not a real reserved channel.
			//    ref count has to be 0.
			if (aChannel.nRefCount > 0)
				continue;
			
			if (aFirstChannel == null)
				aFirstChannel = aChannel;
			
			if (aChannel.nRequestCount < nMinRequestCount)
				aMinUsedChannel = aChannel;
		}
		
		if (aMinUsedChannel != null)
			return aMinUsedChannel;
		
		if (aFirstChannel != null)
			return aFirstChannel;

		return null;
	}
	
	//-------------------------------------------------------------------------
	public interface INotify4OutboundChannel
	{
		public void notifyOutboundChannel (final String sChannelID)
			throws Exception;
	}
	
	//-------------------------------------------------------------------------
	private void impl_asyncWaitForOutboundChannelAndNotify (final ChannelDescriptor       aChannel ,
												            final INotify4OutboundChannel iListener)
	    throws Exception
	{
		if (
			(iListener == null) ||
			(aChannel  == null)
		   )
			return;
		
		// tricky :-)
		// safe this as lock for later synchronize ...
		final Object aSync = this;
		
		new Thread ()
		{
			@Override
			public void run ()
			{
				try
				{
					while (true)
					{
						synchronized(this)
						{
							wait (250);
						}
						
						
						String sChannelID = null;
						long   nRefCount  = 0;
						synchronized (aSync)
						{
							sChannelID = aChannel.sChannelID;
							nRefCount  = aChannel.nRefCount ;
						}

						if (nRefCount < 1)
						{
							iListener.notifyOutboundChannel(sChannelID);
							break;
						}
					}
				}
				catch (Throwable ex)
				{}
			}
		}.start ();
	}

	//-------------------------------------------------------------------------
	private void impl_dumpState ()
	    throws Exception
	{
		final StringBuffer sDump = new StringBuffer (256);
		
		sDump.append ("======================================\n");
		sDump.append ("LB '"+m_sID+"'\n"                        );
		sDump.append ("\n"                                      );
		sDump.append ("channel map : \n"                        );

		final Iterator< ChannelDescriptor > lChannels = mem_Channels ().values ().iterator ();
		while (lChannels.hasNext())
		{
			final ChannelDescriptor aChannel = lChannels.next();
			sDump.append ("channel '"              +aChannel.sChannelID   +"'");
			sDump.append (" : ref-count="          +aChannel.nRefCount        );
			sDump.append (" : request-count="      +aChannel.nRequestCount    );
			sDump.append (" : request-count (sum)="+aChannel.nRequestCountSum );
			sDump.append (" : doc-count="          +aChannel.nDocCount        );
			sDump.append (" : is-active="          +aChannel.bActive          );
			sDump.append ("\n"                                          	  );
		}

		sDump.append ("\n"           );
		sDump.append ("item map : \n");
		final Map< String, String >               lItems = mem_Item2ChannelMap ();
		final Iterator< Entry< String, String > > pItems = lItems.entrySet().iterator();
		
		if (lItems.isEmpty())
			sDump.append ("empty\n");
		
		while (pItems.hasNext())
		{
			final Entry< String, String > aItem    = pItems.next   ();
			final String                  sItem    = aItem.getKey  ();
			final String                  sChannel = aItem.getValue();
			
			sDump.append ("item '"+sItem+"' -> channel '"+sChannel+"'\n");
		}
		
		LOG.trace (sDump.toString ());
	}
	
	//-------------------------------------------------------------------------
	private static Map< String, CalcEngineLoadBalancer > mem_BalancerRegistry ()
	    throws Exception
	{
		if (m_lBalancerRegistry == null)
			m_lBalancerRegistry = new HashMap< String, CalcEngineLoadBalancer >();
		return m_lBalancerRegistry;
	}

	//-------------------------------------------------------------------------
	private Map< String, ChannelDescriptor > mem_Channels ()
	    throws Exception
	{
		if (m_lChannels == null)
			m_lChannels = new HashMap< String, ChannelDescriptor > ();
		return m_lChannels;
	}
	
	//-------------------------------------------------------------------------
	private Map< String, String > mem_Item2ChannelMap ()
	    throws Exception
	{
		if (m_lItem2ChannelMap == null)
			m_lItem2ChannelMap = new HashMap< String, String > ();
		return m_lItem2ChannelMap;
	}

	//-------------------------------------------------------------------------
	private class ChannelDescriptor
	{
		/** the id of a port, process, queue where all requests has to be routed for sticky calc engine requests
		 *  Its the key of the map where this channel descriptor is bound as value.
		 */
		public String sChannelID = null;

		/** count all references to this channel.
		 *  At least it's the count of all bound items for this channel.
		 */
		public long nRefCount = 0;
		
		/** count requests to know if a calc engine needs to be restarted (before it becomes dirty)
		 */
		public long nRequestCount = 0;
		
		/** calculate sum of all requests ignoring resets
		 */
		public long nRequestCountSum = 0;
		
		/** count documents bound to this channel ...
		 *  Used to decide if a worker needs a restart after X documents.
		 */
		public long nDocCount = 0;

		/** mark a channel as active/inactive.
		 *  This is define from outside.
		 *  We use that information for load balancing calculation only.
		 */
		public boolean bActive = false;
	}
	
	//-------------------------------------------------------------------------
	private static Map< String, CalcEngineLoadBalancer > m_lBalancerRegistry = null;

	//-------------------------------------------------------------------------
	private String m_sID = null;
	
	//-------------------------------------------------------------------------
	private Map< String, ChannelDescriptor > m_lChannels = null;

	//-------------------------------------------------------------------------
	private Map< String, String > m_lItem2ChannelMap = null;

	//-------------------------------------------------------------------------
	private long m_nThreshold4ChannelExchange_RequestCount = 0;
	
	//-------------------------------------------------------------------------
	private long m_nThreshold4ChannelExchange_DocCount = 0;

	//-------------------------------------------------------------------------
	private int m_nRoundRobinIndex = 0;
}
