/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools;

import java.util.HashMap;
import com.openexchange.office.tools.doc.DocumentType;

/**
 * Helper class to provide certain document based mime types and
 * function to determine mime-types from file extensions.
 *
 * @author Carsten Driesner
 *
 */
public class MimeTypeHelper {
    private static HashMap<String, String> m_mapExtensionsToMimeType = new HashMap<String, String>();
    private static HashMap<String, DocumentType> m_mapMimeTypeToDocumentType = new HashMap<String, DocumentType>();

    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";
    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_MACRO = "application/vnd.ms-excel.sheet.macroEnabled.12";
    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE = "application/vnd.openxmlformats-officedocument.spreadsheetml.template";
    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO = "application/vnd.ms-excel.template.macroEnabled.12";

    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION = "application/vnd.openxmlformats-officedocument.presentationml.presentation";
    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_MACRO = "application/vnd.ms-powerpoint.presentation.macroEnabled.12";
    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE = "application/vnd.openxmlformats-officedocument.presentationml.template";
    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO = "application/vnd.ms-powerpoint.presentation.macroEnabled.12";

    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING = "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_MACRO = "application/vnd.ms-word.document.macroEnabled.12";
    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE = "application/vnd.openxmlformats-officedocument.wordprocessingml.template";
    public static String MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE_MACRO = "application/vnd.ms-word.template.macroEnabled.12";

    public static String MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT = "application/vnd.oasis.opendocument.text";
    public static String MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE = "application/vnd.oasis.opendocument.text-template";

    public static String MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET = "application/vnd.oasis.opendocument.spreadsheet";
    public static String MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE = "application/vnd.oasis.opendocument.spreadsheet-template";

    public static String MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION = "application/vnd.oasis.opendocument.presentation";
    public static String MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE = "application/vnd.oasis.opendocument.presentation-template";

    public static String MIMETYPE_MSEXCEL = "application/vnd.ms-excel";
    public static String MIMETYPE_MSPOWERPOINT = "application/vnd.ms-powerpoint";
    public static String MIMETYPE_MSWORD = "application/msword";

    public static String EXCEL_TEMPLATE_EXTENSIONS[] = { "xltx", "xltm", "xlt" };
    public static String WORD_TEMPLATE_EXTENSIONS[] = { "dotx", "dotm", "dot" };
    public static String POWERPOINT_TEMPLATE_EXTENSIONS[] = { "potx", "pptm", "pot" };
    public static String ODF_CALC_TEMPLATE_EXTENSIONS[] = { "ots" };
    public static String ODF_WRITER_TEMPLATE_EXTENSIONS[] = { "ott" };
    public static String ODF_IMPRESS_TEMPLATE_EXTENSIONS[] = { "otp" };
    
    public static String SPREADSHEET_EXTENSIONS[] = { "xlsx", "xlsm", "xltx", "xltm" };
    public static String TEXT_EXTENSIONS[] = { "docx", "docm", "dotx", "dotm", "odt" };
    public static String PRESENTATION_EXTENSIONS[] = { "pptx", "pptm", "potx", "potm", "otp" };

    /**
     * Provides the mime type of the a known template file extension.
     *
     * @param templateExt
     *  A template document file extension where the mime type should be
     *  retrieved for.
     *
     * @return
     *  The mime type for the template document file extensions or null
     *  if the extension is unknown, null or not a template document extension.
     */
    public static String getMimeTypeFromTemplateExtension(String templateExt) {
        synchronized (m_mapExtensionsToMimeType) {
            initStaticMimeTypeMap();
            return m_mapExtensionsToMimeType.get(templateExt);
        }
    }

    /**
     * Provides the document type for a provided template mime type.
     *
     * @param templateMimeType
     *  A template document mime type.
     *
     * @return
     *  The document type or null if the mime type couldn't be recognized as
     *  a supported document template type.
     */
    public static DocumentType getDocumentTypeFromTemplateMimeType(String templateMimeType) {
        synchronized (m_mapMimeTypeToDocumentType) {
            initStaticDocumentTypeMap();
            return m_mapMimeTypeToDocumentType.get(templateMimeType);
        }
    }

    /**
     * Initializes the static maps to speed up the search.
     */
    private static void initStaticMimeTypeMap() {

        if ((null != m_mapExtensionsToMimeType) && (m_mapExtensionsToMimeType.isEmpty())) {
            // Excel
            m_mapExtensionsToMimeType.put(EXCEL_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE);
            m_mapExtensionsToMimeType.put(EXCEL_TEMPLATE_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO);
            m_mapExtensionsToMimeType.put(EXCEL_TEMPLATE_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSEXCEL);
            // Powerpoint
            m_mapExtensionsToMimeType.put(POWERPOINT_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE);
            m_mapExtensionsToMimeType.put(POWERPOINT_TEMPLATE_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO);
            m_mapExtensionsToMimeType.put(POWERPOINT_TEMPLATE_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSPOWERPOINT);
            // Word
            m_mapExtensionsToMimeType.put(WORD_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE);
            m_mapExtensionsToMimeType.put(WORD_TEMPLATE_EXTENSIONS[1], MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE_MACRO);
            m_mapExtensionsToMimeType.put(WORD_TEMPLATE_EXTENSIONS[2], MimeTypeHelper.MIMETYPE_MSWORD);
            // Calc
            m_mapExtensionsToMimeType.put(ODF_CALC_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE);
            // Impress
            m_mapExtensionsToMimeType.put(ODF_IMPRESS_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE);
            // Writer
            m_mapExtensionsToMimeType.put(ODF_WRITER_TEMPLATE_EXTENSIONS[0], MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE);
        }
    }

    private static void initStaticDocumentTypeMap() {

        if ((null != m_mapMimeTypeToDocumentType) && (m_mapMimeTypeToDocumentType.isEmpty())) {
            // Excel/Calc
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE, DocumentType.SPREADSHEET);
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO, DocumentType.SPREADSHEET);
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_MSEXCEL, DocumentType.SPREADSHEET);
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE, DocumentType.SPREADSHEET);
            // Powerpoint/Impress
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE, DocumentType.PRESENTATION);
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO, DocumentType.PRESENTATION);
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_MSPOWERPOINT, DocumentType.PRESENTATION);
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE, DocumentType.PRESENTATION);
            // Word/Writer
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE, DocumentType.TEXT);
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE_MACRO, DocumentType.TEXT);
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_MSWORD, DocumentType.TEXT);
            m_mapMimeTypeToDocumentType.put(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE, DocumentType.TEXT);
        }
    }
}
