/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.realtime.impl;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.annotation.concurrent.NotThreadSafe;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.google.common.base.Optional;
import com.openexchange.capabilities.CapabilityService;
import com.openexchange.capabilities.CapabilitySet;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStoragePermission;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.groupware.attach.Attachments;
import com.openexchange.groupware.ldap.User;
import com.openexchange.mail.MailServletInterface;
import com.openexchange.mail.dataobjects.MailPart;
import com.openexchange.office.realtime.debug.SaveDebugProperties;
import com.openexchange.office.realtime.doc.DocumentEventHelper;
import com.openexchange.office.realtime.doc.DocumentStateHelper;
import com.openexchange.office.realtime.doc.ImExportHelper;
import com.openexchange.office.realtime.doc.OXDocument;
import com.openexchange.office.realtime.doc.OXDocument.ResolvedStreamInfo;
import com.openexchange.office.realtime.doc.OXDocument.SaveResult;
import com.openexchange.office.realtime.doc.OXDocumentException;
import com.openexchange.office.realtime.tools.BackupFileHelper;
import com.openexchange.office.realtime.tools.ConnectionHelper;
import com.openexchange.office.realtime.tools.DebugHelper;
import com.openexchange.office.realtime.tools.ExtraData;
import com.openexchange.office.realtime.tools.MessageData;
import com.openexchange.office.tools.ApplicationType;
import com.openexchange.office.tools.RecentFileListManager;
import com.openexchange.office.tools.SessionUtils;
import com.openexchange.office.tools.StorageHelper;
import com.openexchange.office.tools.UserConfigurationHelper;
import com.openexchange.office.tools.UserConfigurationHelper.Mode;
import com.openexchange.office.tools.doc.DocumentFormatHelper;
import com.openexchange.office.tools.doc.DocumentMetaData;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.office.tools.files.DocFileHelper;
import com.openexchange.office.tools.files.DocFileHelper.RenameResult;
import com.openexchange.office.tools.files.FileDescriptor;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.FolderHelper;
import com.openexchange.office.tools.json.JSONHelper;
import com.openexchange.office.tools.message.MessageChunk;
import com.openexchange.office.tools.message.MessageHelper;
import com.openexchange.office.tools.message.MessagePropertyKey;
import com.openexchange.office.tools.message.OperationHelper;
import com.openexchange.office.tools.monitoring.SaveType;
import com.openexchange.office.tools.resource.Resource;
import com.openexchange.office.tools.resource.ResourceHelper;
import com.openexchange.office.tools.resource.ResourceManager;
import com.openexchange.office.tools.rt.IDUtils;
import com.openexchange.office.tools.user.UserHelper;
import com.openexchange.realtime.Asynchronous;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.Duration;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.server.services.ServerServiceRegistry;
import com.openexchange.session.Session;
import com.openexchange.sessiond.SessiondService;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;
import com.openexchange.tools.encoding.Base64;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;


/**
 * The abstract DocumentConnection class implements the necessary specialties of a OX Document
 * instance. This includes loading/saving documents, apply changes, switching the edit
 * rights between clients and send update messages to the connected clients. The
 * class doesn't implement application specific parts.
 * {@link Connection}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public @NotThreadSafe
abstract class DocumentConnection extends Connection {
    private final static org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(DocumentConnection.class);

    protected static final long ACTIONS_UNTIL_FORCE_UPDATE = 20;
    private static final int DEFAULT_MAX_TIME_FOR_SENDINGLASTACTIONS = 20000; // time in milliseconds
    private static final int WAIT_FOR_SAVEINPROGRESS_LOCK = 5000;
    private static final int WAIT_FOR_SAVEINPROGRESS_EDITRIGHTS = 10000; // time in milliseconds

    private final String m_resourceID;
    private DocFileHelper m_fileHelper = null;
    private DocumentMetaData m_lastKnownMetaData = null;
    private StorageHelper m_storageHelper = null;
    private ResourceManager m_resourceManager = null;
    private ScheduledTimerTask m_prepareLosingEditRightsTimer = null;
    private ScheduledTimerTask m_completeSwitchingEditRightsTimer = null;
    private CompleteEditRightsTimeoutRunnable m_completeEditRightsRunnable = null;
    private final ConnectionStatus m_connectionStatus = new ConnectionStatus();
    private Sync m_syncStatus = null;
    private ArrayList<MessageChunk> m_messageChunkList = new ArrayList<MessageChunk>();
    private ApplicationType m_appType = ApplicationType.APP_NONE;
    private boolean m_newDocLoaded = false;
    private boolean m_createVersion = true;
    private boolean m_debugOperations = false;
    private boolean m_slowSave = false; // for debugging purpose only (enable/disable) slow save for users
    private int m_slowSaveTime = 0; // default time for slow save delay, must be defined in office.properties
    private long m_nActionsReceived = 0;
    private int m_nMaxTimeForSendingLastActions = DEFAULT_MAX_TIME_FOR_SENDINGLASTACTIONS;
    private final AtomicBoolean m_bBackupFileWritten = new AtomicBoolean();
    private final UserDataManager userDataManager = new UserDataManager();
    private FlushDocumentRenameRunnable asyncFlushDocRunnable = null;

    public DocumentConnection(ServiceLookup serviceLookup, ID id, ActionHandler handler, String componentID) {
        super(serviceLookup, id, handler, componentID);

        LOG.trace("RT connection: DocumentConnection created: " + id.toString());

        // initialize ServiceLookup and ResourceManager
        m_resourceManager = new ResourceManager(serviceLookup);
        // No need to support a user-based max time for sending last actions.
        m_nMaxTimeForSendingLastActions = getConfigurationService().getIntProperty(
            "io.ox/office//module/switchingEditRightsTimeout",
            (DEFAULT_MAX_TIME_FOR_SENDINGLASTACTIONS / 1000));
        // set time in milli seconds
        m_nMaxTimeForSendingLastActions *= 1000;
        m_debugOperations = getConfigurationService().getBoolProperty("io.ox/office//module/debugoperations", false);
        m_debugOperations |= getConfigurationService().getBoolProperty("io.ox/office//module/debugavailable", false);
        m_slowSave = getConfigurationService().getBoolProperty("io.ox/office//module/debugslowsave", false) & m_debugOperations;
        m_slowSaveTime = getConfigurationService().getIntProperty("io.ox/office//module/debugslowsavetime", 0) * 1000; // time in milliseconds
        m_resourceID = IDUtils.getResourceId(id);

        setCreateVersion(true);
    }

    /**
     * Handles the "firstjoined" notification provided by the Realtime-Framework. This method is called
     * when the first client joins/creates the document.
     *
     * @param id The ID of the first client that joins/creates to the document (chat-room). (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#firstJoined(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void firstJoined(ID id) {
        super.firstJoined(id);

        LOG.debug("RT connection: First client joined RT connection: " + (isValidID(id) ? id.toString() : "n/a"));
        LOG.debug("RT connection: Thread Id = " + String.valueOf(getCreateThreadId()));

        setCreateVersion(true);
        m_resourceManager.lockResources(true);

        // Check the global (hazelcast backed) save state of this document
        if (isGlobalSaveStateSet()) {
            setGlobalSaveState();
        }
    }

    /**
     * Handles the "join" notification provided by the Realtime-Framework. This method is called when a client joins the document.
     *
     * @param id The ID of the client that want to join the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#onJoin(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onJoin(ID id) {
        super.onJoin(id);

        int curClientCount = 0;

        LOG.debug("RT connection: Client joined: " + (isValidID(id) ? id.toString() : ""));

        checkForDisposed();
        checkForThreadId("onJoin");

        ServerSession session = null;
        try {
            session = id.toSession();
        }
        catch (OXException e) {
            LOG.error("Cannot retrieve a valid session from the client - exception catched", e);
            throw new RuntimeException("Cannot add user without valid session");
        }

        synchronized (m_connectionStatus) {
            try {
                final int userId = SessionUtils.getUserId(session);
                final String displayName = UserHelper.getFullName(session.getUser());
                final boolean guest = session.getUser().isGuest();
                m_connectionStatus.addActiveUser(id.toString(), displayName, userId, guest, new JSONObject());
            } catch (Exception e) {
                LOG.error("RT connection: onJoin catches exception while retrieving session from joining user ID. ", e);
            }
            curClientCount = m_connectionStatus.getActiveClients();
            if (LOG.isDebugEnabled()) {
                LOG.debug("RT connection: Number of current clients = " + String.valueOf(curClientCount));
                DebugHelper.debugLogActiveClients(LOG, m_connectionStatus);
            }
        }
    }

    /**
     * Handles the "join" notification from the Realtime-Framework. This method is called when a client joins and provides additional
     * information via the stanza.
     *
     * @param id The ID of the client that want to join the document (chat-room).
     * @param stanza
     */
    @Override
    protected void onJoin(ID id, Stanza stanza) {
        super.onJoin(id, stanza);

        checkForDisposed();
        checkForThreadId("onJoin");

        final Session session = MessageHelper.getServerSession(null, id);
        final JSONObject connectData = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "connectData"), true);
        UserData userData = null;
        
        if ((null != session) && (null != connectData)) {
            final ConnectDataHelper connectDataHelper = new ConnectDataHelper(connectData);

            if ((null == m_fileHelper) && (connectDataHelper.hasValidResource())) {
                // retrieve complete resource data from the connect data object and init helper objects
                m_fileHelper = new DocFileHelper(getServices());
                m_syncStatus = new Sync(LOG, connectDataHelper.getFolderId(), connectDataHelper.getFileId());
                m_storageHelper = new StorageHelper(getServices(), session, connectDataHelper.getFolderId());
            }

            // store application type to be used later
            final ApplicationType app = ApplicationType.stringToEnum(connectData.optString("app", null));
            if (ApplicationType.APP_NONE == m_appType) {
                m_appType = app;
            }
            // store if the document was newly created or not
            m_newDocLoaded = connectData.optBoolean("newDocument");

            // store association between rt-user-id & userData to have access to
            // user-specific data
            userData = new UserData(connectDataHelper.getFolderId(), connectDataHelper.getFileId(), connectDataHelper.getOldRTId(), connectDataHelper.getSessionId());
            setUserData(id, userData);
        }

        if (null == m_fileHelper) {
            LOG.error("RT connection: Invalid resource, connection NOT established for resource-id: " + id.toString());
            throw new RuntimeException("Invalid resource detected, not able to initialize Connection instance!");
        }

        LOG.debug("RT connection: onJoin called for document: " + DebugHelper.getDocumentFolderAndFileId(userData.getFolderId(), userData.getFileId()) + ", identity: " + getIdentity());
    }

    /**
     * Sets the user data of the joined user.
     *
     * @param id the id of the client joining the connection instance.
     * @param userData the user data which should be associated with the user.
     */
    private void setUserData(final ID id, final UserData userData) {
        userDataManager.addUser(id, userData);
    }

    /**
     * Must be called whenever the user leaves the connection instance.
     *
     * @param id the id of the client leaving the connection instance.
     */
    protected void removeUserData(final ID id) {
        userDataManager.removeUser(id);
    }

    /**
     * Provides the join data associated with the provided id.
     *
     * @param id the ID for which we want to get the join data.
     * @return the join data as JSONObject or null, if no association is present.
     */
    protected final UserData getUserData(final ID id) {
        return userDataManager.getUserData(id);
    }

    /**
     * Checks that a session id is accessible via session service. This can
     * detect configuration problems regarding session storage and hazelcast.
     *
     * @param sessionId a session id as string
     * @return TRUE, if we can receive the session instance, otherwise FALSE.
     */
    protected boolean isSessionIdAccessbile(final String sessionId) {
        boolean ok = false;

        try {
            final Session session = ServerServiceRegistry.getInstance().getService(SessiondService.class, true).getSession(sessionId);
            ok = (null != session);
        } catch (Exception e) {
            // ok is by default false
        }

        return ok;
    }

    /**
     * Handles the leaving of a client from this document group. This notification is called before
     * the ID is removed from the group and therefore using methods like isValid() is possible.
     *
     * @param id The ID of the client that is about to leave the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#beforeLeave(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void beforeLeave(ID id) {
        super.beforeLeave(id);

        // Update statistics with CLOSE or TIMEOUT type
        if ((null != m_fileHelper) && isValidID(id)) {
            final String uid = id.toString();
            boolean timeout = false;
            synchronized (m_connectionStatus) {
                timeout = (m_connectionStatus.getDurationOfInactivity(uid) >= ConnectionStatus.INACTIVITY_TRESHOLD_TIME);
            }

            if (timeout) {
                DocumentEventHelper.addTimeoutEvent(m_fileHelper);
            } else {
                DocumentEventHelper.addCloseEvent(m_fileHelper);
            }
        }
    }

   /**
     * Handles the "leave" notification provided by the Realtime-Framework. This method is called when a client leaves the document group.
     * ATTENTION: If getSignOffMessage is marked with the annotation "Asynchronous" onLeave is also called from the same background thread.
     * Therefore this method must be thread-safe, too. The notification is sent while the ID itself is not member of the group anymore. So
     * calling isValid() is not possible and will always return FALSE.
     *
     * @param id The ID of the last client that leaves the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#onLeave(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onLeave(ID id) {
        super.onLeave(id);

        final UserData userData = this.getUserData(id);
        if (null != userData) {
            LOG.debug("RT connection: onLeave called for document: " + DebugHelper.getDocumentFolderAndFileId(userData.getFolderId(), userData.getFileId()) + ", identity: " + getIdentity());
            LOG.debug("RT connection: Client " + id.toString() + " leaving => Flushing operations");
        }

        checkForDisposed();

        // handles the leaving of a client from the document connection
        impl_handleClientLeaves(id);
    }

    /**
     * Handles the "leave" notification provided by the Realtime-Framework. This method is called when a client leaves the document group.
     * ATTENTION: If getSignOffMessage is marked with the annotation "Asynchronous" onLeave is also called from the same background thread.
     * Therefore this method must be thread-safe, too. ATTENTION: The onLeave(ID id) method is also called by the RT-framework therefore one
     * has to make sure that nothing is done twice. The notification is sent while the ID itself is not member of the group anymore. So
     * calling isValid() is not possible and will always return FALSE.
     *
     * @param id The ID of the last client that wants to leave the document (chat-room).
     * @param stanza Additional data sent together with the leave request from the client.
     * @see com.openexchange.realtime.group.GroupDispatcher#onLeave(com.openexchange.realtime.packet.ID)
     */
    @Override
    protected void onLeave(ID id, Stanza stanza) {
        super.onLeave(id, stanza);

        checkForDisposed();

        final Session session = MessageHelper.getServerSession(null, id);
        final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "recentfile"), true);
        if ((null != session) && (null != jsonRequest)) {
            final ApplicationType app = ApplicationType.stringToEnum(jsonRequest.optString("app", null));

            JSONObject recentFile = jsonRequest.optJSONObject("file");
            if ((null != recentFile) && (ApplicationType.APP_NONE != app)) {
                DocumentMetaData currentMetaData = null;
                synchronized (getSyncAccess()) {
                    currentMetaData = m_lastKnownMetaData;
                }
                recentFile = FileDescriptor.createJSONObject(currentMetaData, new java.util.Date());

                RecentFileListManager recentFileListManager = new RecentFileListManager(getServices(), session);
                ArrayList<JSONObject> recentFileList = ConnectionHelper.getRecentFileList(recentFileListManager, app);
                if (null != recentFileList) {
                    recentFileListManager.addCurrentFile(recentFileList, recentFile, this.getLastSaveTime());
                    recentFileListManager.writeRecentFileList(app, recentFileList);
                    recentFileListManager.flush();
                }
                logPerformanceData(session, jsonRequest);
            }
        }
    }

    /**
     * Handles the "dispose" notification provided by the Realtime-Framework.
     * This method is called when the last client left the document and the
     * Realtime-Framework wants to dispose the GroupDispatcher instance.
     * Attention: If getSignOffMessage or onLeave is marked with the annotation
     * "Asynchronous" onDispose is also called from the same background thread.
     * Therefore this method must be thread-safe, too.
     *
     * @param id The ID of the last client that left the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#onDispose()
     */
    @Override
    protected void onDispose(ID id) throws OXException {
        // set instance to disposed and reset memory sensitive members
        setDisposed();

        final ResourceManager resourceManager = getResourceManager();
        if (null != resourceManager) {
            resourceManager.lockResources(false);
        }

        synchronized (getSyncAccess()) {
            m_messageChunkList = null;
            m_resourceManager = null;
        }
    }

    /**
     * Reset connection status values, cancels timers on disposing
     * the connection instance.
     */
    protected void impl_resetConnectionStatusOnDispose() {
        final ConnectionStatus connectionStatus = getConnectionStatus();
        synchronized (connectionStatus) {
            cancelPrepareLosingEditRightsTimer();
            connectionStatus.setWantsEditRightsUser(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);
        }
    }

    /**
     * Determines which user ID should be used for the next flushDocument call.
     *
     * @return The ID of the creator for the next flushDocument or null, if the
     *  current
     */
    private ID impl_getCreatorForNextFlush() {
        ID creator = null;
        String currentEditUserId = ConnectionStatus.EMPTY_USERID;

        synchronized (m_connectionStatus) {
            currentEditUserId = m_connectionStatus.getCurrentEditingUserId();
        }

        if (StringUtils.isNotEmpty(currentEditUserId)) {
            creator = new ID(currentEditUserId);
        }

        return creator;
    }

    /**
     * Handles the leaving of a client from the document connection.
     *
     * @param id The user id of the client leaving.
     */
    private void impl_handleClientLeaves(ID id) {
        int curClientCount = 0;
        ConnectionStatus connectionStatus = null;
        String wantsEditRightsUser = ConnectionStatus.EMPTY_USERID;
        String currentEditUser = ConnectionStatus.EMPTY_USERID;

        if (this.isDisposed()) {
            // Check for disposed connection and bail out early - no need to do
            // anything more.
            return;
        }

        synchronized (m_connectionStatus) {
            curClientCount = m_connectionStatus.getActiveClients();
            if (curClientCount > 0) {
                try {
                    boolean removeUserStatus = m_connectionStatus.removeActiveUser(id.toString());
                    if (LOG.isDebugEnabled() && removeUserStatus) {
                        LOG.debug("RT connection: onLeave removed this active user:" + id.toString());
                    }
                } catch (JSONException e) {
                    LOG.error("RT connection: onLeave catches JSON exception while removing active user", e);
                }

                curClientCount = m_connectionStatus.getActiveClients();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("RT connection: Number of current clients = " + String.valueOf(curClientCount));
                    DebugHelper.debugLogActiveClients(LOG, m_connectionStatus);
                }
            }

            // Check if the leaving user wants the edit rights. If yes we have
            // to remove the user to enable others to acquire them.
            wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
            currentEditUser = m_connectionStatus.getCurrentEditingUserId();
            if (StringUtils.isNotEmpty(wantsEditRightsUser)) {
                ID wantsEditRightsUserId = new ID(wantsEditRightsUser);
                if (wantsEditRightsUserId.equals(id)) {
                    // Reset wantsEditRightsUser from connectionStatus
                    m_connectionStatus.setWantsEditRightsUser(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);
                    LOG.debug("RT connection: Wants to edit client: " + wantsEditRightsUser + " leaving, reset wantsEditRightsUser");
                }
            }

            // Handle if the leaving user has edit rights
            if (StringUtils.isNotEmpty(currentEditUser)) {
                ID currentEditUserId = new ID(currentEditUser);
                if (currentEditUserId.equals(id)) {
                    // Reset current editing user from connectionStatus
                    m_connectionStatus.setCurrentEditingUser(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);

                    LOG.debug("RT connection: Editor " + currentEditUser + " leaving, reset editor");
                    if (StringUtils.isNotEmpty(wantsEditRightsUser)) {
                        LOG.debug("RT connection: Editor " + currentEditUser + " left, set wantsEditRightUser " + wantsEditRightsUser + " as new editor");
                        // Set wantsEditRightsUser as new editor if known
                        m_connectionStatus.setCurrentEditingUser(wantsEditRightsUser, m_connectionStatus.getWantsEditRightsUserName());
                        // Reset wantsEditRightsUser so others can acquire the edit rights
                        m_connectionStatus.setWantsEditRightsUser(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);
                        // cancel timer as we already set the new editor
                        this.cancelPrepareLosingEditRightsTimer();
                    }
                }
            }
            connectionStatus = m_connectionStatus.clone();
        }

        // remove client from the user data map
        this.removeUserData(id);

        if (curClientCount > 0) {
            updateClientsExceptSender(new MessageData(null, connectionStatus), id);
        }
    }

    /**
     * Determines if a user has write access to a file checking the locking state
     * and the folder & file permissions.
     *
     * @param session the session of the client which should be checked for access rights
     * @param folderId the folder id which should be checked for access rights
     * @param fileId the file id which should be checked for access rights
     * @param userId the user id of the user that should determine his/her access rights
     * @return TRUE, if the user has access, otherwise FALSE.
     */
    private boolean impl_canUserAccessFile(final Session session, final String folderId, final String fileId, final int userId) {
        return FileHelper.canWriteToFile(getServices(), session, userId, fileId);
    }

    /**
     * Determines who has/receives the edit rights, when a new client joins. In
     * the normal case the first client receives the edit rights, but there are
     * many special cases, where the edit rights depend on folder permissions,
     * file locks and/or other properties.
     * <b>ATTENTION</b>:
     * Must be called after the document meta data have been set by calling setLastKnownMetaData().
     *
     * @param newId The real-time ID of the joining client.
     */
    protected void impl_handleEditRights(final ID newId, Boolean writeAccess) {
        boolean joiningClientReceivesEditRights = false;
        boolean forceSwitch = false;
        int curClientCount = 0;
        int newUserId = -1;
        UserData userData = null;
        Session session = null;


        try {
            session = newId.toSession();
        } catch (Exception e) {
            LOG.warn("Exception while trying to retrieve session from user", e);
        }

        try {
            userData = this.getUserData(newId);
        } catch (Exception e) {
            LOG.warn("Exception while trying to retrieve join data", e);
        }

        try {
            newUserId = IDUtils.getUserIdFromRealTimeId(newId);
        } catch (Exception e) {
            LOG.error("Exception while trying to determine user id from rt-id", e);
        }

        synchronized (m_connectionStatus) {
            final String wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
            final String currentEditUser = m_connectionStatus.getCurrentEditingUserId();

            String oldUserId = StringUtils.EMPTY;
            String oldUserUuid = userData.getOldRTId();
            curClientCount = m_connectionStatus.getActiveClients();

            if (StringUtils.isNotEmpty(oldUserUuid)) {
                try {
                    oldUserId = this.getRealtimeID(oldUserUuid, session).toString();
                } catch (Exception e) { }
            }

            if ((1 == curClientCount) && isValidID(newId)) {
                // first client always receives the edit rights
                joiningClientReceivesEditRights = impl_determineWriteAccess(writeAccess, session, userData, newUserId);
            } else if (ConnectionStatus.isEmptyUser(currentEditUser) && ConnectionStatus.isEmptyUser(wantsEditRightsUser)) {
                // Fix for 30583 & better edit rights handling
                // Currently no one has the edit rights and no one wants to
                // receive the edit rights => next joining client receives the edit rights
                joiningClientReceivesEditRights = impl_determineWriteAccess(writeAccess, session, userData, newUserId);
            } else {
                // Special case using inactivity heuristic to determine, if we can provide
                // edit rights to the new joining user.
                try {
                    if (StringUtils.isNotEmpty(currentEditUser)) {
                        int userId1 = m_connectionStatus.getUserId(currentEditUser);
                        int userId2 = IDUtils.getUserIdFromRealTimeId(newId);

                        if ((userId1 == userId2) && (currentEditUser.equalsIgnoreCase(oldUserId))) {
                            // set new joining client as new editor without sending
                            // preparelosing etc. messages
                            joiningClientReceivesEditRights = impl_determineWriteAccess(writeAccess, session, userData, newUserId);
                            forceSwitch = true;
                        }
                    }
                } catch (Exception e) {
                    // nothing to do -> this is just a heuristic and can be ignored
                }
            }
        }

        if (joiningClientReceivesEditRights) {
            try {
                // No need that acquireEdit sends any update message
                // therefore set silent to true. As stated above
                // the current state is transferred via getWelcomeMessage.
                acquireEdit(null, newId, true, forceSwitch);
            } catch (OXException e) {
                LOG.error("RT connection: onJoin calling acquireEdit failed with exception", e);
            }
        }

        if (curClientCount > 1) {
            // send update to other clients to update their active clients count etc.
            updateClients(new MessageData(null, getConnectionStatusClone(), null), newId);
        }
    }

    /**
     * Determine the write access state using a pre-defined value or by checking
     * the folder & file access rights including locking state.
     *
     * @param writeAccess
     * @param session
     * @param userData
     * @param userId
     * @return
     */
    private boolean impl_determineWriteAccess(final Boolean writeAccess, final Session session, final UserData userData, final int userId) {
        if (null != writeAccess) {
            return writeAccess;
        } else {
            return impl_canUserAccessFile(session, userData.getFolderId(), userData.getFileId(), userId);
        }
    }

    /**
     * Finalize load of the document for both use cases synchronous/asynchronous.
     *
     * @param onBehalfOf The ID of the client which initiated the join to the document group.
     * @param operations Operations of the document, must be null if the client receives special UI operations (in case of Spreadsheet).
     * @param documentOSN The document OSN retrieved directly from the document stream.
     * @param asyncLoad Specifies if the document was loaded by the asynchronous load request queue or not.
     * @param genericHtmlDocument Specifies an optional string containing the document as a generic HTML DOM.
     * @return The final welcome message to be sent to the client (first&last for synchronous loading and the second&last one for
     *         asynchronous loading).
     */
    protected Message impl_createWelcomeMessage(ID onBehalfOf, JSONObject operations, int documentOSN, boolean asyncLoad, String genericHtmlDocument, int ignoreOpsCount, boolean ignoreMessageChunkList, final DocumentMetaData docMetaData) {
        Message welcomeMessage = null;
        int messageChunkListSize = 0;
        JSONObject lastOperation = null;
        JSONObject firstOperation = null;
        boolean opsFromDocument = true;
        boolean openError = false;

        if (isValidID(onBehalfOf)) {
            // Determine the operation state number and append additional operations
            // applied to the document since the last save action.
            synchronized (getSyncAccess()) {
                messageChunkListSize = getMessageQueueCount();

                JSONObject opsObject = operations;
                // set sync object with the current document OSN
                m_syncStatus.updateDocmentOSN(documentOSN);
                m_syncStatus.updateFileVersion((null != docMetaData) ? docMetaData.getVersion() : null);

                if (messageChunkListSize == 0) {
                    // If we don't have additional operations we have to send
                    // the current operation state number for the document. Therefore
                    // calculate or provide the current operation state number.
                    // Otherwise we don't provide the number from here but let
                    // the additional operations send it.
                    try {
                        int operationCount = opsObject.getJSONArray("operations").length();
                        lastOperation = opsObject.getJSONArray("operations").getJSONObject(operationCount - 1);
                        if (!lastOperation.has("osn")) {

                            synchronized (m_connectionStatus) {
                                if (m_connectionStatus.getOperationStateNumber() > 0) {
                                    lastOperation.put("osn", m_connectionStatus.getOperationStateNumber() - 1);
                                } else {
                                    // determine initial osn. we prefer the document OSN but if not available
                                    // fallback to the operation count
                                    int initialOSN = (documentOSN != -1) ? documentOSN : operationCount;
                                    lastOperation.put("osn", initialOSN - 1);
                                    // Set initial operation state number
                                    m_connectionStatus.setOperationStateNumber(initialOSN);
                                }
                                lastOperation.put("opl", 1);
                            }
                        }

                    } catch (JSONException e) {
                        LOG.error("RT connection: Error setting operation state number to last operation", e);
                        return impl_createErrorMessage4WelcomeMessage(
                            onBehalfOf,
                            ErrorCode.LOADDOCUMENT_FAILED_ERROR,
                            getConnectionStatusClone());
                    }
                } else if (messageChunkListSize > 0) {
                    opsFromDocument = false;
                    // in some cases we have to ignore the messages in the chunk list
                    if (!ignoreMessageChunkList) {
                        // Adding operations applied to the document since the last save action.
                        // These operations will provide the current operation state number, too.
                        OperationHelper.appendOperationChunks(operations, m_messageChunkList, false);
                    }

                    if (LOG.isDebugEnabled()) {
                        // Additional analyzing code to find root cause of missing operations
                        firstOperation = OperationHelper.debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                        lastOperation = OperationHelper.debugGetLastOperationFromMessageChunkList(m_messageChunkList);
                    }
                }
            }

            LOG.debug("RT connection: Connection.getWelcomeMessage, message chunk list size = " + Integer.toString(messageChunkListSize));
            if (null != firstOperation) {
                LOG.debug("RT connection: Connection.getWelcomeMessage, first Operation in messsage chunk list: " + OperationHelper.operationToString(firstOperation));
            }
            if (null != lastOperation) {
                if (opsFromDocument) {
                    LOG.debug("RT connection: Connection.getWelcomeMessage, last Operation in document: " + OperationHelper.operationToString(lastOperation));
                } else {
                    LOG.debug("RT connection: Connection.getWelcomeMessage, last Operation in messsage chunk list: " + OperationHelper.operationToString(lastOperation));
                }
            }

            // Check m_fileHelper which can be zero if ctor was called with
            // a invalid ID.
            if ((null != m_fileHelper) && (null != docMetaData)) {
                synchronized (m_connectionStatus) {
                    // copy data from meta data to the connection status
                    m_connectionStatus.setLockState(docMetaData.getLockedUntil() != null);
                    m_connectionStatus.setLockedByUser(docMetaData.getLockedByUser());
                    m_connectionStatus.setLockedByUserId(docMetaData.getLockedByUserId());
                }
            }

            // add additional client startup data
            final ExtraData extraData = new ExtraData(LOG, "impl_createWelcomeMessage");
            synchronized (getSyncAccess()) {
                extraData.setProperty(MessagePropertyKey.KEY_CLIENT_ID, onBehalfOf.toString());
                extraData.setProperty(Sync.SYNC_INFO, m_syncStatus.toJSON());
                extraData.setProperty(MessagePropertyKey.KEY_HTMLDOCUMENT, genericHtmlDocument);
            }

            // putting all operations into an 'actions' object
            final JSONObject actionData = new JSONObject();
            try {
                synchronized (getSyncAccess()) {
                    if (ignoreMessageChunkList) {
                        final JSONArray actions = new JSONArray();
                        if (ignoreOpsCount > 0) {
                            // If we have loaded with preview the preview operations
                            // should not be provided twice.
                            final JSONObject ops = new JSONObject();
                            JSONArray opsArray = operations.getJSONArray("operations");
                            final int operationsCount = opsArray.length();

                            // cut out the old previewoperations to be not sent twice
                            opsArray = JSONHelper.subArray(opsArray, ignoreOpsCount);
                            if (operationsCount == ignoreOpsCount) {
                                // add a synchronization no-op into the empty operations list
                                opsArray.put(OperationHelper.createSyncNoOperation(m_connectionStatus.getOperationStateNumber()));
                            }
                            // put the adapted operations object into the actions array
                            ops.put("operations", opsArray);
                            actions.put(ops);
                        } else {
                            // Other clients need the whole UI operation list
                            actions.put(operations);
                        }
                        actionData.put(MessagePropertyKey.KEY_ACTIONS, actions);
                    } else if (!operations.isEmpty()) {
                        actionData.put(MessagePropertyKey.KEY_ACTIONS, (new JSONArray()).put(operations));
                    } else {
                        extraData.setProperty(MessagePropertyKey.KEY_HAS_ERRORS, true);
                        openError = true;
                    }
                }
            } catch (JSONException e) {
                LOG.error("RT connection: Error adding operations to JSON", e);
                return impl_createErrorMessage4WelcomeMessage(onBehalfOf, ErrorCode.LOADDOCUMENT_FAILED_ERROR, getConnectionStatusClone());
            }

            final ConnectionStatus connectionStatus = getConnectionStatusClone();
            String answerString = "getactions";

            if (asyncLoad) {
                // message will be the final load update
                answerString = "update";
                connectionStatus.setFinalLoad(true);
            } else {
                // message will be the one and only welcome message including
                // document operations
                connectionStatus.setSyncLoad(true);
            }

            ErrorCode errorCode = ErrorCode.NO_ERROR;
            try {
                final UserData userData = this.getUserData(onBehalfOf);
                if (this.isValidMemberAndEditor(onBehalfOf) && !(this.m_bBackupFileWritten.get()) &&
                    !(this.canCreateOrWriteBackupFile(onBehalfOf.toSession(), userData.getFolderId()))) {
                    errorCode = ErrorCode.BACKUPFILE_WONT_WRITE_WARNING;
                }
            } catch (Exception e) {
                LOG.error("RT connection: Exception catched while checking permissions for writing backup file", e);
            }

            // set write protected state for the client now - cannot be sent via normal updates
            connectionStatus.setWriteProtected(docMetaData.getWriteProtectedState());
            // reset write protected state for the meta data again - this is not a persistent state
            // but depends on the client context
            docMetaData.resetWriteProtectedState();

            // build up the welcome message
            final MessageData messageData = new MessageData(
                new MessageChunk(actionData, getId()), connectionStatus, MessageHelper.finalizeJSONResult(errorCode, null, extraData.getJSON()));

            // set operations chunk of collected JSON objects as payload
            welcomeMessage = new Message();
            welcomeMessage.setFrom(getId());
            welcomeMessage.setSelector(getStamp(onBehalfOf));
            welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), answerString)).build()));

            LOG.debug("RT connection: Sending [welcome message] to: " + onBehalfOf.toString());
        }

        // update statistics: if no (error) documentEvent has been created by now, a new
        // document has been successfully opened
        if (openError) {
            DocumentEventHelper.addOpenErrorEvent(m_fileHelper);
        } else {
            DocumentEventHelper.addOpenEvent(m_fileHelper, docMetaData);
        }

        return welcomeMessage;
    }

    /**
     * Handles an error code and updates the recent file according to the provided error code.
     *
     * @param onBehalfOf    The ID of the client requesting the action.
     * @param serverSession The session of the client requesting the action.
     * @param errorCode     The error code describing the problem of a recent action.
     * @param folderId      The folder id of the document file.
     * @param fileId        The file id of the document file.
     * @return The message to be sent to the client side. A recent list has been updated regarding the error code provided. If the error
     *         code specifies that the file is not found, the recent list will not list the file anymore.
     */
    protected Message impl_handleError4Welcome(ID onBehalfOf, Session serverSession, ErrorCode errorCode, String folderId, String fileId) {
        // It's clear that we encountered on an error. Check for some
        // specific errors which need special handling.
        if (errorCode.getCode() == ErrorCode.GENERAL_FILE_NOT_FOUND_ERROR.getCode()) {
            // file was not found => the recent file list must be updated
            RecentFileListManager recentFileListManager = new RecentFileListManager(getServices(), serverSession);
            ConnectionHelper.removeFromRecentFile(recentFileListManager, m_appType, folderId, fileId);
        }

        final ConnectionStatus errConnectionStatus = getConnectionStatusClone();
        errConnectionStatus.setSyncLoad(true);
        return impl_createErrorMessage4WelcomeMessage(onBehalfOf, errorCode, errConnectionStatus);
    }

    /**
     * Provides a message which can be used within the getWelcomeMessage() method to provide error states.
     *
     * @param onBehalfOf The ID of the client that triggered the welcomeMessage means that just have joined the Connection. Must be set
     *                   otherwise the message won't be sent by the Realtime Framework.
     * @param errorCode The initialized ErrorCode to be sent as the result of the getWelcomeMessage() method. Note : This generates an error
     *                   message usable as an answer for getWelcomeMessage() only. If you want to make it more generic you have to make sure the
     *                   payload has to be more flexible extended ...
     */
    protected Message impl_createErrorMessage4WelcomeMessage(ID onBehalfOf, ErrorCode errorCode, ConnectionStatus connectionStatus) {
        final ExtraData extraData = new ExtraData(LOG, "impl_createErrorMessage4WelcomeMessage");
        extraData.setProperty(MessagePropertyKey.KEY_CLIENT_ID, onBehalfOf.toString());

        // create error welcome message
        final MessageData messageData = new MessageData(null, connectionStatus, MessageHelper.finalizeJSONResult(errorCode, null, extraData.getJSON()));
        final Message welcomeMessage = new Message();
        welcomeMessage.setFrom(getId());
        welcomeMessage.setTo(onBehalfOf);
        welcomeMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "getactions")).build()));

        return welcomeMessage;
    }

    /**
     * Resets the edit client to ensure that the client is not able to make any changes on the document anymore. This method is normally
     * used if a "hangup" message is sent to the specific client because it triggered an invalid state (e.g. sent incorrect operations,
     * failed to sent all operations in time while switching edit rights)
     *
     * @param fromId The ID of the editor to be reseted. It must be the current editor ID of this Connection instance otherwise the method
     *               won't do anything.
     * @return TRUE if the editor has been reseted or FALSE if not.
     */
    protected boolean impl_resetEditor(ID fromId) {
        boolean result = false;

        if (isValidID(fromId)) {
            synchronized (m_connectionStatus) {
                final String currentEditUser = m_connectionStatus.getCurrentEditingUserId();
                if (StringUtils.isNotEmpty(currentEditUser)) {
                    ID currentEditUserId = new ID(currentEditUser);
                    if (currentEditUserId.equals(fromId)) {
                        String wantsEditRightsUser = m_connectionStatus.getWantsEditRightsUserId();
                        if (StringUtils.isNotEmpty(wantsEditRightsUser)) {
                            // Set wantsToEditUser as new editor to speed up switching process in this case
                            m_connectionStatus.setCurrentEditingUser(wantsEditRightsUser.toString(), m_connectionStatus.getWantsEditRightsUserName());
                            // cancel a possible switching timeout timer
                            this.cancelPrepareLosingEditRightsTimer();
                            LOG.debug("RT connection: Editor " + currentEditUser + " removed as current editor, set wantsEditRightUser " + wantsEditRightsUser + " as new editor");
                        } else {
                            // Reset edit user
                            m_connectionStatus.setCurrentEditingUser(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);
                            LOG.debug("RT connection: Editor " + currentEditUser + " removed as current editor.");
                        }
                        result = true;
                    }
                }
            }
        }

        return result;
    }

    /**
     * Provides a stanza for the last message sent to the client leaving a document (chat-room). ATTENTION: This method is called by a
     * background thread from the Realtime-Framework. Therefore this method must be implemented thread-safe.
     *
     * @param onBehalfOf The ID of the client which wants to leave the document.
     * @return The stanza sent to the client which wants to leave the document. (non-Javadoc)
     * @see com.openexchange.realtime.group.GroupDispatcher#getSignOffMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    @Asynchronous
    public Stanza getSignOffMessage(ID onBehalfOf) {
        final ID fromId = getId();
        Message signoffMessage = null;
        final UserData signOffUserData = this.getUserData(onBehalfOf);

        LOG.debug("RT connection: [signoff message] called for document: " + ((null != signOffUserData) ? DebugHelper.getDocumentFolderAndFileId(signOffUserData.getFolderId(), signOffUserData.getFileId()) : "unknown"));
        if (isValidID(onBehalfOf)) {
            int activeClients = 0;

            LOG.debug("RT connection: [signoff message] for: " + onBehalfOf.toString());

            synchronized (m_connectionStatus) {
                activeClients = m_connectionStatus.getActiveClients();
            }

            ID creator = impl_getCreatorForNextFlush();
            creator = (null == creator) ? onBehalfOf : creator; // make sure that we use as fallback the leaving user

            final ServerSession serverSession = MessageHelper.getServerSession(null, creator);
            final UserData userData = this.getUserData(creator);

            ErrorCode errorCode = flushDocument(serverSession, userData, false, (activeClients == 1));
            impl_sendFlushInfoOnError(errorCode, onBehalfOf, true);
            if ((activeClients == 1) && errorCode.isError()) {
                // Set save on dispose to false, which prevents an unnecessary save
                // in case the last client leaves and we encountered an error.
                setSaveOnDispose(false);
            }

            final ExtraData extraData = new ExtraData(LOG, "getSignOffMessage");
            synchronized (getSyncAccess()) {
                String fileName = "unknown";
                if (null != m_lastKnownMetaData) {
                    fileName = m_lastKnownMetaData.getFileName();
                }
                m_syncStatus.setFileName(fileName);
                extraData.setProperty(Sync.SYNC_INFO, m_syncStatus.toJSON());
            }

            // send back the JSON result object returned by flushDocument
            signoffMessage = new Message();
            signoffMessage.setFrom(fromId);
            signoffMessage.setTo(onBehalfOf);
            signoffMessage.addPayload(new PayloadTree(
                PayloadTreeNode.builder().withPayload(
                    new PayloadElement(MessageHelper.finalizeJSONResult(errorCode, null, extraData.getJSON()), "json", getComponentID(), "closedocument")).build()));

            LOG.debug("RT connection: Sending [signoff message] to: " + onBehalfOf.toString());
        } else {
            LOG.debug("RT connection: Didn't send [signoff message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
        }

        return signoffMessage;
    }

    /**
     * Handles the inactivity notice of the real-time framework for a client that is a member of this connection instance. Currently the
     * handler disables the remote selection of inactive clients.
     *
     * @param stanza The Stanza containing the inactive client identified by ElementPath 'com.openexchange.realtime.client' and the Duration
     *               of inactivity identified by 'com.openexchange.realtime.client.inactivity'.
     */
    @Override
    public void handleInactivityNotice(Stanza stanza) {
        Optional<ID> inactiveClient = stanza.getSinglePayload(new ElementPath("com.openexchange.realtime", "client"), ID.class);
        Optional<Duration> inactivityDuration = stanza.getSinglePayload(new ElementPath("com.openexchange.realtime.client", "inactivity"), Duration.class);

        if (inactiveClient.isPresent() && inactivityDuration.isPresent()) {
            final String id = inactiveClient.get().toString();
            final Duration timeOfDuration = inactivityDuration.get();
            boolean enableActiveUser = false;

            LOG.debug("RT connection: User=" + inactiveClient.get() + " was inactive for " + inactivityDuration.get());
            if (timeOfDuration.equals(Duration.NONE)) {
                enableActiveUser = true; // enable active user, if we have a duration of zero
            }

            ConnectionStatus statusToSend = null;
            synchronized (m_connectionStatus) {
                boolean isUserEnabled = m_connectionStatus.isActiveUserEnabled(id);
                if (isUserEnabled != enableActiveUser) {
                    if (enableActiveUser) {
                        m_connectionStatus.enableActiveUser(id);
                    } else {
                        // Disable user to remove the remote selection and set
                        // the latest inactivity duration, too. This is used for
                        // the heuristic to give the same user the edit rights, if
                        // 1.) The same user has the edit rights
                        // 2.) All collaboration clients are from the same user
                        // 3.) The edit rights RT-user has at least an inactivity
                        // of 30 seconds.
                        m_connectionStatus.disableActiveUser(id, timeOfDuration.getValueInS());
                    }
                    LOG.debug("RT connection: User=" + id + " enabled=" + String.valueOf(enableActiveUser));
                } else if (!enableActiveUser) {
                    // update inactivity time
                    m_connectionStatus.disableActiveUser(id, timeOfDuration.getValueInS());
                }

                // The status must now be sent on any possible change
                // (update of collaboration clients).
                statusToSend = m_connectionStatus.clone();
                statusToSend.setInactivityChanged(id);
            }

            if (null != statusToSend) {
                updateClients(new MessageData(null, statusToSend, null), inactiveClient.get());
            }
        }
    }

    /**
     * Handles a sync request from a client. If the client is a valid member
     * of the document the latest connection status data is sent via update.
     *
     * Stanzas layout: AcquireEdit { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "sync" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleSync(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        checkForDisposed();
        checkForThreadId("handleSync");

        if (isValidID(fromId)) {
            final ConnectionStatus statusToSend = m_connectionStatus.clone();

            updateClient(new MessageData(null, statusToSend, null), fromId);
        }
    }

    /**
     * Stanzas layout: AcquireEdit { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "acquireedit" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleAcquireEdit(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        checkForDisposed();
        checkForThreadId("handleAcquireEdit");

        if (isValidID(fromId)) {
            final Session serverSession = MessageHelper.getServerSession(null, fromId);
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "state"), false);
            int requesterOSN = 0;

            try {
                requesterOSN = jsonRequest.getInt("osn");
            } catch (JSONException e) {
                LOG.error("RT connection: handleAcquireEdit error retrieving osn from JSON object. ", e);
            }

            if (LOG.isDebugEnabled()) {
                String editor = getConnectionStatusClone().getCurrentEditingUserId();
                LOG.debug("RT connection: Handling [acquireEdit], called from: " + fromId.toString() + " with client-osn: " + String.valueOf(requesterOSN) + " server-osn: " + String.valueOf(this.m_connectionStatus.getOperationStateNumber()));
                LOG.debug("RT connection: Editor: " + ((editor != null) ? editor : "unknown"));
            }

            acquireEdit(serverSession, fromId, false, false);
        } else {
            LOG.debug("RT connection: Didn't handle [acquireEdit] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    /**
     * Stanzas layout: CanLoseEditRights { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "canloseeditrights" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCanLoseEditRights(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        checkForDisposed();
        checkForThreadId("handleCanLoseEditRights");

        if (isValidID(fromId)) {
            final Session serverSession = MessageHelper.getServerSession(null, fromId);
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "state"), false);
            int requesterOSN = 0;

            try {
                requesterOSN = jsonRequest.getInt("osn");
            } catch (JSONException e) {
                LOG.error("RT connection: handleCanLoseEditRights cannot retrieve osn from JSON request. ", e);
            }

            LOG.debug("RT connection: Handling [canLoseEditRights], called from: " + fromId.toString() + " with client-osn: " + String.valueOf(requesterOSN) + " server-osn: " + String.valueOf(this.m_connectionStatus.getOperationStateNumber()));
            canLoseEditRights(serverSession, fromId, requesterOSN, false);

        } else {
            LOG.debug("Didn't handle [canLoseEditRights] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
        }
    }

    /**
     * Stanzas layout: AddResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "addresource" }, { namespace: getComponentID(), element: "resource", data: "{... : ...,}" }]}
     *
     * @param stanza
     * @throws OXException
     */
    public void handleAddResource(Stanza stanza) throws OXException {
        final ID fromId = retrieveRealID(stanza);

        checkForDisposed();
        checkForThreadId("handleAddResource");

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (this.isValidMemberAndEditor(fromId)) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "resource"), false);
            final ServerSession serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = addResource(serverSession, jsonRequest);

            // send back the JSON result object returned by addResource
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                        "json", getComponentID(), "resource")).build()));

            LOG.debug("RT connection: Handling [addResource], called from: " + fromId.toString());

            // send the message to the one who requested to add the resource
            send(returnMessage);
        } else {
            LOG.debug("RT connection: Didn't handle [addResource] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: FlushDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "flushdocument" }, { namespace: getComponentID(), element: "document", data:
     * "{... : ...,}" }] }
     *
     * @param stanza
     * @throws OXException
     */
    public void handleFlushDocument(Stanza stanza) throws OXException {
        final ID fromId = retrieveRealID(stanza);

        checkForDisposed();
        checkForThreadId("handleFlushDocument");

        if (isMember(fromId)) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "document"), false);
            final ID creator = this.impl_getCreatorForNextFlush();

            ErrorCode errorCode = ErrorCode.NO_ERROR;
            if (null != creator) {
                final Session serverSession = MessageHelper.getServerSession(null, creator);
                final UserData userData = this.getUserData(creator);
                errorCode = flushDocument(serverSession, userData, false, false);

                LOG.debug("RT connection: Handling [flushDocument], called from: " + fromId.toString());
            } else {
                LOG.debug("RT connection: Didn't handle [flushDocument], because editor session could not be retrieved.");
            }

            // send back the JSON result object returned by flushDocument
            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(
                PayloadTreeNode.builder().withPayload(
                    new PayloadElement(MessageHelper.finalizeJSONResult(errorCode, jsonRequest, null), "json", getComponentID(), "flushdocument")).build()));
            send(returnMessage);

            // send possible flushDocument() error code to other clients
            impl_sendFlushInfoOnError(errorCode, fromId, true);
        } else {
            LOG.debug("RT connection: Didn't handle [flushDocument] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: GetResource { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element:
     * "message", payloads: [{ element: "action", data: "getdocument" }, { namespace: getComponentID(), element: "document", data: "{... : ...,}" }]}
     *
     * @param stanza
     * @throws OXException
     */
    public void handleGetDocument(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();

        checkForDisposed();
        checkForThreadId("handleGetDocument");

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (isValidID(fromId)) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "document"), false);
            final ServerSession serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = getDocument(serverSession, jsonRequest);

            // send back the JSON result object returned by getDocument
            returnMessage.setFrom(getId());
            returnMessage.setTo(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(
                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null),
                    "json",
                    getComponentID(),
                    "document")).build()));

            LOG.debug("RT connection: Handling [getDocument], sending to: " + fromId.toString());

            // send the message to the one who requested to get the document
            send(returnMessage);
        } else {
            LOG.debug("RT connection: Didn't handle [getDocument] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: RenameDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "renamedocument" }, { namespace: getComponentID(), element: "document", data:
     * "{... : ...,}" }]}
     *
     * @param stanza
     * @throws OXException
     */
    public void handleRenameDocument(Stanza stanza) throws OXException {
        final ID fromId = this.retrieveRealID(stanza);

        checkForDisposed();
        checkForThreadId("handleRenameDocument");

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (this.isValidMemberAndEditor(fromId)) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "document"), false);
            final Session serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final RenameResult result = renameDocument(serverSession, fromId, jsonRequest);

            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(MessageHelper.finalizeJSONResult(result.jsonResult, jsonRequest, null), "json", getComponentID(), "document")).build()));

            LOG.debug("RT connection: Sending [renameDocument] result to: " + stanza.getFrom().toString());

            // always send the message to the one who requested to rename the document
            send(returnMessage);

            // if rename was successful update all clients with the result, too
            final ErrorCode errorCode = result.errorCode;
            if (!errorCode.isError()) {
                LOG.debug("RT connection: Relaying [renameDocument] result, called from: " + stanza.getFrom().toString());
                // update all Clients with the result, too
                updateClients(new MessageData(null, getConnectionStatusClone(), result.jsonResult), null);
            }
        } else {
            LOG.debug("RT connection: Didn't handle [renameDocument] since the requester was no member");
        }
    }

    /**
     * Stanzas layout: CopyDocument { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     * element: "message", payloads: [{ element: "action", data: "copydocument" }, { namespace: getComponentID(), element: "document", data:
     * "{... : ...,}" }]}
     *
     * @param stanza
     * @throws OXException
     */
    public void handleCopyDocument(Stanza stanza) throws OXException {
        final ID fromId = retrieveRealID(stanza);

        checkForDisposed();
        checkForThreadId("handleCopyDocument");

        // don't check for isMember here, since this message was sent as an
        // internal backend call
        if (this.isValidID(fromId)) {
            final Message returnMessage = new Message();
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "document"), false);
            final Session serverSession = MessageHelper.getServerSession(jsonRequest, fromId);
            final JSONObject jsonResult = copyDocument(serverSession, fromId, jsonRequest);

            returnMessage.setFrom(getId());
            returnMessage.setTo(stanza.getFrom());
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(MessageHelper.finalizeJSONResult(jsonResult, jsonRequest, null), "json", getComponentID(), "document")).build()));

            LOG.debug("RT connection: Sending [handleCopyDocument] result to: " + stanza.getFrom().toString());

            // send the return message to the one who requested to copy document
            send(returnMessage);
        } else {
            LOG.debug("RT connection: Didn't handle [handleCopyDocument] since the requester was no member");
        }
    }

    /**
     * Updates the user data of the provided client e.g. the current selection.
     *
     * @param stanza Stanzas layout: UpdateUserData { to:
     *            "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element: "message", payloads:
     *            [{ namespace: getComponentID(), element: "state", data: "{... : ...,}]" }] }
     * @throws OXException
     * @throws JSONException
     */
    public void handleUpdateUserData(Stanza stanza) {
        final ID fromId = stanza.getFrom();

        checkForDisposed();
        checkForThreadId("handleUpdateUserData");

        // don't check for isMember here, since this message was
        // sent as an internal backend call
        if (isValidID(fromId)) {
            LOG.debug("RT connection: [updateUserData] received by " + fromId.toString());
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "state"), false);
            final String userId = fromId.toString();

            // update user data
            ConnectionStatus statusToSend = null;
            synchronized (m_connectionStatus) {
                m_connectionStatus.updateUserData(userId, jsonRequest);
                statusToSend = m_connectionStatus.clone();
            }

            statusToSend.setSelectionChanged(userId);

            LOG.debug("RT connection: relaying [updateUserData]");
            updateClientsExceptSender(new MessageData(null, statusToSend, null), fromId);

        } else {
            LOG.debug("RT connection: Didn't handle [updateUserData] since the requester was no member");
        }
    }

    /**
     * Writes a log message sent from a client to the sevrer log file.
     *
     * @param stanza Stanzas layout: LogMessage { to:
     *            "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666" element: "message", payloads:
     *            [{ namespace: getComponentID(), element: "state", data: "{... : ...,}]" }] }
     */
    public void handleLogMessage(Stanza stanza) {
        final ID fromId = stanza.getFrom();

        checkForDisposed();
        checkForThreadId("handleLogMessage");

        if (isValidID(fromId)) {
            LOG.debug("RT connection: [logMessage] received from " + fromId.toString());
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), "state"), false);
            final String message = (null != jsonRequest) ? jsonRequest.optString("message") : "no message from client";

            // write client message to log file for Connection
            if (null != message) {
                LOG.error("Client message: " + message);
            }
        } else {
            LOG.debug("RT connection: Didn't handle [logMessage] since the requester was no member");
        }
    }

    /**
     * Tries to flush a modified document due to a dispose notification from the
     * Realtime framework.
     *
     * @return TRUE if the saving the document was successful otherwise FALSE.
     *         The method also provides TRUE if there was no need to save
     *         the document as there are no pending operations.
     */
    @Override
    public boolean saveDocumentOnDispose() {
        String editingUserId = null;
        boolean result = false;

        LOG.debug("RT connection: Save document on dispose initiated");
        synchronized (m_connectionStatus) {
            editingUserId = m_connectionStatus.getCurrentEditingUserId();
        }


        // Try to use the session information from the edit user.
        // If a client leaves the document it will be saved, therefore
        // there must be an editing user if we have operations in the
        // message chunk list.
        if (isModified() && !isDisposed() && StringUtils.isNotEmpty(editingUserId)) {
            ID userId = null;
            try {
                userId = new ID(editingUserId);
            } catch (Exception e) {
                LOG.error("RT connection: Save document on dispose => no user for saving found!", e);
            }

            if ((null != userId) && isSaveOnDispose()) {
                try {
                    final Session session = userId.toSession();
                    final UserData userData = this.getUserData(userId);
                    // No need to check result - we cannot reach any client here
                    flushDocument(session, userData, true, true);
                } catch (OXException e) {
                    LOG.error("RT connection: Flushing document on dispose resulted in exception!", e);
                }
            }
        } else {
            LOG.debug("RT connection: Save document on dispose - no need to flush document");
        }

        return result;
    }

    /**
     * Make a fail safe revision of the current document. This method is called by a timer
     * thread and therefore must be implemented thread-safe. Attention: There are members
     * which are set to null in onDispose() to return memory to the Java VM as soon as possible.
     * Therefore access to these member must be surrounded by a check for null!
     *
     * @return The error code describing if the fail safe save was successful or not it describes
     * what's the root cause of a possible failure.
     */
    @Override
    public ErrorCode failSafeSaveDocument(FailSafeSaveReason failSafeSaveReason) {
        final String editingUserId = m_connectionStatus.getCurrentEditingUserId();
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        // Check current editing user, can happen that we lost
        // our editing user and then we cannot process the request.
        if (ConnectionStatus.isNotEmptyUser(editingUserId)) {
            final ID userId = new ID(editingUserId);
            final UserData userData = this.getUserData(userId);
            Session session = null;

            // retrieve session information from the edit user
            // if there is no editing user we don't need to make a
            // fail safe save
            try {
                session = userId.toSession();
            } catch (OXException e) {
                LOG.error("RT connection: failSafeSaveDocument catches exception while retrieving session from user ID. ", e);
            }

            if (null != session) {
                // make sure we are the only method to trigger the saving process
                if (incAndCheckSaveInProgress() > 0) {
                    // there is a saving in progress so just leave
                    LOG.debug("RT connection: Fail safe save triggered: saving document in progress => nothing to do");
                    decSaveInProgress();
                    return errorCode;
                }

                final long startTime = System.currentTimeMillis();

                // close document in every case to write left over operations,
                // if nothing needs to be written, newFileVersion will be the empty string
                try {
                    int osn = -1;
                    int messageChunkListSize = 0;
                    JSONObject outputOperation = null;

                    ArrayList<MessageChunk> currentMessages = null;
                    ResourceManager resourceManager = null;
                    synchronized (getSyncAccess()) {
                        resourceManager = m_resourceManager;
                        currentMessages = getMessageChunkClone();
                        synchronized (m_connectionStatus) {
                            osn = m_connectionStatus.getOperationStateNumber();
                        }
                    }

                    // update global (hazelcast) based document state
                    DocumentStateHelper.updateSaveState(this.getDocumentDirectory(), this.getId().getContext(), this.getId().getResource(), true);

                    if ((null != currentMessages) && (null != resourceManager) && !currentMessages.isEmpty()) {
                        final boolean createVersion = this.getCreateVersion();
                        final String revisionless = createVersion ? "with new version" : "revisionless";

                        LOG.debug("RT connection: Fail safe save triggered: Trying to save document " + revisionless);

                        if (LOG.isDebugEnabled()) {
                            // Additional analyzing code to find root cause of missing operations
                            LOG.debug("RT connection: Fail safe save (before save): Current message chunk list size = " + currentMessages.size());
                            outputOperation = OperationHelper.debugGetLastOperationFromMessageChunkList(currentMessages);
                            if (null != outputOperation) {
                                LOG.debug("RT connection: Fail safe save (before save): Last Operation in messsage chunk list = " + OperationHelper.operationToString(outputOperation));
                                outputOperation = null;
                            }
                        }

                        final SaveDebugProperties dbgProps = (m_slowSave) ? new SaveDebugProperties() : null;
                        if (null != dbgProps) {
                            setupSaveDebugProperties(dbgProps, session);
                        }

                        final OXDocument oxDocument = new OXDocument(session, getServices(), userData, m_fileHelper, m_newDocLoaded, resourceManager, m_storageHelper, isDebugOperations(), dbgProps);

                        // check error code to detect loading problems
                        errorCode = oxDocument.getLastError();
                        if (errorCode.isError()) {
                            this.decSaveInProgress();
                            return errorCode;
                        }

                        oxDocument.setUniqueDocumentId(osn);
                        final SaveResult saveResult = oxDocument.save(ImExportHelper.getExporter(getServices(), m_fileHelper), resourceManager, currentMessages, !createVersion, false);

                        // read out return values
                        errorCode = saveResult.errorCode;

                        if ((ErrorCode.NO_ERROR == errorCode) && (null != failSafeSaveReason)) {
                            // update Statistics with saved document count based
                            // on the given reason  (either OPS100 or 15MINS type)
                            DocumentEventHelper.addSaveEvent(m_fileHelper,
                                (FailSafeSaveReason.TOO_MANY_OPS == failSafeSaveReason) ? SaveType.OPS_100 : SaveType.OPS_15MINS);
                        }

                        // write up-to-date sync information
                        synchronized (getSyncAccess()) {
                            this.m_syncStatus.updateSyncInfo((saveResult != null) ? saveResult.version : null, osn);
                        }

                        if (saveResult.backupFileWritten) {
                            // store successful write backup file
                            m_bBackupFileWritten.set(true);
                        }

                        // clean messageChunkList in case the messageChunkList has been flushed
                        if (!errorCode.isError()) {
                            if (createVersion) {
                                // after saving a new version we save revisionless
                                setCreateVersion(false);
                            }

                            int currSize = 0;
                            synchronized (getSyncAccess()) {
                                if (null != m_messageChunkList) {
                                    currSize = m_messageChunkList.size();

                                    // Remove all operations which have been saved recently
                                    // leaving just the new ones added during the save process.
                                    int index = currentMessages.size() - 1;

                                    if (index == m_messageChunkList.size() - 1) {
                                        m_messageChunkList.clear(); // optimization
                                    } else if ((index > 0) && ((index + 1) < m_messageChunkList.size())) {
                                        // New operation list must only hold the new operations
                                        // therefore we need to remove the old ones.
                                        ArrayList<MessageChunk> copy = getMessageChunkClone();
                                        List<MessageChunk> newPart = copy.subList(index + 1, m_messageChunkList.size());
                                        m_messageChunkList.clear();
                                        m_messageChunkList.addAll(newPart);
                                    }
                                    if (LOG.isDebugEnabled()) {
                                        messageChunkListSize = m_messageChunkList.size();
                                        outputOperation = OperationHelper.debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                                    }
                                }
                            }

                            setLastSaveTime();
                            LOG.debug("RT connection: Fail safe save (after save): Before adjustment, message chunk list size = " + currSize);
                            LOG.debug("RT connection: Fail safe save (after save): After adjustment, message chunk list size = " + messageChunkListSize);
                            LOG.debug("RT connection: Fail safe save (after save): First operation in messsage chunk list = " + OperationHelper.operationToString(outputOperation));
                        } else if (saveResult.errorCode.getErrorClass() == ErrorCode.ERRORCLASS_FATAL_ERROR) {
                            throw new OXDocumentException("", saveResult.errorCode);
                        }
                    } else {
                        LOG.debug("RT connection: Fail safe save: No need to save document");
                    }

                    LOG.debug("RT connection: Fail safe  save: Flushing operations to native document succeeded");
                } catch (OXDocumentException e) {
                    LOG.error("RT connection: Fail safe save failed to save document correctly: " + errorCode.getDescription(), e);
                    if (e.getErrorcode().getErrorClass() == ErrorCode.ERRORCLASS_FATAL_ERROR) {
                        // In case of a fatal error we clear the message chunk list
                        // to prevent saving a backup file again and again
                        syncClearMessageChunkList();
                    }
                } catch (Exception e) {
                    LOG.error("RT connection: Exception detected while doing a fail safe save", e);
                    // setErrorClass provides a clone with changed error class!
                    errorCode = ErrorCode.setErrorClass(ErrorCode.SAVEDOCUMENT_FAILED_ERROR, ErrorCode.ERRORCLASS_FATAL_ERROR);
                } finally {
                    this.decSaveInProgress();

                    long saveDocumentTime = (System.currentTimeMillis() - startTime);
                    LOG.trace("RT connection: Fail safe save, TIME to flushDocument: " + saveDocumentTime + "ms");

                    // reset global save state to prevent a second access via a new rt group dispatcher instance
                    DocumentStateHelper.updateSaveState(this.getDocumentDirectory(), this.getId().getContext(), this.getId().getResource(), false);

                   // in case of an error we send a flush info error to all clients
                    this.impl_sendFlushInfoOnError(errorCode, userId, false);
                }

                // sets the latest error code for this fail safe save operation
                this.setLastFailSafeSaveError(errorCode);

                // check for result and trigger update if save was successful
                // which means (warning or no error detected)
                if (errorCode.getErrorClass() <= ErrorCode.ERRORCLASS_WARNING) {
                    // Trigger update message so clients can update files view
                    m_connectionStatus.setFailSafeSaveDone(true);
                    updateClients(new MessageData(null, getConnectionStatusClone()), null);
                    m_connectionStatus.setFailSafeSaveDone(false);
                }
            }
        }

        return errorCode;
    }

    /**
     * Removes all pending operations from the queue. This is for testing
     * purpose only and should never be called in a production system.
     */
    @Override
    public void resetOperationQueue() {
        syncClearMessageChunkList();
        // set last save time as fail safe save does
        setLastSaveTime();
    }

    /**
     * Sends a decline edit rights message to the client requesting the edit rights.
     *
     * @param fromId the client that requested to receive the edit rights
     * @param errorCode the error code describing the cause of the declination
     */
    private void sendDeclineEditRights(final ID fromId, final ErrorCode errorCode) throws OXException {
        // Asynchronous rename in progress - send decline acquire edit rights to requesting client and bail out
        final Message returnMessage = new Message();
        final MessageData messageData = new MessageData(null, getConnectionStatusClone(), MessageHelper.finalizeJSONResult(ErrorCode.SWITCHEDITRIGHTS_NOT_POSSIBLE_SAVE_IN_PROGRESS_WARNING));

        returnMessage.setFrom(fromId);
        returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
            new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "declinedacquireeditrights")).build()));
        relayToID(returnMessage, fromId);

        LOG.debug("RT connection: Relaying [declinedacquireeditrights] to client requested edit rights - save in progress, applied from: " + fromId.toString());
        return;
    }

    /**
     * Implementation method to process a "acquireedit" request from a client. Sending the required "preparelosingeditrights" message to the
     * current editor and setup a timeout thread to handle non-responsive clients.
     *
     * @param session The session of the user who wants to receive edit rights.
     * @param fromId  The ID of the user who wants to receive edit rights.
     * @param silent  If set to TRUE the method won't send any update nor preparelosingeditrights notification. Used by methods which already
     *                know that switching edit rights can be done immediately.
     * @param force   If set to TRUE the method forces to set the provided client as the new editor regardless of other properties. Should be
     *                called carefully as this switches edit rights without any hint to the current editor.
     */
    private void acquireEdit(Session session, ID fromId, boolean silent, boolean forceSwitch) throws OXException {
        Session curSession = session;
        String displayUserName = ConnectionStatus.EMPTY_USERNAME;

        if ((null == curSession) && isValidID(fromId)) {
            try {
                curSession = fromId.toSession();
            } catch (OXException e) {
                LOG.error("RT connection: acquireEdit catches exception while retrieving session from sender ID. ", e);
            }
        }

        if ((null != curSession) && (curSession instanceof ServerSession)) {
            final User curUser = ((ServerSession) curSession).getUser();
            displayUserName = (null != curUser) ? curUser.getDisplayName(): ConnectionStatus.EMPTY_USERNAME;
        }

        if (isValidID(fromId)) {
            String wantsEditRightsUserId = ConnectionStatus.EMPTY_USERID;
            String hasEditRightsUserId = ConnectionStatus.EMPTY_USERID;

            synchronized (m_connectionStatus) {
                wantsEditRightsUserId = m_connectionStatus.getWantsEditRightsUserId();
                hasEditRightsUserId = m_connectionStatus.getCurrentEditingUserId();
            }

            FlushDocumentRenameRunnable currentFlushDocumentRunnable;
            synchronized(getSyncAccess()) {
                currentFlushDocumentRunnable = asyncFlushDocRunnable;
            }

            // Check, if a asynchronous rename is in progress
            if (null != currentFlushDocumentRunnable) {
                LOG.debug("RT connection: Asynchronous rename in progress - cannot change edit rights.");
                sendDeclineEditRights(fromId, ErrorCode.SWITCHEDITRIGHTS_NOT_POSSIBLE_SAVE_IN_PROGRESS_WARNING);
                return;
            }

            // Check if saving is currently in progress - we don't want to transfer
            // edit rights while saving is in progress. A save operation can need
            // a good amount of time and we cannot handle more than one save request
            // at a time. Therefore we lower the risk of complications, if we refuse
            // changing edit rights in this scenario.
            if (isSaveInProgressWithWait(WAIT_FOR_SAVEINPROGRESS_EDITRIGHTS)) {
                LOG.debug("RT connection: Save in progress - not possible to change edit rights.");
                sendDeclineEditRights(fromId, ErrorCode.SWITCHEDITRIGHTS_NOT_POSSIBLE_SAVE_IN_PROGRESS_WARNING);
                return;
            }

            // Check if user tries to acquire edit rights although she has it already.
            // Can be related to a slow connection and the user tries again and again
            // to acquire the rights.
            if (StringUtils.isNotEmpty(hasEditRightsUserId)) {
                ID currentEditorId = new ID(hasEditRightsUserId);
                if (fromId.equals(currentEditorId)) {
                    LOG.debug("RT connection: Ignore [acquireedit] from editor");
                    return;
                }
            }

            if (StringUtils.isEmpty(hasEditRightsUserId) || forceSwitch) {
                String wantsToEdit = "";
                ConnectionStatus connectionStatus = null;

                LOG.debug("RT connection: acquireedit no current editor found, editor now: " + fromId.toString() + ", wants to edit:" + wantsEditRightsUserId);

                synchronized (m_connectionStatus) {
                    // first acquire edit rights on a non-acquired document can be directly approved
                    m_connectionStatus.setCurrentEditingUser(fromId.toString(), displayUserName);
                    if (wantsEditRightsUserId.equals(fromId.toString())) {
                        // Special case: If user1 acquires edit rights and the editor leaves and afterwards
                        // the user1 again acquires edit rights, user1 will be editor and also the user who
                        // wants to get these rights. So we have to reset wantsToEditRightsUser!
                        m_connectionStatus.setWantsEditRightsUser(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);
                    }
                    // cancels a possible prepare losing edit rights timer
                    cancelPrepareLosingEditRightsTimer();
                    wantsToEdit = m_connectionStatus.getWantsEditRightsUserId();
                    connectionStatus = m_connectionStatus.clone();
                }
                if (!silent) {
                    LOG.debug("RT connection: Relaying [updateClients] to all clients, editor:" + fromId.toString() + ", wants to edit:" + wantsToEdit);
                    updateClients(new MessageData(null, connectionStatus), null);
                    flushDocumentAndSetCreateVersion(curSession, fromId, false, false);
                }
            } else if (StringUtils.isEmpty(wantsEditRightsUserId)) {
                ID currentEditorId = new ID(hasEditRightsUserId);
                Message returnMessageLose = null;
                Message returnMessageGet = null;
                MessageData messageData = null;

                synchronized (m_connectionStatus) {
                    // Start "preparelosingeditrights" timeout timer to prevent that we have a document
                    // where the edit rights are pending forever (e.g. the current editor loses connection
                    // while we are switching the edit rights - waiting for canLoseEditRights)
                    cancelPrepareLosingEditRightsTimer();
                    startPrepareLosingEditRightsTimer(session, currentEditorId);

                    // store user who wants to receive the edit rights
                    m_connectionStatus.setWantsEditRightsUser(fromId.toString(), displayUserName);

                    if (!silent) {
                        // Send the message to the client who will lose the edit rights
                        returnMessageLose = new Message();
                        returnMessageGet = new Message();
                        messageData = new MessageData(null, m_connectionStatus.clone(), null);
                    }
                }

                if (!silent) {
                    if (null == returnMessageGet) {
                        returnMessageGet = new Message();
                    }

                    // send info notification to the client which will receive the edit rights
                    returnMessageGet.setFrom(fromId);
                    returnMessageGet.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "acceptedacquireeditrights")).build()));

                    LOG.debug("RT connection: Relaying [acceptedacquireeditrights] to client requested edit rights, applied from: " + fromId.toString());

                    relayToID(returnMessageGet, fromId);

                    if (null == returnMessageLose) {
                        returnMessageLose = new Message();
                    }

                    // send message to the client which will lose the edit rights for preparation
                    returnMessageLose.setFrom(currentEditorId);
                    returnMessageLose.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "preparelosingeditrights")).build()));

                    LOG.debug("RT connection: Relaying [preparelosingeditrights] to client with edit rights, to: " + currentEditorId.toString());
                    relayToID(returnMessageLose, currentEditorId);
                }
            } else {
                ID wantsEditorRightsId = new ID(wantsEditRightsUserId);

                if (!fromId.equals(wantsEditorRightsId) && !silent) {
                    // Someone else already wants to acquire the edit rights. Therefore we
                    // have to decline this request until the edit rights changed.

                    final Message returnMessage = new Message();
                    final MessageData messageData = new MessageData(null, getConnectionStatusClone(), MessageHelper.finalizeJSONResult(ErrorCode.SWITCHEDITRIGHTS_IN_PROGRESS_WARNING));

                    returnMessage.setFrom(fromId);
                    returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                        new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "declinedacquireeditrights")).build()));

                    LOG.debug("RT connection: Relaying [declinedacquireeditrights] to client requested edit rights while there is a switch in progress, applied from: " + fromId.toString());
                    LOG.debug("RT connection: Relaying [declinedacquireeditrights], wantsEditorRights=" + wantsEditRightsUserId + ", editor=" + hasEditRightsUserId);
                    relayToID(returnMessage, fromId);
                }
            }
        }
    }

    /**
     * Implementation method to handle the response from the current editor client to give up the edit rights. This method is called by the
     * "preparelosingeditrights" timer thread and the RT core. Therefore it must be thread-safe implemented.
     *
     * @param session The session of the user who was notified via "preparelosingeditrights"
     * @param fromId The id who was notified via "preparelosingeditrights".
     * @param clientOSN The osn of the client responding to the "preparelosingeditrights" notification.
     * @param timeout Specifies that the method is called from the timeout callback.
     */
    protected void canLoseEditRights(Session session, ID fromId, int clientOSN, boolean timeout) {
        int serverOSN = clientOSN;
        ID currentEditorId = null;
        ID newEditorId = null;
        boolean editRightsChanged = false;

        synchronized (m_connectionStatus) {
            currentEditorId = new ID(this.m_connectionStatus.getCurrentEditingUserId());
            if (timeout || currentEditorId.equals(fromId)) {
                editRightsChanged = true;
                newEditorId = new ID(m_connectionStatus.getWantsEditRightsUserId());
                serverOSN = m_connectionStatus.getOperationStateNumber();

                this.cancelPrepareLosingEditRightsTimer();
            }

            if (timeout) {
                // set inactivity in case of a non-responding client losing edit rights
                m_connectionStatus.disableActiveUser(fromId.toString(), m_nMaxTimeForSendingLastActions / 1000);
            }
        }

        // prevent processing of an old canloseeditrights notification
        if (!editRightsChanged || (null == newEditorId)) {
            return;
        }

        // Check the current server osn and determine, if the client losing
        // edit rights is out of sync. Do this only, if we don't have a timeout
        // scenario.
        if (!timeout && (clientOSN > serverOSN) && (null != currentEditorId)) {
            try {
                sendHangUp(fromId, getConnectionStatusClone(), ErrorCode.HANGUP_INVALID_OSN_DETECTED_ERROR,
                    HangUpReceiver.HANGUP_ID, "Relaying [hangup] to client that has recently lost edit rights due to inconsitent osn, called from: " + fromId.toString() + " with client-osn: " + String.valueOf(clientOSN) + " server-osn: " + String.valueOf(serverOSN));
            } catch (Exception e) {
                LOG.error("RT connection: Exception while sending [hangup] to editor client due to inconsistent OSN", e);
            }
        }

        // We now define that losing the edit rights should be the time to save the
        // current document state. See US 94297034.
        startCompleteEditRightsTimer(session, newEditorId);
        flushDocumentAndSetCreateVersion(session, fromId, false, false);
        cancelCompleteEditRightsTimer();

        // finalize the switching of the edit rights to the new editor
        completeEditRightsSwitch(session, newEditorId);
    }

    /**
     * Completes the switching of the edit rights.
     *
     * @param session The session of the client that will lose the edit rights.
     * @param newEditorId The id of the client that will receive the edit rights.
     */
    private void completeEditRightsSwitch(final Session session, final ID newEditorId) {
        ConnectionStatus statusToSend = null;

        synchronized (m_connectionStatus) {
            // Switch the edit rights.
            m_connectionStatus.setCurrentEditingUser(newEditorId.toString(), m_connectionStatus.getWantsEditRightsUserName());
            // Remove the requester so others can acquire the edit rights
            m_connectionStatus.setWantsEditRightsUser(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);

            statusToSend = m_connectionStatus.clone();
            LOG.debug("RT connection: Switching edit rights to client: " + newEditorId);
        }

        if (statusToSend != null) {
            updateClientsExceptSender(new MessageData(null, statusToSend), newEditorId);

            // Checks the permissions regarding a possible backup file and sends
            // a special update-message to the new editor client
            ErrorCode errorCode = ErrorCode.NO_ERROR;
            try {
                final UserData userData = this.getUserData(newEditorId);
                if (!this.canCreateOrWriteBackupFile(newEditorId.toSession(), userData.getFolderId())) {
                    errorCode = ErrorCode.BACKUPFILE_WONT_WRITE_WARNING;
                }
            } catch (Exception e) {
                LOG.error("Exception catched while trying to determine permission for possible backup file", e);
            }
            this.updateClient(new MessageData(null, statusToSend, MessageHelper.finalizeJSONResult(errorCode)), newEditorId);
        }
    }

    /**
     * Sends a waiting message to the client receiving the edit rights
     * @param newEditorId
     */
    protected void sendWaitingMessageToNewEditor(final ID newEditorId) {
        if (null != newEditorId) {
            try {
                // Someone else already wants to acquire the edit rights. Therefore we
                // have to decline this request until the edit rights changed.
                final Message returnMessage = new Message();
                final MessageData messageData = new MessageData(null, getConnectionStatusClone(), MessageHelper.finalizeJSONResult(ErrorCode.SWITCHEDITRIGHTS_NEEDS_MORE_TIME_WARNING));

                returnMessage.setFrom(newEditorId);
                returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "infomessage")).build()));

                LOG.debug("RT connection: Relaying [infomessage] to client requested edit rights, because a flushDocument needs more time" + newEditorId.toString());
                relayToID(returnMessage, newEditorId);
            } catch (Exception e) {
                LOG.error("RT connection: Exception catched while trying to send message to future editor to wait, because of a lengthly save document", e);
            }
        }
    }

    /**
     * Adds a resource to the distributed managed file resource, so it can be used by the client via a media url.
     *
     * @param session The session of the user who wants to add a resource
     * @param jsonRequest The addResource request which includes the resourceid
     * @return A json object containing the result of the addResource request.
     */
    private JSONObject addResource(Session session, JSONObject jsonRequest) {
        final JSONObject jsonResult = new JSONObject();

        if ((null != session) && (null != jsonRequest)) {
            final String managedResourceId = jsonRequest.optString("resourceid", "");

            // we got a file id to create a distributed managed file resource from
            if (StringUtils.isNotEmpty(managedResourceId)) {
                final Resource managedResource = m_resourceManager.createManagedResource(managedResourceId);

                if (null != managedResource) {
                    m_resourceManager.addResource(managedResource);
                    LOG.debug("RT connection: addResource added resource " + managedResourceId + "to resourceManager.");

                    try {
                        jsonResult.put("resourceid", managedResource.getManagedId());
                    } catch (JSONException e) {
                        LOG.error("RT connection: addResource - could not retrieve managed resource id");
                    }
                } else {
                    LOG.error("RT connection: addResource - managed resource could not be retrieved from id: " + managedResourceId);
                }
            } else {
                // try to create the resource from the given data
                final String data64 = jsonRequest.optString("dataurl", "");
                byte[] data = null;

                if (null != data64) {
                    final String searchPattern = "base64,";
                    int pos = data64.indexOf(searchPattern);

                    if ((-1 != pos) && ((pos += searchPattern.length()) < (data64.length() - 1))) {
                        data = Base64.decode(data64.substring(pos));
                    }
                }

                if ((data != null) && (data.length > 0)) {
                    final String fileName = jsonRequest.optString("filename", "");
                    final String fileExtension = FileHelper.getExtension(fileName);

                    String newExtension = jsonRequest.optString("add_ext", "");

                    // extension to add to resource fileName
                    if (newExtension.length() < 1) {
                        newExtension = fileExtension;
                    }

                    // crc32 of resource data
                    int crc32 = jsonRequest.optInt("add_crc32", 0);

                    if (0 == crc32) {
                        crc32 = Resource.getCRC32(data);
                    }

                    // resource id
                    final long uid = m_resourceManager.addResource(crc32, data);
                    LOG.debug("RT connection: addResource added resource " + Long.toString(uid) + "to resourceManager.");

                    try {
                        String resourceName = ResourceHelper.getResourceName(m_fileHelper, uid, newExtension);
                        if (null != resourceName) {
                            jsonResult.put("added_filename", resourceName);
                        }
                    } catch (Exception e) {
                        LOG.error("RT connection: addResource catches exception adding filename to JSON result. ", e);
                    }
                }
            }
        }

        return MessageHelper.checkJSONResult(jsonResult);
    }

    /**
     * Provides the stream and mimetype of a requested document.
     *
     * @param session The session of the user who wants to add a resource
     * @param jsonRequest The request containing the folder/file id and the target format.
     * @return A json object containing the stream and mimetype.
     */
    private JSONObject getDocument(ServerSession session, JSONObject jsonRequest) {
        final JSONObject jsonResult = new JSONObject();

        if ((null != session) && (null != jsonRequest)) {
            final String documentFormat = jsonRequest.optString("documentformat", "native"); // ["native", "pdf"]
            final String returnType = jsonRequest.optString("returntype", "dataurl"); // ["dataurl", "resourceid"]
            final String source = jsonRequest.optString("source", "file"); //
            final String folder = jsonRequest.optString("folder");
            final String id = jsonRequest.optString("id");
            final String module = jsonRequest.optString("module");
            final String attachment = jsonRequest.optString("attachment");
            ArrayList<MessageChunk> currentMessages = null;
            MailServletInterface mailInterface = null;
            InputStream pdfDocumentStm = null;
            InputStream documentStm = null;

            synchronized (getSyncAccess()) {
                currentMessages = getMessageChunkClone();
            }

            try {
                ErrorCode errorCode = ErrorCode.NO_ERROR;

                if (source.equals("mail")) {
                    try {
                        final MailPart part = MailServletInterface.getInstance(session).getMessageAttachment(folder, id, attachment, false);

                        if (part != null) {
                            documentStm = part.getInputStream();
                        }
                    } catch (final Exception e) {
                        LOG.error("RT connection: getDocument catches exception trying to retrieve input stream from mail part. ", e);
                    }
                } else if (source.equals("tasks") || source.equals("calendar") || source.equals("contacts")) {
                    try {
                        documentStm = Attachments.getInstance().getAttachedFile(
                            session,
                            Integer.parseInt(folder),
                            Integer.parseInt(attachment),
                            Integer.parseInt(module),
                            Integer.parseInt(id),
                            session.getContext(),
                            session.getUser(),
                            session.getUserConfiguration());
                    } catch (final Exception e) {
                        LOG.error("RT connection: getDocument catches exception trying to retrieve input stream from task attachment. ", e);
                    }
                } else {
                    OXDocument oxDocument = new OXDocument(session, getServices(), folder, id, m_fileHelper, m_newDocLoaded, m_resourceManager, m_storageHelper, isDebugOperations(), null);
                    ResolvedStreamInfo docStreamInfo = oxDocument.getResolvedDocumentStream(ImExportHelper.getExporter(getServices(), m_fileHelper), getResourceManager(), currentMessages, null, false, false);

                    if (null != docStreamInfo) {
                        documentStm = docStreamInfo.resolvedStream;
                        errorCode = docStreamInfo.errorCode;
                    } else {
                        errorCode = ErrorCode.SAVEDOCUMENT_FAILED_ERROR;
                    }
                }

                // convert document stream content to PDF, if requested
                if (null != documentStm) {
                    String fileName = jsonRequest.optString("filename", "document");
                    String mimeType = jsonRequest.optString("mimetype", "");

                    if (documentFormat.equals("pdf") && (errorCode == ErrorCode.NO_ERROR)) {
                        final IManager dcManager = getServices().getService(IManager.class);

                        // convert content of document stream to PDF
                        if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                            final HashMap<String, Object> jobProperties = new HashMap<String, Object>(8);
                            final HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);

                            jobProperties.put(Properties.PROP_INPUT_STREAM, documentStm);
                            jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);

                            try {
                                final CapabilityService capService = getServices().getService(CapabilityService.class);
                                final CapabilitySet capSet = (null != capService) ? capService.getCapabilities(session) : null;

                                // enable sharepoint conversion, depending on the user capability set
                                if ((null != capSet) && capSet.contains("sharepointconversion")) {
                                    jobProperties.put(Properties.PROP_FEATURES_ID, Integer.valueOf(Feature.SHAREPOINT_PDFCONVERSION.id()));
                                }
                            } catch (OXException e) {
                                LOG.error("RT connection: getDocument catches exception trying to retrieve 'sharepointconversion' capability.", e);
                            }

                            pdfDocumentStm = dcManager.convert("pdf", jobProperties, resultProperties);
                            IOUtils.closeQuietly(documentStm);
                            documentStm = pdfDocumentStm;
                            pdfDocumentStm = null;

                            fileName = fileName + ".pdf";
                            mimeType = "application/pdf";
                        }
                    }

                    if (mimeType.length() < 1) {
                        mimeType = DocFileHelper.getMimeType(fileName);
                    }

                    // set resulting data at JSONObject; document
                    // stream might be null after a conversion error
                    if (null != documentStm) {
                        // add resourceid or dataurl to the result object, depending on the value of returnType
                        OperationHelper.appendData(m_resourceManager, jsonResult, documentStm, returnType, mimeType);

                        // add filename and mimetype properties to the valid result object
                        if (!jsonResult.isEmpty()) {
                            try {
                                jsonResult.put("resourcename", fileName);
                                jsonResult.put("mimetype", mimeType);
                            } catch (JSONException e) {
                                LOG.error("RT connection: getDocument catches exception setting resource name/mimetype to JSON result. ", e);
                            }
                        }

                        IOUtils.closeQuietly(documentStm);
                    }
                }
            } finally {
                IOUtils.closeQuietly(mailInterface);
                IOUtils.closeQuietly(documentStm);
                IOUtils.closeQuietly(pdfDocumentStm);
            }
        }

        return MessageHelper.checkJSONResult(jsonResult);
    }

    /**
     * Renames the name of the document. This doesn't influence the folder&file id. ATTENTION: The assumption that folder & file id are not
     * influenced are not true. At least there are storage-plugins (e.g. Dropbox) which use a uri to the file/folder. Therefore changing the
     * name invalidates the file id!!!!
     *
     * @param session     The session of the user who wants to rename a document.
     * @param jsonRequest The request containing the folder/file id and the new name.
     * @return A json object containing the result of the rename request.
     */
    private RenameResult renameDocument(final Session session, final ID id, final JSONObject jsonRequest) {
        String requestedNewFileName = "";
        RenameResult result = new RenameResult();

        try {
            if ((null != session) && (null != jsonRequest) && (null != m_fileHelper) && jsonRequest.has(MessagePropertyKey.KEY_FILE_NAME)) {
                final UserData userData = this.getUserData(id);
                final StorageHelper storageHelper = new StorageHelper(getServices(), session, userData.getFolderId());
                FlushDocumentRenameRunnable currentAsyncFlushDocRunnable;
                
                synchronized(getSyncAccess()) {
                    currentAsyncFlushDocRunnable = asyncFlushDocRunnable;
                }
                
                requestedNewFileName = jsonRequest.getString(MessagePropertyKey.KEY_FILE_NAME);
                if (storageHelper.supportsPersistentIDs()) {
                    result = m_fileHelper.renameDocument(session, requestedNewFileName, userData.getFolderId(), userData.getFileId());
                } else if (null == currentAsyncFlushDocRunnable) {
                    // We want to support rename on storage systems that don't support transparent
                    // file ids. In this cases we have to do a complex transaction process which starts
                    // with saving the document, before we rename it.
                    synchronized(getSyncAccess()) {
                        currentAsyncFlushDocRunnable = new FlushDocumentRenameRunnable(this, session, id, userData, requestedNewFileName);
                        asyncFlushDocRunnable = currentAsyncFlushDocRunnable;
                    }
                    final Thread thread = new Thread(asyncFlushDocRunnable);

                    // start the thread to flush the document
                    thread.start();

                    // Set a save in progress as result for the rename request
                    // The clients are notified by a specific message that the rename was done or
                    // or an error occurred.
                    result.errorCode = ErrorCode.RENAMEDOCUMENT_SAVE_IN_PROGRESS_WARNING;
                    result.jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, result.errorCode.getAsJSON());
                } else {
                    // rename already in progress - error message
                    result.errorCode = ErrorCode.RENAMEDOCUMENT_NOT_POSSIBLE_IN_PROGRESS_ERROR;
                    result.jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, result.errorCode.getAsJSON());
                }
            }

            if ((null != result) && (result.errorCode == ErrorCode.NO_ERROR) && (null != result.metaData)) {
                result.jsonResult.put(MessagePropertyKey.KEY_FILE_NAME, result.metaData.getFileName());
                result.jsonResult.put(MessagePropertyKey.KEY_FILE_VERSION, result.metaData.getVersion());
                result.jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, result.errorCode.getAsJSON());
                // update current document meta data for access
                synchronized (getSyncAccess()) {
                    final int lockedByUserId = m_lastKnownMetaData.getLockedByUserId();
                    final String lockedByUser = m_lastKnownMetaData.getLockedByUser();
                    m_lastKnownMetaData = new DocumentMetaData(result.metaData, lockedByUser, lockedByUserId);
                }
            } else {
                // In case of an error set at least the change result entry
                result.jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, result.errorCode.getAsJSON());
            }
        } catch (JSONException e) {
            LOG.warn("RT connection: renameDocument - failed to rename document to " + requestedNewFileName, e);

            // Fix for bug 29261: Don't return an empty jsonResult which results in a response
            // where the property "hasErrors" is set to "true". This forces the client to immediately
            // show an error bubble and stop accepting server messages.
            try {
                // provide change result to the client
                result.errorCode = ErrorCode.RENAMEDOCUMENT_FAILED_ERROR;
                result.jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.RENAMEDOCUMENT_FAILED_ERROR.getAsJSON());
            } catch (JSONException je) {
                LOG.error("RT connection: renameDocument catches exception trying to set rename result to JSON object. ", je);
            }
        }

        MessageHelper.checkJSONResult(result.jsonResult);
        return result;
    }

    /**
     * Completes an asynchronous rename. ATTENTION: This method is called by a separate thread and
     * therefore must be implemented thread-safe.
     *
     * @param session the session of the user requested the rename request
     * @param id the id of the user requested the rename request
     * @param userData the user data of the user requested the rename request
     * @param errorCode the error code of the save document request
     */
    protected void completeAsyncRename(final Session session, final ID id, final UserData userData, final String newFileName, final ErrorCode errorCode) {
        RenameResult result = new RenameResult();

        if (errorCode.isNoError()) {
            try {
                if ((null != session) && (null != userData) && (StringUtils.isNotEmpty(newFileName))) {
                    result = m_fileHelper.renameDocument(session, newFileName, userData.getFolderId(), userData.getFileId());
                } else {
                    result.errorCode = ErrorCode.RENAMEDOCUMENT_FAILED_ERROR;
                }

                final JSONObject jsonResult;
                final Message message = new Message();

                jsonResult = result.errorCode.getAsJSONResultObject();
                try {
                    jsonResult.put("fileId", result.newFileId);
                    if (null != result.metaData) {
                        jsonResult.put("fileName", result.metaData.getFileName());
                    }
                } catch (JSONException e) {
                    // nothing to do
                }

                final MessageData messageData = new MessageData(null, null, jsonResult);
                message.setFrom(id);
                message.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(messageData, MessageData.class.getName(), this.getComponentID(), "renameandreload")).build()));

                // send "renameandreload" to editor first, so it should be the editor on the renamed doc
                relayToID(message, id);

                try {
                    // delay sending the message to the other clients
                    Thread.sleep(2000);
                } catch (InterruptedException e) {
                    // do nothing
                }

                // send the message to all other clients
                relayToAllExceptSender(message);
            } catch (OXException e) {
                // we are lost here - message cannot be sent to clients
                LOG.error("RT connection: async rename of document failed due to exception", e);
            }
        } else {
            final JSONObject jsonResult;
            final Message message = new Message();
    
            jsonResult = errorCode.getAsJSONResultObject();
            final MessageData messageData = new MessageData(null, null, jsonResult);
            message.setFrom(id);
            message.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), this.getComponentID(), "renameandreload")).build()));

            // send "renameandreload" to editor, so an error message is displayed
            // not necessary to send it any other client / rename process has not been notified yet
            try {
                relayToID(message, id);
            } catch (OXException e) {
                // we are lost here - message cannot be sent to clients
                LOG.error("RT connection: cannot sent message that document save for async rename failed", e);
            }
        }

        // reset async flush document runnable
        synchronized(getSyncAccess()) {
            asyncFlushDocRunnable = null;
        }
    }

    /**
     * Saves the current document including the latest changes in the message chunk list to a specified folder. The current state of the
     * connection including the message chunk list, editor etc. are not changed.
     *
     * @param session     The session of the user that requested to copy the current document to a sepcified folder.
     * @param id          The id of the client requested to copy the current document to a specific folder.
     * @param jsonRequest The request containing the folder/file id, the target folder id, the new file name. The method uses the folder of
     *                    the document if the target folder id is missing, if this is not possible, the user folder is used. If the new file name is
     *                    missing the original file name is used.
     * @return A json object containing the result of the copy document request.
     */
    private JSONObject copyDocument(final Session session, final ID id, final JSONObject jsonRequest) {
        final JSONObject jsonResult = new JSONObject();
        final UserData userData = this.getUserData(id);
        String targetFolder = "";
        boolean asTemplate = false;

        try {
            String newFileName = null;

            if ((null != session) && (null != jsonRequest) && (null != m_fileHelper) && (null != userData)) {
                boolean bCreateDocumentCopy = true;
                targetFolder = jsonRequest.optString(MessagePropertyKey.KEY_TARGET_FOLDER);
                newFileName = jsonRequest.optString(MessagePropertyKey.KEY_FILE_NAME);
                asTemplate = jsonRequest.optBoolean(MessagePropertyKey.KEY_AS_TEMPLATE);

                if ((null == targetFolder) || (targetFolder.length() == 0)) {
                    // There is no target folder
                    boolean bCanUseFolder = false;
                    if (!asTemplate) {
                        // First we try to use the source folder and check the
                        // permissions. If we cannot write, we take the users
                        // default folder as fallback.
                        final IDBasedFolderAccessFactory folderFactory = getServices().getService(IDBasedFolderAccessFactory.class);
                        final IDBasedFolderAccess folderAccess = ((null != folderFactory) && (null != session)) ? folderFactory.createAccess(session) : null;

                        targetFolder = userData.getFolderId();
                        int[] permissions = FolderHelper.getFolderPermissions(folderAccess, targetFolder);
                        if (null != permissions) {
                            // Check folder permissions that we can create a copy
                            // of the document file and write to it.
                            bCanUseFolder = (permissions[FolderHelper.FOLDER_PERMISSIONS_FOLDER] >= FileStoragePermission.CREATE_OBJECTS_IN_FOLDER) && (permissions[FolderHelper.FOLDER_PERMISSIONS_WRITE] > FileStoragePermission.NO_PERMISSIONS);
                        }
                    }

                    // Fallback use the users documents folder to create the document
                    // For template this is the default case!
                    if (!bCanUseFolder) {
                        targetFolder = DocFileHelper.getUserDocumentsFolderId(session, getServices());
                        if (null == targetFolder) {
                            jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR.getAsJSON());
                            bCreateDocumentCopy = false;
                        }
                    }
                }

                if (StringUtils.isEmpty(newFileName)) {
                    // We don't have a new file name for the copy of the
                    // document. Therefore we retrieve the original name.
                    try {
                        newFileName = m_fileHelper.getFilenameFromFile(session, userData.getFileId());
                    } catch (OXException e) {
                        newFileName = null;
                    }
                    if (null == newFileName) {
                        jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.COPYDOCUMENT_FILENAME_UNKNOWN_ERROR.getAsJSON());
                        bCreateDocumentCopy = false;
                    }
                }

                if (bCreateDocumentCopy) {
                    ErrorCode errorCode = impl_copyDocument(session, userData.getFolderId(), userData.getFileId(), targetFolder, newFileName, asTemplate, jsonResult);
                    jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON());
                }
            } else {
                try {
                    // provide change result to the client
                    jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.COPYDOCUMENT_FAILED_ERROR.getAsJSON());
                } catch (JSONException je) {
                    LOG.error("RT connection: copyDocument catches exception trying to set result to JSON object. ", je);
                }
            }
        } catch (JSONException e) {
            LOG.warn("RT connection: copyDocument - failed to copy document.", e);

            try {
                // provide change result to the client
                jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, ErrorCode.COPYDOCUMENT_FAILED_ERROR.getAsJSON());
            } catch (JSONException je) {
                LOG.error("RT connection: copyDocument catches exception trying to set result to JSON object. ", je);
            }
        }

        return MessageHelper.checkJSONResult(jsonResult);
    }

    /**
     * Creates a copy of the current document including changes within the message chunk list.
     *
     * @param session      The session of the user that requested to create a copy of the current document.
     * @param folderId     The folder id of the file to be copied.
     * @param fileId       The id of the file to be copied.
     * @param targetFolder The target folder where the document should be stored to. Must not be null or an empty string.
     * @param fileName     The file name of the new document without extension. Must not be null or an empty string.
     * @param asTemplate   Specifies if the new document should be a template.
     * @param jsonResult   Contains the json part of the copy document method. If the result of the method is no error, these values can be
     *                     used to retrieve the new file id, folder id, version, mime type and file name.
     * @return The error code of the request. Must be ErrorCode.NO_ERROR if no error occurred, otherwise it describes the problem that
     *         prevented to store the document in the target folder.
     */
    private ErrorCode impl_copyDocument(final Session session, final String folderId, final String fileId, String targetFolder, String fileName, boolean asTemplate, JSONObject jsonResult) {
        ErrorCode errorCode = ErrorCode.COPYDOCUMENT_FAILED_ERROR;

        if ((null != session) && (null != targetFolder) && (null != fileName)) {
            final IDBasedFileAccess fileAccess = getFileAccess(session);
            InputStream documentStm = null;
            boolean rollback = false;
            boolean giveUp = false;

            if (null != fileAccess) {

                try {
                    ArrayList<MessageChunk> currentMessages = null;
                    int osn = -1;

                    // make sure that no save operation interrupts our copy document processing
                    boolean canContinue = waitForSaveInProgressAvailableWithTimeout(WAIT_FOR_SAVEINPROGRESS_LOCK);
                    if (!canContinue) {
                        // We are not able to continue although waiting for WAIT_FOR_SAVEINPROGRESS_LOCK milliseconds
                        // bail out and provide a error to the client.
                        errorCode = ErrorCode.GENERAL_SYSTEM_BUSY_ERROR;
                        return errorCode;
                    }

                    // from here we can be sure that no one can interrupt us storing this document
                    synchronized (getSyncAccess()) {
                        currentMessages = getMessageChunkClone();
                        synchronized (m_connectionStatus) {
                            osn = m_connectionStatus.getOperationStateNumber();
                        }
                    }

                    // set up the new file object used to create a new infostore item
                    final File targetFile = new DefaultFile();
                    final File sourceFile = m_fileHelper.getMetaDataFromFile(session, fileId, FileStorageFileAccess.CURRENT_VERSION);
                    String extension = FileHelper.getExtension(sourceFile.getFileName());
                    String mimeType = sourceFile.getFileMIMEType();

                    // We want to copy the document and store it as a template
                    // document. Make sure that the document is not already a
                    // docment template.
                    if (asTemplate && !DocumentFormatHelper.isSupportedTemplateFormat(mimeType, extension)) {
                        // Document should be copied and "transformed" into a document template.
                        HashMap<String, String> templateFormatInfo = DocumentFormatHelper.getTemplateFormatInfoForDocument(mimeType, extension);
                        if (null != templateFormatInfo) {
                            mimeType = templateFormatInfo.get(Properties.PROP_MIME_TYPE);
                            extension = templateFormatInfo.get(Properties.PROP_INPUT_TYPE);
                        } else {
                            // We don't know target template format. Give up and provide error code.
                            errorCode = ErrorCode.COPYDOCUMENT_TEMPLATE_FORMAT_UNKOWN_ERROR;
                            giveUp = true;
                        }
                    }

                    if (!giveUp) {
                        // use provided values to initialize file object
                        targetFile.setId(FileStorageFileAccess.NEW);
                        targetFile.setFileName(fileName + "." + extension);
                        targetFile.setFolderId(targetFolder);
                        targetFile.setFileMIMEType(mimeType);
                        targetFile.setDescription(sourceFile.getDescription());

                        // Create the new document stream containing the latest changes
                        // from the message chunk list.
                        final SaveDebugProperties dbgProps = (m_slowSave) ? new SaveDebugProperties() : null;
                        if (null != dbgProps) {
                            setupSaveDebugProperties(dbgProps, session);
                        }

                        final OXDocument oxDocument = new OXDocument(session, getServices(), folderId, fileId, m_fileHelper, m_newDocLoaded, m_resourceManager, m_storageHelper, isDebugOperations(), dbgProps);
                        oxDocument.setUniqueDocumentId(osn);

                        final OXDocument.ResolvedStreamInfo docStreamInfo = oxDocument.getResolvedDocumentStream(
                            ImExportHelper.getExporter(getServices(), m_fileHelper),
                            m_resourceManager,
                            currentMessages,
                            null,
                            false,
                            asTemplate);

                        if ((null != docStreamInfo) && (null != docStreamInfo.resolvedStream) && (docStreamInfo.debugStream == null)) {
                            documentStm = docStreamInfo.resolvedStream;
                            fileAccess.startTransaction();
                            rollback = true;
                            LOG.trace("RT connection: [copyDocument] saveDocument using meta data: " + ((null != targetFile) ? targetFile.toString() : "null"));
                            fileAccess.saveDocument(targetFile, documentStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
                            fileAccess.commit();
                            rollback = false;

                            // return actual parameters of new file
                            jsonResult.put("id", targetFile.getId());
                            jsonResult.put("folder_id", targetFile.getFolderId());
                            jsonResult.put("version", targetFile.getVersion());
                            jsonResult.put("filename", targetFile.getFileName());
                            jsonResult.put("file_mimetype", targetFile.getFileMIMEType());
                            errorCode = ErrorCode.NO_ERROR;
                        } else {
                            // extract error code from the result object
                            errorCode = (null != docStreamInfo) ? docStreamInfo.errorCode : ErrorCode.COPYDOCUMENT_FAILED_ERROR;
                        }
                    }
                } catch (OXException e) {
                    LOG.error("RT connection: impl_copyDocument - failed to copy document.", e);
                    errorCode = ExceptionToErrorCode.map(e, ErrorCode.GENERAL_PERMISSION_CREATE_MISSING_ERROR, true);
                } catch (JSONException e) {
                    LOG.error("RT connection: impl_copyDocument - failed to setup result json object");
                } finally {
                    // Roll-back (if needed) and finish
                    if (rollback) {
                        TransactionAwares.rollbackSafe(fileAccess);
                    }

                    TransactionAwares.finishSafe(fileAccess);
                    IOUtils.closeQuietly(documentStm);

                    // decrement counter to enable saving again
                    this.decSaveInProgress();
                }
            }
        }

        return errorCode;
    }

    /**
     * Provides thread-safe the current connection status as a shallow clone.
     *
     * @return The current connection status as a shallow clone.
     */
    public ConnectionStatus getConnectionStatusClone() {
        synchronized (m_connectionStatus) {
            return m_connectionStatus.clone();
        }
    }

    /**
     * Saves the lastest changes of the document to the infostore.
     *
     * @param session   The session of the client that requested to store the lastest changes.
     * @param fromId    The user id of the client that requested to store the latest changes.
     * @param forceSave Forces a save even if a concurrent save is in progress. This should only be used in special situations, like a
     *                  shutdown and an emergency save is needed.
     * @param finalSave Specifies if this is the last flush so the filter can do cleanup work.
     * @return          An error code describing if and why the method has failed. In case of success ErrorCode.NO_ERROR is returned.
     */
    protected ErrorCode flushDocumentAndSetCreateVersion(final Session session, final ID fromId, boolean forceSave, boolean finalSave) {
        final ErrorCode errorCode = flushDocument(session, fromId, false, false);
        // Check if storage supports versions - if true we want to have a version for
        // every editor leaving the document. Otherwise we want to preserve the version
        // that existed when the first user opened the document.
        if (m_storageHelper.supportsFileVersions()) {
            // if editor changes the next save must create a new version
            setCreateVersion(true);
        }
        // send possible flushDocument() error code to other clients
        impl_sendFlushInfoOnError(errorCode, fromId, false);

        return errorCode;
    }

    /**
     * Save the latest changes of the document to the infostore.
     *
     * @param session   The session of the client that requested to store the lastest changes.
     * @param id     
     * @param forceSave Forces a save even if a concurrent save is in progress. This should only be used in special situations, like a
     *                  shutdown and an emergency save is needed.
     * @param finalSave Specifies if this is the last flush so the filter can do cleanup work.
     * @return          An error code describing if and why the method has failed. In case of success ErrorCode.NO_ERROR is returned.
     */
    protected ErrorCode flushDocument(final Session session, final ID id, boolean forceSave, boolean finalSave) {
        final UserData userData = this.getUserData(id);

        return flushDocument(session, userData, forceSave, finalSave);
    }

    /**
     * Save the latest changes of the document to the infostore.
     *
     * @param session   The session of the client that requested to store the lastest changes.
     * @param forceSave Forces a save even if a concurrent save is in progress. This should only be used in special situations, like a
     *                  shutdown and an emergency save is needed.
     * @param finalSave Specifies if this is the last flush so the filter can do cleanup work.
     * @return          An error code describing if and why the method has failed. In case of success ErrorCode.NO_ERROR is returned.
     */
    protected ErrorCode flushDocument(Session session, final UserData userData, boolean forceSave, boolean finalSave) {
        ErrorCode errorCode = ErrorCode.NO_ERROR;
        long startTime = System.currentTimeMillis();

        if (null != session) {
            LOG.debug("RT connection: flushDocument called.");

            // Check dispose flag and bail out if set - could be possible as the
            // signOffMessage is marked as asynchronous.
            if (this.isDisposed()) {
                LOG.debug("RT connection: flushDocument detected disposed instance - no need to flush");
                return errorCode;
            }

            // We only want to have just ONE save document process with
            // only ONE exception: if forceSave is set (only if the connection
            // is going to be disposed) we have to write the latest state.
            final int saveCount = this.incAndCheckSaveInProgress();
            if ((saveCount > 0) && !forceSave) {
                this.decSaveInProgress();
                // save is in progress - try to wait a little bit to acquire save lock
                boolean canContinue = waitForSaveInProgressAvailableWithTimeout(WAIT_FOR_SAVEINPROGRESS_LOCK);
                if (!canContinue) {
                    LOG.error("RTConnection: Waiting for save lock timed out - previous save is still in progress - giving up, save not possible!");
                    return ErrorCode.SAVEDOCUMENT_SAVE_IN_PROGRESS_ERROR;
                }
            }

            // From here we can be sure that just one thread is active
            boolean pendingOperations = false;
            ArrayList<MessageChunk> currentMessages = null;
            int osn = -1;
            int currSize = 0;
            final SaveDebugProperties dbgProps = (m_slowSave) ? new SaveDebugProperties() : null;
            if (null != dbgProps) {
                setupSaveDebugProperties(dbgProps, session);
            }

            // make a shallow copy of the message chunk list to have a defined
            // point where we save the document. handleApplyActions can be
            // called from a different thread and manipulates the message chunk
            // list, too.
            synchronized (getSyncAccess()) {
                pendingOperations = !m_messageChunkList.isEmpty();
                if (pendingOperations) {
                    currentMessages = getMessageChunkClone();
                }
                synchronized (m_connectionStatus) {
                    osn = m_connectionStatus.getOperationStateNumber();
                }
            }

            if (!pendingOperations) {
                // no pending operations to save -> bail out early
                LOG.debug("RT connection: No changes in message chunk list found => no need to flush document");
                this.decSaveInProgress();
                return errorCode;
            }

            // For debugging purposes we log some data about the current pending operations and server osn.
            LOG.debug("RT connection: Flushing operations, current message chunk list size = " + currentMessages.size());
            LOG.debug("RT connection: Flushing operations, current server osn = " + String.valueOf(osn));
            LOG.debug("RT connection: Flushing operations (before save): last operation in current message chunk list = " + OperationHelper.operationToString(OperationHelper.debugGetLastOperationFromMessageChunkList(currentMessages)));

            // we now try to save the document using the copied pending operations
            final boolean createVersion = this.getCreateVersion();
            OXDocument oxDocument = null;
            SaveResult saveResult = null;
            try {
                oxDocument = new OXDocument(session, getServices(), userData, m_fileHelper, m_newDocLoaded, m_resourceManager, m_storageHelper, isDebugOperations(), dbgProps);
                oxDocument.setUniqueDocumentId(osn);
                errorCode = oxDocument.getLastError();

                if (errorCode.isNoError()) {
                    saveResult = oxDocument.save(ImExportHelper.getExporter(getServices(), m_fileHelper), m_resourceManager, currentMessages, !createVersion, finalSave);
                    if (saveResult.errorCode.getErrorClass() == ErrorCode.ERRORCLASS_FATAL_ERROR) {
                        throw new OXDocumentException("Fatal error detected while saving document", saveResult.errorCode);
                    } else {
                        // take over non-fatal errors/warnings to deliver them to the clients
                        errorCode = saveResult.errorCode;
                    }
                } else {
                    // We were not able to read the document bail out throwing
                    // exception with the provided error code.
                    throw new OXDocumentException("Error reading document stream via OXDocuments", errorCode);
                }
            } catch (OXDocumentException e) {
                LOG.error("RT connection: Exception catched while trying to flush document", e);
                // Document exception detected -> this is a fatal error
                errorCode = e.getErrorcode();
                if (errorCode.getErrorClass() == ErrorCode.ERRORCLASS_FATAL_ERROR) {
                    // Fatal error detected on save which means that a backup file
                    // has been created. Further saving won't help, therefore we
                    // clear the message chunk list
                    syncClearMessageChunkList();
                }
            } catch (Exception e) {
                LOG.error("RT connection: Exception catched while trying to flush document", e);
                // Exception detected -> this is a fatal error
                errorCode = ErrorCode.setErrorClass(ErrorCode.SAVEDOCUMENT_FAILED_ERROR, ErrorCode.ERRORCLASS_FATAL_ERROR);
                // Fatal error detected on save which means that a backup file
                // has been created. Further saving won't help, therefore we
                // clear the message chunk list
                syncClearMessageChunkList();
            } finally {
                // if we encountered an error just bail out here
                if (errorCode.isError()) {
                    this.decSaveInProgress();
                    return errorCode;
                }
            }

            // Check dispose state as we can return from a lengthy save and now
            // work on a disposed instance. The following code must not run on
            // a disposed instance.
            if (!this.isDisposed()) {
                try {
                    // clean messageChunkList in case the messageChunkList has been flushed
                    int messageChunkListSize = 0;
                    JSONObject outputOperation = null;

                    if (createVersion) {
                        // After a new version we want to save revisionless until an editor leaves the document.
                        setCreateVersion(false);
                    }

                    synchronized (getSyncAccess()) {
                        currSize = m_messageChunkList.size();

                        // update sync status to the latest saved document
                        this.m_syncStatus.updateSyncInfo((saveResult != null) ? saveResult.version : null, osn);

                        // Remove all operations which have been saved recently
                        // leaving just the new ones added during the save process.
                        int index = currentMessages.size() - 1;

                        if (index == m_messageChunkList.size() - 1) {
                            m_messageChunkList.clear(); // optimization
                        } else if ((index >= 0) && ((index + 1) < m_messageChunkList.size())) {
                            // New operation list must only hold the new operations
                            // therefore we need to remove the old ones.
                            ArrayList<MessageChunk> copy = getMessageChunkClone();
                            List<MessageChunk> newPart = copy.subList(index + 1, m_messageChunkList.size());
                            m_messageChunkList.clear();
                            m_messageChunkList.addAll(newPart);
                        }
                        if (LOG.isDebugEnabled()) {
                            messageChunkListSize = m_messageChunkList.size();
                            outputOperation = OperationHelper.debugGetFirstOperationFromMessageChunkList(m_messageChunkList);
                        }
                        this.setLastSaveTime();
                        synchronized (m_connectionStatus) {
                            osn = m_connectionStatus.getOperationStateNumber();
                        }
                    }

                    LOG.debug("RT connection: Flushing operations completed.");
                    LOG.debug("RT connection: Flushing operations, current server osn = " + String.valueOf(osn));
                    LOG.debug("RT connection: Flushing operations (after save): Before adjustment, message chunk list size = " + currSize);
                    LOG.debug("RT connection: Flushing operations (after save): After adjustment, message chunk list size = " + messageChunkListSize);
                    LOG.debug("RT connection: Flushing operation (after save): First operation in messsage chunk list = " + OperationHelper.operationToString(outputOperation));

                    // update Statistics with saved document count (standard CLOSE type)
                    DocumentEventHelper.addSaveEvent(m_fileHelper, SaveType.CLOSE);
                } catch (Throwable e) {
                    LOG.error("RT connection: flushDocument catched exception while cleaning up - state could be inconsistent!", e);
                    // Exception detected while updating the message chunk list
                    // This will likely result in a inconsistency between document file
                    // and operations, therefore stop the clients to make further changes.
                    errorCode = ErrorCode.setErrorClass(ErrorCode.SAVEDOCUMENT_FAILED_ERROR, ErrorCode.ERRORCLASS_FATAL_ERROR);
                } finally {
                    // decrement the atomic integer
                    this.decSaveInProgress();

                    // logging time for flushing the document
                    long flushDocumentTime = (System.currentTimeMillis() - startTime);
                    LOG.trace("RT connection: TIME to flushDocument: " + flushDocumentTime + "ms");
                }
            }
        } else {
            // no session - therefore saving is not possible
            LOG.error("RT connection: flushDocument called without session - saving is not possible");
        }

        return errorCode;
    }

    /**
     * Starts a new "preparelosingeditrights" timer thread which ensures that a non-responsive client won't block switchting edit rights.
     *
     * @param session The session of the current editor client
     * @param id The ID of the current editor client.
     */
    protected void startPrepareLosingEditRightsTimer(final Session session, final ID id) {
        TimerService timerService = getServices().getService(TimerService.class);

        if ((timerService != null) && (m_nMaxTimeForSendingLastActions > 0)) {
            // zero or negative values mean there is no time out at all
            m_prepareLosingEditRightsTimer = timerService.schedule(
                new PrepareLosingEditRightsTimeoutRunnable(this, session, id),
                m_nMaxTimeForSendingLastActions);
        }
    }

    /**
     * Cancels a running "preparelosingeditrights" timer. Won't do anything if there is no
     * preparelosingeditrights timer running.
     */
    protected void cancelPrepareLosingEditRightsTimer() {
        if (m_prepareLosingEditRightsTimer != null) {
            m_prepareLosingEditRightsTimer.cancel();
            m_prepareLosingEditRightsTimer = null;
        }
    }

    /**
     * Starts a new "complete edit rights" timer thread which ensures that a long-
     * running flushDocument doesn't prevent us from sending the new editor client
     * a status message with more information about the switch process.
     *
     * @param session The session of the old editor client.
     * @param newEditorId The ID of the new editor client.
     */
    protected void startCompleteEditRightsTimer(final Session session, final ID newEditorId) {
        TimerService timerService = getServices().getService(TimerService.class);

        if (timerService != null) {
            m_completeEditRightsRunnable = new CompleteEditRightsTimeoutRunnable(this, newEditorId);
            m_completeSwitchingEditRightsTimer = timerService.schedule(m_completeEditRightsRunnable, WAIT_FOR_SAVEINPROGRESS_EDITRIGHTS);
        }
    }

    /**
     * Cancels a running "complete edit rights" timer. Won't do anything, if there is no
     * "complete edit rights" timer running.
     */
    protected void cancelCompleteEditRightsTimer() {
        if (m_completeEditRightsRunnable != null) {
            m_completeEditRightsRunnable.setToComplete();
        }
        if (m_completeSwitchingEditRightsTimer != null) {
            m_completeSwitchingEditRightsTimer.cancel();
            m_completeSwitchingEditRightsTimer = null;
            m_completeEditRightsRunnable = null;
        }
    }

    /**
     * Determines, if the current client is able to create/write the backup file.
     *
     * @param session The session of the current client
     * @param folderId The ID of the folder containing the associated document.
     * @return TRUE, if the backup file can be written/created by the current client, otherwise FALSE.
     */
    protected boolean canCreateOrWriteBackupFile(final Session session, final String folderId) {
        DocumentMetaData metaData = null;
        synchronized(getSyncAccess()) {
            metaData = this.getLastKnownMetaData();
        }
        return BackupFileHelper.canCreateOrWriteBackupFile(getServices(), session, this.getStorageHelper(), folderId, metaData);
    }

    /**
     * Checks that the provided ID is member and editor of this document.
     *
     * @param id A ID to be checked for membership and edit rights.
     * @return TRUE, if the id is member and editor, otherwise FALSE.
     */
    protected boolean isValidMemberAndEditor(final ID id) {
        boolean result = false;

        if (null != id) {
            final String editRightsID = getConnectionStatusClone().getCurrentEditingUserId();
            ID editorId = ConnectionStatus.isNotEmptyUser(editRightsID) ? new ID(editRightsID) : null;
            result = (null != editorId) && (editorId.equals(id));
        }
        return result;
    }

    /**
     * Provides the number of operations not applied to a saved revision of the document.
     *
     * @return The number of pending operations (not saved in the document file yet).
     */
    @Override
    public long getPendingOperationsCount() {
        long num = 0;

        synchronized (getSyncAccess()) {
            if (null != m_messageChunkList) {
                num = this.m_messageChunkList.size();
            }
        }
        return num;
    }

    /**
     * Clears the message chunk list in case of errors where saving a complete
     * document is not possible anymore. E.g. when the filter detects a problem
     * with the operations.
     */
    private void syncClearMessageChunkList() {
        synchronized (getSyncAccess()) {
            if (null != m_messageChunkList) {
                m_messageChunkList.clear();
            }
        }
    }

    protected int getMessageQueueCount() {
        return (null != m_messageChunkList) ? m_messageChunkList.size() : 0;
    }

    protected boolean addMessageChunkToQueue(MessageChunk messageChunk) {
        if (null != m_messageChunkList) {
            m_messageChunkList.add(messageChunk);
            return true;
        }
        return false;
    }

    protected boolean isModified() {
        return (null != m_messageChunkList) ? (m_messageChunkList.size() > 0) : false;
    }

    /**
     * Sets the create version state to the provided value in a thread-safe way.
     *
     * @param newValue The new value of the create version state.
     * @return The old value of the create version state.
     */
    synchronized private boolean setCreateVersion(boolean newValue) {
        boolean oldValue = m_createVersion;
        m_createVersion = newValue;
        return oldValue;
    }

    protected void setupSaveDebugProperties(final SaveDebugProperties props, final Session session) {
        try {
            UserConfigurationHelper userConfHelper = new UserConfigurationHelper(this.getServices(), session, "io.ox/office", Mode.WRITE_BACK);

            props.setProperty(SaveDebugProperties.DEBUG_PROP_SLOWSAVE, m_slowSave);
            props.setProperty(SaveDebugProperties.DEBUG_PROP_SLOWSAVETIME, m_slowSaveTime);
            props.setProperty(SaveDebugProperties.DEBUG_PROP_SLOWSAVEUSER, userConfHelper.getBoolean("module/debugslowsaveuser", false));
        } catch (Throwable e) { }
    }

    /**
     * Provides a clone of the current message chunk list.
     * ATTENTION: Must be synchronized with getSyncAccess() form outside.
     *
     * @return A clone of the current message chunk list.
     */
    @SuppressWarnings("unchecked")
    public ArrayList<MessageChunk> getMessageChunkClone() {
        return (null != m_messageChunkList) ? (ArrayList<MessageChunk>) m_messageChunkList.clone() : null;
    }

    protected final String getResourceID() { return m_resourceID; }

    synchronized public boolean getCreateVersion() { return m_createVersion; }

    protected DocFileHelper getFileHelper() { return m_fileHelper; }

    public ResourceManager getResourceManager() { return m_resourceManager; }

    public StorageHelper getStorageHelper() { return m_storageHelper; }

    public ConnectionStatus getConnectionStatus() { return m_connectionStatus; }

    public DocumentMetaData getLastKnownMetaData() { return m_lastKnownMetaData; }

    protected void setLastKnownMetaData(final DocumentMetaData metaData) { m_lastKnownMetaData = metaData; }

    public boolean isNewDocLoaded() { return m_newDocLoaded; }

    public long getActionsReceived() { return m_nActionsReceived; }

    public boolean isActionUpdateNecessary() { return (m_nActionsReceived >= ACTIONS_UNTIL_FORCE_UPDATE); }

    public long incActionsReceived() { return (m_nActionsReceived++); }

    public void clearActionsReceived() { m_nActionsReceived = 0; }

    public boolean isDebugOperations() { return m_debugOperations; }

}
