/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.CTDataValidation;
import org.xlsx4j.sml.CTDataValidation_legacy;
import org.xlsx4j.sml.CTDataValidations;
import org.xlsx4j.sml.CTDataValidations_legacy;
import org.xlsx4j.sml.IDataValidation;
import org.xlsx4j.sml.IDataValidations;
import org.xlsx4j.sml.STDataValidationErrorStyle;
import org.xlsx4j.sml.STDataValidationOperator;
import org.xlsx4j.sml.STDataValidationType;
import org.xlsx4j.sml.SmlUtils;
import org.xlsx4j.sml.Worksheet;

import com.openexchange.office.ooxml.xlsx.OperationDocument;

public class ValidationUtils {

    final static String[] enumValidationType = { "all", "integer", "number", "list", "date", "time", "length", "custom", "source" };
    final static String[] enumCompare        = { "between", "notBetween", "equal", "notEqual", "less", "lessEqual", "greater", "greaterEqual" };
    final static String[] enumErrorType      = { "error", "warning", "info" };

    // transforming string token "a,b,c" to our style "a","b","c"
    public static String formulaAddQuotations(String value) {
	    if(value.length()>2&&value.charAt(0)=='"'&&value.charAt(value.length()-1)=='"') {
	    	final StringTokenizer tokenizer = new StringTokenizer(value.substring(1, value.length()-1), ",");
	    	final StringBuffer buffer = new StringBuffer();
	    	while(tokenizer.hasMoreTokens()) {
	    		if(buffer.length()>0) {
	    			buffer.append(',');
	    		}
	    		buffer.append('"').append(tokenizer.nextToken()).append('"');
	    	}
	    	value = buffer.toString();
	    }
	    return value;
    }

	// transforming string token "a","b","c" to "a,b,c"
    public static String formulaRemoveQuotations(String value) {
        if(value.length()>2&&value.charAt(0)=='"'&&value.charAt(value.length()-1)=='"') {
        	final StringBuffer buffer = new StringBuffer(value);
        	for(int i=buffer.length()-2; i>1; i--) {
        		if(buffer.charAt(i)=='"') {
        			buffer.deleteCharAt(i);
        		}
        	}
        	return value = buffer.toString();
        }
    	return value;
    }

    public static void addInsertValidationOperation(JSONArray operationsArray, int sheetIndex, int validationIndex, List<SmlUtils.CellRefRange> rangeList, STDataValidationType type, STDataValidationOperator operator, String value1, String value2,
                                            boolean showInfo, String infoTitle, String infoText, boolean showError, String errorTitle, String errorText, STDataValidationErrorStyle errorStyle, boolean showDropDown, boolean ignoreBlank)
        throws JSONException {

        if(value1==null) {
            return;
        }
        final JSONObject insertValidationObject = new JSONObject();
        insertValidationObject.put("name", "insertValidation");
        insertValidationObject.put("sheet", sheetIndex);
        insertValidationObject.put("index", validationIndex);
        insertValidationObject.put("ranges", Utils.createJsonArrayFromCellRefRange(rangeList));
        int val = type.ordinal();
        if(type==STDataValidationType.LIST) {
            if(!value1.isEmpty()&&value1.charAt(0)!='"') {
                val = enumValidationType.length - 1;    // using type source if value does not start with "
            }
        }
        if(val>=enumValidationType.length)
            val = 0;
        insertValidationObject.put("type", enumValidationType[val]);
        if(operator!=STDataValidationOperator.BETWEEN) {
            val = operator.ordinal();
            if(val>=enumCompare.length)
                val = 0;
            insertValidationObject.put("compare", enumCompare[val]);
        }
        insertValidationObject.put("value1", formulaAddQuotations(value1));
        if(value2!=null) {
            insertValidationObject.put("value2", formulaAddQuotations(value2));
        }
        if(showInfo==false) {
            insertValidationObject.put("showInfo", showInfo);
        }
        if(infoTitle!=null) {
            insertValidationObject.put("infoTitle", infoTitle);
        }
        if(infoText!=null) {
            insertValidationObject.put("infoText",  infoText);
        }
        if(showError==false) {
            insertValidationObject.put("showError", showError);
        }
        if(errorTitle!=null) {
            insertValidationObject.put("errorTitle", errorTitle);
        }
        if(errorText!=null) {
            insertValidationObject.put("errorText", errorText);
        }
        if(errorStyle!=STDataValidationErrorStyle.STOP) {
            val = errorStyle.ordinal();
            if(val>=enumErrorType.length)
                val = 0;
            insertValidationObject.put("errorType", enumErrorType[val]);
        }
        if(type==STDataValidationType.LIST&&showDropDown==false) {
            insertValidationObject.put("showDropDown", showDropDown);
        }
        if(ignoreBlank==false) {
            insertValidationObject.put("ignoreEmpty", ignoreBlank);
        }
        operationsArray.put(insertValidationObject);
    }

    private static List<IDataValidations> getDataValidations(OperationDocument operationDocument, int sheetIndex) {
        return OperationDocument.getDataValidations(operationDocument.getWorksheet(sheetIndex));
    }

    private static IDataValidation getDataValidation(OperationDocument operationDocument, int sheetIndex, int index) {

        int i = 0;
        final List<IDataValidations> dataValidationsList = getDataValidations(operationDocument, sheetIndex);
    	for(IDataValidations iDataValidations:dataValidationsList) {
    		if(iDataValidations.getDataValidation().size()+i<=index) {
    			i+=iDataValidations.getDataValidation().size();
    		}
    		else {
    			return iDataValidations.getDataValidation().get(index - i);
    		}
    	}
    	return null;
	}

	private static void fillValidation(IDataValidation iDataValidation, JSONObject op, boolean insert)
		throws JSONException {

		// TODO: check for required attributes if "insert" is true

		if(iDataValidation!=null) {
	        final Iterator<String> keys = op.keys();
	        while(keys.hasNext()) {
	            final String attr = keys.next();
	            final Object value = op.get(attr);
	            if(attr.equals("ranges")) {
	            	Utils.applySqrefRangeFromJsonArray(iDataValidation.getsqref(), value);
	            }
	            else if(attr.equals("type")) {
            		STDataValidationType dataValidationType = STDataValidationType.NONE;
	            	if(value instanceof String) {
	            		final String type = (String)value;
	            		if(type.equals("all")) {
	            			dataValidationType = STDataValidationType.NONE;
	            		} else if(type.equals("integer")) {
	            			dataValidationType = STDataValidationType.WHOLE;
	            		} else if(type.equals("number")) {
	            			dataValidationType = STDataValidationType.DECIMAL;
	            		} else if(type.equals("list")) {
	            			dataValidationType = STDataValidationType.LIST;
	            		} else if(type.equals("data")) {
	            			dataValidationType = STDataValidationType.DATE;
	            		} else if(type.equals("time")) {
	            			dataValidationType = STDataValidationType.TIME;
	            		} else if(type.equals("length")) {
	            			dataValidationType = STDataValidationType.TEXT_LENGTH;
	            		} else if(type.equals("custom")) {
	            			dataValidationType = STDataValidationType.CUSTOM;
	            		} else if(type.equals("source")) {
	            			dataValidationType = STDataValidationType.LIST;
	            		}
	               	}
            		iDataValidation.setType(dataValidationType);
	            }
	            else if(attr.equals("compare")) {
	            	STDataValidationOperator dataValidationOperator = STDataValidationOperator.BETWEEN;
	            	if(value instanceof String) {
	            		final String operator = (String)value;
	            		if(operator.equals("between")) {
	            			dataValidationOperator = STDataValidationOperator.BETWEEN;
	            		} else if(operator.equals("notBetween")) {
	            			dataValidationOperator = STDataValidationOperator.NOT_BETWEEN;
	            		} else if(operator.equals("equal")) {
	            			dataValidationOperator = STDataValidationOperator.EQUAL;
	            		} else if(operator.equals("notEqual")) {
	            			dataValidationOperator = STDataValidationOperator.NOT_EQUAL;
	            		} else if(operator.equals("less")) {
	            			dataValidationOperator = STDataValidationOperator.LESS_THAN;
	            		} else if(operator.equals("lessEqual")) {
	            			dataValidationOperator = STDataValidationOperator.LESS_THAN_OR_EQUAL;
	            		} else if(operator.equals("greater")) {
	            			dataValidationOperator = STDataValidationOperator.GREATER_THAN;
	            		} else if(operator.equals("greaterEqual")) {
	            			dataValidationOperator = STDataValidationOperator.GREATER_THAN_OR_EQUAL;
	            		}
	            	}
	            	iDataValidation.setOperator(dataValidationOperator);
	            }
	            else if(attr.equals("value1")) {
            		iDataValidation.setFormula1(formulaRemoveQuotations(value instanceof String?(String)value:null));
	            }
	            else if(attr.equals("value2")) {
            		iDataValidation.setFormula2(formulaRemoveQuotations(value instanceof String?(String)value:null));
	            }
	            else if(attr.equals("showInfo")) {
            		iDataValidation.setShowInputMessage(value instanceof Boolean?(Boolean)value:null);
	            }
	            else if(attr.equals("infoTitle")) {
	            	iDataValidation.setPromptTitle(value instanceof String?(String)value:null);
	            }
	            else if(attr.equals("infoText")) {
	            	iDataValidation.setPrompt(value instanceof String?(String)value:null);
	            }
	            else if(attr.equals("showError")) {
            		iDataValidation.setShowErrorMessage(value instanceof Boolean?(Boolean)value:null);
	            }
	            else if(attr.equals("errorTitle")) {
	            	iDataValidation.setErrorTitle(value instanceof String?(String)value:null);
	            }
	            else if(attr.equals("errorText")) {
	            	iDataValidation.setError(value instanceof String?(String)value:null);
	            }
	            else if(attr.equals("errorType")) {
	            	STDataValidationErrorStyle dataValidationErrorStyle = STDataValidationErrorStyle.STOP;
	            	if(value instanceof String) {
	            		final String errorStyle = (String)value;
	            		if(errorStyle.equals("error")) {
	            			dataValidationErrorStyle = STDataValidationErrorStyle.STOP;
	            		} else if(errorStyle.equals("warning")) {
	            			dataValidationErrorStyle = STDataValidationErrorStyle.WARNING;
	            		} else if(errorStyle.equals("info")) {
	            			dataValidationErrorStyle = STDataValidationErrorStyle.INFORMATION;
	            		}
	            	}
	            	iDataValidation.setErrorStyle(dataValidationErrorStyle);
	            }
	            else if(attr.equals("showDropDown")) {
	            	iDataValidation.setShowDropDown(value instanceof Boolean?!(Boolean)value:false);
	            }
	            else if(attr.equals("ignoreEmpty")) {
	            	iDataValidation.setAllowBlank(value instanceof Boolean?(Boolean)value:true);
	            }
	        }
		}
	}


	public static void insertValidation(OperationDocument operationDocument, JSONObject op)
		throws JSONException {

		final int sheet = op.getInt("sheet");
		final int index = op.optInt("index", -1);

		// the dataValidations might exist within two different lists, so we have to create and insert the correct type
		IDataValidation  iDataValidation = null;
		IDataValidations iDataValidations = null;
		if(index!=-1) {
			iDataValidation = getDataValidation(operationDocument, sheet, index);
			if(iDataValidation!=null) {
				iDataValidations = (IDataValidations)iDataValidation.getParent();
			}
		}
		if(iDataValidations==null) {
			// append the dataValidation
			final List<IDataValidations> dataValidationsList = getDataValidations(operationDocument, sheet);
			if(!dataValidationsList.isEmpty()) {
				iDataValidations = dataValidationsList.get(dataValidationsList.size()-1);
			}
			else {
				final Worksheet worksheet = operationDocument.getWorksheet(sheet);
				final CTDataValidations dataValidations = Context.getsmlObjectFactory().createCTDataValidations();
				dataValidations.setParent(worksheet);
				worksheet.setDataValidations(dataValidations);
				iDataValidations = dataValidations;
			}
		}
		if(iDataValidations instanceof CTDataValidations) {
			final CTDataValidations dataValidations = (CTDataValidations)iDataValidations;
			final List<CTDataValidation> validationList = dataValidations.getDataValidation();
			final int listIndex = iDataValidation!=null?validationList.indexOf(iDataValidation):validationList.size();
			final CTDataValidation dataValidation = Context.getsmlObjectFactory().createCTDataValidation();
			dataValidation.setParent(dataValidations);
			validationList.add(listIndex, dataValidation);
			iDataValidation = dataValidation;
		} else if (iDataValidations instanceof CTDataValidations_legacy) {
			final CTDataValidations_legacy dataValidations = (CTDataValidations_legacy)iDataValidations;
			final List<CTDataValidation_legacy> validationList = dataValidations.getDataValidation();
			final int listIndex = iDataValidation!=null?validationList.indexOf(iDataValidation):validationList.size();
			final CTDataValidation_legacy dataValidation = Context.getsmlObjectFactory().createCTDataValidation_legacy();
			dataValidation.setParent(dataValidations);
			validationList.add(listIndex, dataValidation);
			iDataValidation = dataValidation;
		}
		if(iDataValidation!=null) {
			fillValidation(iDataValidation, op, true);
		}
	}

    public static void changeValidation(OperationDocument operationDocument, JSONObject op)
		throws JSONException {

		fillValidation(getDataValidation(operationDocument, op.getInt("sheet"), op.getInt("index")), op, false);
	}

	public static void deleteValidation(OperationDocument operationDocument, int sheet, int index) {
		final IDataValidation iDataValidation = getDataValidation(operationDocument, sheet, index);
		if(iDataValidation!=null) {
			Object parent = iDataValidation.getParent();
			if(parent instanceof IDataValidations) {
				((IDataValidations)parent).getDataValidation().remove(iDataValidation);
			}
		}
	}
}
