/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.ArrayList;
import java.util.List;

import org.docx4j.dml.BaseStyles;
import org.docx4j.dml.CTColorScheme;
import org.docx4j.dml.Theme;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.CTBorder;
import org.xlsx4j.sml.CTBorderPr;
import org.xlsx4j.sml.CTCellAlignment;
import org.xlsx4j.sml.CTCellProtection;
import org.xlsx4j.sml.CTColor;
import org.xlsx4j.sml.CTColors;
import org.xlsx4j.sml.CTFill;
import org.xlsx4j.sml.CTGradientStop;
import org.xlsx4j.sml.CTIndexedColors;
import org.xlsx4j.sml.CTNumFmt;
import org.xlsx4j.sml.CTPatternFill;
import org.xlsx4j.sml.CTRgbColor;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.STBorderStyle;
import org.xlsx4j.sml.STHorizontalAlignment;
import org.xlsx4j.sml.STPatternType;
import org.xlsx4j.sml.STVerticalAlignment;
import org.xlsx4j.sml.SmlUtils;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.OperationDocument;


public class Utils {

    // CTNumFmt

    public static void createNumberFormat(JSONObject dest, long numberFormatId, CTNumFmt numberFormat)
        throws JSONException {

        if(numberFormatId<0&&numberFormat==null) {
            return;
        }
        JSONObject jsonNumberFormatObject = new JSONObject();
        jsonNumberFormatObject.put("id", numberFormatId);
        if(numberFormat!=null) {
            String code = numberFormat.getFormatCode();
            if(code!=null) {
                jsonNumberFormatObject.put("code", code);
            }
        }
        dest.put("numberFormat", jsonNumberFormatObject);
    }

    public static void createFill(JSONObject dest, CTStylesheet stylesheet, CTFill fill)
        throws JSONException {

        if(fill!=null) {
            if (null == fill.getPatternFill() && null != fill.getGradientFill()) {
                //fallback for gradients! taking its first color
                List<CTGradientStop> stop = fill.getGradientFill().getStop();
                if (null != stop && stop.size() > 0) {
                    CTColor color = stop.get(0).getColor();
                    JSONObject jsonColor = createColor(stylesheet, color);
                    dest.put("fillColor", jsonColor);
                }
            }else {
                createPatternFill(dest, stylesheet, fill.getPatternFill());
            }
//          createGradientFill(dest, fill.getGradientFill());
        }
    }

    public static void createPatternFill(JSONObject dest, CTStylesheet stylesheet, CTPatternFill patternFill)
        throws JSONException {

        if(patternFill!=null) {
        	STPatternType patternType = patternFill.getPatternType();
        	if(patternType==null) {
        		patternType = STPatternType.SOLID;
        	}
            switch(patternType) {
                case NONE: {
                    JSONObject jsonColor = new JSONObject(1);
                    jsonColor.put("type", "auto");
                	dest.put("fillColor", jsonColor);
                	break;
                }
                case SOLID :
                default: {
                    JSONObject jsonColor = createColor(stylesheet, patternFill.getFgColor()!=null?patternFill.getFgColor():patternFill.getBgColor());
                    if(jsonColor!=null) {
                        dest.put("fillColor", jsonColor);
                    }
                }
            }
        }
    }

    final static String themeColors[] = {

/* OOXML specification at a glance... if accessing theme colors
 * as described in the specification, then at least black and
 * white are swapped. Some says that only the first two pairs
 * within the index are swapped, other think (like in poi) that each
 * non alpha rbg white and black color is switched
 *
 * TODO: I have to test this. For now I only switch the index into
 * the color scheme.
 *

        "dark1",
        "light1",
        "dark2",
        "light2",
*/
        "light1",
        "dark1",
        "light2",
        "dark2",

        "accent1",
        "accent2",
        "accent3",
        "accent4",
        "accent5",
        "accent6",
        "hyperlink",
        "followedHyperlink"
    };

    final static String[] indexedColors = {
        "000000","FFFFFF","FF0000","00FF00","0000FF","FFFF00","FF00FF","00FFFF",
        "000000","FFFFFF","FF0000","00FF00","0000FF","FFFF00","FF00FF","00FFFF",
        "800000","008000","000080","808000","800080","008080","C0C0C0","808080",
        "9999FF","993366","FFFFCC","CCFFFF","660066","FF8080","0066CC","CCCCFF",
        "000080","FF00FF","FFFF00","00FFFF","800080","800000","008080","0000FF",
        "00CCFF","CCFFFF","CCFFCC","FFFF99","99CCFF","FF99CC","CC99FF","FFCC99",
        "3366FF","33CCCC","99CC00","FFCC00","FF9900","FF6600","666699","969696",
        "003366","339966","003300","333300","993300","993366","333399","333333"
    };

    public static JSONObject createColor(CTStylesheet stylesheet, CTColor color)
        throws JSONException {

        JSONObject jsonColor = new JSONObject();

        if (color!=null) {

            Boolean auto    = color.isAuto();
            Long    theme   = color.getTheme();
            Long    indexed = color.getIndexed();
            byte[]  rgb     = color.getRgb();
            byte    alpha   = -1;

            if(theme!=null&&theme.longValue()<themeColors.length) {
                jsonColor.put("type",  "scheme");
                jsonColor.put("value", themeColors[theme.intValue()]);
            }
            else if(auto!=null&&auto.booleanValue()) {
                jsonColor.put("type", "auto");
            }
            else if(rgb!=null&&rgb.length==3) {
                jsonColor.put("type", "rgb");
                jsonColor.put("value", Commons.bytesToHexString(rgb, 0, 3));
            }
            else if(rgb!=null&&rgb.length==4) {
//                alpha = rgb[0];
//            	Alpha is ignored now because of excel and calc, they ignore it too
                jsonColor.put("type", "rgb");
                jsonColor.put("value", Commons.bytesToHexString(rgb, 1, 3));
            }
            else if(indexed!=null) {
                if(indexed.longValue()>=indexedColors.length) {
                    jsonColor.put("type",  "auto");
                }
                else {
                    jsonColor.put("type", "rgb");
                    String colorValue = indexedColors[indexed.intValue()];
                    if(stylesheet!=null) {
                        // check if indexedColor is overwritten
	                    final CTColors ctColors = stylesheet.getColors();
	                    if(ctColors!=null) {
	                        CTIndexedColors _indexedColors = ctColors.getIndexedColors();
	                        if(_indexedColors!=null) {
	                            List<CTRgbColor> _indexedColorList = _indexedColors.getRgbColor();
	                            if(_indexedColorList.size()>indexed.intValue()) {
	                                CTRgbColor _indexedRgbColor = _indexedColorList.get(indexed.intValue());
	                                if(_indexedRgbColor!=null) {
	                                    byte[] _indexedRgbColorArray = _indexedRgbColor.getRgb();
	                                    if(_indexedRgbColorArray!=null&&_indexedRgbColorArray.length==4) {
	                                        // alpha = indRgbArray[0]; don't know if alpha is used here, value seems to be zero always
	                                        colorValue = Commons.bytesToHexString(_indexedRgbColorArray, 1, 3);
	                                    }
	                                }
	                            }
	                        }
	                    }
                    }
                    jsonColor.put("value", colorValue);
                }
            }
            final JSONArray transformations = createTransformations(color, alpha);
            if (transformations!=null) {
                jsonColor.put("transformations", transformations);
            }
            return jsonColor;
        }
        return null;
    }

    public static JSONObject createDefaultColor()
    	throws JSONException {

    	JSONObject jsonColor = new JSONObject(2);
        jsonColor.put("type",  "scheme");
        jsonColor.put("value", themeColors[1]);
        return jsonColor;
    }

    private static JSONArray createTransformations(CTColor color, byte alpha)
        throws JSONException {

        JSONArray jsonColorTransformations = new JSONArray();
        if(alpha!=-1) {
            jsonColorTransformations.put(createAlphaTransformation(alpha));
        }
        if (color!=null) {
            final Double tint = color.getTint();
            if (tint!=null) {
            	final JSONObject tintTransformation = createTintTransformation(tint.doubleValue());
            	if(tintTransformation!=null) {
            		jsonColorTransformations.put(tintTransformation);
            	}
            }
        }
        return jsonColorTransformations.length()>0?jsonColorTransformations:null;
    }

    private static JSONObject createAlphaTransformation(byte alpha)
        throws JSONException {

        JSONObject alphaTransform = new JSONObject();
        alphaTransform.put("type", "alpha");
        alphaTransform.put("value", (alpha&0xff)/255.0*100000.0);
        return alphaTransform;
    }

    private static JSONObject createTintTransformation(double _tint)
        throws JSONException {

        if(_tint >= 0) {
        	final int tint = 100000 - (int)(_tint * 100000.0);
        	if(tint!=100000) {
	            final JSONObject tintTransform = new JSONObject();
	            tintTransform.put("type", "tint");
	            tintTransform.put("value", tint);
	            return tintTransform;
        	}
        }
        else {
            final int shade = 100000 - (int)(_tint *-100000.0);
            if(shade!=100000) {
                final JSONObject tintTransform = new JSONObject();
	            tintTransform.put("type", "shade");
	            tintTransform.put("value", shade);
	            return tintTransform;
            }
        }
        return null;
    }

    // CTBorder

    public static void createBorders(JSONObject jsonDest, CTStylesheet stylesheet, CTBorder borders)
        throws JSONException {

        if(borders!=null) {
            Commons.jsonPut(jsonDest, "borderLeft", createJSONfromCTBorder(stylesheet, borders.getLeft()));
            Commons.jsonPut(jsonDest, "borderTop", createJSONfromCTBorder(stylesheet, borders.getTop()));
            Commons.jsonPut(jsonDest, "borderRight", createJSONfromCTBorder(stylesheet, borders.getRight()));
            Commons.jsonPut(jsonDest, "borderBottom", createJSONfromCTBorder(stylesheet, borders.getBottom()));
            Commons.jsonPut(jsonDest, "borderInsideHor", createJSONfromCTBorder(stylesheet, borders.getHorizontal()));
            Commons.jsonPut(jsonDest, "borderInsideVert", createJSONfromCTBorder(stylesheet, borders.getVertical()));
        }
    }

    public static int pixelToHmm(double pixel) {
    	return (int) Math.round(pixel / 96.0 * 2540.0);
    }

    public static double hmmToPixel(int hmm) {
    	return hmm * 96.0 / 2540.0;
    }

    public static int HAIR_WIDTH_HMM= pixelToHmm(0.5);
    public static int THIN_WIDTH_HMM = pixelToHmm(1);
    public static int MEDIUM_WIDTH_HMM = pixelToHmm(2);
    public static int THICK_WIDTH_HMM = pixelToHmm(3);

    public static JSONObject createJSONfromCTBorder(CTStylesheet stylesheet, CTBorderPr borderPr)
        throws JSONException {

        if (borderPr==null)
            return null;

        JSONObject jsonBorder = new JSONObject();
        String style = "single";
        double width = THIN_WIDTH_HMM;
        STBorderStyle borderStyle = borderPr.getStyle();
        if(borderStyle!=null) {
            if(borderStyle==STBorderStyle.NONE) {
                style = "none";
            }
            else if(borderStyle==STBorderStyle.HAIR) {
                style = "single";
                width = HAIR_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.THIN) {
                style = "single";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DOTTED) {
                style = "dotted";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DASH_DOT_DOT) {
                style = "dashDotDot";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DASH_DOT) {
                style = "dashDot";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DASHED) {
                style = "dashed";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.MEDIUM) {
                style = "single";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.MEDIUM_DASH_DOT_DOT) {
                style = "dashDotDot";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.MEDIUM_DASH_DOT) {
                style = "dashDot";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.MEDIUM_DASHED) {
                style = "dashed";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.SLANT_DASH_DOT) {
                style = "dashDot";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.THICK) {
                style = "single";
                width = THICK_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DOUBLE) {
                style = "double";
                width = THICK_WIDTH_HMM;
            }
        }
        if(borderStyle!=STBorderStyle.NONE) {
            jsonBorder.put("width", width);
        }
        jsonBorder.put("style", style);

        // creating border color
        Commons.jsonPut(jsonBorder, "color", createColor(stylesheet, borderPr.getColor()));

        return jsonBorder;
    }

    public static void createAlignment(JSONObject jsonDest, CTCellAlignment cellAlignment)
        throws JSONException {

        if(cellAlignment!=null) {
            STHorizontalAlignment horzAlignment = cellAlignment.getHorizontal();
            if(horzAlignment!=null) {
                String hAlign;
                switch(horzAlignment) {
                    case LEFT: hAlign = "left"; break;
                    case CENTER_CONTINUOUS:             // !!PASSTHROUGH INTENDED
                    case CENTER : hAlign = "center"; break;
                    case RIGHT : hAlign = "right"; break;
                    case JUSTIFY : hAlign = "justify"; break;
                    default: hAlign = "auto"; break;
                }
                jsonDest.put("alignHor", hAlign);
            }

            STVerticalAlignment vertAlignment = cellAlignment.getVertical();
            if(vertAlignment!=null) {
                String vAlign;
                switch(vertAlignment) {
                    case TOP: vAlign = "top"; break;
                    case CENTER : vAlign = "middle"; break;
                    case JUSTIFY : vAlign = "justify"; break;
                    default: vAlign = "bottom"; break;
                }
                jsonDest.put("alignVert", vAlign);
            }
            Boolean wrapText = cellAlignment.isWrapText();
            if(wrapText!=null) {
                jsonDest.put("wrapText", wrapText.booleanValue());
            }
        }
    }

    public static void createProtection(JSONObject jsonDest, CTCellProtection cellProtection)
        throws JSONException {

        if(cellProtection!=null) {
            if(cellProtection.isHidden()!=null && cellProtection.isHidden().booleanValue()) {
                jsonDest.put("hidden", true);
            }
            if(cellProtection.isLocked()!=null&&!cellProtection.isLocked().booleanValue()) {
                jsonDest.put("unlocked", true);
            }
        }
        else
            jsonDest.put("unlocked", false);
    }

    public static byte[] themeColorToBytes(Theme theme, String themeColor) {
        byte[] rgbValue = null;
        if(theme!=null) {
            BaseStyles themeElements = theme.getThemeElements();
            if(themeElements!=null) {
                CTColorScheme clrScheme = themeElements.getClrScheme();
                if(clrScheme!=null) {
                    org.docx4j.dml.CTColor ctColor = null;
                    if(themeColor.equals("light1")) {
                        ctColor = clrScheme.getLt1();
                    }
                    else if(themeColor.equals("dark1")) {
                        ctColor = clrScheme.getDk1();
                    }
                    else if(themeColor.equals("light2")) {
                        ctColor = clrScheme.getLt1();
                    }
                    else if(themeColor.equals("dark2")) {
                        ctColor = clrScheme.getDk2();
                    }
                    else if(themeColor.equals("accent1")) {
                        ctColor = clrScheme.getAccent1();
                    }
                    else if(themeColor.equals("accent2")) {
                        ctColor = clrScheme.getAccent2();
                    }
                    else if(themeColor.equals("accent3")) {
                        ctColor = clrScheme.getAccent3();
                    }
                    else if(themeColor.equals("accent4")) {
                        ctColor = clrScheme.getAccent4();
                    }
                    else if(themeColor.equals("accent5")) {
                        ctColor = clrScheme.getAccent5();
                    }
                    else if(themeColor.equals("accent6")) {
                        ctColor = clrScheme.getAccent6();
                    }
                    else if(themeColor.equals("hyperlink")) {
                        ctColor = clrScheme.getHlink();
                    }
                    else if(themeColor.equals("followedHyperlink")) {
                        ctColor = clrScheme.getFolHlink();
                    }
                    else if(themeColor.equals("text1")) {
                        ctColor = clrScheme.getDk1();
                    }
                    else if(themeColor.equals("text2")) {
                        ctColor = clrScheme.getDk2();
                    }
                    else if(themeColor.equals("background1")) {
                        ctColor = clrScheme.getLt1();
                    }
                    else if(themeColor.equals("background2")) {
                        ctColor = clrScheme.getLt2();
                    }
                    if(ctColor!=null) {
                        rgbValue = Commons.ctColorToBytes(ctColor);
                    }
                }
            }
        }
        return rgbValue;
    }

    public static void applyColor(OperationDocument operationDocument, CTColor color, JSONObject jsonColor)
        throws FilterException, JSONException {

        if(color!=null&&jsonColor.has("type")) {
            String type = jsonColor.getString("type");
            color.setTint(null);
            if(type.equals("rgb")&&jsonColor.has("value")) {
                org.docx4j.dml.CTColor ctColor = null;
                color.setRgb(Commons.hexStringToBytes(jsonColor.getString("value")));
                color.setTheme(null);
            } else if (type.equals("scheme")&&jsonColor.has("value")) {
                String themeColor = jsonColor.getString("value");

                long themeColorIndex = 0;
                for(String c:themeColors) {
                    if(c.equals(themeColor)) {
                        break;
                    }
                    themeColorIndex++;
                }
                if(themeColorIndex >= themeColors.length) {
                    if(themeColor.equals("text1")) {
                        themeColorIndex = 1;
                    }
                    else if(themeColor.equals("text2")) {
                        themeColorIndex = 3;
                    }
                    else if(themeColor.equals("background1")) {
                        themeColorIndex = 0;
                    }
                    else if(themeColor.equals("background2")) {
                        themeColorIndex = 2;
                    }
                    else {
                        themeColorIndex = 1;
                    }
                }
                color.setTheme(themeColorIndex);
                if (jsonColor.has("transformations")) {
                    JSONArray transformations = jsonColor.getJSONArray("transformations");
                    addTransformationsToColor(color, transformations);
                }
                color.setRgb(themeColorToBytes(operationDocument.getTheme(), themeColor));
            } else if (type.equals("auto")) {
                color.setAuto(true);
                color.setTheme(null);
            }
        }
    }

    private static void addTransformationsToColor(CTColor color, JSONArray transformations) {
        if ((transformations != null) && (transformations.length() > 0)) {
            Object temp = transformations.opt(0);
            if (temp instanceof JSONObject) {
                JSONObject transformation = (JSONObject)temp;
                String type = transformation.optString("type");
                Double value = transformation.optDouble("value");

                if ((type != null) && (value != null)) {
                    if (type.equals("tint")) {
                        color.setTint((100000.0-value) / 100000.0);
                    }
                    else if (type.equals("shade")) {
                        color.setTint((-(100000.0-value)) / 100000.0);
                    }
                }
            }
        }
    }

    public static void createColumnAttributeRanges(Cols cols, long start, long size) {
        List<Col> colList = cols.getCol();

        int i=0;
        for(;i<colList.size();i++) {
            Col col = colList.get(i);

            if(col.getMax()<start) {
                continue;
            }
            if(col.getMin()>((start+size)-1))
                break;
            long colLength = col.getMax() - col.getMin() + 1;
            if(col.getMin()>start) {
                Col newCol = Context.getsmlObjectFactory().createCol();
                newCol.setMin(start);
                long newEnd = start + size - 1;
                if(newEnd>=col.getMin()) {
                    newEnd = col.getMin() - 1;
                }
                newCol.setMax(newEnd);
                start = newEnd + 1;
                size -= (newCol.getMax() - newCol.getMin()) + 1;
                colList.add(i, newCol);
            }
            else if(col.getMin()==start) {
                if(colLength<size) {
                    start += colLength;
                    size -= colLength;
                }
                else if(colLength>size) {
                    final Col newCol = col.clone();
                    col.setMax(col.getMin()+size-1);
                    newCol.setMin(col.getMax()+1);
                    colList.add(i+1, newCol);
                    size = 0;
                    break;
                }
                else {
                    size = 0;
                    break;
                }
            }
            // colMin() < start
            else if(col.getMax()>=start) {
                final Col newCol = col.clone();
                newCol.setMin(start);
                newCol.setMax(col.getMax());
                col.setMax(start-1);
                colList.add(i+1, newCol);
            }
        }
        if(size>0) {
            Col newCol = Context.getsmlObjectFactory().createCol();
            newCol.setMin(start);
            newCol.setMax((start+size)-1);
            colList.add(i, newCol);
        }
    }

    public static void applySqrefRangeFromJsonArray(List<String> sqrefList, Object value)
    	throws JSONException {

    	if(value instanceof JSONArray) {
            final JSONArray ranges = (JSONArray)value;
            sqrefList.clear();
            for(int i=0; i<ranges.length();i++) {
                final JSONObject range = ranges.getJSONObject(i);
                final JSONArray start = range.getJSONArray("start");
                final JSONArray end = range.optJSONArray("end");
                final SmlUtils.CellRefRange cellRefRange = new SmlUtils.CellRefRange(
                		new SmlUtils.CellRef(start.getInt(0), start.getInt(1)),
                		end!=null ? new SmlUtils.CellRef(end.getInt(0), end.getInt(1)) : new SmlUtils.CellRef(start.getInt(0), start.getInt(1)));
                sqrefList.add(SmlUtils.getCellRefRange(cellRefRange));
            }
    	}
    	else {
    		sqrefList.clear();
    	}
    }

    public static JSONArray createJsonArrayFromCellRefRange(List<SmlUtils.CellRefRange> rangeList)
    	throws JSONException {

    	final JSONArray ranges = new JSONArray();
        for(SmlUtils.CellRefRange cellRefRange:rangeList) {
            final JSONObject range = new JSONObject(2);
            range.put("start", cellRefRange.getStart().getJSONArray());
            if((cellRefRange.getEnd().getColumn()!=cellRefRange.getStart().getColumn())||(cellRefRange.getEnd().getRow()!=cellRefRange.getStart().getRow())) {
                range.put("end", cellRefRange.getEnd().getJSONArray());
            }
            ranges.put(range);
        }
        return ranges;
    }

    public static List<SmlUtils.CellRefRange> createCellRefRangeListFromSqrefList(List<String> sqrefList) {
		final List<SmlUtils.CellRefRange> cellRefRangeList = new ArrayList<SmlUtils.CellRefRange>();
	    for(String sqRef:sqrefList) {
	        final SmlUtils.CellRefRange cellRefRange = SmlUtils.createCellRefRange(sqRef);
	        if(cellRefRange!=null) {
	            cellRefRangeList.add(cellRefRange);
	        }
	    }
	    return cellRefRangeList;
    }
}
