/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;

import org.apache.commons.lang.mutable.MutableInt;
import org.docx4j.dml.Theme;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.packages.OpcPackage;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.ThemePart;
import org.docx4j.openpackaging.parts.SpreadsheetML.SharedStrings;
import org.docx4j.openpackaging.parts.SpreadsheetML.Styles;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.CTCalcPr;
import org.xlsx4j.sml.CTDataValidations;
import org.xlsx4j.sml.CTDataValidations_legacy;
import org.xlsx4j.sml.CTExtension;
import org.xlsx4j.sml.CTExtensionList;
import org.xlsx4j.sml.CTFontSize;
import org.xlsx4j.sml.CTFonts;
import org.xlsx4j.sml.CTRElt;
import org.xlsx4j.sml.CTRst;
import org.xlsx4j.sml.CTSheetFormatPr;
import org.xlsx4j.sml.CTSst;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTXstringWhitespace;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.IDataValidations;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.SheetView;
import org.xlsx4j.sml.SheetViews;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;

import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.IResourceManager;
import com.openexchange.office.ooxml.xlsx.operations.ApplyOperationHelper;
import com.openexchange.office.ooxml.xlsx.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.xlsx.tools.AutoFilterHelper;
import com.openexchange.office.ooxml.xlsx.tools.ConditionalFormattings;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;
import com.openexchange.office.ooxml.xlsx.tools.TableHelper;
import com.openexchange.office.ooxml.xlsx.tools.ValidationUtils;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.doc.DocumentComplexity;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class OperationDocument extends com.openexchange.office.ooxml.OperationDocument {

    final private long FACTOR_FOR_REQUIRED_HEAPSPACE = 32;
    private SpreadsheetMLPackage opcPackage;
    protected CTStylesheet stylesheet = null;
    protected CTSst sharedStrings = null;
    private ApplyOperationHelper applyOperationHelper = null;
    private CreateOperationHelper _createOperationHelper = null;
    private Double defaultFontSize = null;

	private final static Logger log = LoggerFactory.getLogger(OperationDocument.class);

    public OperationDocument(Session session, ServiceLookup _services, InputStream _inputDocumentStream, IResourceManager _resourceManager, DocumentProperties _documentProperties)
        throws FilterException {

        super(session, _services, null, _resourceManager, _documentProperties);
        try {
        	final long maxAllowedXmlSizeMatchingComplexity = ConfigurationHelper.getIntegerOfficeConfigurationValue(getServiceLookup(), getSession(), "//spreadsheet/maxCells", 500000) * 100;
        	final long maxAllowedXmlSizeMatchingMemory = _documentProperties.optLong(DocumentProperties.PROP_MAX_MEMORY_USAGE, -1) / FACTOR_FOR_REQUIRED_HEAPSPACE;
        	inputDocumentStream = DocumentComplexity.checkDocumentComplexity(_inputDocumentStream, maxAllowedXmlSizeMatchingComplexity, maxAllowedXmlSizeMatchingMemory);
            opcPackage = (SpreadsheetMLPackage)SpreadsheetMLPackage.load(inputDocumentStream);

            // styles
            final Styles styles = getPackage().getWorkbookPart().getStyles();
            if(styles!=null)
                stylesheet = styles.getJaxbElement();

            // shared strings
            final SharedStrings _sharedStrings = getPackage().getWorkbookPart().getSharedStrings();
            if(_sharedStrings!=null)
                sharedStrings = _sharedStrings.getJaxbElement();

            final String initialSheetName = getDocumentProperties().optString(DocumentProperties.PROP_INITIAL_SHEETNAME, "Sheet1");
            final Workbook workbook = getPackage().getWorkbookPart().getJaxbElement();
            final Sheet sheet = workbook.getSheets().getSheet().get(0);
            if(sheet.getName().equals("empty_sheet")) {
                sheet.setName(initialSheetName);
            }
        }
        catch(Throwable e) {
            removeMemoryListener();
            throw getRethrowException(e);
        }
    }

    // the constructor without inputStream creates an empty document
    public OperationDocument(ServiceLookup _services, DocumentProperties _documentProperties)
        throws FilterException {

        super(null, _services, null, null, _documentProperties);
        try {
            opcPackage = SpreadsheetMLPackage.createPackage();

            final ThemePart themePart = getDefaultThemePart(new PartName("/xl/theme/theme1.xml"));
            getPackage().getWorkbookPart().addTargetPart(themePart);
            getPackage().createWorksheetPart(new PartName("/xl/worksheets/sheet1.xml"), getDocumentProperties().optString(DocumentProperties.PROP_INITIAL_SHEETNAME, "Sheet1"), 1);

            // applying the default column width...
            final Sheet sheet = SheetUtils.getSheet(opcPackage, 0);
            final RelationshipsPart relationshipPart = opcPackage.getWorkbookPart().getRelationshipsPart();
            final WorksheetPart worksheetPart = (WorksheetPart)relationshipPart.getPart(sheet.getId());
            final Worksheet worksheet = worksheetPart.getJaxbElement();
            CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
            if(sheetFormatPr==null) {
                sheetFormatPr = Context.getsmlObjectFactory().createCTSheetFormatPr();
                worksheet.setSheetFormatPr(sheetFormatPr);
                sheetFormatPr.setBaseColWidth(new Long(10L));
                sheetFormatPr.setDefaultRowHeight(14.5);
            }

            getStylesheet(true);
        }
        catch (Throwable e) {
            removeMemoryListener();
            throw getRethrowException(e);
        }
    }

    @Override
	public Part getContextPart() {
    	return contextPart!=null ? contextPart : getPackage().getWorkbookPart();
    }

    public CTStylesheet getStylesheet(boolean createIfMissing) {

        if(stylesheet==null&&createIfMissing) {
            try {
                final Styles styles = new Styles(new PartName("/xl/styles.xml"));
                stylesheet = createStylesheet();
                styles.setJaxbElement(stylesheet);
                getPackage().getWorkbookPart().addTargetPart(styles);
            }
            catch(final Exception e) {
                stylesheet = null;
            }
        }
        return stylesheet;
    }

    public CTSst getSharedStrings(boolean createIfMissing)
        throws Exception {

        if(sharedStrings==null&&createIfMissing) {
            final SharedStrings _sharedStrings = new SharedStrings(new PartName("/xl/sharedStrings.xml"));
            sharedStrings = Context.getsmlObjectFactory().createCTSst();
            _sharedStrings.setJaxbElement(sharedStrings);
            getPackage().getWorkbookPart().addTargetPart(_sharedStrings);
        }
        return sharedStrings;
    }

    // returns null if something goes wrong ...
    public String getSharedString(String index) {

        if(sharedStrings!=null) {
            final List<CTRst> si = sharedStrings.getSi();
            try  {
                final int i = Integer.parseInt(index);
                if(i>=0&&i<si.size()) {
                    final CTRst ctRst = si.get(i);
                    final CTXstringWhitespace ctxString = ctRst.getT();
                    if(ctxString!=null) {
                        return ctxString.getValue();
                    }
                    final List<CTRElt> ctREltList = ctRst.getR();
                    if(ctREltList!=null) {
                        final StringBuffer stringBuffer = new StringBuffer();
                        for(final CTRElt ctRElt:ctREltList) {
                            final String t = ctRElt.getT().getValue();
                            if(t!=null) {
                                stringBuffer.append(t);
                            }
                        }
                        return stringBuffer.toString();
                    }
                }
            }
            catch(final NumberFormatException e) {
                // TODO: what to do now ?
            }
        }
        return null;
    }

    public String applySharedString(String v)
    	throws InvalidFormatException {

    	if(sharedStrings==null) {
    		final SharedStrings sharedStringsPart = new SharedStrings();
    		sharedStrings = Context.getsmlObjectFactory().createCTSst();
    		sharedStringsPart.setJaxbElement(sharedStrings);
            getPackage().getWorkbookPart().addTargetPart(sharedStringsPart);
    	}
    	int i;
        final List<CTRst> si = sharedStrings.getSi();
        for(i=0;i<si.size();i++) {
        	final CTRst crst = si.get(i);
        	if(crst.getT()!=null&&crst.getT().getValue()!=null&&crst.getT().getValue().equals(v)) {
        		return String.valueOf(i);
        	}
        }
        final CTRst rst = Context.getsmlObjectFactory().createCTRst();
        final CTXstringWhitespace n = Context.getsmlObjectFactory().createCTXstringWhitespace();
        n.setValue(v);
        if(v.charAt(0)==' '||v.charAt(v.length()-1)==' ') {
            n.setSpace("preserve");
        }
        rst.setT(n);
        si.add(rst);
        return String.valueOf(si.size()-1);
    }

    public static List<IDataValidations> getDataValidations(Worksheet worksheet) {
        final List<IDataValidations> dataValidations = new ArrayList<IDataValidations>();

        final CTDataValidations validations = worksheet.getDataValidations();
        if(validations!=null) {
            dataValidations.add(validations);
        }
        final CTExtensionList extLst = worksheet.getExtLst();
        if(extLst!=null) {
            for(final CTExtension extension:extLst.getExt()) {
                final Object any = extension.getAny();
                if(any instanceof JAXBElement<?>) {
                    if(extension.getUri().equals("{CCE6A557-97BC-4b89-ADB6-D9C93CAAB3DF}")) {
                        final Object value = ((JAXBElement<?>)any).getValue();
                        if(value instanceof CTDataValidations_legacy) {
                            dataValidations.add((CTDataValidations_legacy)value);
                        }
                    }
                }
            }
        }
        return dataValidations;
    }

    //
    public String escapeString(String s) {
        if(s==null||s.length()==0)
            return s;
        final char char0 = s.charAt(0);
        return ((char0=='=')||(char0=='#')) ? "'" + s : s;
    }

    public ApplyOperationHelper getApplyOperationHelper() {
        return applyOperationHelper;
    }

    public CreateOperationHelper getCreateOperationHelper()
    	throws FilterException {
    	if(_createOperationHelper==null) {
    		_createOperationHelper = new CreateOperationHelper(this, new JSONArray());
    	}
    	_createOperationHelper.setOperationsArray(new JSONArray());
    	return _createOperationHelper;
    }

    @Override
    public void applyOperations(String applyOperations) throws FilterException {
    	if (applyOperations != null){
            int i = 0;
            JSONObject op = null;
            try {
                applyOperationHelper = new ApplyOperationHelper(this);
                final JSONArray aOperations = new JSONArray(new JSONTokener(applyOperations));
                for (i = 0; i < aOperations.length(); i++) {
                	successfulAppliedOperations = i;
                    op = (JSONObject) aOperations.get(i);
                    final String opName = op.getString("name");
                    switch(opName) {
	                    case "setCellContents": {
	                        applyOperationHelper.setCellContents(op.getInt("sheet"), op.getJSONArray("start"), op.getJSONArray("contents"));
	                        break;
	                    }
	                    case "fillCellRange": {
	                        applyOperationHelper.fillCellRange(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.opt("value"), op.opt("result"), op.optJSONObject("attrs"), op.optInt("shared", -1), op.optJSONArray("ref"));
	                        break;
	                    }
	                    case "clearCellRange": {
	                        applyOperationHelper.clearCellRange(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.optBoolean("remove", false));
	                        break;
	                    }
	                    case "setRowAttributes": {
	                        applyOperationHelper.setRowAttributes(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.getJSONObject("attrs"));
	                        break;
	                    }
	                    case "setColumnAttributes": {
	                        applyOperationHelper.setColumnAttributes(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.getJSONObject("attrs"));
	                        break;
	                    }
	                    case "insertRows": {
	                        applyOperationHelper.insertRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
	                        break;
	                    }
	                    case "deleteRows": {
	                        applyOperationHelper.deleteRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
	                        break;
	                    }
	                    case "insertColumns": {
	                        applyOperationHelper.insertColumns(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
	                        break;
	                    }
	                    case "deleteColumns": {
	                        applyOperationHelper.deleteColumns(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
	                        break;
	                    }
	                    case "insertSheet": {
	                        applyOperationHelper.insertSheet(op.getInt("sheet"), op.getString("sheetName"), op.optJSONObject("attrs"));
	                        break;
	                    }
	                    case "moveSheet": {
	                        applyOperationHelper.moveSheet(op.getInt("sheet"), op.getInt("to"));
	                        break;
	                    }
	                    case "deleteSheet": {
	                        applyOperationHelper.deleteSheet(op.getInt("sheet"));
	                        break;
	                    }
	                    case "copySheet": {
	                    	applyOperationHelper.copySheet(op.getInt("sheet"), op.getInt("to"), op.getString("sheetName"));
	                    	break;
	                    }
	                    case "setSheetName": {
	                        applyOperationHelper.setSheetName(op.getInt("sheet"), op.getString("sheetName"));
	                        break;
	                    }
	                    case "setSheetAttributes": {
	                        applyOperationHelper.setSheetAttributes(op.getInt("sheet"), op.getJSONObject("attrs"));
	                        break;
	                    }
	                    case "mergeCells": {
	                        applyOperationHelper.mergeCells(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.optBoolean("keepContent", false), op.optString("type", "merge"));
	                        break;
	                    }
	                    case "insertDrawing": {
	                        applyOperationHelper.insertDrawing(op.getJSONArray("start"), op.getString("type"), op.optJSONObject("attrs"));
	                        break;
	                    }
	                    case "deleteDrawing": {
	                        applyOperationHelper.deleteDrawing(op.getJSONArray("start"));
	                        break;
	                    }
	                    case "setDrawingAttributes": {
	                        applyOperationHelper.applyDrawingAttributes(op.getJSONArray("start"), op.getJSONObject("attrs"));
	                        break;
	                    }
	                    case "insertStyleSheet": {
	                        applyOperationHelper.insertStyleSheet(op.getString("type"), op.getString("styleId"), op.getString("styleName"), op.getJSONObject("attrs"), op.optString("parent", null), op.optBoolean("hidden", false), op.optInt("uiPriority", 0), op.optBoolean("default", false), op.optBoolean("auto", false));
	                        break;
	                    }
	                    case "insertChartDataSeries": {
	                        applyOperationHelper.insertChartDataSeries(op.getJSONArray("start"), op.getInt("series"), op.optJSONObject("attrs"));
	                        break;
	                    }
	                    case "setChartDataSeriesAttributes": {
	                        applyOperationHelper.setChartDataSeriesAttributes(op.getJSONArray("start"), op.getInt("series"), op.optJSONObject("attrs"));
	                        break;
	                    }
	                    case "deleteChartDataSeries": {
	                        applyOperationHelper.deleteChartDataSeries(op.getJSONArray("start"), op.getInt("series"));
	                        break;
	                    }
	                    case "insertName": {
	                        applyOperationHelper.insertName(op.optLong("sheet", -1), op.getString("exprName"), op.getString("formula"), null);
	                        break;
	                    }
	                    case "changeName": {
	                        applyOperationHelper.changeName(op.optLong("sheet", -1), op.getString("exprName"), op.optString("formula", null), op.optString("newName", null));
	                        break;
	                    }
	                    case "deleteName": {
	                        applyOperationHelper.deleteName(op.optLong("sheet", -1), op.getString("exprName"));
	                        break;
	                    }
	                    case "setChartAxisAttributes": {
	                        applyOperationHelper.setChartAxisAttributes(op.getJSONArray("start"), op.getString("axis"), op.optJSONObject("attrs"));
	                        break;
	                    }
	                    case "setChartGridlineAttributes": {
	                        applyOperationHelper.setChartGridlineAttributes(op.getJSONArray("start"), op.getString("axis"), op.optJSONObject("attrs"));
	                        break;
	                    }
	                    case "setChartTitleAttributes": {
	                        applyOperationHelper.setChartTitleAttributes(op.getJSONArray("start"), op.getString("axis"), op.optJSONObject("attrs"));
	                        break;
	                    }
	                    case "setChartLegendAttributes": {
	                        applyOperationHelper.setChartLegendAttributes(op.getJSONArray("start"), op.optJSONObject("attrs"));
	                        break;
	                    }
	                    case "insertValidation": {
	                    	ValidationUtils.insertValidation(this, op);
	                    	break;
	                    }
	                    case "changeValidation": {
	                    	ValidationUtils.changeValidation(this, op);
	                    	break;
	                    }
	                    case "deleteValidation": {
	                    	ValidationUtils.deleteValidation(this, op.getInt("sheet"), op.getInt("index"));
	                    	break;
	                    }
	                    case "setDocumentAttributes": {
	                    	applyOperationHelper.setDocumentAttributes(op.getJSONObject("attrs"));
	                    	break;
	                    }
	                    case "insertTable": {
	                    	final String tableName = op.optString("table", null);
	                    	if(tableName==null||tableName.isEmpty()) {
	                    		AutoFilterHelper.insertTable(applyOperationHelper, op.getInt("sheet"), null, op.getJSONArray("start"), op.getJSONArray("end"), op.optJSONObject("attrs"));
	                    	}
	                    	else {
	                    		TableHelper.changeOrInsertTable(applyOperationHelper, op.getInt("sheet"), tableName, op.getJSONArray("start"), op.getJSONArray("end"), op.optJSONObject("attrs"), true);
	                    	}
	                        break;
	                    }
	                    case "changeTable": {
	                    	final String tableName = op.optString("table", null);
	                    	if(tableName==null||tableName.isEmpty()) {
	                    		AutoFilterHelper.changeTable(applyOperationHelper, op.getInt("sheet"), null, op.optJSONArray("start"), op.optJSONArray("end"), op.optJSONObject("attrs"));
	                    	}
	                    	else {
	                    		TableHelper.changeOrInsertTable(applyOperationHelper, op.getInt("sheet"), tableName, op.optJSONArray("start"), op.optJSONArray("end"), op.optJSONObject("attrs"), false);
	                    	}
	                        break;
	                    }
	                    case "deleteTable": {
	                    	final String tableName = op.optString("table", null);
	                    	if(tableName==null||tableName.isEmpty()) {
	                    		AutoFilterHelper.deleteTable(applyOperationHelper, op.getInt("sheet"), null);
	                    	}
	                    	else {
	                    		TableHelper.deleteTable(applyOperationHelper, op.getInt("sheet"), tableName);
	                    	}
	                        break;
	                    }
	                    case "changeTableColumn": {
	                    	final String tableName = op.optString("table", null);
	                    	if(tableName==null||tableName.isEmpty()) {
	                    		AutoFilterHelper.changeTableColumn(applyOperationHelper, op.getInt("sheet"), null, op.getLong("col"), op.getJSONObject("attrs"));
	                    	}
	                    	else {
	                    		TableHelper.changeTableColumn(applyOperationHelper, op.getInt("sheet"), tableName, op.getLong("col"), op.getJSONObject("attrs"));
	                    	}
	                        break;
	                    }
	                    case "insertCondFormat": {
	                    	ConditionalFormattings.insertCondFormat(this, op.getInt("sheet"), op.getInt("index"), op.getJSONArray("ranges"));
	                    	break;
	                    }
	                    case "changeCondFormat": {
	                    	ConditionalFormattings.changeCondFormat(this, op.getInt("sheet"), op.getInt("index"), op.getJSONArray("ranges"));
	                    	break;
	                    }
	                    case "deleteCondFormat": {
	                    	ConditionalFormattings.deleteCondFormat(this, op.getInt("sheet"), op.getInt("index"));
	                    	break;
	                    }
	                    case "insertCondFormatRule": {
	                    	ConditionalFormattings.changeOrinsertCondFormatRule(this, true, op.getInt("sheet"), op.getInt("index"), op.getInt("ruleIndex"), op.optString("type", "formula"),
	                    		op.has("value1") ? op.get("value1") : "", op.optString("value2", ""), op.optInt("priority"), op.optBoolean("stop", false), op.has("attrs") ? op.getJSONObject("attrs") : new JSONObject());
	                    	break;
	                    }
	                    case "changeCondFormatRule": {
	                    	ConditionalFormattings.changeOrinsertCondFormatRule(this, false, op.getInt("sheet"), op.getInt("index"), op.getInt("ruleIndex"), op.optString("type", null),
	                    		op.opt("value1"), op.optString("value2", null), op.optInt("priority"), op.optBoolean("stop"), op.optJSONObject("attrs"));
	                    	break;
	                    }
	                    case "deleteCondFormatRule": {
	                    	ConditionalFormattings.deleteCondFormatRule(this, op.getInt("sheet"), op.getInt("index"), op.getInt("ruleIndex"));
	                    	break;
	                    }
	                    case "insertHyperlink": {
	                    	applyOperationHelper.insertHyperlink(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.getString("url"));
	                    	break;
	                    }
	                    case "deleteHyperlink": {
	                    	applyOperationHelper.deleteHyperlink(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"));
	                    	break;
	                    }
	                    case "createError": {
	                        throw new FilterException("createError operation detected: " + opName, ErrorCode.UNSUPPORTED_OPERATION_USED);
	                    }
	                    default: {
	                        logMessage("warn", "Ignoring unsupported operation: " + opName);
	                        break;
	                    }
                    }
                }
                successfulAppliedOperations=aOperations.length();
                final WorkbookPart workbookPart = getPackage().getWorkbookPart();
                final Workbook workbook = workbookPart.getJaxbElement();
                CTCalcPr calcPr = workbook.getCalcPr();
                if(calcPr==null) {
                    calcPr = Context.getsmlObjectFactory().createCTCalcPr();
                    workbook.setCalcPr(calcPr);
                }
                // our formula updates should be always up to date now, so it is no longer necessary to
                if(log.isInfoEnabled()) {
                	checkFormulaResults();
                }
                calcPr.setFullCalcOnLoad(true);

                // removing calcChain which might be invalid now
                final RelationshipsPart workbookRelationshipsPart = workbookPart.getRelationshipsPart();
                if(workbookRelationshipsPart!=null) {
                    final Relationship relationship = workbookRelationshipsPart.getRelationshipByType("http://schemas.openxmlformats.org/officeDocument/2006/relationships/calcChain");
                    if(relationship!=null) {
                        workbookRelationshipsPart.removeRelationship(relationship);
                    }
                }

                // removing tabSelected from other worksheets and finalizing tables
                final Sheets sheets = workbook.getSheets();
                final List<Sheet> sheetList = sheets.getSheet();
                final MutableInt uniqueTableId = new MutableInt(1000);
                for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
                    final Sheet sheet = sheetList.get(sheetIndex);
                    final Part part = getSheetPart(getPackage(), sheet);
                    final String sheetType = getSheetType(part);
                    if(sheetType.equals("worksheet")) {
                        final Worksheet worksheet = ((WorksheetPart)part).getJaxbElement();
                        TableHelper.finalizeTables(this, (WorksheetPart)part, uniqueTableId);
                        final SheetViews sheetViews = worksheet.getSheetViews();
                        if(sheetViews!=null) {
                            final List<SheetView> sheetViewList = sheetViews.getSheetView();
                            for(SheetView sheetView:sheetViewList) {
                                sheetView.setTabSelected(null);
                            }
                        }
                    }
                }
            }
            catch(final Exception e) {
                String message = e.getMessage();
                if(op!=null) {
                    message += ", operation:" + Integer.toString(i) + " " + op.toString();
                }
                throw new FilterException(message, e, ErrorCode.CRITICAL_ERROR);
            }
        }
    }

    /**
     * checkFormulaResults is logging the formula cells that do not have a result
     */
    private void checkFormulaResults() {
        final WorkbookPart workbookPart = getPackage().getWorkbookPart();
        final Workbook workbook = workbookPart.getJaxbElement();
        final RelationshipsPart workbookRelationshipsPart = workbookPart.getRelationshipsPart();
        for (final Sheet sheet:workbook.getSheets().getSheet()) {
            final Part part = workbookRelationshipsPart.getPart(sheet.getId());
            if(part instanceof WorksheetPart) {
                final WorksheetPart sheetPart = (WorksheetPart)part;
                if(sheetPart.isUnmarshalled()) {
	                final Worksheet worksheet = sheetPart.getJaxbElement();
	                final SheetData sheetData = worksheet.getSheetData();
	                if(sheetData!=null) {
	                    final Iterator<Row> rowIterator = sheetData.createRowIterator();
	                    while(rowIterator.hasNext()) {
	                        final Row row = rowIterator.next();
	                        final Iterator<Cell> cellIterator = row.createCellIterator();
	                        while(cellIterator.hasNext()) {
	                            final Cell cell = cellIterator.next();
	                            if(cell.getF()!=null&&cell.getV()==null) {
	                            	log.info("XLSX: Cell:" + cell.getColumn() + " formula without result!");
	                            }
	                        }
	                    }
	                }
                }
            }
        }
    }

    @Override
    public JSONObject getOperations()
        throws Exception {

        final CreateOperationHelper createOperationHelper = getCreateOperationHelper();
    	createOperationHelper.createOperations();

/*
        Writer writer = new FileWriter(new File("/tmp/jsonout.txt"));
        for(int i=0;i<createOperationHelper.getOperationsArray().length();i++) {
            createOperationHelper.getOperationsArray().getJSONObject(i).write(writer);
            writer.append(System.getProperty("line.separator"));
        }
        writer.close();
*/
        final JSONObject aOperations = new JSONObject();
        aOperations.put("operations", createOperationHelper.getOperationsArray());
        return aOperations;
    }

    public JSONObject getPreviewOperations()
        throws Exception {

        final CreateOperationHelper createOperationHelper = getCreateOperationHelper();
    	createOperationHelper.createPreviewOperations(getDocumentProperties());
        final JSONObject aOperations = new JSONObject();
        aOperations.put("operations", createOperationHelper.getOperationsArray());
        return aOperations;
    }

    public JSONObject getOutstandingOperations()
        throws Exception {

        final CreateOperationHelper createOperationHelper = getCreateOperationHelper();
        final int activeSheetIndex = (Integer)getDocumentProperties().get(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX);
        createOperationHelper.createOutstandingOperations(activeSheetIndex);
        final JSONObject aOperations = new JSONObject();
        aOperations.put("operations", createOperationHelper.getOperationsArray());
        return aOperations;
    }

    @Override
    public SpreadsheetMLPackage getPackage() {
        return opcPackage;
    }

    @Override
    public void setPackage(OpcPackage p) {
       opcPackage = (SpreadsheetMLPackage)p;
    }

    @Override
    public Theme getTheme()
    	throws FilterException {

    	try {
    		final PartName themePartName = new PartName("/xl/theme/theme1.xml");
	    	ThemePart themePart = (ThemePart)getPackage().getParts().get(themePartName);
	    	if(themePart==null) {
	    		themePart = getDefaultThemePart(themePartName);
	    	}
	    	return themePart.getJaxbElement();
    	}
    	catch(InvalidFormatException e) {
    		throw new FilterException("xlsx filter, could not create default theme", ErrorCode.CRITICAL_ERROR, e);
    	}
    }

    public double getDefaultFontSize() {

        if(defaultFontSize==null) {

            try {
                final CTStylesheet stylesheet = getStylesheet(false);
                if(stylesheet!=null) {
                    final CTFonts fonts = stylesheet.getFonts();
                    if(fonts!=null&&!fonts.getFont().isEmpty()) {
                        final CTFontSize fontSize = fonts.getFont().get(0).getSz();
                        if(fontSize!=null) {
                            defaultFontSize = fontSize.getVal();
                        }
                    }
                }
            }
            catch(final Exception e) {
                // nothing ...
            }
            if(defaultFontSize==null) {
                defaultFontSize = new Double(11.0);
            }
        }
        return defaultFontSize.doubleValue();
    }

    public Worksheet getWorksheet(int sheetIndex) {
        final Sheet sheet = SheetUtils.getSheet(getPackage(), sheetIndex);
        final RelationshipsPart relationshipPart = getPackage().getWorkbookPart().getRelationshipsPart();
        return ((WorksheetPart)relationshipPart.getPart(sheet.getId())).getJaxbElement();
    }

    public double getDefaultColumnWidth(Worksheet worksheet) {

        final CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
        if(sheetFormatPr==null) {
            return 8.7d;
        }
        final Double defaultColWidth = sheetFormatPr.getDefaultColWidth();
        if(defaultColWidth!=null) {
            return defaultColWidth.doubleValue();
        }
        return sheetFormatPr.getBaseColWidth() + 0.7d;
    }

    public double getDefaultRowHeight(Worksheet worksheet) {

        final CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
        return sheetFormatPr!=null ? sheetFormatPr.getDefaultRowHeight() : 14.5d;
    }

    public static Part getSheetPart(SpreadsheetMLPackage spreadsheetPackage, final Sheet sheet) {
        final RelationshipsPart relationshipPart = spreadsheetPackage.getWorkbookPart().getRelationshipsPart();
        return relationshipPart.getPart(sheet.getId());
    }

    public static String getSheetType(final Part part)
        throws FilterException {

        String sheetType = null;
        if(part.getRelationshipType().equals("http://schemas.openxmlformats.org/officeDocument/2006/relationships/worksheet")) {
            sheetType = "worksheet";
        }
        else if(part.getRelationshipType().equals("http://schemas.microsoft.com/office/2006/relationships/xlMacrosheet")) {
            sheetType = "macrosheet";
        }
        else if(part.getRelationshipType().equals("http://schemas.microsoft.com/office/2006/relationships/xlIntlMacrosheet")) {
            sheetType = "macrosheet";
        }
        else if(part.getRelationshipType().equals("http://schemas.openxmlformats.org/officeDocument/2006/relationships/dialogsheet")) {
            sheetType = "dialogsheet";
        }
        else if(part.getRelationshipType().equals("http://schemas.openxmlformats.org/officeDocument/2006/relationships/chartsheet")) {
            sheetType = "chartsheet";
        }
        if(sheetType==null) {
            throw new FilterException("xlsx import: unknown sheettype detected.", ErrorCode.CRITICAL_ERROR);
        }
        return sheetType;
    }

    private CTStylesheet createStylesheet() throws JAXBException {

        final String defaultStylesheet = "<styleSheet xmlns=\"http://schemas.openxmlformats.org/spreadsheetml/2006/main\"" +
        		"    xmlns:mc=\"http://schemas.openxmlformats.org/markup-compatibility/2006\"" +
        		"    mc:Ignorable=\"x14ac\"" +
        		"    xmlns:x14ac=\"http://schemas.microsoft.com/office/spreadsheetml/2009/9/ac\">" +
        		"    <fonts count=\"1\" x14ac:knownFonts=\"1\">" +
        		"        <font>" +
        		"            <sz val=\"11\" />" +
        		"            <color theme=\"1\" />" +
        		"            <name val=\"Arial\" />" +
        		"            <family val=\"2\" />" +
        		"            <scheme val=\"minor\" />" +
        		"        </font>" +
        		"    </fonts>" +
        		"    <fills count=\"2\">" +
        		"        <fill>" +
        		"            <patternFill patternType=\"none\" />" +
        		"        </fill>" +
        		"        <fill>" +
        		"            <patternFill patternType=\"gray125\" />" +
        		"        </fill>" +
        		"    </fills>" +
        		"    <borders count=\"1\">" +
        		"        <border>" +
        		"            <left />" +
        		"            <right />" +
        		"            <top />" +
        		"            <bottom />" +
        		"            <diagonal />" +
        		"        </border>" +
        		"    </borders>" +
        		"    <cellStyleXfs count=\"1\">" +
        		"        <xf numFmtId=\"0\" fontId=\"0\" fillId=\"0\" borderId=\"0\" />" +
        		"    </cellStyleXfs>" +
        		"    <cellXfs count=\"2\">" +
        		"        <xf numFmtId=\"0\" fontId=\"0\" fillId=\"0\" borderId=\"0\" xfId=\"0\" />" +
        		"    </cellXfs>" +
        		"    <cellStyles count=\"1\">" +
        		"        <cellStyle name=\"Standard\" xfId=\"0\" builtinId=\"0\" />" +
        		"    </cellStyles>" +
        		"    <dxfs count=\"0\" />" +
        		"    <tableStyles count=\"0\" defaultTableStyle=\"TableStyleMedium2\"" +
        		"        defaultPivotStyle=\"PivotStyleLight16\" />" +
        		"</styleSheet>";

        final java.util.HashMap<String, String> mappings = new java.util.HashMap<String, String>();
        final Object o = org.docx4j.XmlUtils.unmarshallFromTemplate(defaultStylesheet, mappings, getPackage().getWorkbookPart().getJAXBContext());
        return o instanceof JAXBElement ? (CTStylesheet)((JAXBElement<?>)o).getValue() : (CTStylesheet)o;
    }
}
