/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.math.BigInteger;
import java.util.List;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;

import org.docx4j.IndexedNodeList;
import org.docx4j.dml.BaseStyles;
import org.docx4j.dml.CTColorScheme;
import org.docx4j.dml.Theme;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.w15.CTPeople;
import org.docx4j.w15.CTPerson;
import org.docx4j.w15.CTPresenceInfo;
import org.docx4j.wml.Body;
import org.docx4j.wml.CTBorder;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.CTTrackChange;
import org.docx4j.wml.Color;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Document;
import org.docx4j.wml.STShd;
import org.docx4j.wml.STThemeColor;
import org.docx4j.wml.SectPr;
import org.docx4j.wml.SectPr.PgMar;
import org.docx4j.wml.SectPr.PgSz;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.tools.Commons;

public class Utils {

    public static String themeColorToString(Theme theme, STThemeColor themeColor) {
        String rgbValue = null;
        if(theme!=null) {
            BaseStyles themeElements = theme.getThemeElements();
            if(themeElements!=null) {
                CTColorScheme clrScheme = themeElements.getClrScheme();
                if(clrScheme!=null) {
                    org.docx4j.dml.CTColor ctColor = null;
                    switch(themeColor) {
                        case ACCENT_1 : ctColor = clrScheme.getAccent1(); break;
                        case ACCENT_2 : ctColor = clrScheme.getAccent2(); break;
                        case ACCENT_3 : ctColor = clrScheme.getAccent3(); break;
                        case ACCENT_4 : ctColor = clrScheme.getAccent4(); break;
                        case ACCENT_5 : ctColor = clrScheme.getAccent5(); break;
                        case ACCENT_6 : ctColor = clrScheme.getAccent6(); break;
                        case BACKGROUND_1 : ctColor = clrScheme.getLt1(); break;
                        case BACKGROUND_2 : ctColor = clrScheme.getLt2(); break;
                        case DARK_1 : ctColor = clrScheme.getDk1(); break;
                        case DARK_2 : ctColor = clrScheme.getDk2(); break;
                        case FOLLOWED_HYPERLINK : ctColor = clrScheme.getFolHlink(); break;
                        case HYPERLINK : ctColor = clrScheme.getHlink(); break;
                        case LIGHT_1 : ctColor = clrScheme.getLt1(); break;
                        case LIGHT_2 : ctColor = clrScheme.getLt2(); break;
                        case TEXT_1 : ctColor = clrScheme.getDk1(); break;
                        case TEXT_2 : ctColor = clrScheme.getDk2(); break;
                        case NONE : break;
                    }
                    if(ctColor!=null) {
                        rgbValue = Commons.ctColorToString(ctColor);
                    }
                }
            }
        }
        return rgbValue;
    }

    private static class TintShadeTransformation {
    	STThemeColor 	themeColor;
    	String 			themeTint;
    	String 			themeShade;
    	String 			Color;
    }

    public static void initBorderColorFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor, CTBorder border)
        throws FilterException, JSONException {

    	if(border==null||!jsonColor.has("type")) {
    		return;
    	}
    	final TintShadeTransformation tintShadeTransformation = createTintShadeTransformationFromJSONColor(operationDocument, jsonColor);
    	border.setThemeColor(tintShadeTransformation.themeColor);
        border.setThemeTint(tintShadeTransformation.themeTint);
        border.setThemeShade(tintShadeTransformation.themeShade);
        border.setColor(tintShadeTransformation.Color);
    }

    public static void initShdFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor, CTShd shd)
    	throws FilterException, JSONException {

    	if(shd==null||!jsonColor.has("type")) {
    		return;
    	}
    	final TintShadeTransformation tintShadeTransformation = createTintShadeTransformationFromJSONColor(operationDocument, jsonColor);
    	shd.setThemeFill(tintShadeTransformation.themeColor);
    	shd.setThemeFillTint(tintShadeTransformation.themeTint);
    	shd.setThemeFillShade(tintShadeTransformation.themeShade);
    	shd.setFill(tintShadeTransformation.Color);
        shd.setVal(STShd.CLEAR);
    }

    public static void initColorFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor, Color color)
        throws FilterException, JSONException {

    	if(color==null||!jsonColor.has("type")) {
    		return;
    	}
    	final TintShadeTransformation tintShadeTransformation = createTintShadeTransformationFromJSONColor(operationDocument, jsonColor);
    	color.setThemeColor(tintShadeTransformation.themeColor);
    	color.setThemeTint(tintShadeTransformation.themeTint);
    	color.setThemeShade(tintShadeTransformation.themeShade);
    	color.setVal(tintShadeTransformation.Color);
    }

    private static TintShadeTransformation createTintShadeTransformationFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor)
    	throws FilterException, JSONException {

    	final TintShadeTransformation tintShadeTransformation = new TintShadeTransformation();

		final String type = jsonColor.getString("type");
		boolean hasRgbValue = false;
        if(type.equals("rgb")&&jsonColor.has("value")) {
        	tintShadeTransformation.Color = jsonColor.getString("value");
        	hasRgbValue = true;
        } else if (type.equals("scheme")&&jsonColor.has("value")) {
            tintShadeTransformation.themeColor = STThemeColor.fromValue(jsonColor.getString("value"));
            tintShadeTransformation.Color = themeColorToString(operationDocument.getTheme(), tintShadeTransformation.themeColor);
            hasRgbValue = true;
        } else if (type.equals("auto")) {
            tintShadeTransformation.Color = "auto";
        }
        if (jsonColor.has("transformations")) {
        	final JSONArray transformations = jsonColor.getJSONArray("transformations");
            if ((transformations != null) && (transformations.length() > 0)) {
                final Object temp = transformations.opt(0);
                if (temp instanceof JSONObject) {
                    JSONObject transformation = (JSONObject)temp;
                    final String transformationType = transformation.optString("type");
                    final Double value = transformation.optDouble("value");
                    if ((transformationType != null) && (value != null)) {
                        if (transformationType.equals("tint")) {
                        	if(hasRgbValue) {
                        		final double[] hsl = Commons.rgbToHsl(tintShadeTransformation.Color);
                        		hsl[2] = (hsl[2]-1)*(value/100000) + 1;
                        		tintShadeTransformation.Color = Commons.hslToRgb(hsl);
                        	}
                        	if(tintShadeTransformation.themeColor!=null) {
                        		tintShadeTransformation.themeTint = Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv));
                        	}
                        }
                        else if (transformationType.equals("shade")) {
                        	if(hasRgbValue) {
                        		final double[] hsl = Commons.rgbToHsl(tintShadeTransformation.Color);
                        		hsl[2] = hsl[2] * (value/100000);
                        		tintShadeTransformation.Color = Commons.hslToRgb(hsl);
                        	}
                        	if(tintShadeTransformation.themeColor!=null) {
                        		tintShadeTransformation.themeShade = Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv));
                        	}
                        }
                    }
                }
            }
        }
	    return tintShadeTransformation;
    }

    public static boolean fillJsonColor(JSONObject jsonColor, STThemeColor themeColor, String STHexColor)
        throws JSONException {

        if(themeColor!=null&&themeColor.value()!=null&&themeColor.value().length()>0) {
            jsonColor.put("type", "scheme");
            jsonColor.put("value", themeColor.value());
        } else if (STHexColor!=null&&STHexColor.length()>0) {
            if(STHexColor.equals("auto"))
                jsonColor.put("type", "auto");
            else if(Commons.highlightColorToRgb.containsKey(STHexColor)){
                jsonColor.put("type", "rgb");
                jsonColor.put("value", Commons.highlightColorToRgb.get(STHexColor));
            }
            else {
                jsonColor.put("type", "rgb");
                jsonColor.put("value", STHexColor);
            }
        }
        return jsonColor.length()>0;
    }

    public static JSONObject createColor(STThemeColor themeColor, String STHexColor)
        throws JSONException {

        JSONObject jsonColor = new JSONObject();
        return fillJsonColor(jsonColor, themeColor, STHexColor) ? jsonColor : null;
    }

    public static JSONObject createFillColor(CTShd shd)
        throws JSONException {

        if (shd!=null) {
            JSONObject jsonColor = new JSONObject();
            JSONArray transformations = Utils.createFillTransformations(shd);
            if (transformations!=null) {
                jsonColor.put("transformations", transformations);
            }
            return fillJsonColor(jsonColor, shd.getThemeFill(), shd.getFill()) ? jsonColor : null;
        }
        return null;
    }

    public static JSONObject createColor(Color color)
        throws JSONException {

        if (color!=null) {
            JSONObject jsonColor = new JSONObject();
            JSONArray transformations = Utils.createTransformations(color);
            if (transformations!=null) {
                jsonColor.put("transformations", transformations);
            }
            return fillJsonColor(jsonColor, color.getThemeColor(), color.getVal()) ? jsonColor : null;
        }
        return null;
    }

    private static JSONArray createFillTransformations(CTShd shd)
        throws JSONException {

        JSONArray jsonColorTransformations = null;
        if (shd!=null) {
            String shade = shd.getThemeFillShade();
            if (shade!=null) {
                jsonColorTransformations = new JSONArray();
                jsonColorTransformations.put(Utils.createShadeTransformation(shade));
            }
            String tint = shd.getThemeFillTint();
            if (tint!=null) {
                jsonColorTransformations = (jsonColorTransformations == null) ? new JSONArray() : jsonColorTransformations;
                jsonColorTransformations.put(Utils.createTintTransformation(tint));
            }
        }
        return jsonColorTransformations;
    }

    private static JSONArray createTransformations(Color color)
        throws JSONException {

        JSONArray jsonColorTransformations = null;
        if (color!=null) {
            String shade = color.getThemeShade();
            if (shade!=null) {
                jsonColorTransformations = new JSONArray();
                jsonColorTransformations.put(Utils.createShadeTransformation(shade));
            }
            String tint = color.getThemeTint();
            if (tint!=null) {
                jsonColorTransformations = (jsonColorTransformations == null) ? new JSONArray() : jsonColorTransformations;
                jsonColorTransformations.put(Utils.createTintTransformation(tint));
            }
        }
        return jsonColorTransformations;
    }

    private static JSONObject createShadeTransformation(String shade)
        throws JSONException {

        JSONObject shadeTransform = null;
        if (shade!=null) {
            shadeTransform = new JSONObject();
            shadeTransform.put("type", "shade");
            shadeTransform.put("value", Math.round(Integer.parseInt(shade, 16) * Commons.themeTransformFactor));
        }
        return shadeTransform;
    }

    private static JSONObject createTintTransformation(String tint)
        throws JSONException {

        JSONObject tintTransform = null;
        if (tint!=null) {
            tintTransform = new JSONObject();
            tintTransform.put("type", "tint");
            tintTransform.put("value", Math.round(Integer.parseInt(tint, 16) * Commons.themeTransformFactor));
        }
        return tintTransform;
    }

    public static SectPr getDocumentProperties(MainDocumentPart mainDocPart, boolean createIfMissing) {

        SectPr sectPr = null;
        if(mainDocPart!=null) {
            final Document doc = mainDocPart.getJaxbElement();
            if(doc!=null) {
                final Body body = doc.getBody();
                if(body!=null) {
                    sectPr = body.getSectPr();
                }
                if(sectPr==null&&createIfMissing) {
                	sectPr = Context.getWmlObjectFactory().createSectPr();
                	sectPr.setParent(body);
                	body.setSectPr(sectPr);
                }
            }
        }
        return sectPr;
    }

    public static long getPageWidth(MainDocumentPart mainDocPart, long defaultWidth) {
    	final SectPr sectPr = getDocumentProperties(mainDocPart, false);
    	if(sectPr!=null) {
	        PgMar pgMar = sectPr.getPgMar();
	        PgSz pgSz = sectPr.getPgSz();
	        if(pgSz!=null&&pgMar!=null) {
	            long sz = pgSz.getW().longValue();
	            sz -= pgMar.getLeft().longValue();
	            sz -= pgMar.getRight().longValue();
	            return sz * 2540 / 1440;
	        }
    	}
    	return defaultWidth;
    }

    public static long mapTwipTo100THMM(BigInteger bigInt) {
        return bigInt!=null?((bigInt.longValue() * 2540) / 72) / 20:0;
    }

    public static long mapTwipTo100THMM(long longValue) {
        return ((longValue * 2540) / 72) / 20;
    }

    public static BigInteger map100THMMToTwip(long longValue) {
        return BigInteger.valueOf( longValue * 1440 /2540 );
    }

    public static BigInteger map100THMMToTwip(double doubleValue) {
        return BigInteger.valueOf( (long)(doubleValue * 1440. /2540.) );
    }

    public static boolean equals(final String s1, final String s2) {
        return (s1==null&&s2==null)||((s1!=null)&&s1.equals(s2));
    }

    public static IndexedNodeList<Object> getContent(Object o) {
    	IndexedNodeList<Object> r = null;
        if(o instanceof ContentAccessor) {
        	final List<Object> content = ((ContentAccessor)o).getContent();
        	if(content instanceof IndexedNodeList) {
        		r = (IndexedNodeList<Object>)content;
        	}
        }
        return r;
    }

    public static JSONObject createJSONFromTrackInfo(com.openexchange.office.ooxml.docx.OperationDocument operationDocument, CTTrackChange changeTrack)
    	throws JSONException {

		final JSONObject trackInfoDesc = new JSONObject(2);
		if(changeTrack!=null) {
			final String author = changeTrack.getAuthor();
			if(author!=null) {
				trackInfoDesc.put("author", changeTrack.getAuthor());
				operationDocument.addChangeTrackAuthor(changeTrack.getAuthor(), changeTrack.getDate());
	
				final CTPeople ctPeople = operationDocument.getPeopleList(false);
				if(ctPeople!=null) {
					final List<CTPerson> peopleList = ctPeople.getPerson();
					if(peopleList!=null) {
						for(CTPerson person:peopleList) {
							if(person.getAuthor()!=null&&person.getAuthor().equals(author)) {
								final CTPresenceInfo presenceInfo = person.getPresenceInfo();
								if(presenceInfo!=null) {
									final String providerId = presenceInfo.getProviderId();
									final String userId = presenceInfo.getUserId();
									if(providerId!=null&&userId!=null) {
										if(providerId.equals("None")&&userId.startsWith("OXText_")) {
											trackInfoDesc.put("uid", userId.substring(7));
										}
									}
								}
							}
						}
					}
				}
			}
			if(changeTrack.getDate()!=null) {
				trackInfoDesc.put("date", changeTrack.getDate().toXMLFormat());
			}
		}
		return trackInfoDesc;
    }

    public static void applyTrackInfoFromJSON(com.openexchange.office.ooxml.docx.OperationDocument operationDocument, JSONObject json, CTTrackChange changeTrack) {
    	if(json!=null) {
    		applyTrackInfoFromJSON(operationDocument, json.opt("author"), json.opt("date"), json.opt("uid"), changeTrack);
    	}
    }

    public static void applyTrackInfoFromJSON(com.openexchange.office.ooxml.docx.OperationDocument operationDocument, Object author, Object date, Object uid, CTTrackChange changeTrack) {

		final org.docx4j.w15.ObjectFactory peopleObjectFactory = new org.docx4j.w15.ObjectFactory();

    	try {
    		String uidAuthor = changeTrack.getAuthor();
			if(author!=null) {
    			if(author instanceof String) {
    				uidAuthor = (String)author;
    				changeTrack.setAuthor(uidAuthor);
    			}
    			else {
    				changeTrack.setAuthor(null);
    			}
			}
			if(date!=null) {
    			if(date instanceof String) {
    				final XMLGregorianCalendar xgc=DatatypeFactory.newInstance().newXMLGregorianCalendar((String)date);
    				changeTrack.setDate(xgc);
    			}
    			else {
    				changeTrack.setDate(null);
    			}
			}
			if(uid!=null&&uidAuthor!=null&&!uidAuthor.isEmpty()) {
				if(uid instanceof String) {
					final CTPeople ctPeople = operationDocument.getPeopleList(true);
					if(ctPeople!=null) {
    					final List<CTPerson> peopleList = ctPeople.getPerson();
    					if(peopleList!=null) {
    						int i=0;
    						for(;i<peopleList.size();i++) {
    							final CTPerson person = peopleList.get(i);
    							if(person.getAuthor()!=null&&person.getAuthor().equals(uidAuthor)) {
    								final CTPresenceInfo presenceInfo = person.getPresenceInfo();
    								if(presenceInfo!=null) {
	    								final String providerId = presenceInfo.getProviderId();
	    								final String userId = presenceInfo.getUserId();
	    								if(providerId.equals("None")&&userId.startsWith("OXText_")) {
	    									if(userId.substring(7).equals((String)uid)) {
		    									break;
		    								}
	    								}
    								}
    							}
    						}
    						if(i==peopleList.size()) {
	    						final CTPerson newPerson = peopleObjectFactory.createCTPerson();
	    						newPerson.setParent(ctPeople);
	    						newPerson.setAuthor(uidAuthor);
	    						peopleList.add(newPerson);
								final CTPresenceInfo presenceInfo = peopleObjectFactory.createCTPresenceInfo();
								presenceInfo.setParent(newPerson);
								presenceInfo.setProviderId("None");
								presenceInfo.setUserId("OXText_"+(String)uid);
								newPerson.setPresenceInfo(presenceInfo);
    						}
						}
    				}
				}
				else {
					// TODO: removing uid for this author
				}
			}
    	}
    	catch(DatatypeConfigurationException e) {
    		// ups
    	};
	}
}
