/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.math.BigDecimal;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;

import org.docx4j.IndexedNode;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.relationships.Relationship;
import org.docx4j.vml.CTArc;
import org.docx4j.vml.CTCurve;
import org.docx4j.vml.CTFill;
import org.docx4j.vml.CTGroup;
import org.docx4j.vml.CTImage;
import org.docx4j.vml.CTImageData;
import org.docx4j.vml.CTLine;
import org.docx4j.vml.CTOval;
import org.docx4j.vml.CTPolyLine;
import org.docx4j.vml.CTRect;
import org.docx4j.vml.CTRoundRect;
import org.docx4j.vml.CTShape;
import org.docx4j.vml.CTShapetype;
import org.docx4j.vml.CTStroke;
import org.docx4j.vml.CTTextbox;
import org.docx4j.vml.VmlCore;
import org.docx4j.vml.VmlShapeCore;
import org.docx4j.vml.officedrawing.STHrAlign;
import org.docx4j.vml.officedrawing.STTrueFalse;
import org.docx4j.vml.wordprocessingDrawing.CTWrap;
import org.docx4j.vml.wordprocessingDrawing.STWrapType;
import org.docx4j.wml.CTTxbxContent;
import org.docx4j.wml.Color;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.docx.tools.Component.IShape;
import com.openexchange.office.ooxml.docx.tools.Component.ShapeType;
import com.openexchange.office.ooxml.docx.tools.Component.Type;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.tools.doc.DocumentImageHelper;

public class PictVML {

    public static void createImageAttrs(CreateOperationHelper createOperationHelper, VML_Shape_base vmlShapeImpl, JSONObject attrs)
    	throws JSONException {

    	final CTImageData imageData = vmlShapeImpl.getImageData(false);
    	if(imageData!=null) {
    		final String rId = imageData.getId();
    		if(rId!=null) {
            	final Part contextPart = createOperationHelper.getOperationDocument().getContextPart();
    			final String imageUrl = Commons.getUrl(contextPart, rId);
    			if(!imageUrl.isEmpty()) {
                	JSONObject imageAttrs = attrs.optJSONObject("image");
                	if(imageAttrs==null) {
                		imageAttrs = new JSONObject(1);
                		attrs.put("image", imageAttrs);
                	}
                	imageAttrs.put("imageUrl", imageUrl);
    			}
    		}
    	}
    }

    public static class VMLShape {

        private final Map<String, VML_Shape_type> shapeTypes;
        private VML_Shape_base shape;

        VMLShape(List<Object> content) {

            shapeTypes = new HashMap<String, VML_Shape_type>();
            shape = null;
            if(content!=null) {
                for(Object o : content) {
                    if(o instanceof JAXBElement)
                        o = ((JAXBElement<?>)o).getValue();

                    VML_Shape_base vml_Shape_base = createPictVMLShape(this, o, null);
                    if(vml_Shape_base instanceof VML_Shape_type) {
                        String Id = ((VML_Shape_type)vml_Shape_base).getId();
                        if(Id!=null&&Id.length()>0)
                            getShapeTypes().put(Id, (VML_Shape_type)vml_Shape_base);
                    } else if(vml_Shape_base!=null) {
                        if(shape!=null)
                            System.out.println("OOXML - VMLShape, not assuming group shapes...");
                        else
                            shape = vml_Shape_base;
                    }
                }
            }
        }
        Map<String, VML_Shape_type> getShapeTypes() {
            return shapeTypes;
        }
        VML_Shape_base getVMLShapeImpl() {
            return shape;
        }
    }

    protected static abstract class VML_Shape_base implements IShape {

        final VmlCore  o;
        final VML_Shape_group parent;
        final VML_Shape_type referenceShape;
        final List<JAXBElement<?>> shapeElements;

        protected VML_Shape_base(VmlCore _o, List<JAXBElement<?>> _shapeElements, VML_Shape_type _referenceShape, VML_Shape_group _parent) {
            o = _o;
            parent = _parent;
            referenceShape = _referenceShape;
            shapeElements = _shapeElements;
        }
        public VmlCore getObject() {
            return o;
        }

        // returns the wrap mode, can be null (not a live object)
        public CTWrap getCTWrap(boolean createIfNotAvailable) {
            CTWrap ctWrap = null;
            final JAXBElement<?> jaxbElement = Commons.findElement(shapeElements, "wrap");
            if(jaxbElement!=null) {
                ctWrap = (CTWrap)jaxbElement.getValue();
            }
            else if(createIfNotAvailable) {
            	final org.docx4j.vml.wordprocessingDrawing.ObjectFactory vmlWordprocessingDrawingObjectFactory
            		= Context.getVmlWordprocessingDrawingObjectFactory();
                ctWrap = vmlWordprocessingDrawingObjectFactory.createCTWrap();
                shapeElements.add(vmlWordprocessingDrawingObjectFactory.createWrap(ctWrap));
            }
            return ctWrap;
        }

        public CTFill getFill(boolean createIfNotAvailable) {
        	CTFill fill = null;
        	JAXBElement<?> jaxbElement = Commons.findElement(shapeElements, "fill");
        	if(jaxbElement!=null) {
        		fill = (CTFill)jaxbElement.getValue();
        		return fill;
        	}
        	if(createIfNotAvailable) {
        		final org.docx4j.vml.ObjectFactory vmlObjectFactory = new org.docx4j.vml.ObjectFactory();
        		fill = vmlObjectFactory.createCTFill();
        		shapeElements.add(vmlObjectFactory.createFill(fill));
        	}
        	return fill;
        }

        public CTStroke getStroke(boolean createIfNotAvailable) {
        	CTStroke stroke = null;
        	JAXBElement<?> jaxbElement = Commons.findElement(shapeElements, "stroke");
        	if(jaxbElement!=null) {
        		return (CTStroke)jaxbElement.getValue();
        	}
        	if(createIfNotAvailable) {
        		final org.docx4j.vml.ObjectFactory vmlObjectFactory = new org.docx4j.vml.ObjectFactory();
        		stroke = vmlObjectFactory.createCTStroke();
        		shapeElements.add(vmlObjectFactory.createStroke(stroke));
        	}
        	return stroke;
        }

        public CTImageData getImageData(boolean createIfNotAvailable) {
        	CTImageData imageData = null;
        	final JAXBElement<?> jaxbElement = Commons.findElement(shapeElements, "imagedata");
        	if(jaxbElement!=null) {
        		imageData = (CTImageData)jaxbElement.getValue();
        	}
        	else if(createIfNotAvailable) {
        		final org.docx4j.vml.ObjectFactory vmlObjectFactory = new org.docx4j.vml.ObjectFactory();
        		imageData = vmlObjectFactory.createCTImageData();
        		shapeElements.add(vmlObjectFactory.createImagedata(imageData));
        	}
        	return imageData;
        }

        public CTTextbox getTextbox(boolean createIfNotAvailable) {
            final JAXBElement<?> jaxbElement = Commons.findElement(shapeElements, "textbox");
            if(jaxbElement!=null) {
                return (CTTextbox)jaxbElement.getValue();
            }
            else if(createIfNotAvailable) {
                final org.docx4j.vml.ObjectFactory vmlObjectFactory = new org.docx4j.vml.ObjectFactory();
                final CTTextbox textbox = vmlObjectFactory.createCTTextbox();
                textbox.setParent(getObject());
                Commons.insertElement(shapeElements, "textbox", vmlObjectFactory.createTextbox(textbox));
                return textbox;
            }
            return null;
        }

        public ShapeType getType() {
        	if(o instanceof VmlShapeCore) {
        		return ((VmlShapeCore)o).getOle()!=null?ShapeType.OLE:ShapeType.SHAPE;
        	}
        	return ShapeType.SHAPE;
        }

		public void postFixNonConformance(AtomicInteger id) {
			//
		}

    	public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
    	    throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

            final ShapeType shapeType = getType();

    	    // fill attributes
            final JSONObject drawingAttrs = attrs.optJSONObject("drawing");
            if(drawingAttrs!=null) {
                VMLStyle vmlStyle = new VMLStyle(this);

                int marginLeft = 0;
                int marginTop = 0;
                int marginRight = 0;
                int marginBottom = 0;
                if(drawingAttrs.hasAndNotNull("marginLeft")) {
                    marginLeft = drawingAttrs.getInt("marginLeft");
                }
                if(drawingAttrs.hasAndNotNull("marginTop")) {
                    marginTop = drawingAttrs.getInt("marginTop");
                }
                if(drawingAttrs.hasAndNotNull("marginRight")) {
                    marginRight = drawingAttrs.getInt("marginRight");
                }
                if(drawingAttrs.hasAndNotNull("marginBottom")) {
                    marginBottom = drawingAttrs.getInt("marginBottom");
                }

                boolean isInline = true;
                final String pos = vmlStyle.get("position");
                if(pos!=null)
                    isInline = !pos.equals("absolute");
                Object inline = drawingAttrs.opt("inline");
                if(inline!=null) {
                    if(inline instanceof Boolean) {
                        isInline = ((Boolean)inline).booleanValue();
                    }
                    else {
                        isInline = true;
                    }
                }
                if (isInline) {
                    vmlStyle.set("position", "static");
                    vmlStyle.remove("mso-position-horizontal");
                    vmlStyle.remove("mso-position-horizontal-relative");
                    vmlStyle.remove("mso-position-vertical");
                    vmlStyle.remove("mso-position-vertical-relative");
                } else {
                    vmlStyle.set("position",  "absolute");
                    if(drawingAttrs.hasAndNotNull("anchorHorBase")) {
                        vmlStyle.setHorzBase(drawingAttrs.getString("anchorHorBase"));
                    }
                    String anchorHorAlign = null;
                    if(drawingAttrs.hasAndNotNull("anchorHorAlign")) {
                        anchorHorAlign = drawingAttrs.getString("anchorHorAlign");
                        vmlStyle.setHorzAlign(anchorHorAlign);
                    }

                    if(drawingAttrs.hasAndNotNull("anchorVertBase")) {
                        vmlStyle.setVertBase(drawingAttrs.getString("anchorVertBase"));
                    }
                    String anchorVertAlign = null;
                    if(drawingAttrs.hasAndNotNull("anchorVertAlign")) {
                        anchorVertAlign = drawingAttrs.getString("anchorVertAlign");
                        vmlStyle.setVertAlign(anchorVertAlign);
                    }

                    if(anchorHorAlign!=null&&anchorHorAlign.equals("offset")) {
                        if(drawingAttrs.hasAndNotNull("anchorHorOffset"))
                            marginLeft += drawingAttrs.getInt("anchorHorOffset");
                    }
                    if(anchorVertAlign!=null&&anchorVertAlign.equals("offset")) {
                        if(drawingAttrs.hasAndNotNull("anchorVertOffset"))
                            marginTop += drawingAttrs.getInt("anchorVertOffset");
                    }
                    if(drawingAttrs.hasAndNotNull("textWrapMode")) {
                        CTWrap ctWrap = getCTWrap(true);
                        String wrapMode = drawingAttrs.getString("textWrapMode");
                        if(wrapMode.equals("square"))
                            ctWrap.setType(STWrapType.SQUARE);
                        else if(wrapMode.equals("tight"))
                            ctWrap.setType(STWrapType.TIGHT);
                        else if(wrapMode.equals("topAndBottom"))
                            ctWrap.setType(STWrapType.TOP_AND_BOTTOM);
                        else if(wrapMode.equals("through"))
                            ctWrap.setType(STWrapType.THROUGH);
                        else
                            ctWrap.setType(STWrapType.NONE);
                    }
                }

                if(marginLeft!=0)
                    vmlStyle.setLength("margin-left", marginLeft);
                else
                    vmlStyle.remove("margin-left");
                if(marginTop!=0)
                    vmlStyle.setLength("margin-top", marginTop);
                else
                    vmlStyle.remove("margin-top");
                if(marginRight!=0)
                    vmlStyle.setLength("margin-right", marginRight);
                else
                    vmlStyle.remove("margin-right");
                if(marginBottom!=0)
                    vmlStyle.setLength("margin-bottom", marginBottom);
                else
                    vmlStyle.remove("margin-bottom");

                if(drawingAttrs.has("width")||drawingAttrs.has("height")) {
                    int width = drawingAttrs.optInt("width", 0);
                    int height = drawingAttrs.optInt("height", 0);
                    if(parent!=null) {
                        vmlStyle.set("width", Integer.toString(width));
                        vmlStyle.set("height", Integer.toString(height));
                    }
                    else {
                        // root object with anchor
                        vmlStyle.setLength("width", width);
                        vmlStyle.setLength("height", height);
                    }
                    if((shapeType==ShapeType.GROUP)&&(!drawingAttrs.has("coordWidth")||!drawingAttrs.has("coordHeight"))) {
                        o.setCoordsize(Integer.toString(width) + "," + Integer.toString(height));
                    }
                }
                if(drawingAttrs.has("coordWidth")||drawingAttrs.has("coordHeight")) {
                    o.setCoordsize(Integer.toString(drawingAttrs.optInt("coordWidth", 0)) + "," + Integer.toString(drawingAttrs.optInt("coordHeight", 0)));
                }
                if(parent!=null&&(drawingAttrs.has("left")||drawingAttrs.has("top"))) {
                    vmlStyle.set("left", Integer.toString(drawingAttrs.optInt("left", 0)));
                    vmlStyle.set("top", Integer.toString(drawingAttrs.optInt("top", 0)));
                }
                Iterator<String> keys = drawingAttrs.keys();
                while(keys.hasNext()) {
                    String attr = keys.next();
                    Object value = drawingAttrs.get(attr);
                    if(shapeType == ShapeType.HORIZONTAL_LINE && (attr.equals("anchorHorAlign") || attr.equals("width"))) {
                        if(o instanceof CTRect) {
                            CTRect rect = (CTRect)o;
                            try{
                                if(attr.equals("anchorHorAlign") ){
                                    STHrAlign hrAlign = STHrAlign.fromValue((String)value);
                                    rect.setHralign(hrAlign);
                                } else {
                                    rect.setHrpct(new Float(0.));
                                }
                            } catch( IllegalArgumentException i ){
                                //
                            }
                        }
                    }
                }
            }

    		// fill attributes
    	    final JSONObject fillAttrs = attrs.optJSONObject("fill");
    	    if(fillAttrs!=null) {
                final CTFill fill = getFill(false);
    	        final String type = fillAttrs.optString("type", null);
    	        if(type!=null) {
    	            boolean filled = false;
    	            if(type.equals("solid")) {
    	                filled = true;
    	            }
    	            final org.docx4j.vml.STTrueFalse trueFalse = filled ? org.docx4j.vml.STTrueFalse.T : org.docx4j.vml.STTrueFalse.F;
    	            if(fill!=null) {
    	                fill.setOn(trueFalse);
    	            }
    	            else {
    	                o.setFilled(trueFalse);
    	            }
    	        }
    	        final JSONObject color = fillAttrs.optJSONObject("color");
    	        if(color instanceof JSONObject) {
    	            String newColor = null;
    	            final String fallBack = color.optString("fallbackValue", null);
    	            if(fallBack instanceof String) {
    	                newColor = "#" + fallBack.toUpperCase();
    	            }
    	            else if(color.optString("type", "").equals("rgb")) {
    	                final String rgb = color.optString("value");
    	                if(rgb!=null) {
    	                    newColor = "#" + rgb.toUpperCase();
    	                }
    	            }
    	            if(newColor!=null) {
                        if(fill!=null) {
                            fill.setColor(newColor);
                        }
                        else {
                            o.setFillcolor(newColor);
                        }
    	            }
    	        }
    	    }

    		// line attributes
    		final JSONObject lineAttrs = attrs.optJSONObject("line");
    		if(lineAttrs!=null) {
                CTStroke stroke = getStroke(false);
                final String type = lineAttrs.optString("type", null);
                if(type!=null) {
                    boolean stroked = false;
                    if(type.equals("solid")) {
                        stroked = true;
                    }
                    final org.docx4j.vml.STTrueFalse trueFalse = stroked ? org.docx4j.vml.STTrueFalse.T : org.docx4j.vml.STTrueFalse.F;
                    if(stroke!=null) {
                        stroke.setOn(trueFalse);
                    }
                    else if(o instanceof VmlShapeCore) {
                        ((VmlShapeCore)o).setStroked(trueFalse);
                    }
                }
                final JSONObject color = lineAttrs.optJSONObject("color");
                if(color instanceof JSONObject) {
                    String newColor = null;
                    final String fallBack = color.optString("fallbackValue", null);
                    if(fallBack instanceof String) {
                        newColor = "#" + fallBack.toUpperCase();
                    }
                    else if(color.optString("type", "").equals("rgb")) {
                        final String rgb = color.optString("value");
                        if(rgb!=null) {
                            newColor = "#" + rgb.toUpperCase();
                        }
                    }
                    if(newColor!=null) {
                        if(stroke!=null) {
                            stroke.setColor(newColor);
                        }
                        else if(o instanceof VmlShapeCore) {
                            ((VmlShapeCore)o).setStrokecolor(newColor);
                        }
                    }
                }
                final Object weight = lineAttrs.opt("width");
                if(weight!=null&&weight instanceof Number) {
                    final long w = ((Number)weight).longValue();
                    String newWidth = VMLStyle.getVMLLength(w);
                    if(stroke!=null) {
                        stroke.setWeight(newWidth);
                    }
                    else if(o instanceof VmlShapeCore) {
                        ((VmlShapeCore)o).setStrokeweight(newWidth);
                    }
                }
                // dash style
                final Object style = lineAttrs.opt("style");
                if(style!=null) {
                    if(stroke==null) {
                        stroke = getStroke(true);
                        if(o instanceof VmlShapeCore) {
                            stroke.setOn(((VmlShapeCore)o).getStroked());
                        }
                    }
                    if(style instanceof String) {
                        if(((String)style).equals("solid")) {
                            stroke.setDashstyle("Solid");
                        }
                        else if(((String)style).equals("dotted")) {
                            stroke.setDashstyle("Dot");
                        }
                        else if(((String)style).equals("dashed")) {
                            stroke.setDashstyle("Dash");
                        }
                        else if(((String)style).equals("dashDot")) {
                            stroke.setDashstyle("LongDashDot");
                        }
                        else if(((String)style).equals("dashDotDot")) {
                            stroke.setDashstyle("LongDashDotDot");
                        }
                    }
                    else {
                        stroke.setDashstyle(null);
                    }
                }
    		}

    		// line attributes
            final JSONObject shapeAttrs = attrs.optJSONObject("shape");
            if(shapeAttrs!=null) {
                final Object paddingLeft = shapeAttrs.opt("paddingLeft");
                final Object paddingTop = shapeAttrs.opt("paddingTop");
                final Object paddingRight = shapeAttrs.opt("paddingRight");
                final Object paddingBottom = shapeAttrs.opt("paddingBottom");
                CTTextbox textbox = null;
                if(paddingLeft!=null||paddingTop!=null||paddingRight!=null||paddingBottom!=null) {
                    if(textbox==null) {
                        textbox = getTextbox(true);
                    }
                    final String[] paddingArray = new String[4];
                    final String oldInset = textbox.getInset();
                    if(oldInset!=null) {
                        final String[] pArray = oldInset.split(",");
                        for (int i=0; i<pArray.length&&i<4; i++) {
                            final String pEntry = pArray[i];
                            if(pEntry!=null&&pArray.length!=0) {
                                paddingArray[i] = pArray[i];
                            }
                        }
                    }
                    if(paddingLeft!=null) {
                        if(paddingLeft instanceof Number) {
                            final int pLeft = ((Number)paddingLeft).intValue();
                            paddingArray[0] = pLeft==250?null:VMLStyle.getVMLLength(pLeft);
                        }
                        else {
                            paddingArray[0] = null;
                        }
                    }
                    if(paddingTop!=null) {
                        if(paddingTop instanceof Number) {
                            final int pTop = ((Number)paddingTop).intValue();
                            paddingArray[1] = pTop==125?null:VMLStyle.getVMLLength(pTop);
                        }
                        else {
                            paddingArray[1] = null;
                        }
                    }
                    if(paddingRight!=null) {
                        if(paddingRight instanceof Number) {
                            final int pRight = ((Number)paddingRight).intValue();
                            paddingArray[2] = pRight==250?null:VMLStyle.getVMLLength(pRight);
                        }
                        else {
                            paddingArray[2] = null;
                        }
                    }
                    if(paddingBottom!=null) {
                        if(paddingBottom instanceof Number) {
                            final int pBottom = ((Number)paddingBottom).intValue();
                            paddingArray[3] = pBottom==125?null:VMLStyle.getVMLLength(pBottom);
                        }
                        else {
                            paddingArray[3] = null;
                        }
                    }
                    if(paddingArray[0]==null&&paddingArray[1]==null&&paddingArray[2]==null&&paddingArray[3]==null) {
                        textbox.setInset(null);
                    }
                    else {
                        final StringBuilder stringBuilder = new StringBuilder();
                        for(int i=3;i>=0;i--) {
                            if(paddingArray[i]!=null) {
                                if(stringBuilder.length()>0) {
                                    stringBuilder.insert(0, paddingArray[i] + ",");
                                }
                                else {
                                    stringBuilder.insert(0, paddingArray[i]);
                                }
                            }
                        }
                    }
                }
                HashMap<String, String> textboxStyle = null;
                final Object autoResizeHeight = shapeAttrs.opt("autoResizeHeight");
                if(autoResizeHeight!=null) {
                    if(textbox==null) {
                        textbox = getTextbox(true);
                    }
                    textboxStyle = textbox.getStyleMap();
                    if((autoResizeHeight instanceof Boolean)&&((Boolean)autoResizeHeight).booleanValue()) {
                        textboxStyle.put("mso-fit-shape-to-text", "t");
                    }
                    else {
                        textboxStyle.remove("mso-fit-shape-to-text");
                    }
                }
            }
    	}

    	private static JSONObject createWMLColorFromVML(String vmlColor, String rgbDefault)
    		throws JSONException, ParseException {

    		final JSONObject color = new JSONObject(2);
			if(vmlColor!=null) {
                if(vmlColor.startsWith("#")) {
                	if(vmlColor.length()>=7) {
                        color.put("type", "rgb");
                        color.put("value", vmlColor.substring(1, 7).toUpperCase());
                	}
                }
                else {
                	color.put("type", "preset");
                    color.put("value", vmlColor);
                }
			}
			if(color.isEmpty()) {
				color.put("type", "rgb");
				color.put("value", rgbDefault);
			}
    		return color;
    	}

    	public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
    		throws JSONException, ParseException {

            final ShapeType shapeType = getType();

            final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
            final JSONObject drawingAttrs = initialDrawingAttrs!=null ? initialDrawingAttrs : new JSONObject();

            VMLStyle vmlStyle = new VMLStyle(this);

            final String coordSize = o.getCoordsize();
            if(coordSize!=null) {
                String[] result = coordSize.split(",");
                if(result.length==2&&!result[0].isEmpty()&&!result[1].isEmpty()) {
                    drawingAttrs.put("coordWidth", (int)VMLStyle.getLength(result[0], UnitType.COORDSIZE));
                    drawingAttrs.put("coordHeight", (int)VMLStyle.getLength(result[1], UnitType.COORDSIZE));
                }
            }
            if(parent!=null) {
                // childs are absolute having no dimension
                final String width = vmlStyle.get("width");
                if(width!=null) {
                    drawingAttrs.put("width", (int)VMLStyle.getLength(width, UnitType.COORDSIZE));
                }
                final String height = vmlStyle.get("height");
                if(height!=null) {
                    drawingAttrs.put("height", (int)VMLStyle.getLength(height, UnitType.COORDSIZE));
                }
                final String left = vmlStyle.get("left");
                if(left!=null) {
                    drawingAttrs.put("left", (int)VMLStyle.getLength(left, UnitType.COORDSIZE));
                }
                final String top = vmlStyle.get("top");
                if(top!=null) {
                    drawingAttrs.put("top", (int)VMLStyle.getLength(top, UnitType.COORDSIZE));
                }
            }
            else {
                String width = vmlStyle.get("width");
                if(VMLStyle.getDimensionType(width)==DimensionType.LENGTH) {
                    drawingAttrs.put("width", (int)VMLStyle.getLength(width, UnitType.PX));
                }
                String height = vmlStyle.get("height");
                if(VMLStyle.getDimensionType(height)==DimensionType.LENGTH) {
                    drawingAttrs.put("height", (int)VMLStyle.getLength(height, UnitType.PX));
                }
                boolean inline = true;
                String position = vmlStyle.get("position");
                if(position!=null) {
                    if(position.equals("absolute"))
                        inline = false;
                }
                drawingAttrs.put("inline", inline);
                if(!inline) {
                    drawingAttrs.put("anchorHorBase", vmlStyle.getHorzBase() );
                    String anchorHAlign = vmlStyle.getHorzAlign();
                    drawingAttrs.put("anchorHorAlign", anchorHAlign);
                    if(anchorHAlign.equals("offset")) {
                        int horzOffset = 0;
                        String marginLeft = vmlStyle.get("margin-left");
                        if(VMLStyle.getDimensionType(marginLeft)==DimensionType.LENGTH) {
                            horzOffset = (int)VMLStyle.getLength(marginLeft, UnitType.PX);
                        }
                        drawingAttrs.put("anchorHorOffset", (Integer)horzOffset);
                    }
                    drawingAttrs.put("anchorVertBase", vmlStyle.getVertBase() );
                    String anchorVAlign = vmlStyle.getVertAlign();
                    drawingAttrs.put("anchorVertAlign", anchorVAlign);
                    if(anchorVAlign.equals("offset")) {
                        int vertOffset = 0;
                        String marginTop = vmlStyle.get("margin-top");
                        if(VMLStyle.getDimensionType(marginTop)==DimensionType.LENGTH) {
                            vertOffset = (int)VMLStyle.getLength(marginTop, UnitType.PX);
                        }
                        drawingAttrs.put("anchorVertOffset", (Integer)vertOffset);
                    }
                    CTWrap ctWrap = getCTWrap(false);
                    if(ctWrap!=null) {
                        String wrapMode = "none";
                        if(ctWrap.getType()==STWrapType.SQUARE)
                            wrapMode = "square";
                        else if(ctWrap.getType()==STWrapType.THROUGH)
                            wrapMode = "through";
                        else if(ctWrap.getType()==STWrapType.TIGHT)
                            wrapMode = "tight";
                        else if(ctWrap.getType()==STWrapType.TOP_AND_BOTTOM)
                            wrapMode = "top-and-bottom";
                        drawingAttrs.put("textWrapMode", wrapMode);
                    }
                }
                if(shapeType == ShapeType.HORIZONTAL_LINE) {
                    //convert o:hr-properties to shape properties
                    //o:hralign, o:hrpct, o:hrnoshade, fillcolor, stroked
                    if(!drawingAttrs.has("width"))
                        drawingAttrs.put("width", 0);

                    drawingAttrs.put( "textWrapMode", "square" );
                    drawingAttrs.put( "anchorHorBase", "column" );
                    drawingAttrs.put( "anchorHorOffset", 0 );
                    drawingAttrs.put( "inline", false );

                    JSONObject topBorder = new JSONObject();
                    topBorder.put("width", 10);
                    topBorder.put("style", "solid");

                    if(o instanceof CTRect) {
                        CTRect rect = (CTRect)o;
                        STHrAlign hrAlign = rect.getHralign();
                        if(hrAlign != null){
                            drawingAttrs.put("anchorHorAlign", hrAlign.value());
                            if( hrAlign.value().equals("right") )
                                drawingAttrs.put( "textWrapSide", "left" );
                        }
                        String fillColor = rect.getFillcolor();
                        if(fillColor != null){
                            Color color = new Color();
                            if(fillColor.startsWith("#")) {
                                if(fillColor.length()==7) {
                                    // #RRGGBB
                                    color.setVal(fillColor.substring(1, 7).toUpperCase());
                                }
                                // TODO: there exists also the a #RGB
                            }
                            else
                                color.setVal(fillColor);
                            Commons.jsonPut(topBorder, "color", Utils.createColor(color));
                            drawingAttrs.put("borderTop", topBorder);
                        }
                    }
                    drawingAttrs.put("borderTop", topBorder);
                }
            }
            if(initialDrawingAttrs==null&&!drawingAttrs.isEmpty()) {
                attrs.put("drawing", drawingAttrs);
            }


    		// fill attributes
    		// ---------------
    		final JSONObject initialFillAttrs = attrs.optJSONObject("fill");
    		final JSONObject fillAttrs = initialFillAttrs!=null ? initialFillAttrs : new JSONObject();

    		// type
    		Boolean filled = null;
    		CTFill fill = getFill(false);
    		if(fill!=null&&fill.getOn()!=null) {
    			filled = fill.getOn().is();
    		}
    		if(filled==null&&o.getFilled()!=null) {
    			filled = o.getFilled().is();
    		}
    		if(filled==null&&referenceShape!=null) {
    			fill = referenceShape.getFill(false);
    			if(fill!=null&&fill.getOn()!=null) {
        			filled = fill.getOn().is();
    			}
    			if(filled==null&&referenceShape.o.getFilled()!=null) {
            		filled = referenceShape.o.getFilled().is();
    			}
    		}
    		if(filled==null||filled.booleanValue()) {
    			fillAttrs.put("type", "solid");
    		}

    		// color
    		String fillColor = null;
    		if(fill!=null) {
    			fillColor = fill.getColor();
    		}
    		if(fillColor==null) {
    			fillColor = o.getFillcolor();
    		}
    		fillAttrs.put("color", createWMLColorFromVML(fillColor, "FFFFFF"));
            if(initialFillAttrs==null&&!fillAttrs.isEmpty()) {
    			attrs.put("fill", fillAttrs);
    		}

            // line attributes
            // ---------------
    		final JSONObject initialLineAttrs = attrs.optJSONObject("line");
    		final JSONObject lineAttrs = initialLineAttrs!=null ? initialLineAttrs : new JSONObject();

    		// type...
    		// on/off inheritance order:
    		// this Shape.stroke.on != null
    		// this Shape.stroked != null
    		// referenced Shape.stroke.on != null
    		// referenced Shape.stroked !=null
    		// otherwise true..
    		Boolean stroked = null;
    		CTStroke stroke = getStroke(false);
    		if(stroke!=null&&stroke.getOn()!=null) {
    			stroked = stroke.getOn().is();
    		}
    		if(stroked==null&&o instanceof VmlShapeCore&&((VmlShapeCore)o).getStroked()!=null) {
        		stroked = ((VmlShapeCore)o).getStroked().is();
    		}
    		if(stroked==null&&referenceShape!=null) {
    			stroke = referenceShape.getStroke(false);
    			if(stroke!=null&&stroke.getOn()!=null) {
        			stroked = stroke.getOn().is();
    			}
    			if(stroked==null&&referenceShape.o instanceof VmlShapeCore&&((VmlShapeCore)referenceShape.o).getStroked()!=null) {
    				stroked = ((VmlShapeCore)referenceShape.o).getStroked().is();
    			}
    		}
    		if(stroked==null||stroked.booleanValue()) {
    			lineAttrs.put("type", "solid");
    		}
            // TODO: the following code is superfluous if line type "auto" is removed and the default is "none"
    		else {
    		    lineAttrs.put("type", "none");
    		}
    		// dashStyle
    		if(stroke!=null) {
    		    final String dashStyle = stroke.getDashstyle();
    		    if(dashStyle!=null) {
    		        String style = null;
    		        if(dashStyle.contains("DashDotDot")) {
    		            style = "dashDotDot";
    		        }
    		        else if(dashStyle.contains("DashDot")) {
    		            style = "dashDot";
    		        }
    		        else if(dashStyle.contains("Dash")) {
    		            style = "dashed";
    		        }
    		        else if(dashStyle.contains("Dot")) {
    		            style = "dotted";
    		        }
    		        if(style!=null) {
                        lineAttrs.put("style", style);
    		        }
    		    }
    		}
    		// color
    		String strokeColor = null;
    		String strokeWeight = null;
    		if(stroke!=null) {
    			strokeColor = stroke.getColor();
    			strokeWeight = stroke.getWeight();
    		}
    		if(o instanceof VmlShapeCore) {
    			if(strokeColor==null) {
    				strokeColor = ((VmlShapeCore)o).getStrokecolor();
    			}
    			if(strokeWeight==null) {
    				strokeWeight = ((VmlShapeCore)o).getStrokeweight();
    			}
    		}
    		lineAttrs.put("color", createWMLColorFromVML(strokeColor, "000000"));
    		lineAttrs.put("width", Math.round(VMLStyle.getLength(strokeWeight==null ? "1px" : strokeWeight, UnitType.EMU)));

            if(initialLineAttrs==null&&!lineAttrs.isEmpty()) {
    			attrs.put("line", lineAttrs);
    		}

            // Shape attributes
            //-----------------
            if(shapeType==ShapeType.SHAPE) {
                final CTTextbox textbox = getTextbox(false);
                if(textbox!=null) {
                    final JSONObject initialShapeAttrs = attrs.optJSONObject("shape");
                    final JSONObject shapeAttrs = initialShapeAttrs!=null ? initialShapeAttrs : new JSONObject();

                    final String inset = textbox.getInset();
                    if(inset!=null) {
                        final String[] paddings = inset.split(",");
                        if(paddings.length>=1&&!paddings[0].isEmpty()) {
                            shapeAttrs.put("paddingLeft", Double.valueOf(Math.round(VMLStyle.getLength(paddings[0], UnitType.EMU))).intValue());
                        }
                        if(paddings.length>=2&&!paddings[1].isEmpty()) {
                            shapeAttrs.put("paddingTop", Double.valueOf(Math.round(VMLStyle.getLength(paddings[1], UnitType.EMU))).intValue());
                        }
                        if(paddings.length>=3&&!paddings[2].isEmpty()) {
                            shapeAttrs.put("paddingRight", Double.valueOf(Math.round(VMLStyle.getLength(paddings[2], UnitType.EMU))).intValue());
                        }
                        if(paddings.length>=4&&!paddings[3].isEmpty()) {
                            shapeAttrs.put("paddingRight", Double.valueOf(Math.round(VMLStyle.getLength(paddings[3], UnitType.EMU))).intValue());
                        }
                    }

                    final HashMap<String, String> textboxStyle = textbox.getStyleMap();
                    if(textboxStyle.containsKey("mso-fit-shape-to-text")) {
                        final String val = textboxStyle.get("mso-fit-shape-to-text");
                        if(val.equals("t")) {
                            shapeAttrs.put("autoResizeHeight", true);
                        }
                    }
                    if(initialShapeAttrs==null&&!shapeAttrs.isEmpty()) {
                        attrs.put("shape", shapeAttrs);
                    }
                }
            }

            // Image attributes
            //-----------------
            if(shapeType==ShapeType.OLE||shapeType==ShapeType.IMAGE) {
            	createImageAttrs(createOperationHelper, this, attrs);
            }
    		return attrs;
    	}

        public IShape getNext() {
            if(parent==null) {
                return null;
            }
            final List<VML_Shape_group.Entry> shapes = parent.getShapes();
            int index = -1;
            for(int i=0; i<shapes.size();i++) {
                if(shapes.get(i).shape==this) {
                    index = i;
                    break;
                }
            }
            index++;
            if(index>0&&index<shapes.size()) {
                return shapes.get(index).shape;
            }
            return null;
        }

        public Object getChild() {

            final CTTextbox textbox = getTextbox(false);
            if(textbox==null) {
                return null;
            }
            final CTTxbxContent textboxContent = textbox.getTxbxContent();
            if(textboxContent==null) {
                return null;
            }
            final Component.RootComponent rootComponent = new Component.RootComponent(textboxContent);
            return rootComponent.getNextChildComponent(null, null);
    	}

    	public Object insertChild(int number, Component.Type type) {

    	    final CTTextbox textbox = getTextbox(true);
            CTTxbxContent textboxContent = textbox.getTxbxContent();
            if(textboxContent==null) {
                textboxContent = Context.getWmlObjectFactory().createCTTxbxContent();
                textboxContent.setParent(textbox);
                textbox.setTxbxContent(textboxContent);
            }
            final Component.RootComponent rootComponent = new Component.RootComponent(textboxContent);
            Component c = rootComponent.getNextChildComponent(null, null);
            if(c!=null) {
                c = c.getComponent(number);
            }
            return Component.insertChildComponent(null, new IndexedNode<Object>(textboxContent), number, c, type);
    	}
    }

    public static class VML_Shape_group extends VML_Shape_base {

        public static class Entry {
            VML_Shape_base  shape;
            int index;

            public Entry(VML_Shape_base _shape, int _index) {
                shape = _shape;
                index = _index;
            }
        }

        final List<Entry> shapes;

        public VML_Shape_group(CTGroup o, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), null, parent);
            shapes = new ArrayList<Entry>();
        }
        public void addShape(Entry entry) {
            shapes.add(entry);
        }
        public List<Entry> getShapes() {
        	return shapes;
        }
        @Override
        public ShapeType getType() {
        	return ShapeType.GROUP;
        }
        @Override
        public Object getChild() {
        	if(shapes.isEmpty()) {
        		return null;
        	}
        	return shapes.get(0).shape;
        }
        @Override
        public Object insertChild(int number, Component.Type type) {

            final org.docx4j.vml.ObjectFactory vmlObjectFactory = new org.docx4j.vml.ObjectFactory();

            VML_Shape_base child = null;
            JAXBElement<?> jaxbChild = null;
            if(type==Type.AC_GROUP) {
                final CTGroup groupImpl = vmlObjectFactory.createCTGroup();
                groupImpl.setParent(o);
                groupImpl.setVmlId("Group 1");
                groupImpl.setStyle("position:absolute;width:100;height:100");
                child = new VML_Shape_group(groupImpl, this);
                jaxbChild = vmlObjectFactory.createGroup(groupImpl);
            }
            else if(type==Type.AC_SHAPE) {
                final CTRect rectImpl = vmlObjectFactory.createCTRect();
                rectImpl.setParent(o);
                rectImpl.setVmlId("Rectangle 1");
                rectImpl.setStyle("position:absolute;width:100;height:100;visibility:visible;mso-wrap-style:square;v-text-anchor:middle");
                rectImpl.setFilled(org.docx4j.vml.STTrueFalse.F);
                rectImpl.setStroked(org.docx4j.vml.STTrueFalse.F);
                rectImpl.setStrokeweight("1pt");
                child = new VML_Shape_rect(rectImpl, this);
                jaxbChild = vmlObjectFactory.createRect(rectImpl);
            }
            else if(type==Type.AC_IMAGE) {
                final CTImage imageImpl = vmlObjectFactory.createCTImage();
                imageImpl.setParent(o);
                imageImpl.setVmlId("Image 1");
                imageImpl.setStyle("position:absolute;width:100;height:100;visibility:visible;mso-wrap-style:square");
                child = new VML_Shape_image(imageImpl, this);
                jaxbChild = vmlObjectFactory.createImage(imageImpl);
            }
            if(child!=null) {
                int new_index = number<shapes.size()?shapes.get(number).index:shapes.size();
                for(int i=number; i<shapes.size();i++) {
                    shapes.get(i).index++;  // fixing shape entries
                }
                shapes.add(number, new VML_Shape_group.Entry(child, new_index));
                ((CTGroup)o).getEGShapeElements().add(new_index, jaxbChild);
            }
            return child;
        }
        @Override
        public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
            throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

            super.applyAttrsFromJSON(operationDocument, attrs);
        }
    }

    // VML_Shape_type may be referenced by the VML_Shape_shape via type attribute
    public static class VML_Shape_type extends VML_Shape_base {

        final String Id;

        VML_Shape_type(CTShapetype o) {
            super(o, o.getEGShapeElements(), null, null);
            Id = o.getVmlId();
        }
        String getId() {
            return Id;
        }
    }
    public static class VML_Shape_shape extends VML_Shape_base {

        public VML_Shape_shape(CTShape o, VML_Shape_type vml_Shape_type, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), vml_Shape_type, parent);
        }
    }
    public static class VML_Shape_line extends VML_Shape_base {

        public VML_Shape_line(CTLine o, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), null, parent);
        }
    }
    public static class VML_Shape_polyline extends VML_Shape_base {

        public VML_Shape_polyline(CTPolyLine o, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), null, parent);
        }
    }
    public static class VML_Shape_curve extends VML_Shape_base {

        public VML_Shape_curve(CTCurve o, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), null, parent);
        }
    }
    public static class VML_Shape_rect extends VML_Shape_base {

        public VML_Shape_rect(CTRect o, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), null, parent);
        }
        @Override
        public ShapeType getType() {
    		final STTrueFalse stTrueFalse = ((CTRect)getObject()).getHr();
            if(stTrueFalse == STTrueFalse.T||stTrueFalse == STTrueFalse.TRUE) {
                return ShapeType.HORIZONTAL_LINE;
            }
            return ShapeType.SHAPE;
        }
    }
    public static class VML_Shape_roundrect extends VML_Shape_base {

        public VML_Shape_roundrect(CTRoundRect o, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), null, parent);
        }
    }
    public static class VML_Shape_oval extends VML_Shape_base {

        public VML_Shape_oval(CTOval o, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), null, parent);
        }
    }
    public static class VML_Shape_arc extends VML_Shape_base {

        public VML_Shape_arc(CTArc o, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), null, parent);
        }
    }
    public static class VML_Shape_image extends VML_Shape_base {

        public VML_Shape_image(CTImage o, VML_Shape_group parent) {
            super(o, o.getEGShapeElements(), null, parent);
        }
        @Override
        public ShapeType getType() {
    		return ShapeType.IMAGE;
        }
        @Override
        public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
            throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

            super.applyAttrsFromJSON(operationDocument, attrs);

            final JSONObject imageAttrs = attrs.optJSONObject("image");
            if(imageAttrs!=null) {
                String imageUrl = DocumentImageHelper.getImageUrlFromImageData(operationDocument.getResourceManager(), imageAttrs.optString("imageData"), "word/media/");
                if(imageUrl==null) {
                    imageUrl = imageAttrs.optString("imageUrl", null);
                }
                if(imageUrl!=null) {
                    final Relationship relationship = Commons.createGraphicRelation(operationDocument, operationDocument.getContextPart(), imageUrl);
                    if(relationship!=null) {
                        final CTImageData imageData = getImageData(true);
                        imageData.setId(relationship.getId());
                        imageData.setTitle("");
                    }
                }
            }
        }
    }

    protected static VML_Shape_base createPictVMLShape(VMLShape vmlShape, Object o, VML_Shape_group parent) {
        VML_Shape_base vml_Shape_base = null;
        if(o instanceof CTShapetype)
            vml_Shape_base = new VML_Shape_type((CTShapetype)o);
        else if(o instanceof CTGroup) {
            List<JAXBElement<?>> content = ((CTGroup)o).getEGShapeElements();
            vml_Shape_base = new VML_Shape_group((CTGroup)o, parent);
            for(int i=0; i<content.size();i++) {
                Object group_o = content.get(i);
                if(group_o instanceof JAXBElement)
                    group_o = ((JAXBElement<?>)group_o).getValue();
                VML_Shape_base group_vml = createPictVMLShape(vmlShape, group_o, (VML_Shape_group)vml_Shape_base);
                if(group_vml instanceof VML_Shape_type) {
                    String Id = ((VML_Shape_type)group_vml).getId();
                    if(Id.length()>0)
                        vmlShape.getShapeTypes().put(Id, (VML_Shape_type)group_vml);
                }
                else if (group_vml!=null) {
                    ((VML_Shape_group)vml_Shape_base).addShape(new VML_Shape_group.Entry(group_vml, i));
                }
            }
        }
        else if(o instanceof CTShape) {
            VML_Shape_type vmlShapetype = null;
            String type = ((CTShape)o).getType();
            if(type!=null&&type.length()>1) {
                type = type.substring(1);
                vmlShapetype = vmlShape.getShapeTypes().get(type);
            }
            vml_Shape_base = new VML_Shape_shape((CTShape)o, vmlShapetype, parent);
        }
        else if(o instanceof CTLine)
            vml_Shape_base = new VML_Shape_line((CTLine)o, parent);
        else if(o instanceof CTPolyLine)
            vml_Shape_base = new VML_Shape_polyline((CTPolyLine)o, parent);
        else if(o instanceof CTCurve)
            vml_Shape_base = new VML_Shape_curve((CTCurve)o, parent);
        else if(o instanceof CTRect)
            vml_Shape_base = new VML_Shape_rect((CTRect)o, parent);
        else if(o instanceof CTRoundRect)
            vml_Shape_base = new VML_Shape_roundrect((CTRoundRect)o, parent);
        else if(o instanceof CTOval)
            vml_Shape_base = new VML_Shape_oval((CTOval)o, parent);
        else if(o instanceof CTArc)
            vml_Shape_base = new VML_Shape_arc((CTArc)o, parent);
        else if(o instanceof CTImage)
            vml_Shape_base = new VML_Shape_image((CTImage)o, parent);

        return vml_Shape_base;
    }

    enum DimensionType {
        UNDEFINED,
        AUTO,
        LENGTH,
        PERCENT,
        INHERIT
    }

    enum UnitType {
        UNDEFINED,
        PT,
        PC,
        IN,
        MM,
        CM,
        PX,
        EM,
        EX,
        PERCENT,
        EMU,
        COORDSIZE
    }

    public static class VMLStyle {

        final VmlCore shape;
        final Map<String, String> values;

        VMLStyle(VML_Shape_base o) {
            shape = o.getObject();
            values = shape.getStyleMap();
        }
        String get(String key) {
            return values.get(key);
        }
        void set(String key, String value) {
            values.put(key, value);
        }
        void remove(String key) {
            values.remove(key);
        }

        String getHorzBase() {
            String value = get("mso-position-horizontal-relative"); // margin, page, text or char possible
            if(value==null)
                return "margin";
            else if(value.equals("page"))
                return "page";
            else if(value.equals("text"))
                return "column";
            else if(value.equals("char"))
                return "character";
            return "margin";
        }

        String getHorzAlign() {
            String value = get("mso-position-horizontal");          // absolute, left, center, right, inside or outside possible
            if(value==null)
                return "offset";
            else if(value.equals("absolute"))
                return "offset";
            else if(value.equals("center"))
                return "center";
            else if(value.equals("right"))
                return "right";
            else if(value.equals("inside"))
                return "inside";
            else if(value.equals("outside"))
                return "outside";
            return "left";
        }

        void setHorzBase(final String horzBase) {                   // page, column, character, leftMargin, insideMargin or outsideMargin
            if(horzBase.equals("page")) {
                set("mso-position-horizontal-relative", "page");
            }
            else if(horzBase.equals("column")) {
                set("mso-position-horizontal-relative", "text");
            }
            else if(horzBase.equals("character")) {
                set("mso-position-horizontal-relative", "char");
            }
            else if(horzBase.equals("leftMargin")) {
                set("mso-position-horizontal-relative", "margin");
            }
            else if(horzBase.equals("insideMargin")) {
                set("mso-position-horizontal-relative", "text");
            }
            else if(horzBase.equals("outsideMargin")) {
                set("mso-position-horizontal-relative", "text");
            }
        }

        void setHorzAlign(final String horzAlign) {                 // left, right, center, inside, outside or offset
            if(horzAlign.equals("left")) {
                set("mso-position-horizontal", "left");
            }
            if(horzAlign.equals("center")) {
                set("mso-position-horizontal", "center");
            }
            if(horzAlign.equals("right")) {
                set("mso-position-horizontal", "right");
            }
            if(horzAlign.equals("inside")) {
                set("mso-position-horizontal", "inside");
            }
            if(horzAlign.equals("outside")) {
                set("mso-position-horizontal", "outside");
            }
            if(horzAlign.equals("offset")) {                        // offset, we have to add the offset to the left margin..
                remove("mso-position-horizontal");
            }
        }

        String getVertBase() {
            String value = get("mso-position-vertical-relative");   // margin, page, text or line possible
            if(value==null)
                return "margin";
            else if(value.equals("page"))
                return "page";
            else if(value.equals("text"))
                return "paragraph";
            else if(value.equals("line"))
                return "line";
            return "margin";
        }

        String getVertAlign() {

            String value = get("mso-position-vertical");            // absolute, top, center, bottom, inside or outside possible
            if(value==null)
                return "offset";
            else if(value.equals("absolute"))
                return "offset";
            else if(value.equals("center"))
                return "center";
            else if(value.equals("bottom"))
                return "right";
            else if(value.equals("inside"))
                return "inside";
            else if(value.equals("outside"))
                return "outside";
            return "top";
        }

        void setVertBase(final String vertBase) {                   // margin, page, paragraph, line, topMargin, bottomMargin, insideMargin or outsideMargin
            if(vertBase.equals("margin")) {
                set("mso-position-vertical-relative", "margin");
            }
            else if(vertBase.equals("page")) {
                set("mso-position-vertical-relative", "page");
            }
            else if(vertBase.equals("paragraph")) {
                set("mso-position-vertical-relative", "text");
            }
            else if(vertBase.equals("line")) {
                set("mso-position-vertical-relative", "line");
            }
            else if(vertBase.equals("topMargin")) {
                set("mso-position-vertical-relative", "margin");
            }
            else if(vertBase.equals("bottomMargin")) {
                set("mso-position-vertical-relative", "margin");    // TODO: offset has to be added to the margin
            }
            else if(vertBase.equals("insideMargin")) {
                set("mso-position-vertical-relative", "inside");
            }
            else if(vertBase.equals("outsideMargin")) {
                set("mso-position-vertical-relative", "outside");
            }
        }

        void setVertAlign(final String vertAlign) {                 // top, bottom, center, inside, outside or offset
            if(vertAlign.equals("top")) {
                set("mso-position-vertical", "top");
            }
            else if(vertAlign.equals("bottom")) {
                set("mso-position-vertical", "bottom");
            }
            else if(vertAlign.equals("center")) {
                set("mso-position-vertical", "center");
            }
            else if(vertAlign.equals("inside")) {
                set("mso-position-vertical", "inside");
            }
            else if(vertAlign.equals("outside")) {
                set("mso-position-vertical", "outside");
            }
            else if(vertAlign.equals("offset")) {
                remove("mso-position-vertical");
            }
        }

        static DimensionType getDimensionType(String value) {
            if(value==null)
                return DimensionType.UNDEFINED;
            else if(value.equals("auto"))
                return DimensionType.AUTO;
            else if (value.equals("inherit"))
                return DimensionType.INHERIT;
            else if (value.contains("%"))
                return DimensionType.PERCENT;
            else if(value.matches(".*?pt|.*?pc|.*?in|.*?mm|.*?cm|.*?px"))
                return DimensionType.LENGTH;
            return DimensionType.UNDEFINED;
        }
        static UnitType getUnitType(String value) {
            if(value==null)
                return UnitType.UNDEFINED;
            else if(value.contains("pt"))
                return UnitType.PT;
            else if(value.contains("pc"))
                return UnitType.PC;
            else if(value.contains("in"))
                return UnitType.IN;
            else if(value.contains("mm"))
                return UnitType.MM;
            else if(value.contains("cm"))
                return UnitType.CM;
            else if(value.contains("px"))
                return UnitType.PX;
            else if(value.contains("em"))
                return UnitType.EM;
            else if(value.contains("ex"))
                return UnitType.EX;
            else if(value.contains("%"))
                return UnitType.PERCENT;
            else if(value.contains("emu"))
                return UnitType.EMU;
            else
                return UnitType.UNDEFINED;
        }
        // returns the length in 1/100mm
        public static double getLength(String value, UnitType defaultUnit) throws ParseException {
            double length = 0;

            length = NumberFormat.getInstance(Locale.US).parse(value).doubleValue();
            if(defaultUnit==UnitType.COORDSIZE) {
                return length;
            }
            switch(getUnitType(value)) {
                case EM :
                case EX :
                case PERCENT :
                    return 1;

                default:
                case UNDEFINED :
                case EMU : {
                    length /= 360.0;
                }
                break;
                case PT : length = (length/72)*2540; break;
                case PC : length = (length/6)*2540; break;
                case IN : length = length*2540; break;
                case MM : length = length*100; break;
                case CM : length = length*1000; break;
                case PX : length = (length/96)*2540; break;
            }
            return length;
        }

        public static String getVMLLength(long length100ThMM) {
            BigDecimal dec = new BigDecimal(((double)length100ThMM*72/2540));
            dec = dec.setScale(2, BigDecimal.ROUND_HALF_UP);
            return dec.toString() + "pt";
        }

        // sets length which is assumed to be in 1/100mm
        void setLength(String key, int length) {
            values.put(key, getVMLLength(length));
        }
    }
}
