/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.apache.commons.io.IOUtils;
import org.docx4j.dml.Theme;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.packages.OpcPackage;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.PeoplePart;
import org.docx4j.openpackaging.parts.ThemePart;
import org.docx4j.openpackaging.parts.WordprocessingML.CommentsPart;
import org.docx4j.openpackaging.parts.WordprocessingML.DocumentSettingsPart;
import org.docx4j.openpackaging.parts.WordprocessingML.FontTablePart;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.openpackaging.parts.WordprocessingML.StyleDefinitionsPart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.docx4j.relationships.Relationships;
import org.docx4j.w15.CTPeople;
import org.docx4j.wml.CTLanguage;
import org.docx4j.wml.CTSettings;
import org.docx4j.wml.Comments;
import org.docx4j.wml.Comments.Comment;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.DocDefaults;
import org.docx4j.wml.DocDefaults.RPrDefault;
import org.docx4j.wml.Fonts;
import org.docx4j.wml.Fonts.Font;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.P;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.SectPr;
import org.docx4j.wml.SectPr.PgMar;
import org.docx4j.wml.Styles;
import org.docx4j.wml.Styles.LatentStyles;
import org.docx4j.wml.Styles.LatentStyles.LsdException;
import org.docx4j.wml.Tr;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;

import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.IResourceManager;
import com.openexchange.office.ooxml.docx.operations.ApplyOperationHelper;
import com.openexchange.office.ooxml.docx.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.docx.tools.Component;
import com.openexchange.office.ooxml.docx.tools.Component.ParagraphComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextRun_Base;
import com.openexchange.office.ooxml.docx.tools.Component.TrComponent;
import com.openexchange.office.ooxml.docx.tools.Utils;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class OperationDocument extends com.openexchange.office.ooxml.OperationDocument {

    private WordprocessingMLPackage opcPackage;

    public OperationDocument(Session session, ServiceLookup _services, InputStream _inputDocumentStream, IResourceManager _resourceManager, DocumentProperties _documentProperties)
        throws FilterException {

        super(session, _services, _inputDocumentStream, _resourceManager, _documentProperties);
        try {

        	inputDocumentStream = checkDocumentComplexity(_inputDocumentStream);

            opcPackage = WordprocessingMLPackage.load(inputDocumentStream);
        }
        catch(Throwable e) {
            removeMemoryListener();
            throw getRethrowException(e);
        }
    }

    // the constructor without inputStream creates an empty document
    public OperationDocument(ServiceLookup _services, DocumentProperties _documentProperties)
        throws FilterException {

        super(null, _services, null, null, _documentProperties);
        try {
            opcPackage = WordprocessingMLPackage.createPackage();

            final MainDocumentPart mainDocumentPart = getPackage().getMainDocumentPart();
            final Styles styles = mainDocumentPart.getStyleDefinitionsPart().getJaxbElement();
            resetDocumentLanguage(styles);
            insertPredefinedLatentStyles(styles);
            final ThemePart themePart = getDefaultThemePart(new PartName("/word/theme/theme1.xml"));
            mainDocumentPart.addTargetPart(themePart);
            final DocumentSettingsPart documentSettingsPart = getDefaultSettingsPart(new PartName("/word/settings.xml"));
            mainDocumentPart.addTargetPart(documentSettingsPart);
            final SectPr sectPr = Utils.getDocumentProperties(mainDocumentPart, true);
            PgMar pgMar = sectPr.getPgMar();
            if(pgMar==null) {
            	pgMar = Context.getWmlObjectFactory().createSectPrPgMar();
            	pgMar.setParent(sectPr);
            	sectPr.setPgMar(pgMar);
            }
            pgMar.setLeft(BigInteger.valueOf(1440));
            pgMar.setRight(BigInteger.valueOf(1440));
            pgMar.setTop(BigInteger.valueOf(1440));
            pgMar.setBottom(BigInteger.valueOf(1440));
            pgMar.setHeader(BigInteger.valueOf(708));
            pgMar.setFooter(BigInteger.valueOf(708));

            // creating a fontlist (QuickOffice won't load documents without this part)
            getFontList(true);
        }
        catch(Throwable e) {
            removeMemoryListener();
            throw getRethrowException(e);
        }
    }

    private void setContext(String target) {
        if(target.isEmpty()) {
            setContextPart(null);
        }
        else if (target.startsWith("cmt")) {
        	// not to be allowed to be zero... a comment has to be inserted
        	// before it could be used, so this part must be there
     		final CommentsPart commentsPart = getPackage().getMainDocumentPart().getCommentsPart();
     		final Comments comments = commentsPart.getJaxbElement();
     		comments.setCommentId(BigInteger.valueOf(Integer.valueOf(target.substring(3))));
     		setContextPart(commentsPart);
        }
        else {
            setContextPart(getPackage().getMainDocumentPart().getRelationshipsPart(false).getPart(target));
        }
    }

    @Override
    public void applyOperations(String applyOperations) throws FilterException {

        if (applyOperations != null){
            int i = 0;
            JSONObject op = null;
            try {
                ApplyOperationHelper applyOperationHelper = new ApplyOperationHelper(this);
                final JSONArray aOperations = new JSONArray(new JSONTokener(applyOperations));
                for (i = 0; i < aOperations.length(); i++) {
                	successfulAppliedOperations = i;
                    op = (JSONObject) aOperations.get(i);
                    setContext(op.optString("target", ""));
                    final String opName = op.getString("name");
                    switch(opName) {
                    	case "insertParagraph": {
                        	applyOperationHelper.insertParagraph(op.getJSONArray("start"), op.optJSONObject("attrs"));
                        	break;
                    	}
                    	case "delete": {
                    		applyOperationHelper.delete(op.getJSONArray("start"), op.optJSONArray("end"));
                    		break;
                    	}
                    	case "move": {
                    		applyOperationHelper.move(op.getJSONArray("start"), op.optJSONArray("end"), op.getJSONArray("to"));
                    		break;
                    	}
                    	case "splitParagraph": {
                    		applyOperationHelper.splitParagraph(op.getJSONArray("start"));
                    		break;
                    	}
                    	case "mergeParagraph": {
                    		applyOperationHelper.mergeParagraph(op.getJSONArray("start"));
                    		break;
                    	}
                    	case "insertText": {
                    		applyOperationHelper.insertText(op.getJSONArray("start"), op.getString("text").replaceAll("\\p{C}", " "), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertTab": {
                    		applyOperationHelper.insertTab(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertHardBreak": {
                    		applyOperationHelper.insertHardBreak(op.getJSONArray("start"), op.optString("type"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertTable": {
                    		applyOperationHelper.insertTable(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertRows": {
                    		applyOperationHelper.insertRows(op.getJSONArray("start"), op.optInt("count", 1), op.optBoolean("insertDefaultCells", false), op.optInt("referenceRow", -1), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertCells": {
                    		applyOperationHelper.insertCells(op.getJSONArray("start"), op.optInt("count", 1), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertColumn": {
                    		applyOperationHelper.insertColumn(op.getJSONArray("start"), op.getJSONArray("tableGrid"), op.getInt("gridPosition"), op.optString("insertMode", "before"));
                    		break;
                    	}
                    	case "deleteColumns": {
                    		applyOperationHelper.deleteColumns(op.getJSONArray("start"), op.getInt("startGrid"), op.optInt("endGrid", op.getInt("startGrid")));
                    		break;
                    	}
                    	case "setAttributes": {
                    		applyOperationHelper.setAttributes(op.getJSONObject("attrs"), op.getJSONArray("start"), op.optJSONArray("end"));
                    		break;
                    	}
                    	case "insertField": {
                    		applyOperationHelper.insertField(op.getJSONArray("start"), op.getString("type"), op.getString("representation"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertDrawing": {
                    		applyOperationHelper.insertDrawing(op.getJSONArray("start"), op.getString("type"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertStyleSheet": {
                    		applyOperationHelper.insertStyleSheet(op.getString("type"), op.getString("styleId"), op.getString("styleName"), op.getJSONObject("attrs"), op.optString("parent"), op.optBoolean("hidden"), op.getInt("uiPriority"), op.optBoolean("default"), op.optBoolean("custom"));
                    		break;
                    	}
                    	case "changeStyleSheet": {
                    		applyOperationHelper.changeStyleSheet(op.getString("type"), op.getString("styleId"), op.optString("styleName"), op.optJSONObject("attrs"), op.optString("parent"), op.optBoolean("hidden"), op.optInt("uiPriority"));
                    		break;
                    	}
                    	case "deleteStyleSheet": {
                    		applyOperationHelper.deleteStyleSheet(op.getString("type"), op.getString("styleId"));
                    		break;
                    	}
                    	case "insertListStyle": {
                    		applyOperationHelper.insertListStyle(op.getString("listStyleId"), op.getJSONObject("listDefinition"));
                    		break;
                    	}
                    	case "splitTable": {
                    		applyOperationHelper.splitTable(op.getJSONArray("start"));
                    		break;
                    	}
                    	case "mergeTable": {
                    		applyOperationHelper.mergeTable(op.getJSONArray("start"));
                    		break;
                    	}
                    	case "setDocumentAttributes": {
                    		applyOperationHelper.setDocumentAttributes(op.getJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertHeaderFooter": {
                    		applyOperationHelper.insertHeaderFooter(op.getString("id"), op.getString("type"));
                    		break;
                    	}
                    	case "deleteHeaderFooter": {
                    		applyOperationHelper.deleteHeaderFooter(op.getString("id"));
                    		break;
                    	}
                    	case "insertComment": {
                    		applyOperationHelper.insertComment(op.getJSONArray("start"), op.getString("id"), op.optString("uid"), op.optString("author"), op.optString("date"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertRange": {
                    		applyOperationHelper.insertRange(op.getJSONArray("start"), op.getString("id"), op.optString("type", "comment"), op.optString("position", "start"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "insertComplexField": {
                    		applyOperationHelper.insertComplexField(op.getJSONArray("start"), op.getString("instruction"), op.optJSONObject("attrs"));
                    		break;
                    	}
                    	case "createError": {
                    		throw new FilterException("createError operation detected: " + opName, ErrorCode.UNSUPPORTED_OPERATION_USED);
                    	}
                    	default: {
                    		logMessage("warn", "Ignoring unsupported operation: " + opName);
                    	}
                    }
                }
                successfulAppliedOperations=aOperations.length();

                // creating fontListPart ...for QuickOffice
                getFontList(true);

                // set user language if not already set in the document
                applyOperationHelper.setDocumentLanguageIfNotSet(getUserLanguage());

                // postfix non conformance of paragraphs in tables and unique ids (maindocumentPart/headerPart/footerPart/commentsPart)
                final AtomicInteger uniqueId = new AtomicInteger(1000000);
                final MainDocumentPart mainDocumentPart = getPackage().getMainDocumentPart();
                setContextPart(null);
                final Component.RootComponent rootComponent = new Component.RootComponent(mainDocumentPart);
                applyOperationHelper.postFixNonConformance(rootComponent.getNextChildComponent(null, null), uniqueId);
                final RelationshipsPart mainRelationshipsPart = mainDocumentPart.getRelationshipsPart();
                final Relationships relationships = mainRelationshipsPart.getRelationships();
                if(relationships!=null) {
                    for(Relationship relationship:relationships.getRelationship()) {
                        final String type = relationship.getType();
                        if(type.equals(Namespaces.HEADER)||type.equals(Namespaces.FOOTER)) {
                            setContextPart(mainRelationshipsPart.getPart(relationship));
                            final Component.RootComponent headerFooterComponent = new Component.RootComponent((ContentAccessor)getContextPart());
                            applyOperationHelper.postFixNonConformance(headerFooterComponent.getNextChildComponent(null, null), uniqueId);
                        }
                        else if(type.equals(Namespaces.COMMENTS)) {
                        	final CommentsPart commentsPart = (CommentsPart)mainRelationshipsPart.getPart(relationship);
                        	setContextPart(commentsPart);
                        	final Comments comments = commentsPart.getContents();
                        	for(Comment c:comments.getComment()) {
                        		comments.setCommentId(c.getId());
	                            final Component.RootComponent commentComponent = new Component.RootComponent((ContentAccessor)getContextPart());
	                            applyOperationHelper.postFixNonConformance(commentComponent.getNextChildComponent(null, null), uniqueId);
                        	}
                        }
                    }
                }
            }
            catch(Exception e) {
                String message = e.getMessage();
                if(op!=null) {
                    message += ", operation:" + Integer.toString(i) + " " + op.toString();
                }
                throw new FilterException(message, e, ErrorCode.CRITICAL_ERROR);
            }
        }
    }

    @Override
    public JSONObject getOperations()
        throws Exception {

        final JSONObject aOperations = new JSONObject();
        final JSONArray operationsArray = new JSONArray();

        //set dummy for addSetDocumentAttributes-Operation called in createDocumentDefaults
        final JSONObject noOpSync = new JSONObject();
        noOpSync.put("name", "noOp");
        operationsArray.put(noOpSync);

        CreateOperationHelper createOperationHelper = new CreateOperationHelper(this, operationsArray);
        createOperationHelper.CreateFontDescriptions();
        createOperationHelper.CreateStyleOperations();
        createOperationHelper.CreateThemeOperations();
        createOperationHelper.CreateListOperations();
        createOperationHelper.CreateHeaderFooterOperations();
        createOperationHelper.createOperations();
        createOperationHelper.createDocumentDefaults(getUserLanguage());
        aOperations.put("operations", operationsArray);

        return aOperations;
    }

    @Override
    public WordprocessingMLPackage getPackage() {
        return opcPackage;
    }

    @Override
    public void setPackage(OpcPackage p) {
       opcPackage = (WordprocessingMLPackage)p;
    }

    @Override
    public Part getContextPart() {
    	return contextPart!=null ? contextPart : getPackage().getMainDocumentPart();
    }

    @Override
    public Theme getTheme()
    	throws FilterException {

    	ThemePart themePart = getPackage().getMainDocumentPart().getThemePart();
        if(themePart == null) {
        	try {
        		themePart = getDefaultThemePart(new PartName("/word/theme/theme1.xml"));
        	}
        	catch(InvalidFormatException e) {
        		throw new FilterException("docx filter, could not create default theme", ErrorCode.CRITICAL_ERROR, e);
        	}
        }
        return themePart.getJaxbElement();
    }

    public Styles getStyles(boolean createIfMissing)
        throws Exception {

        StyleDefinitionsPart styleDefinitionsPart = getPackage().getMainDocumentPart().getStyleDefinitionsPart();
        if(styleDefinitionsPart==null&&createIfMissing) {
            styleDefinitionsPart = new StyleDefinitionsPart(new PartName("/word/styles.xml"));
            styleDefinitionsPart.setJaxbElement(Context.getWmlObjectFactory().createStyles());
            getPackage().getMainDocumentPart().addTargetPart(styleDefinitionsPart);
        }
        return styleDefinitionsPart!=null?styleDefinitionsPart.getJaxbElement() : null;
    }

    public CTSettings getSettings(boolean createIfMissing)
    	throws Exception {

        DocumentSettingsPart settingsPart = getPackage().getMainDocumentPart().getDocumentSettingsPart();
        if(settingsPart==null&&createIfMissing) {
        	settingsPart = new DocumentSettingsPart(new PartName("/word/settings.xml"));
        	settingsPart.setJaxbElement(Context.getWmlObjectFactory().createCTSettings());
        	getPackage().getMainDocumentPart().addTargetPart(settingsPart);
        }
    	return settingsPart!=null?settingsPart.getJaxbElement():null;
    }

    public CommentsPart getCommentsPart(boolean createIfMissing)
    	throws InvalidFormatException {

 		CommentsPart commentsPart = getPackage().getMainDocumentPart().getCommentsPart();
    	if(commentsPart==null&&createIfMissing) {
    		commentsPart = new CommentsPart();
    		commentsPart.setJaxbElement(Context.getWmlObjectFactory().createComments());
    		getPackage().getMainDocumentPart().addTargetPart(commentsPart);
    	}
    	return commentsPart;
    }

    public Comment getComment(BigInteger id, boolean createIfMissing)
    	throws InvalidFormatException {

 		CommentsPart commentsPart = getCommentsPart(createIfMissing);
    	if(commentsPart!=null) {
    		final Comments comments = commentsPart.getJaxbElement();
    		for(Comment comment:comments.getComment()) {
    			if(comment.getId().equals(id)) {
    				return comment;
    			}
    		}
    		if(createIfMissing) {
    			final Comment comment = Context.getWmlObjectFactory().createCommentsComment();
    			comment.setParent(comments);
    			comment.setId(id);
    			comment.setAuthor("");
    			comments.getComment().add(comment);
    			return comment;
    		}
    	}
    	return null;
    }

    public List<Font> getFontList(boolean createIfMissing)
        throws Exception {

        FontTablePart fontTablePart = getPackage().getMainDocumentPart().getFontTablePart();
        if(fontTablePart==null&&createIfMissing) {
            fontTablePart = new FontTablePart(new PartName("/word/fontTable.xml"));
            fontTablePart.setJaxbElement(Context.getWmlObjectFactory().createFonts());
            getPackage().getMainDocumentPart().addTargetPart(fontTablePart);
        }
        if(fontTablePart!=null) {
            final Fonts fonts = fontTablePart.getJaxbElement();
            if(fonts!=null) {
                final List<Font> fontList = fonts.getFont();
                if(createIfMissing) {
                    int i;
                    // we put at least one dummy font into this list
                    for(i=0;i<fontList.size();i++) {
                        final String fontName = fontList.get(i).getName();
                        if(fontName!=null&&fontName.equals("Times New Roman")) {
                            break;
                        }
                    }
                    if(i==fontList.size()) {
                        final Font timesNewRoman = Context.getWmlObjectFactory().createFontsFont();
                        timesNewRoman.setName("Times New Roman");
                        fontList.add(timesNewRoman);
                    }
                }
            }
        }
        return null;
    }

    public CTPeople getPeopleList(boolean createIfMissing) {

    	try {
	        PeoplePart peoplePart = getPackage().getMainDocumentPart().getPeoplePart();
	        if(peoplePart==null&&createIfMissing) {
	        	peoplePart = new PeoplePart(new PartName("/word/people.xml"));
	        	peoplePart.setJaxbElement(new org.docx4j.w15.ObjectFactory().createCTPeople());
	        	getPackage().getMainDocumentPart().addTargetPart(peoplePart);
	        }
	        return peoplePart!=null?peoplePart.getJaxbElement():null;
    	}
    	catch(InvalidFormatException e) {
    		return null;
    	}
    }

    /**
     * Reset document language.
     *
     * @param styles {Styles}
     *  The WordML styles with or without language settings
     */
    private void resetDocumentLanguage(Styles styles) {

        if (styles != null) {
            final ObjectFactory objectFactory = new ObjectFactory();
            DocDefaults docDefaults = styles.getDocDefaults();
            if(docDefaults==null) {
                docDefaults = objectFactory.createDocDefaults();
                docDefaults.setParent(styles);
            }
            RPrDefault rDefault = docDefaults.getRPrDefault();
            if(rDefault==null) {
                rDefault = objectFactory.createDocDefaultsRPrDefault();
                rDefault.setParent(docDefaults);
            }
            RPr rPrDefault = rDefault.getRPr();
            if(rPrDefault==null) {
                rPrDefault = objectFactory.createRPr();
                rPrDefault.setParent(rDefault);
            }
            CTLanguage ctLanguage = rPrDefault.getLang();
            if(ctLanguage==null) {
                ctLanguage = objectFactory.createCTLanguage();
                ctLanguage.setParent(rPrDefault);
            }
            // reset language
            ctLanguage.setVal("");
            ctLanguage.setEastAsia("");
            ctLanguage.setBidi("");
        }
    }

    @Override
    protected void switchToNonTemplateDocument() {
    	super.switchToNonTemplateDocument();
    }

    @Override
    protected void switchToTemplateDocument() {
    	super.switchToTemplateDocument();

    	final MainDocumentPart mainDocumentPart = getPackage().getMainDocumentPart();
        final RelationshipsPart mainRelationshipsPart = mainDocumentPart.getRelationshipsPart();
        removePart(mainRelationshipsPart, "/word/stylesWithEffects.xml");
        removePart(mainRelationshipsPart, "/word/webSettings.xml");
        removePart(mainRelationshipsPart, "/word/printerSettings/printerSettings1.bin");
        removePart(mainRelationshipsPart, "/word/printerSettings/printerSettings2.bin");
        removePart(mainRelationshipsPart, "/word/printerSettings/printerSettings3.bin");

        // removing changeIds from the document
        final Component.RootComponent rootComponent = new Component.RootComponent(mainDocumentPart);
    	removeRSids(rootComponent.getNextChildComponent(null, null));

    	// removing rsIds from settings if available
    	final DocumentSettingsPart documentSettingsPart = mainDocumentPart.getDocumentSettingsPart();
    	if(documentSettingsPart!=null) {
    		final CTSettings settings = documentSettingsPart.getJaxbElement();
    		settings.setRsids(null);
    	}
    	final SectPr secPr = mainDocumentPart.getJaxbElement().getBody().getSectPr();
    	if(secPr!=null) {
    		secPr.setRsidDel(null);
    		secPr.setRsidR(null);
    		secPr.setRsidRPr(null);
    		secPr.setRsidSect(null);
    	}
    }

    private void removeRSids(Component c) {
    	while(c!=null) {
    		removeRSids(c.getNextChildComponent(null, null));
    		if(c instanceof ParagraphComponent) {
    			final P p = (P)c.getObject();
    			p.setRsidDel(null);
    			if(p.getRsidP()!=null&&!p.getRsidP().equals("47110815")) {
        			p.setRsidP(null);
    			}
    			p.setRsidR(null);
    			p.setRsidRDefault(null);
    			p.setRsidRPr(null);
    		}
    		else if (c instanceof TextRun_Base) {
    			final R r = ((TextRun_Base)c).getTextRun();
    			r.setRsidDel(null);
    			r.setRsidR(null);
    			r.setRsidRPr(null);
    		}
    		else if(c instanceof TrComponent) {
    			final Tr tr = (Tr)c.getObject();
    			tr.setRsidDel(null);
    			tr.setRsidR(null);
    			tr.setRsidRPr(null);
    			tr.setRsidTr(null);
    		}
    		c = c.getNextComponent();
    	}
    }

    private InputStream checkDocumentComplexity(InputStream _inputStream)
        	throws FilterException {

    	InputStream returnValue = null;

    	final int maxWordCount  = ConfigurationHelper.getIntegerOfficeConfigurationValue(getServiceLookup(), getSession(), "//text/maxWordCount", 50000);
        if(maxWordCount<=0) {
        	return _inputStream;
        }

        int xmlFileSize = 0;

        byte[] bytes;

        try {
			bytes = IOUtils.toByteArray(_inputStream);
		} catch (IOException e1) {
            throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
		} finally {
	        IOUtils.closeQuietly(_inputStream);
		}

        final InputStream _zipInputStream = new ByteArrayInputStream(bytes);
        final ZipInputStream zipInputStream = new ZipInputStream(_zipInputStream);

        ZipEntry zipEntry = null;
        final byte[] buffer = new byte[0x10000];  // 64kb buffer

        try {
			while((zipEntry = zipInputStream.getNextEntry())!=null) {

				int read;

				final String zipEntryName = zipEntry.getName();
				if(zipEntryName.endsWith(".xml")) {
		        	while((read = zipInputStream.read(buffer))>=0) {
		        		xmlFileSize += read;
		        	}
		        }
			    zipInputStream.closeEntry();

			    // the document.xml size of a document with 50000 words -> 3,5mb
			    // the same document with only 25000 words -> 2,4mb
			    // so we assume 25000 words are about 1,1mb and 1mb is for the bottom
			    // of the barrel such as styles etc
			    if(xmlFileSize>maxWordCount*45+1000000) {
		            throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
			    }
			}
			returnValue = new ByteArrayInputStream(bytes);
        } catch (IOException e) {
            throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
		} finally {
			try {
				zipInputStream.close();
			} catch (IOException e) {

			}
			IOUtils.closeQuietly(_zipInputStream);
		}
        return returnValue;
    }


    private void insertPredefinedLatentStyles(Styles styles) {

        final ObjectFactory objectFactory = new ObjectFactory();
        // name, semiHidden, uiPriority, unhideWhenUsed, qFormat
        final String[] predefinedLatentStyles = {
            "Normal, 0, 0, 0, 1",
            "heading 1, 0, 9, 0, 1",
            "heading 2,, 9,, 1",
            "heading 3,, 9,, 1",
            "heading 4,, 9,, 1",
            "heading 5,, 9,, 1",
            "heading 6,, 9,, 1",
            "heading 7,, 9,, 1",
            "heading 8,, 9,, 1",
            "heading 9,, 9,, 1",
            "toc 1,, 39",
            "toc 2,, 39",
            "toc 3,, 39",
            "toc 4,, 39",
            "toc 5,, 39",
            "toc 6,, 39",
            "toc 7,, 39",
            "toc 8,, 39",
            "toc 9,, 39",
            "caption,, 35,, 1",
            "Title, 0, 10, 0, 1",
            "Default Paragraph Font, , 1",
            "Subtitle, 0, 11, 0, 1",
            "Strong, 0, 22,, 0, 1",
            "Emphasis, 0, 20,, 0, 1",
            "Table Grid, 0, 59, 0",
            "Placeholder Text,,,0",
            "No Spacing, 0, 1, 0, 1",
            "Light Shading, 0, 60,0",
            "Light List, 0 ,61, 0",
            "Light Grid, 0 ,62, 0",
            "Medium Shading 1, 0 ,63, 0",
            "Medium Shading 2, 0 ,64, 0",
            "Medium List 1, 0 ,65, 0",
            "Medium List 2, 0 ,66, 0",
            "Medium Grid 1, 0 ,67, 0",
            "Medium Grid 2, 0 ,68, 0",
            "Medium Grid 3, 0 ,69, 0",
            "Dark List, 0 ,70, 0",
            "Colorful Shading, 0 ,71, 0",
            "Colorful List, 0 ,72, 0",
            "Colorful Grid, 0 ,73, 0",
            "Light Shading Accent 1, 0 ,60, 0",
            "Light List Accent 1, 0 ,61, 0",
            "Light Grid Accent 1, 0 ,62, 0",
            "Medium Shading 1 Accent 1, 0 ,63, 0",
            "Medium Shading 2 Accent 1, 0 ,64, 0",
            "Medium List 1 Accent 1, 0 ,65, 0",
            "Revision, 0",
            "List Paragraph, 0 ,34, 0, 1",
            "Quote, 0 ,29, 0, 1",
            "Intense Quote, 0 ,30, 0, 1",
            "Medium List 2 Accent 1, 0 ,66, 0",
            "Medium Grid 1 Accent 1, 0 ,67, 0",
            "Medium Grid 2 Accent 1, 0 ,68, 0",
            "Medium Grid 3 Accent 1, 0 ,69, 0",
            "Dark List Accent 1, 0 ,70, 0",
            "Colorful Shading Accent 1, 0 ,71, 0",
            "Colorful List Accent 1, 0 ,72, 0",
            "Colorful Grid Accent 1, 0 ,73, 0",
            "Light Shading Accent 2, 0 ,60, 0",
            "Light List Accent 2, 0 ,61, 0",
            "Light Grid Accent 2, 0 ,62, 0",
            "Medium Shading 1 Accent 2, 0 ,63, 0",
            "Medium Shading 2 Accent 2, 0 ,64, 0",
            "Medium List 1 Accent 2, 0 ,65, 0",
            "Medium List 2 Accent 2, 0 ,66, 0",
            "Medium Grid 1 Accent 2, 0 ,67, 0",
            "Medium Grid 2 Accent 2, 0 ,68, 0",
            "Medium Grid 3 Accent 2, 0 ,69, 0",
            "Dark List Accent 2, 0 ,70, 0",
            "Colorful Shading Accent 2, 0 ,71, 0",
            "Colorful List Accent 2, 0 ,72, 0",
            "Colorful Grid Accent 2, 0 ,73, 0",
            "Light Shading Accent 3, 0 ,60, 0",
            "Light List Accent 3, 0 ,61, 0",
            "Light Grid Accent 3, 0 ,62, 0",
            "Medium Shading 1 Accent 3, 0 ,63, 0",
            "Medium Shading 2 Accent 3, 0 ,64, 0",
            "Medium List 1 Accent 3, 0 ,65, 0",
            "Medium List 2 Accent 3, 0 ,66, 0",
            "Medium Grid 1 Accent 3, 0 ,67, 0",
            "Medium Grid 2 Accent 3, 0 ,68, 0",
            "Medium Grid 3 Accent 3, 0 ,69, 0",
            "Dark List Accent 3, 0 ,70, 0",
            "Colorful Shading Accent 3, 0 ,71, 0",
            "Colorful List Accent 3, 0 ,72, 0",
            "Colorful Grid Accent 3, 0 ,73, 0",
            "Light Shading Accent 4, 0 ,60, 0",
            "Light List Accent 4, 0 ,61, 0",
            "Light Grid Accent 4, 0 ,62, 0",
            "Medium Shading 1 Accent 4, 0 ,63, 0",
            "Medium Shading 2 Accent 4, 0 ,64, 0",
            "Medium List 1 Accent 4, 0 ,65, 0",
            "Medium List 2 Accent 4, 0 ,66, 0",
            "Medium Grid 1 Accent 4, 0 ,67, 0",
            "Medium Grid 2 Accent 4, 0 ,68, 0",
            "Medium Grid 3 Accent 4, 0 ,69, 0",
            "Dark List Accent 4, 0 ,70, 0",
            "Colorful Shading Accent 4, 0 ,71, 0",
            "Colorful List Accent 4, 0 ,72, 0",
            "Colorful Grid Accent 4, 0 ,73, 0",
            "Light Shading Accent 5, 0 ,60, 0",
            "Light List Accent 5, 0 ,61, 0",
            "Light Grid Accent 5, 0 ,62, 0",
            "Medium Shading 1 Accent 5, 0 ,63, 0",
            "Medium Shading 2 Accent 5, 0 ,64, 0",
            "Medium List 1 Accent 5, 0 ,65, 0",
            "Medium List 2 Accent 5, 0 ,66, 0",
            "Medium Grid 1 Accent 5, 0 ,67, 0",
            "Medium Grid 2 Accent 5, 0 ,68, 0",
            "Medium Grid 3 Accent 5, 0 ,69, 0",
            "Dark List Accent 5, 0 ,70, 0",
            "Colorful Shading Accent 5, 0 ,71, 0",
            "Colorful List Accent 5, 0 ,72, 0",
            "Colorful Grid Accent 5, 0 ,73, 0",
            "Light Shading Accent 6, 0 ,60, 0",
            "Light List Accent 6, 0 ,61, 0",
            "Light Grid Accent 6, 0 ,62, 0",
            "Medium Shading 1 Accent 6, 0 ,63, 0",
            "Medium Shading 2 Accent 6, 0 ,64, 0",
            "Medium List 1 Accent 6, 0 ,65, 0",
            "Medium List 2 Accent 6, 0 ,66, 0",
            "Medium Grid 1 Accent 6, 0 ,67, 0",
            "Medium Grid 2 Accent 6, 0 ,68, 0",
            "Medium Grid 3 Accent 6, 0 ,69, 0",
            "Dark List Accent 6, 0 ,70, 0",
            "Colorful Shading Accent 6, 0 ,71, 0",
            "Colorful List Accent 6, 0 ,72, 0",
            "Colorful Grid Accent 6, 0 ,73, 0",
            "Subtle Emphasis, 0 ,19, 0, 1",
            "Intense Emphasis, 0 ,21, 0, 1",
            "Subtle Reference, 0 ,31, 0, 1",
            "Intense Reference, 0 ,32, 0, 1",
            "Book Title, 0 ,33, 0, 1",
            "Bibliography,, 37",
            "TOC Heading,, 39,, 1"
        };

        if (styles != null) {
            // set attributes for the latent styles part
            LatentStyles latentStyles = styles.getLatentStyles();
            latentStyles.setDefLockedState(false);
            latentStyles.setDefUIPriority(BigInteger.valueOf(99));
            latentStyles.setDefSemiHidden(true);
            latentStyles.setDefUnhideWhenUsed(true);
            latentStyles.setDefQFormat(false);

            // Extract predefined latent styles and put into the styles parts
            List<LsdException> styleList = latentStyles.getLsdException();
            // We will add all known latent MS styles therefore remove preset
            // to prevent duplicates
            styleList.clear();

            for (String style: predefinedLatentStyles) {
                String[] styleParts = style.split(",");

                LsdException lsdException = objectFactory.createStylesLatentStylesLsdException();
                lsdException.setParent(styleList);
                for (int i = 0; i < styleParts.length; i++) {

                    String value = styleParts[i].trim();
                    if (value.length() > 0) {
                        switch (i) {
                            case 0: lsdException.setName(value); break;
                            case 1: lsdException.setSemiHidden(Boolean.parseBoolean(value)); break;
                            case 2: lsdException.setUiPriority(BigInteger.valueOf(Long.parseLong(value))); break;
                            case 3: lsdException.setUnhideWhenUsed(Boolean.parseBoolean(value)); break;
                            case 4: lsdException.setQFormat(Boolean.parseBoolean(value)); break;
                        }
                    }
                }
                styleList.add(lsdException);
            }
        }
    }
}
