/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

window.chai.use(function (chai, utils) {

    'use strict';

    var EPSILON = Math.pow(2, -51);

    // private global functions ===============================================

    /**
     * Adds a new property to Chai's message chain.
     *
     * @param {String} name
     *  The name of the new property.
     *
     * @param {Function} handler
     *  The callback function invoked for the property. Will be called in the
     *  context of the assertion instance.
     */
    function addProperty(name, handler) {
        utils.addProperty(chai.Assertion.prototype, name, handler);
    }

    /**
     * Adds a new method to Chai's message chain.
     *
     * @param {String} name
     *  The name of the new method.
     *
     * @param {Function} handler
     *  The callback function invoked for the method. Will be called in the
     *  context of the assertion instance. Receives the parameters passed to
     *  the method in the language chain of the assertion.
     */
    function addMethod(name, handler) {
        utils.addMethod(chai.Assertion.prototype, name, handler);
    }

    /**
     * Returns whether the passed value is a finite number.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed value is a finite number. Does NOT return true for
     *  arrays containing a number element, as the native function isFinite()
     *  does.
     */
    function isFiniteNumber(value) {
        return (typeof value === 'number') && isFinite(value);
    }

    // properties =============================================================

    /**
     * A property asserting whether the actual value is zero, or almost zero.
     * Precisely, returns whether the absolute value is less than or equal to
     * 2^-51 which is twice of Number.EPSILON.
     *
     * @example
     *  expect(Math.sin(Math.PI)).to.be.almostZero;
     */
    addProperty('almostZero', function () {
        this.assert(
            isFiniteNumber(this._obj) && (Math.abs(this._obj) <= EPSILON),
            'expected #{this} to be almost zero',
            'expected #{this} to not be almost zero'
        );
    });

    /**
     * A property asserting whether the actual value is a finite number.
     *
     * @example
     *  expect(Math.sin(0)).to.be.finite;
     */
    addProperty('finite', function () {
        this.assert(
            isFiniteNumber(this._obj),
            'expected #{this} to be finite',
            'expected #{this} to not be finite'
        );
    });

    /**
     * A property asserting whether the actual value is NaN.
     *
     * @example
     *  expect(Math.log(-1)).to.be.NaN;
     */
    addProperty('NaN', function () {
        this.assert(
            // isNaN() would return true for the array [NaN]
            (typeof this._obj === 'number') && isNaN(this._obj),
            'expected #{this} to be NaN',
            'expected #{this} to not be NaN'
        );
    });

    // methods ================================================================

    /**
     * A method asserting whether the tested number and the specified number
     * are almost equal, with a relative error less than or equal to 2^-51
     * which is twice of Number.EPSILON.
     *
     * @example
     *  expect(Math.sin(Math.PI/2)).to.almostEqual(1);
     *
     * @param {Number} expected
     *  The expected number that will be compared with the tested number. MUST
     *  NOT be zero (use the property 'almostZero' for that).
     */
    addMethod('almostEqual', function (expected) {
        this.assert(
            isFiniteNumber(this._obj) && (Math.abs((this._obj - expected) / expected) <= EPSILON),
            'expected #{this} to almost equal #{exp}',
            'expected #{this} to not almost equal #{exp}',
            expected
        );
    });

    /**
     * A method asserting whether the tested value, converted to a string,
     * equals the passed expectation.
     *
     * @example
     *  expect([1, 2]).to.stringifyTo('1,2');
     *
     * @param {String} expected
     *  The expected string that will be compared with the result of passing
     *  the tested value to the String() constructor.
     */
    addMethod('stringifyTo', function (expected) {
        var result = String(this._obj);
        this.assert(
            result === expected,
            'expected #{this} to stringify to #{exp} but got \'' + result + '\'',
            'expected #{this} to not stringify to #{exp}',
            expected
        );
    });

    // ========================================================================
});
