/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/model/formula/impl/datetimefuncs'
], function (DateTimeFuncs) {

    'use strict';

    // module DateTimeFuncs ===================================================

    describe('Spreadsheet module DateTimeFuncs', function () {

        it('should exist', function () {
            expect(DateTimeFuncs).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        // converts a date string 'YYYY-MM-DD' to an instance of the Date class
        function d(date) {
            var matches = /^(\d{4})-(\d{2})-(\d{2})$/.exec(date);
            return new Date(Date.UTC(parseInt(matches[1]), parseInt(matches[2]) - 1, parseInt(matches[3])));
        }

        // function implementations -------------------------------------------

        // TODO: more test cases to cover all execution paths
        describe('function "YEARFRAC"', function () {
            var YEARFRAC = DateTimeFuncs.YEARFRAC;
            it('should exist', function () {
                expect(YEARFRAC).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(YEARFRAC).itself.to.respondTo('resolve');
            });
            it('should return correct results for date mode 0 (30/360 US)', function () {
                expect(YEARFRAC.resolve(d('2011-01-01'), d('2011-01-01'), 0)).to.be.almostZero;
                expect(YEARFRAC.resolve(d('2011-02-28'), d('2012-02-29'), 0)).to.almostEqual(1);
            });
            it('should return correct results for date mode 1 (actual/actual)', function () {
                expect(YEARFRAC.resolve(d('2011-01-01'), d('2011-01-01'), 1)).to.be.almostZero;
                expect(YEARFRAC.resolve(d('2011-02-28'), d('2012-02-29'), 1)).to.almostEqual(732 / 731);
            });
            it('should return correct results for date mode 2 (actual/360)', function () {
                expect(YEARFRAC.resolve(d('2011-01-01'), d('2011-01-01'), 2)).to.be.almostZero;
                expect(YEARFRAC.resolve(d('2011-02-28'), d('2012-02-29'), 2)).to.almostEqual(61 / 60);
            });
            it('should return correct results for date mode 3 (actual/365)', function () {
                expect(YEARFRAC.resolve(d('2011-01-01'), d('2011-01-01'), 3)).to.be.almostZero;
                expect(YEARFRAC.resolve(d('2011-02-28'), d('2012-02-29'), 3)).to.almostEqual(366 / 365);
            });
            it('should return correct results for date mode 4 (30E/360)', function () {
                expect(YEARFRAC.resolve(d('2011-01-01'), d('2011-01-01'), 4)).to.be.almostZero;
                expect(YEARFRAC.resolve(d('2011-02-28'), d('2012-02-29'), 4)).to.almostEqual(361 / 360);
            });
            it('should default to date mode 0 (30/360 US)', function () {
                expect(YEARFRAC.resolve(d('2011-01-01'), d('2011-01-01'))).to.be.almostZero;
                expect(YEARFRAC.resolve(d('2011-02-28'), d('2012-02-29'))).to.almostEqual(1);
            });
        });

    });

    // ========================================================================
});
