/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (SheetUtils, FormulaUtils) {

    'use strict';

    // static class FormulaUtils ==============================================

    describe('Spreadsheet module FormulaUtils', function () {

        it('should exist', function () {
            expect(FormulaUtils).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var sqrt = Math.sqrt,
            exp = Math.exp,
            log = Math.log,
            ErrorCodes = SheetUtils.ErrorCodes;

        // workaround to test for throwing an error code (seems that expect(...).to.throw()
        // does not work with custom exception values that are not instances of Error)
        function expectErrorCode(expected, func) {
            try {
                var result = func.apply(null, _.toArray(arguments).slice(expectErrorCode.length));
                // error code may be returned instead of thrown
                expect(result).to.equal(expected);
            } catch (ex) {
                expect(ex).to.equal(expected);
            }
        }

        // constants ----------------------------------------------------------

        describe('constant "EPSILON"', function () {
            it('should exist', function () {
                expect(FormulaUtils.EPSILON).to.be.a('number');
                expect(FormulaUtils.EPSILON).to.be.above(0);
            });
        });

        describe('constant "MIN_NUMBER"', function () {
            it('should exist', function () {
                expect(FormulaUtils.MIN_NUMBER).to.be.a('number');
                expect(FormulaUtils.MIN_NUMBER).to.be.above(0);
                expect(FormulaUtils.MIN_NUMBER).to.be.below(FormulaUtils.EPSILON);
            });
        });

        describe('constant "MAX_PARAM_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils.MAX_PARAM_COUNT).to.be.a('number');
                expect(FormulaUtils.MAX_PARAM_COUNT).to.be.above(0);
            });
        });

        describe('constant "MAX_MATRIX_ROW_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils.MAX_MATRIX_ROW_COUNT).to.be.a('number');
                expect(FormulaUtils.MAX_MATRIX_ROW_COUNT).to.be.above(0);
            });
        });

        describe('constant "MAX_MATRIX_COL_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils.MAX_MATRIX_COL_COUNT).to.be.a('number');
                expect(FormulaUtils.MAX_MATRIX_COL_COUNT).to.be.above(0);
            });
        });

        describe('constant "MAX_REF_LIST_SIZE"', function () {
            it('should exist', function () {
                expect(FormulaUtils.MAX_REF_LIST_SIZE).to.be.a('number');
                expect(FormulaUtils.MAX_REF_LIST_SIZE).to.be.above(0);
            });
        });

        describe('constant "MAX_CELL_ITERATION_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils.MAX_CELL_ITERATION_COUNT).to.be.a('number');
                expect(FormulaUtils.MAX_CELL_ITERATION_COUNT).to.be.above(0);
            });
        });

        describe('constant "UNSUPPORTED_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils.UNSUPPORTED_ERROR).to.be.an('object');
                expect(FormulaUtils.UNSUPPORTED_ERROR).to.have.property('code');
            });
        });

        describe('constant "CIRCULAR_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils.CIRCULAR_ERROR).to.be.an('object');
                expect(FormulaUtils.CIRCULAR_ERROR).to.have.property('code');
            });
        });

        // public methods -----------------------------------------------------

        describe('method "trunc"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('trunc');
            });
            it('should round down positive numbers', function () {
                expect(FormulaUtils.trunc(1.1)).to.equal(1);
                expect(FormulaUtils.trunc(5.5)).to.equal(5);
                expect(FormulaUtils.trunc(9.9)).to.equal(9);
            });
            it('should round up negative numbers', function () {
                expect(FormulaUtils.trunc(-1.1)).to.equal(-1);
                expect(FormulaUtils.trunc(-5.5)).to.equal(-5);
                expect(FormulaUtils.trunc(-9.9)).to.equal(-9);
            });
            it('should return integers unmodified', function () {
                expect(FormulaUtils.trunc(-1)).to.equal(-1);
                expect(FormulaUtils.trunc(0)).to.equal(0);
                expect(FormulaUtils.trunc(1)).to.equal(1);
            });
        });

        describe('method "add"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('add');
            });
            it('should return the sum of two numbers', function () {
                expect(FormulaUtils.add(3, 2)).to.equal(5);
                expect(FormulaUtils.add(0, 2)).to.equal(2);
            });
            it('should return the concatenation of two strings', function () {
                expect(FormulaUtils.add('a', 'b')).to.equal('ab');
            });
        });

        describe('method "multiply"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('multiply');
            });
            it('should return the product of two numbers', function () {
                expect(FormulaUtils.multiply(3, 2)).to.equal(6);
            });
        });

        describe('method "divide"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('divide');
            });
            it('should return the quotient of two numbers', function () {
                expect(FormulaUtils.divide(3, 2)).to.equal(1.5);
                expect(FormulaUtils.divide(0, 2)).to.equal(0);
            });
            it('should throw the #DIV/0! error, if divisor is 0', function () {
                expectErrorCode(ErrorCodes.DIV0, FormulaUtils.divide, 3, 0);
                expectErrorCode(ErrorCodes.DIV0, FormulaUtils.divide, 0, 0);
            });
        });

        describe('method "modulo"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('modulo');
            });
            it('should return the remainder, if divisor is positive', function () {
                expect(FormulaUtils.modulo(-5, 3)).to.equal(1);
                expect(FormulaUtils.modulo(-4, 3)).to.equal(2);
                expect(FormulaUtils.modulo(-3, 3)).to.equal(0);
                expect(FormulaUtils.modulo(-2, 3)).to.equal(1);
                expect(FormulaUtils.modulo(-1, 3)).to.equal(2);
                expect(FormulaUtils.modulo(0, 3)).to.equal(0);
                expect(FormulaUtils.modulo(1, 3)).to.equal(1);
                expect(FormulaUtils.modulo(2, 3)).to.equal(2);
                expect(FormulaUtils.modulo(3, 3)).to.equal(0);
                expect(FormulaUtils.modulo(4, 3)).to.equal(1);
                expect(FormulaUtils.modulo(5, 3)).to.equal(2);
            });
            it('should return the remainder, if divisor is negative', function () {
                expect(FormulaUtils.modulo(-5, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(-4, -3)).to.equal(-1);
                expect(FormulaUtils.modulo(-3, -3)).to.equal(0);
                expect(FormulaUtils.modulo(-2, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(-1, -3)).to.equal(-1);
                expect(FormulaUtils.modulo(0, -3)).to.equal(0);
                expect(FormulaUtils.modulo(1, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(2, -3)).to.equal(-1);
                expect(FormulaUtils.modulo(3, -3)).to.equal(0);
                expect(FormulaUtils.modulo(4, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(5, -3)).to.equal(-1);
            });
            it('should throw the #DIV/0! error, if divisor is 0', function () {
                expectErrorCode(ErrorCodes.DIV0, FormulaUtils.modulo, 3, 0);
                expectErrorCode(ErrorCodes.DIV0, FormulaUtils.modulo, 0, 0);
            });
        });

        describe('method "power"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('power');
            });
            it('should return the power of two numbers', function () {
                expect(FormulaUtils.power(3, 0)).to.equal(1);
                expect(FormulaUtils.power(3, 1)).to.equal(3);
                expect(FormulaUtils.power(3, 2)).to.equal(9);
                expect(FormulaUtils.power(3, 3)).to.equal(27);
                expect(FormulaUtils.power(9, 0.5)).to.almostEqual(3);
                expect(FormulaUtils.power(81, 0.25)).to.almostEqual(3);
                expect(FormulaUtils.power(4, -1)).to.almostEqual(0.25);
                expect(FormulaUtils.power(2, -2)).to.almostEqual(0.25);
                expect(FormulaUtils.power(0, 1)).to.equal(0);
                expect(FormulaUtils.power(0, 2)).to.equal(0);
                expect(FormulaUtils.power(0, 0.5)).to.equal(0);
            });
            it('should throw the #NUM! error, if both numbers are 0', function () {
                expectErrorCode(ErrorCodes.NUM, FormulaUtils.power, 0, 0);
            });
            it('should throw the #DIV/0! error, if base is 0, and exponent is negative', function () {
                expectErrorCode(ErrorCodes.DIV0, FormulaUtils.power, 0, -1);
            });
        });

        describe('method "arctan2"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('arctan2');
            });
            it('should return the arc tangent of two numbers', function () {
                expect(FormulaUtils.arctan2(1, 0)).to.be.almostZero;
                expect(FormulaUtils.arctan2(1, 1)).to.almostEqual(0.25 * Math.PI);
                expect(FormulaUtils.arctan2(0, 1)).to.almostEqual(0.5 * Math.PI);
                expect(FormulaUtils.arctan2(-1, 1)).to.almostEqual(0.75 * Math.PI);
                expect(FormulaUtils.arctan2(-1, 0)).to.almostEqual(Math.PI);
                expect(FormulaUtils.arctan2(-1, -1)).to.almostEqual(-0.75 * Math.PI);
                expect(FormulaUtils.arctan2(0, -1)).to.almostEqual(-0.5 * Math.PI);
                expect(FormulaUtils.arctan2(1, -1)).to.almostEqual(-0.25 * Math.PI);
            });
            it('should throw the #DIV/0! error, if both numbers are 0', function () {
                expectErrorCode(ErrorCodes.DIV0, FormulaUtils.arctan2, 0, 0);
            });
        });

        describe('method "sinh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('sinh');
            });
            it('should return the hyperbolic sine', function () {
                expect(FormulaUtils.sinh(-2)).to.almostEqual((exp(-2) - exp(2)) / 2);
                expect(FormulaUtils.sinh(-1)).to.almostEqual((exp(-1) - exp(1)) / 2);
                expect(FormulaUtils.sinh(0)).to.equal(0);
                expect(FormulaUtils.sinh(1)).to.almostEqual((exp(1) - exp(-1)) / 2);
                expect(FormulaUtils.sinh(2)).to.almostEqual((exp(2) - exp(-2)) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.sinh) { expect(FormulaUtils.sinh).to.equal(Math.sinh); }
            });
        });

        describe('method "cosh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('cosh');
            });
            it('should return the hyperbolic cosine', function () {
                expect(FormulaUtils.cosh(-2)).to.almostEqual((exp(-2) + exp(2)) / 2);
                expect(FormulaUtils.cosh(-1)).to.almostEqual((exp(-1) + exp(1)) / 2);
                expect(FormulaUtils.cosh(0)).to.equal(1);
                expect(FormulaUtils.cosh(1)).to.almostEqual((exp(1) + exp(-1)) / 2);
                expect(FormulaUtils.cosh(2)).to.almostEqual((exp(2) + exp(-2)) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.cosh) { expect(FormulaUtils.cosh).to.equal(Math.cosh); }
            });
        });

        describe('method "tanh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('tanh');
            });
            it('should return the hyperbolic tangent', function () {
                expect(FormulaUtils.tanh(-2)).to.almostEqual((exp(-4) - 1) / (exp(-4) + 1));
                expect(FormulaUtils.tanh(-1)).to.almostEqual((exp(-2) - 1) / (exp(-2) + 1));
                expect(FormulaUtils.tanh(0)).to.equal(0);
                expect(FormulaUtils.tanh(1)).to.almostEqual((exp(2) - 1) / (exp(2) + 1));
                expect(FormulaUtils.tanh(2)).to.almostEqual((exp(4) - 1) / (exp(4) + 1));
            });
            it('should use the native implementation', function () {
                if (Math.tanh) { expect(FormulaUtils.tanh).to.equal(Math.tanh); }
            });
        });

        describe('method "asinh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('asinh');
            });
            it('should return the argument resulting in the hyperbolic sine', function () {
                expect(FormulaUtils.asinh(-2)).to.almostEqual(log(sqrt(5) - 2));
                expect(FormulaUtils.asinh(-1)).to.almostEqual(log(sqrt(2) - 1));
                expect(FormulaUtils.asinh(0)).to.equal(0);
                expect(FormulaUtils.asinh(1)).to.almostEqual(log(sqrt(2) + 1));
                expect(FormulaUtils.asinh(2)).to.almostEqual(log(sqrt(5) + 2));
            });
            it('should use the native implementation', function () {
                if (Math.asinh) { expect(FormulaUtils.asinh).to.equal(Math.asinh); }
            });
        });

        describe('method "acosh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('acosh');
            });
            it('should return the argument resulting in the hyperbolic cosine', function () {
                expect(FormulaUtils.acosh(-1)).not.to.be.finite;
                expect(FormulaUtils.acosh(0)).not.to.be.finite;
                expect(FormulaUtils.acosh(1)).to.equal(0);
                expect(FormulaUtils.acosh(2)).to.almostEqual(log(2 + sqrt(3)));
                expect(FormulaUtils.acosh(3)).to.almostEqual(log(3 + sqrt(8)));
            });
            it('should use the native implementation', function () {
                if (Math.acosh) { expect(FormulaUtils.acosh).to.equal(Math.acosh); }
            });
        });

        describe('method "atanh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('atanh');
            });
            it('should return the argument resulting in the hyperbolic tangent', function () {
                expect(FormulaUtils.atanh(-2)).not.to.be.finite;
                expect(FormulaUtils.atanh(-1)).not.to.be.finite;
                expect(FormulaUtils.atanh(-0.5)).to.almostEqual(log(1 / 3) / 2);
                expect(FormulaUtils.atanh(0)).to.equal(0);
                expect(FormulaUtils.atanh(0.5)).to.almostEqual(log(3) / 2);
                expect(FormulaUtils.atanh(1)).not.to.be.finite;
                expect(FormulaUtils.atanh(2)).not.to.be.finite;
            });
            it('should use the native implementation', function () {
                if (Math.atanh) { expect(FormulaUtils.atanh).to.equal(Math.atanh); }
            });
        });

        describe('method "acoth"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('acoth');
            });
            it('should return the argument resulting in the hyperbolic tangent', function () {
                expect(FormulaUtils.acoth(-3)).to.almostEqual(log(1 / 2) / 2);
                expect(FormulaUtils.acoth(-2)).to.almostEqual(log(1 / 3) / 2);
                expect(FormulaUtils.acoth(-1)).not.to.be.finite;
                expect(FormulaUtils.acoth(0)).not.to.be.finite;
                expect(FormulaUtils.acoth(1)).not.to.be.finite;
                expect(FormulaUtils.acoth(2)).to.almostEqual(log(3) / 2);
                expect(FormulaUtils.acoth(3)).to.almostEqual(log(2) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.acoth) { expect(FormulaUtils.acoth).to.equal(Math.acoth); }
            });
        });

        describe('method "compareNumbers"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareNumbers');
            });
            it('should return -1, if first number is less than second number', function () {
                expect(FormulaUtils.compareNumbers(-3, 0)).to.equal(-1);
                expect(FormulaUtils.compareNumbers(0, 3)).to.equal(-1);
            });
            it('should return 0, if the numbers are equal', function () {
                expect(FormulaUtils.compareNumbers(-3, -3)).to.equal(0);
                expect(FormulaUtils.compareNumbers(0, 0)).to.equal(0);
                expect(FormulaUtils.compareNumbers(3, 3)).to.equal(0);
                expect(FormulaUtils.compareNumbers(0, FormulaUtils.MIN_NUMBER / 2)).to.equal(0);
            });
            it('should return 1, if first number is greater than second number', function () {
                expect(FormulaUtils.compareNumbers(3, 0)).to.equal(1);
                expect(FormulaUtils.compareNumbers(0, -3)).to.equal(1);
            });
            it('should return NaN, if either number is not finite', function () {
                expect(FormulaUtils.compareNumbers(3, 1 / 0)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(1 / 0, 3)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(1 / 0, 1 / 0)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(3, Number.NaN)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(Number.NaN, 3)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(Number.NaN, Number.NaN)).to.be.NaN;
            });
        });

        describe('method "compareStrings"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareStrings');
            });
            it('should return -1, if first string is less than second string (ignoring case)', function () {
                expect(FormulaUtils.compareStrings('a', 'b')).to.equal(-1);
                expect(FormulaUtils.compareStrings('A', 'b')).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'B')).to.equal(-1);
                expect(FormulaUtils.compareStrings('A', 'B')).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'aa')).to.equal(-1);
                expect(FormulaUtils.compareStrings('ab', 'ba')).to.equal(-1);
                expect(FormulaUtils.compareStrings('', 'a')).to.equal(-1);
            });
            it('should return 0, if the strings are equal (ignoring case)', function () {
                expect(FormulaUtils.compareStrings('a', 'a')).to.equal(0);
                expect(FormulaUtils.compareStrings('a', 'A')).to.equal(0);
                expect(FormulaUtils.compareStrings('A', 'a')).to.equal(0);
                expect(FormulaUtils.compareStrings('A', 'A')).to.equal(0);
                expect(FormulaUtils.compareStrings('abc', 'abc')).to.equal(0);
                expect(FormulaUtils.compareStrings('', '')).to.equal(0);
            });
            it('should return 1, if first string is greater than second string (ignoring case)', function () {
                expect(FormulaUtils.compareStrings('b', 'a')).to.equal(1);
                expect(FormulaUtils.compareStrings('B', 'a')).to.equal(1);
                expect(FormulaUtils.compareStrings('b', 'A')).to.equal(1);
                expect(FormulaUtils.compareStrings('B', 'A')).to.equal(1);
                expect(FormulaUtils.compareStrings('aa', 'a')).to.equal(1);
                expect(FormulaUtils.compareStrings('ba', 'ab')).to.equal(1);
                expect(FormulaUtils.compareStrings('a', '')).to.equal(1);
            });
            it('should return -1, if first string is less than second string (regarding case)', function () {
                expect(FormulaUtils.compareStrings('A', 'a', true)).to.equal(-1);
                expect(FormulaUtils.compareStrings('B', 'a', true)).to.equal(-1);
            });
            it('should return 1, if first string is greater than second string (regarding case)', function () {
                expect(FormulaUtils.compareStrings('a', 'A', true)).to.equal(1);
                expect(FormulaUtils.compareStrings('b', 'A', true)).to.equal(1);
            });
        });

        describe('method "compareBooleans"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareBooleans');
            });
            it('should return the result of Boolean comparison', function () {
                expect(FormulaUtils.compareBooleans(false, false)).to.equal(0);
                expect(FormulaUtils.compareBooleans(false, true)).to.equal(-1);
                expect(FormulaUtils.compareBooleans(true, false)).to.equal(1);
                expect(FormulaUtils.compareBooleans(true, true)).to.equal(0);
            });
        });

        describe('method "compareErrorCodes"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareErrorCodes');
            });
            it('should return the result of error code comparison', function () {
                expect(FormulaUtils.compareErrorCodes(ErrorCodes.NULL, ErrorCodes.NULL)).to.equal(0);
                expect(FormulaUtils.compareErrorCodes(ErrorCodes.NULL, ErrorCodes.DIV0)).to.equal(-1);
                expect(FormulaUtils.compareErrorCodes(ErrorCodes.DIV0, ErrorCodes.NULL)).to.equal(1);
            });
        });

        describe('method "compareValues"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareValues');
            });
            it('should compare null to other types', function () {
                expect(FormulaUtils.compareValues(null, 0)).to.equal(-1);
                expect(FormulaUtils.compareValues(null, '')).to.equal(-1);
                expect(FormulaUtils.compareValues(null, false)).to.equal(-1);
                expect(FormulaUtils.compareValues(null, ErrorCodes.NULL)).to.equal(-1);
            });
            it('should compare numbers to other types', function () {
                expect(FormulaUtils.compareValues(0, null)).to.equal(1);
                expect(FormulaUtils.compareValues(0, '0')).to.equal(-1);
                expect(FormulaUtils.compareValues(0, false)).to.equal(-1);
                expect(FormulaUtils.compareValues(0, ErrorCodes.NULL)).to.equal(-1);
            });
            it('should compare strings to other types', function () {
                expect(FormulaUtils.compareValues('', null)).to.equal(1);
                expect(FormulaUtils.compareValues('0', 0)).to.equal(1);
                expect(FormulaUtils.compareValues('false', false)).to.equal(-1);
                expect(FormulaUtils.compareValues('#NULL!', ErrorCodes.NULL)).to.equal(-1);
            });
            it('should compare Booleans to other types', function () {
                expect(FormulaUtils.compareValues(false, null)).to.equal(1);
                expect(FormulaUtils.compareValues(false, 0)).to.equal(1);
                expect(FormulaUtils.compareValues(false, 'false')).to.equal(1);
                expect(FormulaUtils.compareValues(false, ErrorCodes.NULL)).to.equal(-1);
            });
            it('should compare error codes to other types', function () {
                expect(FormulaUtils.compareValues(ErrorCodes.NULL, null)).to.equal(1);
                expect(FormulaUtils.compareValues(ErrorCodes.NULL, 0)).to.equal(1);
                expect(FormulaUtils.compareValues(ErrorCodes.NULL, '#NULL!')).to.equal(1);
                expect(FormulaUtils.compareValues(ErrorCodes.NULL, false)).to.equal(1);
            });
            it('should compare null values', function () {
                expect(FormulaUtils.compareValues(null, null)).to.equal(0);
            });
            it('should compare numbers', function () {
                expect(FormulaUtils.compareValues(0, 1)).to.equal(-1);
                expect(FormulaUtils.compareValues(1, 1)).to.equal(0);
                expect(FormulaUtils.compareValues(1, 0)).to.equal(1);
                expect(FormulaUtils.compareValues(3, 1 / 0)).to.be.NaN;
            });
            it('should compare strings (ignoring case)', function () {
                expect(FormulaUtils.compareValues('a', 'a')).to.equal(0);
                expect(FormulaUtils.compareValues('a', 'A')).to.equal(0);
                expect(FormulaUtils.compareValues('a', 'b')).to.equal(-1);
                expect(FormulaUtils.compareValues('a', 'B')).to.equal(-1);
                expect(FormulaUtils.compareValues('b', 'a')).to.equal(1);
                expect(FormulaUtils.compareValues('B', 'a')).to.equal(1);
            });
            it('should compare strings (regarding case)', function () {
                expect(FormulaUtils.compareValues('a', 'a', true)).to.equal(0);
                expect(FormulaUtils.compareValues('a', 'A', true)).to.equal(1);
                expect(FormulaUtils.compareValues('a', 'b', true)).to.equal(-1);
                expect(FormulaUtils.compareValues('a', 'B', true)).to.equal(1);
                expect(FormulaUtils.compareValues('b', 'a', true)).to.equal(1);
                expect(FormulaUtils.compareValues('B', 'a', true)).to.equal(-1);
            });
            it('should compare Booleans', function () {
                expect(FormulaUtils.compareValues(false, false)).to.equal(0);
                expect(FormulaUtils.compareValues(false, true)).to.equal(-1);
                expect(FormulaUtils.compareValues(true, false)).to.equal(1);
                expect(FormulaUtils.compareValues(true, true)).to.equal(0);
            });
            it('should compare error codes', function () {
                expect(FormulaUtils.compareValues(ErrorCodes.NULL, ErrorCodes.NULL)).to.equal(0);
                expect(FormulaUtils.compareValues(ErrorCodes.NULL, ErrorCodes.DIV0)).to.equal(-1);
                expect(FormulaUtils.compareValues(ErrorCodes.DIV0, ErrorCodes.NULL)).to.equal(1);
            });
        });

        // logger -------------------------------------------------------------

        describe('should support the logger interface"', function () {
            expect(FormulaUtils).itself.to.respondTo('log');
            expect(FormulaUtils).itself.to.respondTo('info');
            expect(FormulaUtils).itself.to.respondTo('warn');
            expect(FormulaUtils).itself.to.respondTo('error');
        });
    });

    // ========================================================================
});
