/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/combofield', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/textfield',
    'io.ox/office/tk/control/menumixin',
    'io.ox/office/tk/popup/listmenu'
], function (Utils, KeyCodes, Forms, TextField, MenuMixin, ListMenu) {

    'use strict';

    // class ComboField =======================================================

    /**
     * Creates a text field control with attached drop-down list showing
     * predefined values for the text field.
     *
     * @constructor
     *
     * @extends TextField
     * @extends MenuMixin
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the TextField base class
     *  constructor, and the ListMenu class constructor used for the drop-down
     *  menu element. Additionally, the following options are supported:
     *  @param {Boolean} [initOptions.typeAhead=false]
     *      If set to true, the label of the first list item that starts with
     *      the text currently edited will be inserted into the text field.
     *      The remaining text appended to the current text will be selected.
     */
    function ComboField(initOptions) {

        var // self reference
            self = this,

            // search the list items and insert label into text field while editing
            typeAhead = Utils.getBooleanOption(initOptions, 'typeAhead', false),

            // the drop-down button that will be added to the group
            menuButton = $(Forms.createButtonMarkup({ focusable: false })),

            // the drop-down menu instance (must be created after TextField base constructor!)
            menu = null;

        // base constructors --------------------------------------------------

        TextField.call(this, Utils.extendOptions(initOptions, { attributes: { role: 'combobox', 'aria-autocomplete': 'inline' } }));
        menu = new ListMenu(Utils.extendOptions({ expandWidth: true }, initOptions, { anchor: this.getNode(), itemDesign: 'list' }));
        MenuMixin.call(this, menu, { button: menuButton, ariaOwner: this.getInputNode() });

        // private methods ----------------------------------------------------

        /**
         * Returns browser focus to the text field element.
         */
        function returnFocusToTextField() {
            // not on touch devices to prevent flickering keyboard etc.
            if (!Utils.TOUCHDEVICE) {
                // deferred to not confuse focus listeners due to mixed focus events triggered from mouse click and focus() call
                self.executeDelayed(function () {
                    self.getInputNode().focus();
                }, undefined, 'TK: ComboField');
            }
        }

        /**
         * Scrolls the drop-down menu to make the specified list item visible.
         */
        function scrollToListItem(buttonNode) {
            if ((buttonNode.length > 0) && self.isMenuVisible()) {
                menu.scrollToChildNode(buttonNode);
            }
        }

        /**
         * Handles 'popup:show' events, moves the focus to the text field and disables the tooltip.
         */
        function popupShowHandler() {
            Forms.disableToolTip(self.getNode());
            returnFocusToTextField();
            scrollToListItem(menu.getSelectedItemNodes().first());
        }

        /**
         * Handles 'popup:hide' events and enables the tooltip.
         */
        function popupHideHandler() {
            Forms.enableToolTip(self.getNode());
        }

        /**
         * Update handler that activates a list item if it matches the passed
         * value.
         */
        function itemUpdateHandler(value) {

            var // activate a button representing a list item
                selectedButton = menu.selectMatchingItemNodes(value);

            // scroll to make the element visible
            scrollToListItem(selectedButton);
        }

        /**
         * Handles keyboard events in the input control. Moves the active list
         * entry according to cursor keys.
         */
        function inputKeyDownHandler(event) {

            var // item node currently selected
                buttonNode = menu.getSelectedItemNodes().first(),
                // the <input> control
                inputNode = self.getInputNode();

            // open drop-down menu on specific key sequence
            if (KeyCodes.matchKeyCode(event, 'SPACE', { alt: true, ctrl: true })) {
                self.showMenu();
                return;
            }

            // open drop-down menu on specific navigation keys
            switch (event.keyCode) {
            case KeyCodes.UP_ARROW:
            case KeyCodes.DOWN_ARROW:
            case KeyCodes.PAGE_UP:
            case KeyCodes.PAGE_DOWN:
                self.showMenu();
                break;
            }

            // do nothing, if the drop-down menu is not visible
            if (!self.isMenuVisible()) { return; }

            switch (event.keyCode) {
            case KeyCodes.ESCAPE:
                self.hideMenu();
                // Bug 28215: IE needs explicit selection again, otherwise text cursor is hidden
                Forms.setInputSelection(inputNode, inputNode.val().length);
                // let the Group base class not trigger the 'group:cancel' event
                event.preventDefault();
                break;
            case KeyCodes.SPACE:
                // Bug 28208: SPACE key with open drop-down menu and selected list item: trigger change
                if (buttonNode.length > 0) {
                    self.hideMenu();
                    self.triggerChange(Forms.getButtonValue(buttonNode), { sourceEvent: event, preserveFocus: true });
                    // do not insert the space character into the text field
                    return false;
                }
                break;
            }

            // move selection in drop-down list
            buttonNode = menu.getItemNodeForKeyEvent(event, buttonNode);
            if (buttonNode.length > 0) {
                // call the update handler to update the text field and list selection
                self.setValue(Forms.getButtonValue(buttonNode));
                // select entire text field
                inputNode.select();
                return false;
            }
        }

        /**
         * Handler that will be called after the text field has been validated
         * while editing. Will try to insert auto-completion text according to
         * existing entries in the drop-down list.
         */
        function groupValidateHandler(event, prevText) {

            var // the text field element
                inputNode = self.getInputNode(),
                // current text of the text field
                currText = inputNode.val(),
                // current selection of the text field
                selection = Forms.getInputSelection(inputNode),
                // the list item button containing the text of the text field
                buttonNode = $(),
                // the button value
                buttonValue = null,
                // the textual representation of the button value
                buttonText = null;

            // find the first button whose text representation starts with the current text
            buttonNode = menu.getItemNodes().filter(function () {
                var itemText = self.valueToText(Forms.getButtonValue(this));
                return _.isString(itemText) && (itemText.length >= currText.length) &&
                    (itemText.substr(0, currText.length).toLowerCase() === currText.toLowerCase());
            }).first();

            // get value and text representation from the button
            if (buttonNode.length > 0) {
                buttonValue = Forms.getButtonValue(buttonNode);
                buttonText = self.valueToText(buttonValue);
            }

            // try to add the remaining text of an existing list item, but only
            // if the text field does not contain a selection, and something
            // has been appended to the old text
            if (typeAhead && (selection.start === currText.length) &&
                (prevText.length < currText.length) && (prevText === currText.substr(0, prevText.length)) &&
                _.isString(buttonText) && (currText.length < buttonText.length)
            ) {
                inputNode.val(buttonText);
                Forms.setInputSelection(inputNode, currText.length, buttonText.length);
            }

            // select entry in drop-down list, if value (not text representation) is equal
            itemUpdateHandler(self.getFieldValue());
        }

        // methods ------------------------------------------------------------

        /**
         * Adds a new string entry to the drop-down list.
         *
         * @param value
         *  The value to be shown in the drop-down list. Will be converted to
         *  a string using the current validator of the text field.
         *
         * @param {Object} [options]
         *  Additional options for the list entry. Supports all options that
         *  are supported by the method ListMenu.createItemNode(). If the
         *  option 'label' is not specified, the label of the list entry will
         *  be set to the string value provided by the current validator of the
         *  text field.
         *
         * @returns {ComboField}
         *  A reference to this instance.
         */
        this.createListEntry = function (value, options) {
            options = Utils.extendOptions({ label: this.valueToText(value) }, options);
            menu.createItemNode(value, options);
            return this;
        };

        // initialization -----------------------------------------------------

        // add the drop-down button to the group (do not make it focusable)
        this.addChildNodes(menuButton);

        // add special marker class used to adjust formatting
        this.getNode().addClass('combo-field').attr({ role: 'form' });

        // highlight list entry
        this.registerUpdateHandler(itemUpdateHandler);

        // register event handlers
        this.on('group:validate', groupValidateHandler);
        this.getInputNode().on('keydown', inputKeyDownHandler);

        // handle click events in the drop-down list
        menu.getNode().on(Forms.DEFAULT_CLICK_TYPE, Forms.BUTTON_SELECTOR, function (event) {
            self.triggerChange(Forms.getButtonValue(this), { sourceEvent: event });
        });

        // register pop-up event handlers
        menu.on({ 'popup:show': popupShowHandler, 'create:item': function () { self.refresh(); }, 'popup:hide': popupHideHandler });

        // register input control for focus handling (to keep menu visible)
        menu.registerFocusableNodes(this.getInputNode());

        // keep focus in text field when clicking in the drop-down menu
        menu.getNode().on('focusin', returnFocusToTextField);

        // destroy all class members
        this.registerDestructor(function () {
            initOptions = self = menuButton = menu = null;
        });

    } // class ComboField

    // exports ================================================================

    // derive this class from class TextField
    return TextField.extend({ constructor: ComboField });

});
