/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/plaintext/spellchecker', [
    'io.ox/office/text/utils/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/text/dom',
    'io.ox/office/text/position',
    'io.ox/office/text/utils/textutils'
], function (Config, Utils, TriggerObject, DOM, Position, TextUtils) {

    'use strict';

    // class SpellChecker =====================================================

    /**
     * The spell checker object.
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {Object} model
     *  The text editor model.
     */
    function SpellChecker(app, model) {

        var // self reference for local functions
            self = this,
            // online spelling mode off
            onlineSpelling = Config.SPELLING_ENABLED && app.getUserSettingsValue('onlineSpelling'),
            // the scroll node
            scrollNode = null,
            // collects replacements of misspelled words by locale
            spellReplacementsByLocale = {},
            // the cache for the modified paragraphs, that need to be spell checked
            paragraphCache = $();

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Helper function for receiving character styles
         */
        function getCharacterStyles() {
            return model.getStyleCollection('character');
        }

        /**
         * Starting spell checking by calling the worker function 'implCheckSpelling'
         */
        function implStartOnlineSpelling() {
            if (app.isImportFinished() && onlineSpelling === true && model.getEditMode()) { implCheckSpelling(); }
        }

        /**
         * A debounced function for starting spell checking by calling 'implStartOnlineSpelling'
         */
        var implStartOnlineSpellingDebounced = model.createDebouncedMethod($.noop, implStartOnlineSpelling, { delay: 1000, infoString: 'Text: spellchecker.implStartOnlineSpelling' });

        /**
         * Receiving a list of replacement words for a specified word in a specified language from
         * the global replacement collection.
         *
         * @param {String} word
         *  The word to be replaced.
         *
         * @param {String} language
         *  The language of the word to be replaced.
         *
         * @returns {String[]}
         *  An array containing all replacement words for the specified word in the specified language.
         *  If there is no replacement word, the array will be empty.
         */
        function getSpellReplacements(word, language) {
            return spellReplacementsByLocale[language] ? spellReplacementsByLocale[language][word] : [];
        }

        /**
         * Checking, whether two different spell results are identical. A spell check result is an array containing
         * spell check error objects. One object has the following format:
         * Example: {"start":0,"length":2,"replacements":["da","dB","DSL"],"locale":"de_DE","word":"ds"}
         *
         * @param {Object[]} result1
         *  An spell check result container.
         *
         * @param {Object[]} result2
         *  Another spell check result container.
         *
         * @returns {Number}
         *  The number of the first difference. If number is '-1', there is no difference.
         */
        function sameSpellResults(result1, result2) {

            var sameResults = 0,
                counter = 0,
                length1 = result1.length,
                length2 = result2.length,
                minLength = Math.min(length1, length2),
                sameError = minLength > 0 ? true : false;

            // comparing two single result objects
            function compareResultObjects(res1, res2) {
                return res1.start === res2.start && res1.length === res2.length && res1.word === res2.word && res1.locale === res2.locale;
            }

            while (sameError && result1[counter] && result2[counter]) {

                sameError = compareResultObjects(result1[counter], result2[counter]);

                if (sameError) {
                    // comparing the next error objects
                    counter++;
                } else {
                    // finding the first different position
                    sameResults = Math.min(result1[counter].start, result2[counter].start);
                }
            }

            // reaching the end of at least one error array
            if (sameError) {
                if (length1 === length2) {
                    // result arrays are identical
                    sameResults = -1;
                } else {
                    // using the previous error
                    counter--;
                    // increasing the minimum position for the same results
                    sameResults = result1[counter].start + result1[counter].length;
                }
            }

            return sameResults;
        }

        /**
         * Try to merge a span, that is no longer marked with spell error attribute. This can
         * happen for example after inserting a missing letter or after removing a superfluous
         * letter, so that several different text span can be merged.
         *
         * @param {Node[]|jQuery} spans
         *  The list of spans, that need to be checked, if they can be merged with their neighbors.
         */
        function mergeNonSpellErrorSpans(spans) {
            if (spans && _.isNumber(spans.length) && spans.length > 0) {
                _.each(spans, function (span) {
                    if (!DOM.isSpellerrorNode(span)) {
                        TextUtils.mergeSiblingTextSpans(span);
                        TextUtils.mergeSiblingTextSpans(span, true);
                    }
                });
            }
        }

        /**
         * Removing all spell errors inside a paragraph
         *
         * @param {Node|jQuery} paragraph
         *  The DOM node from which the class shall be removed.
         */
        function removeAllSpellErrorsInParagraph(paragraph) {

            var // a container for all spans with spell error
                allSpans = $(paragraph).find('span.spellerror');

            _.each(allSpans, function (span) {
                self.clearSpellErrorAttributes(span); // remove spell attributes - if there are any
            });

            // try to merge with neighbors after removing the spell check attribute
            mergeNonSpellErrorSpans(allSpans);
        }

        /**
         * Helper function that applies the results received from the server to the DOM.
         *
         * @param {Object[]} spellResult
         *  A collection of spell results for the specified paragraph received from the server.
         *
         * @param {Node[]|jQuery} textSpans
         *  The collected text spans inside the specified paragraph.
         *
         * @param {Number} minPos
         *  The minimum position that will be evaluated inside the paragraph.
         */
        function applySpellcheckResults(spellResult, textSpans, minPos) {

            var // a collector for all 'error spans'
                errorSpanArray = [],
                // the offset of the span inside the paragraph
                spanOffset = 0,
                // the length of a text span
                spanLength,
                // a collector for text spans, that can maybe be merged later
                allClearedSpans = [];

            // saving the results in the global collector for all replacements
            _(spellResult).each(function (result) {
                if (result.replacements && result.locale && result.word) {
                    if (!spellReplacementsByLocale[result.locale]) {
                        spellReplacementsByLocale[result.locale] = {};
                    }
                    spellReplacementsByLocale[result.locale][result.word] = result.replacements;
                }
            });

            // iterate over all text spans to generate an array for the error spans
            _(textSpans).each(function (currentSpan) {

                var result,
                    localErrors = [],
                    ignoreError = false;

                spanLength = currentSpan.firstChild.nodeValue.length;

                // ignore all spans at positions left of the minimal position
                if (spanOffset + spanLength < minPos) { ignoreError = true; }

                if (!ignoreError) {

                    // removing an existing spell-error attribute from the text span
                    if ($(currentSpan).hasClass(DOM.SPELLERRORNODE_CLASS)) {
                        // clearing existing spell-error attributes from text spans behind 'minpos'
                        self.clearSpellErrorAttributes(currentSpan);
                        // collecting this span to enable later merging of text spans
                        allClearedSpans.push(currentSpan);
                    }

                    // iterating over all errors, to find precise position inside the current text span
                    for (result = 0; result < spellResult.length; ++result) {

                        var currErrorStart = spellResult[result].start,
                            currErrorLength = spellResult[result].length,
                            maxError = currErrorStart + currErrorLength,
                            minError = currErrorStart;

                        if (maxError <= spanOffset) {
                            continue;
                        }

                        if (minError >= spanOffset + spanLength) {
                            break;
                        }

                        if (minError < spanOffset) {
                            currErrorLength -= spanOffset - minError;
                            minError = 0;
                        }

                        currErrorStart = (minError <= spanOffset) ? 0 : (minError - spanOffset);

                        if (maxError > spanOffset + spanLength) {
                            currErrorLength = spanLength - (currErrorStart - spanOffset);
                        }

                        // saving start and length of error range
                        localErrors.push({ start: currErrorStart, length: currErrorLength });
                    }

                    // collecting all error ranges for the current span
                    if (localErrors.length > 0) { errorSpanArray.push({ span: currentSpan, errors: localErrors }); }
                }

                spanOffset += spanLength;
            });

            // iterating over all error spans, to make them visible
            _.each(errorSpanArray, function (errorSpan) {

                var spanPositionOffset = 0,
                    newSpan,
                    currentSpan = errorSpan.span;

                _.each(errorSpan.errors, function (localError) {

                    newSpan = null;

                    if (localError.start > spanPositionOffset && (localError.start - spanPositionOffset) < currentSpan.textContent.length) {
                        DOM.splitTextSpan(currentSpan, localError.start - spanPositionOffset);
                        spanPositionOffset += localError.start - spanPositionOffset;
                    }

                    // split end of text span NOT covered by the error
                    if (localError.length > 0 && currentSpan.textContent.length > localError.length) {
                        newSpan = DOM.splitTextSpan(currentSpan, localError.length, { append: true });
                        spanPositionOffset += localError.length;
                    }

                    // setting the 'spellerror' attribute to the span, so that it will be underlined
                    getCharacterStyles().setElementAttributes(currentSpan, { character: { spellerror: true } }, { special: true });

                    if (newSpan !== null) { currentSpan = newSpan[0]; }
                });
            });

            // try to merge collected text spans with their neighbors after removing the spell check attribute
            mergeNonSpellErrorSpans(allClearedSpans);
        }

        /**
         * The handler function, that evaluates the results received from the server.
         *
         * @param {Object[]} spellResult
         *  A collection of spell results for the specified paragraph received from the server.
         *
         * @param {Node[]|jQuery} textSpans
         *  The collected text spans inside the specified paragraph.
         *
         * @param {Node} paragraph
         *  The paragraph, to which the spell results will be applied.
         */
        var spellResultHandler = Utils.profileMethod('SpellChecker: Applying spell check results: ', function (spellResult, textSpans, paragraph) {

            var // the first position with a changed spell check error
                firstChangedPos = 0,
                // the spell check results of a previous spell check process
                oldSpellResult = null,
                // the selection object
                selection = model.getSelection();

            // do nothing, if the paragraph was modified in the meantime (before the answer was received)
            if (!$(paragraph).hasClass('p_spelled')) { return; }

            // checking for the old spelling results
            oldSpellResult = $(paragraph).data(SpellChecker.SPELL_RESULT_CACHE);

            // handling empty result array (will be handled in fail handler)
            if (spellResult.length === 0) {
                if (oldSpellResult) {
                    // no errors now, but some errors before
                    removeAllSpellErrorsInParagraph(paragraph);
                    self.clearSpellResultCache(paragraph);
                } else {
                    // no errors now and no errors before
                    return;
                }
            }

            // finding differences to a previous spell check process
            if (oldSpellResult) {
                // do nothing, if the spell results have not changed
                firstChangedPos = sameSpellResults(spellResult, oldSpellResult);
                if (firstChangedPos === -1) { return; }
            }

            // applying the results received from the server
            // -> the array with textSpans is still valid, because 'p_spelled' is still set at the paragraph
            applySpellcheckResults(spellResult, textSpans, firstChangedPos);

            // saving spell results at the paragraph to compare it with the next spell check results
            $(paragraph).data(SpellChecker.SPELL_RESULT_CACHE, spellResult);

            // invalidate an existing cached text point
            selection.setInsertTextPoint(null);

            // restore the selection, but not if a drawing or a table cell is selected (Task 26214)
            if (app.getView().hasAppFocus() && model.isTextOnlySelected()) {
                selection.restoreBrowserSelection({ preserveFocus: true });
            }
        });

        /**
         * The worker function, that checks the paragraphs, generates server calls and
         * applies the response from the server to the paragraphs.
         */
        function implCheckSpelling() {

            var // the vertical offset of the specific node
                topOffset = 0,
                // the maximum offset for paragraphs that will be spell checked (one and a half screen downwards)
                maxOffset = 1.5 * (scrollNode.height() + scrollNode.offset().top),
                // the minimum offset for paragraphs that will be spell checked (a half screen upwards)
                minOffset = -0.3 * maxOffset,
                // iterating over all paragraphs, that are not marked with class 'p_spelled'. If the paragraph cache was filled, use it
                allParagraphs = paragraphCache.length > 0 ? paragraphCache : model.getNode().find(DOM.PARAGRAPH_NODE_SELECTOR).not('.p_spelled');

            // Slicing, so that there are not too many parallel server calls
            // -> it is useful to profile the sliced function using 'Utils.profileMethod' in following performance tests.
            model.iterateArraySliced(allParagraphs, function (paragraph) {

                var // all text spans in the paragraph, as array
                    textSpans = [],
                    // all spans of the paragraph - each span is a word/language object
                    paraSpans = [];

                if (!$(paragraph).hasClass('p_spelled')) {

                    // calculating if this paragraph is in or near the visible region (for performance reasons)
                    topOffset = Utils.round($(paragraph).offset().top, 1);

                    // TODO: This check is not sufficient for long paragraphs
                    if (topOffset < maxOffset && minOffset < topOffset) {

                        // marking paragraph immediately as spell checked, so that directly following changes can be recognized
                        $(paragraph).addClass('p_spelled');

                        // collect all non-empty text spans in the paragraph
                        Position.iterateParagraphChildNodes(paragraph, function (node) {

                            // iterating over all text spans
                            DOM.iterateTextSpans(node, function (span) {
                                var charAttributes = null;
                                if (!DOM.isListLabelNode(span.parentNode)) {
                                    charAttributes = getCharacterStyles().getElementAttributes(span);
                                    if (charAttributes.character.url === '') {
                                        textSpans.push(span);
                                        paraSpans.push({ word: span.firstChild.nodeValue, locale: charAttributes.character.language });
                                    }
                                }
                            });

                        }, undefined, { allNodes: true });

                        // preparing server call for the paragraph, if text spans were found
                        if (paraSpans.length > 0) {

                            app.sendRequest('spellchecker', {
                                action: 'spellparagraph',
                                paragraph: JSON.stringify(paraSpans)
                            }, {
                                method: 'POST',
                                resultFilter: function (data) {
                                    // returning undefined rejects the entire request
                                    return Utils.getArrayOption(data, 'spellResult', undefined, true);
                                }
                            })
                            .done(function (spellResult) {
                                spellResultHandler(spellResult, textSpans, paragraph);
                            })
                            .fail(function (data) {

                                // Handling empty result array (should not be in fail handler)
                                if (data && _.isArray(data.spellResult) && data.spellResult.length === 0) {

                                    if ($(paragraph).data(SpellChecker.SPELL_RESULT_CACHE)) {
                                        // no errors now, but some errors before
                                        removeAllSpellErrorsInParagraph(paragraph);
                                        self.clearSpellResultCache(paragraph);
                                    }
                                }
                            });
                        }
                    }
                }
            }, { delay: 'immediate', infoString: 'Spellchecker: implCheckSpelling' })
            .always(function () {
                paragraphCache = $();
            });
        }

        /**
         * Initialization of spell checker after document is loaded successfully.
         */
        function documentLoaded() {

            if (self.isOnlineSpelling()) { self.setOnlineSpelling(true, true); }

            self.listenTo(model, 'operations:success', implStartOnlineSpellingDebounced);

            // after receiving edit privileges it is necessary to check the document
            self.listenTo(model, 'change:editmode', implStartOnlineSpellingDebounced);

            // setting scroll node and register listener for 'scroll' events
            scrollNode = app.getView().getContentRootNode();
            self.listenTo(scrollNode, 'scroll', implStartOnlineSpellingDebounced);
        }

        // public methods -----------------------------------------------------

        this.getSpellErrorWord = function () {

            var // the selection object
                selection = model.getSelection(),
                // the documents character styles
                characterStyles = getCharacterStyles(),
                // the returning object containing the spellchecker infos
                result = { url: null, spellError: false, word: '', replacements: [] },
                // the selected text span
                span = null,
                // an object containing an expanded word selection
                newSelection = null,
                // the locale of the selected text span
                locale = null,
                // the character attributes of the selected text span
                attributes = null,
                // the logical start position of the selection
                startPosition = null,
                // the node info object at the logical start position
                obj = null,
                // whether the logical start position is at the beginning of the paragraph
                isParaStart = false,
                // Chrome on osx makes a text-range-selection on right mouseclick
                chromeOnOSX = (_.browser.MacOS && _.browser.Chrome);

            if (!selection.hasRange() || chromeOnOSX) {
                // find out a possible URL set for the current position
                startPosition = selection.getStartPosition();
                // on this special case, we need to count up the start-position. Otherwise,
                // the startposition of the selection breaks the detection of wrong spelled words
                if (chromeOnOSX) {
                    startPosition = Position.increaseLastIndex(startPosition);
                }
                obj = Position.getDOMPosition(model.getCurrentRootNode(), startPosition);
                isParaStart = startPosition[startPosition.length - 1] === 0;

                if (!isParaStart && obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {

                    attributes = characterStyles.getElementAttributes(obj.node.parentNode).character;

                    if (attributes.url.length > 0) {
                        // Now we have to check some edge cases to prevent to show
                        // the popup for a paragraph which contains only an empty span
                        // having set the url attribute.
                        span = $(obj.node.parentNode);
                        if ((span.text().length > 0) || (span.next().length > 0) || DOM.isTextSpan(span.next())) {
                            result.url = attributes.url;
                        }
                    } else if ($(obj.node.parentNode).hasClass(DOM.SPELLERRORNODE_CLASS) === true) {
                        // Now we have to check some edge cases to prevent to show
                        // the popup for a paragraph which contains only an empty span
                        // having set the url attribute.
                        span = $(obj.node.parentNode);
                        if ((span.text().length > 0) || (span.prev().length > 0) || DOM.isTextSpan(span.next())) {
                            result.spellError = true;
                        }

                        newSelection = Position.getWordSelection(selection.getEnclosingParagraph(), startPosition[startPosition.length - 1]);

                        if (newSelection) {
                            locale = attributes.language.replace(/-/, '_');
                            result.word = newSelection.text;
                            result.start = newSelection.start;
                            result.end = newSelection.end;

                            result.replacements = getSpellReplacements(result.word, locale);
                        }
                    }
                }
            }

            return result;
        };

        this.replaceWord = function (paraPosition, startEndIndex, replacement) {
            var // the selection object
                selection = model.getSelection(),
                // the undo manager
                undoManager = model.getUndoManager(),

                startPosition = _.clone(paraPosition),
                endPosition = _.clone(paraPosition);

            startPosition[endPosition.length - 1] = startEndIndex.start;
            endPosition[endPosition.length - 1] = startEndIndex.end;

            undoManager.enterUndoGroup(function () {
                selection.setTextSelection(startPosition, endPosition);
                model.deleteSelected();  // this selection can never contain unrecoverable content -> no deferred required
                model.insertText(replacement, selection.getStartPosition(), model.getPreselectedAttributes()); // restoring start position is required!
                selection.setTextSelection(endPosition);
            });
        };

        /**
         *
         */
        this.reset = function (attributes, paragraph) {
            //reset spelling status on language changes
            if (attributes.character && attributes.character.language) {
                this.resetDirectly(paragraph);
            }
        };

        /**
         *
         */
        this.resetClosest = function (newAttributes, element) {
            // invalidate spell result if language attribute changes
            if (newAttributes.character && newAttributes.character.language) {
                this.resetDirectly($(element).closest(DOM.PARAGRAPH_NODE_SELECTOR));
            }
        };

        /**
         * Removing the class marker for spell-checked paragraphs
         *
         * @param {Node|jQuery} paragraph
         *  The DOM node from which the class shall be removed.
         */
        this.resetDirectly = function (paragraph) {
            // reset to 'not spelled'
            $(paragraph).removeClass('p_spelled');
            // ... and caching paragraph for performance reasons
            paragraphCache = paragraphCache.add(paragraph);
        };

        /**
         * Removing classes & runtimeclasses, multi selection possible.
         * Supported are div elements and text spans.
         *
         * @param {jQuery} node
         *  The jQuerified DOM node from which the class shall be removed.
         */
        this.clearSpellcheckHighlighting = function (node) {
            // removing classes, multi selection possible -> no 'else if'
            if (node.is('span.' + DOM.SPELLERRORNODE_CLASS)) {
                // remove spellcheck attribute and class
                self.clearSpellErrorAttributes(node);
            } else if (node.is('div')) {
                // remove runtime classes and cached result data
                node.removeClass('selected p_spelled');
                self.clearSpellResultCache(node);
            }
        };

        /**
         * Removes the character attribute 'spellerror', so that
         * the dotted underline of a text span disappears.
         *
         * @param {Node|jQuery} span
         *  The DOM node from which the attribute shall be removed.
         */
        this.clearSpellErrorAttributes = function (span) {

            var // the character styles of the document
                characterStyles = getCharacterStyles();

            // remove spell attributes - if there are any
            characterStyles.setElementAttributes(span, { character: { spellerror: null } }, { special: true });
        };

        /**
         * Toggle automatic spell checking in the document.
         */
        this.setOnlineSpelling = function (state, noConfig) {
            var rootNode = model.getNode();
            if (state !== onlineSpelling || noConfig === true) {
                onlineSpelling = state;
                if (noConfig !== true) {
                    app.setUserSettingsValue('onlineSpelling', onlineSpelling);
                }
                rootNode.toggleClass('spellerror-visible', onlineSpelling);

                if (onlineSpelling === true) {
                    // deleting the paragraph cache, so that all paragraphs are checked
                    paragraphCache = $();
                    // start timeout handler
                    implStartOnlineSpellingDebounced();
                }
            }
        };

        /**
         * Spellcheck the current selection - works ATM only on selected word
         */
        this.checkSpelling = function () {

            var // the current selection object
                selection = model.getSelection(),
                // the documents character styles
                characterStyles = getCharacterStyles(),
                // the text of the current selection
                selectionText = '',
                // the language, defaulting to en-US
                language = 'en-US',
                // a helper text string
                text = '';

            if (!selection.hasRange()) {
                //TODO: select the word under/nearest to the cursor
            }

            if (this.hasEnclosingParagraph()) {
                // use range to retrieve text and possible url
                if (selection.hasRange()) {

                    // Find out the texts/locales of the selected text to provide them to the
                    // spell checking
                    selection.iterateNodes(function (node, pos, start, length) {
                        if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                            var nodeText = $(node).text();
                            if (nodeText) {
                                selectionText += text.concat(nodeText.slice(start, start + length));
                            }
                            var charAttributes = characterStyles.getElementAttributes(node).character;
                            if (charAttributes.language.length > 0) {
                                language = charAttributes.language;
                            }
                        }
                    });
                }
            }

            app.sendRequest('spellchecker', {
                action: 'spellwordreplacements',
                word: encodeURIComponent(selectionText),
                locale: language
            }, {
                resultFilter: function (data) {
                    // returning undefined rejects the entire request
                    return Utils.getArrayOption(data, 'SpellReplacements', undefined, true);
                }
            })
            .done(function (spellReplacements) {
                Utils.log(selectionText + ': ' + spellReplacements.join(' '));
            });
        };

        /**
         * Checking, if spell checking is enabled.
         *
         * @return {Boolean}
         *  Whether spellcheck in config and in user-hud is enabled
         */
        this.isOnlineSpelling = function () {
            return onlineSpelling;
        };

        /**
         * Checking, whether a specified paragraph has cached spell results
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node, that is checked for existing spell results.
         *
         * @return {Boolean}
         *  Whether a specified paragraph has cached spell results
         */
        this.hasSpellResultCache = function (paragraph) {
            return $(paragraph).data(SpellChecker.SPELL_RESULT_CACHE);
        };

        /**
         * Clearing the spell result cache at a specified paragraph
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node, whose cached spell results will be removed.
         */
        this.clearSpellResultCache = function (paragraph) {
            $(paragraph).removeData(SpellChecker.SPELL_RESULT_CACHE);
        };

        // initialization -----------------------------------------------------

        app.onImportSuccess(documentLoaded);

    } // class SpellChecker

    // constants --------------------------------------------------------------

    /**
     * The name of the jQuery data key that is used as connection between a comment thread
     * and its bubble node.
     */
    SpellChecker.SPELL_RESULT_CACHE = 'spellresult';

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: SpellChecker });

});
