/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/text/model/numberFormatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/formatter'
], function (Utils, IO, DateUtils, Formatter) {

    'use strict';

    // class NumberFormatter ==================================================

    /**
     * A number formatter for a text document.
     *
     * @constructor
     *
     * @extends Formatter
     */
    function NumberFormatter(docModel) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        Formatter.call(this);

        // public methods -----------------------------------------------------

        /**
         * Returns an object containing the date components of the passed date
         * value as single numeric properties.
         *
         * @param {Date} date
         *  The date to be converted, as UTC or local date value.
         *
         * @param {Boolean} [local=false]
         *  Whether to extract the UTC date components (false, default), or the
         *  local date components (true).
         *
         * @returns {Object}
         *  The date components, in the properties 'Y' (full year), 'M' (zero-based
         *  month), and 'D' (one-based day in month).
         */
        this.getDateComponents = function (date, local) {
            return {
                Y: local ? date.getFullYear() : date.getUTCFullYear(),
                M: local ? date.getMonth() : date.getUTCMonth(),
                D: local ? date.getDate() : date.getUTCDate()
            };
        };

        /**
         * Returns an object containing the date and time components of the passed
         * date value as single numeric properties.
         *
         * @param {Date} date
         *  The date to be converted, as UTC or local date value.
         *
         * @param {Boolean} [local=false]
         *  Whether to extract the UTC date/time components (false, default), or
         *  the local date/time components (true).
         *
         * @returns {Object}
         *  The date and time components, in the properties 'Y' (full year), 'M'
         *  (zero-based month), 'D' (one-based day in month), 'h' (hours), 'm'
         *  (minutes), 's' (seconds), and 'ms' (milliseconds).
         */
        this.getDateTimeComponents = function (date, local) {
            var comps = self.getDateComponents(date, local);
            comps.h = local ? date.getHours() : date.getUTCHours();
            comps.m = local ? date.getMinutes() : date.getUTCMinutes();
            comps.s = local ? date.getSeconds() : date.getUTCSeconds();
            comps.ms = local ? date.getMilliseconds() : date.getUTCMilliseconds();
            return comps;
        };

        /**
         * Creates a UTC date value from the specified date and time components.
         *
         * @param {Object} comps
         *  The date/time components, in the properties 'Y', 'M', 'D', 'h', 'm',
         *  's', and 'ms', as returned by the function getDateTimeComponents().
         *
         * @returns {Date}
         *  The UTC date value representing the specified date and time.
         */
        this.makeDateTime = function (comps) {
            return new Date(Date.UTC(comps.Y, comps.M, comps.D, comps.h, comps.m, comps.s, comps.ms));
        };

        /**
         * Creates a UTC date value from the specified date components.
         *
         * @param {Object} comps
         *  The date components, in the properties 'Y', 'M', and 'D', as returned
         *  by the function getDateComponents().
         *
         * @returns {Date}
         *  The UTC date value representing the specified date. The time will be
         *  set to midnight.
         */
        this.makeDate = function (comps) {
            return new Date(Date.UTC(comps.Y, comps.M, comps.D, 0, 0, 0, 0));
        };

        /**
         * Converts decimal into Roman number format.
         *
         * @param {Number} number
         *  Decimal number.
         *
         * @returns {String} romanNumber
         *  Converted decimal number into string with Roman formatting.
         *
         */
        this.toRoman = function (number) {
            var romanNumber = '',
                decimalTemplates = [1000, 900, 500, 400, 100, 90, 50, 40, 10, 9, 5, 4, 1],
                dtLength = decimalTemplates.length,
                romanLetters = ['M', 'CM', 'D', 'CD', 'C', 'XC', 'L', 'XL', 'X', 'IX', 'V', 'IV', 'I'];

            for (var i = 0; i < dtLength; i++) {
                while (number >= decimalTemplates[i]) {
                    romanNumber += romanLetters[i];
                    number -= decimalTemplates[i];
                }
            }
            return romanNumber;
        };

        /**
         * Converts decimal into alphabetic letter format.
         *
         * @param {Number} number
         *  Decimal number.
         *
         * @returns {String}
         *  Converted decimal number into string with alphabetic formatting.
         */
        this.toAlphabetic = function (number) {
            var alphabet = '',
                multiAlphabet = '',
                charArray = 'abcdefghijklmnopqrstuvwxyz',
                factor = null;

            if (!_.isNumber(number) || number < 1) {
                Utils.error('numberFormatter.toAlphabetic(): argument is not valid number: ', number);
                return;
            }
            if (number > 26) {
                factor = ~~(number / 26);
                number = number % 26;
            }
            alphabet = charArray[number - 1]; // Zero based index of charArray!

            if (factor) {
                for (var i = 0; i <= factor; i++) {
                    multiAlphabet += alphabet;
                }
                return multiAlphabet;
            }
            return alphabet;
        };

        /**
         * Formats decimal number to have one dash before and one after number, separated by one whitespace.
         * Example: 1 to - 1 -
         *
         * @param {Number} number
         *  Decimal number.
         *
         * @returns {String}
         *  Converted decimal number into string with dash formatting.
         */
        this.toArabicDash = function (number) {
            return '- ' + number + ' -';
        };

        /**
         * Converts decimal number to hexadecimal.
         *
         * @param {Number} number
         *  Number to be converted to hexadecimal.
         * @returns {String|null}
         *  Converted hex string.
         */
        this.decToHex = function (number) {
            if (!_.isNumber(number)) {
                Utils.error('numberFormatter.decToHex(): argument is not valid number: ', number);
                return;
            }
            return number.toString(16);
        };

        /**
         * Converts hexadecimal string to decimal number.
         *
         * @param {String} hexString
         *  String to be converted into decimal.
         * @returns {Number}
         *  Converted hex string to decimal number.
         */
        this.hexToDec = function (hexString) {
            return parseInt(hexString, 16);
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = self = null;
        });

    } // class NumberFormatter

    // exports ================================================================

    return Formatter.extend({ constructor: NumberFormatter });

});
