/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/selectionrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, TriggerObject, TimerMixin, Config, SheetUtils, PaneUtils, RenderUtils) {

    'use strict';

    var // convenience shortcuts
        Range = SheetUtils.Range;

    // class SelectionRenderer ================================================

    /**
     * Renders the own selection, and the selections of remote users into the
     * DOM layers of a single grid pane.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this selection layer renderer.
     */
    function SelectionRenderer(gridPane) {

        var // self reference
            self = this,

            // the spreadsheet model and view
            docView = gridPane.getDocView(),
            docModel = docView.getDocModel(),

            // the remote layer node for displaying selections of other users
            remoteLayerNode = gridPane.createLayerNode('remote-layer'),

            // the selection layer (container for the selected ranges)
            selectionLayerNode = gridPane.createLayerNode('selection-layer'),

            // the cell range and absolute position covered by the layer nodes in the sheet area
            layerRange = null,
            layerRectangle = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Updates the position of the auto-fill handle and the resize handles
         * for cell selection, when entire columns or rows are selected.
         */
        function updateResizeHandlePositions() {

            var // the auto-fill resizer handle node
                autoFillHandleNode = selectionLayerNode.find('.autofill.resizers>[data-pos]'),
                // the range node containing the auto-fill handle
                autoFillRangeNode = autoFillHandleNode.closest('.range'),
                // visible area of the grid pane
                visibleRect = gridPane.getVisibleRectangle();

            // converts the passed offset in the visible area to an offset relative to 'parentNode'
            function setRelativeOffset(targetNode, offsetName, parentNode, visibleOffset) {
                var offset = visibleRect[offsetName] - layerRectangle[offsetName] - parentNode.position()[offsetName] + visibleOffset;
                targetNode.css(offsetName, offset + 'px');
            }

            // adjust position of the auto-fill handle node for entire column or row selection
            switch (autoFillHandleNode.attr('data-pos')) {
            case 'r':
                setRelativeOffset(autoFillHandleNode, 'top', autoFillRangeNode, 0);
                break;
            case 'b':
                setRelativeOffset(autoFillHandleNode, 'left', autoFillRangeNode, 0);
                break;
            }

            // adjust position of the selection handle node for entire column or row selection
            selectionLayerNode.find('.select.resizers>[data-pos]').each(function () {

                var handleNode = $(this),
                    rangeNode = handleNode.closest('.range');

                switch (handleNode.attr('data-pos')) {
                case 'l':
                case 'r':
                    setRelativeOffset(handleNode, 'top', rangeNode, visibleRect.height / 2);
                    break;
                case 't':
                case 'b':
                    setRelativeOffset(handleNode, 'left', rangeNode, visibleRect.width / 2);
                    break;
                }
            });
        }

        /**
         * Renders all visible selection ranges according to the current pane
         * layout data received from a view layout update notification.
         */
        function renderCellSelection() {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) {
                selectionLayerNode.empty();
                return;
            }

            var // the entire cell selection
                selection = docView.getSelection(),
                // the range occupied by the active cell (will not be filled)
                activeRange = null,
                // the position of the active cell, extended to the merged range
                activeRectangle = null,
                // additional data for active auto-fill tracking
                autoFillData = docView.getSheetViewAttribute('autoFillData'),
                // whether the active cell touches the borders of selection ranges
                activeBorders = { left: false, top: false, right: false, bottom: false },
                // hide left/top border of selection ranges, if first column/row is hidden
                firstColHidden = !docView.getColCollection().isEntryVisible(0),
                firstRowHidden = !docView.getRowCollection().isEntryVisible(0),
                // the HTML mark-up for all selection ranges but the active range
                rangesMarkup = '',
                // the HTML mark-up for the active selection range
                activeMarkup = '';

            // returns the mark-up of a resizer handler according to the type of the passed range
            function createResizerMarkup(range, leadingCorner) {
                var hPos = leadingCorner ? 'l' : 'r', vPos = leadingCorner ? 't' : 'b';
                return '<div data-pos="' + (docModel.isColRange(range) ? hPos : docModel.isRowRange(range) ? vPos : (vPos + hPos)) + '"></div>';
            }

            // add active flag to range object
            selection.ranges[selection.active].active = true;

            // adjust single range for auto-fill tracking
            if ((selection.ranges.length === 1) && _.isObject(autoFillData)) {

                var // the selected cell range
                    firstRange = selection.ranges.first(),
                    // whether to expand/shrink the leading or trailing border
                    leading = /^(left|top)$/.test(autoFillData.border),
                    // whether to expand/shrink columns or rows
                    columns = /^(left|right)$/.test(autoFillData.border);

                activeRange = firstRange.clone();
                if (autoFillData.count >= 0) {
                    // adjust range for auto-fill mode
                    if (leading) {
                        firstRange.start.move(-autoFillData.count, columns);
                    } else {
                        firstRange.end.move(autoFillData.count, columns);
                    }
                } else {
                    // adjust range for deletion mode
                    firstRange.collapsed = true;
                    if (-autoFillData.count < firstRange.size(columns)) {
                        // range partly covered
                        if (leading) {
                            activeRange.start.move(-autoFillData.count, columns);
                        } else {
                            activeRange.end.move(autoFillData.count, columns);
                        }
                    } else {
                        // special marker for deletion of the entire range
                        activeRange = Utils.BREAK;
                    }
                }

                // add tracking style effect
                firstRange.tracking = true;
            }

            // the range covered by the active cell (or any other range in auto-fill tracking mode)
            activeRange = (activeRange === Utils.BREAK) ? null : activeRange ? activeRange :
                docView.getMergeCollection().expandRangeToMergedRanges(new Range(selection.address));

            // convert active range to a rectangle relative to the layer root node
            if (_.isObject(activeRange)) {
                gridPane.iterateRangesForRendering(activeRange, function (range, rectangle) {
                    activeRectangle = rectangle;
                }, { alignToGrid: true });
            }

            // render all ranges that are visible in this grid pane
            gridPane.iterateRangesForRendering(selection.ranges, function (range, rectangle, index) {

                var // position of active range, relative to current range
                    relActiveRect = null,
                    // mark-up for the current range
                    rangeMarkup = '';

                // hide left/top border of the range, if the first column/row is hidden
                if (firstColHidden && (range.start[0] === 0)) {
                    rectangle.left -= 5;
                    rectangle.width += 5;
                }
                if (firstRowHidden && (range.start[1] === 0)) {
                    rectangle.top -= 5;
                    rectangle.height += 5;
                }

                // generate the HTML mark-up for the range
                rangeMarkup = '<div class="range';
                if (range.active) { rangeMarkup += ' active'; }
                if (range.tracking) { rangeMarkup += ' tracking-active'; }
                if (range.collapsed) { rangeMarkup += ' collapsed'; }
                rangeMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '" data-index="' + index + '">';

                // insert the semi-transparent fill elements (never cover the active range with the fill elements)
                if (activeRectangle && range.overlaps(activeRange)) {

                    // initialize position of active cell, relative to selection range
                    relActiveRect = _.clone(activeRectangle);
                    relActiveRect.left -= rectangle.left;
                    relActiveRect.top -= rectangle.top;
                    relActiveRect.right = rectangle.width - relActiveRect.left - relActiveRect.width;
                    relActiveRect.bottom = rectangle.height - relActiveRect.top - relActiveRect.height;

                    // insert fill element above active cell
                    if (relActiveRect.top > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:0;height:' + relActiveRect.top + 'px;"></div>';
                    }

                    // insert fill element left of active cell
                    if (relActiveRect.left > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;width:' + relActiveRect.left + 'px;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element right of active cell
                    if (relActiveRect.right > 0) {
                        rangeMarkup += '<div class="fill" style="left:' + (relActiveRect.left + relActiveRect.width) + 'px;right:0;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element below active cell
                    if (relActiveRect.bottom > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:' + (relActiveRect.top + relActiveRect.height) + 'px;bottom:0;"></div>';
                    }

                    // update border flags for the active cell
                    activeBorders.left = activeBorders.left || (range.start[0] === activeRange.start[0]);
                    activeBorders.top = activeBorders.top || (range.start[1] === activeRange.start[1]);
                    activeBorders.right = activeBorders.right || (range.end[0] === activeRange.end[0]);
                    activeBorders.bottom = activeBorders.bottom || (range.end[1] === activeRange.end[1]);

                } else {
                    rangeMarkup += '<div class="abs fill"></div>';
                }

                // generate the HTML mark-up for the selection range
                rangeMarkup += '<div class="border"></div>';

                // additional mark-up for single-range selection
                if (selection.ranges.length === 1) {
                    if (Utils.TOUCHDEVICE && !_.isObject(autoFillData)) {
                        // add resize handlers for selection on touch devices
                        rangeMarkup += '<div class="select resizers">' + createResizerMarkup(range, true) + createResizerMarkup(range, false) + '</div>';
                    } else {
                        // add the auto-fill handler in the bottom right corner of the selection
                        rangeMarkup += '<div class="autofill resizers">' + createResizerMarkup(range, false) + '</div>';
                    }
                }

                // close the range node
                rangeMarkup += '</div>';

                // and the generated mark-up to the appropriate mark-up list
                if (range.active) { activeMarkup += rangeMarkup; } else { rangesMarkup += rangeMarkup; }
            }, { alignToGrid: true });

            // additions for the active cell (or active range in auto-fill)
            if (activeRectangle) {

                // add thin border for active cell on top of the selection
                rangesMarkup += '<div class="active-cell';
                _.each(activeBorders, function (isBorder, borderName) { if (!isBorder) { rangesMarkup += ' ' + borderName; } });
                rangesMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(activeRectangle) + '"></div>';
            }

            // insert entire HTML mark-up into the selection container node
            selectionLayerNode[0].innerHTML = rangesMarkup + activeMarkup;

            // update the visibility/position of additional DOM elements
            updateResizeHandlePositions();

            // notify listeners for additional rendering depending on the selection
            self.trigger('render:cellselection');
        }

        /**
         * Debounced version of the method renderCellSelection().
         */
        var renderCellSelectionDebounced = this.createDebouncedMethod($.noop, renderCellSelection, { delay: 100 });

        /**
         * Renders selections of all remote users which have the same sheet
         * activated.
         */
        var renderRemoteSelections = Config.SHOW_REMOTE_SELECTIONS ? function () {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) {
                remoteLayerNode.empty();
                return;
            }

            var // prepare equal mark-up in all selection ranges
                borderMarkup = '<div class="borders">' + _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('') + '</div>',
                // top-left visible address of the grid pane, to detect if the name-layer outside of the view
                firstRow = gridPane.getTopLeftAddress()[1],
                // the HTML mark-up for all user selections
                markup = '';

            // generate selections of all remote users on the same sheet
            docModel.getViewAttribute('remoteClients').forEach(function (client) {

                var // the JSON selection data of the remote client
                    selection = client.userData.selection;

                // skip users that have another sheet activated
                if (client.userData.sheet !== docModel.getActiveSheet()) { return; }

                var // the escaped user name, ready to be inserted into HTML mark-up
                    userName = Utils.escapeHTML(_.noI18n(client.userName)),
                    // create the array of cell range addresses
                    ranges = docModel.createRangeArray(selection.ranges);

                // render all ranges of the user
                gridPane.iterateRangesForRendering(ranges, function (range, rectangle) {

                    var // the CSS classes to be set at the range (bug 35604: inside range if on top of the grid pane)
                        classes = 'range' + ((range.start[1] === firstRow) ? ' badge-inside' : '');

                    // create the mark-up for the range
                    markup += '<div class="' + classes + '" data-style="' + client.colorIndex + '" data-username="' + userName + '"';
                    markup += ' style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '">' + borderMarkup + '</div>';
                }, { alignToGrid: true });
            });

            // insert entire HTML mark-up into the container node
            remoteLayerNode[0].innerHTML = markup;
        } : $.noop;

        /**
         * Updates the layer nodes after specific document view attributes have
         * been changed.
         */
        function changeViewAttributesHandler(event, attributes) {

            // display remote selections
            if ('remoteClients' in attributes) {
                renderRemoteSelections();
            }
        }

        /**
         * Updates the layer nodes after specific sheet view attributes have
         * been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // render cell selection (triggers a 'render:cellselection' event)
            if (Utils.hasProperty(attributes, /^(selection$|split|autoFillData$)/)) {
                renderCellSelection();
            }

            // trigger a 'render:cellselection' event if active pane changes
            // (no actual rendering needed, border colors will change via CSS)
            if ('activePane' in attributes) {
                self.trigger('render:cellselection');
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Changes the layers according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('SelectionRenderer.setLayerRange()', function (newLayerRange, newLayerRectangle) {

            // store new layer range and rectangle for convenience
            layerRange = newLayerRange;
            layerRectangle = newLayerRectangle;

            // render own selection, and selections of all remote editors
            renderCellSelection();
            renderRemoteSelections();
        });

        /**
         * Resets this renderer, clears the DOM layer nodes.
         */
        this.hideLayerRange = function () {
            layerRange = layerRectangle = null;
            selectionLayerNode.empty();
            remoteLayerNode.empty();
        };

        // initialization -----------------------------------------------------

        // render cell selection again when document edit mode changes
        gridPane.listenToWhenVisible(this, docModel, 'change:editmode', renderCellSelection);

        // render cell selection after merging/unmerging cell ranges (size of active cell may change)
        gridPane.listenToWhenVisible(this, docView, 'insert:merged delete:merged', renderCellSelectionDebounced);

        // update layers according to changed view attributes (only, if this grid pane is visible)
        gridPane.listenToWhenVisible(this, docModel, 'change:viewattributes', changeViewAttributesHandler);
        gridPane.listenToWhenVisible(this, docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

        // post-processing after scrolling, or after selection has been rendered
        this.listenTo(gridPane, 'change:scrollpos', updateResizeHandlePositions);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = gridPane = docModel = docView = null;
            remoteLayerNode = selectionLayerNode = null;
        });

    } // class SelectionRenderer

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: SelectionRenderer });

});
