/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/highlightmixin', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // class HighlightRangeDescriptor =========================================

    /**
     * A descriptor object for a highlighted cell range in the document view.
     *
     * @constructor
     *
     * @property {Range3D} range
     *  The cell range address of the highlighted range, with sheet indexes.
     *
     * @property {Number} arrayIndex
     *  Internal array index of the token array containing this highlighted
     *  range.
     *
     * @property {Number} tokenIndex
     *  Internal index of the formula token containing this highlighted range,
     *  in the token array specified by the property 'arrayIndex'. Different
     *  highlighted ranges may origin from the same formula token, e.g. from a
     *  defined name in the formula containing multiple range addresses.
     *
     * @property {String} tokenKey
     *  A specific identifier of the originating formula token, built from the
     *  token array index (property 'arrayIndex'), and the token index
     *  (property 'tokenIndex').
     *
     * @property {Boolean} draggable
     *  Whether this highlighted range is intended to be movable and resizable
     *  with mouse or touch gestures. Needed to render the highlighted range
     *  correctly.
     *
     * @property {Boolean} tracking
     *  Whether this highlighted range is currently dragged around in the GUI.
     *  The range will be rendered in a special way to indicate its special
     *  state.
     */
    function HighlightRangeDescriptor(rangeInfo, arrayIndex, draggable) {

        // the range address, with sheet indexes
        this.range = rangeInfo.range;

        // array index of the token array containing the highlighted range
        this.arrayIndex = arrayIndex;

        // index of the formula token containing the highlighted range
        this.tokenIndex = rangeInfo.index;

        // unique identifier of the originating formula token
        this.tokenKey = arrayIndex + ',' + rangeInfo.index;

        // ranges of defined names are not draggable
        this.draggable = draggable && (rangeInfo.type === 'ref');

        // the range may be tracked in the GUI currently
        this.tracking = false;

    } // class HighlightRangeDescriptor

    // mix-in class HighlightMixin ============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides extensions for
     * showing and manipulating highlighted cell ranges in the active sheet
     * based on specific formula token arrays. Highlighted cell ranges will be
     * rendered beside the regular cell selection. They are used for example in
     * formula edit mode to visualize the ranges and defined names used in the
     * formula, or to show the source ranges of a selected chart object.
     *
     * @constructor
     */
    function HighlightMixin() {

        var // self reference (the document view)
            self = this,

            // the unique identifier of the current highlighting mode
            highlightUid = null,

            // whether the current highlighting mode has high priority (cannot be overridden)
            highPriority = false,

            // the token arrays currently used for highlighted ranges
            highlightTokenArrays = [],

            // all options used to highlight cell ranges
            highlightOptions = null,

            // range descriptors representing all highlighted ranges
            highlightRanges = [],

            // array index of the range currently tracked
            trackedRangeIndex = -1;

        // private methods ----------------------------------------------------

        /**
         * Handles change events from any of the token arrays, and notifies the
         * own event listeners.
         */
        function changeTokenArrayHandler() {

            // whether dragging the cell ranges is enabled
            var draggable = Utils.getBooleanOption(highlightOptions, 'draggable', false);

            // process all registered token arrays
            highlightRanges = [];
            highlightTokenArrays.forEach(function (tokenArray, arrayIndex) {

                // array of range info objects from the current token array
                var rangeInfos = tokenArray.extractRanges(highlightOptions);

                // append highlight descriptor objects to the resulting array
                rangeInfos.forEach(function (rangeInfo) {
                    highlightRanges.push(new HighlightRangeDescriptor(rangeInfo, arrayIndex, draggable));
                });
            });

            // restore the tracking flag
            var rangeData = highlightRanges[trackedRangeIndex];
            if (rangeData) { rangeData.tracking = true; }

            // notify change listeners
            self.trigger('change:highlight:ranges');
        }

        /**
         * Registers or unregisters the change listener at all current token
         * arrays.
         */
        function registerTokenArrayListeners(enable) {
            _.invoke(highlightTokenArrays, enable ? 'on' : 'off', 'triggered', changeTokenArrayHandler);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether the range highlighting mode is currently active.
         *
         * @returns {Boolean}
         *  Whether the range highlighting mode is currently active.
         */
        this.isRangeHighlightingActive = function () {
            return _.isString(highlightUid);
        };

        /**
         * Registers token arrays used to render highlighted cell ranges in the
         * active sheet.
         *
         * @param {Array<TokenArray>|TokenArray} tokenArrays
         *  An array of token arrays, or a single token array instance, to be
         *  used for range highlighting.
         *
         * @param {Boolean} [options]
         *  Optional parameters. Supports all options also supported by the
         *  method TokenArray.extractRanges(), and the following options:
         *  @param {Boolean} [options.priority=false]
         *      If set to true, the range highlighting mode will be started as
         *      high-priority mode, meaning that no other range highlighting
         *      mode can be started as long as it is active. If another range
         *      highlighting mode with high priority is currently running when
         *      invoking this method, it will not do anything and will return
         *      the value null.
         *  @param {Boolean} [options.draggable=false]
         *      If set to true, the cell ranges representing reference tokens
         *      will be marked as draggable. Draggable ranges will be rendered
         *      with additional GUI handle elements that can be used to move or
         *      resize the respective range.
         *
         * @returns {String|Null}
         *  A unique identifier for the highlighting mode started with this
         *  method call. This identifier needs to be passed to the method
         *  HighlighMixin.endRangeHighlighting() in order to prevent canceling
         *  a highlighting mode started from another part of the code. If a
         *  range highlighting mode with high priority is currently active (see
         *  option 'priority' above), null will be returned instead.
         */
        this.startRangeHighlighting = function (tokenArrays, options) {

            // do not break active highlighting mode with high priority
            if (this.isRangeHighlightingActive() && highPriority) {
                return null;
            }

            // store tracking options for usage in event handlers
            highlightOptions = _.clone(options);

            // unregister event handlers at old token arrays
            registerTokenArrayListeners(false);

            // create a unique identifier for the new highlighting mode
            highlightUid = _.uniqueId('highlight');
            highPriority = Utils.getBooleanOption(options, 'priority', false);

            // store the token arrays
            highlightTokenArrays = _.getArray(tokenArrays).slice(0);
            registerTokenArrayListeners(true);

            // notify initial ranges to listeners (e.g. renderer)
            changeTokenArrayHandler();
            return highlightUid;
        };

        /**
         * Leaves the specified range highlighting mode.
         *
         * @param {String|Null} uid
         *  The unique identifier of the range highlighting mode returned by
         *  the method HighlighMixin.startRangeHighlighting(). If another
         *  highlighting mode (with another unique identifier) has been started
         *  in the meantime, this method will do nothing.
         *
         * @returns {Boolean}
         *  Whether the specified highlighting mode was still active, and has
         *  been left successfully.
         */
        this.endRangeHighlighting = function (uid) {

            // do nothing if the passed identifier does not match
            if (!uid || (uid !== highlightUid)) { return false; }

            // unregister event handlers at old token arrays
            registerTokenArrayListeners(false);

            // reset the identifier and token arrays
            highlightUid = null;
            highlightTokenArrays = [];

            // notify listeners (e.g. renderer)
            changeTokenArrayHandler();
            return true;
        };

        /**
         * Returns the cell range address of the specified highlighted range.
         *
         * @param {Number} rangeIndex
         *  The array index of the highlighted range.
         *
         * @returns {Range3D|Null}
         *  The cell range address of the specified highlighted range, if it
         *  exists; otherwise null.
         */
        this.getHighlightedRange = function (rangeIndex) {
            var rangeData = highlightRanges[rangeIndex];
            return rangeData ? rangeData.range : null;
        };

        /**
         * Invokes the passed callback function for all highlighted ranges.
         *
         * @param {Function} callback
         *  The callback function that will be invoked for every highlighted
         *  range. Receives the following parameters:
         *  (1) {HighlightRangeDescriptor} rangeDesc
         *      A descriptor object containing all properties of the
         *      highlighted range.
         *  (2) {Number} rangeIndex
         *      The array index of the highlighted range.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {HighlightMixin}
         *  A reference to this instance.
         */
        this.iterateHighlightedRanges = function (callback, context) {
            highlightRanges.forEach(callback, context);
            return this;
        };

        /**
         * Marks the specified highlighted range as the tracked range, used
         * when moving or resizing a highlighted range in the GUI. The tracked
         * range will be rendered with a special glow effect.
         *
         * @param {Number} rangeIndex
         *  The array index of the highlighted range to be tracked, or -1 to
         *  disable the tracking mode.
         *
         * @returns {HighlightMixin}
         *  A reference to this instance.
         */
        this.trackHighlightedRange = function (rangeIndex) {

            // nothing to do, if the passed index does not differ from current index
            if (trackedRangeIndex === rangeIndex) { return this; }

            // reset tracking flag in old highlighted range
            var rangeData = highlightRanges[trackedRangeIndex];
            if (rangeData) { rangeData.tracking = false; }
            trackedRangeIndex = rangeIndex;

            // set tracking flag in new highlighted range
            rangeData = highlightRanges[trackedRangeIndex];
            if (rangeData) { rangeData.tracking = true; }

            // notify change listeners
            this.trigger('change:highlight:ranges');
            return this;
        };

        /**
         * Returns whether a highlighted range is currently marked as tracking
         * range (see method HighlightMixin.trackHighlightedRange() for more
         * details).
         *
         * @returns {Boolean}
         *  Whether a highlighted range is currently marked as tracking range.
         */
        this.hasTrackedHighlightedRange = function () {
            return trackedRangeIndex in highlightRanges;
        };

        /**
         * Modifies the cell range address of a single reference token inside a
         * single token array currently registered for range highlighting.
         *
         * @param {Number} rangeIndex
         *  The array index of a highlighted range.
         *
         * @param {Range} range
         *  The new cell range address to be inserted into the reference token.
         *  The absolute/relative state of the reference components will not be
         *  changed.
         *
         * @returns {HighlightMixin}
         *  A reference to this instance.
         */
        this.changeHighlightedRange = function (rangeIndex, range) {

            var // the range descriptor containing all information to find the formula token to modify
                rangeData = highlightRanges[rangeIndex],
                // the token array instance
                tokenArray = rangeData ? highlightTokenArrays[rangeData.arrayIndex] : null;

            // modify the reference token, if the indexes are valid
            if (tokenArray) {
                tokenArray.modifyToken(rangeData.tokenIndex, 'ref', function (refToken) {
                    return refToken.setRange(range);
                });
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            registerTokenArrayListeners(false);
            self = highlightTokenArrays = highlightOptions = highlightRanges = null;
        });

    } // class HighlightMixin

    // exports ================================================================

    return HighlightMixin;

});
