/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/gridpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/clipboard',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/popup/cellcontextmenu',
    'io.ox/office/spreadsheet/view/popup/drawingcontextmenu',
    'io.ox/office/spreadsheet/view/popup/tablecolumnmenu',
    'io.ox/office/spreadsheet/view/popup/validationlistmenu',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/cellrenderer',
    'io.ox/office/spreadsheet/view/render/selectionrenderer',
    'io.ox/office/spreadsheet/view/render/drawingrenderer',
    'io.ox/office/spreadsheet/view/render/formrenderer',
    'io.ox/office/spreadsheet/view/render/highlightrenderer',
    'io.ox/office/spreadsheet/view/mixin/gridtrackingmixin',
    'io.ox/office/spreadsheet/view/chartcreator',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, KeyCodes, Forms, Tracking, TriggerObject, TimerMixin, ImageUtil, SheetUtils, PaneUtils, Clipboard, Labels, CellContextMenu, DrawingContextMenu, TableColumnMenu, ValidationListMenu, RenderUtils, CellRenderer, SelectionRenderer, DrawingRenderer, FormRenderer, HighlightRenderer, GridTrackingMixin, ChartCreator, gt) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Range = SheetUtils.Range,
        RangeArray = SheetUtils.RangeArray,

        // Bug 41353: Chrome 45 cannot render border lines around very large nodes
        MAX_RECTANGLE_SIZE = 100000;

    // class GridPane =========================================================

    /**
     * Represents a single scrollable grid pane in the spreadsheet view. If the
     * view has been split or frozen at a specific cell position, the view will
     * consist of up to four panes (top-left, top-right, bottom-left, and
     * bottom-right pane).
     *
     * Instances of this class trigger the following events:
     * - 'change:scrollpos':
     *      After the scroll position of the grid pane has been changed. Event
     *      listeners may want to update the position of additional DOM content
     *      inserted into the grid pane.
     * - 'render:cellselection'
     *      After the current cell selection has actually been rendered into
     *      the DOM selection layer node (rendering may happen debounced after
     *      several 'change:selection' view events).
     * - 'render:drawingselection'
     *      After the current drawing selection has actually been rendered into
     *      the DOM drawing layer (rendering may happen debounced after several
     *      'change:selection' view events). Event handlers receive the DOM
     *      drawing frames currently selected in this grid pane, as jQuery
     *      collection.
     * - 'select:start'
     *      After selection tracking has been started. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Address} address
     *          The address of the start cell.
     *      (3) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys.
     *          - 'append': Append new range to current selection (CTRL/META).
     *          - 'extend': Extend current active range (SHIFT).
     * - 'select:move'
     *      After the address of the tracked cell has changed while selection
     *      tracking is active. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Address} address
     *          The address of the current cell.
     * - 'select:end'
     *      After selection tracking has been finished (either successfully or
     *      not). Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Address} address
     *          The address of the last cell.
     *      (3) {Boolean} success
     *          True if selecting has finished successfully, false if selecting
     *          has been canceled.
     * - 'select:change'
     *      While selection tracking on touch devices is active, and a
     *      selection range has been modified by tracking the resizer handles.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} rangeIndex
     *          The array index of the modified range in the current selection.
     *      (3) {Range} range
     *          The new address of the specified range.
     *      (4) {Address} [activeCell]
     *          The address of the active cell. If omitted, the current active
     *          cell will not be changed, unless the new active range does not
     *          contain it anymore.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     * @extends GridTrackingMixin
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this header pane.
     *
     * @param {String} panePos
     *  The position of this grid pane.
     */
    function GridPane(docView, panePos) {

        var // self reference
            self = this,

            // column and row pane side identifiers
            colPaneSide = PaneUtils.getColPaneSide(panePos),
            rowPaneSide = PaneUtils.getRowPaneSide(panePos),

            // the application instance
            app = docView.getApp(),

            // the spreadsheet model
            docModel = docView.getDocModel(),

            // the spreadsheet view, and the header panes associated to the grid pane
            colHeaderPane = docView.getColHeaderPane(panePos),
            rowHeaderPane = docView.getRowHeaderPane(panePos),

            // the container node of this grid pane
            rootNode = $('<div class="abs grid-pane f6-target" data-pos="' + panePos + '">'),

            // the scrollable node of this pane (embed into rootNode to allow to hide scroll bars if needed)
            // (attribute unselectable='on' is needed for IE to prevent focusing the node when clicking on a scroll bar)
            scrollNode = $('<div class="grid-scroll" unselectable="on">').appendTo(rootNode),

            // helper node controlling the size and position of the scroll bars
            scrollSizeNode = $('<div class="grid-scroll-size">').appendTo(scrollNode),

            // the root node for all layer nodes, positioned dynamically in the scrollable area
            layerRootNode = $('<div class="grid-layer-root noI18n">').appendTo(scrollSizeNode),

            // the clipboard node that carries the browser focus for copy/paste events
            clipboardNode = docView.createClipboardNode().attr('tabindex', 1).text('\xa0').appendTo(rootNode),
            clipboardFocusMethod = null,

            // the target node for the browser focus (either clipboard node, or the text area in cell edit mode)
            focusTargetNode = clipboardNode,

            // the ratio between (public) absolute scroll position, and internal scroll bars
            scrollLeftRatio = 1,
            scrollTopRatio = 1,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenWidth = 0,
            hiddenHeight = 0,

            // the registered layer renderers, mapped by renderer key
            layerRenderers = {},

            // the cell range and absolute position covered by the layer nodes in the sheet area
            layerRange = null,
            layerRectangle = null,

            // context menu for cells
            cellContextMenu = null,

            // the context menu for drawing objects
            drawingContextMenu = null,

            // all registered embedded cell pop-up menus, mapped by type
            cellMenuRegistry = { menus: {}, order: [] },

            // the cell button element caused to open the active cell menu
            activeCellButton = null,

            // the embedded cell menu instance currently activated (visible)
            activeCellMenu = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Registers a new layer renderer.
         *
         * @param {String} rendererId
         *  Unique identifier for the renderer across all renderers of this
         *  grid pane.
         *
         * @param {BaseRenderer} layerRenderer
         *  The new renderer instance. This grid pane takes ownership!
         */
        function registerLayerRenderer(rendererId, layerRenderer) {

            // store the new layer renderer internally
            layerRenderers[rendererId] = layerRenderer;

            // forward layer renderer events to own listeners
            if (_.isFunction(layerRenderer.on)) {
                layerRenderer.on('triggered', function () {
                    self.trigger.apply(self, _.toArray(arguments).slice(1));
                });
            }
        }

        /**
         * Refreshes the appearance of this grid pane according to the position
         * of the grid pane that is currently active.
         */
        function updateFocusDisplay() {

            var // which header pane is currently focused (while selection tracking is active)
                focusPaneSide = docView.getSheetViewAttribute('activePaneSide'),
                // which grid pane is currently focused
                focusPanePos = docView.getSheetViewAttribute('activePane'),
                // whether this grid pane appears focused
                focused = false;

            if (docView.isCellEditMode()) {
                // never highlight selection in cell in-place edit mode
                focused = false;
            } else if (docView.hasFrozenSplit()) {
                // always highlight all panes in frozen split mode
                focused = true;
            } else if (_.isString(focusPaneSide)) {
                // a header pane is active (selection tracking): focus this
                // grid pane if it is related to the active header pane
                focused = (colPaneSide === focusPaneSide) || (rowPaneSide === focusPaneSide);
            } else {
                // a single grid pane is focused
                focused = focusPanePos === panePos;
            }

            rootNode.toggleClass(Forms.FOCUSED_CLASS, focused);
        }

        /**
         * Handles changed size of the scroll area received from the column or
         * row header pane.
         */
        function updateScrollAreaSize() {

            var // the available size according to existence of scroll bars
                clientWidth = scrollNode[0].clientWidth,
                clientHeight = scrollNode[0].clientHeight,
                // difference of scroll area size between header pane and this grid pane
                widthCorrection = rootNode.width() - clientWidth,
                heightCorrection = rootNode.height() - clientHeight,
                // the new absolute width and height of the scrollable area
                scrollWidth = Math.max(0, colHeaderPane.getScrollSize() - widthCorrection),
                scrollHeight = Math.max(0, rowHeaderPane.getScrollSize() - heightCorrection),
                // the real size of the scroll size node (may be smaller due to browser limits)
                effectiveSize = null;

            // initialize the size of the scroll sizer node, the resulting size may be smaller
            effectiveSize = Utils.setContainerNodeSize(scrollSizeNode, scrollWidth, scrollHeight);

            // recalculate ratio between absolute and effective scroll area size
            scrollLeftRatio = (effectiveSize.width > clientWidth) ? ((scrollWidth - clientWidth) / (effectiveSize.width - clientWidth)) : 0;
            scrollTopRatio = (effectiveSize.height > clientHeight) ? ((scrollHeight - clientHeight) / (effectiveSize.height - clientHeight)) : 0;
        }

        /**
         * Handles changed scroll position received from the column or row
         * header pane.
         */
        function updateScrollPosition() {

            var // new size of the layer root node
                layerRootWidth = layerRectangle.width,
                layerRootHeight = layerRectangle.height,
                // the new DOM scroll position
                scrollLeft = (scrollLeftRatio > 0) ? Math.round(colHeaderPane.getScrollPos() / scrollLeftRatio) : 0,
                scrollTop = (scrollTopRatio > 0) ? Math.round(rowHeaderPane.getScrollPos() / scrollTopRatio) : 0,
                // the effective DOM node offsets according to the scroll position
                offsetLeft = 0,
                offsetTop = 0;

            // Only set new DOM scroll position if it differs from current position. This test
            // MUST be done, otherwise Firefox scrolls extremely slow when using the mouse wheel!
            if (scrollLeft !== scrollNode[0].scrollLeft) { scrollNode[0].scrollLeft = scrollLeft; }
            if (scrollTop !== scrollNode[0].scrollTop) { scrollNode[0].scrollTop = scrollTop; }

            // restrict size of layer root node to size of sheet
            layerRootWidth = Math.min(layerRootWidth, docView.getColCollection().getTotalSize() - layerRectangle.left);
            layerRootHeight = Math.min(layerRootHeight, docView.getRowCollection().getTotalSize() - layerRectangle.top);

            // calculate the effective offsets according to current scroll position
            offsetLeft = Math.max(-2 * layerRootWidth, layerRectangle.left - colHeaderPane.getScrollPos() + scrollLeft - hiddenWidth);
            offsetTop = Math.max(-2 * layerRootHeight, layerRectangle.top - rowHeaderPane.getScrollPos() + scrollTop - hiddenHeight);

            // workaround IE positioning limitations (not possible to set an
            // absolute left/top position of more than Utils.MAX_NODE_SIZE pixels)
            Utils.setPositionInContainerNode(scrollSizeNode, layerRootNode, offsetLeft, offsetTop);

            // set size of the layer root node (should always be inside the node size limits)
            layerRootNode.css({ width: layerRootWidth, height: layerRootHeight });

            // notify listeners
            self.trigger('change:scrollpos');
        }

        /**
         * Lets the cell collection prefetch more cells around the visible area
         * in this grid pane (in shape of a 'plus' sign).
         */
        var prefetchOuterCellRanges = this.createDebouncedMethod($.noop, function () {

            var // the model of the active sheet
                sheetModel = docView.getSheetModel(),
                // the visible rectangle, in pixels
                visibleRect = self.getVisibleRectangle(),
                // the rectangles to be fetched
                fetchRects = [
                    { left: visibleRect.left, top: visibleRect.top - visibleRect.height, width: visibleRect.width, height: 3 * visibleRect.height },
                    { left: visibleRect.left - visibleRect.width, top: visibleRect.top, width: 3 * visibleRect.width, height: visibleRect.height }
                ],
                // the cell ranges covered by the rectangles
                fetchRanges = fetchRects.map(function (fetchRect) {
                    return sheetModel.getRangeFromRectangle(fetchRect, { pixel: true });
                });

            // fetch all cell ranges with a slight delay to not disturb the rendering process
            docView.getCellCollection().fetchMissingCellRanges(fetchRanges, { visible: true, merged: true });

        }, { delay: 2000 });

        /**
         * Registers a cell menu instance to be shown for specific drop-down
         * buttons embedded in the active sheet.
         *
         * @param {String} menuType
         *  The type specifier for the button elements in the form layer, that
         *  are actually triggering the menu.
         *
         * @param {BaseMenu} menu
         *  The menu instance that will be shown when clicking the specified
         *  cell drop-down buttons. This instance takes ownership of the passed
         *  menu (the menu does not have to be destroyed manually)! The passed
         *  menu instance may provide a public method initialize() that will be
         *  invoked before opening the menu. The method will receive the
         *  following parameters:
         *  (1) {Address} address
         *      The address of the cell containing the button node that caused
         *      opening the menu.
         *  (2) {Any} userData
         *      Additional user data that have been passed to the method
         *      FormRenderer.createCellButton().
         *  The method may return a promise indicating that initialization of
         *  the drop-down menu has been finished. The promise returned by the
         *  method may be abortable; in this case, when opening the drop-down
         *  menu repeatedly, or opening another cell drop-down menu quickly,
         *  old initialization requests will be aborted, before calling the
         *  initialization callback function again.
         */
        function registerCellMenu(menuType, menu) {

            // add the menu to the registry map
            cellMenuRegistry.menus[menuType] = menu;
            cellMenuRegistry.order.push(menuType);

            // only the active button must be included in the focus handling of the menu
            menu.registerFocusableNodes(function () { return activeCellButton; });

            // dynamic anchor position according to the clicked drop-down button
            menu.setAnchor(function () {

                var // current page position of the button node
                    buttonRect = Utils.getNodePositionInPage(activeCellButton),
                    // the width of cell and button rectangle, used for anchor of the drop-down menu node
                    anchorWidth = activeCellButton.data('anchorWidth');

                // create the anchor rectangle according to the current position of the button node
                return {
                    left: buttonRect.left + buttonRect.width - anchorWidth,
                    top: buttonRect.top,
                    width: anchorWidth,
                    height: buttonRect.height
                };
            });

            // hide the menu automatically, if the anchor leaves the visible area of the grid pane
            menu.setAnchorBox(scrollNode);

            // remember active menu for usage in other methods
            self.listenTo(menu, 'popup:show', function () {
                activeCellButton.addClass('dropdown-open');
                activeCellMenu = menu;
            });

            // abort initialization if pop-up menu will be closed quickly
            self.listenTo(menu, 'popup:hide', function () {
                activeCellButton.removeClass('dropdown-open');
                activeCellMenu = null;
            });
        }

        /**
         * Opens or closes the embedded cell pop-up menu associated to the
         * passed cell drop-down button.
         *
         * @param {jQuery} buttonNode
         *  The cell button node whose menu will be opened or closed.
         */
        var toggleCellMenu = (function () {

            var // current asynchronous initialization of the cell menu contents
                initPromise = null;

            // aborts current menu initialization
            function abortInitialization() {
                if (_.isObject(initPromise) && _.isFunction(initPromise.abort)) {
                    initPromise.abort();
                }
            }

            // the actual toggleCellMenu() method to be returned from local scope
            function toggleCellMenu(buttonNode) {

                var // the pop-up menu instance
                    menu = Utils.getObjectOption(cellMenuRegistry.menus, buttonNode.attr('data-type')),
                    // the result of the initialization callback function
                    initResult = null;

                // check existence of menu registration
                if (!menu) {
                    Utils.warn('GridPane.toggleCellMenu(): missing menu registration for clicked cell drop-down button');
                    return;
                }

                // abort another initialization of a cell menu currently running
                abortInitialization();

                // always leave cell edit mode (do not open the menu, if formula validation fails)
                if (!docView.leaveCellEditMode('auto', { validate: true })) { return; }

                // if the menu is currently visible, close it and return
                if (menu.isVisible()) {
                    menu.hide();
                    docView.grabFocus();
                    return;
                }

                // remember current button node, will be returned as focusable node to prevent
                // closing the menu too fast (the menu may be associated to different buttons)
                activeCellButton = buttonNode;

                // invoke initialization callback handler
                if (_.isFunction(menu.initialize)) {
                    initResult = menu.initialize(buttonNode.data('address'), buttonNode.data('userData'));
                }

                // show the menu (unless the promise has already been rejected synchronously)
                initPromise = $.when(initResult);
                if (initPromise.state() !== 'rejected') { menu.busy().show(); }

                // show warning alerts if available
                docView.yellOnPromise(initPromise);

                // further initialization for the open menu
                if (menu.isVisible()) {
                    self.listenTo(initPromise, 'fail', function () { menu.hide(); });
                    menu.one('popup:hide', abortInitialization);
                }

                // final clean-up
                self.listenTo(initPromise, 'always', function () {
                    menu.idle();
                    initPromise = null;
                });
            }

            return toggleCellMenu;
        }());

        /**
         * Hides the cell pop-up menu currently shown.
         *
         * @returns {Boolean}
         *  Whether a cell list menu was actually open and has been hidden.
         */
        function hideActiveCellMenu() {
            if (activeCellMenu && activeCellMenu.isVisible()) {
                activeCellMenu.hide();
                return true;
            }
            return false;
        }

        /**
         * Tries to activate an embedded cell pop-up menu attached to the
         * specified cell. If the cell contains multiple drop-down menus, the
         * method searches for a cell menu currently activated for that cell,
         * and activates the next registered menu. The order of the menus is
         * equal to the registration order via the method registerCellMenu().
         *
         * @param {Address} address
         *  The address of a cell to open an embedded pop-up menu for.
         *
         * @returns {Boolean}
         *  Whether a drop-down menu has been found and activated for the cell.
         */
        function activateCellMenu(address) {

            var // all cell button nodes attached to the specified cell
                cellButtons = layerRenderers.form.getCellButtons(address),
                // all menu types occurring in cellButtons, in registration order
                menuTypes = null,
                // type of the pop-up menu currently open (needed to select the next menu)
                activeMenuType = null;

            // no cell drop-down button found: nothing to do
            if (cellButtons.length === 0) { return false; }

            // always scroll to the cell containing the pop-up menu
            self.scrollToCell(address);

            // filter the menu types of the existing cell buttons in registration order
            menuTypes = cellMenuRegistry.order.filter(function (menuType) {
                return cellButtons.filter('[data-type="' + menuType + '"]').length > 0;
            });

            // get type of the pop-up menu currently opened and attached to the specified cell
            if (activeCellButton && activeCellMenu && activeCellButton.data('address').equals(address)) {
                activeMenuType = activeCellButton.attr('data-type');
            }

            // do nothing if the active menu is the only menu for the cell
            if (_.isString(activeMenuType) && (menuTypes.length === 1)) { return true; }

            // always hide the active cell menu before opening the next menu
            hideActiveCellMenu();

            // pick the next menu type from the list of available menu types
            activeMenuType = menuTypes[(_.indexOf(menuTypes, activeMenuType) + 1) % menuTypes.length];

            // open the menu
            toggleCellMenu(cellButtons.filter('[data-type="' + activeMenuType + '"]'));
            return true;
        }

        /**
         * Handles a click on an embedded cell drop-down button, and opens the
         * associated registered menu.
         */
        function cellButtonClickHandler(event, buttonNode) {
            toggleCellMenu(buttonNode);
        }

        /**
         * Updates the grid pane after the document edit mode has changed.
         */
        function editModeHandler(event, editMode) {
            // hide the active pop-up menu, if the document changes to read-only mode
            if (!editMode && app.isImportFinished()) {
                hideActiveCellMenu();
            }
        }

        /**
         * Updates the layer nodes after specific sheet view attributes have
         * been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // refresh focus display ('activePane', and 'activePaneSide')
            if (Utils.hasProperty(attributes, /^activePane/)) {
                updateFocusDisplay();
            }

            // hide the active cell pop-up menu after specific attributes have been changed
            if (Utils.hasProperty(attributes, /^(selection$|split|activePane|autoFillData$)/)) {
                hideActiveCellMenu();
            }
        }

        /**
         * Prepares this grid pane for entering the in-place cell edit mode.
         */
        function cellEditEnterHandler() {
            // disable global F6 focus traveling into this grid pane while in-place cell edit mode is active in another grid pane
            rootNode.toggleClass('f6-target', self === docView.getActiveGridPane());
            updateFocusDisplay();
            // always hide the active cell pop-up menu when cell edit mode starts
            hideActiveCellMenu();
        }

        /**
         * Prepares this grid pane for leaving the in-place cell edit mode.
         */
        function cellEditLeaveHandler() {
            rootNode.addClass('f6-target');
            updateFocusDisplay();
        }

        /**
         * Handles DOM 'scroll' events of the scrollable node.
         */
        function scrollHandler() {
            colHeaderPane.scrollTo(Math.round(scrollNode[0].scrollLeft * scrollLeftRatio));
            rowHeaderPane.scrollTo(Math.round(scrollNode[0].scrollTop * scrollTopRatio));
        }

        /**
         * Handles 'keydown' events from the root node.
         */
        function keyDownHandler(event) {

            // ESCAPE key (only one action per key event)
            if (event.keyCode === KeyCodes.ESCAPE) {

                // cancel current tracking, but do nothing else
                if (Tracking.cancelTracking()) {
                    return false;
                }

                // cancel custom selection mode (before deselecting drawings)
                if (docView.cancelCustomSelectionMode()) {
                    return false;
                }

                // close cell pop-up menu currently open
                if (hideActiveCellMenu()) {
                    return false;
                }

                // drawing selection: back to cell selection
                if (docView.hasDrawingSelection()) {
                    docView.removeDrawingSelection();
                    return false;
                }

                // otherwise: let ESCAPE key bubble up
                return;
            }

            // show cell drop-down menu of active cell on ALT+DOWN
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { alt: true })) {
                activateCellMenu(docView.getActiveCell());
                return false;
            }

            // handle keyboard shortcuts for the cell pop-up menu currently visible
            if (activeCellMenu && _.isFunction(activeCellMenu.handleKeyDownEvent) && activeCellMenu.handleKeyDownEvent(event)) {
                return;
            }
        }

        /**
         * Handles 'drop' events from the root node.
         */
        function dropHandler(event) {

            // nothing to do in locked sheets
            // TODO: the error message needs to be adjusted when dropping cell data will be supported
            if (!docView.requireUnlockedActiveSheet('drawing:insert:locked')) { return; }

            var // promises for the image descriptors (URL and name)
                promises = [],
                // whether any of the dropped data is not an image
                unsupported = false;

            if (event.originalEvent.dataTransfer.files.length > 0) {
                // data originates from local file system
                _.each(event.originalEvent.dataTransfer.files, function (eventFile) {
                    if (ImageUtil.hasFileImageExtension(eventFile.name)) {
                        promises.push(ImageUtil.insertFile(app, eventFile));
                    } else {
                        unsupported = true;
                    }
                });
            } else if (event.originalEvent.dataTransfer.getData('Url').length > 0) {
                // data originates from another browser window
                var url = event.originalEvent.dataTransfer.getData('Url');
                if (ImageUtil.hasUrlImageExtension(url)) {
                    promises.push(ImageUtil.insertURL(url));
                } else {
                    unsupported = true;
                }
            }

            // show a warning if nothing of the dropped data may be an image
            if (unsupported && (promises.length === 0)) {
                docView.yell({ type: 'info', message: gt('File type not supported.') });
            }

            // nothing more to do without pending images
            if (promises.length === 0) { return; }

            // fail with empty resolved promise to be able to use $.when() which would
            // otherwise abort immediately if ANY of the promises fails
            promises = promises.map(function (promise) {
                return promise.then(null, _.constant($.when()));
            });

            // immediately go into busy mode
            docView.enterBusy();

            // wait for all image descriptors, insert the images afterwards
            $.when.apply($, promises).then(function () {
                // $.when() passes the results as function arguments, filter out invalid images
                return docView.insertImages(_.filter(arguments, _.identity));
            }).always(function () {
                docView.leaveBusy();
            });
        }

        /**
         * Sets the browser focus to the clipboard node, and selects all its
         * contents.
         */
        function grabClipboardFocus() {

            var // the browser selection
                selection = window.getSelection(),
                // a browser selection range object
                docRange = null;

            // calls the native focus() method at the clipboard node
            function grabNativeFocus() {
                clipboardFocusMethod.call(clipboardNode[0]);
            }

            // bug 29948: prevent grabbing to clipboard node while in cell in-place edit mode
            if (docView.isCellEditMode()) { return; }

            // on touch devices, selecting text nodes in a content-editable node will
            // always show the virtual keyboard
            // TODO: need to find a way to provide clipboard support
            if (Utils.TOUCHDEVICE) {
                grabNativeFocus();
                return;
            }

            // Bug 26283: Browsers are picky how to correctly focus and select text
            // in a node that is content-editable: Chrome and IE will change the
            // scroll position of the editable node, if it has been focused
            // explicitly while not having an existing browser selection (or even
            // if the selection is not completely visible). Furthermore, the node
            // will be focused automatically when setting the browser selection.
            // Firefox, on the other hand, wants to have the focus already set
            // before the browser selection can be changed, otherwise it may throw
            // exceptions. Additionally, changing the browser selection does NOT
            // automatically set the focus into the editable node.
            if (_.browser.Firefox) {
                grabNativeFocus();
            }

            // Clear the old browser selection.
            // Bug 28515, bug 28711: IE fails to clear the selection (and to modify
            // it afterwards), if it currently points to a DOM node that is not
            // visible anymore (e.g. the 'Show/hide side panel' button). Workaround
            // is to move focus to an editable DOM node which will cause IE to update
            // the browser selection object. The target container node cannot be used
            // for that, see comments above for bug 26283. Using another focusable
            // node (e.g. the body element) is not sufficient either. Interestingly,
            // even using the (editable) clipboard node does not work here. Setting
            // the new browser selection below will move the browser focus back to
            // the application pane.
            Utils.clearBrowserSelection();

            // set the browser selection
            try {
                docRange = window.document.createRange();
                docRange.setStart(clipboardNode[0], 0);
                docRange.setEnd(clipboardNode[0], clipboardNode[0].childNodes.length);
                selection.addRange(docRange);
            } catch (ex) {
                Utils.error('GridPane.grabClipboardFocus(): failed to select clipboard node: ' + ex);
            }
        }

        /**
         * Handles 'cut' events of the clipboard node.
         * Invokes the copyHandler, then deletes the values and clears
         * the attributes of the selected cells.
         */
        function cutHandler(event) {

            var // the clipboard event data
                clipboardData = Utils.getClipboardData(event);

            // check for cell protection
            if (!docView.requireEditableSelection({ lockTables: 'header' })) {
                event.preventDefault();
                return false;
            }

            // invoke clipboard copy handler first
            if (copyHandler(event) === false) { return false; }

            var selectedDrawingModels = docView.getSelectedDrawingModels();

            if (clipboardData) {
                // prevent default cut handling for desktop browsers, but not for touch devices
                if (!Utils.TOUCHDEVICE) {
                    event.preventDefault();
                }

                docView.executeControllerItem((selectedDrawingModels && (selectedDrawingModels.length > 0)) ?
                    'drawing/delete' : 'cell/clear/all');
            } else {
                // delete values and clear attributes of the selected cells (bug 39533: via controller for busy screen etc.)
                self.executeDelayed(function () {
                    return docView.executeControllerItem((selectedDrawingModels && (selectedDrawingModels.length > 0)) ?
                        'drawing/delete' : 'cell/clear/all');
                });
            }
        }

        /**
         * Handles 'copy' events of the clipboard node.
         * Creates a HTML table of the active range of the current selection
         * and generates the client clipboard id. Adds both to the system clipboard.
         * And invokes the Calcengine copy operation.
         */
        var copyHandler = Utils.profileMethod('GridPane.copyHandler()', function (event) {

            var // model of the active sheet
                sheetModel = docView.getSheetModel(),
                // current selection in the active sheet
                selection = docView.getSelection(),
                // the active cell range in the selection
                activeRange = selection.activeRange(),
                // the clipboard event data
                clipboardData = Utils.getClipboardData(event),
                // the client clipboard id to identify the copy data when pasting
                clientClipboardId = null,
                // get drawing selection, if available, first
                selectedDrawingModels = docView.getSelectedDrawingModels(),
                // the HTML mark-up promise, whose resolved value is to be to copied
                htmlMarkup = null,
                // the plain text Promise, whose resolved value is to be to copied
                plainText = null;

            if (selectedDrawingModels.length > 0) {

                // generate HTML representation of drawing(s), no plain-text can be generated
                Utils.log('generating HTML mark-up for ' + selectedDrawingModels.length + ' drawing objects');
                htmlMarkup = Clipboard.convertDrawingModels(app, selectedDrawingModels);

            } else {

                // restrict the number of cells to copy at the same time
                if (activeRange.cells() > SheetUtils.MAX_FILL_CELL_COUNT) {
                    docView.yell({ type: 'info', message: gt('It is not possible to copy more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)) });
                    event.preventDefault();
                    return false;
                }

                var promise = docView.getCellCollection().queryCellContents([new RangeArray(activeRange)]);
                if (promise.state() !== 'resolved') {
                    Utils.log('selected cell range not completely known locally, preparing server-side clipboard handling');
                    clientClipboardId = docView.createClientClipboardId();
                    docView.clipboardServerCopy(activeRange);
                }

                // generate HTML table representation
                Utils.log('generating HTML mark-up for selected cell range');
                htmlMarkup = Clipboard.getHtmlMarkupFromRange(sheetModel, activeRange, clientClipboardId);

                // generate plain text representation
                Utils.log('generating plain-text representation of selected cell range');
                plainText = Clipboard.getPlainTextFromRange(sheetModel, activeRange);
            }

            // set clipboard debug pane content
            self.trigger('debug:clipboard', htmlMarkup);

            // if browser supports the clipboard API, add the generated data to the event
            if (clipboardData) {

                // clear clipboard and add the generated plain-text contents and and/or the HTML mark-up
                Utils.log('adding data to system clipboard');
                clipboardData.clearData();
                if (plainText) { clipboardData.setData('text/plain', plainText); }
                if (htmlMarkup) { clipboardData.setData('text/html', htmlMarkup); }

                // prevent default copy handling for desktop browsers, but not for touch devices
                if (!Utils.TOUCHDEVICE) {
                    event.preventDefault();
                }

            } else if (htmlMarkup) {

                // add Excel XML name spaces for custom tag support
                $('html').attr({
                    xmlns: 'http://www.w3.org/TR/REC-html40',
                    'xmlns:o': 'urn:schemas-microsoft-com:office:office',
                    'xmlns:x': 'urn:schemas-microsoft-com:office:excel'
                });

                // insert contents into the internal clipboard node, and focus it
                Utils.log('adding copied data to internal clipboard DOM node');
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlMarkup);
                grabClipboardFocus();

                // remove the XML namespaces after the browser has copied the data from the clipboard node
                _.defer(function () { $('html').removeAttr('xmlns xmlns:o xmlns:x'); });
            }
        });

        /**
         * Handles 'paste' events of the clipboard node.
         */
        var pasteHandler = Utils.profileMethod('GridPane.pasteHandler()', function (event) {

            var // model of the active sheet
                sheetModel = docView.getSheetModel(),
                // the drawing collection of the active sheet
                drawingCollection = sheetModel.getDrawingCollection(),
                // current selection in the active sheet
                selection = docView.getSelection(),
                // address of the active cell
                activeCell = docView.getActiveCell(),
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the cleaned up HTML data
                htmlData,
                // the HTML data attached to the event
                htmlRawData,
                // the plain text data attached to the event
                textData;

            /**
             * Tries to add the 'imageData' or 'imageUrl' attribute to the
             * passed image attribute set, if possible.
             */
            function addImageSourceToAttributes(imgNode, attributes) {

                // ignore all images without valid source URL
                var imageUrl = imgNode.attr('src');
                if (!imageUrl) { return false; }

                // add sub-object for image attributes
                if (!attributes.image) { attributes.image = {}; }

                // special handling for data URL, ignore all images without valid file extension
                if (/^data:image\//.test(imageUrl)) {
                    attributes.image.imageData = imageUrl;
                    return true;
                }

                // URL must contain valid file extension
                if (ImageUtil.hasFileImageExtension(imageUrl)) {
                    attributes.image.imageUrl = imageUrl;
                    return true;
                }

                return false;
            }

            /**
             * Inserts all drawing objects described in the passed drawing data
             * array, as expected by the method ViewFuncMixin.insertDrawings(),
             * with blocked user interface.
             *
             * @param {Array<Object>} drawingData
             *  The type and formatting attributes of the drawing objects to be
             *  inserted. See method ViewFuncMixin.insertDrawings() for more
             *  details.
             *
             * @returns {jQuery.Promise}
             *  A promise that will be resolved after all drawing objects have
             *  been inserted into the document, or that will be rejected on
             *  any error.
             */
            function insertDrawings(drawingData) {

                // nothing to do without any valid drawing objects
                if (drawingData.length === 0) { return; }

                // do not paste drawing objects into locked sheets
                if (!docView.requireUnlockedActiveSheet('drawing:insert:locked')) { return; }

                // index to be inserted into the names of the pasted objects
                var nameIndex = drawingCollection.getModelCount({ deep: true });

                // generate a name for each drawing (with effective type that may have changed e.g. from chart to image)
                drawingData.forEach(function (data) {
                    nameIndex += 1;
                    if (!data.attrs.drawing) { data.attrs.drawing = {}; }
                    data.attrs.drawing.name = _.noI18n(Labels.getDrawingTypeLabel(data.type)) + ' ' + nameIndex;
                });

                // insert the new charts into the document
                return docView.insertDrawings(drawingData, function (generator, sheet, position, data) {
                    if (data.type === 'chart') {
                        ChartCreator.generateOperationsFromModelData(app, generator, position, data.modelData);
                    }
                });
            }

            /**
             * tries to paste all drawing objects contained in the passed
             * drawing container node.
             */
            var pasteDrawingClipboard = Utils.profileAsyncMethod('GridPane.pasteDrawingClipboard()', function (containerNode, otherAppGuid) {

                var // pixel position of the active cell
                    activeCellRect = sheetModel.getCellRectangle(activeCell),
                    // the drawing attributes to be passed to insertDrawings()
                    drawingData = [];

                // generate the formatting attributes to be passed to insertDrawings()
                containerNode.children().each(function () {

                    var // current drawing data node, as jQuery object
                        dataNode = $(this),
                        // the drawing type stored in the data node
                        drawingType = dataNode.attr('data-type'),
                        // the location of the drawing object (relative to the bounding rectangle)
                        rectangle = Clipboard.getRectangleFromNode(dataNode, activeCellRect.left, activeCellRect.top),
                        // the explicit formatting attributes of the object
                        attributes = Clipboard.getAttributesFromNode(dataNode),
                        // additional JSON data from the DOM node
                        data = Clipboard.getJSONDataFromNode(dataNode),
                        // child image node (for image objects, or as fall-back replacement image)
                        imgNode = dataNode.find('>img'),
                        // whether the data node has been processed successfully (ready to be pasted)
                        success = false;

                    // merge the formatting attributes for the target rectangle
                    attributes = docModel.extendAttributes(attributes, drawingCollection.getAttributeSetForRectangle(rectangle));

                    // TODO: other drawing types

                    // paste charts only if they have been copied from this document
                    if (drawingType === 'chart') {
                        success = _.isObject(data) && _.isObject(data.modelData) && (app.getGlobalUid() === otherAppGuid);
                        if (!success) { delete attributes.chart; }
                    }

                    // process image objects, or try to get a replacement image node as last fall-back
                    if (!success && (imgNode.length > 0)) {
                        drawingType = 'image';
                        success = addImageSourceToAttributes(imgNode, attributes);
                    }

                    // collect the attributes for all valid objects for asynchronous insertion
                    if (success) {
                        drawingData.push(_.extend({ type: drawingType, attrs: attributes }, data));
                    }
                });

                // lock the GUI, and insert the new drawing objects into the document
                return insertDrawings(drawingData);
            });

            var pasteImageClipboard = Utils.profileAsyncMethod('GridPane.pasteImageClipboard()', function (htmlData, jqImgElements) {

                var // pixel position of the active cell
                    activeCellRect = sheetModel.getCellRectangle(activeCell),
                    // the drawing attributes to be passed to insertDrawings()
                    drawingData = [];

                // generate the formatting attributes to be passed to insertDrawings()
                jqImgElements.each(function () {

                    var // create the formatting attributes for the target rectangle
                        attributes = drawingCollection.getAttributeSetForRectangle({
                            left: activeCellRect.left,
                            top: activeCellRect.top,
                            width: this.width || this.naturalWidth || 256,
                            height: this.height || this.naturalHeight || 192
                        });

                    // add generic drawing attributes
                    attributes.drawing.anchorType = 'twoCellAsOneCell';

                    // add image source link
                    if (addImageSourceToAttributes($(this), attributes)) {
                        drawingData.push({ type: 'image', attrs: attributes });
                    }
                });

                // insert the new images into the document
                return insertDrawings(drawingData);
            });

            function createRangeFromParserResult(contents) {
                var range = new Range(selection.address);
                range.end[0] += ((contents.length > 0) && (contents[0].length > 0)) ? (contents[0].length - 1) : 0;
                range.end[1] += (contents.length > 0) ? (contents.length - 1) : 0;
                return range;
            }

            function handleParserResult(result) {

                if (!result.contents) {
                    Utils.warn('no content');
                    return $.when();
                }

                var // the target range of the paste
                    pasteRange = createRangeFromParserResult(result.contents),
                    // the number of cells to paste
                    pasteCellCount = pasteRange.cells(),
                    // the resulting promise
                    promise = null;

                if (pasteRange.end[0] > docModel.getMaxCol() || pasteRange.end[1] > docModel.getMaxRow()) {
                    promise = SheetUtils.makeRejected('paste:outside');
                } else {
                    // restrict the number of cells to paste at the same time, check for protected cells
                    if (pasteCellCount > SheetUtils.MAX_FILL_CELL_COUNT) {
                        promise = SheetUtils.makeRejected('paste:overflow');
                    } else {
                        promise = docView.areRangesEditable(pasteRange, { lockTables: 'header', error: 'paste:locked' });
                    }
                }

                // paste the contents contained in the passed result object
                promise = promise.then(function () {

                    function processRanges(ranges, callback) {

                        var // column offset for pasting contents that are available relative to column A
                            colOff = selection.address[0],
                            // row offset for pasting contents that are available relative to row 1
                            rowOff = selection.address[1];

                        // transform the ranges to the paste position
                        ranges = RangeArray.map(ranges, function (range) {
                            return docModel.getCroppedMovedRange(range, colOff, rowOff);
                        });

                        // invoke callback handler, if any valid ranges are left
                        return ranges.empty() ? null : callback(ranges);
                    }

                    // create a single undo action for all affected operations
                    return docModel.getUndoManager().enterUndoGroup(function () {

                        // the promise that chains all operations performed while pasting
                        var operationPromise = $.when();

                        // unmerge cells if necessary
                        if ((pasteCellCount > 1) && sheetModel.getMergeCollection().rangesOverlapAnyMergedRange(pasteRange)) {
                            operationPromise = operationPromise.then(function () {
                                return sheetModel.mergeRanges(pasteRange, 'unmerge');
                            });
                        }

                        // merge cells
                        if (result.mergedRanges) {
                            operationPromise = operationPromise.then(function () {
                                return processRanges(result.mergedRanges, function (mergedRanges) {
                                    return sheetModel.mergeRanges(mergedRanges, 'merge');
                                });
                            });
                        }

                        // set cell contents
                        operationPromise = operationPromise.then(function () {
                            return sheetModel.setCellContents(activeCell, result.contents, { parse: true });
                        });

                        // add hyperlinks
                        _.each(result.urlMap, function (ranges, url) {
                            operationPromise = operationPromise.then(function () {
                                return processRanges(ranges, function (urlRanges) {
                                    return sheetModel.fillCellRanges(urlRanges, undefined, undefined, { url: url });
                                });
                            });
                        });

                        // select the new range and update the cell heights
                        operationPromise = operationPromise.then(function () {
                            docView.selectRange(pasteRange);
                            return docView.setOptimalRowHeight();
                        });

                        return operationPromise;
                    });
                });

                // show alert messages on error
                return docView.yellOnPromise(promise);
            }

            function pasteHtmlClipboard(html) {

                var // the client clipboard id to identify the copy data when pasting
                    clientClipboardId = Clipboard.getClientClipboardId(html),
                    // the server clipboard id to identify the copy data when pasting
                    serverClipboardId = docView.getServerClipboardId(clientClipboardId);

                if (serverClipboardId) {
                    // set clipboard debug pane content
                    self.trigger('debug:clipboard', 'Server side paste with clipboard ID: ' + serverClipboardId);
                    // server paste call
                    Utils.log('pasting server-side clipboard contents');
                    return docView.clipboardServerPaste(selection.activeRange(), serverClipboardId);
                }

                // parse HTML
                Utils.log('pasting HTML clipboard contents');
                return handleParserResult(Clipboard.parseHTMLData(docView, html));
            }

            function pasteTextClipboard(text) {
                // set clipboard debug pane content
                self.trigger('debug:clipboard', text);
                // parse text
                Utils.log('pasting plain-text clipboard contents');
                return handleParserResult(Clipboard.parseTextData(text));
            }

            // check if we have edit rights before pasting
            if (!docModel.getEditMode()) {
                event.preventDefault();
                app.rejectEditAttempt();
                return false;
            }

            htmlRawData = clipboardData && clipboardData.getData('text/html');
            htmlData = _.isString(htmlRawData) ? Utils.parseAndSanitizeHTML(htmlRawData) : $();

            var drawingContainerNode = Clipboard.getDrawingContainerNode(htmlData);
            var jqTableElements = Clipboard.getContainedHtmlElements(htmlData, 'table');
            var jqImgElements = !jqTableElements ? Clipboard.getContainedHtmlElements(htmlData, 'img') : null;

            if (drawingContainerNode || jqTableElements || jqImgElements) {
                // check if clipboard event data contains HTML

                // prevent default paste handling for desktop browsers, but not for touch devices
                if (!Utils.TOUCHDEVICE) {
                    event.preventDefault();
                }

                // set clipboard debug pane content
                self.trigger('debug:clipboard', htmlRawData);
                // render the HTML to apply CSS styles provided by Excel or Calc
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlData);

                // enable busy blocker screen
                docView.enterBusy();

                // first, try to paste drawing objects that have been copied by ourselves
                var promise = null;
                if (drawingContainerNode) {
                    promise = pasteDrawingClipboard(drawingContainerNode, Clipboard.getApplicationGuid(htmlData));
                } else if (jqImgElements) {
                    promise = pasteImageClipboard(htmlData, jqImgElements);
                } else {
                    promise = pasteHtmlClipboard(clipboardNode);
                }

                // leave busy blocker screen
                promise.always(function () { docView.leaveBusy().grabFocus(); });

                // clear clipboard
                clipboardNode.text('\xa0');
                return false;
            }

            // focus and select the clipboard node
            grabClipboardFocus();

            // check if clipboard event data contains plain text
            // to be used in case the clipboard node contains no HTML table to parse
            textData = (_.browser.IE < 12) ? window.clipboardData && window.clipboardData.getData('text') : clipboardData && clipboardData.getData('text/plain');

            // read and parse pasted data
            self.executeDelayed(function () {
                self.trigger('debug:clipboard', clipboardNode);

                drawingContainerNode = Clipboard.getDrawingContainerNode(clipboardNode);
                jqTableElements = Clipboard.getContainedHtmlElements(clipboardNode, 'table');
                jqImgElements = !jqTableElements ? Clipboard.getContainedHtmlElements(clipboardNode, 'img') : null;

                // enable busy blocker screen
                docView.enterBusy();

                // first, try to paste drawing objects that have been copied by ourselves
                var promise = null;
                if (drawingContainerNode) {
                    promise = pasteDrawingClipboard(drawingContainerNode, Clipboard.getApplicationGuid(clipboardNode));
                } else if (jqImgElements) {
                    promise = pasteImageClipboard(clipboardNode, jqImgElements);
                } else if (jqTableElements) {
                    promise = pasteHtmlClipboard(clipboardNode);
                } else if (textData) {
                    promise = pasteTextClipboard(textData);
                } else {
                    promise = $.when();
                }

                // leave busy blocker screen
                promise.always(function () { docView.leaveBusy().grabFocus(); });

                // clear clipboard
                clipboardNode.text('\xa0');
            });
        });

        // public methods -----------------------------------------------------

        /**
         * Returns the document view that contains this grid pane.
         *
         * @returns {SpreadsheetView}
         *  The document view that contains this grid pane.
         */
        this.getDocView = function () {
            return docView;
        };

        /**
         * Returns the root DOM node of this grid pane.
         *
         * @returns {jQuery}
         *  The root node of this grid pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns the scrollable container node of all content layers in this
         * grid pane.
         *
         * @returns {jQuery}
         *  The scrollable container node of this grid pane, as jQuery object.
         */
        this.getScrollNode = function () {
            return scrollNode;
        };

        /**
         * Returns the root container node of all rendering layers in this grid
         * pane.
         *
         * @returns {jQuery}
         *  The root container node of all rendering layers in this grid pane,
         *  as jQuery object.
         */
        this.getLayerRootNode = function () {
            return layerRootNode;
        };

        /**
         * Returns the column header pane associated to this grid pane.
         *
         * @returns {HeaderPane}
         *  The column header pane associated to this grid pane.
         */
        this.getColHeaderPane = function () {
            return colHeaderPane;
        };

        /**
         * Returns the row header pane associated to this grid pane.
         *
         * @returns {HeaderPane}
         *  The row header pane associated to this grid pane.
         */
        this.getRowHeaderPane = function () {
            return rowHeaderPane;
        };

        /**
         * Creates a new DOM rendering layer node, and inserts it into the
         * layer root node of this grid pane.
         *
         * @param {String} className
         *  Name of a CSS class that will be added to the created layer node.
         *
         * @returns {jQuery} layerNode
         *  The new DOM rendering layer node.
         */
        this.createLayerNode = function (className) {
            return $('<div class="grid-layer ' + className + '">').appendTo(layerRootNode);
        };

        /**
         * Returns whether the root node of this grid pane is currently focused
         * or contains a DOM node that is currently focused (e.g. the text area
         * node while edit mode is active).
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently focused.
         */
        this.hasFocus = function () {
            return focusTargetNode.is(window.document.activeElement);
        };

        /**
         * Sets the browser focus into this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            focusTargetNode.focus();
            return this;
        };

        /**
         * Temporarily registers a custom focus target node for this grid pane.
         * The passed node will be used when focusing the grid pane instead of
         * the own internal clipboard node.
         *
         * @param {jQuery|HTMLElement} targetNode
         *  The DOM element to be used as focus target node for this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.registerFocusTargetNode = function (targetNode) {
            focusTargetNode = $(targetNode).first().focus();
            clipboardNode.hide();
            return this;
        };

        /**
         * Restores the default focus target node of this grid pane (the own
         * internal clipboard node). Can be called after the focus target node
         * has been changed with the method GridPane.registerFocusTargetNode().
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.unregisterFocusTargetNode = function () {
            focusTargetNode = clipboardNode.show();
            grabClipboardFocus();
            return this;
        };

        /**
         * Initializes the settings and layout of this grid pane.
         *
         * @param {Object} colSettings
         *  The view settings of the horizontal pane side (left or right).
         *  Supports the following properties:
         *  @param {Number} colSettings.offset
         *      The absolute horizontal offset of the grid pane in the view
         *      root node, in pixels.
         *  @param {Number} colSettings.size
         *      The total outer width of the grid pane, in pixels. The value
         *      zero will have the effect to hide the entire grid pane.
         *  @param {Boolean} colSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) horizontally.
         *  @param {Boolean} colSettings.showOppositeScroll
         *      Whether the vertical scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} colSettings.hiddenSize
         *      The total width of the hidden columns in front of the sheet in
         *      frozen view mode.
         *
         * @param {Object} rowSettings
         *  The view settings of the vertical pane side (top or bottom).
         *  Supports the following properties:
         *  @param {Number} rowSettings.offset
         *      The absolute vertical offset of the grid pane in the view root
         *      node, in pixels.
         *  @param {Number} rowSettings.size
         *      The total outer height of the grid pane, in pixels. The value
         *      zero will have the effect to hide the entire grid pane.
         *  @param {Boolean} rowSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) vertically.
         *  @param {Boolean} rowSettings.showOppositeScroll
         *      Whether the horizontal scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} rowSettings.hiddenSize
         *      The total height of the hidden rows in front of the sheet in
         *      frozen view mode.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (colSettings, rowSettings) {

            // initialize according to visibility
            if ((colSettings.size > 0) && (rowSettings.size > 0)) {

                // show pane root node and initialize auto-scrolling
                rootNode.show();
                Tracking.enableTracking(layerRootNode, Utils.extendOptions(PaneUtils.DEFAULT_TRACKING_OPTIONS, {
                    autoScroll: colSettings.frozen ? (rowSettings.frozen ? false : 'vertical') : (rowSettings.frozen ? 'horizontal' : true),
                    borderNode: rootNode,
                    dblClick: 'mouse',
                    dblClickDelay: _.browser.IE ? 1800 : null // TODO: IE needs too much time to process tracking events
                }));

                // position and size
                rootNode.css({ left: colSettings.offset, top: rowSettings.offset, width: colSettings.size, height: rowSettings.size });

                // initialize the scroll node containing the scroll bars
                scrollNode.css({
                    overflowX: colSettings.frozen ? 'hidden' : '',
                    overflowY: rowSettings.frozen ? 'hidden' : '',
                    bottom: (!colSettings.frozen && !rowSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_HEIGHT) : 0,
                    right: (!rowSettings.frozen && !colSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_WIDTH) : 0
                });

                // additional layout information for frozen splits
                hiddenWidth = colSettings.hiddenSize;
                hiddenHeight = rowSettings.hiddenSize;

                // update scroll area size and CSS classes at root node for focus display
                updateScrollAreaSize();
                updateFocusDisplay();

            } else {

                // hide pane root node and deinitialize auto-scrolling
                rootNode.hide();
                Tracking.disableTracking(layerRootNode);
            }

            return this;
        };

        /**
         * Updates the DOM layer nodes according to the passed layer range.
         *
         * @param {RangeBoundary} layerBoundary
         *  The address of the new cell range covered by this grid pane, and
         *  the exact rectangle to be covered by the DOM layers. If this object
         *  is invalid (all properties are null), all rendering layers will be
         *  hidden.
         *
         * @param {Array} dirtyRanges
         *  An array with the addresses of all dirty cell ranges in the sheet
         *  that must be recalculated and repainted (e.g. after one or more
         *  document operations such as inserting or deleting columns or rows).
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.setLayerRange = function (layerBoundary, dirtyRanges) {

            if (layerBoundary.isValid()) {

                // update visible sheet contents according to the new layer range
                layerRange = layerBoundary.range;
                layerRectangle = layerBoundary.rectangle;

                // update the scroll position (exact position of the layer root node in the DOM)
                scrollNode.show();
                updateScrollAreaSize();
                updateScrollPosition();

                // invoke all layer renderers
                RenderUtils.takeTime('GridPane.setLayerRange(): update rendering layers, pos=' + panePos, function () {
                    RenderUtils.log('range=' + layerRange + ' rectangle=' + JSON.stringify(layerRectangle));
                    _.invoke(layerRenderers, 'setLayerRange', layerRange, layerRectangle, dirtyRanges);
                });

                // prefetch more cell ranges around the visible area in the grid pane (in shape of a 'plus' sign)
                prefetchOuterCellRanges();

            } else if (layerRange) {

                // clear the layer rectangle
                layerRange = layerRectangle = null;

                // invoke all layer renderers
                _.invoke(layerRenderers, 'hideLayerRange');

                // hide the scroll node to prevent any left-overs if all columns/rows are hidden in a visible grid pane
                scrollNode.hide();
            }

            return this;
        };

        /**
         * Returns whether a layer range is currently available (and the DOM
         * rendering layers are visible).
         *
         * @returns {Boolean}
         *  Whether a layer range is currently available.
         */
        this.isVisible = function () {
            return _.isObject(layerRange);
        };

        /**
         * Registers an event handler at the specified event source object that
         * will only be invoked when this grid pane is visible. See method
         * BaseObject.listenTo() for details about the parameters.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.listenToWhenVisible = function (target, source, type, handler) {
            target.listenTo(source, type, function () {
                if (layerRange) { return handler.apply(self, arguments); }
            });
            return this;
        };

        /**
         * Returns the address of the cell range currently covered by the
         * rendering layer nodes, including the ranges around the visible area.
         *
         * @returns {Range|Null}
         *  The address of the cell range covered by the layer nodes, if the
         *  layers are visible; otherwise null.
         */
        this.getLayerRange = function () {
            return layerRange.clone();
        };

        /**
         * Returns the offset and size of the entire sheet rectangle covered by
         * the rendering layer nodes, including the ranges around the visible
         * area.
         *
         * @returns {Object|Null}
         *  The offset and size of the sheet area covered by the layer nodes,
         *  in pixels, if the layers are visible; otherwise null.
         */
        this.getLayerRectangle = function () {
            return _.copy(layerRectangle, true);
        };

        /**
         * Converts the passed absolute sheet rectangle to a rectangle relative
         * to the current rendering layer nodes.
         *
         * @param {Object} rectangle
         *  An absolute sheet rectangle, in pixels.
         *
         * @returns {Object}
         *  The resulting layer rectangle, relative to the current layer nodes.
         */
        this.convertToLayerRectangle = function (rectangle) {
            return {
                left: rectangle.left - (layerRectangle ? layerRectangle.left : 0),
                top: rectangle.top - (layerRectangle ? layerRectangle.top : 0),
                width: layerRectangle ? rectangle.width : 0,
                height: layerRectangle ? rectangle.height : 0
            };
        };

        /**
         * Returns the CSS position and size attributes for the passed absolute
         * sheet rectangle in pixels, as HTML mark-up value for the 'style'
         * element attribute, adjusted to the current position of the layer
         * nodes in this grid pane.
         *
         * @param {Object} rectangle
         *  The absolute sheet rectangle, in pixels.
         *
         * @returns {String}
         *  The value for the 'style' element attribute in HTML mark-up text.
         */
        this.getLayerRectangleStyleMarkup = function (rectangle) {
            return PaneUtils.getRectangleStyleMarkup(this.convertToLayerRectangle(rectangle));
        };

        /**
         * Invokes the passed iterator function for all cell ranges that are
         * located inside the current layer range. The iterator function will
         * receive additional data needed for rendering those ranges in some
         * way.
         *
         * @param {RangeArray|Range}
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Function} iterator
         *  The iterator function that will be invoked for each range that is
         *  inside the visible area of this grid pane. Receives
         *  the following parameters:
         *  (1) {Range} range
         *      The address of the cell range to be rendered.
         *  (2) {Object} rectangle
         *      The location of the current range, relative to the layer root
         *      node of this grid pane.
         *  (3) {Number} index
         *      The array index of the current cell range.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.alignToGrid=false]
         *      If set to true, the rectangles passed to the iterator callback
         *      function will be expanded by one pixel to the left and to the
         *      top, so that all its outer pixels are located on the grid lines
         *      surrounding the range.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.iterateRangesForRendering = function (ranges, callback, options) {

            // do nothing, if all columns or rows are hidden
            if (!this.isVisible()) { return this; }

            var // the bounding rectangle for range nodes (prevent oversized DOM nodes)
                boundRectangle = { width: MAX_RECTANGLE_SIZE, height: MAX_RECTANGLE_SIZE },
                // whether to enlarge rectangle by one pixel to the left and top, to place it on grid lines
                alignToGrid = Utils.getBooleanOption(options, 'alignToGrid', false);

            // get start position of the bounding rectangle for range nodes, to prevent oversized DOM nodes
            boundRectangle.left = Math.max(0, layerRectangle.left - Math.floor((boundRectangle.width - layerRectangle.width) / 2));
            boundRectangle.top = Math.max(0, layerRectangle.top - Math.floor((boundRectangle.height - layerRectangle.height) / 2));

            // restrict to ranges in the current cell area to prevent oversized DOM nodes
            // that would collapse to zero size, e.g. when entire columns or rows are selected
            // (for details, see the comments for the Utils.MAX_NODE_SIZE constant)
            RangeArray.forEach(ranges, function (range, index) {

                var // position and size of the selection range, restricted to the bounding rectangle
                    rectangle = Utils.getIntersectionRectangle(docView.getRangeRectangle(range), boundRectangle);

                // skip ranges located completely outside the bounding rectangle
                if (!rectangle) { return; }

                // enlarge rectangle to match the grid lines between the cells
                if (alignToGrid) {
                    rectangle.left -= 1;
                    rectangle.top -= 1;
                    rectangle.width += 1;
                    rectangle.height += 1;
                }

                callback.call(this, range, this.convertToLayerRectangle(rectangle), index);
            }, this);
            return this;
        };

        /**
         * Returns the size of the leading hidden area that cannot be shown in
         * this grid pane in a frozen split view.
         *
         * @returns {Object}
         *  The size of the of the leading hidden area that cannot be shown in
         *  this grid pane, in the following properties:
         *  - {Number} width
         *      The width of the hidden area left of this grid pane. Will be 0,
         *      if the view does not contain a frozen split.
         *  - {Number} height
         *      The height of the hidden area left of this grid pane. Will be
         *      0, if the view does not contain a frozen split.
         */
        this.getHiddenSize = function () {
            return { width: hiddenWidth, height: hiddenHeight };
        };

        /**
         * Returns the effective offset and size of the sheet area that is
         * really visible in this grid pane.
         *
         * @returns {Object}
         *  The offset and size of the sheet area visible in this grid pane, in
         *  pixels.
         */
        this.getVisibleRectangle = function () {
            return {
                left: colHeaderPane.getVisiblePosition().offset,
                top: rowHeaderPane.getVisiblePosition().offset,
                width: scrollNode[0].clientWidth,
                height: scrollNode[0].clientHeight
            };
        };

        /**
         * Returns the address of the top-left cell visible in this grid pane.
         * The cell is considered visible, it its column and row are at least
         * half visible.
         *
         * @returns {Address}
         *  The address of the top-left cell in this grid pane.
         */
        this.getTopLeftAddress = function () {
            return new Address(colHeaderPane.getFirstVisibleIndex(), rowHeaderPane.getFirstVisibleIndex());
        };

        /**
         * Changes the scroll position of this grid pane relative to the
         * current scroll position.
         *
         * @param {Number} leftDiff
         *  The difference for the horizontal scroll position, in pixels.
         *
         * @param {Number} topDiff
         *  The difference for the vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollRelative = function (leftDiff, topDiff) {
            colHeaderPane.scrollRelative(leftDiff);
            rowHeaderPane.scrollRelative(topDiff);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified rectangle in the sheet
         * visible.
         *
         * @param {Object} rectangle
         *  The sheet rectangle to be made visible, in pixels.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceTop=false]
         *      If set to true, the rectangle will always be scrolled into the
         *      upper border of the grid pane, also if it is already visible in
         *      the bottom area.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToRectangle = function (rectangle, options) {

            var // whether to force the rectangle to the top border
                forceTop = Utils.getBooleanOption(options, 'forceTop', false);

            colHeaderPane.scrollToPosition(rectangle.left, rectangle.width);
            rowHeaderPane.scrollToPosition(rectangle.top, rectangle.height, { forceLeading: forceTop });
            return this;
        };

        // cell layer ---------------------------------------------------------

        /**
         * Returns all data for the cell covering the passed page coordinates.
         * It does not matter whether the specified position is visible in the
         * scroll node.
         *
         * @param {Number} pageX
         *  The horizontal page position, in pixels.
         *
         * @param {Number} pageY
         *  The vertical page position, in pixels.
         *
         * @param {Object} [options]
         *  Optional parameters that will be forwarded to the method
         *  ColRowCollection.getEntryByOffset().
         *
         * @returns {Object}
         *  A result descriptor with the following properties:
         *  - {Number} left
         *      The horizontal absolute sheet position, in pixels.
         *  - {Number} top
         *      The vertical absolute sheet position, in pixels.
         *  - {Object} colDesc
         *      The column descriptor received from the column collection.
         *  - {Object} rowDesc
         *      The row descriptor received from the row collection.
         *  - {Address} address
         *      The address of the cell covering the specified position.
         */
        this.getCellDataByOffset = function (pageX, pageY, options) {

            var // the absolute position of the layer root node
                layerOffset = layerRootNode.offset(),
                // the resulting cell data object
                result = {};

            // calculate sheet offset in pixels
            result.left = pageX - layerOffset.left + layerRectangle.left;
            result.top = pageY - layerOffset.top + layerRectangle.top;

            // add the column and row collection entry
            options = _.extend({}, options, { pixel: true });
            result.colDesc = docView.getColCollection().getEntryByOffset(result.left, options);
            result.rowDesc = docView.getRowCollection().getEntryByOffset(result.top, options);

            // add the cell address
            result.address = new Address(result.colDesc.index, result.rowDesc.index);

            return result;
        };

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Address} address
         *  The cell position to be made visible.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceTop=false]
         *      If set to true, the cell will always be scrolled into the upper
         *      border of the grid pane, also if it is already visible in the
         *      bottom area.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address, options) {

            var // the position of the specified cell
                rectangle = docView.getCellRectangle(address, { expandMerged: true });

            return this.scrollToRectangle(rectangle, options);
        };

        // drawing layer ------------------------------------------------------

        /**
         * Returns the DOM drawing frame node at the passed document position.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object.
         *
         * @returns {jQuery}
         *  The DOM node of the drawing frame, as jQuery object; or an empty
         *  jQuery collection, if the drawing frame does not exist.
         */
        this.getDrawingFrame = function (position) {
            return layerRenderers.drawing.getDrawingFrame(position);
        };

        /**
         * Scrolls this grid pane to make the specified drawing frame visible.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing frame.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceTop=false]
         *      If set to true, the cell will always be scrolled into the upper
         *      border of the grid pane, also if it is already visible in the
         *      bottom area.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToDrawingFrame = function (position, options) {

            var // the drawing model at the specified document position
                drawingModel = docView.getDrawingCollection().findModel(position, { deep: true }),
                // the position of the drawing model (will be null for hidden drawing objects)
                rectangle = drawingModel.getRectangle();

            if (rectangle) { this.scrollToRectangle(rectangle, options); }
            return this;
        };

        // initialization -----------------------------------------------------

        // tracking mix-in expects fully initialized grid pane
        GridTrackingMixin.call(this);

        // marker for touch devices and browser types
        Forms.addDeviceMarkers(rootNode);

        // handle DOM focus requests at the clipboard node (select clipboard contents)
        clipboardFocusMethod = clipboardNode[0].focus;
        clipboardNode[0].focus = grabClipboardFocus;

        // on touch devices, disable the content-editable mode of the clipboard node,
        // to prevent showing the virtual keyboard all the time
        if (Utils.TOUCHDEVICE) { clipboardNode.removeAttr('contenteditable'); }

        // create the DOM layer nodes and layer renderers
        registerLayerRenderer('cell', new CellRenderer(this));
        registerLayerRenderer('selection', new SelectionRenderer(this));
        registerLayerRenderer('form', new FormRenderer(this));
        registerLayerRenderer('drawing', new DrawingRenderer(this));
        registerLayerRenderer('highlight', new HighlightRenderer(this));

        // update grid pane when document edit mode changes
        this.listenToWhenVisible(this, docModel, 'change:editmode', editModeHandler);

        // handle in-place cell edit mode
        this.listenTo(docView, 'celledit:enter', cellEditEnterHandler);
        this.listenTo(docView, 'celledit:leave', cellEditLeaveHandler);

        // update layers according to changed view attributes (only, if this grid pane is visible)
        this.listenToWhenVisible(this, docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

        // listen to changed scroll position/size from header panes
        this.listenToWhenVisible(this, colHeaderPane, 'change:scrollsize', updateScrollAreaSize);
        this.listenToWhenVisible(this, colHeaderPane, 'change:scrollpos', updateScrollPosition);
        this.listenToWhenVisible(this, rowHeaderPane, 'change:scrollsize', updateScrollAreaSize);
        this.listenToWhenVisible(this, rowHeaderPane, 'change:scrollpos', updateScrollPosition);

        // listen to DOM scroll events
        scrollNode.on('scroll', scrollHandler);

        // activate this pane on focus change
        rootNode.on('focusin', function () {
            // do not change active pane while in cell edit mode (range selection mode in formulas)
            if (!docView.isCellEditMode()) {
                docView.setSheetViewAttribute('activePane', panePos);
            }
        });

        // handle key and drop events
        rootNode.on({ keydown: keyDownHandler, drop: dropHandler });

        // handle system clipboard events
        clipboardNode.on({ cut: cutHandler, copy: copyHandler, paste: pasteHandler });

        // show cell menu after a click on a cell drop-down button
        layerRenderers.form.on('cellbutton:click', cellButtonClickHandler);

        // immediately hide the active pop-up menu on selection tracking
        this.on('select:start', function () { hideActiveCellMenu(); });

        // create the context menus for cells and drawing objects
        cellContextMenu = new CellContextMenu(this);
        drawingContextMenu = new DrawingContextMenu(this, layerRenderers.drawing.getLayerNode());

        // drop-down menu for table columns (sorting and filtering)
        registerCellMenu('table', new TableColumnMenu(docView));

        // drop-down menu for data validation of active cell
        registerCellMenu('validation', new ValidationListMenu(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () {

            clipboardNode[0].focus = clipboardFocusMethod;

            _.invoke(layerRenderers, 'destroy');
            _.invoke(cellMenuRegistry.menus, 'destroy');

            cellContextMenu.destroy();
            drawingContextMenu.destroy();

            app.destroyImageNodes(clipboardNode);

            self = app = docModel = docView = colHeaderPane = rowHeaderPane = null;
            rootNode = scrollNode = scrollSizeNode = layerRootNode = null;
            clipboardNode = clipboardFocusMethod = layerRenderers = null;
            cellContextMenu = drawingContextMenu = null;
            cellMenuRegistry = activeCellButton = activeCellMenu = null;
        });

    } // class GridPane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: GridPane });

});
