/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/range3darray', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/arraytemplate',
    'io.ox/office/spreadsheet/utils/range3d'
], function (Utils, ArrayTemplate, Range3D) {

    'use strict';

    // class Range3DArray =====================================================

    /**
     * Represents an array of cell range addresses with sheet indexes. The
     * array elements are instances of the class Range3D.
     *
     * @constructor
     *
     * @extends Array
     */
    var Range3DArray = ArrayTemplate.create(Range3D);

    // static methods ---------------------------------------------------------

    /**
     * Creates an array of cell range address with sheet indexes from the
     * passed simple cell range addresses, and the passed sheet indexes. The
     * sheet indexes in the resulting cell range address will be ordered
     * automatically. The start and end address of the original cell range
     * addresses will be deeply cloned when creating the new range addresses.
     *
     * @param {RangeArray|Range} ranges
     *  An array of simple cell range addresses (without sheet indexes, or a
     *  simple cell range address (without sheet indexes).
     *
     * @param {Number} sheet1
     *  The zero-based index of the first sheet of the created cell range
     *  addresses.
     *
     * @param {Number} [sheet2]
     *  The zero-based index of the second sheet of the created cell range
     *  addresses. If omitted, the new ranges will refer to a single sheet.
     *
     * @returns {Range3DArray}
     *  The cell range addresses created from the passed parameters.
     */
    Range3DArray.createFromRanges = function (ranges, sheet1, sheet2) {
        return Range3DArray.map(ranges, function (range) { return Range3D.createFromRange(range, sheet1, sheet2); });
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the number of cells covered by all cell range address.
     *
     * @returns {Number}
     *  The number of cells covered by all cell range address.
     */
    Range3DArray.prototype.cells = function () {
        return this.reduce(function (count, range) { return count + range.cells(); }, 0);
    };

    /**
     * Returns whether all cell range addresses in this array refer to the same
     * single sheet.
     *
     * @returns {Boolean}
     *  Whether all cell range addresses in this array refer to the same single
     *  sheet.
     */
    Range3DArray.prototype.singleSheet = function () {
        if (this.empty()) { return false; }
        var sheet = this.first().sheet1;
        return this.every(function (range) { return range.isSheet(sheet); });
    };

    /**
     * Returns whether any cell range address in this array overlaps with any
     * cell range address in the passed array.
     *
     * @param {Range3DArray|Range3D} ranges
     *  The other cell range addresses to be checked. This method also accepts
     *  a single cell range address as parameter.
     *
     * @returns {Boolean}
     *  Whether any cell range address in this array overlaps with any cell
     *  range address in the passed array.
     */
    Range3DArray.prototype.overlaps = function (ranges) {
        return this.some(function (range1) {
            return Range3DArray.some(ranges, function (range2) {
                return range1.overlaps(range2);
            });
        });
    };

    /**
     * Returns the address of the bounding cell range of all range addresses in
     * this array (the smallest range that contains all ranges in the array).
     *
     * @returns {Range3D|Null}
     *  The address of the bounding cell range containing all range addresses;
     *  or null, if this range array is empty.
     */
    Range3DArray.prototype.boundary = function () {

        switch (this.length) {
        case 0:
            return null;
        case 1:
            // this method must not return original array elements
            return this[0].clone();
        }

        var result = this[0].clone();
        this.forEach(function (range) {
            result.sheet1 = Math.min(result.sheet1, range.sheet1);
            result.sheet2 = Math.max(result.sheet2, range.sheet2);
            result.start[1] = Math.min(result.start[1], range.start[1]);
            result.start[0] = Math.min(result.start[0], range.start[0]);
            result.start[1] = Math.min(result.start[1], range.start[1]);
            result.end[0] = Math.max(result.end[0], range.end[0]);
            result.end[1] = Math.max(result.end[1], range.end[1]);
        });
        return result;
    };

    /**
     * Returns the cell range addresses covered by this array, and the passed
     * array of cell range addresses. More precisely, returns the existing
     * intersection ranges from all pairs of the cross product of the arrays of
     * cell range addresses.
     *
     * @param {Range3DArray|Range3D} ranges
     *  The other cell range addresses to be intersected with the cell range
     *  addresses in this array. This method also accepts a single cell range
     *  address as parameter.
     *
     * @returns {Range3DArray}
     *  The cell range addresses covered by this array and the passed array.
     */
    Range3DArray.prototype.intersect = function (ranges) {
        var result = new Range3DArray();
        this.forEach(function (range1) {
            Range3DArray.forEach(ranges, function (range2) {
                var range = range1.intersect(range2);
                if (range) { result.push(range); }
            });
        });
        return result;
    };

    /**
     * Returns a shortened array of cell range addresses containing at most as
     * many cells as specified. If a cell range address in this array does not
     * fit completely due to the remaining number of allowed cells, it will be
     * shortened to the appropriate number of leading sheets and/or upper rows
     * of the range. If the range cannot be shortened exactly to entire sheets,
     * it will be split into two or more ranges (the latter range being the
     * leading part of an inner sheet, and optionally the leading part of an
     * inner row). Overlapping cell range addresses will NOT be merged.
     *
     * Example: Shortening the single range Sheet1:Sheet3!A1:C3 to 26 cells
     * (all cells but the very last cell Sheet3!C3) results in the ranges
     * Sheet1:Sheet2!A1:C3,Sheet3!A1:C2,Sheet3!A3:B3.
     *
     * @param {Number} maxCells
     *  The maximum number of cells allowed in the resulting array.
     *
     * @returns {Range3DArray}
     *  The shortened array of cell range addresses.
     */
    Range3DArray.prototype.shortenTo = function (maxCells) {

        // the resulting cell ranges
        var result = new Range3DArray();

        Utils.iterateArray(this, function (range) {

            // no more cells left (e.g., previous range has occupied the last cell)
            if (maxCells <= 0) { return Utils.BREAK; }

            var // number of cells in the current range
                cells = range.cells();

            // push entirely fitting range, continue with next range
            if (cells <= maxCells) {
                result.push(range);
                maxCells -= cells;
                return;
            }

            // shorten the range to entire sheets
            var sheets = Math.floor(maxCells / range.cols() / range.rows());
            if (sheets > 0) {
                range = range.clone();
                range.sheet2 = range.sheet1 + sheets - 1;
                result.push(range);
                maxCells -= sheets * range.cols() * range.rows();
            }

            // adjust the range to refer to the split sheet only
            range = range.clone();
            range.sheet1 = range.sheet2 = range.sheet1 + sheets;

            // shorten the remaining range to entire rows
            var rows = Math.floor(maxCells / range.cols());
            if (rows > 0) {
                range.end[1] = range.start[1] + rows - 1;
                result.push(range);
                maxCells -= range.cols() * rows;
            }

            // adjust the range to refer to the split row only
            range = range.clone();
            range.start[1] = range.end[1] = range.start[1] + rows;

            // add remaining part of the last row
            if (maxCells > 0) {
                range.end[0] = range.start[0] + maxCells - 1;
                result.push(range);
            }

            // last range has been pushed partly, stop iteration
            return Utils.BREAK;
        });

        return result;
    };

    // exports ================================================================

    return Range3DArray;

});
