/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/intervalarray', [
    'io.ox/office/tk/object/arraytemplate',
    'io.ox/office/spreadsheet/utils/interval'
], function (ArrayTemplate, Interval) {

    'use strict';

    // class IntervalArray ====================================================

    /**
     * Represents an array of column/row index intervals. The array elements
     * are instances of the class Interval.
     *
     * @constructor
     *
     * @extends Array
     */
    var IntervalArray = ArrayTemplate.create(Interval);

    // static methods ---------------------------------------------------------

    /**
     * Creates an index interval array from the passed array of single column
     * or row indexes. The index intervals in the resulting array will be
     * merged, i.e. no index interval overlaps with or adjoins to another index
     * interval. The index intervals will be sorted in ascending order.
     *
     * @param {Array<Number>} indexes
     *  An unordered array of column/row indexes to be merged to an array of
     *  index intervals.
     *
     * @returns {IntervalArray}
     *  A sorted array of unified index intervals exactly covering the passed
     *  column/row indexes.
     */
    IntervalArray.mergeIndexes = function (indexes) {

        // simplifications for small arrays
        switch (indexes.length) {
        case 0:
            return new IntervalArray();
        case 1:
            return new IntervalArray(new Interval(indexes[0]));
        }

        // the resulting intervals
        var result = new IntervalArray();

        // create the resulting intervals (expand last interval, or create a new interval)
        var interval = null;
        _.sortBy(indexes).forEach(function (index) {
            if ((interval = _.last(result)) && (index <= interval.last + 1)) {
                interval.last = index;
            } else {
                result.push(new Interval(index));
            }
        });

        return result;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the number of column/row indexes covered by all index intervals
     * in this array.
     *
     * @returns {Number}
     *  The number of column/row indexes covered by all index intervals.
     */
    IntervalArray.prototype.size = function () {
        return this.reduce(function (count, interval) { return count + interval.size(); }, 0);
    };

    /**
     * Returns whether an index interval in this array contains the passed
     * column or row index.
     *
     * @param {Number} index
     *  The column or row index to be checked.
     *
     * @returns {Boolean}
     *  Whether an index interval in this array contains the passed index.
     */
    IntervalArray.prototype.containsIndex = function (index) {
        return this.some(function (interval) { return interval.containsIndex(index); });
    };

    /**
     * Returns whether all index intervals in the passed array are contained
     * completely in any of the index intervals in this array.
     *
     * @param {IntervalArray|Interval} intervals
     *  The other index intervals to be checked. This method also accepts a
     *  single index interval as parameter.
     *
     * @returns {Boolean}
     *  Whether all index intervals in the passed array are contained
     *  completely in any of the index intervals in this array.
     */
    IntervalArray.prototype.contains = function (intervals) {
        return IntervalArray.every(intervals, function (interval2) {
            return this.some(function (interval1) {
                return interval1.contains(interval2);
            });
        }, this);
    };

    /**
     * Returns whether any index interval in this array overlaps with any index
     * interval in the passed array.
     *
     * @param {IntervalArray|Interval} intervals
     *  The other index intervals to be checked. This method also accepts a
     *  single index interval as parameter.
     *
     * @returns {Boolean}
     *  Whether any index interval in this array overlaps with any index
     *  interval in the passed array.
     */
    IntervalArray.prototype.overlaps = function (intervals) {
        return this.some(function (interval1) {
            return IntervalArray.some(intervals, function (interval2) {
                return interval1.overlaps(interval2);
            });
        });
    };

    /**
     * Returns whether any pair of index intervals in this array overlap each
     * other.
     *
     * @returns {Boolean}
     *  Whether any pair of index intervals in this array overlap each other.
     */
    IntervalArray.prototype.overlapsSelf = function () {

        // simplifications for small arrays
        switch (this.length) {
        case 0:
        case 1:
            return false;
        case 2:
            return this[0].overlaps(this[1]);
        }

        // work on a sorted clone for performance (check adjacent array elements)
        return this.clone().sort().some(function (interval, index, array) {
            return (index > 0) && (array[index - 1].overlaps(interval));
        });
    };

    /**
     * Returns the bounding interval of the index intervals in this array (the
     * smallest index interval that contains all intervals in the array).
     *
     * @returns {Interval|Null}
     *  The bounding interval containing all index intervals; or null, if this
     *  interval array is empty.
     */
    IntervalArray.prototype.boundary = function () {

        switch (this.length) {
        case 0:
            return null;
        case 1:
            // this method must not return original array elements
            return this[0].clone();
        }

        var result = this[0].clone();
        this.forEach(function (interval) {
            result.first = Math.min(result.first, interval.first);
            result.last = Math.max(result.last, interval.last);
        });
        return result;
    };

    /**
     * Returns a shallow copy of this index interval array that does not
     * contain any duplicate index intervals.
     *
     * @returns {IntervalArray}
     *  A shallow copy of this index interval array without any duplicates.
     */
    IntervalArray.prototype.unify = function () {
        var result = new IntervalArray(), map = {};
        this.forEach(function (interval) {
            var key = interval.key();
            if (!map[key]) {
                result.push(interval);
                map[key] = true;
            }
        });
        return result;
    };

    /**
     * Merges the index intervals in this array so that no index interval in
     * the result overlaps with or adjoins to another index interval. The
     * resulting index intervals will be sorted in ascending order.
     *
     * @returns {IntervalArray}
     *  A sorted array of merged index intervals exactly covering the index
     *  intervals in this array.
     */
    IntervalArray.prototype.merge = function () {

        // make deep copy, and sort the intervals by start index
        var result = this.clone(true).sort();

        // merge overlapping intervals in-place
        var interval2 = null;
        result.forEach(function (interval1, index) {
            // extend current interval, as long as adjacent or overlapping with next interval
            while ((interval2 = result[index + 1]) && (interval1.last + 1 >= interval2.first)) {
                interval1.last = Math.max(interval1.last, interval2.last);
                result.splice(index + 1, 1);
            }
        });

        return result;
    };

    /**
     * Returns the index intervals covered by this array, and the passed array
     * of index intervals. More precisely, returns the existing intersection
     * intervals from all interval pairs of the cross product of the interval
     * arrays.
     *
     * @param {IntervalArray|Interval} intervals
     *  The other index intervals to be intersected with this index intervals.
     *  This method also accepts a single index interval as parameter.
     *
     * @returns {IntervalArray}
     *  The index intervals covered by this array, and the passed array of
     *  index intervals.
     */
    IntervalArray.prototype.intersect = function (intervals) {
        var result = new IntervalArray();
        this.forEach(function (interval1) {
            IntervalArray.forEach(intervals, function (interval2) {
                var interval = interval1.intersect(interval2);
                if (interval) { result.push(interval); }
            });
        });
        return result;
    };

    /**
     * Returns the difference of the index intervals in this array, and the
     * passed index intervals (all index intervals contained in this array,
     * that are not contained in the passed array).
     *
     * @param {IntervalArray|Interval} intervals
     *  The index intervals to be removed from this array. This method also
     *  accepts a single index interval as parameter.
     *
     * @returns {IntervalArray}
     *  All index intervals that are contained in this array, but not in the
     *  passed array. May be an empty array, if the passed index intervals
     *  completely cover all index intervals in this array.
     */
    IntervalArray.prototype.difference = function (intervals) {

        // the resulting intervals
        var result = this;

        // reduce intervals in 'result' with each interval from the passed array
        IntervalArray.some(intervals, function (interval2) {

            // initialize the arrays for this iteration
            var source = result;
            result = new IntervalArray();

            // process each interval in the source interval list
            source.forEach(function (interval1) {

                // check if the intervals cover each other at all
                if (!interval1.overlaps(interval2)) {
                    result.push(interval1.clone());
                    return;
                }

                // do nothing if interval2 covers interval1 completely (delete interval1 from the result)
                if (interval2.contains(interval1)) {
                    return;
                }

                // if interval1 starts before interval2, extract the leading part of interval1
                if (interval1.first < interval2.first) {
                    result.push(new Interval(interval1.first, interval2.first - 1));
                }

                // if interval1 ends after interval2, extract the trailing part of interval1
                if (interval1.last > interval2.last) {
                    result.push(new Interval(interval2.last + 1, interval1.last));
                }
            });

            // early exit the loop, if all source intervals have been deleted already
            return result.length === 0;
        });

        // ensure to return a clone
        return (result === this) ? this.clone() : result;
    };

    /**
     * Modifies the indexes of all index intervals in this array relatively.
     *
     * @param {Number} distance
     *  The distance to move the indexes of all index intervals in this array.
     *  Negative values will decrease the indexes.
     *
     * @returns {IntervalArray}
     *  A reference to this instance.
     */
    IntervalArray.prototype.move = function (distance) {
        this.forEach(function (interval) { interval.move(distance); });
        return this;
    };

    /**
     * Returns the string representation of the index intervals as columns.
     *
     * @param {String} [separator=',']
     *  The separator text inserted between the intervals.
     *
     * @returns {String}
     *  The upper-case string representation of the index intervals as columns.
     */
    IntervalArray.prototype.stringifyAsCols = function (separator) {
        return _.invoke(this, 'stringifyAsCols').join(separator);
    };

    /**
     * Returns the string representation of the index intervals as rows.
     *
     * @param {String} [separator=',']
     *  The separator text inserted between the intervals.
     *
     * @returns {String}
     *  The one-based string representation of the index intervals as rows.
     */
    IntervalArray.prototype.stringifyAsRows = function (separator) {
        return _.invoke(this, 'stringifyAsRows').join(separator);
    };

    /**
     * Returns the string representation of the index intervals as columns, or
     * as rows.
     *
     * @param {Boolean} columns
     *  Whether to return the column representation (true), or the row
     *  representation (false) of the index intervals.
     *
     * @param {String} [separator=',']
     *  The separator text inserted between the intervals.
     *
     * @returns {String}
     *  The string representation of this index interval.
     */
    IntervalArray.prototype.stringifyAs = function (columns, separator) {
        return columns ? this.stringifyAsCols(separator) : this.stringifyAsRows(separator);
    };

    // exports ================================================================

    return IntervalArray;

});
