/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/numberformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/utils/fontmetrics',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, DateUtils, FontMetrics, LocaleData, Parser, Formatter, SheetUtils) {

    'use strict';

    var // the exponent character
        EXP = 'E',

        // default built-in formats used in OOXML
        DEFAULT_BUILTIN_FORMAT_MAP = {
            0: 'GENERAL',
            1: '0',
            2: '0.00',
            3: '#,##0',
            4: '#,##0.00',
            9: '0%',
            10: '0.00%',
            11: '0.00E+00',
            12: '# ?/?',
            13: '# ??/??',
            14: LocaleData.SHORT_DATE,
            48: '0.00E+0',
            49: '@'
        };

    // private global functions ===============================================

    /**
     * Reverses the passed map of built-in format codes.
     */
    function reverseBuiltInFormatMap(builtInFormats) {
        return Utils.mapProperties(_.invert(builtInFormats), function (id) { return parseInt(id, 10); });
    }

    // class NumberFormatter ==================================================

    /**
     * A number formatter for a spreadsheet document. Supports the format code
     * syntax used in document operations, and allows to deal with all data
     * types of a spreadsheet document: numbers, strings, Boolean values, and
     * error codes.
     *
     * @constructor
     *
     * @extends Formatter
     *
     * @param {SpreadsheetModel} docModel
     *  The document model containing this instance.
     */
    function NumberFormatter(docModel) {

        var // self reference
            self = this,

            // the application instance
            app = docModel.getApp(),

            // all predefined number format codes, mapped by category
            categoryCodesDatabase = {},

            // all built-in number format codes, mapped by identifier
            builtInFormats = _.clone(DEFAULT_BUILTIN_FORMAT_MAP),

            // all built-in number format codes, mapped by format code
            invBuiltInFormats = reverseBuiltInFormatMap(builtInFormats),

            // the standard format code
            standardCode = 'General',

            // the localized Boolean literals
            falseLiteral = null,
            trueLiteral = null,

            // lazy initialized helper parsing the date
            unformatDate = null,

            // lazy initialized helper parsing the time
            unformatTime = null;

        // base constructor ---------------------------------------------------

        Formatter.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns the decimal string representation of the passed number.
         *
         * @param {Number} number
         *  The number whose decimal string representation will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed number
         *  will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} number
         *      The resulting rounded number.
         *  - {String} text
         *      The complete decimal string representation of the number.
         *  - {String} abs
         *      The decimal string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      number.
         */
        function formatAsDecimal(number, digits) {

            var // the absolute number, rounded to the specified digit count after decimal separator
                round = Utils.round(Math.abs(number), Math.pow(10, -digits)),
                // the integral part of the absolute number
                int = Math.floor(round),
                // the fractional part of the absolute number
                frac = Math.round((round - int) * Math.pow(10, digits)),
                // the text representation of the integral part
                intText = String(int),
                // the text representation of the significant fractional part (without leading zeros)
                fracText = String(frac),
                // the zeros between decimal separator and significant fractional part
                fillText = null,
                // the result returned by this method
                result = { abs: intText, digits: (int === 0) ? 0 : intText.length };

            // add the rounded number to the result
            result.number = (number < 0) ? -round : round;

            // no fraction available: do not add decimal separator
            if (frac > 0) {

                // leading zeros for the fractional part
                fillText = Utils.repeatString('0', digits - fracText.length);
                // remove trailing zeros
                fracText = fracText.replace(/0+$/, '');
                // concatenate integer and fractional part, remove trailing zeros
                result.abs += LocaleData.DEC + fillText + fracText;
                // update number of significant digits
                if (int > 0) { result.digits += fillText.length; }
                result.digits += fracText.length;
            }

            // add final string representation (with sign)
            result.text = (number < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Returns the scientific string representation of the passed number.
         *
         * @param {Number} mant
         *  The mantissa of the number whose scientific string representation
         *  will be returned. Must be in the half-open intervals +-[1, 10).
         *
         * @param {Number} exp
         *  The exponent of the number whose scientific string representation
         *  will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed
         *  mantissa will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} number
         *      The resulting rounded number.
         *  - {String} text
         *      The complete scientific string representation of the number.
         *  - {String} abs
         *      The scientific string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      mantissa.
         */
        function formatAsScientific(mant, exp, digits) {

            var // get the formatted mantissa
                result = formatAsDecimal(mant, digits);

            // if mantissa has been rounded up to 10, increase the exponent
            if (Math.abs(result.number) === 10) {
                result.number /= 10;
                result.abs = '1';
                exp += 1;
            }

            // update the rounded number in the result
            result.number *= Math.pow(10, exp);

            // add the exponent (at least two digits, but may be three digits)
            result.abs += EXP + ((exp < 0) ? '-' : '+');
            exp = Math.abs(exp);
            if (exp < 10) { result.abs += '0'; }
            result.abs += String(exp);

            // add final string representation (with sign)
            result.text = (mant < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Formats a number with increasing number of digits following the
         * decimal separator, as long as the resulting number fits according to
         * the provided result validator.
         *
         * @param {Number} minDigits
         *  The minimum number of digits after the decimal separator.
         *
         * @param {Number} maxDigits
         *  The maximum number of digits after the decimal separator.
         *
         * @param {Function} formatCallback
         *  The callback function that receives the current number of digits
         *  following the decimal separator, and must return a result object
         *  containing the properties 'number', 'text', 'abs', and 'digits'.
         *  See methods formatAsDecimal() and formatAsScientific() above for a
         *  detailed description about this result object.
         *
         * @param {Function} validateCallback
         *  The callback function that checks the current result object it
         *  receives. Must return a Boolean value specifying whether the result
         *  object is valid.
         *
         * @returns {Object}
         *  The last matching result object returned by the callback function.
         */
        function findLongestFormat(minDigits, maxDigits, formatCallback, validateCallback) {

            var currDigits = minDigits,
                currResult = null,
                lastResult = null;

            do {
                currResult = formatCallback(currDigits);
                if (!validateCallback(currResult)) { break; }
                lastResult = currResult;
                currDigits += 1;
            } while (currDigits <= maxDigits);

            return lastResult;
        }

        /**
         * create helper object for converting in-place-formated values back to cell-format
         * takes the editCodesDatabase' date & time
         * parses the special chars between the alphanumercis ('mm.dd.yy' -> '.') and create a regexp for "x.x.x" & "x.x" formats
         * gets the index of the chars in the format ('mm.dd.yy -> m=0 d=1 y=2)
         */
        function initUnformatData() {
            function prepareUnformat(source/*, key0, key1, key2*/) {
                var args = arguments;
                var splitFinder = /\w+((?=[\S])\W)\w+/;
                var splitSymbol = splitFinder.exec(source);

                if (splitSymbol) {
                    var result = {};

                    result.splitSymbol = splitSymbol[1];
                    result.finder = new RegExp('\\d+\\' + result.splitSymbol + '\\d+\\' + result.splitSymbol + '\\d+|\\d+\\' + result.splitSymbol + '\\d+');

                    var possKeys = source.split(new RegExp('\\' + result.splitSymbol, 'g'));
                    _.each(possKeys, function (value, index) {
                        for (var i = 1; i < args.length; i++) {
                            var argKey = args[i];
                            if (value && (value[0].toLowerCase() === argKey)) {
                                result[argKey] = index;
                                break;
                            }
                        }
                    });
                    return result;
                }
            }

            unformatDate = prepareUnformat(LocaleData.SHORT_DATE, 'd', 'm', 'y');
            unformatTime = prepareUnformat(LocaleData.LONG_TIME, 'h', 'm', 's');
        }

        // public methods -----------------------------------------------------

        /**
         * Returns all available number format codes for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {Array|Null}
         *  The format codes for the specified category, or null if no format
         *  codes have been defined for the category. Each element in the
         *  returned array is an object with the following properties:
         *  - {String} value
         *      The actual format code.
         *  - {String} label
         *      A caption label with a formatted example number, intended to be
         *      used in GUI form controls.
         *  - {Boolean} [red=false]
         *      If set to true, the format code contains a [RED] color modifier
         *      for negative numbers.
         */
        this.getCategoryCodes = function (category) {
            var categoryFormats = categoryCodesDatabase[category.toLocaleLowerCase()];
            return _.isArray(categoryFormats) ? _.copy(categoryFormats, true) : null;
        };

        /**
         * Returns the default number format code for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {String|Null}
         *  The default format code for the specified category, or null if no
         *  format code has been found.
         */
        this.getCategoryDefaultCode = function (category) {
            var categoryCodes = categoryCodesDatabase[category.toLocaleLowerCase()];
            return (_.isArray(categoryCodes) && _.isObject(categoryCodes[0])) ? categoryCodes[0].value : null;
        };

        /**
         * Returns the standard format code.
         *
         * @returns {String}
         *  The standard format code.
         */
        this.getStandardCode = function () {
            return standardCode;
        };

        /**
         * Returns the predefined number format code for the specified format
         * code identifier.
         *
         * @param {Number} formatId
         *  The format code identifier.
         *
         * @returns {String|Null}
         *  The number format code for the specified identifier if existing,
         *  otherwise null.
         */
        this.getBuiltInFormatCode = function (formatId) {
            return (formatId in builtInFormats) ? builtInFormats[formatId] : null;
        };

        /**
         * Returns the predefined number format code for the specified format
         * code identifier.
         *
         * @param {String} formatCode
         *  The number format code to be converted to a built-in identifier.
         *
         * @returns {Number|Null}
         *  The built-in identifier of the passed format code if available,
         *  otherwise null.
         */
        this.getBuiltInFormatId = function (formatCode) {
            return (formatCode in invBuiltInFormats) ? invBuiltInFormats[formatCode] : null;
        };

        /**
         * Returns the format code of the passed value of a 'numberFormat'
         * formatting attribute.
         *
         * @param {Object} [numberFormat]
         *  The value of a 'numberFormat' formatting attribute. Must contain at
         *  least one of the properties 'id' (numeric identifier of a built-in
         *  format code), and 'code' (explicit format code, as string). If an
         *  identifier and an explicit format code exists, the format code will
         *  be ignored, and the built-in format code of the identifier will be
         *  returned. If omitted, the standard format code will be returned.
         *
         * @returns {String}
         *  The effective format code of the passed attribute value.
         */
        this.resolveFormatCode = function (numberFormat) {

            var // the built-in identifier
                id = Utils.getIntegerOption(numberFormat, 'id'),
                // explicit format code
                code = Utils.getStringOption(numberFormat, 'code');

            // prefer built-in identifier over format code
            if (_.isNumber(id) && (id in builtInFormats)) { return builtInFormats[id]; }

            // return existing explicit format code
            if (_.isString(code) && (code.length > 0)) { return code; }

            Utils.warn('NumberFormatter.resolveFormatCode(): invalid number format: ' + JSON.stringify(numberFormat));
            return standardCode;
        };

        /**
         * Returns the parsed number format descriptor of the passed value of a
         * 'numberFormat' formatting attribute.
         *
         * @param {Object} numberFormat
         *  The value of a 'numberFormat' formatting attribute. Must contain at
         *  least one of the properties 'id' (numeric identifier of a built-in
         *  format code), and 'code' (explicit format code, as string). If an
         *  identifier and an explicit format code exists, the format code will
         *  be ignored, and the built-in format code of the identifier will be
         *  returned.
         *
         * @returns {ParsedFormat}
         *  The parsed number format descriptor of the passed attribute value.
         */
        this.parseNumberFormat = function (numberFormat) {
            return Parser.parseFormatCode(this.resolveFormatCode(numberFormat));
        };

        /**
         * Converts the passed format code to a value that can be used as
         * 'numberFormat' formatting attribute. Tries to resolve the identifier
         * of a built-in number format if possible.
         *
         * @param {String} formatCode
         *  A format code.
         *
         * @returns {Object}
         *  A number format attribute value. The object will contain the
         *  property 'id', if the format code represents a built-in number
         *  format; otherwise the property 'code' with the passed format code.
         */
        this.createNumberFormat = function (formatCode) {
            // replace format codes of built-in number formats with identifier
            var id = this.getBuiltInFormatId(formatCode);
            return _.isNumber(id) ? { id: id } : { code: formatCode };
        };

        /**
         * Tries to convert the passed text to a cell result literal (either a
         * floating-point number, a Boolean value, an error code, or a string).
         *
         * @param {String} text
         *  The string to be converted to a cell result value.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.keepApos=false]
         *      If set to true, a leading apostrophe in the passed text (used
         *      to protect a string from conversion to other data types) will
         *      be kept in the result. by default, the result will be a string
         *      without the leading apostrophe.
         *
         * @returns {Number|Boolean|String|ErrorCode}
         *  The resulting converted result value (the original string, if the
         *  passed text cannot be converted to another data type).
         */
        this.parseResultValue = function (text, options) {

            // try to parse Boolean literals
            if (text.toUpperCase() === falseLiteral) { return false; }
            if (text.toUpperCase() === trueLiteral) { return true; }

            // try to parse error code literals
            if (app.isLocalizedErrorCode(text)) {
                return app.convertStringToErrorCode(text);
            }

            // try to convert passed text to a floating-point number
            var parseResult = this.parseFormattedNumber(text);
            if (parseResult) { return parseResult.number; }

            // everything else is a regular string
            return Utils.getBooleanOption(options, 'keepApos', false) ? text : text.replace(/^'/, '');
        };

        /**
         * Returns the display string of the passed number formatted with the
         * 'General' number format, fitting into the specified text width.
         *
         * @param {Number} number
         *  The number whose string representation in 'General' number format
         *  will be returned.
         *
         * @param {Number} maxLength
         *  The maximum number of characters allowed for the absolute part of
         *  the passed number, including the decimal separator and the complete
         *  exponent in scientific notation.
         *
         * @param {FontDescriptor} [fontDesc]
         *  A font descriptor with font settings influencing the text width. If
         *  omitted, the result object returned by this method will not contain
         *  the effective pixel width of the formatted number. MUST be
         *  specified, if the parameter 'maxWidth' (see below) is set.
         *
         * @param {Number} [maxWidth]
         *  The maximum width of the resulting string representation, in
         *  pixels. If omitted, the number will be formatted without a pixel
         *  width restriction (but still with the restriction to a maximum
         *  number of text characters specified in the 'maxLength' parameter).
         *
         * @returns {String|Null}
         *  The resulting display text of the number, formatted with the
         *  'General' number format. If a passed maximum pixel width has been
         *  passed to this method, and if it is too small to take the passed
         *  number in either decimal or scientific notation, the null value
         *  will be returned instead.
         */
        this.formatStandardNumber = function (number, maxLength, fontDesc, maxWidth) {

            var // mantissa and exponent (will be -INF for value zero)
                normalized = Utils.normalizeNumber(number),
                // whether to calculate text widths in pixels
                hasFontDesc = _.isObject(fontDesc),
                // whether to restrict the result to a pixel width
                useMaxWidth = hasFontDesc && _.isNumber(maxWidth),
                // the maximum pixel width of a digit
                digitWidth = useMaxWidth ? Math.max(1, FontMetrics.getDigitWidth(fontDesc)) : null,
                // the pixel width of the decimal separator character
                decWidth = useMaxWidth ? FontMetrics.getTextWidth(fontDesc, LocaleData.DEC) : null,
                // the pixel width of the exponent with two digits
                exp2Width = useMaxWidth ? (FontMetrics.getTextWidth(fontDesc, EXP + '+') + 2 * digitWidth) : null,
                // available, minimum, and maximum number of digits for the passed width
                availDigits = 0, minDigits = 0, maxDigits = 0,
                // the formatter callback function (decimal or scientific)
                formatterFunc = null,
                // the validator callback function (depending on pixel mode)
                validateFunc = null,
                // conversion result in decimal notation
                decResult = null,
                // conversion result in scientific notation
                scientResult = null,
                // the result to be used (either decimal or scientific)
                finalResult = null;

            // validates the passed result by character count only
            function validateLength(result) {
                return result.abs.length <= maxLength;
            }

            // validates the passed result by character count and pixel width
            function validateLengthAndWidth(result) {
                return (result.abs.length <= maxLength) && (FontMetrics.getTextWidth(fontDesc, result.text) <= maxWidth);
            }

            // set the validator callback function, depending on pixel mode
            validateFunc = useMaxWidth ? validateLengthAndWidth : validateLength;

            // find the best decimal representation
            if (normalized.exp < 2 - maxLength) {
                // start with zero representation for numbers equal to or very close to zero
                decResult = { number: number, abs: '0', text: (number < 0) ? '-0' : '0', digits: 0 };
                if (!validateFunc(decResult)) { decResult = null; }
            } else if (normalized.exp <= maxLength - 1) {
                // try to find the best explicit representation in a specific interval of exponents
                availDigits = useMaxWidth ? Math.floor((maxWidth - decWidth) / digitWidth) : maxLength;
                minDigits = Math.max(0, -(normalized.exp + 2));
                maxDigits = Math.min(15, Math.max(0, 14 - normalized.exp), maxLength - Math.max(2, normalized.exp), availDigits - Math.max(0, normalized.exp) - 1);
                formatterFunc = _.partial(formatAsDecimal, number);
                decResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateFunc);
            }

            // find the best scientific representation (exclude zero)
            if (_.isFinite(normalized.exp) && ((normalized.exp <= -4) || (normalized.exp >= 4))) {
                availDigits = maxLength - 5;
                if (useMaxWidth) { availDigits = Math.min(availDigits, Math.floor((maxWidth - decWidth - exp2Width) / digitWidth)); }
                minDigits = Utils.minMax(availDigits - 2, 0, 13);
                maxDigits = Utils.minMax(availDigits - 1, 0, 14);
                formatterFunc = _.partial(formatAsScientific, normalized.mant, normalized.exp);
                scientResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateFunc);
            }

            // prefer decimal notation, if it shows at least as much significant digits as the scientific notation
            finalResult = (decResult && (!scientResult || (decResult.digits >= scientResult.digits))) ? decResult : scientResult;
            return finalResult ? finalResult.text : null;
        };

        /**
         * Returns the passed cell value, formatted with the specified number
         * format.
         *
         * @param {Number|String|Boolean|ErrorCode} value
         *  The typed cell value to be formatted.
         *
         * @param {String} [formatCode]
         *  The format code used to format the passed cell value. If omitted,
         *  the standard format code will be used instead.
         *
         * @returns {String|Null}
         *  The formatted value; or null, if the passed format code is invalid,
         *  or if the value cannot be formatted with the format code (for
         *  example, a number is too large to be formatted as date/time).
         */
        this.formatValue = _.wrap(this.formatValue, function (formatValue, value, formatCode) {

            // error codes will never be formatted
            if (SheetUtils.isErrorCode(value)) {
                return app.convertErrorCodeToString(value);
            }

            // format Booleans as text
            if (_.isBoolean(value)) {
                value = app.getBooleanLiteral(value);
            }

            // fall back to standard format code
            if (!_.isString(formatCode)) { formatCode = standardCode; }

            // use base class method to format numbers and strings
            return formatValue.call(this, value, formatCode, { generalLength: SheetUtils.MAX_LENGTH_STANDARD_CELL });
        });

        /**
         * converts the assigned formated numbers to the source-value
         * with help of the unformat-data & the assigned category
         */
        this.parseEditString = function (value, category) {
            switch (category) {

            case 'percent':
                if (_.isString(value)) {
                    value = Parser.stringToNumber(value);
                    if (_.last(value) === '%') {
                        value /= 100;
                    }
                }
                break;

            case 'date':
            case 'time':
            case 'datetime':
                if (unformatDate) {
                    var date = new Date(0);

                    var dateInfo = unformatDate.finder.exec(value);
                    if (dateInfo) {
                        dateInfo = dateInfo[0].split(/\W/g);

                        date.setUTCDate(parseInt(dateInfo[unformatDate.d]));
                        date.setUTCMonth(parseInt(dateInfo[unformatDate.m]) - 1);
                        date.setUTCFullYear(parseInt(dateInfo[unformatDate.y]));
                    }

                    var timeInfo = unformatTime.finder.exec(value);
                    if (timeInfo) {
                        timeInfo = timeInfo[0].split(/\W/g);

                        date.setUTCSeconds(parseInt(timeInfo[unformatTime.s]));
                        date.setUTCMinutes(parseInt(timeInfo[unformatTime.m]));
                        date.setUTCHours(parseInt(timeInfo[unformatTime.h]));
                    }

                    if (dateInfo || timeInfo) {
                        value = self.convertDateToNumber(date);
                    }
                }

                break;
            }
            return value;
        };

        // initialization -----------------------------------------------------

        // initialize document-dependent settings
        app.onImportStart(function () {
            // TODO: disable negative dates handling for OOXML when formatting numbers locally
            this.configure({ negativeDates: true }); // { negativeDates: app.isODF() }
        }, this);

        // initialize locale-dependent settings
        app.onInitLocale(function () {
            falseLiteral = app.getBooleanLiteral(false);
            trueLiteral = app.getBooleanLiteral(true);
        });

        // get localized predefined format codes
        this.listenTo(LocaleData.loadResource('io.ox/office/spreadsheet/resource/numberformats'), 'done', function (data) {
            _.extend(categoryCodesDatabase, data.categories);
            _.extend(builtInFormats, data.builtInFormats);
            invBuiltInFormats = reverseBuiltInFormatMap(builtInFormats);
            standardCode = self.getCategoryDefaultCode('standard') || standardCode;
            initUnformatData();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = app = docModel = categoryCodesDatabase = null;
        });

    } // class NumberFormatter

    // exports ================================================================

    return Formatter.extend({ constructor: NumberFormatter });

});
