/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokenizer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/sheetref',
    'io.ox/office/spreadsheet/model/formula/cellref',
    'io.ox/office/spreadsheet/model/formula/tokens'
], function (Utils, LocaleData, Parser, ModelObject, SheetUtils, FormulaUtils, SheetRef, CellRef, Tokens) {

    'use strict';

    var // convenience shortcuts
        ErrorCodes = SheetUtils.ErrorCodes,
        Address = SheetUtils.Address,
        Range = SheetUtils.Range,

        // the English prefix characters for R1C1 reference notation
        ENGLISH_R1C1_PREFIX_CHARS = 'RC',

        // the localized prefix characters for R1C1 reference notation
        LOCAL_R1C1_PREFIX_CHARS = (function () {

            // R1C1 prefix characters for all supported languages
            var DATABASE = {
                    ar: 'RC',
                    ca: 'FC',
                    cs: 'RC',
                    da: 'RC',
                    de: 'ZS',
                    en: 'RC',
                    es: 'FC',
                    et: 'RC',
                    fi: 'RS',
                    fr: 'LC',
                    hu: 'SO',
                    is: 'RC',
                    it: 'RC',
                    ja: 'RC',
                    ko: 'RC',
                    lv: 'RK',
                    nl: 'RK',
                    nn: 'RC',
                    pl: 'WK',
                    pt: 'LC',
                    ro: 'RC',
                    ru: 'RC',
                    sk: 'RC',
                    sl: 'RC',
                    sv: 'RC',
                    zh: 'RC'
                };

            // pick the current language
            if (LocaleData.LANGUAGE in DATABASE) { return DATABASE[LocaleData.LANGUAGE]; }

            // fall-back to English for unknown languages
            Utils.warn('Tokenizer: missing R1C1 notation characters, falling back to English');
            return 'RC';
        }());

    // private global functions ===============================================

    /**
     * Little helper function that calls the toUpperCase() method at its string
     * argument and returns the result.
     */
    function toUpperCase(text) {
        return text.toUpperCase();
    }

    /**
     * Returns a regular expression that matches strings in R1C1 notation with
     * optional row and column parts.
     *
     * @param {String} prefixChars
     *  The actual prefix characters for the row and column part. MUST be a
     *  string with exactly two characters (row and column).
     *
     * @returns {RegExp}
     *  A regular expression that matches absolute references in R1C1 notation.
     *  Either part may be missing completely, or the prefix character may be
     *  present without index. Matching strings are for example 'R2C3', 'R2C',
     *  'RC3', 'RC', 'R2', 'C3', 'R', 'C', or the empty string (!).
     */
    var getR1C1RegExp = _.memoize(function (prefixChars) {
        return new RegExp('^(?:' + prefixChars[0] + '(\\d*))?(?:' + prefixChars[1] + '(\\d*))?$', 'i');
    });

    // class GrammarConfig ====================================================

    /**
     * @constructor
     *
     * @property {SpreadsheetModel} docModel
     *  The document model owning this configuration.
     *
     * @property {NumberFormatter} formatter
     *  The number formatter of the document model, as convenience shortcut.
     *
     * @property {Number} MAXCOL
     *  The maximum column index supported in the sheets of the spreadsheet
     *  document.
     *
     * @property {Number} MAXROW
     *  The maximum row index supported in the sheets of the spreadsheet
     *  document.
     */
    function GrammarConfig(docModel, grammar) {

        this.docModel = docModel;
        this.formatter = docModel.getNumberFormatter();
        this.MAXCOL = docModel.getMaxCol();
        this.MAXROW = docModel.getMaxRow();

        // the application instance needed to resolve the string literals
        var app = docModel.getApp();

        // grammar dependent configuration items
        switch (grammar) {

        case 'op':
            _.extend(this, {
                // the decimal separator character
                DEC: '.',
                // the list separator character
                SEP: ',',
                // the Boolean TRUE literal
                TRUE: 'TRUE',
                // the Boolean FALSE literal
                FALSE: 'FALSE',
                // all supported error code literals, as strings
                ERRORS: app.getNativeErrorCodes(),
                // maps operator identifiers to operator names
                OPERATORS: {
                    eq: '=',
                    ne: '<>',
                    lt: '<',
                    le: '<=',
                    gt: '>',
                    ge: '>=',
                    con: '&',
                    pct: '%',
                    add: '+',
                    sub: '-',
                    mul: '*',
                    div: '/',
                    pow: '^',
                    list: ',',
                    isect: ' ',
                    range: ':'
                },
                // the row separator character in constant matrixes
                MAT_ROW: ';',
                // the column separator character in constant matrixes
                MAT_COL: ',',
                // the row prefix character in R1C1 references
                R1C1_R: 'R',
                // the column prefix character in R1C1 references
                R1C1_C: 'C',
                // converts error code object to text representation
                getErrorCodeText: _.property('code'),
                // converts text representation to error code object
                getErrorCodeValue: SheetUtils.makeErrorCode,
                // converts native function name to text representation
                getFunctionText: toUpperCase,
                // converts text representation to native function name
                getFunctionValue: toUpperCase
            });
            break;

        case 'ui':
            _.extend(this, {
                // the decimal separator character
                DEC: LocaleData.DEC,
                // the list separator character
                SEP: app.getListSeparator(),
                // the Boolean TRUE literal
                TRUE: app.getBooleanLiteral(true),
                // the Boolean FALSE literal
                FALSE: app.getBooleanLiteral(false),
                // all supported error code literals, as strings
                ERRORS: app.getLocalizedErrorCodes(),
                // maps operator identifiers to operator names
                OPERATORS: {
                    eq: '=',
                    ne: '<>',
                    lt: '<',
                    le: '<=',
                    gt: '>',
                    ge: '>=',
                    con: '&',
                    pct: '%',
                    add: '+',
                    sub: '-',
                    mul: '*',
                    div: '/',
                    pow: '^',
                    list: app.getListSeparator(),
                    isect: ' ',
                    range: ':'
                },
                // the row separator character in constant matrixes
                MAT_ROW: '|',
                // the column separator character in constant matrixes
                MAT_COL: ';',
                // the row prefix character in R1C1 references
                R1C1_R: LOCAL_R1C1_PREFIX_CHARS[0],
                // the column prefix character in R1C1 references
                R1C1_C: LOCAL_R1C1_PREFIX_CHARS[1],
                // converts error code object to text representation
                getErrorCodeText: app.convertErrorCodeToString.bind(app),
                // converts text representation to error code object
                getErrorCodeValue: app.convertStringToErrorCode.bind(app),
                // converts native function name to text representation
                getFunctionText: app.getLocalizedFunctionName.bind(app),
                // converts text representation to native function name
                getFunctionValue: app.getNativeFunctionName.bind(app)
            });
            break;

        default:
            throw new RangeError('Tokenizer.createConfig(): invalid formula grammar "' + grammar + '"');
        }

        // shortcut: the error code text for #REF! errors
        this.REF_ERROR = this.getErrorCodeText(ErrorCodes.REF);

        var // RE pattern snippet for the leading character of a simple sheet name (without apostrophes).
            SHEET_LEADING_CHAR_TOKEN = '[a-z_\xa1-\u2027\u202a-\uffff]',

            // RE pattern snippet for other characters of a simple sheet name (without apostrophes).
            SHEET_SIMPLE_CHAR_TOKEN = '[\\w.\xa1-\u2027\u202a-\uffff]',

            // RE pattern snippet for any character of a complex sheet name (enclosed in apostrophes).
            SHEET_COMPLEX_CHAR_TOKEN = '[^\x00-\x1f\x80-\x9f\\[\\]\'*?:/\\\\]',

            // RE pattern snippet for a characters of a simple external file name (without apostrophes).
            FILE_SIMPLE_CHAR_TOKEN = '[\\w.\\\\\xa1-\u2027\u202a-\uffff]',

            // RE pattern snippet for the leading character of a name or identifier.
            NAME_LEADING_CHAR_TOKEN = '[a-z_\\\\\xa1-\u2027\u202a-\uffff]',

            // RE pattern snippet for other characters of a name or identifier.
            NAME_INNER_CHAR_TOKEN = '[\\w.\\\\\\?\xa1-\u2027\u202a-\uffff]',

            // RE pattern for a simple sheet name.
            SHEET_NAME_SIMPLE_PATTERN = SHEET_LEADING_CHAR_TOKEN + SHEET_SIMPLE_CHAR_TOKEN + '*',

            // RE pattern for a complex sheet name (with embedded double-apostrophes).
            SHEET_NAME_COMPLEX_PATTERN = SHEET_COMPLEX_CHAR_TOKEN + '+(?:(?:\'\')+' + SHEET_COMPLEX_CHAR_TOKEN + '+)*',

            // RE pattern for a simple external file name (enclosed in brackets, without apostrophes).
            // - Group 1: The name of the external file.
            FILE_NAME_SIMPLE_PATTERN = '(?:\\[(' + FILE_SIMPLE_CHAR_TOKEN + '+)\\])',

            // RE pattern for a #REF! error code
            REF_ERROR_PATTERN = _.escapeRegExp(this.REF_ERROR),

            // RE look-ahead pattern to exclude an opening parenthesis and an exclamation mark
            // after another pattern. All valid inner characters of names must be excluded too,
            // otherwise the previous groups would match less characters than they should.
            TERMINATE_REF_PATTERN = '(?!\\(|!|' + NAME_INNER_CHAR_TOKEN + ')',

            // RE pattern for a sheet reference used in formulas, e.g. Sheet1! in the formula =Sheet1!A1,
            // or a broken sheet reference after deleting the referenced sheet, e.g. #REF!! in the formula =#REF!!A1.
            // - Group 1: The name of the sheet (simple sheet name).
            // - Group 2: The name of the sheet (complex sheet name).
            // - Group 3: The error code of a broken sheet reference.
            // Either group 1 or group 2 will match the sheet name, or group 3 will contain the error code (the other groups will be empty).
            SHEET_REF_PATTERN = '(?:(' + SHEET_NAME_SIMPLE_PATTERN + ')|\'(' + SHEET_NAME_COMPLEX_PATTERN + ')\'|(' + REF_ERROR_PATTERN + '))!',

            // RE pattern for an optional sheet reference. See SHEET_REF_PATTERN for details.
            SHEET_REF_PATTERN_OPT = '(?:' + SHEET_REF_PATTERN + ')?',

            // RE pattern for a sheet range reference used in formulas, e.g. Sheet1:Sheet2! in the formula =SUM(Sheet1:Sheet2!A1).
            // - Group 1: The name of the first sheet (simple sheet names).
            // - Group 2: The name of the second sheet (simple sheet names).
            // - Group 3: The name of the first sheet (complex sheet names).
            // - Group 4: The name of the second sheet (complex sheet names).
            // Either group 1 or group 3 will match the first sheet name (the other group will be empty).
            // Either group 2 or group 4 will match the second sheet name (the other group will be empty).
            SHEET_RANGE_REF_PATTERN = '(?:(' + SHEET_NAME_SIMPLE_PATTERN + '):(' + SHEET_NAME_SIMPLE_PATTERN + ')|\'(' + SHEET_NAME_COMPLEX_PATTERN + '):(' + SHEET_NAME_COMPLEX_PATTERN + ')\')!',

            // RE pattern for a column name (absolute or relative).
            // - Group 1: The absolute marker.
            // - Group 2: The column name.
            COL_REF_PATTERN = '(\\$)?([a-z]+)',

            // RE pattern for a row name (absolute or relative).
            // - Group 1: The absolute marker.
            // - Group 2: The row name.
            ROW_REF_PATTERN = '(\\$)?([0-9]+)',

            // RE pattern for a cell address.
            // - Group 1: The absolute marker for the column.
            // - Group 2: The name of the column.
            // - Group 3: The absolute marker for the row.
            // - Group 4: The name of the row.
            CELL_REF_PATTERN = COL_REF_PATTERN + ROW_REF_PATTERN,

            // RE pattern for a cell range address.
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column.
            // - Group 3: The absolute marker for the first row.
            // - Group 4: The name of the first row.
            // - Group 5: The absolute marker for the second column.
            // - Group 6: The name of the second column.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            CELL_RANGE_REF_PATTERN = CELL_REF_PATTERN + ':' + CELL_REF_PATTERN,

            // RE pattern for a column range reference, e.g. $C:$C.
            // - Group 1: The absolute marker for the first column.
            // - Group 2: The name of the first column.
            // - Group 3: The absolute marker for the second column.
            // - Group 4: The name of the second column.
            COL_RANGE_REF_PATTERN = COL_REF_PATTERN + ':' + COL_REF_PATTERN,

            // RE pattern for a row range reference, e.g. $2:$2.
            // - Group 1: The absolute marker for the first row.
            // - Group 2: The name of the first row.
            // - Group 3: The absolute marker for the second row.
            // - Group 4: The name of the second row.
            ROW_RANGE_REF_PATTERN = ROW_REF_PATTERN + ':' + ROW_REF_PATTERN,

            // RE pattern for a complete name or identifier used in formulas.
            // - Group 1: The entire name.
            NAME_REF_PATTERN = '(' + NAME_LEADING_CHAR_TOKEN + NAME_INNER_CHAR_TOKEN + '*)',

            // RE pattern for remaining invalid characters
            // - Group 1: The matched invalid character.
            INVALID_CHAR_PATTERN = '([!§$?°~#;,.|])';

        // create the map of all regular expressions
        this.RE = {

            // Any whitespace or other non-printable characters.
            WHITESPACE: /^[\s\x00-\x1f\x80-\x9f]+/,

            // String literals: Enclosed in double quotes, embedded double quotes
            // are represented by a sequence of two double quote characters.
            STRING_LIT: /^(".*?")+/,

            // Boolean literals: Must not be followed by any character valid inside an
            // identifier. This prevents to match 'trueA' or 'false_1' or similar name tokens.
            BOOLEAN_LIT: new RegExp('^(' + _.escapeRegExp(this.TRUE) + '|' + _.escapeRegExp(this.FALSE) + ')(?!' + NAME_INNER_CHAR_TOKEN + ')', 'i'),

            // Error code literals (only supported error codes, not any string
            // starting with a hash character).
            ERROR_LIT: new RegExp('^(' + _.map(this.ERRORS, _.escapeRegExp).join('|') + ')(?!' + NAME_INNER_CHAR_TOKEN + ')', 'i'),

            // Opening parenthesis of a matrix literal.
            MATRIX_OPEN: /^{/,

            // The column separator in a matrix literal.
            MATRIX_COL_SEPARATOR: new RegExp('^' + _.escapeRegExp(this.MAT_COL)),

            // The row separator in a matrix literal.
            MATRIX_ROW_SEPARATOR: new RegExp('^' + _.escapeRegExp(this.MAT_ROW)),

            // Closing parenthesis of a matrix literal.
            MATRIX_CLOSE: /^}/,

            // The list/parameter separator character.
            SEPARATOR: new RegExp('^' + _.escapeRegExp(this.SEP)),

            // All unary and binary operators, as array sorted by string length.
            OPERATORS: _.map(this.OPERATORS, function (op, id) {
                return { id: id, op: op, re: new RegExp('^' + _.escapeRegExp(op)) };
            }).sort(function (desc1, desc2) {
                return desc2.op.length - desc1.op.length;
            }),

            // The opening parenthesis.
            OPEN: /^\(/,

            // The closing parenthesis.
            CLOSE: /^\)/,

            // A cell reference, e.g. A1 or Sheet1!$A$1, but without trailing parenthesis, e.g. F2().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The absolute marker for the column.
            // - Group 5: The column name.
            // - Group 6: The absolute marker for the row.
            // - Group 7: The row name.
            CELL_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + CELL_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A cell reference in multiple sheets, e.g. Sheet1:Sheet2!$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the column.
            // - Group 6: The column name.
            // - Group 7: The absolute marker for the row.
            // - Group 8: The row name.
            CELL_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + CELL_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference, e.g. A1:B2 or Sheet1!$A$1:$B$2, but without trailing parenthesis, e.g. A1:F2().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The absolute marker for the first column.
            // - Group 5: The name of the first column.
            // - Group 6: The absolute marker for the first row.
            // - Group 7: The name of the first row.
            // - Group 8: The absolute marker for the second column.
            // - Group 9: The name of the second column.
            // - Group 10: The absolute marker for the second row.
            // - Group 11: The name of the second row.
            CELL_RANGE_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + CELL_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A range reference in multiple sheets, e.g. Sheet1:Sheet2!$A$1:$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column.
            // - Group 11: The absolute marker for the second row.
            // - Group 12: The name of the second row.
            CELL_RANGE_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + CELL_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column range, e.g. A:B or Sheet1!$C:$C (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The absolute marker for the first column.
            // - Group 5: The name of the first column.
            // - Group 6: The absolute marker for the second column.
            // - Group 7: The name of the second column.
            COL_RANGE_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + COL_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A column range in multiple sheets, Sheet1:Sheet2!$C:$C (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the second column.
            // - Group 8: The name of the second column.
            COL_RANGE_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + COL_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row range, e.g. 1:2 or $3:$3 (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The absolute marker for the first row.
            // - Group 5: The name of the first row.
            // - Group 6: The absolute marker for the second row.
            // - Group 7: The name of the second row.
            ROW_RANGE_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + ROW_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A row range in multiple sheets, e.g. Sheet1:Sheet2!$3:$3 (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first row.
            // - Group 6: The name of the first row.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            ROW_RANGE_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + ROW_RANGE_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A reference error with sheet reference, e.g. Sheet1!#REF!.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            ERROR_REF: new RegExp('^' + SHEET_REF_PATTERN + REF_ERROR_PATTERN, 'i'),

            // A reference error in multiple sheets, e.g. Sheet1:Sheet2!#REF!.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            ERROR_CUBE_REF: new RegExp('^' + SHEET_RANGE_REF_PATTERN + REF_ERROR_PATTERN, 'i'),

            // Any function name (names followed by an opening parenthesis), e.g. SUM() or Module1!macro().
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The name of the function.
            FUNCTION_REF: new RegExp('^' + SHEET_REF_PATTERN_OPT + NAME_REF_PATTERN + '(?=\\()', 'i'),

            // Defined names with sheet reference, e.g. Sheet1!my_name.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The error code of a broken sheet reference.
            // - Group 4: The name.
            SHEET_NAME_REF: new RegExp('^' + SHEET_REF_PATTERN + NAME_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // Defined names without sheet reference, e.g. my_name.
            // - Group 1: The name.
            GLOBAL_NAME_REF: new RegExp('^' + NAME_REF_PATTERN + TERMINATE_REF_PATTERN, 'i'),

            // A single unsupported character.
            INVALID_CHAR: new RegExp('^' + INVALID_CHAR_PATTERN),

            // Matches a valid defined name (full string match).
            NAME_COMPLETE: new RegExp('^' + NAME_REF_PATTERN + '$', 'i'),

            // Matches a simple sheet name that does not need to be enclosed in apostrophes (full string match).
            UNENCODED_SHEET_NAME_COMPLETE: new RegExp('^' + SHEET_NAME_SIMPLE_PATTERN + '$', 'i'),

            // Matches a simple sheet name or a sheet range that does not need to be enclosed in apostrophes
            // (full string match). Supports an optional leading reference to an external file (simple name),
            // enclosed in brackets. Examples: 'Sheet1', 'Sheet1:Sheet2', '[file1.xlsx]Sheet1', '[path\to\file1.xlsx]Sheet1:Sheet2'
            UNENCODED_EXT_SHEET_COMPLETE: new RegExp('^' + FILE_NAME_SIMPLE_PATTERN + '?' + SHEET_NAME_SIMPLE_PATTERN + '(:' + SHEET_NAME_SIMPLE_PATTERN + ')?$', 'i')
        };

        FormulaUtils.info('GrammarConfig(): grammar=' + grammar + ' config=', this);

    } // class GrammarConfig

    // static methods ---------------------------------------------------------

    /**
     * Creates the configuration structure for the specified formula grammar.
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model.
     *
     * @param {String} grammar
     *  The formula grammar identifier (see the class constructor of the class
     *  Tokenizer for details).
     *
     * @returns {GrammarConfig}
     *  The configuration for the specified formula grammar.
     */
    GrammarConfig.create = (function () {

        // cache for grammar configurations, mapped by document UID, then by grammar
        var grammarConfigCache = {};

        function create(docModel, grammar) {

            // create a cache entry for new document, delete it on destruction of the document
            var modelUid = docModel.getUid();
            if (!(modelUid in grammarConfigCache)) {
                grammarConfigCache[modelUid] = {};
                docModel.registerDestructor(function () {
                    delete grammarConfigCache[modelUid];
                });
            }

            // immediately return existing cache entry
            var config = grammarConfigCache[modelUid][grammar];
            if (config) { return config; }

            // create a new configuration object, and store it in the cache
            config = new GrammarConfig(docModel, grammar);
            grammarConfigCache[modelUid][grammar] = config;
            return config;
        }

        return create;
    }());

    // public methods ---------------------------------------------------------

    /**
     * Generates the display text for the passed literal value, according to
     * the formula grammar represented by this configuration instance.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The value to be converted to a number.
     *
     * @returns {String|Null}
     *  The string representation of the value, if it is valid; otherwise null.
     */
    GrammarConfig.prototype.generateLiteralText = function (value) {

        // numbers
        if (_.isNumber(value)) {
            return !_.isFinite(value) ? this.getErrorCodeText(ErrorCodes.NUM) :
                (Math.abs(value) < FormulaUtils.MIN_VALUE) ? '0' :
                this.formatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT).replace(LocaleData.DEC, this.DEC);
        }

        // strings
        if (_.isString(value)) {
            // duplicate inner quotes, enclose in quotes
            return '"' + value.replace(/"/g, '""') + '"';
        }

        // Boolean values
        if (_.isBoolean(value)) {
            return value ? this.TRUE : this.FALSE;
        }

        // error codes
        if (SheetUtils.isErrorCode(value)) {
            return this.getErrorCodeText(value);
        }

        // empty function parameters
        if (!_.isNull(value)) {
            Utils.warn('GrammarConfig.generateLiteralText(): unsupported value type');
        }
        return '';
    };

    /**
     * Returns the string representation of the passed sheet references. Will
     * be used as prefix by cell references and defined names.
     *
     * @param {SheetRef} sheet1Ref
     *  A descriptor for the first sheet in the reference.
     *
     * @param {SheetRef|Null} sheet2Ref
     *  A descriptor for the second sheet in the reference, e.g. 'Sheet2' in
     *  the reference Sheet1:Sheet2!A1. Can be set to null for a single sheet
     *  reference.
     *
     * @returns {String}
     *  The string representation for the passed sheet references, including
     *  the trailing exclamation mark.
     */
    GrammarConfig.prototype.generateSheetName = function (sheet1Ref, sheet2Ref) {

        // convert first sheet name
        if (!sheet1Ref) { return ''; }
        var sheet1Name = _.isNumber(sheet1Ref.sheet) ? this.docModel.getSheetName(sheet1Ref.sheet) : sheet1Ref.sheet;
        if (!_.isString(sheet1Name)) { return this.REF_ERROR + '!'; }

        // decide whether both sheet names are simple (no enclosing apostrophes needed)
        var simpleNames = this.RE.UNENCODED_SHEET_NAME_COMPLETE.test(sheet1Name);

        // convert second sheet name
        if (sheet2Ref && !sheet1Ref.equals(sheet2Ref)) {
            var sheet2Name = _.isNumber(sheet2Ref.sheet) ? this.docModel.getSheetName(sheet2Ref.sheet) : sheet2Ref.sheet;
            if (!_.isString(sheet2Name)) { return this.REF_ERROR + '!'; }
            simpleNames = simpleNames && this.RE.UNENCODED_SHEET_NAME_COMPLETE.test(sheet2Name);
            sheet1Name += ':' + sheet2Name;
        }

        // enclose complex names in apostrophes
        return simpleNames ? (sheet1Name + '!') : ('\'' + sheet1Name.replace(/'/g, '\'\'') + '\'!');
    };

    /**
     * Generates the string representation for the passed references.
     *
     * @param {CellRef} cell1Ref
     *  A descriptor for the first cell address in the reference, e.g. 'A1' in
     *  the reference Sheet1!A1:B2.
     *
     * @param {CellRef|Null} cell2Ref
     *  A descriptor for the second cell address in the reference, e.g. 'B2' in
     *  the reference Sheet1!A1:B2. Can be set to null for a single cell
     *  reference.
     *
     * @param {SheetRef} sheet1Ref
     *  A descriptor for the first sheet in the reference.
     *
     * @param {SheetRef|Null} sheet2Ref
     *  A descriptor for the second sheet in the reference, e.g. 'Sheet2' in
     *  the reference Sheet1:Sheet2!A1. Can be set to null for a single sheet
     *  reference.
     *
     * @returns {String}
     *  The string representation for the passed references.
     */
    GrammarConfig.prototype.generateReference = function (cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {

        var range = cell1Ref ? Range.createFromAddresses(cell1Ref.address, cell2Ref && cell2Ref.address) : null,
            text = this.generateSheetName(sheet1Ref, sheet2Ref);

        if (!range) {
            text += this.REF_ERROR;

        // generate row interval (preferred over column interval for entire sheet range)
        } else if (cell1Ref.absCol && cell2Ref && cell2Ref.absCol && this.docModel.isRowRange(range)) {
            text += cell1Ref.rowText() + ':' + cell2Ref.rowText();

        // generate column interval
        } else if (cell1Ref.absRow && cell2Ref && cell2Ref.absRow && this.docModel.isColRange(range)) {
            text += cell1Ref.colText() + ':' + cell2Ref.colText();

        // generate range or cell address
        } else {
            text += cell1Ref.refText();
            if (cell2Ref) { text += ':' + cell2Ref.refText(); }
        }

        return text;
    };

    // class TokenDescriptor ==================================================

    /**
     * A descriptor for a single formula token parsed from a formula expression
     * string, together with additional information such as the original
     * display text of the token.
     *
     * @property {BaseToken} token
     *  The formula token.
     *
     * @property {Number} index
     *  The array index of the formula token.
     *
     * @property {String} text
     *  The display text of the token from the original formula expression.
     *
     * @property {Number} start
     *  The start position of the text in the original formula expression.
     *
     * @property {Number} end
     *  The end position of the text in the original formula expression.
     */
    function TokenDescriptor(token, text) {

        this.token = token;
        this.text = text;
        this.start = 0;
        this.end = 0;

    } // class TokenDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Returns a text description of the token for debugging (!) purposes.
     */
    TokenDescriptor.prototype.toString = function () {
        return this.token.toString();
    };

    // class Tokenizer ========================================================

    /**
     * Parses formula expressions to arrays of formula tokens.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     *
     * @param {String} grammar
     *  The identifier of the formula grammar to be used to parse formula
     *  expressions. Supported values are:
     *  - 'op': Fixed token representations as used in operations (matches the
     *      formula grammar used in OOXML spreadsheet files).
     *  - 'ui': The localized token representations according to the current
     *      GUI language of the application.
     */
    function Tokenizer(docModel, grammar) {

        var // the formula grammar configuration
            config = GrammarConfig.create(docModel, grammar);

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Returns the identifier of the formula grammar used by this tokenizer
         * instance.
         *
         * @returns {String}
         *  The formula grammar. See class constructor for details.
         */
        this.getGrammar = function () {
            return grammar;
        };

        /**
         * Returns the configuration for the formula grammar used by this
         * tokenizer instance.
         *
         * @returns {GrammarConfig}
         *  The configuration for the formula grammar used by this tokenizer.
         */
        this.getConfig = function () {
            return config;
        };

        /**
         * Checks if the passed string can be used as label for a named formula
         * expression (a.k.a. defined name a.k.a. named range).
         *
         * @param {String} label
         *  The string to be checked.
         *
         * @return {String}
         *  The empty string, if the passed text is a valid name; otherwise one
         *  of the following error codes:
         *  - 'name:empty': The passed label is empty.
         *  - 'name:invalid': The passed label contains invalid characters.
         *  - 'name:address': The passed label would be a valid, but conflicts
         *      with the representation of a relative cell reference in A1
         *      notation, or a cell reference in R1C1 notation (either English,
         *      e.g. 'R1C1', or according to current UI language, e.g. 'Z1S1'
         *      in German).
         */
        this.validateNameLabel = function (label) {

            // returns whether the text is a valid R1C1 reference for the passed prefix characters
            function isValidR1C1(prefixChars) {

                // check that text matches the regular expression
                var matches = getR1C1RegExp(prefixChars).exec(label);
                if (!matches) { return false; }

                // extract column and row index (row before column in R1C1 notation!)
                var row = matches[1] ? (parseInt(matches[1], 10) - 1) : 0,
                    col = matches[2] ? (parseInt(matches[2], 10) - 1) : 0;

                // column and row index must be valid in the document
                return docModel.isValidAddress(new Address(col, row));
            }

            // the passed string must not be empty
            if (label.length === 0) { return 'name:empty'; }

            // check that the passed string does not contain invalid characters
            if (!config.RE.NAME_COMPLETE.test(label)) { return 'name:invalid'; }

            // bug 38786: in OOXML documents (bug 40203: not in ODF documents):
            // check that label does not look like a cell address
            if (docModel.getApp().isOOXML()) {

                // check that the string is not a valid relative reference in A1 notation
                var address = Address.parse(label);
                if (address && docModel.isValidAddress(address)) { return 'name:address'; }

                // check that the string is not a valid reference in R1C1 notation
                // (neither English nor localized)
                if (isValidR1C1(ENGLISH_R1C1_PREFIX_CHARS) || isValidR1C1(LOCAL_R1C1_PREFIX_CHARS)) {
                    return 'name:address';
                }
            }

            // the passed text is a valid label for a defined name
            return '';
        };

        /**
         * Encloses the passed sheet name in apostrophes, if it contains any
         * characters that would be ambiguous in a formula expression (e.g.
         * whitespace or any operator characters). Apostrophes embedded in the
         * sheet name will be doubled.
         *
         * @param {String} sheetName
         *  The sheet name to be encoded.
         *
         * @returns {String}
         *  The encoded sheet name ready to be used in a formula expression.
         */
        this.encodeExtendedSheetRef = function (sheetName) {

            var // whether encoding is not needed
                simpleName = config.RE.UNENCODED_EXT_SHEET_COMPLETE.test(sheetName);

            // enclose complex names in apostrophes
            return simpleName ? sheetName : ('\'' + sheetName.replace(/'/g, '\'\'') + '\'');
        };

        /**
         * Converts the specified formula expression to an array of formula
         * tokens.
         *
         * @param {String} formula
         *  The formula string, without the leading equality sign.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.extendSheet]
         *      If specified, all cell references without a sheet reference,
         *      and all references to existing sheet-local names will be
         *      extended with a sheet reference pointing to the sheet contained
         *      in this option.
         *      Example: The formula expression 'A1+global+local' extended to
         *      the first sheet may become 'Sheet1!A1+global+Sheet1!local'.
         *
         * @returns {Array<TokenDescriptor>}
         *  An array of descriptors for all parsed formula tokens.
         */
        this.parseFormula = function (formula, options) {

            var // index of the sheet to extend local references to
                extendSheet = Utils.getIntegerOption(options, 'extendSheet', -1),
                // the sheet model of the specified extension sheet
                extendSheetModel = docModel.getSheetModel(extendSheet),
                // the matches of the regular expressions
                matches = null,
                // whether a matrix literal is currently being parsed
                matrix = false,
                // start and end reference structures for sheet references
                sheet1Ref = null, sheet2Ref = null,
                // start and end reference structures for column/row references
                cell1Ref = null, cell2Ref = null,
                // parsed literal
                value = null,
                // the descriptors of the resulting formula tokens
                tokenDescs = [];

            // pushes the passed token and the text currently matched in 'matches'
            function pushToken(token) {
                tokenDescs.push(new TokenDescriptor(token, matches[0]));
            }

            // pushes a reference token according to the current values of the reference variables
            function pushReferenceToken() {
                // bug 40293: add sheet reference to sheet-local cell references if specified
                if (!sheet1Ref && extendSheetModel) { sheet1Ref = new SheetRef(extendSheet, true); }
                pushToken(new Tokens.ReferenceToken(docModel, cell1Ref, cell2Ref, sheet1Ref, sheet2Ref));
            }

            // pushes a name token according to the current values of the reference variables
            function pushNameToken(isFunc) {
                // bug 40293: add sheet reference to existing sheet-local names if specified
                if (!isFunc && !sheet1Ref && extendSheetModel && extendSheetModel.getNameCollection().hasNameModel(value)) {
                    sheet1Ref = new SheetRef(extendSheet, true);
                }
                pushToken(new Tokens.NameToken(docModel, value, sheet1Ref, isFunc));
            }

            // callback must be defined outside the while-loop
            function matchOperator(desc) {
                if ((matches = desc.re.exec(formula))) {
                    pushToken(new Tokens.OperatorToken(desc.id));
                    return true;
                }
            }

            // extends the last 'bad' token, or appends a new 'bad' token
            function appendBadText() {
                var tokenDesc = _.last(tokenDescs);
                if (tokenDesc && tokenDesc.token.isType('bad')) {
                    tokenDesc.token.appendValue(matches[0]);
                    tokenDesc.text += matches[0];
                } else {
                    pushToken(new Tokens.FixedToken('bad', matches[0]));
                }
            }

            // extract all supported tokens from start of formula string
            FormulaUtils.info('Tokenizer.parseFormula(): formula="' + formula + '"');
            while (formula.length > 0) {

                // try/finally to be able to continue the while-loop at any place
                try {

                    // whitespace (MUST be parsed before operators)
                    if ((matches = config.RE.WHITESPACE.exec(formula))) {
                        pushToken(new Tokens.FixedToken('ws', matches[0]));
                        continue;
                    }

                    // the following tokens are invalid in matrix literals
                    if (!matrix) {

                        // opening parenthesis of a matrix literal
                        if ((matches = config.RE.MATRIX_OPEN.exec(formula))) {
                            pushToken(new Tokens.MatrixDelimiterToken('mat_open'));
                            matrix = true;
                            continue;
                        }

                        // list/parameter separator (MUST be done before operators)
                        if ((matches = config.RE.SEPARATOR.exec(formula))) {
                            pushToken(new Tokens.SeparatorToken());
                            continue;
                        }

                        // unary/binary operator
                        if (config.RE.OPERATORS.some(matchOperator)) {
                            continue;
                        }

                        // opening parenthesis
                        if ((matches = config.RE.OPEN.exec(formula))) {
                            pushToken(new Tokens.ParenthesisToken(true));
                            continue;
                        }

                        // closing parenthesis
                        if ((matches = config.RE.CLOSE.exec(formula))) {
                            pushToken(new Tokens.ParenthesisToken(false));
                            continue;
                        }

                        // cell range reference in multiple sheets (before other references)
                        if ((matches = config.RE.CELL_RANGE_CUBE_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[3]);
                            sheet2Ref = SheetRef.create(docModel, matches[2], matches[4]);
                            cell1Ref = CellRef.create(docModel, matches[5], matches[6], matches[7], matches[8]);
                            cell2Ref = CellRef.create(docModel, matches[9], matches[10], matches[11], matches[12]);
                            if (sheet1Ref && sheet2Ref && cell1Ref && cell2Ref) {
                                pushReferenceToken();
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a local name such as 'Sheet1:Sheet2!A1:name42'
                        }

                        // cell reference in multiple sheets
                        if ((matches = config.RE.CELL_CUBE_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[3]);
                            sheet2Ref = SheetRef.create(docModel, matches[2], matches[4]);
                            cell1Ref = CellRef.create(docModel, matches[5], matches[6], matches[7], matches[8]);
                            cell2Ref = null;
                            if (cell1Ref) {
                                pushReferenceToken();
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a local name such as 'Sheet1:Sheet2!name42'
                        }

                        // column range reference in multiple sheets
                        if ((matches = config.RE.COL_RANGE_CUBE_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[3]);
                            sheet2Ref = SheetRef.create(docModel, matches[2], matches[4]);
                            cell1Ref = CellRef.create(docModel, matches[5], matches[6], true, 0);
                            cell2Ref = CellRef.create(docModel, matches[7], matches[8], true, config.MAXROW);
                            if (cell1Ref && cell2Ref) {
                                pushReferenceToken();
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a range of names such as 'Sheet1:Sheet2!name:name'
                        }

                        // row range reference in multiple sheets
                        if ((matches = config.RE.ROW_RANGE_CUBE_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[3]);
                            sheet2Ref = SheetRef.create(docModel, matches[2], matches[4]);
                            cell1Ref = CellRef.create(docModel, true, 0, matches[5], matches[6]);
                            cell2Ref = CellRef.create(docModel, true, config.MAXCOL, matches[7], matches[8]);
                            if (cell1Ref && cell2Ref) {
                                pushReferenceToken();
                                continue;
                            }
                            // otherwise, continue with other REs
                        }

                        // reference error in multiple sheets
                        if ((matches = config.RE.ERROR_CUBE_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[3]);
                            sheet2Ref = SheetRef.create(docModel, matches[2], matches[4]);
                            cell1Ref = cell2Ref = null;
                            pushReferenceToken();
                            continue;
                        }

                        // cell range reference (before cell references)
                        if ((matches = config.RE.CELL_RANGE_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[2], matches[3]);
                            sheet2Ref = null;
                            cell1Ref = CellRef.create(docModel, matches[4], matches[5], matches[6], matches[7]);
                            cell2Ref = CellRef.create(docModel, matches[8], matches[9], matches[10], matches[11]);
                            if (cell1Ref && cell2Ref) {
                                pushReferenceToken();
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a local name such as 'A1:name42'
                        }

                        // cell reference
                        if ((matches = config.RE.CELL_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[2], matches[3]);
                            sheet2Ref = null;
                            cell1Ref = CellRef.create(docModel, matches[4], matches[5], matches[6], matches[7]);
                            cell2Ref = null;
                            if (cell1Ref) {
                                pushReferenceToken();
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a local name such as 'name42'
                        }

                        // column range reference
                        if ((matches = config.RE.COL_RANGE_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[2], matches[3]);
                            sheet2Ref = null;
                            cell1Ref = CellRef.create(docModel, matches[4], matches[5], true, 0);
                            cell2Ref = CellRef.create(docModel, matches[6], matches[7], true, config.MAXROW);
                            if (cell1Ref && cell2Ref) {
                                pushReferenceToken();
                                continue;
                            }
                            // otherwise, continue with other REs, it may be a range of names such as 'name:name'
                        }

                        // row range reference
                        if ((matches = config.RE.ROW_RANGE_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[2], matches[3]);
                            sheet2Ref = null;
                            cell1Ref = CellRef.create(docModel, true, 0, matches[4], matches[5]);
                            cell2Ref = CellRef.create(docModel, true, config.MAXCOL, matches[6], matches[7]);
                            if (cell1Ref && cell2Ref) {
                                pushReferenceToken();
                                continue;
                            }
                            // otherwise, continue with other REs
                        }

                        // reference error
                        if ((matches = config.RE.ERROR_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[2], matches[3]);
                            sheet2Ref = cell1Ref = cell2Ref = null;
                            pushReferenceToken();
                            continue;
                        }

                        // defined name with sheet reference
                        if ((matches = config.RE.SHEET_NAME_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[2], matches[3]);
                            value = matches[4];
                            pushNameToken(false);
                            continue;
                        }

                        // function call (with and without sheet reference)
                        if ((matches = config.RE.FUNCTION_REF.exec(formula))) {
                            sheet1Ref = SheetRef.create(docModel, matches[1], matches[2], matches[3]);
                            value = sheet1Ref ? matches[4] : config.getFunctionValue(matches[4]);
                            pushNameToken(true);
                            continue;
                        }
                    }

                    // tokens valid exclusively in matrix literals
                    if (matrix) {

                        // row separator in a matrix literal
                        if ((matches = config.RE.MATRIX_ROW_SEPARATOR.exec(formula))) {
                            pushToken(new Tokens.MatrixDelimiterToken('mat_row'));
                            continue;
                        }

                        // column separator in a matrix literal
                        if ((matches = config.RE.MATRIX_COL_SEPARATOR.exec(formula))) {
                            pushToken(new Tokens.MatrixDelimiterToken('mat_col'));
                            continue;
                        }

                        // closing parenthesis of a matrix literal
                        if ((matches = config.RE.MATRIX_CLOSE.exec(formula))) {
                            pushToken(new Tokens.MatrixDelimiterToken('mat_close'));
                            matrix = false;
                            continue;
                        }
                    }

                    // literals are valid in regular formula structure, and in matrix literals

                    // number literal (with leading sign character in matrix literals),
                    // using the parser method (returns a different result object)
                    if ((matches = Parser.parseLeadingNumber(formula, { sep: config.DEC, sign: matrix }))) {
                        value = matches.number;
                        // simulate the result array of the RegExp.exec() method for pushToken() and the 'finally' block
                        matches = [matches.text];
                        pushToken(new Tokens.LiteralToken(value));
                        continue;
                    }

                    // string literal
                    if ((matches = config.RE.STRING_LIT.exec(formula))) {
                        value = matches[0].slice(1, matches[0].length - 1).replace(/""/g, '"');
                        pushToken(new Tokens.LiteralToken(value));
                        continue;
                    }

                    // Boolean literal (before defined names)
                    if ((matches = config.RE.BOOLEAN_LIT.exec(formula))) {
                        value = matches[0].toUpperCase() === config.TRUE;
                        pushToken(new Tokens.LiteralToken(value));
                        continue;
                    }

                    // error code literal
                    if ((matches = config.RE.ERROR_LIT.exec(formula))) {
                        value = config.getErrorCodeValue(matches[0]);
                        pushToken(new Tokens.LiteralToken(value));
                        continue;
                    }

                    // defined name without sheet reference (weakest pattern, try last)
                    if (!matrix && (matches = config.RE.GLOBAL_NAME_REF.exec(formula))) {
                        sheet1Ref = null;
                        value = matches[1];
                        pushNameToken(false);
                        continue;
                    }

                    // invalid characters
                    if ((matches = config.RE.INVALID_CHAR.exec(formula))) {
                        appendBadText();
                        continue;
                    }

                    // error, or other unexpected/unsupported tokens:
                    // push entire remaining formula text as 'bad' token
                    matches = [formula];
                    appendBadText();
                }

                // remove the current match from the formula string
                finally {
                    formula = formula.slice(matches[0].length);
                }
            }
            FormulaUtils.logTokens('\xa0 tokenize', tokenDescs);

            // post-process the token array
            Utils.iterateArray(tokenDescs, function (tokenDesc, index) {

                var // the current token
                    token = tokenDesc.token,
                    // the token preceding and following the current token
                    prevTokenDesc = tokenDescs[index - 1],
                    nextTokenDesc = tokenDescs[index + 1],
                    // the next non-whitespace token preceding the current token
                    prevNonWsTokenDesc = (prevTokenDesc && prevTokenDesc.token.isType('ws')) ? tokenDescs[index - 2] : prevTokenDesc;

                // replace combination of 'op[-] lit[number]' with a negative number, if the
                // minus is the leading token or preceded by an operator or opening parenthesis
                var nextValue = null;
                if (token.isType('op') && (token.getValue() === '-') &&
                    nextTokenDesc && nextTokenDesc.token.isType('lit') && _.isNumber(nextValue = nextTokenDesc.token.getValue()) && (nextValue > 0) &&
                    (!prevNonWsTokenDesc || prevNonWsTokenDesc.token.matchesType(/^(op|sep|open)$/))
                ) {
                    tokenDesc.token = new Tokens.LiteralToken(-nextValue);
                    tokenDesc.text += nextTokenDesc.text;
                    tokenDescs.splice(index + 1, 1);
                }

                // if the range intersection operator is a space character, insert an operator
                // token for a white-space token containing a SPACE character and surrounded
                // by reference/name tokens or other subexpressions in parentheses
                var spaceIndex = null;
                if ((config.OPERATORS.isect === ' ') && token.isType('ws') && ((spaceIndex = tokenDesc.text.indexOf(' ')) >= 0) &&
                    prevTokenDesc && prevTokenDesc.token.matchesType(/^(ref|name|close)$/) &&
                    nextTokenDesc && nextTokenDesc.token.matchesType(/^(ref|name|open|func)$/)
                ) {

                    // insert the remaining white-space following the first SPACE character
                    if (spaceIndex + 1 < tokenDesc.text.length) {
                        var trailingSpace = tokenDesc.text.slice(spaceIndex + 1);
                        tokenDescs.splice(index + 1, 0, new TokenDescriptor(new Tokens.FixedToken('ws', trailingSpace), trailingSpace));
                    }

                    // insert an intersection operator for the SPACE character after the current token
                    tokenDescs.splice(index + 1, 0, new TokenDescriptor(new Tokens.OperatorToken('isect'), ' '));

                    // shorten the current white-space token to the text preceding the first SPACE character, or remove it
                    if (spaceIndex > 0) {
                        token.setValue(tokenDesc.text.slice(0, spaceIndex));
                        tokenDesc.text = token.getValue();
                    } else {
                        tokenDescs.splice(index, 1);
                    }
                }

            }, { reverse: true, context: this });
            FormulaUtils.logTokens('\xa0 postprocess', tokenDescs);

            // add text positions to the token descriptors
            tokenDescs.forEach(function (tokenDesc, index) {
                tokenDesc.index = index;
                tokenDesc.start = (index === 0) ? 0 : tokenDescs[index - 1].end;
                tokenDesc.end = tokenDesc.start + tokenDesc.text.length;
            });

            return tokenDescs;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = config = null;
        });

    } // class Tokenizer

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: Tokenizer });

});
