/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/referencefuncs', [
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SheetUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with cell range
     * references used as values (cell address arithmetics), and functions that
     * look up specific values in a range of cells.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, cell range references can be given literally,
     * or can be the result of functions or operators. The client-side formula
     * engine provides the function signature type 'ref', and the class
     * Reference whose instances represent a single cell range address, or a
     * list of several cell range addresses (e.g. as result of the reference
     * list operator).
     *
     *************************************************************************/

    var // convenience shortcuts
        ErrorCodes = SheetUtils.ErrorCodes,
        Address = SheetUtils.Address;

    // exports ================================================================

    return {

        ADDRESS: {
            category: 'reference',
            minParams: 2,
            maxParams: 5,
            type: 'val',
            signature: 'val:int val:int val:int val:bool val:str',
            resolve: function (row, col, absMode, a1Style, sheetRef) {

                var // document model
                    docModel = this.getDocModel(),
                    // the formula tokenizer for the current UI language
                    tokenizer = this.getTokenizer(),
                    // the grammar configuration of the formula tokenizer
                    config = tokenizer.getConfig(),
                    // absolute columns or row index
                    absCol = false, absRow = false,
                    // the resulting address string
                    result = '';

                // returns the string representation of the passed column/row index for A1 notation
                function createA1Index(index, absRef, columns) {
                    var maxIndex = docModel.getMaxIndex(columns);
                    if ((index >= 1) && (index <= maxIndex + 1)) {
                        return (absRef ? '$' : '') + Address.stringifyIndex(index - 1, columns);
                    }
                    throw ErrorCodes.VALUE;
                }

                // returns the string representation of the passed column/row index for RC notation
                function createRCIndex(index, abs, columns) {
                    var maxIndex = docModel.getMaxIndex(columns);
                    if (abs) {
                        // absolute mode: always insert one-based column/row index (e.g. R1C4)
                        if ((index >= 1) && (index <= maxIndex + 1)) { return String(index); }
                    } else {
                        // relative mode: leave out zero values (e.g. RC[-1])
                        if (index === 0) { return ''; }
                        if ((-maxIndex <= index) && (index <= maxIndex)) { return '[' + index + ']'; }
                    }
                    throw ErrorCodes.VALUE;
                }

                // default for missing or empty absolute mode (always defaults to 1)
                if (this.isMissingOrEmptyOperand(2)) {
                    absMode = 1;
                } else if ((absMode < 1) || (absMode > 4)) {
                    throw ErrorCodes.VALUE;
                }
                absCol = (absMode % 2) > 0;
                absRow = absMode <= 2;

                // default for missing or empty A1 style (always defaults to TRUE)
                if (this.isMissingOrEmptyOperand(3)) {
                    a1Style = true;
                }

                if (a1Style) {
                    result = createA1Index(col, absCol, true) + createA1Index(row, absRow, false);
                } else {
                    result = config.R1C1_R + createRCIndex(row, absRow, false) + config.R1C1_C + createRCIndex(col, absCol, true);
                }

                // add encoded sheet name (enclosed in apostrophes if needed)
                // (not for missing or empty parameter, but for explicit empty string)
                if (!this.isMissingOrEmptyOperand(4)) {
                    result = ((sheetRef.length > 0) ? tokenizer.encodeExtendedSheetRef(sheetRef) : '') + '!' + result;
                }

                return result;
            }
        },

        AREAS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref',
            resolve: function (ranges) { return ranges.length; }
        },

        CHOOSE: {
            category: 'reference',
            minParams: 2,
            type: 'any',
            signature: 'val:int any',
            resolve: function (index) {
                return this.getOperand(index - 1);
            }
        },

        COLUMN: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref:single',
            resolve: function (range) {
                // use reference address, if parameter is missing
                var address = range ? range.start : this.getRefAddress();
                // function returns one-based column index
                return address[0] + 1;
            }
        },

        COLUMNS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function (operand) {
                switch (operand.getType()) {
                case 'val':
                    var value = operand.getValue();
                    return SheetUtils.isErrorCode(value) ? value : 1;
                case 'mat':
                    return operand.getMatrix().cols();
                case 'ref':
                    return operand.getReference().getSingleRange().cols();
                default:
                    throw 'fatal';
                }
            }
        },

        DDE: {
            category: 'reference',
            supported: 'odf',
            hidden: true,
            minParams: 3,
            maxParams: 4,
            type: 'val'
        },

        FORMULA: {
            category: 'reference',
            supported: 'odf', // OOXML name: FORMULATEXT
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        FORMULATEXT: {
            category: 'reference',
            supported: 'ooxml', // ODF name: FORMULA
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        GETPIVOTDATA: {
            category: 'reference',
            minParams: 2,
            repeatParams: 2,
            type: 'val'
        },

        HLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any'
        },

        HYPERLINK: {
            category: 'reference',
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        INDEX: {
            category: 'reference',
            minParams: 2,
            maxParams: 4,
            type: 'any'
        },

        INDIRECT: {
            category: 'reference',
            minParams: 1,
            maxParams: 2,
            type: 'ref'
        },

        LOOKUP: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'any'
        },

        MATCH: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'val'
        },

        OFFSET: {
            category: 'reference',
            minParams: 3,
            maxParams: 5,
            type: 'ref',
            signature: 'ref val:int val:int val:int val:int',
            resolve: function (ranges, rows, cols, height, width) {

                var // reference must contain a single range address
                    range = this.getSingleRange(ranges, { valueError: true });

                // apply offset
                range.start[0] += cols;
                range.start[1] += rows;
                range.end[0] += cols;
                range.end[1] += rows;

                // modify size (but not if 'height' or 'width' parameters exist but are empty)
                if (!this.isMissingOrEmptyOperand(4)) { range.end[0] = range.start[0] + width - 1; }
                if (!this.isMissingOrEmptyOperand(3)) { range.end[1] = range.start[1] + height - 1; }

                // check that the range does not left the sheet boundaries
                if (this.getDocModel().isValidRange(range)) { return range; }
                throw ErrorCodes.REF;
            }
        },

        ROW: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref:single',
            resolve: function (range) {
                // use reference address, if parameter is missing
                var address = range ? range.start : this.getRefAddress();
                // function returns one-based row index
                return address[1] + 1;
            }
        },

        ROWS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function (operand) {
                switch (operand.getType()) {
                case 'val':
                    var value = operand.getValue();
                    return SheetUtils.isErrorCode(value) ? value : 1;
                case 'mat':
                    return operand.getMatrix().rows();
                case 'ref':
                    return operand.getReference().getSingleRange().rows();
                default:
                    throw 'fatal';
                }
            }
        },

        SHEET: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'any', // string (sheet name) or reference
            resolve: function (operand) {
                // use reference sheet, if parameter is missing
                if (_.isUndefined(operand)) { return this.getRefSheet() + 1; }
                // use sheet index of a reference
                if (operand.isReference()) { return operand.getReference().getSingleRange().sheet1 + 1; }
                // convert values to strings (sheet name may look like a number or Boolean)
                if (operand.isValue()) {
                    var sheet = this.getDocModel().getSheetIndex(this.convertToString(operand.getValue()));
                    return (sheet < 0) ? ErrorCodes.NA : (sheet + 1);
                }
                // always fail for matrix operands, regardless of context
                return ErrorCodes.NA;
            }
        },

        SHEETS: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref:multi',
            resolve: function (range) {
                // return total number of sheets in document, if parameter is missing
                return range ? range.sheets() : this.getDocModel().getSheetCount();
            }
        },

        VLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any'
        }
    };

});
