/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/informationfuncs', [
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SheetUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions providing some kind of
     * information about the contents of sheet cells.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes,

        // shortcuts to mathematical functions
        floor = Math.floor,
        abs = Math.abs;

    // exports ================================================================

    return {

        CELL: {
            category: 'information',
            minParams: 1,
            maxParams: 2,
            type: 'val'
        },

        CURRENT: {
            category: 'information',
            supported: 'odf',
            hidden: true,
            minParams: 0,
            maxParams: 0,
            type: 'any'
        },

        ERRORTYPE: {
            category: 'information',
            supported: 'odf', // OOXML name: ERROR.TYPE
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        'ERROR.TYPE': {
            category: 'information',
            supported: 'ooxml', // ODF name: ERRORTYPE
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        INFO: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        ISBLANK: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isNull // error codes result in FALSE
        },

        ISERR: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: function (value) {
                return SheetUtils.isErrorCode(value) && !ErrorCodes.NA.equals(value);
            }
        },

        ISERROR: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: SheetUtils.isErrorCode
        },

        ISEVEN: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return (floor(abs(number)) % 2) === 0;
            }
        },

        ISFORMULA: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref:single',
            resolve: function (range) {

                // use the first cell in the range (TODO: matrix context)
                var address = range.start;

                // accept reference to own cell (no circular reference error)
                if (this.isRefSheet(range.sheet1) && this.isRefAddress(address)) {
                    return true; // do not check the cell, a new formula has not been inserted yet
                }

                // getCellFormula() returns null for value cells
                return _.isString(this.getCellFormula(range.sheet1, address));
            }
        },

        ISLOGICAL: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isBoolean // error codes and empty cells result in FALSE
        },

        ISNA: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.bind(ErrorCodes.NA.equals, ErrorCodes.NA)
        },

        ISNONTEXT: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: function (value) {
                return !_.isString(value); // error codes and empty cells result in TRUE
            }
        },

        ISNUMBER: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isNumber // error codes and empty cells result in FALSE
        },

        ISODD: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (number) {
                return (floor(abs(number)) % 2) !== 0;
            }
        },

        ISREF: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any',
            resolve: function (operator) { return operator.isReference(); } // error codes result in FALSE
        },

        ISTEXT: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any',
            resolve: _.isString // error codes and empty cells result in FALSE
        },

        NA: {
            category: 'information',
            minParams: 0,
            maxParams: 0,
            type: 'val',
            resolve: _.constant(ErrorCodes.NA)
        },

        STYLE: {
            category: 'information',
            supported: 'odf',
            hidden: true,
            minParams: 1,
            maxParams: 3,
            type: 'val'
        },

        TYPE: {
            category: 'information',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:any', // resolve cell references to values, accept error codes
            resolve: function (value) {
                // array literals always result in 64 regardless of their contents
                if (this.getOperand(0).isMatrix()) { return 64; }
                // determine type of value parameters and cell contents
                if (_.isNumber(value)) { return 1; }
                if (_.isString(value)) { return 2; }
                if (_.isBoolean(value)) { return 4; }
                if (SheetUtils.isErrorCode(value)) { return 16; }
                // default (reference to empty cell, or cell not fetched from server)
                return 1;
            }
        }
    };

});
