/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/colrowcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/cellattributesmodel'
], function (Utils, TimerMixin, ModelObject, SheetUtils, CellAttributesModel) {

    'use strict';

    var // convenience shortcuts
        Interval = SheetUtils.Interval,
        IntervalArray = SheetUtils.IntervalArray;

    // private static functions ===============================================

    /**
     * Converts the passed zoom-independent length in 1/100 of millimeters to a
     * length in pixels according to the specified sheet zoom factor.
     *
     * @param {Number} length
     *  An arbitrary length, in 1/100 of millimeters.
     *
     * @param {Number} zoom
     *  The current sheet zoom factor.
     *
     * @returns {Number}
     *  The passed length, converted to pixels.
     */
    function convertHmmToPixels(length, zoom) {
        return Utils.convertHmmToLength(length * zoom, 'px', 1);
    }

    /**
     * Converts the passed length in pixels according to the current sheet
     * zoom factor to a zoom-independent length in 1/100 of millimeters.
     *
     * @param {Number} length
     *  An arbitrary length, in pixels.
     *
     * @param {Number} zoom
     *  The current sheet zoom factor.
     *
     * @returns {Number}
     *  The passed length, converted to 1/100 of millimeters.
     */
    function convertPixelsToHmm(length, zoom) {
        return Utils.convertLengthToHmm(length / zoom, 'px');
    }

    // class ColRowModel ======================================================

    /**
     * Represents the entries of a ColRowCollection instance: an interval of
     * columns or rows with equal size and default formatting for all cells.
     *
     * @constructor
     *
     * @extends CellAttributesModel
     *
     * @property {Interval} interval
     *  The column/row interval represented by this instance.
     *
     * @property {Boolean} columns
     *  The orientation of this instance: true for columns, false for rows.
     *
     * @property {Number} offsetHmm
     *  Absolute position of the first column/row represented by this instance,
     *  in 1/100 of millimeters.
     *
     * @property {Number} sizeHmm
     *  Size of a single column/row represented by this instance, in 1/100 of
     *  millimeters.
     *
     * @property {Number} offset
     *  Absolute position of the first column/row represented by this instance,
     *  in pixels.
     *
     * @property {Number} size
     *  Size of a single column/row represented by this instance, in pixels.
     */
    var ColRowModel = CellAttributesModel.extend({ constructor: function (sheetModel, interval, columns, initAttributes, initOptions) {

        // helper properties: names of attributes and attribute family
        this.FAMILY_NAME = columns ? 'column' : 'row';
        this.SIZE_ATTR_NAME = columns ? 'width' : 'height';
        this.CUSTOM_SIZE_ATTR_NAME = columns ? 'customWidth' : 'customHeight';

        // base constructor
        CellAttributesModel.call(this, sheetModel, initAttributes, _.extend({
            families: this.FAMILY_NAME,
            parentModel: sheetModel,
            listenToParent: false
        }, initOptions));

        // column/row interval covered by this model
        this.interval = interval.clone();
        this.columns = columns;

        // effective position/size of the columns/rows, in 1/100 mm
        this.offsetHmm = 0;
        this.sizeHmm = 0;

        // effective position/size of the columns/rows, in pixels
        this.offset = 0;
        this.size = 0;

    }}); // class ColRowModel

    // protected methods ------------------------------------------------------

    /**
     * Creates and returns a cloned instance of this entry model for the
     * specified sheet.
     *
     * @internal
     *  Used by the class ColRowCollection during clone construction. DO NOT
     *  CALL from external code!
     *
     * @returns {ColRowModel}
     *  A clone of this entry model, initialized for ownership by the passed
     *  sheet model.
     */
    ColRowModel.prototype.clone = function (sheetModel) {
        var newModel = new ColRowModel(sheetModel, this.interval, this.columns);
        newModel.cloneFrom(this); // clone auto style and explicit attributes
        newModel.offsetHmm = this.offsetHmm;
        newModel.sizeHmm = this.sizeHmm;
        newModel.offset = this.offset;
        newModel.size = this.size;
        return newModel;
    };

    // public methods ---------------------------------------------------------

    /**
     * Updates the effective pixel size of a single column/row, according to
     * the current formatting attributes of this entry model.
     */
    ColRowModel.prototype.updateSize = function (zoom) {

        var // the merged attributes of the column/row family
            attributes = this.getMergedAttributes()[this.FAMILY_NAME],
            // whether this model is visible (filtered rows are hidden too)
            visible = attributes.visible && (this.columns || !attributes.filtered);

        this.sizeHmm = visible ? Math.max(0, Math.round(attributes[this.SIZE_ATTR_NAME])) : 0;
        this.size = (this.sizeHmm === 0) ? 0 : Math.max(SheetUtils.MIN_CELL_SIZE, convertHmmToPixels(this.sizeHmm, zoom));
    };

    /**
     * Returns the absolute offset of the passed index (interpreted relative to
     * the start index of this entry model) in 1/100 mm, independent from the
     * current sheet zoom factor.
     */
    ColRowModel.prototype.getOffsetHmm = function (index) {
        return this.offsetHmm + this.sizeHmm * index;
    };

    /**
     * Returns the end offset of this entry model in 1/100 mm, independent from
     * the current sheet zoom factor.
     */
    ColRowModel.prototype.getEndOffsetHmm = function () {
        return this.getOffsetHmm(this.interval.size());
    };

    /**
     * Returns the absolute offset of the passed index (interpreted relative to
     * the start index of this entry model) in pixels, according to the current
     * sheet zoom factor.
     */
    ColRowModel.prototype.getOffset = function (index) {
        return this.offset + this.size * index;
    };

    /**
     * Returns the end offset of this entry model in pixels, according to the
     * current sheet zoom factor.
     */
    ColRowModel.prototype.getEndOffset = function () {
        return this.getOffset(this.interval.size());
    };

    /**
     * Returns the effective column/row attributes of this entry model.
     *
     * @returns {Object}
     *  The attributes values of the 'column' attribute family, if this model
     *  represents columns, otherwise the attributes values of the 'row'
     *  attribute family.
     */
    ColRowModel.prototype.getColRowAttributes = function () {
        return this.getMergedAttributes()[this.FAMILY_NAME];
    };

    /**
     * Returns whether the custom size flag is set in the attributes of this
     * entry model. For columns, the attribute 'column:customWidth' will be
     * returned; and for rows the attribute 'row:customHeight'.
     *
     * @returns {Boolean}
     *  Whether the custom size flag is set in the formatting attributes.
     */
    ColRowModel.prototype.hasCustomSize = function () {
        return this.getColRowAttributes()[this.CUSTOM_SIZE_ATTR_NAME];
    };

    // class ColRowDescriptor =================================================

    /**
     * A simple descriptor object for a column or row, returned by the public
     * methods of the class ColRowCollection.
     *
     * @constructor
     *
     * @property {Number} index
     *  The zero-based index of the column or row.
     *
     * @property {Number} offsetHmm
     *  The absolute position of the column/row, in 1/100 mm.
     *
     * @property {Number} sizeHmm
     *  The effective size (zero for hidden columns/rows), in 1/100 mm.
     *
     * @property {Number} offset
     *  The absolute position of the column/row, in pixels.
     *
     * @property {Number} size
     *  The effective size (zero for hidden columns/rows), in pixels.
     *
     * @property {String|Null} style
     *  The identifier of the auto style containing the character and cell
     *  formatting attributes for all undefined cells in the column/row.
     *
     * @property {Object} explicit
     *  The explicit formatting attribute set, including the cell formatting
     *  attributes for all undefined cells in the column/row.
     *
     * @property {Object} attributes
     *  The merged attribute set, including the cell formatting attributes for
     *  all undefined cells in the column/row.
     */
    function ColRowDescriptor(entryModel, index) {

        // column/row index
        this.index = index;

        // position and size
        var relIndex = index - entryModel.interval.first;
        this.offsetHmm = entryModel.getOffsetHmm(relIndex);
        this.sizeHmm = entryModel.sizeHmm;
        this.offset = entryModel.getOffset(relIndex);
        this.size = entryModel.size;

        // formatting attributes
        this.style = entryModel.getAutoStyleId();
        this.explicit = entryModel.getExplicitAttributes();
        this.attributes = entryModel.getMergedAttributes();

    } // class ColRowDescriptor

    // class ColRowCollection =================================================

    /**
     * Collects information about all columns or all rows of a single sheet in
     * a spreadsheet document.
     *
     * Triggers the following events:
     * - 'change:entries'
     *      After the formatting attributes of a column/row interval have been
     *      changed. The event handler receives the index interval of the
     *      changed entries, and the explicit attributes set at the interval
     *      (not the merged attribute sets which may be different in the
     *      various entries of the interval). The attributes parameter may be
     *      missing.
     * - 'insert:entries'
     *      After new columns/rows have been inserted into the sheet. The event
     *      handler receives the index interval of the inserted entries.
     * - 'delete:entries'
     *      After columns/rows have been deleted from the sheet. The event
     *      handler receives the index interval of the deleted entries.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     *
     * @param {Boolean} columns
     *  Whether this collection contains column headers (true) or row headers
     *  (false).
     */
    function ColRowCollection(sheetModel, columns) {

        var // self reference
            self = this,

            // the document model
            docModel = sheetModel.getDocModel(),

            // sorted sparse array of index ranges (instances of ColRowModel)
            entryModels = [],

            // the largest valid column/row index
            maxIndex = docModel.getMaxIndex(columns),

            // the current zoom factor of the document
            zoom = sheetModel.getEffectiveZoom(),

            // default model for entries missing in the collection
            defaultModel = new ColRowModel(sheetModel, new Interval(0, maxIndex), columns, null, { listenToParent: true });

        // base constructor ---------------------------------------------------

        // bug 38436: events must be triggered during import, in order to update position of drawing objects
        ModelObject.call(this, docModel, { trigger: 'always' });
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Dumps the entire collection to the debug console.
         */
        var debugDump = SheetUtils.isLoggingActive() ? (function () {

            function dumpMsg(msg) {
                SheetUtils.info(sheetModel.getName() + '.' + (columns ? 'Col' : 'Row') + 'Collection: ' + msg);
            }

            var dumpContents = docModel.getApp().createDebouncedActionsMethodFor(self, $.noop, function () {
                dumpMsg('new contents');
                entryModels.forEach(function (entryModel) {
                    SheetUtils.log('\xa0 interval=' + entryModel.interval.stringifyAs(columns) + ' offset=' + entryModel.offset + ' size=' + entryModel.size + ' style=' + entryModel.getAutoStyleId() + ' attrs=' + Utils.stringifyForDebug(entryModel.getExplicitAttributes(true)));
                });
            });

            return self.createDebouncedMethod(dumpMsg, dumpContents);
        }()) : $.noop;

        /**
         * Searches for an entry model that contains or follows the passed
         * column/row index.
         *
         * @param {Number} index
         *  The zero-based column/row index. Must be located in the valid range
         *  covered by this collection.
         *
         * @returns {Number}
         *  The array index of the entry model containing the passed index. If
         *  the index is located in a gap between two entries, returns the
         *  array index of the entry starting after the passed index. This may
         *  also be an array index pointing after the last existing entry.
         */
        function findModelIndex(index) {
            var index = Utils.findFirstIndex(entryModels, function (entryModel) {
                return index <= entryModel.interval.last;
            }, { sorted: true });
            return (index < 0) ? entryModels.length : index;
        }

        /**
         * Updates the dirty entry model offsets and sizes up to the end of the
         * collection, after changing the collection, or the sheet attributes.
         *
         * @param {Number} startIndex
         *  The zero-based column/row index where updating the offsets starts.
         *
         * @param {Boolean} [updateSize=false]
         *  If set to true, the size of the entries will be updated too. By
         *  default, only the offset positions of the entries will be updated.
         */
        function updateModelGeometry(startIndex, updateSize) {

            var // the array index of the first dirty entry model
                arrayIndex = findModelIndex(startIndex),
                // the last valid entry model
                prevModel = entryModels[arrayIndex - 1],
                // the current entry model to be updated
                currModel = null;

            // update start offsets of all dirty collection entries
            for (; arrayIndex < entryModels.length; arrayIndex += 1) {

                // set offset of current entry to the end offset of the previous entry
                currModel = entryModels[arrayIndex];
                currModel.offsetHmm = prevModel ? prevModel.getEndOffsetHmm() : 0;
                currModel.offset = prevModel ? prevModel.getEndOffset() : 0;

                // add size of the gap between previous and current entry
                var currFirst = currModel.interval.first,
                    prevLast = prevModel ? (prevModel.interval.last + 1) : 0;
                if ((prevLast === 0) || (prevLast < currFirst)) {
                    currModel.offsetHmm += defaultModel.sizeHmm * (currFirst - prevLast);
                    currModel.offset += defaultModel.size * (currFirst - prevLast);
                }

                // update size of the current entry model (first update the merged
                // attributes, parent sheet attributes may have been changed)
                if (updateSize) { currModel.refreshMergedAttributeSet().updateSize(zoom); }

                prevModel = currModel;
            }
        }

        /**
         * Updates the default column/row size according to the current
         * formatting attributes of the sheet model.
         */
        function updateDefaultSize() {

            var // the old default size, in 1/100 mm, and in pixels
                oldDefSizeHmm = defaultModel.sizeHmm,
                oldDefSize = defaultModel.size;

            // calculate the new default size
            defaultModel.updateSize(zoom);

            // do nothing, if the effective size has not changed (check both sizes to catch any rounding errors)
            if ((defaultModel.sizeHmm !== oldDefSizeHmm) || (defaultModel.size !== oldDefSize)) {
                updateModelGeometry(0, true);
                self.trigger('change:entries', self.getFullInterval());
                debugDump('default size changed');
            }
        }

        /**
         * Helper function to get the descriptor of a specific column/row,
         * without needing to search for the affected entry model.
         *
         * @param {Number} arrayIndex
         *  The array index of the entry model that contains the passed index,
         *  or that adjoins the gap containing the passed index.
         *
         * @param {Number} index
         *  The index of the column/row whose descriptor will be returned. Must
         *  be contained in the entry model with the passed index, or must be
         *  contained in the gap between that entry model and its predecessor
         *  or successor.
         *
         * @returns {ColRowDescriptor}
         *  A descriptor object for the column/row.
         */
        function makeDescriptor(arrayIndex, index) {

            // array is empty, or entry points into gap before first entry: use default size
            if ((entryModels.length === 0) || (arrayIndex < 0) || ((arrayIndex === 0) && (index < entryModels[0].interval.first))) {
                return new ColRowDescriptor(defaultModel, index);
            }

            // index points into a gap between two entries, or to the gap after the last entry
            if ((arrayIndex === entryModels.length) || (index < entryModels[arrayIndex].interval.first)) {
                var entryModel = entryModels[arrayIndex - 1],
                    entryDesc = new ColRowDescriptor(defaultModel, index);
                // adjust entry offsets (default entry is always relative to index 0 instead of start index of current gap)
                entryDesc.offsetHmm = entryModel.getEndOffsetHmm() + defaultModel.sizeHmm * (index - entryModel.interval.last - 1);
                entryDesc.offset = entryModel.getEndOffset() + defaultModel.size * (index - entryModel.interval.last - 1);
                return entryDesc;
            }

            // index points into a collection entry
            return new ColRowDescriptor(entryModels[arrayIndex], index);
        }

        /**
         * Inserts the passed entry model into the internal array.
         *
         * @param {Number} arrayIndex
         *  The array index for the new entry model.
         *
         * @param {ColRowModel} entryModel
         *  The new entry model to be inserted.
         */
        function insertModel(arrayIndex, entryModel) {
            entryModels.splice(arrayIndex, 0, entryModel);
        }

        /**
         * Deletes the specified entry models from the array.
         *
         * @param {Number} arrayIndex
         *  The array index of the first entry model to be deleted.
         *
         * @param {Number} [count]
         *  The number of entry models to be deleted. If omitted, all entry
         *  models up to the end of the array will be deleted.
         */
        function deleteModels(arrayIndex, count) {
            // call destructor for all deleted entries
            _.invoke(entryModels.splice(arrayIndex, count), 'destroy');
        }

        /**
         * Splits the entry model at the passed array index, and inserts the
         * new entry model after the existing entry model.
         *
         * @param {Number} arrayIndex
         *  The array index of the entry model to be split.
         *
         * @param {Number} leadingCount
         *  The new interval size of the existing entry model. Must be less
         *  than the current interval size of the entry model.
         *
         * @returns {ColRowModel}
         *  The new entry model inserted after the existing entry model.
         */
        function splitModel(arrayIndex, leadingCount) {

            var // the existing collection entry
                oldModel = entryModels[arrayIndex],
                // the clone of the existing entry
                newModel = oldModel.clone(sheetModel);

            // adjust start and end position of the entries
            oldModel.interval.last = oldModel.interval.first + leadingCount - 1;
            newModel.interval.first = oldModel.interval.first + leadingCount;
            newModel.offsetHmm = oldModel.offsetHmm + leadingCount * oldModel.sizeHmm;
            newModel.offset = oldModel.offset + leadingCount * oldModel.size;

            // insert the new entry model into the collection
            insertModel(arrayIndex + 1, newModel);
            return newModel;
        }

        /**
         * Tries to merge the entry model at the passed array index with its
         * predecessor and/or successor, if these entries are equal. If
         * successful, the preceding and/or following entry models will be
         * removed from this collection.
         *
         * @param {Number} arrayIndex
         *  The array index of the entry model to be merged with its
         *  predecessor and /or successor. May be an invalid index (no entry
         *  exists in the collection), or may refer to the first or last entry
         *  model (no predecessor or successor exists in the collection).
         *
         * @param {Object} [options]
         *  Optional parameters specifying which entry models will be merged
         *  with the specified entry model:
         *  @param {Boolean} [options.prev=false]
         *      If set to true, the entry and its predecessor will be tried to
         *      merge.
         *  @param {Boolean} [options.next=false]
         *      If set to true, the entry and its successor will be tried to
         *      merge.
         *
         * @returns {Object}
         *  An object containing the Boolean flags 'prev' and 'next' specifying
         *  whether the entry model has been merged with its predecessor and/or
         *  successor respectively.
         */
        function mergeModel(arrayIndex, options) {

            var // the result object
                result = {};

            // tries to merge the entry with its predecessor, removes the preceding entry
            function tryMerge(currIndex) {

                var // the preceding collection entry
                    prevModel = entryModels[currIndex - 1],
                    // the current collection entry
                    thisModel = entryModels[currIndex];

                // check that the entries are equal and do not have a gap
                if (prevModel && thisModel && (prevModel.interval.last + 1 === thisModel.interval.first) && prevModel.hasEqualAttributes(thisModel)) {
                    thisModel.interval.first = prevModel.interval.first;
                    thisModel.offsetHmm = prevModel.offsetHmm;
                    thisModel.offset = prevModel.offset;
                    deleteModels(currIndex - 1, 1);
                    return true;
                }
                return false;
            }

            // try to merge the entry with its successor/predecessor
            result.next = Utils.getBooleanOption(options, 'next', false) && tryMerge(arrayIndex + 1);
            result.prev = Utils.getBooleanOption(options, 'prev', false) && tryMerge(arrayIndex);
            return result;
        }

        /**
         * Invokes the passed callback function for all entry models that cover
         * the specified column/row interval.
         *
         * @param {Interval} interval
         *  The column/row interval to be iterated.
         *
         * @param {Function} callback
         *  The callback function called for every entry model covered by the
         *  specified interval. Receives the following parameters:
         *  (1) {ColRowModel} entryModel
         *      The current entry model.
         *  (2) {Number} first
         *      The zero-based index of the first column/row. May be located in
         *      the middle of the visited collection entry, if it is the first
         *      entry of the interval passed to this method.
         *  (3) {Number} last
         *      The zero-based index of the last column/row. May be located in
         *      the middle of the visited collection entry, if it is the last
         *      entry of the interval passed to this method.
         *  If the callback function returns the Utils.BREAK object, the
         *  iteration process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, visits the covered collection entries from last
         *      to first. Must not be used together with the option 'modify'.
         *  @param {Boolean} [options.modify=false]
         *      If set to true, the callback function wants to modify the entry
         *      models. Entries not yet existing (gaps between existing entry
         *      models) will be created and inserted into the collection
         *      before. Entries not covered completely by the first or last
         *      index of the passed interval will be split before. After
         *      iteration is finished, adjacent entry models with equal
         *      formatting attributes will be merged to one entry model. Must
         *      not be used together with the option 'reverse'.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the callback has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateModels(interval, callback, options) {

            var // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether the callback modifies the collection entries
                modify = Utils.getBooleanOption(options, 'modify', false),
                // current column/row index for the next invocation
                index = reverse ? interval.last : interval.first,
                // array index of the next collection entry to be visited
                arrayIndex = findModelIndex(index),
                // the current entry model in the loop
                entryModel = null,
                // whether to break the iteration
                breakLoop = false;

            // invokes the callback function
            function invokeCallback(currModel, otherIndex) {
                breakLoop = breakLoop || (callback.call(self, currModel, reverse ? otherIndex : index, reverse ? index : otherIndex) === Utils.BREAK);
                index = reverse ? (otherIndex - 1) : (otherIndex + 1);
            }

            // invokes the callback function for the gap before a collection entry
            function invokeCallbackForGap(other) {

                var // the entry passed to the callback function
                    entryModel = defaultModel.clone(sheetModel),
                    // helper descriptor to calculate offsets
                    entryDesc = null;

                if (!modify) {
                    // read-only iteration: create a dummy entry that will not be inserted into the collection
                    delete entryModel.setAttributes;
                }

                // initialize interval and start offset of the new entry
                entryModel.interval = Interval.create(index, other);
                entryDesc = makeDescriptor(arrayIndex, entryModel.interval.first);
                entryModel.offsetHmm = entryDesc.offsetHmm;
                entryModel.offset = entryDesc.offset;

                // insert new entry into array in modifying mode
                if (modify) {
                    insertModel(arrayIndex, entryModel);
                    arrayIndex += 1;
                }

                // invoke the callback function
                invokeCallback(entryModel, other);

                // remove entry from the collection, if it does not contain any explicit attributes
                if (modify && !entryModel.isFormatted()) {
                    arrayIndex -= 1;
                    deleteModels(arrayIndex, 1);
                    return;
                }

                // try to merge the new entry with its predecessor
                // (arrayIndex points behind the new entry, after merging, it has to be decreased)
                if (modify && mergeModel(arrayIndex - 1, { prev: true }).prev) {
                    arrayIndex -= 1;
                }

                // in case of read-only, destroy clone
                if (!modify) {
                    entryModel.destroy();
                }
            }

            // invokes the callback function for an existing entry model
            function invokeCallbackForModel(other) {

                // invoke the callback function
                invokeCallback(entryModels[arrayIndex], other);

                // Remove entry model from the collection, if it does not contain any explicit attributes
                // anymore. On success, do not change the array index (it already points to the next
                // entry model after deleting the current entry model).
                if (modify && !entryModels[arrayIndex].isFormatted()) {
                    deleteModels(arrayIndex, 1);
                    return;
                }

                // Try to merge the entry model with its predecessor. On success, do not change the
                // array index (it already points to the next entry model after merging the entries).
                if (modify && mergeModel(arrayIndex, { prev: true }).prev) {
                    return;
                }

                // go to next (or previous) array element
                arrayIndex += reverse ? -1 : 1;
            }

            // modify reversely not implemented
            if (reverse && modify) {
                Utils.error('ColRowCollection.iterateModels(): modify collection reversely not implemented');
                return Utils.BREAK;
            }

            // split the first entry model not covered completely by the interval
            if (modify && (arrayIndex < entryModels.length) && (entryModels[arrayIndex].interval.first < index)) {
                splitModel(arrayIndex, index - entryModels[arrayIndex].interval.first);
                arrayIndex += 1;
            }

            // reverse mode: back to previous entry model, if interval end points to gap before current entry
            if (reverse && ((arrayIndex === entryModels.length) || (interval.last < entryModels[arrayIndex].interval.first))) {
                arrayIndex -= 1;
            }

            // process all existing collection entries and the gaps covered by the interval
            while (!breakLoop && (arrayIndex >= 0) && (arrayIndex < entryModels.length) && (reverse ? (interval.first <= entryModels[arrayIndex].interval.last) : (entryModels[arrayIndex].interval.first <= interval.last))) {

                // the collection entry to be visited
                entryModel = entryModels[arrayIndex];

                // visit the gap between current index and start of the next entry model
                // (reverse mode: gap between end of the previous entry model and current index)
                if (reverse ? (entryModel.interval.last < index) : (index < entryModel.interval.first)) {
                    invokeCallbackForGap(reverse ? (entryModel.interval.last + 1) : (entryModel.interval.first - 1));
                }

                // split the last entry model not covered completely by the interval
                if (modify && (interval.last < entryModel.interval.last)) {
                    splitModel(arrayIndex, interval.last - index + 1);
                }

                // visit the entry model
                invokeCallbackForModel(reverse ? Math.max(interval.first, entryModel.interval.first) : Math.min(interval.last, entryModel.interval.last));
            }

            // try to merge last visited entry model with its successor
            // (arrayIndex already points to the successor of the last visited entry)
            if (modify) {
                mergeModel(arrayIndex, { prev: true });
            }

            // visit the gap after the last existing entry model (reverse mode: before the first existing entry model)
            if (!breakLoop && (reverse ? (interval.first <= index) : (index <= interval.last))) {
                invokeCallbackForGap(reverse ? interval.first : interval.last);
            }

            return breakLoop ? Utils.BREAK : undefined;
        }

        /**
         * Updates the effective offsets and sizes of all columns/rows in this
         * collection, after the zoom factor of the sheet has been changed.
         */
        function changeViewAttributesHandler(event, attributes) {
            if ('zoom' in attributes) {
                zoom = sheetModel.getEffectiveZoom();
                defaultModel.updateSize(zoom);
                updateModelGeometry(0, true);
                debugDump('zoom changed');
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this.getCloneData = function () {
            return { entryModels: entryModels, zoom: zoom };
        };

        /**
         * Clones all contents from the passed collection into this collection.
         *
         * @internal
         *  Used by the class SheetModel during clone construction. DO NOT CALL
         *  from external code!
         *
         * @param {ColRowCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.cloneFrom = function (collection) {

            var // the internal contents of the source collection
                cloneData = collection.getCloneData();

            entryModels = _.invoke(cloneData.entryModels, 'clone', sheetModel);
            zoom = cloneData.zoom;

            return this;
        };

        /**
         * Callback handler for the document operations 'insertColumns' and
         * 'insertRows'. Inserts new columns/rows into this collection, and
         * triggers an 'insert:entries' event containing the inserted
         * column/row interval.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the document operation.
         */
        this.applyInsertOperation = function (context) {

            var // the column/row interval to be inserted
                interval = context.getInterval(columns),
                // the array index of the first entry model to be moved
                arrayIndex = findModelIndex(interval.first),
                // the current entry model
                entryModel = entryModels[arrayIndex],
                // the previous entry model used to copy formatting attributes
                prevModel = null,
                // the number of columns/rows to be inserted
                delta = interval.size();

            // split entry model if it is not covered completely
            if (entryModel && (entryModel.interval.first < interval.first)) {
                splitModel(arrayIndex, interval.first - entryModel.interval.first);
                prevModel = entryModel;
                arrayIndex += 1;
            } else if ((arrayIndex > 0) && (entryModels[arrayIndex - 1].interval.last + 1 === interval.first)) {
                // use formatting of previous entry, if it ends exactly before the interval
                prevModel = entryModels[arrayIndex - 1];
            }

            // move existing entries
            for (var index = arrayIndex; index < entryModels.length; index += 1) {

                // update index interval of the entry
                entryModel = entryModels[index];
                entryModel.interval.first += delta;
                entryModel.interval.last += delta;

                // delete following entries moved outside the collection limits
                if (entryModel.interval.first > maxIndex) {
                    deleteModels(arrayIndex);
                } else if (entryModel.interval.last >= maxIndex) {
                    deleteModels(arrayIndex + 1);
                    entryModel.interval.last = maxIndex;
                }
            }

            // do not create a new entry model, if the preceding column/row has default format
            if (prevModel) {

                // the explicit attributes to be used for the new entries
                var prevAttributeSet = prevModel.getExplicitAttributes();

                // prepare the explicit attributes (always visible, never filtered)
                var prevAttributes = prevAttributeSet[prevModel.FAMILY_NAME];
                if (prevAttributes) {
                    if (prevAttributes.visible === false) {
                        delete prevAttributes.visible;
                    }
                    if (!columns && (prevAttributes.filtered === true)) {
                        delete prevAttributes.filtered;
                    }
                }

                // insert a new entry model, try to merge with adjacent entries
                var newModel = new ColRowModel(sheetModel, interval, columns, prevAttributeSet);
                newModel.updateSize(zoom);
                insertModel(arrayIndex, newModel);
                mergeModel(arrayIndex, { prev: true, next: true });
            }

            // update pixel offsets of the entries
            updateModelGeometry(interval.first);

            // notify insert listeners
            this.trigger('insert:entries', interval);
            debugDump('interval ' + interval.stringifyAs(columns) + ' inserted');
        };

        /**
         * Callback handler for the document operations 'deleteColumns' and
         * 'deleteRows'. Removes existing columns/rows from this collection,
         * and triggers a 'delete:entries' event containing the deleted
         * column/row interval.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the document operation.
         */
        this.applyDeleteOperation = function (context) {

            var // the column/row interval to be deleted
                interval = context.getInterval(columns),
                // the array index of the first entry model to be deleted
                arrayIndex = findModelIndex(interval.first),
                // the current entry model
                entryModel = entryModels[arrayIndex],
                // the number of elements to be deleted from the array
                deleteCount = 0,
                // the number of columns/rows to be removed
                delta = interval.size();

            // split the part of the entry located before the interval
            if (entryModel && (entryModel.interval.first < interval.first)) {
                entryModel = splitModel(arrayIndex, interval.first - entryModel.interval.first);
                arrayIndex += 1;
            }

            // remove and move the remaining entries
            for (var index = arrayIndex; index < entryModels.length; index += 1) {

                // update number of entries to be deleted, update index interval of moved entries
                entryModel = entryModels[index];
                if (entryModel.interval.last <= interval.last) {
                    deleteCount += 1;
                } else {
                    entryModel.interval.first = Math.max(interval.first, entryModel.interval.first - delta);
                    entryModel.interval.last -= delta;
                }
            }

            // delete the array elements
            if (deleteCount > 0) {
                deleteModels(arrayIndex, deleteCount);
            }

            // try to merge with previous entry
            mergeModel(arrayIndex, { prev: true });

            // update pixel offsets of the entries
            updateModelGeometry(interval.first);

            // notify delete listeners
            this.trigger('delete:entries', interval);
            debugDump('interval ' + interval.stringifyAs(columns) + ' deleted');
        };

        /**
         * Handler for the document operations 'setColumnAttributes' and
         * 'setRowAttributes'. Changes the formatting attributes of all entries
         * covered by the operation, and triggers a 'change:entries' event
         * containing the changed column/row interval.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the document operation.
         */
        this.applyChangeOperation = function (context) {
            this.setAttributes(context.getInterval(columns), context.getObj('attrs'), {
                rangeBorders: context.getOptBool('rangeBorders'),
                visibleBorders: context.getOptBool('visibleBorders')
            });
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the maximum valid column/row index available for an entry in
         * this collection.
         *
         * @returns {Number}
         *  The maximum column/row index.
         */
        this.getMaxIndex = function () {
            return maxIndex;
        };

        /**
         * Returns a new column/row interval covering the entire collection.
         *
         * @returns {Interval}
         *  A new column/row interval covering the entire collection.
         */
        this.getFullInterval = function () {
            return new Interval(0, maxIndex);
        };

        /**
         * Returns the total size of all columns/rows represented by this
         * collection, in 1/100 of millimeters, independent from the current
         * sheet zoom factor.
         *
         * @returns {Number}
         *  The total size of all columns/rows in the sheet, in 1/100 of
         *  millimeters.
         */
        this.getTotalSizeHmm = function () {
            // calculate the offset behind the last valid column/row
            return makeDescriptor(entryModels.length, maxIndex + 1).offsetHmm;
        };

        /**
         * Returns the total size of all columns/rows represented by this
         * collection, in pixels according to the current sheet zoom factor.
         *
         * @returns {Number}
         *  The total size of all columns/rows in the sheet, in pixels.
         */
        this.getTotalSize = function () {
            // calculate the offset behind the last valid column/row
            return makeDescriptor(entryModels.length, maxIndex + 1).offset;
        };

        /**
         * Returns a descriptor object for the column/row with the specified
         * index.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {ColRowDescriptor}
         *  A descriptor object for the specified column/row.
         */
        this.getEntry = function (index) {
            return makeDescriptor(findModelIndex(index), index);
        };

        /**
         * Invokes the passed callback function for all visible columns/rows
         * covered by the passed index intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Function} callback
         *  The callback function called for every column/row covered by the
         *  specified interval. Receives the following parameters:
         *  (1) {ColRowDescriptor} entryDesc
         *      A descriptor object for the current column/row.
         *  (2) {Interval} uniqueInterval
         *      The column/row interval containing all columns/rows with the
         *      same formatting attributes as the column/row currently visited.
         *      Useful in 'unique' mode (see option 'unique' below) where only
         *      the first entry (or last entry in reverse mode) of such an
         *      interval will be visited. Will be shrunken to the intervals
         *      passed to this method, regardless how large the entire equally
         *      formatted interval is.
         *  (3) {Interval} origInterval
         *      The original interval from the interval array passed to this
         *      method that contains the column/row currently visited.
         *  (4) {Number} intervalIndex
         *      The array index of the interval contained in the 'origInterval'
         *      parameter.
         *  If the callback function returns the Utils.BREAK object, the
         *  iteration process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the callback function will be called with this
         *      context (the symbol 'this' will be bound to the context inside
         *      the callback function).
         *  @param {String} [options.hidden='none']
         *      Specifies how to handle hidden columns/rows (with a size of
         *      zero). Must be one of the following values:
         *      - 'none' (or omitted): Only visible columns/rows will be
         *          visited.
         *      - 'all': All (visible and hidden) columns/rows will be visited.
         *      - 'last': All visible columns/rows, and hidden columns/rows
         *          that precede a visible column/row will be visited.
         *  @param {Boolean|Null} [options.customFormat=null]
         *      Only valid for row collections. If set to true or false, visits
         *      only the rows which have the row attribute 'customFormat' set
         *      to the specified value.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, visits the passed intervals, and the single
         *      column/row entries in each interval from last to first.
         *  @param {Boolean} [options.unique=false]
         *      If set to true, the callback function will be invoked once only
         *      for an entire column/row interval with the same formatting
         *      attributes. In forward mode, the first available column/row of
         *      such an interval will be visited; in reverse mode, the last
         *      available column/row will be visited.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the callback has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateEntries = function (intervals, callback, options) {

            var // the calling context for the callback function
                context = Utils.getOption(options, 'context'),
                // whether to visit hidden entries
                hiddenMode = Utils.getStringOption(options, 'hidden', 'none'),
                // whether to visit columns/rows with a specific custom size attribute only
                customSize = Utils.getBooleanOption(options, 'customSize', null),
                // whether to visit rows with a specific customFormat attribute only
                customFormat = columns ? null : Utils.getBooleanOption(options, 'customFormat', null),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to iterate once per formatting interval
                unique = Utils.getBooleanOption(options, 'unique', false),
                // the last processed entry model
                lastModel = null;

            // visit all collection entries covered by the passed interval
            return Utils.iterateArray(IntervalArray.get(intervals), function (interval) {
                return iterateModels(interval, function (entryModel, first, last) {

                    try { // to be able to use a finally block

                        // if specified, skip columns/rows without matching custom size attribute
                        if (_.isBoolean(customSize) && (entryModel.hasCustomSize() !== customSize)) { return; }

                        // if specified, skip rows without matching customFormat attribute
                        if (_.isBoolean(customFormat) && (entryModel.getMergedAttributes().row.customFormat !== customFormat)) { return; }

                        // iterating forward: visit previous hidden entry, if specified
                        if (!reverse && (hiddenMode === 'last') && lastModel && (entryModel.size > 0) && (lastModel.size === 0)) {
                            if (callback.call(context, new ColRowDescriptor(lastModel, first - 1), interval) === Utils.BREAK) {
                                return Utils.BREAK;
                            }
                        }

                        // iterating backward: visit hidden entry, if specified and last entry was visible
                        if (reverse && (hiddenMode === 'last') && lastModel && (entryModel.size === 0) && (lastModel.size > 0)) {
                            if (callback.call(context, new ColRowDescriptor(entryModel, last), interval) === Utils.BREAK) {
                                return Utils.BREAK;
                            }
                        }

                        // do not invoke callback for hidden columns/rows (unless hidden mode is 'all')
                        if ((hiddenMode !== 'all') && (entryModel.size === 0)) { return; }

                        // invoke callback once only in unique mode
                        var uniqueInterval = new Interval(first, last);
                        if (unique) {
                            return callback.call(context, new ColRowDescriptor(entryModel, reverse ? last : first), uniqueInterval, interval);
                        }

                        // invoke callback for every column/row covered by the current entry model
                        return Utils.iterateRange(reverse ? last : first, reverse ? (first - 1) : (last + 1), function (index) {
                            return callback.call(context, new ColRowDescriptor(entryModel, index), uniqueInterval, interval);
                        }, { step: reverse ? -1 : 1 });

                    // always update the variable 'lastModel', regardless where this method returns
                    } finally {
                        lastModel = entryModel;
                    }

                }, { reverse: reverse });
            }, { reverse: reverse });
        };

        /**
         * Returns the mixed column/row attributes of all entries covered by
         * the passed index intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @returns {Object}
         *  The mixed attributes of all columns/rows covered by the passed
         *  intervals, as simple object (NOT mapped as a 'column' or 'row'
         *  sub-object). The result does not contain any cell formatting
         *  attributes. All attributes that could not be resolved unambiguously
         *  will be set to the value null.
         */
        this.getMixedAttributes = function (intervals) {

            var // the resulting mixed attributes
                mixedAttributes = null;

            // merge passed intervals (do not visit entries twice)
            intervals = IntervalArray.get(intervals).merge();

            // visit all equally formatted intervals, and build the mixed attributes
            Utils.iterateArray(intervals, function (interval) {
                return iterateModels(interval, function (entryModel) {

                    var // the merged attributes of the current interval (type column/row only)
                        attributes = entryModel.getColRowAttributes(),
                        // whether any attribute is still unambiguous
                        hasNonNull = false;

                    // first visited interval: store initial attributes
                    if (!mixedAttributes) {
                        mixedAttributes = _.copy(attributes, true);
                        return;
                    }

                    // mix the attributes of the visited interval into the result
                    _.each(attributes, function (value, name) {
                        if (_.isEqual(value, mixedAttributes[name])) {
                            hasNonNull = true;
                        } else {
                            mixedAttributes[name] = null;
                        }
                    });

                    // stop iteration, if all attributes are ambiguous
                    if (!hasNonNull) { return Utils.BREAK; }
                });
            });

            return mixedAttributes;
        };

        /**
         * Returns whether the column/row with the specified index is visible.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @returns {Boolean}
         *  Whether the column/row is visible.
         */
        this.isEntryVisible = function (index) {
            return this.getEntry(index).size > 0;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; otherwise about the nearest visible column/row that
         * follows the column/row with the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @param {Number} [lastIndex]
         *  If specified, searching for a visible entry will stop after the
         *  column/row with this index. The index MUST be equal or greater than
         *  the passed index. If omitted, searches to the end of the sheet.
         *
         * @returns {ColRowDescriptor|Null}
         *  A descriptor object for the visible column/row; or null, if no more
         *  visible columns/rows are available.
         */
        this.getNextVisibleEntry = function (index, lastIndex) {

            var // the resulting column/row descriptor
                resultDesc = null;

            // use iterator to visit the first visible entry following the passed index
            this.iterateEntries(new Interval(index, _.isNumber(lastIndex) ? lastIndex : maxIndex), function (entryDesc) {
                resultDesc = entryDesc;
                return Utils.BREAK;
            });

            return resultDesc;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; otherwise about the nearest visible column/row that
         * precedes the column/row with the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @param {Number} [firstIndex]
         *  If specified, searching for a visible entry will stop before the
         *  column/row with this index. The index MUST be equal or less than
         *  the passed index. If omitted, searches to the beginning of the
         *  sheet.
         *
         * @returns {ColRowDescriptor|Null}
         *  A descriptor object for the visible column/row; or null, if no more
         *  visible columns/rows are available.
         */
        this.getPrevVisibleEntry = function (index, firstIndex) {

            var // the resulting column/row descriptor
                resultDesc = null;

            // use reverse iterator to visit the first visible entry preceding the passed index
            this.iterateEntries(new Interval(_.isNumber(firstIndex) ? firstIndex : 0, index), function (entryDesc) {
                resultDesc = entryDesc;
                return Utils.BREAK;
            }, { reverse: true });

            return resultDesc;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; optionally about a visible column/row located near
         * the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @param {String} [method='exact']
         *  Specifies how to look for another column/row, if the specified
         *  column/row is hidden. The following lookup methods are supported:
         *  - 'exact':
         *      Does not look for other columns/rows. If the specified
         *      column/row is hidden, returns null.
         *  - 'next':
         *      Looks for a visible column/row following the passed index.
         *  - 'prev':
         *      Looks for a visible column/row preceding the passed index.
         *  - 'nextPrev':
         *      First, looks for visible column/row following the passed index.
         *      If there is none available, looks for a visible column/row
         *      preceding the passed index.
         *  - 'prevNext':
         *      First, looks for visible column/row preceding the passed index.
         *      If there is none available, looks for a visible column/row
         *      following the passed index.
         *
         * @param {Interval} [boundInterval]
         *  If specified, a bounding index interval to restrict the result to.
         *  If no visible entry could be found inside that interval, null will
         *  be returned.
         *
         * @returns {ColRowDescriptor|Null}
         *  A descriptor object for the visible column/row; or null, if no
         *  visible columns/rows are available.
         */
        this.getVisibleEntry = function (index, method, boundInterval) {

            var // the resulting column/row descriptor
                entryDesc = null;

            function isValidDescriptor() {
                return _.isObject(entryDesc) && (entryDesc.size > 0) && (!boundInterval || boundInterval.containsIndex(entryDesc.index));
            }

            switch (method) {
            case 'next':
                entryDesc = this.getNextVisibleEntry(index);
                break;
            case 'prev':
                entryDesc = this.getPrevVisibleEntry(index);
                break;
            case 'nextPrev':
                entryDesc = this.getNextVisibleEntry(index);
                if (!isValidDescriptor()) { entryDesc = this.getPrevVisibleEntry(index); }
                break;
            case 'prevNext':
                entryDesc = this.getPrevVisibleEntry(index);
                if (!isValidDescriptor()) { entryDesc = this.getNextVisibleEntry(index); }
                break;
            default:
                entryDesc = this.getEntry(index);
            }

            return isValidDescriptor() ? entryDesc : null;
        };

        /**
         * Returns whether all columns/rows in the passed interval are hidden.
         *
         * @param {Interval} interval
         *  The index interval to be checked.
         *
         * @returns {Boolean}
         *  Whether all columns/rows in the passed interval are hidden.
         */
        this.isIntervalHidden = function (interval) {
            return this.getIntervalPosition(interval).size === 0;
        };

        /**
         * Returns the visible column/row intervals contained in the passed
         * interval.
         *
         * @param {Interval} interval
         *  The column/row interval to be processed.
         *
         * @param {Object} [options]
         *  The following options are supported:
         *  @param {Number} [options.lastHidden=0]
         *      If set to a number greater than 0, the start positions of the
         *      returned intervals will be decreased by the specified number,
         *      so that they include that number of hidden columns/rows
         *      preceding each interval of visible columns/rows.
         *
         * @returns {IntervalArray}
         *  An array with all visible column/row intervals contained in the
         *  passed interval. If the passed interval is completely hidden, the
         *  returned array will be empty.
         */
        this.getVisibleIntervals = function (interval, options) {

            var // number of preceding hidden column/row in each interval
                lastHidden = Utils.getIntegerOption(options, 'lastHidden', 0, 0),
                // the resulting visible intervals
                visibleIntervals = new IntervalArray(),
                // the last inserted result interval
                lastInterval = null;

            iterateModels(interval, function (entryModel, first, last) {

                if (entryModel.size === 0) { return; }

                // expand start point of the interval
                first = Math.max(interval.first, first - lastHidden);

                // try to expand last existing interval in the result
                if (lastInterval && (lastInterval.last + 1 >= first)) {
                    lastInterval.last = last;
                } else {
                    visibleIntervals.push(lastInterval = new Interval(first, last));
                }
            });

            return visibleIntervals;
        };

        /**
         * Returns a copy of the passed column/row interval. If the first entry
         * of the passed interval is hidden, the returned interval will start
         * at the first visible entry contained in the interval. Same applies
         * to the last entry of the passed interval. If the entire interval is
         * hidden, null will be returned.
         *
         * @param {Interval} interval
         *  The column/row interval to be processed.
         *
         * @returns {Interval|Null}
         *  The visible part of the passed interval (but may contain inner
         *  hidden columns/rows); or null, if the entire interval is hidden.
         */
        this.shrinkIntervalToVisible = function (interval) {

            var // first visible collection entry
                firstEntry = this.getNextVisibleEntry(interval.first, interval.last),
                // last visible collection entry
                lastEntry = firstEntry ? this.getPrevVisibleEntry(interval.last, interval.first) : null;

            // return the resulting interval
            return lastEntry ? new Interval(firstEntry.index, lastEntry.index) : null;
        };

        /**
         * Returns a column/row interval that covers the passed interval, and
         * that has been expanded to all hidden columns/rows directly preceding
         * and following the passed interval.
         *
         * @param {Interval} interval
         *  The column/row interval to be expanded.
         *
         * @returns {Interval}
         *  An expanded column/row interval including all leading and trailing
         *  hidden columns/rows.
         */
        this.expandIntervalToHidden = function (interval) {

            var // nearest visible collection entry preceding the interval
                prevEntry = (interval.first > 0) ? this.getPrevVisibleEntry(interval.first - 1) : null,
                // nearest visible collection entry following the interval
                nextEntry = (interval.last < maxIndex) ? this.getNextVisibleEntry(interval.last + 1) : null;

            // return the resulting interval
            return new Interval(prevEntry ? (prevEntry.index + 1) : interval.first, nextEntry ? (nextEntry.index - 1) : interval.last);
        };

        /**
         * Returns merged column/row intervals covering the visible parts of
         * the passed intervals. First, the intervals will be expanded to the
         * leading and trailing hidden columns/rows. This may reduce the number
         * of intervals in the result, if there are only hidden columns/rows
         * between the passed intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @returns {IntervalArray}
         *  The merged intervals, shrunken to leading/trailing visible
         *  columns/rows (but may contain inner hidden columns/rows).
         */
        this.mergeAndShrinkIntervals = function (intervals) {

            // expand passed intervals to hidden columns/rows
            intervals = IntervalArray.map(intervals, this.expandIntervalToHidden, this);

            // merge and sort the intervals (result may be smaller due to expansion to hidden columns/rows)
            intervals = intervals.merge();

            // reduce merged intervals to visible parts (filter out intervals completely hidden)
            return IntervalArray.map(intervals, this.shrinkIntervalToVisible, this);
        };

        /**
         * Returns the absolute sheet offset of a specific position inside a
         * column/row in 1/100 mm, independent from the current sheet zoom
         * factor.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @param {Number} offsetHmm
         *  The relative offset inside the column/row, in 1/100 mm. If this
         *  offset is larger than the total size of the column/row, the
         *  resulting absolute sheet offset will be located at the end position
         *  of the column/row.
         *
         * @returns {Number}
         *  The absolute position of the specified entry offset, in 1/100 mm.
         */
        this.getEntryOffsetHmm = function (index, offsetHmm) {
            var entryDesc = this.getEntry(index);
            return entryDesc.offsetHmm + Utils.minMax(offsetHmm, 0, entryDesc.sizeHmm);
        };

        /**
         * Returns the absolute sheet offset of a specific position inside a
         * column/row in pixels, according to the current sheet zoom factor.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @param {Number} offsetHmm
         *  The relative offset inside the column/row, in 1/100 mm. If this
         *  offset is larger than the total size of the column/row, the
         *  resulting absolute sheet offset will be located at the end position
         *  of the column/row.
         *
         * @returns {Number}
         *  The absolute position of the specified entry offset, in pixels.
         */
        this.getEntryOffset = function (index, offsetHmm) {
            var entryDesc = this.getEntry(index);
            return entryDesc.offset + Utils.minMax(convertHmmToPixels(offsetHmm, zoom), 0, entryDesc.size);
        };

        /**
         * Returns information about the column/row covering the passed offset
         * in the sheet in pixels according to the current sheet zoom factor,
         * or in 1/100 of millimeters.
         *
         * @param {Number} offset
         *  The absolute offset in the sheet, in pixels, or in 1/100 of
         *  millimeters, according to the 'pixel' option (see below).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.pixel=false]
         *      If set to true, the parameter 'offset' is interpreted as length
         *      in pixels according to the current sheet zoom factor.
         *      Otherwise, the offset is interpreted in 1/100 of millimeters.
         *  @param {Boolean} [options.outerHidden=false]
         *      If set to true, and the passed offset is outside the sheet area
         *      (less than 0, or greater than the last available offset in the
         *      sheet), returns a descriptor for the very first or very last
         *      column/row of the sheet, regardless if this column/row is
         *      hidden or visible.
         *
         * @returns {ColRowDescriptor}
         *  A descriptor object for the column/row, with the additional
         *  properties 'relOffsetHmm' and 'relOffset' (relative offset inside
         *  the column/row). If the passed offset is less than 0, returns
         *  information about the first visible column/row in the sheet; if the
         *  passed offset is greater than the total size of the sheet, returns
         *  information about the last visible column/row in the sheet (unless
         *  the option 'outerHidden' has been set, see above).
         */
        this.getEntryByOffset = function (offset, options) {

            var // whether to use pixels instead of 1/100 mm
                pixel = Utils.getBooleanOption(options, 'pixel', false),
                // the property names for offset and size
                OFFSET_NAME = pixel ? 'offset' : 'offsetHmm',
                SIZE_NAME = pixel ? 'size' : 'sizeHmm',
                // the total size of the sheet
                totalSize = pixel ? self.getTotalSize() : self.getTotalSizeHmm(),
                // the passed offset, restricted to the valid sheet dimension
                currOffset = Utils.minMax(offset, 0, totalSize - 1),
                // the index of the first entry model after the offset
                arrayIndex = _.sortedIndex(entryModels, Utils.makeSimpleObject(OFFSET_NAME, currOffset + 1), OFFSET_NAME),
                // the current entry model
                entryModel = entryModels[arrayIndex - 1],
                // the end offset of the previous entry
                endOffset = !entryModel ? 0 : pixel ? entryModel.getEndOffset() : entryModel.getEndOffsetHmm(),
                // the column/row index, relative to the entry or the gap
                relIndex = 0,
                // the resulting entry descriptor
                entryDesc = null;

            // special handling if all entries are hidden: return first entry
            if (totalSize === 0) {
                entryDesc = self.getEntry(0);
                entryDesc.relOffset = entryDesc.relOffsetHmm = 0;
                return entryDesc;
            }

            // offset outside sheet limits: always return first/last entry model if specified
            if (Utils.getBooleanOption(options, 'outerHidden', false)) {
                if (offset < 0) {
                    entryDesc = self.getEntry(0);
                } else if (offset >= totalSize) {
                    entryDesc = self.getEntry(maxIndex);
                }
                if (entryDesc && (entryDesc.size === 0)) {
                    entryDesc.relOffset = entryDesc.relOffsetHmm = 0;
                    return entryDesc;
                }
            }

            // offset points into the previous entry
            if (entryModel && (currOffset < endOffset)) {
                relIndex = Math.floor((currOffset - entryModel[OFFSET_NAME]) / entryModel[SIZE_NAME]);
                entryDesc = new ColRowDescriptor(entryModel, entryModel.interval.first + relIndex);
            } else {
                // offset points into the gap before the entry model at arrayIndex
                relIndex = Math.floor((currOffset - endOffset) / defaultModel[SIZE_NAME]);
                entryDesc = new ColRowDescriptor(defaultModel, (entryModel ? (entryModel.interval.last + 1) : 0) + relIndex);
                // adjust offsets (default entry is relative to index 0, not to start index of current gap)
                entryDesc.offsetHmm = (entryModel ? entryModel.getEndOffsetHmm() : 0) + defaultModel.sizeHmm * relIndex;
                entryDesc.offset = (entryModel ? entryModel.getEndOffset() : 0) + defaultModel.size * relIndex;
            }

            // add relative offset properties
            if (pixel) {
                entryDesc.relOffset = Utils.minMax(offset - entryDesc.offset, 0, entryDesc.size);
                entryDesc.relOffsetHmm = Utils.minMax(convertPixelsToHmm(entryDesc.relOffset, zoom), 0, entryDesc.sizeHmm);
            } else {
                entryDesc.relOffsetHmm = Utils.minMax(offset - entryDesc.offsetHmm, 0, entryDesc.sizeHmm);
                entryDesc.relOffset = Utils.minMax(convertHmmToPixels(entryDesc.relOffsetHmm, zoom), 0, entryDesc.size);
            }

            return entryDesc;
        };

        /**
         * Converts the passed offset to a scroll anchor position which
         * contains the entry index, and a floating-point ratio describing the
         * exact position inside the entry.
         *
         * @param {Number} offset
         *  The absolute offset in the sheet, in pixels, or in 1/100 of
         *  millimeters, according to the 'pixel' option (see below).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.pixel=false]
         *      If set to true, the parameter 'offset' is interpreted as length
         *      in pixels according to the current sheet zoom factor.
         *      Otherwise, the offset is interpreted in 1/100 of millimeters.
         *  @param {Boolean} [options.outerHidden=false]
         *      If set to true, and the passed offset is outside the sheet area
         *      (less than 0, or greater than the last available offset in the
         *      sheet), returns a scroll anchor for the very first or very last
         *      column/row of the sheet, regardless if this column/row is
         *      hidden or visible.
         *
         * @returns {Number}
         *  A scroll anchor position, as floating-point number. The integral
         *  part represents the zero-based column/row index of the collection
         *  entry that contains the passed offset. The fractional part
         *  represents the ratio inside the collection entry.
         */
        this.getScrollAnchorByOffset = function (offset, options) {
            var entryDesc = this.getEntryByOffset(offset, options);
            return entryDesc.index + ((entryDesc.sizeHmm > 0) ? (entryDesc.relOffsetHmm / entryDesc.sizeHmm) : 0);
        };

        /**
         * Converts the passed absolute offset in the sheet given in pixels
         * (according to the current sheet zoom factor) to an absolute offset
         * in 1/100 mm (independent from the current sheet zoom factor).
         *
         * @param {Number} offset
         *  An absolute offset in the sheet in pixels (according to the current
         *  sheet zoom factor).
         *
         * @returns {Number}
         *  The absolute offset in the sheet in 1/100 mm (independent from the
         *  current sheet zoom factor).
         */
        this.convertOffsetToHmm = function (offset) {
            var entryDesc = this.getEntryByOffset(offset, { pixel: true });
            return entryDesc.offsetHmm + entryDesc.relOffsetHmm;
        };

        /**
         * Converts the passed absolute offset in the sheet given in 1/100 mm
         * (independent from the current sheet zoom factor) to an absolute
         * offset in pixels (according to the current sheet zoom factor).
         *
         * @param {Number} offsetHmm
         *  An absolute offset in the sheet in 1/100 mm (independent from the
         *  current sheet zoom factor).
         *
         * @returns {Number}
         *  The absolute offset in the sheet in pixels (according to the
         *  current sheet zoom factor).
         */
        this.convertOffsetToPixel = function (offsetHmm) {
            var entryDesc = this.getEntryByOffset(offsetHmm);
            return entryDesc.offset + entryDesc.relOffset;
        };

        /**
         * Calculates the absolute offset of the passed scroll anchor, in 1/100
         * of millimeters.
         *
         * @param {Number} scrollAnchor
         *  A valid scroll anchor value as floating-point number. See return
         *  value of the method ColRowCollection.getScrollAnchorByOffset() for
         *  details.
         *
         * @returns {Number}
         *  The absolute offset of the passed scroll anchor, in 1/100 mm.
         */
        this.convertScrollAnchorToHmm = function (scrollAnchor) {
            var entryDesc = this.getEntry(Math.floor(scrollAnchor));
            return entryDesc.offsetHmm + Math.round(entryDesc.sizeHmm * (scrollAnchor % 1));
        };

        /**
         * Calculates the absolute offset of the passed scroll anchor, in
         * pixels according to the current sheet zoom factor.
         *
         * @param {Number} scrollAnchor
         *  A valid scroll anchor value as floating-point number. See return
         *  value of the method ColRowCollection.getScrollAnchorByOffset() for
         *  details.
         *
         * @returns {Number}
         *  The absolute offset of the passed scroll anchor, in pixels.
         */
        this.convertScrollAnchorToPixel = function (scrollAnchor) {
            var entryDesc = this.getEntry(Math.floor(scrollAnchor));
            return entryDesc.offset + Math.round(entryDesc.size * (scrollAnchor % 1));
        };

        /**
         * Returns the position and size of the specified column/row interval.
         *
         * @param {Interval} interval
         *  The index interval of the columns/rows, with the zero-based
         *  properties 'first' and 'last'.
         *
         * @returns {Object}
         *  An object with the following properties:
         *  - {Number} position.offsetHmm
         *      The absolute position of the first column/row in 1/100 mm,
         *      independent from the current sheet zoom factor.
         *  - {Number} position.sizeHmm
         *      The total size of the interval in 1/100 mm, independent form
         *      the current sheet zoom factor.
         *  - {Number} position.offset
         *      The absolute position of the first column/row in pixels,
         *      according to the current sheet zoom factor.
         *  - {Number} position.size
         *      The total size of the interval in pixels, according to the
         *      current sheet zoom factor.
         */
        this.getIntervalPosition = function (interval) {

            var // descriptor for the first column/row of the interval
                firstEntryDesc = this.getEntry(interval.first),
                // descriptor for the first column/row following the interval
                lastEntryDesc = this.getEntry(interval.last + 1);

            return {
                offsetHmm: firstEntryDesc.offsetHmm,
                sizeHmm: lastEntryDesc.offsetHmm - firstEntryDesc.offsetHmm,
                offset: firstEntryDesc.offset,
                size: lastEntryDesc.offset - firstEntryDesc.offset
            };
        };

        /**
         * Changes the formatting attributes of all entries contained in the
         * passed index interval, and triggers a 'change:entries' event
         * containing the interval passed to this method.
         *
         * @param {Interval} interval
         *  The index interval of the collection entries to be changed.
         *
         * @param {Object} attributes
         *  The new attributes for the entries in this collection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.silent=false]
         *      If set to true, no event will be triggered by this method.
         *  @param {Boolean} [options.rangeBorders=false]
         *      If set to true, attributes for outer borders will be applied at
         *      the outer borders of the interval only, and attributes for
         *      inner borders will be applied inside the interval. Otherwise,
         *      the outer border attributes will be applied for all borders,
         *      and the inner border attributes will be ignored.
         *  @param {Boolean} [options.visibleBorders=false]
         *      If set to true, border attributes may be incomplete. Existing
         *      border properties will be applied for visible borders in the
         *      interval only.
         *
         * @returns {Boolean}
         *  Whether changing the attributes has succeeded.
         */
        this.setAttributes = function (interval, attributes, options) {

            var // whether any entry model has been changed
                changed = false,
                // index of the first column/row with dirty pixel offset due to changed sizes
                dirtyIndex = null,
                // range border mode (special handling for inner border attributes)
                rangeBorders = Utils.getBooleanOption(options, 'rangeBorders', false) && _.isObject(attributes.cell),
                // special treatment for border attributes applied to entire ranges
                visibleBorders = Utils.getBooleanOption(options, 'visibleBorders', false);

            // update all entry models in the passed interval, move border attributes accordingly
            function updateInterval(currInterval, innerToLeading, innerToTrailing) {

                var // options for special inner/outer border treatment
                    attributeOptions = { visibleBorders: visibleBorders };

                // add options for inner/outer border treatment
                if (rangeBorders) {
                    attributeOptions.innerLeft = !columns || innerToLeading;
                    attributeOptions.innerRight = !columns || innerToTrailing;
                    attributeOptions.innerTop = columns || innerToLeading;
                    attributeOptions.innerBottom = columns || innerToTrailing;
                }

                // update all entry models in the interval
                iterateModels(currInterval, function (entryModel) {

                    var // old size of the entry, in 1/100 mm and in pixels
                        oldSizeHmm = entryModel.sizeHmm,
                        oldSize = entryModel.size;

                    // set the passed attributes; post-processing only if any attributes have changed
                    if (!entryModel.setAttributes(attributes, attributeOptions)) { return; }

                    // collection has changed
                    changed = true;

                    // update effective size of the entry model
                    entryModel.updateSize(zoom);

                    // set index of entries with dirty pixel offsets, if size has been changed
                    if (((oldSizeHmm !== entryModel.sizeHmm) || (oldSize !== entryModel.size)) && _.isNull(dirtyIndex)) {
                        dirtyIndex = entryModel.interval.last + 1;
                    }
                }, { modify: true });
            }

            // update all affected entry models
            if (rangeBorders && (interval.first < interval.last)) {
                updateInterval(new Interval(interval.first), false, true);
                if (interval.first + 1 < interval.last) {
                    updateInterval(new Interval(interval.first + 1, interval.last - 1), true, true);
                }
                updateInterval(new Interval(interval.last), true, false);
            } else {
                updateInterval(interval, false, false);
            }

            // update pixel offsets of following entry models, if the size has been changed
            if (_.isNumber(dirtyIndex)) {
                updateModelGeometry(dirtyIndex);
            }

            // notify change listeners
            if (changed && !Utils.getBooleanOption(options, 'silent', false)) {
                this.trigger('change:entries', interval, attributes, options);
            }

            // log the entire collection
            debugDump('interval ' + interval.stringifyAs(columns) + ' changed attributes');
            return true;
        };

        // initialization -----------------------------------------------------

        // initialize column/row size of the default entry model
        defaultModel.updateSize(zoom);

        // update entry sizes, after the default column/row size defined by the sheet has been changed
        sheetModel.on({
            'change:attributes': updateDefaultSize,
            'change:viewattributes': changeViewAttributesHandler
        });

        // update entry sizes after the document has been loaded
        if (!docModel.getApp().isImportFinished()) {
            // the collection may be destroyed before import finishes (operation 'deleteSheet')
            this.listenTo(docModel.getApp().getImportPromise(), 'done', updateDefaultSize);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(entryModels, 'destroy');
            defaultModel.destroy();
            self = docModel = sheetModel = entryModels = defaultModel = null;
        });

    } // class ColRowCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ColRowCollection });

});
