/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/portal/spreadsheet/fulltour', [
    'io.ox/office/portal/portalutils',
    'io.ox/office/portal/portaltour',
    'gettext!io.ox/office/portal/main'
], function (PortalUtils, PortalTour, gt) {

    'use strict';

    // class SpreadsheetFullTour ==============================================

    /**
     * The guided tour for the Spreadsheet Portal application.
     *
     * @constructor
     *
     * @extends PortalTour
     */
    function SpreadsheetFullTour(options) {

        var // self reference
            self = this,

            // whether to show the short version of the spreadsheet tour
            shortTour = options && options.auto && PortalTour.isAnyPortalTourShown();

        // base constructor ---------------------------------------------------

        PortalTour.call(this, 'spreadsheet');

        // private methods ----------------------------------------------------

        /**
         * Returns the localized version of the passed cell formula. Replaces
         * 'SUM' with the translated function name, and all commas with the
         * local function parameter separator.
         */
        function getLocalizedFormula(formula) {
            return formula
                .replace(/SUM/g, self.getApp().getLocalizedFunctionName('SUM'))
                .replace(/,/g, self.getApp().getListSeparator());
        }

        // initialization -----------------------------------------------------

        // the initial welcome message
        this.welcomeStep()
            .title(gt.pgettext('tour', 'The Spreadsheet app'))
            .content(gt.pgettext('tour', 'Welcome to your new Spreadsheet app. This Guided Tour will introduce you to the most important functions of creating and editing spreadsheets online.'));

        // do not show the following steps in the short version of the tour
        if (!shortTour) {

            // highlight the header of the template area
            this.templatesAreaStep()
                .title(gt.pgettext('tour', 'Templates'))
                .content(gt.pgettext('tour', 'Clicking on the Spreadsheet app in the navigation bar displays an overview of all the templates available to you in Spreadsheet.'));

            // describe the 'Blank document' button
            this.blankDocumentStep()
                .title(gt.pgettext('tour', 'Blank spreadsheet'))
                .content(gt.pgettext('tour', 'You can choose from a variety of templates to help you create professional documents.'))
                .content($('<br>'))
                .content(gt.pgettext('tour', 'Let\'s start with a blank document.'));
        }

        // launch the editor application with a new empty document
        this.launchDocumentStep();

        // do not show the following steps in the short version of the tour
        if (!shortTour) {

            // insert document contents, show the 'Saving changes' label
            this.autoSaveStep(function () {
                this.executeItem('cell/active/value', 'Lorem ipsum dolor sit amet.');
            });

            // describe the 'File' toolbar
            this.fileTabStep();
            this.saveAsStep();
            this.sendAsMailStep();
        }

        // start to type a cell formula
        this.step()
            .title(gt.pgettext('tour', 'Formulas'))
            .content(gt.pgettext('tour', 'You can use all the regular spreadsheet formulas by typing them into a cell.'))
            .on('before:show', function () {
                this.getDocView().cancelCellEditMode();
                this.executeItem('cell/active/value', getLocalizedFormula('\'=SUM'));
            })
            .waitForAndReferTo(this.EDITOR_APP_PANE_SELECTOR + ' .selection-layer .active-cell', { position: 'bottom', hotspot: true })
            // TODO: allow to step back in short tours
            .enableBackButton(!shortTour);

        // show auto-completion drop-down menu for functions
        this.step()
            .title(gt.pgettext('tour', 'Autocompletion'))
            .content(gt.pgettext('tour', 'As soon as you start typing a formula, autocompletion offers you a list of matching functions to choose from.'))
            .on('before:show', function () {
                var docView = this.getDocView();
                docView.cancelCellEditMode();
                docView.enterCellEditMode({ text: getLocalizedFormula('=SUM'), stickyList: true });
            })
            .waitForAndReferToPopup({ position: 'bottom', hotspot: true });

        // describe the formula label in the status pane
        this.step()
            .title(gt.pgettext('tour', 'Displaying hidden formulas'))
            .content(gt.pgettext('tour', 'If a result is displayed in a cell, the calculating formula will be displayed in the lower right hand corner of the screen.'))
            .on('before:show', function () {
                this.getDocView().cancelCellEditMode();
                this.executeItem('cell/active/value', getLocalizedFormula('=SUM(1, 2, 3)'));
            })
            .waitForAndReferToGroup(this.EDITOR_WINDOW_SELECTOR + ' .status-pane', 'view/formula', { position: 'top', hotspot: true });

        // short tour: final tour step closing the document, and describing how to restart the tour
        if (shortTour) {
            this.closeDocumentAndRestartTourStep();
            return;
        }

        // describe the 'Close' button, close the document
        this.closeButtonStep();
        this.closeDocumentStep();

        // final tour step describing how to restart the tour
        this.restartTourStep();

    } // class SpreadsheetFullTour

    // exports ================================================================

    return PortalTour.extend({ constructor: SpreadsheetFullTour });

});
