/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/languagepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/control/radiolist',
    'gettext!io.ox/office/editframework/main'
], function (Utils, IO, RadioList, gt) {

    'use strict';

    var // all supported languages, mapped by locale code
        LOCALE_DEFINITIONS = [
            { locale: 'en-US', label: gt('English (US)') },
            { locale: 'en-GB', label: gt('English (UK)') },
            { locale: 'de-DE', label: gt('German') },
            { locale: 'fr-FR', label: gt('French') },
            { locale: 'es-ES', label: gt('Spanish') },
            { locale: 'cs-CZ', label: gt('Czech') },
            { locale: 'da-DK', label: gt('Danish') },
            { locale: 'nl-NL', label: gt('Dutch (Netherlands)') },
            { locale: 'fi-FI', label: gt('Finnish') },
            { locale: 'el-GR', label: gt('Greek') },
            { locale: 'hu-HU', label: gt('Hungarian') },
            { locale: 'it-IT', label: gt('Italian (Italy)') },
            { locale: 'pl-PL', label: gt('Polish') },
            { locale: 'pt-PT', label: gt('Portuguese (Portugal)') },
            { locale: 'ro-RO', label: gt('Romanian') },
            { locale: 'ru-RU', label: gt('Russian') },
            { locale: 'sv-SE', label: gt('Swedish (Sweden)') }
        ],

        // a Promise that will resolve when the supported locales have been received
        supportedLocalesPromise = IO.sendRequest('spellchecker', { action: 'supportedlocales' }, { method: 'POST' })
        .done(function (data) {
            var supportedLocales = Utils.getArrayOption(data, 'SupportedLocales', []);
            _.each(LOCALE_DEFINITIONS, function (definition) {
                var localeName = definition.locale.replace('-', '_');
                definition.supported = _.contains(supportedLocales, localeName);
            });
        });

    // class LanguagePicker ===================================================

    /**
     * A drop-down list with all supported languages.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the RadioList base class.
     *  Additionally, the following options are supported:
     *  @param {Boolean} [initOptions.showListIcons=true]
     *      Whether to show the check boxes in the list.
     */
    function LanguagePicker(initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions(initOptions, {
            tooltip: gt('Text language'),
            sortItems: true
        }));

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('language-picker');

        this.listenTo(supportedLocalesPromise, 'always', function () {

            var showListIcons = Utils.getBooleanOption(initOptions, 'showListIcons', true);

            LOCALE_DEFINITIONS.forEach(function (definition) {

                var options = { label: definition.label };
                if (showListIcons) {
                    options.icon = definition.supported ? 'fa-check-square-o' : 'fa-square-o';
                }

                self.createOptionButton(definition.locale, options);
            });
        });

        // destroy all members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class LanguagePicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: LanguagePicker });

});
