/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/stylecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/utils/attributeutils'
], function (Utils, TimerMixin, ModelObject, AttributeUtils) {

    'use strict';

    // class StyleCollection ==================================================

    /**
     * Collection for hierarchical style sheets and auto styles of a specific
     * attribute family.
     *
     * Attribute definitions that have been registered with the method
     * EditModel.registerAttributeDefinitions() may contain the following
     * additional properties:
     *  - {Function} [definition.format]
     *      A function that applies the passed attribute value to a DOM element
     *      (usually its CSS formatting). Will be called in the context of this
     *      style sheet collection. The function receives the DOM element as
     *      jQuery object in the first parameter, and the attribute value in
     *      the second parameter. An alternative way to update the element
     *      formatting using a complete map of all attribute values is to
     *      specify a global format handler (see options below).
     *  - {Function|Boolean} [definition.preview]
     *      A function that applies the passed attribute value to a DOM element
     *      used to show a preview of a style sheet in the GUI. Will be called
     *      in the context of this style sheet collection. The function
     *      receives the DOM element as jQuery object in the first parameter,
     *      and the attribute value in the second parameter. An alternative way
     *      to update the preview element formatting using a complete map of
     *      all attribute values is to specify a global preview handler (see
     *      options below). If the boolean value 'true' will be set instead of
     *      a function, the registered 'format' method will be called instead.
     *
     * Triggers the following events:
     *  - 'insert:stylesheet': After a new entry sheet has been inserted into
     *      this collection. The event handler receives the identifier of the
     *      new style.
     *  - 'delete:stylesheet': After an entry has been deleted from this
     *      collection. The event handler receives the identifier of the
     *      deleted style.
     *  - 'change:stylesheet': After the formatting attributes of an entry in
     *      this collection have been changed. The event handler receives the
     *      identifier of the changed style.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     *
     * @param {String} styleFamily
     *  The main attribute family represented by the style sheets contained by
     *  instances of the derived collection class. The style sheets and the DOM
     *  elements referring to the style sheets must support all attributes of
     *  this attribute family.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. All callback functions will be called in the
     *  context of this style sheet collection.
     *  @param {Boolean} [initOptions.styleSheetSupport=true]
     *      If set to false, this collection is not allowed to contain style
     *      sheets. It can only be used to format DOM elements with explicit
     *      formatting attributes. DEPRECATED!
     *  @param {Boolean} [initOptions.autoStyleSupport=false]
     *      If set to true, this collection supports auto styles (internal
     *      hidden style sheets used as explicit formatting at any document
     *      model objects).
     *  @param {String} [initOptions.families]
     *      The names of additional attribute families supported by the style
     *      sheets of this collection (space-separated). It is not needed to
     *      repeat the name of the style family (as passed to the 'styleFamily'
     *      parameter) here.
     *  @param {Object} [initOptions.parentResolvers]
     *      The parent style families whose associated style sheets can contain
     *      attributes of the family supported by this style sheet collection.
     *      The DOM elements referring to the style sheets of the specified
     *      style families must be ancestors of the DOM elements referring to
     *      the style sheets of this container. The passed object maps the
     *      attribute family to an ancestor element resolver function. The
     *      function receives the descendant DOM element as jQuery object in
     *      the first parameter, and returns the ancestor element of that DOM
     *      element which is associated to the parent style family used as map
     *      key.
     *  @param {Function} [initOptions.formatHandler]
     *      A callback function that will be called for every DOM element whose
     *      attributes have been changed, and that will format the DOM element
     *      according to the new attribute values. In difference to the
     *      individual formatter methods specified in the definitions for
     *      single attributes, this handler will be called once for a DOM
     *      element regardless of the number of changed attributes. Receives
     *      the following parameters:
     *      (1) {jQuery} element
     *          The element whose attributes have been changed.
     *      (2) {Object} mergedAttributes
     *          The effective attribute values merged from style sheets and
     *          explicit attributes, as map of attribute maps (name/value
     *          pairs), keyed by attribute family.
     *      (3) {Boolean} async
     *          If set to true, the format handler may execute asynchronously
     *          in a browser timeout loop. In this case, the handler MUST
     *          return a Deferred object that will be resolved after execution.
     *  @param {Function} [initOptions.previewHandler]
     *      A callback function that will be called if a DOM element used in
     *      the GUI to show a preview of a style sheet will be formatted. In
     *      difference to the individual previewer methods specified in the
     *      definitions for single attributes, this handler will be called once
     *      for the DOM element regardless of the number of changed attributes.
     *      Receives the following parameters:
     *      (1) {jQuery} element
     *          The element whose attributes have been changed.
     *      (2) {Object} mergedAttributes
     *          The effective attribute values merged from style sheets and
     *          explicit attributes, as map of attribute maps (name/value
     *          pairs), keyed by attribute family.
     *      (3) {Boolean} async
     *          If set to true, the preview handler may execute asynchronously
     *          in a browser timeout loop. In this case, the handler MUST
     *          return a Deferred object that will be resolved after execution.
     *  @param {Function} [initOptions.styleAttributesResolver]
     *      A function that extracts and returns specific attributes from the
     *      complete 'attributes' object of a style sheet. The attributes
     *      returned by this function may depend on a source element (e.g. on
     *      the position of this source element in its parents). The function
     *      receives the following parameters:
     *      (1) {Object} styleAttributes
     *          The complete original 'attributes' object of the style sheet.
     *          MUST NOT be changed by the callback function!
     *      (2) {jQuery} element
     *          The element referring to the style sheet.
     *      (3) {jQuery} [sourceNode]
     *          The descendant source node.
     *  @param {Function} [initOptions.elementAttributesResolver]
     *      A function that extracts and returns specific attributes from the
     *      explicit attribute map of an element. The attributes returned by
     *      this function may depend on a source element (e.g. on the position
     *      of this source element in its parents). The function receives the
     *      following parameters:
     *      (1) {Object} elementAttributes
     *          The original explicit element attributes of the style sheet, as
     *          map of attribute maps (name/value pairs), keyed by attribute
     *          family. MUST NOT be changed by the callback function!
     *      (2) {jQuery} element
     *          The element whose attributes have been passed in the first
     *          parameter.
     *      (3) {jQuery} [sourceNode]
     *          The descendant source node.
     */
    function StyleCollection(docModel, styleFamily, initOptions) {

        var // self reference
            self = this,

            // style sheets, mapped by identifier
            styleSheets = {},

            // the names of all supported attribute families
            supportedFamilies = null,

            // identifier of the default style sheet
            defaultStyleId = null,

            // identifier of the default auto style
            defaultAutoStyleId = null,

            // whether this collection supports style sheets
            styleSheetSupport = Utils.getBooleanOption(initOptions, 'styleSheetSupport', true),

            // whether this collection supports auto styles
            autoStyleSupport = Utils.getBooleanOption(initOptions, 'autoStyleSupport', false),

            // the resolver functions for DOM ancestor elements, keyed by attribute family
            parentResolvers = Utils.getObjectOption(initOptions, 'parentResolvers', {}),

            // formatting handler, called after attributes of a model element have been changed
            formatHandler = Utils.getFunctionOption(initOptions, 'formatHandler'),

            // preview handler, called after attributes of a GUI preview element have been changed
            previewHandler = Utils.getFunctionOption(initOptions, 'previewHandler'),

            // custom resolver for style attributes depending on a context element
            styleAttributesResolver = Utils.getFunctionOption(initOptions, 'styleAttributesResolver'),

            // custom resolver for explicit element attributes depending on a context element
            elementAttributesResolver = Utils.getFunctionOption(initOptions, 'elementAttributesResolver');

        // base constructors --------------------------------------------------

        ModelObject.call(this, docModel);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns whether the passed style sheet is a descendant of the other
         * passed style sheet.
         */
        function isDescendantStyleSheet(styleSheet, ancestorStyleSheet) {
            while (styleSheet) {
                if (styleSheet === ancestorStyleSheet) { return true; }
                styleSheet = styleSheets[styleSheet.parentId];
            }
            return false;
        }

        /**
         * Delete all cached merged attributes in all style sheets.
         */
        function deleteCachedAttributes() {
            _.each(styleSheets, function (styleSheet) {
                delete styleSheet.mergedAttributes;
            });
        }

        /**
         * Restricts the passed attribute set to the attribute families
         * supported by this style sheet collection.
         *
         * @param {Object} mergedAttributes
         *  The attribute set whose member field not representing a supported
         *  attribute map will be removed.
         */
        function restrictToSupportedFamilies(mergedAttributes) {
            // remove attribute maps of unsupported families
            _.each(mergedAttributes, function (unused, family) {
                if (!_.contains(supportedFamilies, family)) {
                    delete mergedAttributes[family];
                }
            });
        }

        /**
         * Returns the complete attributes of an ancestor of the passed
         * element, if a parent style family has been registered and its
         * element resolver function returns a valid ancestor element, and
         * extends missing attributes by their current default values.
         *
         * @param {HTMLElement|jQuery} [element]
         *  An element whose ancestor will be searched and queried for its
         *  attributes. If this object is a jQuery collection, uses the first
         *  DOM node it contains. If missing, returns just the current default
         *  values of all supported attribute families.
         *
         * @returns {Object}
         *  The formatting attributes of an ancestor of the passed element, as
         *  map of attribute value maps (name/value pairs), keyed by attribute
         *  family. If no ancestor element has been found, returns the current
         *  default values for all supported attribute families.
         */
        function resolveBaseAttributes(element) {

            var // passed element, as jQuery object
                $element = $(element),
                // the matching parent element, its style family, the style sheet collection, and the attributes
                parentElement = null,
                parentFamily = null,
                parentStyleCollection = null,
                parentAttributes = null,
                // the resulting merged attributes of the ancestor element
                mergedAttributes = {};

            // collect attributes from ancestor element if specified (only one parent style family must match at a time)
            if ($element.length > 0) {

                // find a matching ancestor element and its style family (only one parent must match at a time)
                _.each(parentResolvers, function (elementResolver, family) {
                    if ($(parentElement).length === 0) {
                        parentElement = elementResolver.call(self, $element);
                        parentFamily = family;
                    }
                });

                // add the element attributes of the ancestor element (only the supported attribute families)
                if ($(parentElement).length > 0) {
                    parentStyleCollection = docModel.getStyleCollection(parentFamily);
                    parentAttributes = parentStyleCollection.getElementAttributes(parentElement, { special: true, sourceNode: $element });
                    docModel.extendAttributes(mergedAttributes, parentAttributes, { special: true });
                }
            }

            // remove attribute maps of unsupported families (this also removes the style identifier of the parent element)
            restrictToSupportedFamilies(mergedAttributes);

            // add missing default attribute values of supported families not found in the parent element
            supportedFamilies.forEach(function (family) {
                if (!(family in mergedAttributes)) {
                    _.extend(mergedAttributes, docModel.getDefaultAttributes(family));
                }
            });

            return mergedAttributes;
        }

        /**
         * Returns the attributes from the specified style sheet and its parent
         * style sheets. Does not add default attribute values, or attributes
         * from ancestors of the passed element.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {HTMLElement|jQuery} [element]
         *  An element referring to a style sheet in this collection whose
         *  attributes will be extracted. Will be passed to a custom style
         *  attributes resolver (see the option 'styleAttributesResolver'
         *  passed to the constructor). If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @param {HTMLElement|jQuery} [sourceNode]
         *  The source DOM node corresponding to a child attribute family that
         *  has initiated the call to this method. Will be passed to a custom
         *  style attributes resolver (see the option 'styleAttributesResolver'
         *  passed to the constructor). If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @returns {Object}
         *  The formatting attributes contained in the style sheet and its
         *  ancestor style sheets, as map of attribute value maps (name/value
         *  pairs), keyed by attribute family.
         */
        function resolveStyleAttributeSet(styleId, element, sourceNode) {

            var // passed element, as jQuery object
                $element = $(element),
                // passed source node, as jQuery object
                $sourceNode = $(sourceNode),
                // the resulting merged attributes of the style sheet and its ancestors
                mergedAttributes = {};

            // collects style sheet attributes recursively through parents
            function collectStyleAttributes(styleSheet) {

                // exit recursive call chain if no more parent styles are available
                if (!styleSheet) { return; }

                // call recursively to get the attributes of the parent style sheets
                collectStyleAttributes(styleSheets[styleSheet.parentId]);

                // add all attributes of the current style sheet, mapped directly by attribute family
                docModel.extendAttributes(mergedAttributes, styleSheet.attributes);

                // try user-defined resolver for style attributes mapped in non-standard structures
                if (_.isFunction(styleAttributesResolver) && ($element.length > 0)) {
                    docModel.extendAttributes(mergedAttributes, styleAttributesResolver.call(self, styleSheet.attributes, $element, $sourceNode));
                }
            }

            // fall-back to default style sheet if passed identifier is invalid
            styleId = self.getEffectiveStyleId(styleId);

            // collect attributes from the style sheet and its parents
            collectStyleAttributes(styleSheets[styleId]);

            // remove attribute maps of unsupported families
            restrictToSupportedFamilies(mergedAttributes);

            // replace identifiers of auto styles with parent style sheet identifier
            mergedAttributes.styleId = self.isAuto(styleId) ? self.getEffectiveStyleId(styleSheets[styleId].parentId) : styleId;

            return mergedAttributes;
        }

        /**
         * Returns the explicit attributes from the specified element. Does not
         * add default attribute values. Uses a custom element attributes
         * resolver function if specified in the constructor of this style
         * sheet collection.
         *
         * @param {HTMLElement|jQuery} [element]
         *  The element whose explicit attributes will be extracted. Will be
         *  passed to a custom attributes resolver (see the option
         *  'elementAttributesResolver' passed to the constructor). If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @param {HTMLElement|jQuery} [sourceNode]
         *  The source DOM node corresponding to a child attribute family that
         *  has initiated the call to this method. Will be passed to a custom
         *  element attributes resolver (see the option
         *  'elementAttributesResolver' passed to the constructor). If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @returns {Object}
         *  The explicit formatting attributes contained in the element, as map
         *  of attribute value maps (name/value pairs), keyed by attribute
         *  family.
         */
        function resolveElementAttributeSet(element, sourceNode) {

            var // the resulting merged attributes of the element
                mergedAttributes = AttributeUtils.getExplicitAttributes(element, { direct: true });

            // call custom element attribute resolver
            if (_.isFunction(elementAttributesResolver)) {
                mergedAttributes = elementAttributesResolver.call(self, mergedAttributes, $(element), $(sourceNode));
            }

            return mergedAttributes;
        }

        /**
         * Filters the passed attribute set by supported attributes in-place.
         */
        function filterSupportedAttributes(mergedAttributes, options) {

            _.each(mergedAttributes, function (attributeValues, family) {

                var // attribute definitions of the current family
                    definitions = docModel.getAttributeDefinitions(family);

                if (definitions) {
                    _.each(attributeValues, function (value, name) {
                        if (!AttributeUtils.isRegisteredAttribute(definitions, name, options)) {
                            delete attributeValues[name];
                        }
                    });
                } else if (!styleSheetSupport || (family !== 'styleId')) {
                    delete mergedAttributes[family];
                }
            });

            return mergedAttributes;
        }

        /**
         * Updates the element formatting according to the passed attributes.
         *
         * @param {jQuery} element
         *  The element whose formatting will be updated, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A complete attribute set with all attribute values merged from
         *  style sheet and explicit attributes.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the registered format or preview handler will
         *      be asked to execute asynchronously in a browser timeout loop.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  registered format or preview handler has finished. If the
         *  'options.async' options is set, and the format handler supports
         *  asynchronous execution, the Promise will be resolved sometime in
         *  the future. Otherwise, the returned Promise is already resolved
         *  when this method returns.
         */
        function updateElementFormatting(element, mergedAttributes, options) {

            var // definitions of own attributes
                definitions = docModel.getAttributeDefinitions(styleFamily),
                // whether to render a GUI preview element
                preview = Utils.getBooleanOption(options, 'preview', false),
                // whether to call the real format handler or the preview handler
                currentFormatHandler = preview ? previewHandler : formatHandler,
                // the result Deferred object returned from the format or preview handler
                resultDef = null;

            if (!preview) {
                self.DBG_COUNT = (self.DBG_COUNT || 0) + 1;
            }

            // call single format handlers for all attributes of the own style family
            _.each(mergedAttributes[styleFamily], function (value, name) {
                var formatMethod = null;
                if (name in definitions) {
                    formatMethod = Utils.getFunctionOption(definitions[name], 'format');
                    if (preview && (Utils.getBooleanOption(definitions[name], 'preview', false) !== true)) {
                        formatMethod = Utils.getFunctionOption(definitions[name], 'preview');
                    }
                    if (_.isFunction(formatMethod)) {
                        formatMethod.call(self, element, value);
                    }
                }
            });

            // call format handler taking all attributes at once
            if (_.isFunction(currentFormatHandler)) {
                try {
                    resultDef = currentFormatHandler.call(self, element, mergedAttributes, Utils.getBooleanOption(options, 'async', false));
                } catch (ex) {
                    Utils.exception(ex);
                    resultDef = $.Deferred().reject();
                }
            }

            // wrap resultDef in $.when() to convert plain values to a Deferred object
            return $.when(resultDef).promise();
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether this collection supports auto styles (internal
         * hidden style sheets used as explicit formatting at any document
         * model objects).
         *
         * @returns {Boolean}
         *  Whether this collection supports auto styles.
         */
        this.hasAutoStyleSupport = function () {
            return autoStyleSupport;
        };

        /**
         * Returns the style family name of this style sheet collection.
         *
         * @returns {String}
         *  The style family name of this style sheet collection.
         */
        this.getStyleFamily = function () {
            return styleFamily;
        };

        /**
         * Returns an array with the names of all attribute families supported
         * by style sheets contained in this instance, including the main style
         * family.
         *
         * @returns {Array<String>}
         *  A string array containing the supported attribute families.
         */
        this.getSupportedFamilies = function () {
            return supportedFamilies;
        };

        /**
         * Returns the identifier of the default style sheet.
         *
         * @returns {String|Null}
         *  The identifier of the default style sheet.
         */
        this.getDefaultStyleId = function () {
            return defaultStyleId;
        };

        /**
         * Returns the identifier of the default auto style.
         *
         * @returns {String|Null}
         *  The identifier of the default auto style.
         */
        this.getDefaultAutoStyleId = function () {
            return defaultAutoStyleId;
        };

        /**
         * Returns the identifier of the style sheet that will be effectively
         * used for the specified style sheet identifier. If the specified
         * style sheet does not exist, but this collection contains a default
         * style sheet, its unique identifier will be returned instead.
         * Otherwise, the passed style sheet identifier will be returned.
         *
         * @param {String} styleId
         *  The unique identifier of a style sheet.
         *
         * @param {Boolean} [preferAuto=false]
         *  If set to true, and there is no entry with the passed identifier in
         *  this collection, and this collection supports auto styles, and a
         *  default auto style exists, the identifier of the default auto style
         *  will be returned instead of the default style sheet.
         *
         * @returns {String}
         *  The passed style sheet identifier, if such a style sheet exists in
         *  this collection, otherwise the identifier of the default style
         *  sheet.
         */
        this.getEffectiveStyleId = function (styleId, preferAuto) {
            var defaultId = (preferAuto && defaultAutoStyleId) ? defaultAutoStyleId : defaultStyleId;
            return (!(styleId in styleSheets) && (defaultId in styleSheets)) ? defaultId : styleId;
        };

        /**
         * Inserts a new style sheet into this collection. An existing style
         * sheet with the specified identifier will be replaced.
         *
         * @param {String} styleId
         *  The unique identifier of of the new style sheet.
         *
         * @param {String} name
         *  The user-defined name of of the new style sheet.
         *
         * @param {String|Null} parentId
         *  The identifier of of the parent style sheet the new style sheet
         *  will derive undefined attributes from. The parent of an auto style
         *  MUST be a real style sheet.
         *
         * @param {Object} attributes
         *  The formatting attributes contained in the new style sheet. The
         *  structure of this object is dependent on the style family of this
         *  collection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, the style sheet should be displayed in the user
         *      interface.
         *  @param {Boolean} [options.auto=false]
         *      If set to true, the style sheet is used as auto style. Auto
         *      styles are always hidden in the user interface (the option
         *      'hidden' will be ignored). Support of auto styles must be
         *      explicitly enabled for this collection with the constructor
         *      option 'autoStyleSupport', otherwise trying to insert an auto
         *      style into this collection will fail.
         *  @param {String} [options.category]
         *      If specified, the name of a category that can be used to group
         *      style sheets into several sections in the user interface.
         *  @param {Number} [options.priority=0]
         *      The sorting priority of the style (the lower the value the
         *      higher the priority).
         *  @param {Boolean} [options.defStyle=false]
         *      True, if the new style sheet is the default style sheet of this
         *      style sheet collection. The default style will be used for all
         *      elements without explicit style sheet. Only the first style
         *      sheet that will be inserted with this flag will be permanently
         *      registered as default style sheet.
         *  @param {Boolean} [options.dirty=false]
         *      True, if the new style sheet has been added manually and not
         *      by loading a document. Therefore this style sheet needs to be
         *      saved as soon as it will be used.
         *
         * @returns {Boolean}
         *  Whether creating the new style sheet was successful.
         */
        this.insertStyleSheet = function (styleId, name, parentId, attributes, options) {

            var // the new style sheet instance
                styleSheet = null,
                // whether the passed style sheet is an auto style
                autoStyle = Utils.getBooleanOption(options, 'auto', false),
                // whether the passed style sheet is the default style of the attribute family
                defaultStyle = Utils.getBooleanOption(options, 'defStyle', false);

            // check validity of the style name
            if (!_.isString(styleId) || (styleId.length === 0)) {
                Utils.error('StyleCollection.insertStyleSheet(): missing style sheet identifier');
                return false;
            }

            // check that style sheet handling is enabled
            if (!autoStyle && !styleSheetSupport) {
                Utils.error('StyleCollection.insertStyleSheet(): creating style sheets not allowed for style family "' + styleFamily + '"');
                return false;
            }

            // check that auto style handling is enabled
            if (autoStyle && !autoStyleSupport) {
                Utils.error('StyleCollection.insertStyleSheet(): creating auto styles not allowed for style family "' + styleFamily + '"');
                return false;
            }

            // handle existing style entry
            if (styleId in styleSheets) {

                // auto styles cannot be replaced
                if (styleSheets[styleId].auto) {
                    Utils.error('StyleCollection.insertStyleSheet(): auto style "' + styleId + '" exists already');
                    return false;
                }

                // do not replace style sheets with auto styles
                if (autoStyle) {
                    Utils.error('StyleCollection.insertStyleSheet(): style sheet "' + styleId + '" cannot be replaced with an auto style');
                    return false;
                }

                // TODO: disallow multiple insertion of style sheets? (predefined style sheets may be created on demand without checking existence)
                Utils.warn('StyleCollection.insertStyleSheet(): style sheet "' + styleId + '" exists already');
            }

            // auto styles cannot be set as parent of style sheets, or other auto styles
            if (parentId && (parentId in styleSheets) && styleSheets[parentId].auto) {
                Utils.error('StyleCollection.insertStyleSheet(): auto style "' + parentId + '" cannot be set as parent of style "' + styleId + '"');
                return false;
            }

            // ensure that a new auto style does not become the parent of another style
            if (autoStyle && _.findWhere(styleSheets, { parentId: styleId })) {
                Utils.error('StyleCollection.insertStyleSheet(): auto style "' + styleId + '" cannot become the parent of another style');
                return false;
            }

            // get or create a style sheet object, set identifier and user-defined name of the style sheet
            styleSheet = styleSheets[styleId] || (styleSheets[styleId] = {});
            styleSheet.id = styleId;
            styleSheet.name = name || styleId;
            styleSheet.auto = autoStyle;

            // set parent of the style sheet, check for cyclic references
            styleSheet.parentId = parentId;
            if (isDescendantStyleSheet(styleSheets[styleSheet.parentId], styleSheet)) {
                Utils.warn('StyleCollection.insertStyleSheet(): cyclic reference, cannot set style sheet parent "' + parentId + '"');
                styleSheet.parentId = null;
            }

            // set style sheet options
            styleSheet.hidden = autoStyle || Utils.getBooleanOption(options, 'hidden', false);
            styleSheet.category = Utils.getStringOption(options, 'category', styleSheet.category);
            styleSheet.priority = Utils.getIntegerOption(options, 'priority', 0);
            styleSheet.dirty = Utils.getBooleanOption(options, 'dirty', false);
            styleSheet.builtIn = Utils.getBooleanOption(options, 'builtIn', styleSheet.builtIn || false);
            styleSheet.custom = Utils.getBooleanOption(options, 'custom', styleSheet.custom || false);

            // bug 27716: set first visible style as default style, if imported default style is hidden
            if (_.isString(defaultStyleId) && (defaultStyleId in styleSheets) && styleSheets[defaultStyleId].hidden && !styleSheet.hidden) {
                defaultStyleId = styleId;
            }

            // set default style sheet
            if (defaultStyle) {
                if (autoStyle) {
                    if (_.isNull(defaultAutoStyleId)) {
                        defaultAutoStyleId = styleId;
                    } else if (defaultAutoStyleId !== styleId) {
                        Utils.warn('StyleCollection.insertStyleSheet(): multiple default auto styles "' + defaultAutoStyleId + '" and "' + styleId + '"');
                    }
                } else {
                    if (_.isNull(defaultStyleId)) {
                        defaultStyleId = styleId;
                    } else if (defaultStyleId !== styleId) {
                        Utils.warn('StyleCollection.insertStyleSheet(): multiple default style sheets "' + defaultStyleId + '" and "' + styleId + '"');
                    }
                }
            }

            // store a deep clone of the passed attributes
            styleSheet.attributes = _.isObject(attributes) ? _.copy(attributes, true) : {};

            // delete all cached merged attributes after inserting a real style sheet (parent tree
            // may have changed, but not for auto styles which cannot be parents of other styles)
            if (!autoStyle) { deleteCachedAttributes(); }

            // notify listeners
            this.trigger('insert:stylesheet', styleId);
            return true;
        };

        this.changeStyleSheet = function (styleId, name, parentId, attributes/*, options*/) {

            if (!_.isString(styleId) || (styleId.length === 0)) {
                Utils.error('StyleCollection.changeStyleSheet(): missing style sheet identifier');
                return false;
            }

            var styleSheet = styleSheets[styleId];
            if (!styleSheet) {
                Utils.error('StyleCollection.changeStyleSheet(): style sheet "' + styleId + '" does not exist');
                return false;
            }

            // auto styles cannot be changed
            if (styleSheet.auto) {
                Utils.error('StyleCollection.changeStyleSheet(): auto style "' + styleId + '" cannot be changed');
                return false;
            }

            styleSheet.name = name || styleSheet.name;

            styleSheet.attributes = attributes || styleSheet.attributes;

            // notify listeners
            this.trigger('change:stylesheet', styleId);
            return true;
        };

        /**
         * Removes an existing style sheet from this style sheet collection.
         *
         * @param {String} styleId
         *  The unique identifier of of the style sheet to be removed.
         *
         * @returns {Boolean}
         *  Whether removing the style sheet was successful.
         */
        this.deleteStyleSheet = function (styleId) {

            if (!_.isString(styleId) || !(styleId in styleSheets)) {
                Utils.error('StyleCollection.deleteStyleSheet(): style sheet "' + styleId + '" does not exist');
                return false;
            }

            // default style sheet cannot be deleted
            if (styleId === defaultStyleId) {
                Utils.error('StyleCollection.deleteStyleSheet(): cannot remove default style sheet');
                return false;
            }

            // update parent of all style sheets referring to the removed style sheet
            _.each(styleSheets, function (childSheet) {
                if (styleId === childSheet.parentId) {
                    childSheet.parentId = styleSheets[styleId].parentId;
                }
            });

            var // the style sheet to be deleted
                styleSheet = styleSheets[styleId];

            // special behavior: built-in style sheets will not be deleted but invalidated
            if (styleSheet.builtIn) {
                this.setDirty(styleId, true);
                return true;
            }

            // remove style sheet from map
            delete styleSheets[styleId];

            // delete all cached merged attributes after deleting a real style sheet (parent tree
            // may have changed, but not for auto styles which cannot be parents of other styles)
            if (!styleSheet.auto) { deleteCachedAttributes(); }

            // notify listeners
            this.trigger('delete:stylesheet', styleId);
            return true;
        };

        /**
         * Returns whether this collection contains the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean}
         *  Whether this instance contains a style sheet with the passed
         *  identifier.
         */
        this.containsStyleSheet = function (styleId) {
            return styleId in styleSheets;
        };

        /**
         * Returns whether this collection contains a style sheet with the
         * specified name (not identifier).
         *
         * @param {String} styleName
         *  The name of the style sheet.
         *
         * @returns {Boolean}
         *  Whether this instance contains a style sheet with the passed name.
         */
        this.containsStyleSheetByName = function (styleName) {
            styleName = styleName.toLowerCase();
            return _.any(styleSheets, function (styleSheet) {
                return styleSheet.name.toLowerCase() === styleName;
            });
        };

        /**
         * Returns the style identifier of the first style sheet found with the
         * specified name.
         *
         * @param {String} styleName
         *  The name of the style sheet.
         *
         * @returns {String|Null}
         *  Whether this instance contains a style sheet with the passed name.
         */
        this.getStyleIdByName = function (styleName) {
            var resultId = null;
            styleName = styleName.toLowerCase();
            _.any(styleSheets, function (styleSheet, styleId) {
                if (styleSheet.name.toLowerCase() === styleName) {
                    resultId = styleId;
                    return true;
                }
            });
            return resultId;
        };

        /**
         * Returns the names of all style sheets in a map, keyed by their
         * unique identifiers.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.skipHidden=false]
         *      If set to true, the names of hidden style sheets will not be
         *      returned.
         *
         * @returns {Object}
         *  A map with all style sheet names, keyed by style sheet identifiers.
         */
        this.getStyleSheetNames = function (options) {

            var // whether to skip hidden style sheets
                skipHidden = Utils.getBooleanOption(options, 'skipHidden', false),
                // the result map
                names = {};

            _.each(styleSheets, function (styleSheet, styleId) {
                if (!skipHidden || !styleSheet.hidden) {
                    names[styleId] = styleSheet.name;
                }
            });
            return names;
        };

        /**
         * Returns the merged attributes from the specified style sheet and its
         * parent style sheets.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Object}
         *  A complete attribute set (a map of attribute maps with name/value
         *  pairs, keyed by the attribute families, additionally containing the
         *  style sheet reference in the 'styleId' property), containing values
         *  for all supported attributes, collected from the specified style
         *  sheet, and all its parent style sheets, up to the map of default
         *  attributes. For efficiency, a reference to the cached attribute set
         *  will be returned. This object MUST NOT be changed!
         */
        this.getStyleAttributeSet = function (styleId) {

            var // the style sheet entry
                styleSheet = styleSheets[this.getEffectiveStyleId(styleId)],
                // the cached style sheet attributes
                mergedAttributes = _.isObject(styleSheet) ? styleSheet.mergedAttributes : null;

            // calculate the merged attributes if not cached yet
            if (!_.isObject(mergedAttributes)) {
                // start with attribute default values
                mergedAttributes = resolveBaseAttributes();
                // extend with attributes of specified style sheet
                docModel.extendAttributes(mergedAttributes, resolveStyleAttributeSet(styleId));
                // cache in style sheet object
                if (_.isObject(styleSheet)) { styleSheet.mergedAttributes = mergedAttributes; }
            }

            return mergedAttributes;
        };

        /**
         * Returns the merged attribute set of the default style sheet of this
         * collection.
         *
         * @returns {Object}
         *  The complete merged attribute set of the default style sheet of
         *  this collection.
         */
        this.getDefaultStyleAttributeSet = function () {
            return this.getStyleAttributeSet(defaultStyleId);
        };

        /**
         * Returns the UI category for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String}
         *  The UI category. If the style sheet does not exist, returns null.
         */
        this.getUICategory = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].category : null;
        };

        /**
         * Returns the UI priority for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Number}
         *  The UI priority. If the style sheet does not exist, returns 0.
         */
        this.getUIPriority = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].priority : 0;
        };

        /**
         * Returns the identifier of the parent style sheet for the specified
         * style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String|Null}
         *  The parent id of the style sheet; or null, if no parent exists.
         */
        this.getParentId = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].parentId : null;
        };

        /**
         * Returns whether the specified style is an auto style.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean|Null}
         *  True, if the specified style is an auto style; false, if the style
         *  is a 'real' style sheet; or null, if the style does not exist.
         */
        this.isAuto = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].auto : null;
        };

        /**
         * Returns whether the specified style sheet is hidden in the user
         * interface.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean|Null}
         *  True, if the specified style sheet is hidden (always true for auto
         *  styles); false, if the style sheet is visible; or null, if the
         *  style sheet does not exist.
         */
        this.isHidden = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].hidden : null;
        };

        /**
         * Returns the user defined name for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String|Null}
         *  The user defined name of the style sheet; or null, if the style
         *  sheet does not exist.
         */
        this.getName = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].name : null;
        };

        /**
         *
         * @param styleId
         * @returns
         */
        this.isCustom = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].custom : null;
        };

        /**
         * Return whether the specified style sheet is dirty.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean|Null}
         *  The dirty state of the style sheet; or null, if the style sheet
         *  does not exist.
         */
        this.isDirty = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].dirty : null;
        };

        /**
         * Changes the dirty state of the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {Boolean} dirty
         */
        this.setDirty = function (styleId, dirty) {
            if (styleId in styleSheets) {
                styleSheets[styleId].dirty = dirty;
            }
        };

        /**
         * Returns the complete 'attributes' object of the specified style
         * sheet as-is, without resolving the parent style sheets, or
         * converting to the attributes of a specific attribute family.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Object}
         *  The complete 'attributes' object contained in the style sheet, as a
         *  deep clone.
         */
        this.getStyleSheetAttributeMap = function (styleId) {
            return (styleId in styleSheets) ? _.copy(styleSheets[styleId].attributes, true) : {};
        };

        /**
         * Builds an attribute map containing all formatting attributes of the
         * own style family, and of the additional attribute families supported
         * by the style sheets in this collection, set to the null value.
         * Additionally, the property 'styleId' set to null will be added to
         * the attribute set.
         *
         * @returns {Object}
         *  An attribute set with null values for all attributes registered for
         *  the supported attribute families.
         */
        this.buildNullAttributes = function () {
            var attributes = docModel.buildNullAttributes(supportedFamilies);
            attributes.styleId = null;
            return attributes;
        };

        /**
         * Returns the merged values of the formatting attributes from a style
         * sheet and explicit attributes.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set containing an optional style sheet
         *  identifier in the property 'styleId', and explicit attributes.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result set.
         *
         * @returns {Object}
         *  A complete merged attribute set with the values of all formatting
         *  attributes supported by this collection.
         */
        this.getMergedAttributes = function (attributes, options) {

            var // the identifier of the style sheet
                styleId = Utils.getStringOption(attributes, 'styleId', null),
                // resulting merged attributes (start with defaults and style sheet attributes)
                mergedAttributes = this.getStyleAttributeSet(styleId);

            // add the passed explicit attributes (protect effective style sheet identifier)
            styleId = mergedAttributes.styleId;
            docModel.extendAttributes(mergedAttributes, attributes, options);
            mergedAttributes.styleId = styleId;

            // filter by supported attributes
            return filterSupportedAttributes(mergedAttributes, options);
        };

        /**
         * Returns the values of the formatting attributes in the specified DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose attributes will be returned. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {HTMLElement|jQuery} [options.sourceNode]
         *      A descendant of the passed element associated to a child
         *      attribute family. Will be passed to a style attribute resolver
         *      callback function where it might be needed to resolve the
         *      correct attributes according to the position of this source
         *      node.
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result map.
         *
         * @returns {Object}
         *  A complete merged attribute set with the values of all formatting
         *  attributes supported by this collection.
         */
        this.getElementAttributes = function (element, options) {

            var // the descendant source node
                sourceNode = Utils.getOption(options, 'sourceNode'),
                // the identifier of the style sheet referred by the element
                styleId = AttributeUtils.getElementStyleId(element),
                // resulting merged attributes (start with defaults, parent element, and style sheet attributes)
                mergedAttributes = resolveBaseAttributes(element);

            // add attributes of the style sheet and its parents
            docModel.extendAttributes(mergedAttributes, resolveStyleAttributeSet(styleId, element, sourceNode));

            // add the explicit attributes of the element (protect effective style sheet identifier)
            styleId = mergedAttributes.styleId;
            docModel.extendAttributes(mergedAttributes, resolveElementAttributeSet(element, sourceNode), options);
            mergedAttributes.styleId = styleId;

            // filter by supported attributes
            return filterSupportedAttributes(mergedAttributes, options);
        };

        /**
         * Changes specific formatting attributes in the specified DOM element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose attributes will be changed. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set with all formatting attributes to be
         *  changed. To clear an explicit attribute value (thus defaulting to
         *  the current style sheet), the respective value in this attribute
         *  set has to be set explicitly to the null value. Attributes missing
         *  in this attribute set will not be modified at the element.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, explicit element attributes that are equal to
         *      the attributes of the current style sheet will be removed from
         *      the element.
         *  @param {Boolean} [options.special=false]
         *      If set to true, allows to change special attributes (attributes
         *      that are marked with the 'special' flag in the attribute
         *      definitions passed to the constructor).
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Function} [options.changeListener]
         *      If specified, will be called if the attributes of the element
         *      have been changed. Will be called in the context of this style
         *      sheet collection. Receives the passed element as first
         *      parameter, the old explicit attribute set as second parameter,
         *      and the new explicit attribute set as third parameter.
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.setElementAttributes = function (element, attributes, options) {

            var // whether to remove element attributes equal to style attributes
                clear = Utils.getBooleanOption(options, 'clear', false),
                // whether a preview element is rendered
                preview = Utils.getBooleanOption(options, 'preview', false),
                // change listener notified for changed attributes
                changeListener = Utils.getFunctionOption(options, 'changeListener'),
                // new style sheet identifier
                styleId = attributes.styleId,

                // the existing explicit element attributes
                oldElementAttributes = AttributeUtils.getExplicitAttributes(element, { direct: true }),
                // new explicit element attributes (clone, there may be multiple elements pointing to the same data object)
                newElementAttributes = _.copy(oldElementAttributes, true),
                // merged attribute values from style sheets and explicit attributes
                mergedAttributes = resolveBaseAttributes(element);

            // add/remove new style sheet identifier, or get current style sheet identifier
            if (_.isString(styleId)) {
                newElementAttributes.styleId = styleId;
            } else if (_.isNull(styleId)) {
                delete newElementAttributes.styleId;
            } else {
                styleId = AttributeUtils.getElementStyleId(element);
            }

            // collect all attributes of the new or current style sheet, and its parents
            docModel.extendAttributes(mergedAttributes, resolveStyleAttributeSet(styleId, element));

            // add or remove the passed explicit attributes
            attributes = _.copy(attributes, true);
            restrictToSupportedFamilies(attributes);
            _.each(attributes, function (attributeValues, family) {

                var // definitions of own attributes
                    definitions = docModel.getAttributeDefinitions(family),
                    // passed attribute values
                    attributeValues = attributes[family],
                    // add an attribute map for the current family
                    elementAttributeValues = null;

                if (_.isObject(attributeValues)) {

                    // add an attribute map for the current family
                    elementAttributeValues = newElementAttributes[family] || (newElementAttributes[family] = {});

                    // update the attribute map with the passed attributes
                    _.each(attributeValues, function (value, name) {
                        if (AttributeUtils.isRegisteredAttribute(definitions, name, options)) {
                            // check whether to clear the attribute
                            if (_.isNull(value) || (clear && _.isEqual(mergedAttributes[family][name], value))) {
                                delete elementAttributeValues[name];
                            } else {
                                elementAttributeValues[name] = value;
                            }
                        }
                    });

                    // remove empty attribute value maps completely
                    if (_.isEmpty(elementAttributeValues)) {
                        delete newElementAttributes[family];
                    }
                }
            });

            // check if any attributes have been changed
            if (preview || !_.isEqual(oldElementAttributes, newElementAttributes)) {

                // write back new explicit attributes to the element
                $(element).data('attributes', newElementAttributes);

                // Do really update the element formatting only if 'preview' is true or after the import of the
                // document is completed.
                if (preview || docModel.getApp().isImportFinished()) {
                    // merge explicit attributes into style attributes, and update element formatting
                    docModel.extendAttributes(mergedAttributes, resolveElementAttributeSet(element), options);
                    updateElementFormatting($(element), mergedAttributes, { preview: preview });
                }

                // call the passed change listener
                if (_.isFunction(changeListener)) {
                    changeListener.call(this, element, oldElementAttributes, newElementAttributes);
                }
            }

            return this;
        };

        /**
         * Changes all given attributes in the style sheet, does not trigger
         * any listener or change anything in the DOM element is in use for
         * attributes: priority, dirty, builtIn.
         *
         * @param styleId {String}
         *
         * @param options {Object}
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.setStyleOptions = function (styleId, options) {
            var styleSheet = styleSheets[styleId];
            if (styleSheet) {
                _.extend(styleSheet, options);
            }
            return this;
        };

        /**
         * Updates the CSS formatting in the specified DOM element, according
         * to its current attribute and style settings.
         *
         * @param {HTMLElement|jQuery} element
         *  The element to be updated. If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.baseAttributes]
         *      An attribute set with all attribute values of the ancestor of
         *      the passed element, merged from style sheet and explicit
         *      attributes, keyed by attribute family. Used internally while
         *      updating the formatting of all child elements of a node.
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the registered format or preview handler will
         *      be asked to execute asynchronously in a browser timeout loop.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  registered format or preview handler has finished. If the
         *  'options.async' options is set, and the format handler supports
         *  asynchronous execution, the Promise will be resolved sometime in
         *  the future. Otherwise, the returned Promise is already resolved
         *  when this method returns.
         */
        this.updateElementFormatting = function (element, options) {

            var // the identifier of the style sheet referred by the element
                styleId = AttributeUtils.getElementStyleId(element),
                // precalculated base attributes passed to this method
                baseAttributes = Utils.getObjectOption(options, 'baseAttributes'),
                // the merged attributes of the passed element
                mergedAttributes = null;

            if (_.isObject(baseAttributes)) {
                mergedAttributes = docModel.getDefaultAttributes(styleFamily);
                docModel.extendAttributes(mergedAttributes, baseAttributes);
            } else {
                mergedAttributes = resolveBaseAttributes(element);
            }

            // add attributes of the style sheet and its parents, and the explicit attributes of the element
            docModel.extendAttributes(mergedAttributes, resolveStyleAttributeSet(styleId, element));
            docModel.extendAttributes(mergedAttributes, resolveElementAttributeSet(element), { special: true });

            // update the formatting of the element
            return updateElementFormatting($(element), mergedAttributes, options);
        };

        /**
         *
         * @param styleId
         * @param possParentStyleId
         * @returns
         */
        this.isChildOfOther = function (styleId, possParentStyleId) {
            if (styleId === possParentStyleId) {
                return true;
            }
            var parentId = self.getParentId(styleId);
            if (parentId) {
                return self.isChildOfOther(parentId, possParentStyleId);
            } else {
                return false;
            }
        };

        // initialization -----------------------------------------------------

        // add additional attribute families passed in the options
        supportedFamilies = Utils.getTokenListOption(initOptions, 'families', []);
        supportedFamilies.push(styleFamily);
        supportedFamilies = _.unique(supportedFamilies);

        // forward other public methods to methods of EditModel
        ['getCssFontFamily', 'resolveColor', 'resolveTextColor', 'getCssColor', 'getCssTextColor', 'getCssBorderAttributes', 'getCssBorder', 'getCssTextDecoration'].forEach(function (method) {
            this[method] = docModel[method].bind(docModel);
        }, this);

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = docModel = initOptions = styleSheets = null;
            parentResolvers = formatHandler = previewHandler = styleAttributesResolver = elementAttributesResolver = null;
        });

    } // class StyleCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: StyleCollection });

});
