/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/editframework/utils/operationerror',
    'io.ox/office/editframework/utils/operationcontext',
    'io.ox/office/spreadsheet/utils/operationcontext',
    'io.ox/office/spreadsheet/model/model'
], function (OperationError, OperationContext, SheetOperationContext, SpreadsheetModel) {

    'use strict';

    // class SheetOperationContext ============================================

    describe('Spreadsheet class SheetOperationContext', function () {

        it('should exist', function () {
            expect(OperationContext).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var // a fake application
            app = new ox.test.DummySpreadsheetApplication(),
            // a document model, needed to convert JSON data to addresses, ranges, etc.
            docModel = new SpreadsheetModel(app),
            // base properties for the test operation that are not convertible to addresses, ranges, etc.
            operation = { name: 'test', num: 1.5, obj: { a: 1 }, arr: [1] },
            // test context for column/row intervals
            intervalContext = new SheetOperationContext(docModel, _.extend({
                interval1: { start: 1, end: 999999 },
                interval2: { start: 1 },
                interval3: { start: 'a' },
                interval4: { start: Number.NaN },
                start: 2
            }, operation), false),
            // test context for cell addresses
            addrContext = new SheetOperationContext(docModel, _.extend({
                address1: [1, 2],
                address2: [999999, 2],
                address3: [null, 2],
                start: [2, 3]
            }, operation), false),
            // test context for cell range addresses
            rangeContext = new SheetOperationContext(docModel, _.extend({
                range1: { start: [1, 2], end: [3, 4] },
                range2: { start: [1, 2] },
                range3: { start: [1, 2], end: [3, null] },
                start: [2, 3]
            }, operation), false),
            // test context for cell range address arrays
            rangeArrayContext = new SheetOperationContext(docModel, _.extend({
                ranges: [{ start: [1, 2] }],
                ranges2: [{ start: [1, 2] }, { start: [2, 3], end: [3, 4] }],
                ranges3: [{ start: [2, 3], end: [3, null] }],
                ranges4: []
            }, operation), false);

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a context', function () {
                var context = new SheetOperationContext(docModel, operation, false);
                expect(context).to.be.an['instanceof'](SheetOperationContext);
                expect(context).to.be.an['instanceof'](OperationContext);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "getInterval"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getInterval');
                expect(OperationContext).to.not.respondTo('getInterval');
            });
            function getInterval(name, columns) { return intervalContext.getInterval.bind(intervalContext, name, columns); }
            it('should return existing properties', function () {
                expect(intervalContext.getInterval('interval1', false)).to.stringifyTo('2:1000000');
                expect(intervalContext.getInterval('interval2', false)).to.stringifyTo('2:2');
                expect(intervalContext.getInterval('interval2', true)).to.stringifyTo('2:2');
                expect(intervalContext.getInterval(true)).to.stringifyTo('3:3');
            });
            it('should throw on missing properties', function () {
                expect(getInterval('wrong', true)).to['throw'](OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getInterval('num', true)).to['throw'](OperationError);
                expect(getInterval('obj', true)).to['throw'](OperationError);
                expect(getInterval('arr', true)).to['throw'](OperationError);
                expect(getInterval('interval1', true)).to['throw'](OperationError);
                expect(getInterval('interval3', true)).to['throw'](OperationError);
                expect(getInterval('interval4', true)).to['throw'](OperationError);
            });
        });

        describe('method "getOptInterval"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptInterval');
                expect(OperationContext).to.not.respondTo('getOptInterval');
            });
            function getOptInterval(name, columns) { return intervalContext.getOptInterval.bind(intervalContext, name, columns); }
            it('should return existing properties', function () {
                expect(intervalContext.getOptInterval('interval1', false)).to.stringifyTo('2:1000000');
                expect(intervalContext.getOptInterval('interval2', false)).to.stringifyTo('2:2');
                expect(intervalContext.getOptInterval('interval2', true)).to.stringifyTo('2:2');
                expect(intervalContext.getOptInterval(true)).to.stringifyTo('3:3');
            });
            it('should return null for missing properties', function () {
                expect(intervalContext.getOptInterval('wrong', true)).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptInterval('num', true)).to['throw'](OperationError);
                expect(getOptInterval('obj', true)).to['throw'](OperationError);
                expect(getOptInterval('arr', true)).to['throw'](OperationError);
                expect(getOptInterval('interval1', true)).to['throw'](OperationError);
                expect(getOptInterval('interval3', true)).to['throw'](OperationError);
                expect(getOptInterval('interval4', true)).to['throw'](OperationError);
            });
        });

        describe('method "getAddress"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getAddress');
                expect(OperationContext).to.not.respondTo('getAddress');
            });
            function getAddress(name) { return addrContext.getAddress.bind(addrContext, name); }
            it('should return existing properties', function () {
                expect(addrContext.getAddress('address1')).to.stringifyTo('B3');
                expect(addrContext.getAddress('start')).to.stringifyTo('C4');
                expect(addrContext.getAddress()).to.stringifyTo('C4');
            });
            it('should throw on missing properties', function () {
                expect(getAddress('wrong')).to['throw'](OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getAddress('num')).to['throw'](OperationError);
                expect(getAddress('obj')).to['throw'](OperationError);
                expect(getAddress('arr')).to['throw'](OperationError);
                expect(getAddress('address2')).to['throw'](OperationError);
                expect(getAddress('address3')).to['throw'](OperationError);
            });
        });

        describe('method "getOptAddress"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptAddress');
                expect(OperationContext).to.not.respondTo('getOptAddress');
            });
            function getOptAddress(name) { return addrContext.getOptAddress.bind(addrContext, name); }
            it('should return existing properties', function () {
                expect(addrContext.getOptAddress('address1')).to.stringifyTo('B3');
                expect(addrContext.getOptAddress('start')).to.stringifyTo('C4');
                expect(addrContext.getOptAddress()).to.stringifyTo('C4');
            });
            it('should return null for missing properties', function () {
                expect(addrContext.getOptAddress('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptAddress('num')).to['throw'](OperationError);
                expect(getOptAddress('obj')).to['throw'](OperationError);
                expect(getOptAddress('arr')).to['throw'](OperationError);
                expect(getOptAddress('address2')).to['throw'](OperationError);
                expect(getOptAddress('address3')).to['throw'](OperationError);
            });
        });

        describe('method "getRange"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getRange');
                expect(OperationContext).to.not.respondTo('getRange');
            });
            function getRange(name) { return rangeContext.getRange.bind(rangeContext, name); }
            it('should return existing properties', function () {
                expect(rangeContext.getRange('range1')).to.stringifyTo('B3:D5');
                expect(rangeContext.getRange('range2')).to.stringifyTo('B3:B3');
                expect(rangeContext.getRange()).to.stringifyTo('C4:C4');
            });
            it('should throw on missing properties', function () {
                expect(getRange('wrong')).to['throw'](OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getRange('num')).to['throw'](OperationError);
                expect(getRange('obj')).to['throw'](OperationError);
                expect(getRange('arr')).to['throw'](OperationError);
                expect(getRange('range3')).to['throw'](OperationError);
            });
        });

        describe('method "getOptRange"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptRange');
                expect(OperationContext).to.not.respondTo('getOptRange');
            });
            function getOptRange(name) { return rangeContext.getOptRange.bind(rangeContext, name); }
            it('should return existing properties', function () {
                expect(rangeContext.getOptRange('range1')).to.stringifyTo('B3:D5');
                expect(rangeContext.getOptRange('range2')).to.stringifyTo('B3:B3');
                expect(rangeContext.getOptRange()).to.stringifyTo('C4:C4');
            });
            it('should return null for missing properties', function () {
                expect(rangeContext.getOptRange('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptRange('num')).to['throw'](OperationError);
                expect(getOptRange('obj')).to['throw'](OperationError);
                expect(getOptRange('arr')).to['throw'](OperationError);
                expect(getOptRange('range3')).to['throw'](OperationError);
            });
        });

        describe('method "getRangeArray"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getRangeArray');
                expect(OperationContext).to.not.respondTo('getRangeArray');
            });
            function getRangeArray(name) { return rangeArrayContext.getRangeArray.bind(rangeArrayContext, name); }
            it('should return existing properties', function () {
                expect(rangeArrayContext.getRangeArray('ranges')).to.stringifyTo('B3:B3');
                expect(rangeArrayContext.getRangeArray('ranges2')).to.stringifyTo('B3:B3,C4:D5');
                expect(rangeArrayContext.getRangeArray()).to.stringifyTo('B3:B3');
                expect(rangeArrayContext.getRangeArray('ranges4')).to.stringifyTo('');
            });
            it('should throw on missing properties', function () {
                expect(getRangeArray('wrong')).to['throw'](OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getRangeArray('num')).to['throw'](OperationError);
                expect(getRangeArray('obj')).to['throw'](OperationError);
                expect(getRangeArray('arr')).to['throw'](OperationError);
                expect(getRangeArray('ranges3')).to['throw'](OperationError);
            });
        });

        describe('method "getOptRangeArray"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptRangeArray');
                expect(OperationContext).to.not.respondTo('getOptRangeArray');
            });
            function getOptRangeArray(name) { return rangeArrayContext.getOptRangeArray.bind(rangeArrayContext, name); }
            it('should return existing properties', function () {
                expect(rangeArrayContext.getOptRangeArray('ranges')).to.stringifyTo('B3:B3');
                expect(rangeArrayContext.getOptRangeArray('ranges2')).to.stringifyTo('B3:B3,C4:D5');
                expect(rangeArrayContext.getOptRangeArray()).to.stringifyTo('B3:B3');
                expect(rangeArrayContext.getOptRangeArray('ranges4')).to.stringifyTo('');
            });
            it('should return null for missing properties', function () {
                expect(rangeContext.getOptRangeArray('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptRangeArray('num')).to['throw'](OperationError);
                expect(getOptRangeArray('obj')).to['throw'](OperationError);
                expect(getOptRangeArray('arr')).to['throw'](OperationError);
                expect(getOptRangeArray('ranges3')).to['throw'](OperationError);
            });
        });
    });

    // ========================================================================
});
