/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/impl/matrixfuncs'
], function (SheetUtils, FormulaUtils, Matrix, MatrixFuncs) {

    'use strict';

    // module MatrixFuncs =====================================================

    describe('Spreadsheet module MatrixFuncs', function () {

        it('should exist', function () {
            expect(MatrixFuncs).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var ErrorCodes = SheetUtils.ErrorCodes;

        function m(values) { return new Matrix(values); }

        function matrixMatcher(expected) {
            expected = m(expected).array;
            return function (result) {
                return _.isEqual(expected, result.array);
            };
        }

        // workaround to test for throwing an error code (seems that expect(...).to.throw()
        // does not work with custom exception values that are not instances of Error)
        function expectErrorCode(expected, func) {
            try {
                var result = func.apply(null, _.toArray(arguments).slice(expectErrorCode.length));
                // error code may be returned instead of thrown
                expect(result).to.equal(expected);
            } catch (ex) {
                expect(ex).to.equal(expected);
            }
        }

        // function implementations -------------------------------------------

        describe('function "MDETERM"', function () {
            var MDETERM = MatrixFuncs.MDETERM;
            it('should exist', function () {
                expect(MDETERM).to.be.an('object');
            });
            it('should NOT be implemented', function () {
                // enforce implementing a unit test for this function
                expect(MDETERM).itself.not.to.respondTo('resolve');
            });
        });

        describe('function "MINVERSE"', function () {
            var MINVERSE = MatrixFuncs.MINVERSE;
            it('should exist', function () {
                expect(MINVERSE).to.be.an('object');
            });
            it('should NOT be implemented', function () {
                // enforce implementing a unit test for this function
                expect(MINVERSE).itself.not.to.respondTo('resolve');
            });
        });

        describe('function "MMULT"', function () {
            var MMULT = MatrixFuncs.MMULT;
            it('should exist', function () {
                expect(MMULT).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(MMULT).itself.to.respondTo('resolve');
            });
            it('should return the matrix product', function () {
                expect(MMULT.resolve(m([[2]]), m([[3]]))).to.satisfy(matrixMatcher([[6]]));
                expect(MMULT.resolve(m([[2]]), m([[3, 4]]))).to.satisfy(matrixMatcher([[6, 8]]));
                expect(MMULT.resolve(m([[2], [3]]), m([[4]]))).to.satisfy(matrixMatcher([[8], [12]]));
                expect(MMULT.resolve(m([[2, 3]]), m([[4], [5]]))).to.satisfy(matrixMatcher([[23]]));
                expect(MMULT.resolve(m([[2], [3]]), m([[4, 5]]))).to.satisfy(matrixMatcher([[8, 10], [12, 15]]));
                expect(MMULT.resolve(m([[2, 3], [4, 5]]), m([[6, 7], [8, 9]]))).to.satisfy(matrixMatcher([[36, 41], [64, 73]]));
            });
            it('should throw the #VALUE! error, if matrix sizes do not match', function () {
                expectErrorCode(ErrorCodes.VALUE, MMULT.resolve, m([[2]]), m([[3], [4]]));
                expectErrorCode(ErrorCodes.VALUE, MMULT.resolve, m([[2, 3]]), m([[4, 5]]));
            });
        });

        describe('function "MUNIT"', function () {
            var MUNIT = MatrixFuncs.MUNIT;
            it('should exist', function () {
                expect(MUNIT).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(MUNIT).itself.to.respondTo('resolve');
            });
            it('should return the unit matrix', function () {
                expect(MUNIT.resolve(1)).to.satisfy(matrixMatcher([[1]]));
                expect(MUNIT.resolve(2)).to.satisfy(matrixMatcher([[1, 0], [0, 1]]));
                expect(MUNIT.resolve(3)).to.satisfy(matrixMatcher([[1, 0, 0], [0, 1, 0], [0, 0, 1]]));
                expect(MUNIT.resolve(4)).to.satisfy(matrixMatcher([[1, 0, 0, 0], [0, 1, 0, 0], [0, 0, 1, 0], [0, 0, 0, 1]]));
            });
            it('should throw the #VALUE! error, if passed dimension is too small', function () {
                expectErrorCode(ErrorCodes.VALUE, MUNIT.resolve, 0);
                expectErrorCode(ErrorCodes.VALUE, MUNIT.resolve, -1);
            });
            it('should throw the UNSUPPORTED error, if passed dimension is too large', function () {
                expectErrorCode(FormulaUtils.UNSUPPORTED_ERROR, MUNIT.resolve, Math.max(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT) + 1);
            });
        });

        describe('function "SUMX2MY2"', function () {
            var SUMX2MY2 = MatrixFuncs.SUMX2MY2;
            it('should exist', function () {
                expect(SUMX2MY2).to.be.an('object');
            });
            it('should NOT be implemented', function () {
                // enforce implementing a unit test for this function
                expect(SUMX2MY2).itself.not.to.respondTo('resolve');
            });
        });

        describe('function "SUMX2PY2"', function () {
            var SUMX2PY2 = MatrixFuncs.SUMX2PY2;
            it('should exist', function () {
                expect(SUMX2PY2).to.be.an('object');
            });
            it('should NOT be implemented', function () {
                // enforce implementing a unit test for this function
                expect(SUMX2PY2).itself.not.to.respondTo('resolve');
            });
        });

        describe('function "SUMXMY2"', function () {
            var SUMXMY2 = MatrixFuncs.SUMXMY2;
            it('should exist', function () {
                expect(SUMXMY2).to.be.an('object');
            });
            it('should NOT be implemented', function () {
                // enforce implementing a unit test for this function
                expect(SUMXMY2).itself.not.to.respondTo('resolve');
            });
        });

        describe('function "TRANSPOSE"', function () {
            var TRANSPOSE = MatrixFuncs.TRANSPOSE;
            it('should exist', function () {
                expect(TRANSPOSE).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(TRANSPOSE).itself.to.respondTo('resolve');
            });
            it('should return the transposed matrix', function () {
                expect(TRANSPOSE.resolve(m([[1]]))).to.satisfy(matrixMatcher([[1]]));
                expect(TRANSPOSE.resolve(m([[1, 2]]))).to.satisfy(matrixMatcher([[1], [2]]));
                expect(TRANSPOSE.resolve(m([[1], [2]]))).to.satisfy(matrixMatcher([[1, 2]]));
                expect(TRANSPOSE.resolve(m([[1, 2], [3, 4]]))).to.satisfy(matrixMatcher([[1, 3], [2, 4]]));
                expect(TRANSPOSE.resolve(m([[1, 2, 3], [4, 5, 6]]))).to.satisfy(matrixMatcher([[1, 4], [2, 5], [3, 6]]));
            });
        });
    });

    // ========================================================================
});
