/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/control/paragraphstylepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/editframework/view/control/stylesheetpicker',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/tk/control/button',
    'io.ox/office/text/view/labels'
], function (Utils, Forms, StyleSheetPicker, ContextMenu, Button, Labels) {

    'use strict';

    // class ParagraphStylePicker =============================================

    var SECTION_FUNCTIONS = 'functions';

    /**
     * A drop-down menu control for paragraph style sheets.
     *
     * @constructor
     *
     * @extends StyleSheetPicker
     */
    function ParagraphStylePicker(docView) {

        var // self reference
            self = this,

            // paragraph style sheets
            paragraphStyles = docView.getDocModel().getStyleCollection('paragraph'),

            // menu for changing styles
            contextMenu = null;

        // base constructor ---------------------------------------------------

        StyleSheetPicker.call(this, docView, 'paragraph', {
            width: 175,
            icon: 'docs-pilcrow',
            tooltip: /*#. tool tip: predefined styles for paragraphs */ Labels.PARAGRAPH_STYLE_LABEL,
            gridColumns: 4,
            i18nModulePath: 'io.ox/office/text/resource/paragraphstylenames',
            smallerVersion: { css: { width: 50 }, hideLabel: true }
        });

        // private methods ----------------------------------------------------

        /**
         * Formats the passed DOM element according to the specified paragraph
         * style sheet.
         */
        function setParagraphFormatting(buttonNode, styleId) {
            var captionNode = Forms.getCaptionNode(buttonNode);
            paragraphStyles.setElementAttributes(captionNode, { styleId: styleId }, { preview: true });
        }

        /**
         * Initializes a new list item according to the paragraph style sheet.
         */
        function createItemHandler(event, buttonNode, styleId, options) {
            //TODO: hacked code for functions like "create new paragraph stylesheet"
            if (options && options.section === SECTION_FUNCTIONS) {
                //no paragraph styling here
                return;
            }
            setParagraphFormatting(buttonNode, styleId);
        }

        /**
         * Updates the drop-down menu button.
         */
        function updateHandler(styleId) {
            var menuButton = self.getMenuButton();

            setParagraphFormatting(menuButton, styleId);
            Forms.getCaptionNode(menuButton).find('>span').css({
                color: '#333'
            });
        }

        //TODO: hacked code for functions like "create new paragraph stylesheet"
        function initSections() {
            self.createMenuSection('', { gridColumns: 4 });
            self.createMenuSection(SECTION_FUNCTIONS, { gridColumns: 1 });
        }

        function clearSelected() {
            self.getMenu().getNode().find('.selected').removeClass('selected');
        }

        function initContextMenu() {

            contextMenu = new ContextMenu(docView, self.getMenu().getNode(), {
                selector: '.section-container:not([data-section="' + SECTION_FUNCTIONS + '"]) a'
            });

            contextMenu.addGroup('paragraph/stylesheet',       new Button({ label: Labels.APPLY_STYLE_LABEL }));
            contextMenu.addGroup('paragraph/changestylesheet', new Button({ label: Labels.CHANGE_STYLE_LABEL }));
            contextMenu.addGroup('paragraph/renamestylesheet', new Button({ label: Labels.RENAME_STYLE_LABEL, custom: true }));
            contextMenu.addGroup('paragraph/deletestylesheet',  new Button({ label: Labels.DELETE_STYLE_LABEL, custom: true }));

            self.getMenu().registerFocusableNodes(contextMenu.getNode());

            contextMenu.on('contextmenu:prepare', function (evt) {

                var buttonNode = $(evt.sourceEvent.target).closest('a');
                var dataValue = buttonNode.attr('data-value');
                var custom = paragraphStyles.isCustom(dataValue);

                contextMenu.iterateGroups(function (group) {
                    var link = group.getNode().find('a');
                    if (custom || !group.getOptions().custom) {
                        group.show();
                        group.setValue(dataValue);
                        link.data('value', dataValue);
                        link.attr('data-value', dataValue);
                    } else {
                        group.hide();
                    }
                });

                clearSelected();
                buttonNode.addClass('selected');

            });

            contextMenu.on('popup:hide', function () {
                clearSelected();
                self.getMenu().getNode().find('a[aria-selected="true"]').parent().addClass('selected');
            });

        }

        // initialization -----------------------------------------------------

        initContextMenu();
        initSections();
        this.getMenu().on('popup:afterclear', initSections);

        // TODO: hacked code for functions like "create new paragraph stylesheet"
        paragraphStyles.insertStyleSheet('\x00paragraph/createstylesheet', Labels.CREATE_STYLE_LABEL, null, null, { category: SECTION_FUNCTIONS, hidden: false, priority: 1, dirty: false });

        this.getMenu().on('create:item', createItemHandler);
        this.registerUpdateHandler(updateHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            self = docView = paragraphStyles = contextMenu = null;
        });

    } // class ParagraphStylePicker

    // exports ================================================================

    // derive this class from class StyleSheetPicker
    return StyleSheetPicker.extend({ constructor: ParagraphStylePicker });

});
