/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/text/pageLayout', [
    'io.ox/office/text/utils/textutils',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/text/utils/operations',
    'io.ox/office/text/dom',
    'io.ox/office/text/position',
    'io.ox/office/text/table',
    'gettext!io.ox/office/text/main'
], function (Utils, Dialogs, TriggerObject, TimerMixin, DrawingFrame, AttributeUtils, Operations, DOM, Position, Table, gt) {

    'use strict';

    // class PageLayout ========================================================

    /**
     * An instance of this class represents a page layout handler in the edited document.
     *
     * Triggers the following events:
     * - 'updateMarginal:leave': This event is triggered after leaving the edit mode of
     *      header and footer AND after the marginal nodes are completely exchanged.
     *      Listeners can rely on the existence of the nodes, that are now in the header
     *      and footer.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {TextModel} model
     *  The text model instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     */
    function PageLayout(model, rootNode) {

        var // self reference
            self = this,
            // the application instance
            app = model.getApp(),
            // the selection instance
            selection = null,
            // the field manager instance
            fieldManager = null,
            // node reference containing content nodes
            $pageContentNode = DOM.getPageContentNode(rootNode),
            // style values for page metrics
            pageStyles,
            drawingStyles,
            characterStyles,
            pageAttributes,
            pageHeight,
            pageMaxHeight,
            pagePaddingLeft,
            pagePaddingRight,
            pagePaddingTop,
            pagePaddingBottom,
            pageWidth,
            footerMargin = 0,
            headerMargin = 0,
            firstPage = false,
            evenOddPages = false,
            // overlay layer element for field tooltip
            $tooltipOverlay = $(),
            // node reference for holding template headers and footers ready for cloning
            $headerFooterPlaceHolder = $('<div>').addClass('header-footer-placeholder').attr('contenteditable', _.browser.WebKit ? 'false' : '').css('display', Utils.SMALL_DEVICE ? 'none' : ''),
            // collection of headers&footers instances
            headerFooterCollection = [],
            // if header/footer is present in document
            marginalActivated = false,
            // page break nodes collection
            pageBreaksCollection = $(),
            // wrapper object of first header in document
            $firstHeaderWrapper = $(),
            // wrapper object of last footer in document
            $lastFooterWrapper = $(),
            // stored height values of different header&footer types
            heightOfDefHeader = 0,
            heightOfDefFooter = 0,
            heightOfFirstHeader = 0,
            heightOfFirstFooter = 0,
            heightOfEvenHeader = 0,
            heightOfEvenFooter = 0,
            // odf min-height values for h&f
            odfFirstHeader = { height: 0, bottom: 0 },
            odfEvenHeader = { height: 0, bottom: 0 },
            odfDefHeader = { height: 0, bottom: 0 },
            odfFirstFooter = { height: 0, top: 0 },
            odfEvenFooter = { height: 0, top: 0 },
            odfDefFooter = { height: 0, top: 0 },
            // pageHeights with different header&footer heights
            firstPageHeight = 0,
            evenPageHeight = 0,
            defPageHeight = 0,
            // store the value of current type of page: first, even, default
            currentPageType = 'default',
            // global variable containing info for fields update
            pageCount = 1,
            // variable used to generate target id, unique for document namespace
            generatedTargetNumber = 100,
            // store reference to header/footer node during creation, to jump into this node after createHeaderFooter operation
            $storedHFnode = null,
            // collection of Id numbers for header/footer targets used in document - NOT the whole ID string!
            targetIdNumbersCollection = [],
            // context menu for header
            contextHeaderMenu,
            // context menu for footer
            contextFooterMenu,
            // Collection of translated labels for marginal context menu
            translations = {
                header_default: gt('Header'),
                footer_default: gt('Footer'),
                header_odd: gt('Odd header'),
                footer_odd: gt('Odd footer'),
                header_first: gt('First header'),
                footer_first: gt('First footer'),
                header_even: gt('Even header'),
                footer_even: gt('Even footer')
            },
            // holds information if remote selection needs to be repaint after page attributes change
            repaintRemoteSelection = false,
            // save deleted nodes when toggling between first, even-odd types headers. In ODT only!
            savedNodesCollection = {},
            // saves state with reduced document paddings (tablets mode)
            reducedPadding = false,
            // promise for generated delayed page break function
            pbPromise = null,
            // constant for delaying execution of page breaks
            PB_DELAY_TIME = 1000,
            // tooltip for user fields in marginal nodes
            $marginalFieldTooltip = null,
            //
            cleanupElements = [];

        rootNode.append($headerFooterPlaceHolder);

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Initial looping through current 'page content DOM' and doing addition of heights of elements to calculate page breaks positions.
         * This function is called only on document loading, and that's why it loops trough all DOM nodes
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.triggerEvent=false]
         *      If set to true, an event will be triggered after the page breaks are inserted completely.
         */
        function initialPageBreaks(options) {

            var // whether an event shall be triggered after all page breaks are included
                triggerEvent = Utils.getBooleanOption(options, 'triggerEvent', false);

            if (Utils.SMALL_DEVICE || model.isDraftMode()) { return; }
            return Utils.takeTime('pageLayout.initialPageBreaks(): ', function () {
                var
                    processingNodes,
                    contentHeight = 0,
                    elementHeightWithoutMB = 0,
                    elementHeight = 0,
                    nodeHelper,
                    zoomFactor,
                    floatingDrawings,
                    fDrawingHeight,
                    fDrawingWidth = 0,
                    accHeightFromPrevDrawing = 0,
                    heightOfRectPartHelper = 0,
                    elementSpan,
                    offsetPosition = [],
                    newlySplittedSpan,
                    markup = '',
                    pageBreak,
                    elMarginBottom,
                    elMarginTop,
                    diffMargin = 0,
                    prevElMarginBottom = 0,
                    prevPageBreaks,
                    lastPageBreak,
                    $lastPageBreakParent,
                    $drawing,
                    drawingHeightAndOffset,
                    $element,
                    iterationPromise = null;

                if (!pageAttributes) {
                    initializePageAttributes();
                }

                // hack for tablets with re-written reduced css padding after document is loaded
                if (app.getView().getContentRootNode().hasClass('small-device')) {
                    pagePaddingLeft = parseFloat($pageContentNode.parent().css('padding-left'));// 18.898;
                    pagePaddingRight = pagePaddingLeft;
                    pageWidth = $pageContentNode.parent().outerWidth();
                } else {
                    pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 0.001);
                    pagePaddingRight = Utils.convertHmmToLength(pageAttributes.page.marginRight, 'px', 0.001);
                    pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 0.001);
                }

                if ($firstHeaderWrapper.length === 0) {
                    toggleContentEditableTrueNodes();
                    updateStartHeaderFooterStyle();
                    markMarginalNodesInsideHeadFoot($headerFooterPlaceHolder.children());

                    $firstHeaderWrapper = self.generateHeaderElementHtml();
                    rootNode.css({ paddingBottom: 0, paddingTop: 0 });
                }

                // check if there are already inserted page breaks for fast display of first pages, and continue after them
                prevPageBreaks = $pageContentNode.find('.page-break');
                if (prevPageBreaks.length > 0) {
                    lastPageBreak = prevPageBreaks.last();
                    $lastPageBreakParent = lastPageBreak.parent();
                    if ($lastPageBreakParent.is('td')) {
                        contentHeight = lastPageBreak.parentsUntil('.pagecontent', '.tb-split-nb').height() - (lastPageBreak.parentsUntil('.pagecontent', '.pb-row').next().position().top - lastPageBreak.parentsUntil('.pagecontent', '.tb-split-nb').position().top);
                        processingNodes = lastPageBreak.parentsUntil('.pagecontent', '.tb-split-nb').nextAll('.p, table');
                    } else if ($lastPageBreakParent.hasClass('p')) {
                        contentHeight = lastPageBreak.next().length ? ($lastPageBreakParent.height() - lastPageBreak.next().position().top) : 0;
                        processingNodes = $lastPageBreakParent.nextAll('.p, table');
                    } else if (lastPageBreak.next().hasClass('manual-page-break')) {
                        contentHeight = lastPageBreak.next().height();
                        processingNodes = lastPageBreak.next().nextAll('.p, table');
                    } else {
                        processingNodes = lastPageBreak.nextAll('.p, table');
                    }
                } else {
                    processingNodes = $pageContentNode.children('.p, table');
                }
                // if first call of initialPageBreaks is aborted or interrupted, update page breaks collection
                if (!pageBreaksCollection || pageBreaksCollection.length === 0) {
                    pageBreaksCollection = prevPageBreaks;
                    pageCount = pageBreaksCollection.length + 1;
                }
                resolvePageTypeAndHeight(processingNodes.first());

                // triggering event, so that current scroll position can be saved
                if (triggerEvent) { model.trigger('pageBreak:before'); }

                iterationPromise = self.iterateArraySliced(processingNodes, function (element, iterator) {
                    //console.warn('pageMaxHeight: ', pageMaxHeight);
                    //console.warn('  -> contentHeight: ', contentHeight);

                    $element = $(element);
                    // processing node with floated drawing(s) inside of it
                    fDrawingHeight = 0;
                    floatingDrawings = $element.find('.drawing.float');
                    if (floatingDrawings.length > 0) {
                        _.each(floatingDrawings, function (drawing) {
                            $drawing = $(drawing);
                            drawingHeightAndOffset = $drawing.outerHeight(true) + ($drawing.position().top / zoomFactor);
                            if ($drawing.outerWidth(true) !== $pageContentNode.width()) {
                                if (drawingHeightAndOffset > fDrawingHeight) {
                                    fDrawingHeight = drawingHeightAndOffset;
                                    fDrawingWidth = $drawing.outerWidth(true);
                                }
                            }
                        });
                    }
                    if (accHeightFromPrevDrawing > fDrawingHeight) {
                        fDrawingHeight = 0;
                    }
                    elementHeightWithoutMB = element.offsetHeight;
                    elMarginBottom = parseInt($element.css('margin-bottom'), 10);
                    elMarginTop = parseInt($element.css('margin-top'), 10);
                    if (elMarginTop > prevElMarginBottom) { // el has margin top that is greater than margin bottom of prev el, include diff in page height
                        diffMargin =  elMarginTop - prevElMarginBottom;
                        elementHeightWithoutMB += diffMargin;
                    }
                    elementHeight = elementHeightWithoutMB + elMarginBottom;
                    prevElMarginBottom = elMarginBottom;
                    if (DOM.isManualPageBreakNode(element)) {
                        if ($element.prev().length > 0 || $element.find('.ms-hardbreak-page').length > 0) {
                            contentHeight = manualInsertionOfPageBreak(element, contentHeight, elementHeight);
                        } else { // first element in document, can't have page break before - can happen on delete all where last is manual pb paragraph
                            $element.removeClass('manual-page-break');
                        }
                    }
                    // if current content height is bellow max page height;
                    if (contentHeight + elementHeight <= pageMaxHeight  && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                        if (!DOM.isManualPageBreakNode(element) || $element.find('.ms-hardbreak-page').length < 1) {
                            contentHeight += elementHeight;
                        }
                        if (accHeightFromPrevDrawing > elementHeight) {
                            accHeightFromPrevDrawing -= elementHeight;
                        } else {
                            heightOfRectPartHelper = accHeightFromPrevDrawing;
                            accHeightFromPrevDrawing = 0;
                        }
                    // for last element that can fit on page we need to omit margin bottom
                    } else if (contentHeight + elementHeightWithoutMB <= pageMaxHeight  && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                        if (!DOM.isManualPageBreakNode(element) || $element.find('.ms-hardbreak-page').length < 1) {
                            contentHeight += elementHeightWithoutMB;
                        }
                        if (accHeightFromPrevDrawing > elementHeightWithoutMB) {
                            accHeightFromPrevDrawing -= elementHeightWithoutMB;
                        } else {
                            heightOfRectPartHelper = accHeightFromPrevDrawing;
                            accHeightFromPrevDrawing = 0;
                        }
                    } else {
                        prevElMarginBottom = 0;
                        accHeightFromPrevDrawing = 0;
                        if ((DOM.isManualPageBreakNode($element.prev()) && DOM.isPageBreakNode($element.prev().children().last()))) {
                            // do nothing
                        } else if ($element.hasClass('p') && ($element.text().length > 10 || $element.find('.drawing.inline').length > 0)) { //&& floatingDrawings.length < 1) {
                            // paragraph split preparation
                            if (accHeightFromPrevDrawing > 0 && fDrawingWidth > 0) {
                                offsetPosition = getOffsetPositionFromElement(element, elementHeight, contentHeight, false, accHeightFromPrevDrawing, fDrawingWidth);
                            } else {
                                offsetPosition = getOffsetPositionFromElement(element, elementHeight, contentHeight, false);
                            }
                            if (offsetPosition && offsetPosition.length > 0) { // if offset position is returned
                                if (offsetPosition[0] === 0) { // if offset is 0, no need for p split, insert pb before element
                                    offsetPosition.shift();
                                    contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                                    if (offsetPosition.length > 0) { contentHeight = 0; }
                                }
                                if (offsetPosition.length > 0) {
                                    elementSpan = $element.children('span').first();
                                    while (elementSpan.text().length < 1 || !(elementSpan.is('span'))) {
                                        elementSpan = elementSpan.next();
                                    }
                                    if (elementSpan && elementSpan.length > 0) { //fix for #32563, if div.p contains only empty spans
                                        _.each(offsetPosition, function (offset) {
                                            var elTextLength = $(elementSpan).text().length;
                                            while (elTextLength > 0 && elTextLength <= offset) {
                                                offset -= elTextLength;
                                                elementSpan = elementSpan.next();
                                                while (!elementSpan.is('span')) { // we might fetch some inline div nodes, like div.tab or div.linebreak
                                                    elementSpan = elementSpan.next();
                                                }
                                                elTextLength = $(elementSpan).text().length;
                                            }
                                            newlySplittedSpan = DOM.splitTextSpan(elementSpan, offset, { append: true });
                                            newlySplittedSpan.addClass('break-above-span');
                                            $element.addClass('contains-pagebreak');
                                            elementSpan = newlySplittedSpan;
                                        });
                                    }
                                    zoomFactor = app.getView().getZoomFactor() / 100;
                                    pageBreak = $();
                                    var arrBreakAboveSpans = $element.find('.break-above-span'),
                                        upperPartDiv = 0,
                                        totalUpperParts = 0,
                                        diffPosTopAndPrev = 0,
                                        elementLeftIndent = parseInt($element.css('margin-left'), 10) + parseInt($element.css('padding-left'), 10); // for indented elements such as lists, or special paragraphs

                                    _.each(arrBreakAboveSpans, function (breakAboveSpan) {
                                        diffPosTopAndPrev += $(pageBreak).outerHeight(true) + upperPartDiv;
                                        upperPartDiv = $(breakAboveSpan).position().top / zoomFactor - diffPosTopAndPrev;
                                        totalUpperParts += upperPartDiv;
                                        contentHeight += upperPartDiv;
                                        nodeHelper = $element.prev();
                                        while (nodeHelper.hasClass('page-break')) {
                                            nodeHelper = nodeHelper.prev();
                                        }
                                        markup = generatePagebreakElementHtml(contentHeight, _.browser.WebKit, elementLeftIndent);
                                        $(breakAboveSpan).before(markup);
                                        pageBreak = $(breakAboveSpan).prev();
                                        contentHeight = 0;
                                        if (_.browser.Chrome && parseInt($element.css('text-indent')) < 0) { // #40776
                                            pageBreak.css('display', 'inline-block');
                                        }
                                    });
                                    contentHeight = elementHeight - totalUpperParts;
                                }
                            } else {
                                contentHeight = insertPageBreaksBetweenDrawings(element, elementHeight, contentHeight, offsetPosition, zoomFactor);
                                //contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                            }

                        } else if (DOM.isTableNode(element) && element.rows.length > 1) { // table split
                            contentHeight = insertPageBreaksInTable(element, elementHeight, contentHeight, { initial: true, diffMargin: diffMargin });
                        } else { // default insertion of page break between elements
                            contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                        }
                    }
                    if (DOM.isNodeWithPageBreakAfterAttribute(element)) {
                        contentHeight = insertManualPageBreakAfter(element, contentHeight);
                    }

                    if (fDrawingHeight > 0) { // store the diff between floated image height and height of containing p, for next iteration
                        accHeightFromPrevDrawing = fDrawingHeight - elementHeight;
                    }
                    //at the last page, we add extra padding to fill in to the size of max page height
                    if (processingNodes.length === iterator + 1) {
                        $pageContentNode.css({
                            'padding-bottom': pageMaxHeight - contentHeight
                        });
                    }

                }, { delay: 'immediate', infoString: 'Text: pageLayout: initialPageBreaks' })
                .done(function () {
                    if ($lastFooterWrapper.length === 0) {
                        $lastFooterWrapper = self.generateFooterElementHtml();
                    }

                    pageBreaksCollection = $pageContentNode.find('.page-break');
                    // store indexes of page breaks as values for page numbers
                    _.each(pageBreaksCollection, function (pageBreakInCollection, index) {
                        $(pageBreakInCollection).data('page-num', index + 1);
                    });
                    pageCount = pageBreaksCollection.length + 1;

                    populateHeaderFooterPaginationClasses(pageBreaksCollection);
                    if (self.isHeaderFooterInDocument()) {
                        //replaceAllTypesOfHeaderFooters();
                        repairImagesAndTfBordersInMarginal();
                    }
                    if (!fieldManager.isSimpleFieldEmpty()) {
                        updatePageCountFields();
                        updatePageNumberFields();
                    }
                    if (!fieldManager.isComplexFieldEmpty()) {
                        fieldManager.updatePageFieldsInMarginals();
                    }

                    _.each(cleanupElements, function (element) {
                        $(element).remove();
                    });
                    cleanupElements = [];

                    if (triggerEvent) {
                        model.trigger('pageBreak:after');
                    }
                    if (model.isLocalStorageImport()) {
                        // if page breaks are called from local storage, trigger new event, that will be listened for fields update
                        // new event is needed to avoid calling of update of drawings and comment layer that are called after 'pageBreak:after'
                        model.trigger('initialPageBreaks:done');
                    }
                });
                return iterationPromise.promise();
            });
        }

        /**
         * Depending on current page number and presence of different types of header/footer,
         * resolve type and height of current page.
         *
         * @param{jQuery} $firstNode
         *  Node from which page breaks insertion starts.
         *
         */
        function resolvePageTypeAndHeight($firstNode) {
            var currentPageNumber = self.getPageNumber($firstNode);

            if (currentPageNumber === 1 && firstPage && firstPageHeight > 0) {
                pageMaxHeight = firstPageHeight;
                currentPageType = 'first';
            } else if (currentPageNumber % 2 === 0 && evenOddPages && evenPageHeight > 0) {
                pageMaxHeight = evenPageHeight;
                currentPageType = 'even';
            } else if (defPageHeight > 0) {
                pageMaxHeight = defPageHeight;
                currentPageType = 'default';
            } else {
                Utils.warn('pageLayout.initialPageBreaks(): default page height is wrong: ', defPageHeight);
            }
        }

        /**
         * After inserting of page breaks is done, this method will insert correct types of headers&footers
         * inside page breaks. There are four types of headers&footers, listed by priority from highest to lowest:
         *  first   - for the first page only, if defined
         *  even    - for even pages, if defined
         *  default - fallback for all types - lowest prio. Odd pages uses this type
         * Highest priority header/footer, if is defined in document, will overwrite all bellow him.
         */
        function replaceAllTypesOfHeaderFooters() {
            var groupOfHeaderFooterToUpdate = pageBreaksCollection.add($firstHeaderWrapper).add($lastFooterWrapper), //rootNode.find('.page-break, .header-wrapper, .footer-wrapper'),
                placeholderHeaderChildren = self.getHeaderFooterChildrenFromPlaceholder('header_default'),
                placeholderHeaderTarget = self.getIdFromType('header_default'),
                placeholderFooterChildren = self.getHeaderFooterChildrenFromPlaceholder('footer_default'),
                placeholderFooterTarget = self.getIdFromType('footer_default'),
                typesOfHeader = [],
                typesOfHeaderInternal = [],
                typesOfFooter = [],
                typesOfFooterInternal = [],
                selector = '';

            if (firstPage) {
                typesOfHeader.push('header_first');
                typesOfHeaderInternal.push('.header.first');
                typesOfFooter.push('footer_first');
                typesOfFooterInternal.push('.footer.first');
                selector += '.first';
            }
            if (evenOddPages) {
                typesOfHeader.push('header_even');
                typesOfHeaderInternal.push('.header.even');
                typesOfFooter.push('footer_even');
                typesOfFooterInternal.push('.footer.even');
                if (selector.length > 0) {
                    selector += ', .even';
                } else {
                    selector += '.even';
                }
            }

            // default
            if (placeholderHeaderChildren.length > 0) {
                groupOfHeaderFooterToUpdate.find('.header').not(selector).attr('data-container-id', placeholderHeaderTarget).empty().append(placeholderHeaderChildren);
            } else {
                groupOfHeaderFooterToUpdate.find('.header').not(selector).removeAttr('data-container-id').empty();
            }
            if (placeholderFooterChildren.length > 0) {
                groupOfHeaderFooterToUpdate.find('.footer').not(selector).attr('data-container-id', placeholderFooterTarget).empty().append(placeholderFooterChildren);
            } else {
                groupOfHeaderFooterToUpdate.find('.footer').not(selector).removeAttr('data-container-id').empty();
            }

            // all other types
            for (var i = 0, typesLength = typesOfHeader.length; i < typesLength; i += 1) {
                placeholderHeaderChildren = self.getHeaderFooterChildrenFromPlaceholder(typesOfHeader[i]);
                placeholderHeaderTarget = self.getIdFromType(typesOfHeader[i]);
                if (placeholderHeaderChildren.length > 0) {
                    groupOfHeaderFooterToUpdate.find(typesOfHeaderInternal[i]).attr('data-container-id', placeholderHeaderTarget).empty().append(placeholderHeaderChildren);
                }
            }

            for (var i = 0, typesLength = typesOfFooter.length; i < typesLength; i += 1) {
                placeholderFooterChildren = self.getHeaderFooterChildrenFromPlaceholder(typesOfFooter[i]);
                placeholderFooterTarget = self.getIdFromType(typesOfFooter[i]);
                if (placeholderFooterChildren.length > 0) {
                    groupOfHeaderFooterToUpdate.find(typesOfFooterInternal[i]).attr('data-container-id', placeholderFooterTarget).empty().append(placeholderFooterChildren);
                }
            }

            // if there are images inside head/foot, might happen that they are replaced before image resource is loaded in browser
            repairImagesAndTfBordersInMarginal(groupOfHeaderFooterToUpdate.find('.header, .footer'));

            if (!fieldManager.isComplexFieldEmpty()) {
                fieldManager.updatePageFieldsInMarginals();
            }
        }

        /**
         * Repairs images inside marginals, for ones that are replaced before image resource is loaded in browser,
         * and therefore image placeholder is shown. It also repairs canvas borders of text frames in marginals.
         *
         * @param {jQuery} [allHeadersFooters]
         *  Collection of all header and footer nodes in document.
         *
         */
        function repairImagesAndTfBordersInMarginal(allHeadersFooters) {
            var placeholdersOfImages = null,
                textFramesWithBordersInMarginal = null;

            // it is important to assign values only after all replacement is done
            allHeadersFooters = allHeadersFooters || pageBreaksCollection.add($firstHeaderWrapper).add($lastFooterWrapper).find('.header, .footer');
            // empty each placeholder inside image in header/footer, to trigger update formatting (and, replacing with real image resource)
            placeholdersOfImages = allHeadersFooters.find('.drawing').children('.placeholder');
            _.each(placeholdersOfImages, function (placeholder) {
                $(placeholder).empty();
                drawingStyles.updateElementFormatting($(placeholder).parent('.drawing'));
            });
            // remove references for GC
            placeholdersOfImages = null;

            // find text frames with canvas border inside header/footer and repaint them (after cloning they have to be manually repainted) #36576
            textFramesWithBordersInMarginal = allHeadersFooters.find(DrawingFrame.NODE_SELECTOR);
            if (textFramesWithBordersInMarginal.length) {
                model.trigger('drawingHeight:update', textFramesWithBordersInMarginal);
                textFramesWithBordersInMarginal = null;
            }
        }

        /**
         * Initializing page attributes needed for calculation of page breaks.
         * In some scenarios, like i.e. loading from local storage,
         * initialPageBreaks method is not called, and variables are not initialized for later usage.
         *
         */
        function initializePageAttributes() {
            pageStyles = model.getStyleCollection('page');
            drawingStyles = model.getStyleCollection('drawing');
            characterStyles = model.getStyleCollection('character');
            pageAttributes = pageStyles.getElementAttributes(rootNode);
            pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 0.001);
            pagePaddingRight = Utils.convertHmmToLength(pageAttributes.page.marginRight, 'px', 0.001);
            pagePaddingTop = Utils.convertHmmToLength(pageAttributes.page.marginTop, 'px', 0.001);
            pagePaddingBottom = Utils.convertHmmToLength(pageAttributes.page.marginBottom, 'px', 0.001);
            pageHeight = Utils.convertHmmToLength(pageAttributes.page.height, 'px', 0.001);
            pageMaxHeight = pageHeight - pagePaddingTop - pagePaddingBottom;
            pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 0.001);
            headerMargin = app.isODF() ? pagePaddingTop : Utils.convertHmmToLength(pageAttributes.page.marginHeader, 'px', 0.001);
            footerMargin = app.isODF() ? pagePaddingBottom : Utils.convertHmmToLength(pageAttributes.page.marginFooter, 'px', 0.001);
            firstPage = pageAttributes.page.firstPage;
            evenOddPages = pageAttributes.page.evenOddPages;
        }

        /**
         * In a element (paragraph) where there are drawing objects, but no text spans to split,
         * we insert page breaks directly before desired drawing.
         * For now, only inline type drawings are supported.
         *
         * @param {Node|jQuery} element
         *  Element which is being processed
         *
         * @param {Number} elementHeight
         *  Current height value of all processed nodes
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Array} offsetPosition
         *  possible position(s) where to split text in paragraph
         *
         * @param {Number} zoomFactor
         *  Current zoom factor in document, can be number value, or undefined on doc loading
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after processing, in px
         *
         */
        function insertPageBreaksBetweenDrawings(element, elementHeight, contentHeight, offsetPosition, zoomFactor) {
            var
                $element = $(element),
                $elDrawings = $element.find('.drawing'),
                numOfPageBreaks = ~~((elementHeight + contentHeight) / pageMaxHeight), //get int value how many times is element bigger than page
                multiplicator,
                upperPartDiv = 0,
                totalUpperParts = 0,
                diffPosTopAndPrev = 0,
                numOfInserts = 0,
                pageBreak = $(),
                elementLeftIndent = parseInt($element.css('margin-left'), 10) + parseInt($element.css('padding-left'), 10); // for indented elements such as lists, or special paragraphs

            if ($elDrawings.length > 0 && offsetPosition.length < numOfPageBreaks) {
                multiplicator = numOfPageBreaks;
                multiplicator -= 1; // decrement multiplier for 0 based values

                Utils.iterateArray($elDrawings, function (drawing) {
                    if (numOfInserts === numOfPageBreaks) {
                        return Utils.BREAK;
                    }
                    if ($(drawing).hasClass('inline')) {
                        diffPosTopAndPrev += $(pageBreak).outerHeight(true) + upperPartDiv;
                        upperPartDiv = $(drawing).position().top / (zoomFactor || 1) - diffPosTopAndPrev;

                        if (upperPartDiv + $(drawing).outerHeight() > (pageMaxHeight - contentHeight) && multiplicator > -1) {
                            totalUpperParts += upperPartDiv;
                            contentHeight += upperPartDiv;
                            $(drawing).before(generatePagebreakElementHtml(contentHeight, _.browser.WebKit, elementLeftIndent));
                            multiplicator -= 1;
                            numOfInserts += 1;
                            pageBreak = $(drawing).prev();
                            contentHeight = 0;
                            $element.addClass('contains-pagebreak');
                            $(drawing).addClass('break-above-drawing');
                        }
                    }
                });
                contentHeight = elementHeight - totalUpperParts;
            } else {
                contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
            }

            return contentHeight;
        }

        /**
         *  Internal function for page layout rendering!
         *  Creates invisible helper node to calculate position(s) in text where paragraph should be split.
         *  Also stores position of all rest new line beginnings into node with jQuery data()
         *  Words and spaces are occurring in pairs, sequentially. Space can be one character, or more grouped, then nbsp is used
         *
         * @param {Node|jQuery} element
         *  Element whose offset is calculated
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Boolean} modifyingElement
         *  If the current processing node is modified or untouched
         *
         * @param {Number} partHeight - optional
         *  If there is floating drawing from previous paragraphs, use height to create dummy div
         *
         * @param {Number} partWidth - optional
         *  If there is floating drawing from previous paragraphs, use width to create dummy div
         *
         * @returns {Array} offsetPosition
         *  Calculated position(s) in text where paragraph should be split
         *
         */
        function getOffsetPositionFromElement(element, elementHeight, contentHeight, modifyingElement, partHeight, partWidth) {

            var
                offsetPosition = [],
                elementSpan,
                contentHeightCached = contentHeight,
                $newElement,
                $element = $(element),
                elementWidth = $element.width(),
                elementLineHeightStyle = $element.css('line-height'),
                oxoPosInPar,
                numOfPageBreaks = 0,
                cachedDataForLineBreaks = [],
                cachedDataForLineBreaksLength,
                helperOffsetArray = [],
                elSpanPositionTop,
                charLength = 0,
                //elSpans,
                $elDrawings, //paragraph contains image(s)
                //drawingTopPos,
                //drawingBottomPos,
                cachedPosTop,
                markup,
                splitPosForbidenRanges = [],
                allowedValueInRange;

            function calculateOffsetPosition() {
                var diff = 0,
                    factorIncr = 1,
                    prevPositionTop,
                    prevOxoPos,
                    tempContHeight = contentHeightCached;

                helperOffsetArray = [];

                if (elementHeight > pageMaxHeight) { // more than one insertion of pb
                    numOfPageBreaks = ~~((elementHeight + contentHeightCached) / pageMaxHeight); // how many times we insert page breaks in table
                    if (offsetPosition.length === 0 && cachedDataForLineBreaksLength > 0) {
                        Utils.iterateArray(cachedDataForLineBreaks, function (el, iterator) {
                            if (el.distanceTop + diff > (pageMaxHeight * factorIncr) - tempContHeight && (prevPositionTop && prevPositionTop <= (pageMaxHeight * factorIncr) - tempContHeight)) {
                                diff += ((pageMaxHeight * factorIncr) - tempContHeight) - prevPositionTop; // difference that will be shifted down (image for ex.)
                                factorIncr += 1;
                                offsetPosition.push(prevOxoPos);
                            } else if (iterator === 0 && (el.distanceTop + diff > (pageMaxHeight * factorIncr) - tempContHeight)) {// first row cannot fit into free space, it is also shifted down (pb on begining, pos 0)
                                offsetPosition.push(0);
                                tempContHeight = 0;
                            }
                            prevPositionTop = el.distanceTop + diff;
                            prevOxoPos = el.oxoPosInPar;
                        });
                        if (offsetPosition.length < numOfPageBreaks) {
                            // now we know space left is not enough for last line, so we put page break before it
                            offsetPosition.push(cachedDataForLineBreaks[cachedDataForLineBreaksLength - 1].oxoPosInPar);
                        }
                    }

                    // normalize values: ex. [150, 400] to [150, 250]
                    _.each(offsetPosition, function (offPos, iterator) {
                        if (iterator > 0) {
                            offPos -=  offsetPosition[iterator - 1];
                        }
                        helperOffsetArray.push(offPos);
                    });
                    offsetPosition = helperOffsetArray.slice();
                } else {
                    //if its not multi paragraph split
                    if (offsetPosition.length === 0 && cachedDataForLineBreaksLength > 0) {
                        Utils.iterateArray(cachedDataForLineBreaks, function (el, iterator) {
                            if (el.distanceTop <= (pageMaxHeight - contentHeightCached)) {
                                offsetPosition.push(el.oxoPosInPar);
                                return Utils.BREAK;
                            } else if (iterator === 0 && (el.distanceTop > (pageMaxHeight - contentHeightCached))) {// first row (backwards) cannot fit into free space, it is also shifted down (pb on begining, pos 0)
                                offsetPosition.push(0);
                                return Utils.BREAK;
                            }
                        }, { reverse: true });
                    }
                    if (offsetPosition.length === 0) {
                        // now we know space left is not enough for last line, so we put page break before it
                        offsetPosition.push(cachedDataForLineBreaks[cachedDataForLineBreaksLength - 1].oxoPosInPar);
                    }
                }
            }

            function wrapWordsWithSpans(elementSpan) {
                var elText,
                    words,
                    markup,
                    spaces,
                    charAtBegining;

                _.each(elementSpan, function (el, index) {
                    elText = $(el).text();
                    words = elText.match(/\S+/g);
                    markup = '';
                    spaces = elText.match(/(\s+)/gi);

                    charAtBegining = (elText[0]) ? elText[0].match(/\S+/g) : null; // check if text begins with characters or whitespace

                    if (words && words.length > 0) {
                        _.each(words, function (word, i) {
                            if (word.length > 0) {
                                var generatedSpace;
                                if (spaces && spaces[i]) {
                                    if (spaces[i].length > 1) {
                                        generatedSpace = '';
                                        for (var j = 0; j < spaces[i].length; j++) { //  translates for ex. '     ' into -> ' &nbsp; &nbsp; ' (5 whitespaces)
                                            if (j % 2 !== 0) {
                                                generatedSpace += ' ';
                                            } else {
                                                generatedSpace += '&nbsp;';
                                            }
                                        }
                                    } else {
                                        if (index === 0 && i === 0 && !charAtBegining) { // fix for first span with whitespace (it will render as 0px span, so we fake content with dot)
                                            generatedSpace = '.';
                                        } else {
                                            generatedSpace = ' ';
                                        }
                                    }
                                } else {
                                    generatedSpace = '';
                                }
                                if (charAtBegining) { // word, space direction
                                    if (generatedSpace.length > 0) {
                                        markup += '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span><span class="whitespaceData" data-charLength="' +  (charLength += spaces[i].length) + '">' + generatedSpace + '</span>';
                                    } else {
                                        markup += '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>';
                                    }
                                } else { // space, word direction
                                    if (generatedSpace.length > 0) {
                                        markup += '<span class="whitespaceData" data-charLength="' + (charLength += spaces[i].length) + '">' + generatedSpace + '</span><span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>';
                                    } else {
                                        markup += '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>';
                                    }
                                }
                            }
                        });
                        //$(el).empty().append(markup);
                        el.innerHTML = markup;
                    }
                });
            }

            if (_.isEmpty($element.data('lineBreaksData')) || modifyingElement) { // if no cached data, or element is modified, calculate all positions
                $newElement = $element.clone(true);
                if (partHeight > 0 && partWidth > 0) { // if there is drawing leftover from previous paragraphs, it will change p layout
                    markup = '<div style="width:' + partWidth + 'px;height:' + partHeight + 'px;float:left;"></div>';
                    $newElement.prepend(markup);
                }
                $newElement.css({ width: elementWidth, lineHeight: elementLineHeightStyle });
                Utils.insertHiddenNodes($newElement, true);

                // Handling of drawings - 2 groups: inline and floated.
                // inline are treated normally. can split before and after them
                // floating needs to fetch div offset also
                // there has to be range of values inside range of [0, paragraph height] for which splitting text is not valid
                // range itself can be made of ranges, ex. [[0, 25], [166, 333], [400, 550]]
                // Exception: floated drawing at the beginning of doc, in first p, before text
                $elDrawings = $newElement.find('.drawing');

                _.each($elDrawings, function (drawing) {
                    var $drawing = $(drawing),
                        range = [];
                    range[0] = $drawing.position().top;
                    range[1] = range[0] + ($drawing.height() + Utils.getElementCssLength($drawing, 'margin-bottom'));
                    if ($drawing.hasClass('float')) {
                        var diff = $drawing.prev('.offset').outerHeight(true);
                        range[0] = Math.max(0, range[0] - diff);
                    }

                    splitPosForbidenRanges.push(range);
                });

                elementSpan = $newElement.children('span');
                wrapWordsWithSpans(elementSpan);

                cachedPosTop = 2; // ignoring first line, 2 is rounding error for position of spans in first line
                _.each(elementSpan.find('.textData'), function (elSpan) {
                    var $elSpan = $(elSpan),
                        elSpanTextLength = $elSpan.text().length;

                    if (elSpanTextLength) { //if (Utils.getDomNode(elSpan).firstChild)
                        elSpanPositionTop = $elSpan.position().top;
                        if ((elSpanPositionTop > cachedPosTop)) {
                            cachedPosTop = elSpanPositionTop;
                            if (splitPosForbidenRanges.length > 0) {
                                allowedValueInRange = true;
                                _.each(splitPosForbidenRanges, function (range) {
                                    if (elSpanPositionTop > range[0] && elSpanPositionTop < range[1]) {
                                        allowedValueInRange = false;
                                    }
                                });
                                if (allowedValueInRange) {
                                    oxoPosInPar = parseInt($elSpan.attr('data-charLength'), 10) - elSpanTextLength;
                                    cachedDataForLineBreaks.push({ oxoPosInPar: oxoPosInPar, distanceTop: elSpanPositionTop });
                                }
                            } else {
                                oxoPosInPar = parseInt($elSpan.attr('data-charLength'), 10) - elSpanTextLength;
                                cachedDataForLineBreaks.push({ oxoPosInPar: oxoPosInPar, distanceTop: elSpanPositionTop });
                            }
                        }
                    }
                });
                $element.data('lineBreaksData', cachedDataForLineBreaks);
                //clear dom after getting position
                cleanupElements.push($newElement);
                //$newElement.remove();
                //$newElement = null;
            } else {
                cachedDataForLineBreaks = $element.data('lineBreaksData');
            }

            cachedDataForLineBreaksLength = cachedDataForLineBreaks.length;
            if (cachedDataForLineBreaksLength > 0) {
                calculateOffsetPosition();
            }

            return offsetPosition;
        }

        /**
         * Internal function for page layout rendering!
         * Default way for inserting page break node in DOM,
         * in between first level nodes, paragraphs and tables
         *
         * @param {Node|jQuery} element
         *  Element which is being proccessed
         *
         * @param {Number} elementHeight
         *  Current height value of all proccessed nodes
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after proccessing, in px
         *
         */
        function defaultInsertionOfPageBreak(element, elementHeight, contentHeight) {
            var
                $element = $(element),
                nodeHelper = $element.prev();

            while (nodeHelper.hasClass('page-break')) {
                nodeHelper = nodeHelper.prev();
            }
            if (nodeHelper.length) {
                $element.before(generatePagebreakElementHtml(contentHeight, _.browser.WebKit));
            }
            $element.addClass('break-above'); // this is marker for first node bellow pagebreak
            contentHeight = elementHeight; // addition if content is shifted down

            return contentHeight;
        }

        /**
         * Internal function for page layout rendering!
         * Proccesses passed table element node, and appends generated page break in a new row.
         *
         * @param {Node|jQuery} element
         *  Element which is being proccessed
         *
         * @param {Number} elementHeight
         *  Current height value of all proccessed nodes
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.initial=false]
         *      Determines if function is called initialy after document loading.
         *  @param {Number} [options.diffMargin=0]
         *      Difference in margin-top from current element, and margin-bottom from previous,
         *      needs to be stored in accumulator, for more precise calculation.
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after proccessing
         *
         */
        function insertPageBreaksInTable(element, elementHeight, contentHeight, options) {
            var
                $element = $(element),
                // if there is difference in margin top of current element, and margin bottom of previous, add this diff as start value to accumulator
                accumulator = (options && options.diffMargin) ? options.diffMargin : 0,
                totalAcc = 0,
                arrAccumulator = [],
                // helper var for big tables
                tempContentHeight = contentHeight,
                tableRows = element.rows,
                breakAboveRows,
                // helper height value, for tables bigger than one page
                tableHeight = elementHeight,
                // height of current row inside table (used for iteration)
                trOffsetHeight,
                // option flag to check is it called during initial page break rendering
                initial = Utils.getBooleanOption(options, 'initial', false),
                //summed value of all rows above last page break
                summedAttrPBHeight = 0,
                // generated string with page break node
                markup,
                // IE needs 2px more to left for split tables, Firefox 1px - due to css border-collapse
                negativeMarginTablesFix = _.browser.IE ? 2 : (_.browser.Firefox ? 1 : 0);

            _.each(tableRows, function (tr) {
                trOffsetHeight = tr.offsetHeight;
                if (accumulator + trOffsetHeight < pageMaxHeight - tempContentHeight) {
                    accumulator += trOffsetHeight;
                } else {
                    $(tr).addClass('break-above-tr');
                    $element.addClass('tb-split-nb');
                    totalAcc += accumulator;
                    arrAccumulator.push(totalAcc);
                    tempContentHeight = trOffsetHeight;
                    totalAcc = tempContentHeight;
                    accumulator = 0;
                }
            });

            if ($(tableRows[0]).hasClass('break-above-tr')) {
                $(tableRows[0]).removeClass('break-above-tr');
                if ($element.prev().length > 0) {
                    markup = generatePagebreakElementHtml(contentHeight, _.browser.WebKit);
                    $element.before(markup);
                }
                $element.addClass('break-above'); // this is marker for first node bellow pagebreak
                contentHeight = 0;
                arrAccumulator.shift();
            }

            breakAboveRows = $element.find('.break-above-tr');
            if (breakAboveRows.length > 0) {
                _.each(breakAboveRows, function (breakAboveRow, i) {
                    contentHeight += arrAccumulator[i];
                    markup = generateTableRowPagebreakHTML(contentHeight, negativeMarginTablesFix);
                    $(breakAboveRow).before(markup);
                    if (tableHeight > pageMaxHeight) {
                        contentHeight = 0;
                    } else {
                        contentHeight = tableHeight - arrAccumulator[i];
                    }
                    tableHeight = tableHeight - arrAccumulator[i];
                    summedAttrPBHeight += arrAccumulator[i];
                });
                contentHeight = elementHeight - summedAttrPBHeight; // height of part of table transferred to new page
            } else {
                contentHeight = elementHeight;
                $element.removeClass('tb-split-nb');
            }
            if (!initial && _.browser.Firefox) { Table.forceTableRendering(element); } // forcing Firefox to re-render table (32461)

            return contentHeight;
        }

        /**
         * In ineditable mode, tables have to have all elements with contenteditable set to false.
         * Collection saves all nodes for later restoration, in editmode.
         *
         */
        function toggleContentEditableTrueNodes() {
            $headerFooterPlaceHolder.find('[contenteditable="true"]').attr('contenteditable', false);
        }

        /**
         * Internal helper function that marks all children of headerfooter node with marginal class.
         * This helps with evaluating changeTrack nodes.
         *
         * @param {jQuery} $node
         *  container node whose children are marked
         *
         */
        function markMarginalNodesInsideHeadFoot($node) {
            DOM.getMarginalContentNode($node).children('.p, table').addClass('marginal').find('table, tr, td, div, span').addClass('marginal');
        }

        /**
         * Internal function  for page layout rendering!
         * Generating html markup string for Header to insert into page break html markup.
         *
         * @returns {String}
         *  Div header html markup
         */
        function getHeaderHtmlContent(type) {
            var
                type = type ? (' ' + type) : '',
                currentHeight = 0,
                minHeight,
                distanceBottom = 0;

            if (app.isODF()) {
                currentHeight = (type === ' first' && firstPage) ? odfFirstHeader.height : (currentPageType === 'even' ? odfEvenHeader.height : (odfDefHeader.height ? odfDefHeader.height : 0));
                distanceBottom = (type === ' first' && firstPage) ? odfFirstHeader.bottom : (currentPageType === 'even' ? odfEvenHeader.bottom : (odfDefHeader.bottom ? odfDefHeader.bottom : 0));
            }
            minHeight = Math.max(currentHeight + pagePaddingTop, pagePaddingTop);

            if (type === ' first') {
                if (firstPage && firstPageHeight > 0) {
                    currentPageType = 'first';
                } else if (defPageHeight > 0) {
                    currentPageType = 'default';
                }
            } else {
                switchPageTypeAndHeight();
            }

            if (self.isHeaderFooterInDocument() && $headerFooterPlaceHolder.children('.header_' + currentPageType).length) {
                return $headerFooterPlaceHolder.children('.header_' + currentPageType).css({ minHeight: minHeight, paddingTop: headerMargin, paddingRight: pagePaddingRight, paddingBottom: distanceBottom, paddingLeft: pagePaddingLeft }).clone(true);
            } else {
                return '<div class="header inactive-selection' + type + '" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + ' style="min-height: ' + minHeight +
                    'px; padding: ' + headerMargin + 'px ' + pagePaddingRight + 'px ' + distanceBottom + 'px ' + (pagePaddingLeft) + 'px;" data-focus-role="header">' +
                    '<div class="cover-overlay" contenteditable="false"></div><div class="marginalcontent"></div></div>';
            }
        }

        /**
         * Internal function  for page layout rendering!
         * Generating html markup string for Footer to insert into page break html markup.
         *
         * @returns {String}
         *  Div header html markup
         */
        function getFooterHtmlContent(type) {
            var
                type = type || '',
                currentHeight = 0,
                minHeight,
                distanceTop = 0;

            if (app.isODF()) {
                currentHeight = (currentPageType === 'first') ? odfFirstFooter.height : (currentPageType === 'even' ? odfEvenFooter.height : (odfDefFooter.height ? odfDefFooter.height : 0));
                distanceTop = (currentPageType === 'first') ? odfFirstFooter.top : (currentPageType === 'even' ? odfEvenFooter.top : (odfDefFooter.top ? odfDefFooter.top : 0));
            }
            minHeight = Math.max(currentHeight + pagePaddingBottom, pagePaddingBottom);

            if (self.isHeaderFooterInDocument() && $headerFooterPlaceHolder.children('.footer_' + currentPageType).length) {
                return $headerFooterPlaceHolder.children('.footer_' + currentPageType).css({ minHeight: minHeight, paddingTop: Math.max(distanceTop, 5), paddingRight: pagePaddingRight, paddingBottom: footerMargin, paddingLeft: pagePaddingLeft }).clone(true);
            } else {
                return '<div class="footer inactive-selection ' + (_.browser.IE ? '' : 'flexfooter') + type + '" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + ' style="min-height: ' + minHeight +
                    'px; padding: ' + Math.max(distanceTop, 5) + 'px ' + pagePaddingRight + 'px ' + footerMargin + 'px ' + (pagePaddingLeft) + 'px;" data-focus-role="footer"><div class="cover-overlay" contenteditable="false"></div><div class="marginalcontent' + (_.browser.IE ? '' : ' flexcontent') + '"></div></div>';
            }
        }

        /**
         * Called from implCreateHeaderFooter function, generates header DOM element for given operation attributes.
         *
         * @param {String} operationId
         * @param {String} operationType
         */
        function createHeaderContainer(operationId, operationType) {
            return $('<div>').addClass('header inactive-selection ' + operationId + ' ' + operationType)
                .attr('contenteditable', false).attr('data-container-id', operationId).attr('data-focus-role', 'header')
                .data({ id: operationId, type: operationType })
                .prepend($('<div>').addClass('cover-overlay').attr('contenteditable', false))
                .append($('<div>').addClass(DOM.MARGINALCONTENT_NODE_CLASSNAME).append($('<div>').addClass('p').data('implicit', true)));
        }

        /**
         * Called from implCreateHeaderFooter function, generates footer DOM element for given operation attributes.
         *
         * @param {String} operationId
         * @param {String} operationType
         */
        function createFooterContainer(operationId, operationType) {
            return $('<div>').addClass('footer inactive-selection ' + operationId + ' ' + operationType + (_.browser.IE ? '' : ' flexfooter'))
                .attr('contenteditable', false).attr('data-container-id', operationId).attr('data-focus-role', 'footer')
                .data({ id: operationId, type: operationType })
                .prepend($('<div>').addClass('cover-overlay').attr('contenteditable', false))
                .append($('<div>').addClass(DOM.MARGINALCONTENT_NODE_CLASSNAME + (_.browser.IE ? '' : ' flexcontent')).append($('<div>').addClass('p').data('implicit', true)));
        }

        /**
        *  Update styling of header&footer inside placeholder, and fetch their heights.
        */
        function updateStartHeaderFooterStyle() {
            var
                headers = $headerFooterPlaceHolder.find('.header'),
                footers = $headerFooterPlaceHolder.find('.footer');

            if (!pageAttributes) {
                initializePageAttributes();
            }

            $headerFooterPlaceHolder.css({
                width: pageWidth,
                margin: '0 0 150px -' + (pagePaddingLeft) + 'px',
                left: pagePaddingLeft
            });
            headers.css({
                minHeight: pagePaddingTop,
                maxWidth: pageWidth,
                padding: headerMargin + 'px ' + pagePaddingRight + 'px 0 ' + (pagePaddingLeft) + 'px'
            });
            footers.css({
                minHeight: pagePaddingBottom,
                maxWidth: pageWidth,
                padding: '5px ' + pagePaddingRight + 'px ' + footerMargin + 'px ' + (pagePaddingLeft) + 'px'
            });
            if (!_.browser.IE) {
                footers.addClass('flexfooter');
                footers.children('.marginalcontent').addClass('flexcontent');
            }

            rootNode.css({ paddingBottom: 0, paddingTop: 0 });

            // update height vaules of headers&footers
            _.each(headers, function (header) {
                if (header.classList.contains('header_first')) {
                    heightOfFirstHeader = $(header).outerHeight(true);
                } else if (header.classList.contains('header_even')) {
                    heightOfEvenHeader = $(header).outerHeight(true);
                } else {
                    heightOfDefHeader = $(header).outerHeight(true);
                }
            });
            _.each(footers, function (footer) {
                if (footer.classList.contains('footer_first')) {
                    heightOfFirstFooter = $(footer).outerHeight(true);
                } else if (footer.classList.contains('footer_even')) {
                    heightOfEvenFooter = $(footer).outerHeight(true);
                } else {
                    heightOfDefFooter = $(footer).outerHeight(true);
                }
            });

            if (heightOfFirstHeader !== 0 || heightOfFirstFooter !== 0) {
                firstPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfFirstHeader) - Math.max(pagePaddingBottom, heightOfFirstFooter);
                if (app.isODF()) {
                    firstPage = true; // for ODF docs this document attribute isn't set
                }
            }
            if (heightOfEvenHeader !== 0 || heightOfEvenFooter !== 0) {
                evenPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfEvenHeader) - Math.max(pagePaddingBottom, heightOfEvenFooter);
                if (app.isODF()) {
                    evenOddPages = true; // for ODF docs this document attribute isn't set
                }
            }
            if (heightOfDefHeader !== 0 || heightOfDefFooter !== 0) {
                defPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfDefHeader) - Math.max(pagePaddingBottom, heightOfDefFooter);
            } else {
                defPageHeight = self.getDefPageActiveHeight({ convertToPixel: true });
            }

            //console.warn('firstPageHeight: ', firstPageHeight);
            //console.warn('evenPageHeight: ', evenPageHeight);
            //console.warn('defPageHeight: ', defPageHeight);
        }

        /**
         * On switching to new page, change global properties for main page content height, and type of page
         *
         */
        function switchPageTypeAndHeight() {

            switch (currentPageType) {
                case 'first':
                case 'default':
                    if (evenOddPages && evenPageHeight > 0) {
                        pageMaxHeight = evenPageHeight;
                        currentPageType = 'even';
                    } else if (currentPageType === 'first') {
                        currentPageType = 'default';
                        pageMaxHeight = defPageHeight;
                    }
                    break;
                case 'even':
                    pageMaxHeight = defPageHeight;
                    currentPageType = 'default';
                    break;
            }
        }

        /**
         * Internal function for page layout rendering!
         * Generating html markup string to insert as page break div element inside page.
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Boolean} contentEditableProperty
         *  Webkit browsers needs to set contenteditable to false, others to empty
         *
         * @param {Number} [elementLeftIndent]
         *  Left indent of element, in px
         *
         * @returns {String|jQuery}
         *  Div page break html string or node markup
         */
        function generatePagebreakElementHtml(contentHeight, contentEditableProperty, elementLeftIndent) {
            var leftIndent = elementLeftIndent || 0,
                markup;

            if (self.isHeaderFooterInDocument()) {
                markup = $('<div>').addClass('page-break').attr('contenteditable', (contentEditableProperty ? 'false' : ''))
                    .css({ width: pageWidth, margin: (Math.max(pageMaxHeight - contentHeight, 0) + 'px 0px 0px -' + (pagePaddingLeft + leftIndent) + 'px') })
                    .append(getFooterHtmlContent())
                    .append($('<div>').addClass('inner-pb-line'))
                    .append(getHeaderHtmlContent());
            } else {
                markup = '<div class="page-break" contenteditable=' + (contentEditableProperty ? '"false"' : '""') + 'style="width: ' + pageWidth +
                    'px; margin: ' + (Math.max(pageMaxHeight - contentHeight, 0)) + 'px 0px 0px -' + (pagePaddingLeft + leftIndent) + 'px">' +
                    getFooterHtmlContent() + '<div class="inner-pb-line"></div>' + getHeaderHtmlContent() + '</div>';
            }

            return markup;
        }

        /**
         * Internal function for page layout rendering!
         * Generating html markup string to insert inside table as row containing page break div.
         *
         * @param {Number} contentHeight
         *  Height of all content in current page, in px
         *
         * @param {Number} negativeMarginTablesFix
         *  IE needs 2px, FF 1px, due to border collapse
         *
         * @returns {String|jQuery}
         *  Table row containing div page break html string or node markup
         */
        function generateTableRowPagebreakHTML(contentHeight, negativeMarginTablesFix) {
            var markup;

            if (self.isHeaderFooterInDocument()) {
                markup = $('<tr>').addClass('pb-row')
                    .append($('<td>').css({ padding: '0', border: 'none' }).attr('colspan', '1')
                        .append($('<div>').addClass('page-break').attr('contenteditable', (_.browser.WebKit ? 'false' : ''))
                        .css({ width: pageWidth, margin: (Math.max(pageMaxHeight - contentHeight, 0) + 'px 0px 0px -' + (pagePaddingLeft + negativeMarginTablesFix) + 'px') })
                            .append(getFooterHtmlContent())
                            .append($('<div>').addClass('inner-pb-line'))
                            .append(getHeaderHtmlContent())
                        )
                    );
            } else {
                markup = '<tr class="pb-row"><td style="padding: 0px; border: none" colspan="1"><div class="page-break" contenteditable=' +
                    (_.browser.WebKit ? '"false"' : '""') + 'style="width: ' + pageWidth + 'px; margin: ' + (Math.max(pageMaxHeight - contentHeight, 0)) +
                    'px 0px 0px -' + (pagePaddingLeft + negativeMarginTablesFix) + 'px">' + getFooterHtmlContent() + '<div class="inner-pb-line"></div>' + getHeaderHtmlContent() + '</div></td></tr>';
            }
            return markup;
        }

        /**
         * Internal function for page layout rendering!
         * Default insertion of manual page breaks,
         * it adds page break before passed element if it is paragraph style attribute,
         * or after if it is Microsoft's hardbreak type page
         *
         * @param {Node} element
         *  DOM element currently beeing proccessed
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.cleanup=false]
         *      If its neccessary to do pagebreaks cleanup before insertion.
         *      Happens if it's not called from initial page break render function.
         *
         * @param {Number} contentHeight
         *  Height of content in current page before proccessing
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after proccessing
         *
         */
        function manualInsertionOfPageBreak(element, contentHeight, elementHeight, options) {
            var
                $element = $(element),
                cleanup = Utils.getBooleanOption(options, 'cleanup', false),
                msHardbreakPage = $element.find('.ms-hardbreak-page'),
                textSpanAfterHardbreak = msHardbreakPage.last().parent('.hardbreak').next(),
                markup,
                $nodeHelper,
                tempContentHeight = contentHeight,
                isAfterLastElement = false,
                elementLeftIndent = parseInt($element.css('margin-left'), 10) + parseInt($element.css('padding-left'), 10); // for indented elements such as lists, or special paragraphs;

            if (cleanup) {
                $nodeHelper = $element.prev();
                // remove old page break
                while ($nodeHelper.hasClass('page-break')) {
                    $nodeHelper.remove();
                    // TODO
                    pageBreaksCollection = $pageContentNode.find('.page-break');
                    resolvePageTypeAndHeight(element);

                    $nodeHelper = $element.prev();
                }
                $element.removeClass('manual-page-break');

                prepareNodeAndRemovePageBreak(element);
            }
            if ($element.is('table') && $element.find(DOM.MANUAL_PAGE_BREAK_SELECTOR).length < 1) {
                return contentHeight;
            }

            $element.next().addClass('break-above'); // this is marker for first node bellow pagebreak
            if (msHardbreakPage.length > 0) { // MS Word hardbreak page
                tempContentHeight += msHardbreakPage.last().parent('.hardbreak').position().top;

                _.each(msHardbreakPage, function (msBreak) {
                    var spanAfterHardbreak,
                        nextAfterSpan;

                    if ($(msBreak).parent('.hardbreak').prevAll(DOM.LIST_LABEL_NODE_SELECTOR).length > 0) { // #35649 hardbreak between text and list label
                        markup = generatePagebreakElementHtml(tempContentHeight, _.browser.WebKit);
                        $element.before(markup);
                    } else {
                        markup = generatePagebreakElementHtml(tempContentHeight, _.browser.WebKit, elementLeftIndent);
                        // if next element after mpb is empty text span, that is last in paragraph -> insert after it
                        // if next element after mpb is empty text span, that is not last in p, but before next mpb -> insert after it
                        // in other cases -> insert manual pb before span
                        spanAfterHardbreak = $(msBreak).parent('.hardbreak').next();
                        nextAfterSpan = spanAfterHardbreak.next();
                        if (DOM.isEmptySpan(spanAfterHardbreak) && ((!nextAfterSpan.length || nextAfterSpan.is('br')) || (nextAfterSpan.hasClass('.hardbreak') && nextAfterSpan.data('type') === 'page'))) {
                            if (nextAfterSpan.is('br')) {
                                spanAfterHardbreak.addClass('break-above-span');
                                nextAfterSpan.after(markup);
                            } else {
                                spanAfterHardbreak.addClass('break-above-span').after(markup);
                            }
                            if (!nextAfterSpan.length || nextAfterSpan.is('br')) {
                                isAfterLastElement = true; // if manual pb is after last element
                            }
                        } else {
                            spanAfterHardbreak.addClass('break-above-span').before(markup);
                        }
                        $element.addClass('contains-pagebreak');
                    }
                });
                if (elementHeight <= pageMaxHeight && !isAfterLastElement) {
                    contentHeight = textSpanAfterHardbreak.length > 0 ? $element.height() - textSpanAfterHardbreak.position().top : 0;
                } else {
                    contentHeight = 0;
                }
            } else {
                if (!$element.prev().hasClass('manual-page-break') || !$element.prev().find('.ms-hardbreak-page').length) { // #38544
                    markup = generatePagebreakElementHtml(contentHeight, _.browser.WebKit);
                    $element.before(markup);
                }
                contentHeight = 0;
            }

            $element.addClass('manual-page-break');

            return contentHeight;
        }

        /**
         *
         * Internal function for page layout rendering!
         * Support function for Open Document Format, adds manual page break after passed node element
         *
         * @param {Node} element
         *  DOM element currently beeing proccessed
         *
         * @param {Number} contentHeight
         *  Height of content in current page before proccessing; in px
         *
         * @returns {Number} contentHeight
         *  Height of content in current page after proccessing; in px
         *
         */

        function insertManualPageBreakAfter(element, contentHeight) {
            $(element).after(generatePagebreakElementHtml(contentHeight, _.browser.WebKit));
            // new element will always start at the beginning of new page
            contentHeight = 0;
            return contentHeight;
        }

        /**
         * Internal function for page layout rendering!
         * Makes preparation for node proccessing,
         * like cleanup of previous page breaks, and class naming
         *
         * @param {Node} element
         *  DOM element currently beeing proccessed
         *
         */
        function prepareNodeAndRemovePageBreak(element) {
            var
                $element = $(element),
                $nodeHelper = $element.next();

            //if element is last in document
            if ($nodeHelper.length) {
                // remove old page break
                while ($nodeHelper.hasClass('page-break')) {
                    $nodeHelper.remove();
                    $nodeHelper = $element.next();
                }
            }
        }

        /**
         * Loops through collection of page breaks,
         * and determines correct class name for each header and footer,
         * regarding of page number.
         *
         * @param {Number[]|jQuery} pageBreaksCollection
         */
        function populateHeaderFooterPaginationClasses(pageBreaksCollection) {

            _.each(pageBreaksCollection, function (pageBreak, index) {
                if (index === 0) {
                    $(pageBreak).children('.footer').addClass('first');
                    $(pageBreak).children('.header').addClass('even');
                } else if (index % 2 === 0) {
                    $(pageBreak).children('.header').addClass('even');
                    $(pageBreak).children('.footer').addClass('default');
                } else {
                    $(pageBreak).children('.footer').addClass('even');
                    $(pageBreak).children('.header').addClass('default');
                }
            });
            $firstHeaderWrapper.children('.header').addClass('first');
            $lastFooterWrapper.children('.footer').removeClass('first default even');
            if (pageBreaksCollection.length === 0) {
                $lastFooterWrapper.children('.footer').addClass('first');
            } else if (pageBreaksCollection.length % 2 === 0) {
                $lastFooterWrapper.children('.footer').addClass('default');
            } else {
                $lastFooterWrapper.children('.footer').addClass('even');
            }
        }

        /**
         * Internal function to update data inside special fields with type: page-count.
         * Frontend needs to overwrite dummy values backend sends,
         * because backend has no knowledge of page count after rendering doc in browser.
         *
         */
        function updatePageCountFields() {
            _.each(rootNode.find(DOM.PAGECOUNT_FIELD_SELECTOR), function (field) {
                $(field.childNodes).empty().html(pageCount);
            });
        }

        /**
         * Internal function to update data inside special fields with type: page-number.
         * Frontend needs to overwrite dummy values backend sends,
         * because backend has no knowledge of page count after rendering doc in browser.
         *
         */
        function updatePageNumberFields() {
            // update in headers & footers
            _.each($pageContentNode.find('.header').find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                $(field).children().empty().html($(field).parentsUntil('.pagecontent', '.page-break').data('page-num') + 1);
            });

            _.each($pageContentNode.find('.footer').find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                $(field).children().empty().html($(field).parentsUntil('.pagecontent', '.page-break').data('page-num'));
            });
            // first header in doc - page number always 1
            $firstHeaderWrapper.children('.header').find(DOM.PAGENUMBER_FIELD_SELECTOR).children().empty().html(1);
            // last footer in doc gets total page count as page number
            $lastFooterWrapper.children('.footer').find(DOM.PAGENUMBER_FIELD_SELECTOR).children().empty().html(pageCount);

            // update in paragraphs as first level children of pagecontent
            _.each($pageContentNode.children(DOM.PARAGRAPH_NODE_SELECTOR).children(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                $(field).children().empty().html(self.getPageNumber(field));
            });
            // update in paragraphs inside tables as first level children of pagecontent
            _.each($pageContentNode.children(DOM.TABLE_NODE_SELECTOR).find('tr').find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                $(field).children().empty().html(self.getPageNumber(field));
            });
        }

        /**
         * Creates overlay layer for tooltip over user-fields.
         *
         */
        function createTooltipOverlay() {
            $tooltipOverlay = $('<div>').addClass('field-tooltip-layer').attr('contenteditable', 'false');
            rootNode.append($tooltipOverlay);
        }

        /**
         * Generates unique id, used as target parameter for insertHeaderFooter operation
         *
         * @param {String} type
         *  Type of header or footer to be generated.
         *
         * @returns {String} generated target string
         *
         */
        function generateOperationTargetString(type) {
            if (app.isODF()) {
                return type + '_Standard';
            } else {
                if (targetIdNumbersCollection.length) {
                    generatedTargetNumber = _.last(targetIdNumbersCollection) + 1;
                }
                return 'OX_rId' + generatedTargetNumber;
            }
        }

        /**
         * Generate operation for header or footer insertion in document.
         *
         * @param {String} type
         *  Type of header or footer that is going to be created.
         */
        function createOperationInsertHeaderFooter(type) {
            var localGenerator = model.createOperationsGenerator(),
                operation = { id: generateOperationTargetString(type), type: type },
                rootNode,
                paragraphStyles = model.getStyleCollection('paragraph'),
                isHeader = type.indexOf('header') > -1,
                localStyleId = isHeader ? paragraphStyles.getStyleIdByName('Header') : paragraphStyles.getStyleIdByName('Footer'),
                paraAttr = null;

            Utils.info('pageLayout.createOperationInsertHeaderFooter() with type', type);

            if ($headerFooterPlaceHolder.children('.' + type).length) {
                Utils.warn('createOperationInsertHeaderFooter: node with this type already exists!', type);
                replaceAllTypesOfHeaderFooters();
            } else {
                if (paragraphStyles) {
                    if (paragraphStyles.isDirty(localStyleId)) {
                        model.generateInsertStyleOp(localGenerator, 'paragraph', localStyleId);
                    }
                    paraAttr = { styleId: localStyleId };

                    localGenerator.generateOperation(Operations.INSERT_HEADER_FOOTER, operation);
                    localGenerator.generateOperation(Operations.PARA_INSERT, { start: [0], target: operation.id, attrs: paraAttr });
                    model.applyOperations(localGenerator);

                    // at the end mark all nodes with marginal class, in both current node and placeholder.
                    // it is important that placeholder node is also processed, because there can be situation where updateEditingHeaderFooter is not triggered.
                    rootNode = model.getRootNode(operation.id);
                    markMarginalNodesInsideHeadFoot(rootNode);
                    if (!DOM.isMarginalPlaceHolderNode(rootNode.parent())) {
                        $headerFooterPlaceHolder.children('.' + type).empty().append(rootNode.children().clone(true));
                        replaceAllTypesOfHeaderFooters();
                    }
                    //markMarginalNodesInsideHeadFoot($headerFooterPlaceHolder.children('.' + type));
                } else {
                    Utils.error('pageLayout.createOperationInsertHeaderFooter(): missing paragraph styles!');
                }
            }
        }

        /**
         * Generate operation for header/footer insertion in ODF document.
         *
         * @param {jQuery} node
         *  Cached node from which insert header/footer operation
         *  and following content operations are going to be reconstructed.
         * @param {String} type
         *  Type of header or footer that is going to be created.
         */
        function insertHFopODT(node, type) {
            var localGenerator = model.createOperationsGenerator(),
                id = DOM.getTargetContainerId(node),
                operation = { id: id, type: type },
                rootNode,
                paragraphStyles = model.getStyleCollection('paragraph'),
                isHeader = type.indexOf('header') > -1,
                localStyleId = isHeader ? paragraphStyles.getStyleIdByName('Header') : paragraphStyles.getStyleIdByName('Footer'),
                options = null,
                position = [],
                paraAttr = null;

            Utils.info('pageLayout.createOperationInsertHeaderFooter() with type', type);

            if (paragraphStyles) {
                if (paragraphStyles.isDirty(localStyleId)) {
                    model.generateInsertStyleOp(localGenerator, 'paragraph', localStyleId);
                }
                localGenerator.generateOperation(Operations.INSERT_HEADER_FOOTER, operation);

                node.children('.cover-overlay').remove();
                if (node.children('.marginalcontent').children().length) {
                    options = { target: id };
                    localGenerator.generateContentOperations(node, position, options);
                } else {
                    paraAttr = { styleId: localStyleId };
                    localGenerator.generateOperation(Operations.PARA_INSERT, { start: [0], target: id, attrs: paraAttr });
                }
                model.applyOperations(localGenerator);

                // at the end mark all nodes with marginal class, in both current node and placeholder.
                // it is important that placeholder node is also processed, because there can be situation where updateEditingHeaderFooter is not triggered.
                rootNode = model.getRootNode(id);
                markMarginalNodesInsideHeadFoot(rootNode);
                if (!DOM.isMarginalPlaceHolderNode(rootNode.parent())) {
                    $headerFooterPlaceHolder.children('.' + type).empty().append(rootNode.children().clone(true));
                    replaceAllTypesOfHeaderFooters();
                }
                //markMarginalNodesInsideHeadFoot($headerFooterPlaceHolder.children('.' + type));
            } else {
                Utils.error('pageLayout.insertHFopODT(): missing paragraph styles!');
            }
        }

        /**
         * In order not to jump to first header/footer in document,
         * store reference to node user selects, and after successful createHeaderFooter,
         * enter headerFooterEditMode for that node.
         *
         * @param {jQuery} $node
         *  Node instance that is going to be stored.
         */
        function storeNodeForHeaderFooterCreation($node) {
            $storedHFnode = $node;
        }

        /**
         * In order not to jump to first header/footer in document,
         * get the reference to node user selects, and after successful createHeaderFooter,
         * enter headerFooterEditMode for that node.
         *
         * @returns {jQuery}
         *  Returns stored node.
         */
        function getNodeStoredOnHeaderFooterInsertion() {
            return $storedHFnode;
        }

        /**
         * For passed container node, that is not yet header or footer,
         * based on position in page and page number, determine type of node
         * passing to insertHeaserFooter operation.
         *
         * @param {jQuery} $node
         *  Passed node for which we want to get type
         *
         * @returns {String}
         *  Calculated type.
         */
        function getHeadFootTypeForCreateOp($node) {
            var
                isHeader = $node.hasClass('header'),
                headerType = 'header_default',
                footerType = 'footer_default',
                pageNum = self.getPageNumber($node);

            if (firstPage && pageNum === 1) {
                headerType = 'header_first';
                footerType = 'footer_first';
            } else if (evenOddPages && (pageNum % 2 === 0)) {
                headerType = 'header_even';
                footerType = 'footer_even';
            }
            return isHeader ? headerType : footerType;
        }

        /**
         * Create new context menu node to append to parent header or footer container.
         *
         * @param {String} type
         *  Type of header or footer
         * @param {Boolean} [isHeader]
         *  If true, it is header, if not, it's footer.
         * @returns {jQuery} newContextMenu
         *  Created context menu.
         */
        function createContextMenu(type, isHeader) {
            var newContextMenu,
                positionClass = isHeader ? 'below-node' : 'above-node',
                gotoLabel = isHeader ? gt('Go to footer') : gt('Go to header');

            newContextMenu = $('<div>').addClass('marginal-context-menu ' + positionClass)
                .append($('<div>').addClass('marginal-menu-type').html('<div class="marginal-label">' + generateMarginalNodeTypeText(type) + '<i class="fa fa-caret-down"></i></div>')
                    .append($('<div>').addClass('marginal-container').css({ display: 'none' })
                        .append($('<div>').addClass('marginal-same').html(gt('Same across entire document')),
                            $('<div>').addClass('marginal-first').html(gt('Different first page')),
                            $('<div>').addClass('marginal-even').html(gt('Different even & odd pages')),
                            $('<div>').addClass('marginal-first-even').html(gt('Different first, even & odd pages')),
                            $('<div>').addClass('marginal-separator-line'),
                            $('<div>').addClass('marginal-none').html(gt('Remove all headers and footers'))
                        )
                    ),
                    $('<div>').addClass('marginal-menu-goto').html(gotoLabel),
                    $('<div>').addClass('marginal-menu-close').html(gt('Close') + '<i class="fa fa-times"></i>')
                );

            if (_.browser.Chrome) {
                newContextMenu.children().css({ display: 'inline-flex' }); // fix of buttons in chrome, for different browser zoom values
            }

            return newContextMenu;
        }

        /**
         * Gets the context menu node for header.
         * If it's not created yet, it is first created.
         * If already existing, first updates the label with header type,
         * and then context menu is returned.
         *
         * @param {String}
         *  Type of the header for which we are creating menu.
         *
         * @returns {jQuery}
         *  Context menu for header.
         */
        function getContextHeaderMenu(type) {
            if (!contextHeaderMenu) {
                contextHeaderMenu = createContextMenu(type, true);
            } else {
                contextHeaderMenu.find('.marginal-label').empty().html(generateMarginalNodeTypeText(type) + '<i class="fa fa-caret-down"></i>');
            }
            return contextHeaderMenu;
        }

        /**
         * Gets the context menu node for footer.
         * If it's not created yet, it is first created.
         * If already existing, first updates the label with footer type,
         * and then context menu is returned.
         *
         * @param {String}
         *  Type of the header for which we are creating menu.
         *
         * @returns {jQuery}
         *  Context menu for footer.
         */
        function getContextFooterMenu(type) {
            if (!contextFooterMenu) {
                contextFooterMenu = createContextMenu(type);
            } else {
                contextFooterMenu.find('.marginal-label').empty().html(generateMarginalNodeTypeText(type) + '<i class="fa fa-caret-down"></i>');
            }
            return contextFooterMenu;
        }

        /**
         * Helper function to generate label for context menu.
         * It makes lookup in translations dictionary.
         *
         * @param {String} type
         *  Type of header or footer for which label is generated.
         *
         * @returns {String}
         *  Generated label for header or footer.
         *
         */
        function generateMarginalNodeTypeText(type) {
            if (evenOddPages && !(/first/i).exec(type)) {
                if (type === 'header_default') {
                    return translations.header_odd;
                } else if (type === 'footer_default') {
                    return translations.footer_odd;
                }
            }

            return translations[type];
        }

        /**
         * Inserts context menu into active header or footer node.
         *
         * @param {jQuery} $node
         *  Active header or footer node, in which we are inserting context menu.
         *
         * @param {Boolean} isHeader
         *  Wether it is header or footer.
         *
         * @param {String} type
         *  Type of the header or footer node (default, first, even).
         *
         */
        function insertContextMenu($node, isHeader, type) {
            if (!type) {
                Utils.error('pageLayout.insertContextMenu: type is missing!');
                return;
            }
            if (DOM.isMarginalPlaceHolderNode($node.parent())) {
                Utils.warn('pageLayout.insertContextMenu: trying to insert context menu on template node!');
                return;
            }

            if (isHeader) {
                getContextHeaderMenu(type).insertAfter($node);
            } else {
                getContextFooterMenu(type).insertBefore($node);
            }
        }

        /**
         * Local helper to replace active header/footer node with coresponding new node on implToggle functions.
         *
         * @param {Bolean} [removeAll]
         *  If set to true, no replacing with new node, but leaving edit state.
         */
        function replaceActiveNodeOnToggle(removeAll) {
            var $rootNode,
                id = null;

            replaceAllTypesOfHeaderFooters();

            if (model.isHeaderFooterEditState()) {
                if (removeAll) {
                    self.leaveHeaderFooterEditMode();
                    selection.setNewRootNode(model.getNode()); // restore original rootNode
                    selection.setTextSelection(selection.getFirstDocumentPosition());
                } else {
                    $rootNode = model.getHeaderFooterRootNode();
                    id = $rootNode.attr('data-container-id');
                    model.setActiveTarget(id);
                    $rootNode.children('.cover-overlay').remove();
                    // browser specific handling
                    if (_.browser.IE) {
                        _.each($rootNode.find('span'), function (span) {
                            DOM.ensureExistingTextNode(span);
                        });
                    }
                    insertContextMenu($rootNode, $rootNode.hasClass('header'), self.getTypeFromId(id));
                }
            }
        }

        /** Local helper to refresh collected nodes children with replacement children,
         *  and if necessary, trigger drawingHeight update.
         *
         *  @param {jQuery} collectionNodes
         *      Nodes that are updated with replacement children.
         *  @param {jQuery} $replacementChildren
         *      Replacement nodes.
         *  @param {jQuery} $coverOverlay
         *      Overlay node that covers inactive header/footer nodes, also needs to be appended.
         */
        function refreshCollectionNodes(collectionNodes, $replacementChildren, $coverOverlay) {
            var $drawingFrameNodes = null;

            collectionNodes.empty().append($coverOverlay, $replacementChildren.clone(true));
            $drawingFrameNodes = collectionNodes.find(DrawingFrame.NODE_SELECTOR);
            if ($drawingFrameNodes.length) {
                model.trigger('drawingHeight:update', $drawingFrameNodes);
            }
        }

        /**
         * On toggling types of h/f, if dropdown is open, hide it for rerender.
         * If toogling is triggered from menu, and not context menu,
         * can happen that edit mode of h/f is not active. Then jump to header on page where cursor is.
         */
        function prepareMarginalDropdown() {
            var activeDropdown;

            if (!model.isHeaderFooterEditState()) {
                self.insertHeader();
            } else {
                activeDropdown = model.getCurrentRootNode().parent().find('.active-dropdown');
                if (activeDropdown.length) {
                    self.toggleMarginalMenuDropdown(activeDropdown); // if dropdown is active, hide it before re-render
                }
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Updates page by inserting page layout.
         *
         * @param {Object} [options]
         *
         * @returns {promise}
         *
         */
        this.callInitialPageBreaks = function (options) {
            return initialPageBreaks(options);
        };

        /**
         * Selector that queries all headers&footers curently present in document.
         *
         * @returns {jQuery}
         */
        this.getallHeadersAndFooters = function () {
            return $pageContentNode.find('.header, .footer').add($firstHeaderWrapper.children('.header')).add($lastFooterWrapper.children('.footer'));
        };

        /**
         * Queries and returns template node for storing headers and footers.
         *
         * @returns {jQuery}
         */
        this.getHeaderFooterPlaceHolder = function () {
            return $headerFooterPlaceHolder;
        };

        /**
         * If header/footer template node is empty or it contains headers and footers.
         *
         * @returns {Boolean}
         */
        this.hasContentHeaderFooterPlaceHolder = function () {
            return $headerFooterPlaceHolder.children().length > 0;
        };

        /**
         * Checks if first header wrapper node is already created or not.
         *
         * @returns {Boolean}
         */
        this.isExistingFirstHeaderWrapper = function () {
            return $firstHeaderWrapper.length > 0;
        };

        /**
         * Checks if last footer wrapper node is already created or not.
         *
         * @returns {Boolean}
         */
        this.isExistingLastFooterWrapper = function () {
            return $lastFooterWrapper.length > 0;
        };

        /**
         * Returns first header wrapper node in the document. (One right above pagecontent node)
         *
         * @returns {jQuery}
         */
        this.getFirstHeaderWrapperNode = function () {
            return $firstHeaderWrapper;
        };

        /**
         * Returns last footer wrapper node in the document. (One right bellow pagecontent node)
         *
         * @returns {jQuery}
         */
        this.getLastFooterWrapperNode = function () {
            return $lastFooterWrapper;
        };

        /**
         * Return first in collection of marginal nodes in document, with passed id
         *
         * @param {String} id
         *  Id of node we search.
         * @returns {jQuery}
         *  Found node in document.
         */
        this.getFirstMarginalByIdInDoc = function (id) {
            return model.getNode().find('.header, .footer').filter('[data-container-id="' + id + '"]').first();
        };

        /**
         * Public method to check if first header in document exist, and if does, return it's wrapper's height (in px).
         * If element exists, returns height of wrapper element, of first header in document (which is prepended to pagecontent node)
         * If element doesn't exist, returs 0.
         *
         * @returns {Number}
         */
        this.getHeightOfFirstHeaderWrapper = function () {
            return $firstHeaderWrapper ? $firstHeaderWrapper.height() : 0;
        };

        /**
         * If there are headers/footers in document.
         *
         * @returns {Boolean}
         */
        this.isHeaderFooterInDocument = function () {
            return $headerFooterPlaceHolder.children().length > 0;
        };

        /**
         * If headers/footers are activated in document.
         *
         * @returns {Boolean}
         */
        this.isHeaderFooterActivated = function () {
            return marginalActivated;
        };

        /**
         * Gets the header's or footer's id from given type.
         *
         * @param {String} type
         *  Type of header or footer - can be default, first, or even.
         *
         * @returns {String}
         *  Returns found id, or empty string
         *
         */
        this.getIdFromType = function (type) {
            if (type) {
                var obj = _.findWhere(headerFooterCollection, { headFootType: type });

                if (obj && obj.id) {
                    return obj.id;
                }
            }
            return '';
        };

        /**
         * Looks up the headerFooterCollection, and returns type of object for given Id.
         *
         * @param {String} operationId
         * @returns {String}
         *  If found, searched type.
         */
        this.getTypeFromId = function (operationId) {
            var $obj = _.findWhere(headerFooterCollection, { id: operationId });

            if ($obj && $obj.headFootType) {
                return $obj.headFootType;
            }

            Utils.warn('pageLayout.getTypeFromId: type for given id not found!');
            return;
        };

        /**
         * Looks up the headerFooterCollection, and checks if given id belongs to marginal nodes collection.
         *
         * @param {String} id
         *  Queried id.
         * @returns {Boolean}
         *  Whether passed id belongs to marginal nodes.
         */
        this.isIdOfMarginalNode = function (id) {
            return !_.isUndefined(_.findWhere(headerFooterCollection, { id: id }));
        };

        /**
         * Mark paragraph and all it's children as marginal.
         *
         * @param {Node} paragraph
         *
         * @param {String} target
         */
        this.markParagraphAsMarginal = function (paragraph, target) {
            if (self.isIdOfMarginalNode(target)) {
                $(paragraph).addClass(DOM.MARGINAL_NODE_CLASSNAME).find('*').addClass(DOM.MARGINAL_NODE_CLASSNAME);
            }
        };

        /**
         * Public accessor for updating style of template node for header&footer.
         *
         */
        this.updateStartHeaderFooterStyle = function () {
            updateStartHeaderFooterStyle();
        };

        /**
         * Creates generic placeholder for the first header in the document.
         *
         * @returns {jQuery}
         *  Created placeholder header node, after insertion in DOM
         */
        this.generateHeaderElementHtml = function () {
            var
                firstHeaderWrapper = $('<div>').addClass('header-wrapper').attr('contenteditable', (_.browser.WebKit ? 'false' : ''))
                    .css({ width: pageWidth, margin: '0px 0px 0px ' + (-pagePaddingLeft) + 'px' }).
                    append(getHeaderHtmlContent('first'));

            $pageContentNode.before(firstHeaderWrapper);

            return firstHeaderWrapper;
        };

        /**
         * Creates generic placeholder for last footer in the document.
         *
         * @returns {jQuery}
         *  Created placeholder footer node, after insertion in DOM
         */
        this.generateFooterElementHtml = function () {
            var
                lastFooterWrapper = $('<div>').addClass('footer-wrapper').attr('contenteditable', (_.browser.WebKit ? 'false' : ''))
                    .css({ width: pageWidth, margin: '0px 0px 0px ' + (-pagePaddingLeft) + 'px' }).
                    append(getFooterHtmlContent());

            $pageContentNode.after(lastFooterWrapper);

            return lastFooterWrapper;
        };

        /**
         * Impl function triggered when INSERT_HEADER_FOOTER operation is called
         *
         * @param {String} operationId
         *
         * @param {String} operationType
         *
         * @param {Object} attributes
         *
         * @param {Boolean} undoRedoRunning - if undo or redo is running, h/f needs to be updated, so that changes are not lost
         *
         * @returns {Boolean}
         *  True if header or footer is succesfully created in DOM and collection, otherwise false
         *
         */
        this.implCreateHeaderFooter = function (operationId, operationType, attributes, undoRedoRunning) {
            var isHeader = (/header/i).test(operationType),
                localCollection,
                $paragraph,
                minHeight,
                distanceBottom,
                distanceTop,
                nodeHeight;

            localCollection = {
                headFootType: operationType,
                id: operationId,
                node: (isHeader ? createHeaderContainer(operationId, operationType) : createFooterContainer(operationId, operationType))
            };
            if ($headerFooterPlaceHolder.length > 0) {
                $headerFooterPlaceHolder.append(localCollection.node);
            } else {
                $headerFooterPlaceHolder = $('<div>').addClass('header-footer-placeholder').attr('contenteditable', _.browser.WebKit ? 'false' : '').css('display', Utils.SMALL_DEVICE ? 'none' : '').append(localCollection.node);
                rootNode.append($headerFooterPlaceHolder);
            }
            headerFooterCollection.push(localCollection);
            if (operationId.match(/\d+/)) {
                targetIdNumbersCollection.push(parseInt(operationId.match(/\d+/)));
                targetIdNumbersCollection.sort(function (a, b) { return a - b; });
            }
            marginalActivated = true;

            if (app.isODF()) {
                if ((/first/i).test(operationType)) {
                    firstPage = true;
                } else if ((/even/i).test(operationType)) {
                    evenOddPages = true;
                }
                if (attributes && attributes.page) {
                    minHeight = attributes.page.minHeight ? Utils.convertHmmToLength(attributes.page.minHeight, 'px', 0.001) : 0;
                    distanceBottom = attributes.page.marginBottom ? Utils.convertHmmToLength(attributes.page.marginBottom, 'px', 0.001) : 0;
                    distanceTop = attributes.page.marginTop ? Utils.convertHmmToLength(attributes.page.marginTop, 'px', 0.001) : 0;
                    if (isHeader) {
                        if ((/first/i).test(operationType)) {
                            odfFirstHeader.height = minHeight;
                            odfFirstHeader.bottom = distanceBottom;
                        } else if ((/even/i).test(operationType)) {
                            odfEvenHeader.height = minHeight;
                            odfEvenHeader.bottom = distanceBottom;
                        } else {
                            odfDefHeader.height = minHeight;
                            odfDefHeader.bottom = distanceBottom;
                        }
                    } else {
                        if ((/first/i).test(operationType)) {
                            odfFirstFooter.height = minHeight;
                            odfFirstFooter.top = distanceTop;
                        } else if ((/even/i).test(operationType)) {
                            odfEvenFooter.height = minHeight;
                            odfEvenFooter.top = distanceTop;
                        } else {
                            odfDefFooter.height = minHeight;
                            odfDefFooter.top = distanceTop;
                        }
                    }
                }

            }

            if (app.isImportFinished()) {
                $paragraph = DOM.getMarginalContentNode(model.getRootNode(operationId)).children('.p');
                if (!$paragraph.length) {
                    $paragraph = $('<div>').addClass('p marginal').data('implicit', true);
                    DOM.getMarginalContentNode(model.getRootNode(operationId)).append($paragraph);
                }
                model.validateParagraphElement($paragraph);

                // height of inserted node needs to be stored, and active page height updated
                nodeHeight = (localCollection.node).outerHeight(true);
                if (operationType === 'header_first') {
                    heightOfFirstHeader = Math.max(pagePaddingTop, nodeHeight);
                    firstPageHeight = pageHeight - heightOfFirstHeader - Math.max(pagePaddingBottom, heightOfFirstFooter);
                } else if (operationType === 'header_even') {
                    heightOfEvenHeader = Math.max(pagePaddingTop, nodeHeight);
                    evenPageHeight = pageHeight - heightOfEvenHeader - Math.max(pagePaddingBottom, heightOfEvenFooter);
                } else if (operationType === 'header_default') {
                    heightOfDefHeader = Math.max(pagePaddingTop, nodeHeight);
                    defPageHeight = pageHeight - heightOfDefHeader - Math.max(pagePaddingBottom, heightOfDefFooter);
                } else if (operationType === 'footer_first') {
                    heightOfFirstFooter = Math.max(pagePaddingBottom, nodeHeight);
                    firstPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfFirstHeader) - heightOfFirstFooter;
                } else if (operationType === 'footer_even') {
                    heightOfEvenFooter = Math.max(pagePaddingBottom, nodeHeight);
                    evenPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfEvenHeader) - heightOfEvenFooter;
                } else {
                    heightOfDefFooter = Math.max(pagePaddingBottom, nodeHeight);
                    defPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfDefHeader) -  heightOfDefFooter;
                }

                if (undoRedoRunning) {
                    // replacing of all types of h/f, in case of undo/redo happens at the end,
                    // on triggering undo:after event
                    self.updateEditingHeaderFooter();
                } else {
                    replaceAllTypesOfHeaderFooters();
                }
            }

            return true;
        };

        /**
         * Function triggered when operation DELETE_HEADER_FOOTER is called
         *
         * @param {String} operationId
         *
         * @returns {Boolean}
         *  True if header or footer is succesfully removed from DOM and collection, otherwise false.
         *
         */
        this.implDeleteHeaderFooter = function (operationId) {

            if (_.findWhere(headerFooterCollection, { id: operationId }).length < 1) {
                Utils.warn('pageLayout.deleteHeaderFooter(): no object found with passed Id');
                return false;
            }

            var $node = model.getRootNode(operationId),
                pos = null,
                type = self.getTypeFromId(operationId),
                className = null,
                $nodeParent = $node.parent();

            // updating the models for comments (only ODT), range markers and complex fields. This
            // is not necessary for drawings, because header and footer have their own drawing layer.
            model.updateCollectionModels($node, { marginal: true });

            if (model.isHeaderFooterEditState()) {
                self.updateEditingHeaderFooter();
                self.leaveHeaderFooterEditMode();
                selection.setNewRootNode(model.getNode()); // restore original rootNode
            }

            if (type) {
                className = '.' + _.first(type.split('_'));
                if (app.isODF()) {
                    if ((/first/i).test(type)) {
                        firstPage = false;
                    } else if ((/even/i).test(type)) {
                        evenOddPages = false;
                    }
                }
            } else {
                Utils.warn('pageLayout.implDeleteHeaderFooter(): unknown type of node!');
            }

            if (className) {
                $pageContentNode.add($firstHeaderWrapper).add($lastFooterWrapper).find(className).filter('[data-container-id="' + operationId + '"]').removeAttr('data-container-id').empty();
            }

            $headerFooterPlaceHolder.children('[data-container-id="' + operationId + '"]').remove();
            headerFooterCollection = headerFooterCollection.filter(function (el) {
                return el.id !== operationId;
            });
            if (_.isEmpty(headerFooterCollection)) {
                marginalActivated = false;
            }

            // set selection
            if (DOM.isMarginalPlaceHolderNode($nodeParent)) {
                selection.setTextSelection(selection.getFirstDocumentPosition());
            } else {
                selection.setNewRootNode(model.getNode());

                if ($nodeParent.hasClass('header-wrapper')) {
                    $nodeParent = $nodeParent.next().children().first();
                } else if ($nodeParent.hasClass('footer-wrapper')) {
                    $nodeParent = $nodeParent.prev().children().last();
                } else if ($nodeParent.hasClass('page-break')) {
                    if ($nodeParent.parent().is('td')) {
                        $nodeParent = $nodeParent.closest('tr');
                        if ($nodeParent.next().length && !$nodeParent.next().hasClass('pb-row')) {
                            $nodeParent = $nodeParent.next().find('.p').first();
                        } else {
                            $nodeParent = $nodeParent.closest('table').next().find('.p').first();
                        }
                    } else {
                        $nodeParent = $nodeParent.next();
                    }
                }
                if ($nodeParent.length) {
                    pos = Position.getFirstPositionInParagraph(model.getNode(), Position.getOxoPosition(model.getNode(), $nodeParent, 0));
                    if (_.isArray(pos)) {
                        selection.setTextSelection(pos);
                    }
                } else {
                    selection.setTextSelection(selection.getFirstDocumentPosition());
                }
            }

            // height of deleted node needs to reset, to get accurate page height
            if (type === 'header_first') {
                heightOfFirstHeader = pagePaddingTop;
                firstPageHeight = pageHeight - heightOfFirstHeader - Math.max(pagePaddingBottom, heightOfFirstFooter);
            } else if (type === 'header_even') {
                heightOfEvenHeader = pagePaddingTop;
                evenPageHeight = pageHeight - heightOfEvenHeader - Math.max(pagePaddingBottom, heightOfEvenFooter);
            } else if (type === 'header_default') {
                heightOfDefHeader = pagePaddingTop;
                defPageHeight = pageHeight - heightOfDefHeader - Math.max(pagePaddingBottom, heightOfDefFooter);
            } else if (type === 'footer_first') {
                heightOfFirstFooter = pagePaddingBottom;
                firstPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfFirstHeader) - heightOfFirstFooter;
            } else if (type === 'footer_even') {
                heightOfEvenFooter = pagePaddingBottom;
                evenPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfEvenHeader) - heightOfEvenFooter;
            } else {
                heightOfDefFooter = pagePaddingBottom;
                defPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfDefHeader) -  heightOfDefFooter;
            }

            // repaint layout debounced
            model.insertPageBreaks();

            return true;
        };

        /**
         * This method is called after user pressed button to insert header on current page.
         * If header is already created in the document, it will just jump into it.
         *
         */
        this.insertHeader = function () {
            var $node,
                pageNum = self.getPageNumber(),
                options = { calledFromCreate: true },
                headerType = 'header_default',
                footerType = 'footer_default';

            if (pageNum === 1) {
                $node = $firstHeaderWrapper.children('.header');
                if (firstPage) {
                    headerType = 'header_first';
                    footerType = 'footer_first';
                }
            } else {
                $node = $(pageBreaksCollection[pageNum - 2]).children('.header');
                if (evenOddPages && (pageNum % 2 === 0)) {
                    headerType = 'header_even';
                    footerType = 'footer_even';
                }
            }
            if (model.isHeaderFooterEditState()) {
                self.updateEditingHeaderFooter();
                self.leaveHeaderFooterEditMode();
                selection.setNewRootNode(model.getNode()); // restore original rootNode
            }
            if (!$node.attr('data-container-id')) {
                storeNodeForHeaderFooterCreation($node);
                createOperationInsertHeaderFooter($node.hasClass('header') ? headerType : footerType);

                //replaceAllTypesOfHeaderFooters();
                $node = getNodeStoredOnHeaderFooterInsertion();
                storeNodeForHeaderFooterCreation(null); // node is used, clean reference
                if (!$node || !$node.length) {
                    $node = model.getCurrentRootNode();
                }
                self.enterHeaderFooterEditMode($node, options);
                selection.setNewRootNode($node);
                selection.setTextSelection(selection.getFirstDocumentPosition());
            } else {
                self.enterHeaderFooterEditMode($node);
                selection.setNewRootNode($node);
                selection.resetSelection();
                selection.setTextSelection(selection.getFirstDocumentPosition());
            }
        };

        /**
         * This method is called after user pressed button to insert footer on current page.
         * If footer is already created in the document, it will just jump into it.
         */
        this.insertFooter = function () {
            var $node,
                numPages = self.getNumberOfDocumentPages(),
                pageNum = self.getPageNumber(),
                options = { calledFromCreate: true },
                headerType = 'header_default',
                footerType = 'footer_default';

            if (pageNum === numPages) {
                $node = $lastFooterWrapper.children('.footer');
            } else {
                $node = $(pageBreaksCollection[pageNum - 1]).children('.footer');
            }
            if (firstPage && pageNum === 1) {
                headerType = 'header_first';
                footerType = 'footer_first';
            }
            if (evenOddPages && (pageNum % 2 === 0)) {
                headerType = 'header_even';
                footerType = 'footer_even';
            }

            if (model.isHeaderFooterEditState()) {
                self.updateEditingHeaderFooter();
                self.leaveHeaderFooterEditMode();
                selection.setNewRootNode(model.getNode()); // restore original rootNode
            }
            if (!$node.attr('data-container-id')) {
                storeNodeForHeaderFooterCreation($node);
                createOperationInsertHeaderFooter($node.hasClass('footer') ? footerType : headerType);

                //replaceAllTypesOfHeaderFooters();
                $node = getNodeStoredOnHeaderFooterInsertion();
                storeNodeForHeaderFooterCreation(null); // node is used, clean reference
                if (!$node || !$node.length) {
                    $node = model.getCurrentRootNode();
                }
                self.enterHeaderFooterEditMode($node, options);
                selection.setNewRootNode($node);
                selection.setTextSelection(selection.getFirstDocumentPosition());
            } else {
                self.enterHeaderFooterEditMode($node);
                selection.setNewRootNode($node);
                selection.resetSelection();
                selection.setTextSelection(selection.getFirstDocumentPosition());
            }
        };

        /**
         * Method that removes active header or footer, where cursor is currently placed.
         * It creates delete operation, and calls it's impl function deleteHeaderFooter.
         * After removing, cursor is returned to main document.
         *
         */
        this.removeActiveHeaderFooter = function () {
            var $node = model.getHeaderFooterRootNode(),
                localGenerator = model.createOperationsGenerator(),
                target = $node.attr('data-container-id'),
                operation = { id: target };

            localGenerator.generateOperation(Operations.DELETE_HEADER_FOOTER, operation);
            model.applyOperations(localGenerator);
        };

        /**
         * Method that removes all headers and footers from document.
         *
         */
        this.removeAllHeadersFooters = function () {
            var localGenerator = model.createOperationsGenerator();

            if (!headerFooterCollection.length) {
                return;
            }
            if (model.isHeaderFooterEditState()) {
                self.updateEditingHeaderFooter();
                self.leaveHeaderFooterAndSetCursor(model.getCurrentRootNode());
            }

            if (app.isODF()) {
                firstPage = false;
                evenOddPages = false;
            }

            _.each(headerFooterCollection, function (element) {
                localGenerator.generateOperation(Operations.DELETE_HEADER_FOOTER, { id: element.id });
            });
            model.applyOperations(localGenerator);

            // inform listeners, that the marginal nodes were removed
            model.trigger('removed:marginal');
        };

        /**
         * On document loading, using fast-load, references to created nodes in placeholder have to be updated.
         * Function is also used to update references after content update.
         *
         */
        this.headerFooterCollectionUpdate = function () {
            var
                placeholderChildren = $headerFooterPlaceHolder.children(),
                elementData,
                localCollection,
                $element;

            headerFooterCollection = [];
            targetIdNumbersCollection = [];

            _.each(placeholderChildren, function (element) {
                var minHeight,
                    distanceBottom,
                    distanceTop,
                    isHeader;

                $element = $(element);
                elementData = $element.data();

                if (_.isEmpty(elementData)) {
                    Utils.warn('pageLayout.headerFooterCollectionUpdate(): Empty object for:', $element);
                    return;
                } else if (_.isEmpty(elementData.type)) {
                    Utils.warn('pageLayout.headerFooterCollectionUpdate(): Empty Type for:', $element);
                    return;
                }
                if (!$element.children('.cover-overlay').length) {
                    $element.prepend($('<div>').addClass('cover-overlay').attr('contenteditable', false));
                }
                if (!DOM.getMarginalContentNode($element).length) {
                    $element.append($('<div>').addClass(DOM.MARGINALCONTENT_NODE_CLASSNAME));
                }
                if ($element.children('.p, table').length) {
                    $element.children('.p, table').detach().appendTo(DOM.getMarginalContentNode($element));
                }

                isHeader = (/header/i).test(elementData.type);

                localCollection = {
                        headFootType: elementData.type,
                        id: elementData.id,
                        node: $element.attr('data-container-id', elementData.id)
                    };
                headerFooterCollection.push(localCollection);
                if (elementData.id.match(/\d+/)) {
                    targetIdNumbersCollection.push(parseInt(elementData.id.match(/\d+/)));
                }
                if (app.isODF()) {
                    if ((/first/i).test(elementData.type)) {
                        firstPage = true;
                    } else if ((/even/i).test(elementData.type)) {
                        evenOddPages = true;
                    }

                    if (elementData.attributes && elementData.attributes.page) {
                        minHeight = elementData.attributes.page.minHeight ? Utils.convertHmmToLength(elementData.attributes.page.minHeight, 'px', 0.001) : 0;
                        distanceBottom = elementData.attributes.page.marginBottom ? Utils.convertHmmToLength(elementData.attributes.page.marginBottom, 'px', 0.001) : 0;
                        distanceTop = elementData.attributes.page.marginTop ? Utils.convertHmmToLength(elementData.attributes.page.marginTop, 'px', 0.001) : 0;
                        if (isHeader) {
                            if ((/first/i).test(elementData.type)) {
                                odfFirstHeader.height = minHeight;
                                odfFirstHeader.bottom = distanceBottom;
                            } else if ((/even/i).test(elementData.type)) {
                                odfEvenHeader.height = minHeight;
                                odfEvenHeader.bottom = distanceBottom;
                            } else {
                                odfDefHeader.height = minHeight;
                                odfDefHeader.bottom = distanceBottom;
                            }
                        } else {
                            if ((/first/i).test(elementData.type)) {
                                odfFirstFooter.height = minHeight;
                                odfFirstFooter.top = distanceTop;
                            } else if ((/even/i).test(elementData.type)) {
                                odfEvenFooter.height = minHeight;
                                odfEvenFooter.top = distanceTop;
                            } else {
                                odfDefFooter.height = minHeight;
                                odfDefFooter.top = distanceTop;
                            }
                        }
                    }
                }
            });
            targetIdNumbersCollection.sort(function (a, b) { return a - b; });

            // for local storage and fast load, enable flag if document is loaded with h/f
            if (!app.isImportFinished() && headerFooterCollection.length) {
                marginalActivated = true;
            }

        };

        /**
         * Returns information to controller, which state is currently set in document:
         *  - HF with first page different,
         *  - HF with even and odd pages different,
         *  - HF same across all document
         *  - no HF in document
         *
         * @returns {String}
         */
        this.getHeaderFooterTypeInDoc = function () {
            if (app.isODF()) {
                if (firstPage && evenOddPages) {
                    return 'all';
                } else if (firstPage) {
                    return 'first';
                } else if (evenOddPages) {
                    return 'evenodd';
                } else if (self.isHeaderFooterActivated()) {
                    return 'default';
                } else {
                    return 'none';
                }
            } else {
                if (self.getPageAttribute('firstPage') && self.getPageAttribute('evenOddPages')) {
                    return 'all';
                } else if (self.getPageAttribute('firstPage')) {
                    return 'first';
                } else if (self.getPageAttribute('evenOddPages')) {
                    return 'evenodd';
                } else if (self.isHeaderFooterActivated()) {
                    return 'default';
                } else {
                    return 'none';
                }
            }
        };

        /**
         * Sets the state of header&footer in document. Called from controller and context menu.
         *
         * @param {String} value
         *  Deppending on value passed, different h/f types in document.
         */
        this.setHeaderFooterTypeInDoc = function (value) {
            switch (value) {
            case 'none':
                model.getUndoManager().enterUndoGroup(function () {
                    if (firstPage) {
                        self.toggleFirstPageHF(false, { removeAll: true });
                    }
                    if (evenOddPages) {
                        self.toggleEvenOddHF(false, { removeAll: true });
                    }
                    self.removeAllHeadersFooters();
                });
                break;
            case 'default':
                model.getUndoManager().enterUndoGroup(function () {
                    prepareMarginalDropdown();

                    if (firstPage) {
                        self.toggleFirstPageHF(false);
                    }
                    if (evenOddPages) {
                        self.toggleEvenOddHF(false);
                    }
                });
                break;
            case 'first':
                model.getUndoManager().enterUndoGroup(function () {
                    prepareMarginalDropdown();

                    if (evenOddPages) {
                        self.toggleEvenOddHF(false);
                    }
                    self.toggleFirstPageHF(true);
                });
                break;
            case 'evenodd':
                model.getUndoManager().enterUndoGroup(function () {
                    prepareMarginalDropdown();

                    if (firstPage) {
                        self.toggleFirstPageHF(false);
                    }
                    self.toggleEvenOddHF(true);
                });
                break;
            case 'all':
                model.getUndoManager().enterUndoGroup(function () {
                    prepareMarginalDropdown();

                    self.toggleEvenOddHF(true);
                    self.toggleFirstPageHF(true);
                });
                break;
            case 'goto':
                self.insertHeader();
                break;

            default:
                self.insertHeader();
            }
        };

        /**
         * Gets deep copy the header or footer node's children from placeholder.
         *
         * @param {String} type
         *  Type of header/footer - can be default, first, or even.
         *
         * @returns {jQuery|Node}
         *  Returns copy of found node's children, or empty jQuery object
         *
         */
        this.getHeaderFooterChildrenFromPlaceholder = function (type) {
            if (type) {
                var obj = _.findWhere(headerFooterCollection, { headFootType: type });

                if (obj && obj.node) {
                    return (obj.node).children().clone(true);
                }
            }
            return $();
        };

        /**
         * Called from editor's function this.getRootNode, when target is header or footer type.
         * Returns found node, depending of document loading, remote client and passed index in editdiv.
         *
         * @param {String} target
         * @param {Number} index
         * @returns {jQuery} $localNode
         */
        this.getRootNodeTypeHeaderFooter = function (target, index) {
            var $localNode = null;

            function hfRootNodeWithIndex() {
                $localNode = model.getNode().find('.header, .footer').filter('[data-container-id="' + target + '"]').eq(index);
                if (!$localNode.length) {
                    Utils.warn('editor.getRootNode(): node with given index not found, fallback to first in document.');
                    $localNode = model.getNode().find('.header, .footer').filter('[data-container-id="' + target + '"]').first();
                }
            }

            if (app.isImportFinished() && model.getEditMode()) {
                if (model.isHeaderFooterEditState() && model.getActiveTarget() === target) {
                    $localNode = model.getCurrentRootNode(target);
                } else if (_.isNumber(index)) {
                    hfRootNodeWithIndex();
                } else {
                    $localNode = model.getNode().find('.header, .footer').filter('[data-container-id="' + target + '"]').first();
                }
            } else {
                // remote client
                if (_.isNumber(index)) {
                    hfRootNodeWithIndex();
                } else {
                    $localNode = $headerFooterPlaceHolder.children('[data-container-id="' + target + '"]').first();
                }
            }

            return $localNode;
        };

        /**
         * Gets number of the page where passed node, logical position, or selection start belongs to.
         *
         * @param {jQuery|Node|Number[]|undefined} [position]
         *  Can accept node, logical position as array, or no argument (then fetches element's selection where cursor is positioned)
         *
         * @returns {Number}
         *  Returns number of page where node is located. If no valid node is fetched, returns 0.
         *
         */
        this.getPageNumber = function (position) {
            var firstPageBreakAbove,
                parentSplit,
                pb,
                traverseUp = false,
                pageNumber = 1,
                helperNode = null;

            if (!position) {
                position = selection.getStartPosition();
            }

            if (_.isArray(position)) {
                position = Position.getDOMPosition(model.getCurrentRootNode(), position);
                helperNode = position ? position.node : null;
                if (helperNode && helperNode.nodeType === 3) {
                    position = helperNode.parentNode;
                }
            }

            if ($(position).length === 0) {
                Utils.warn('Unknown position of element!');
                pageNumber = 0;
                return pageNumber;
            }
            if (!pageBreaksCollection || pageBreaksCollection.length === 0) {
                pageNumber = 1;
                return pageNumber;
            }

            firstPageBreakAbove = $(position);

            // if header or footer is passed, calculate page num directly, otherwise traverse up to get to first element on page (bellow pagebreak)
            if (firstPageBreakAbove.hasClass('header')) {
                parentSplit = firstPageBreakAbove.parent();
                if (parentSplit.hasClass('header-wrapper')) {
                    pageNumber = 1;
                } else {
                    pageNumber = pageBreaksCollection.index(parentSplit) + 2;
                }
                return pageNumber;
            } else if (firstPageBreakAbove.hasClass('footer')) {
                parentSplit = firstPageBreakAbove.parent();
                if (parentSplit.hasClass('footer-wrapper')) {
                    pageNumber = pageBreaksCollection.length + 1;
                } else {
                    pageNumber =  pageBreaksCollection.index(parentSplit) + 1;
                }
                return pageNumber;
            } else {
                while (firstPageBreakAbove.length > 0 && !firstPageBreakAbove.hasClass('page-break')) { // fetching first page break element above passed element to fetch its index
                    if (DOM.nodeContainsPageBreak(firstPageBreakAbove) || firstPageBreakAbove.hasClass('tb-split-nb')) { // first level node contains page break inside itself
                        firstPageBreakAbove = firstPageBreakAbove.find('.page-break').last();
                        if (firstPageBreakAbove.length < 1) {
                            pageNumber = null;
                            return pageNumber;
                        }
                        if (traverseUp) {
                            pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 2) : 1;
                        } else {
                            pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 1) : 1;
                        }

                        return pageNumber;
                    } else {
                        if (firstPageBreakAbove.parent().hasClass('pagecontent')) { // first level node
                            firstPageBreakAbove = firstPageBreakAbove.prev();
                            traverseUp = true;
                        } else {
                            if (firstPageBreakAbove.parentsUntil('.page', '.pagecontent').length) {
                                parentSplit = firstPageBreakAbove.parentsUntil('.pagecontent').last();
                                pb = parentSplit.find('.page-break').first();
                            } else {
                                parentSplit = $();
                                pb = $();
                            }

                            if (DOM.nodeContainsPageBreak(parentSplit)) { // parent node is paragraph split by page break
                                while (firstPageBreakAbove.length > 0 && !firstPageBreakAbove.hasClass('page-break')) { // search above
                                    firstPageBreakAbove = firstPageBreakAbove.prev();
                                }
                                if (firstPageBreakAbove.length < 1) { // if none found, page break is bellow element
                                    if (pb.length < 1) {
                                        pageNumber = null;
                                        return pageNumber;
                                    }
                                    firstPageBreakAbove = pb;
                                    pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 1) : 1;
                                    return pageNumber;
                                }

                                pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 2) : 1;
                                return pageNumber;
                            } else if (parentSplit.hasClass('tb-split-nb')) {  // parent node is table split by page break
                                if (!firstPageBreakAbove.is('tr')) {
                                    firstPageBreakAbove = firstPageBreakAbove.parentsUntil('.pagecontent', 'tr').last();
                                }
                                while (firstPageBreakAbove.length > 0 && !firstPageBreakAbove.hasClass('pb-row')) { // search above
                                    firstPageBreakAbove = firstPageBreakAbove.prev();
                                }
                                if (firstPageBreakAbove.length < 1) { // if none found, page break is bellow element
                                    firstPageBreakAbove = pb;
                                    if (pb.length < 1) {
                                        pageNumber = null;
                                        return pageNumber;
                                    }

                                    pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 1) : 1;
                                    return pageNumber;
                                }
                                firstPageBreakAbove = firstPageBreakAbove.find('.page-break');

                                pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 2) : 1;
                                return pageNumber;
                            } else if (parentSplit.hasClass('page-break') && firstPageBreakAbove.length > 0) {
                                if (firstPageBreakAbove.parentsUntil(parentSplit, '.header').length || firstPageBreakAbove.hasClass('header')) {
                                    pageNumber = pageBreaksCollection.index(parentSplit) + 2;
                                } else if (firstPageBreakAbove.parentsUntil(parentSplit, '.footer').length || firstPageBreakAbove.hasClass('footer')) {
                                    pageNumber =  pageBreaksCollection.index(parentSplit) + 1;
                                }

                                return pageNumber;
                            } else if (parentSplit.length === 0 && firstPageBreakAbove.length > 0) {
                                if (firstPageBreakAbove.parentsUntil('.page', '.header-wrapper').length || firstPageBreakAbove.hasClass('header-wrapper')) { // first page header
                                    pageNumber = 1;
                                    return pageNumber;
                                } else if (firstPageBreakAbove.parentsUntil('.page', '.footer-wrapper').length || firstPageBreakAbove.hasClass('footer-wrapper')) { // last page footer
                                    pageNumber = pageBreaksCollection.length + 1;
                                    return pageNumber;
                                }
                            }

                            firstPageBreakAbove = parentSplit.prev();
                            traverseUp = true;
                        }
                    }
                } // end of while
            }
            pageNumber = firstPageBreakAbove.length > 0 ? (pageBreaksCollection.index(firstPageBreakAbove) + 2) : 1;
            return pageNumber;
        };

        /**
         * Method to get number of pages in whole document.
         *
         * @returns {Number}
         *
         */
        this.getNumberOfDocumentPages = function () {
            return pageCount;
        };

        /**
         * Method to get first element (line inside page break) on demanded page number.
         *
         * @param {Number} pageNumber
         *  Number of page for which we want begining of page element.
         *
         * @returns {jQuery|Node|Boolean}
         *  If found, it returns line node, if not, returns boolean false or null (if first page is demanded).
         *
         */
        this.getBeginingElementOnSpecificPage = function (pageNumber) {
            if (!_.isNumber(pageNumber) || pageNumber < 1 || pageNumber % 1 !== 0) {
                return false;
            }
            if (pageBreaksCollection.length === 0 || pageNumber === 1) {
                return null;
            }
            var $element = $(pageBreaksCollection[pageNumber - 2]).find(DOM.PAGE_BREAK_LINE_SELECTOR);

            if (!$element || $element.length < 1) {
                Utils.warn('Page doesn\'t exist!');
                return false;
            }

            return $element;

        };

        /**
         * Method for getting index of passed node from collection of all nodes with that target in whole document.
         * Collection is sorted from top to bottom of document.
         *
         * @param {jQuery} $node
         *
         * @param {String} target
         *
         * @returns {Number}
         *  Index (cardinal number) of passed node from collection of all nodes with that target in whole document,
         *  top to bottom; or -1 if no node index is found.
         */
        this.getTargetsIndexInDocument = function ($node, target) {
            return target ? model.getNode().find('[data-container-id="' + target + '"]').index($node) : -1;
        };

        /**
         * If parent container doesn't have style "position:relative",
         * safest way is to calculate difference of position.top,
         * between element and his parent.
         *
         * @returns {Number}
         *  Difference in px between top edge of element and top edge of it's parent.
         *
         */
        this.getDistanceTopRelativeToParent = function (element) {
            var $element = $(element),
                $parent = $element.parent();

            return ($parent.length > 0)  ? ($element.position().top - $parent.position().top) : 0;
        };

        /**
         * Getter method for active height in 1/100 mm, of default page (without margins) defined in page attributes
         *
         *  @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.convertToPixel=true]
         *      If set to true, the page height is calculated in pixel. Otherwise
         *      the returned value is in 1/100 mm.
         *
         * @returns {Number}
         *  The height of the page.
         */
        this.getDefPageActiveHeight = function (options) {
            var // whether the page height shall be returned in pixel
                convertToPixel =  Utils.getBooleanOption(options, 'convertToPixel', false);

            if (!pageAttributes) { // #35499 - if updateDocumentFormatting is not executed (start new empty doc), initialization of pageAttributes is required
                initializePageAttributes();
            }

            return convertToPixel ? (pageHeight - pagePaddingTop - pagePaddingBottom) : pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom;
        };

        /**
         * Getter method for page attributes
         *
         * @param {String} property
         *  The property for that the value shall be returned
         *
         * @returns {Number}
         *  The value for the specified page property.
         */
        this.getPageAttribute = function (property) {

            if (!pageAttributes) {
                initializePageAttributes();
            }

            return pageAttributes && pageAttributes.page && pageAttributes.page[property];
        };

        /**
         * Public function that updates value of field, with type page count,
         * based on frontend pageCount calculus.
         *
         * @param {jQuery|Node} field
         *  Field whose value is going to be updated.
         *
         */
        this.updatePageCountField = function (field) {
            $(field).children().empty().html(pageCount);
        };

        /**
         * Public function that updates value of field, with type page number,
         * based on frontend function getPageNumber.
         *
         * @param {jQuery|Node} field
         *  Field whose value is going to be updated.
         *
         */
        this.updatePageNumberField = function (field) {
            $(field).children().empty().html(self.getPageNumber(field));
        };

        /**
         * Callback function for mouseenter event on user type fields.
         * Creates popup with name of user defined field.
         *
         * @param event
         */
        this.createTooltipForField = function (event) {
            var
                $fieldNode = $(event.currentTarget),
                name = $fieldNode.data('name') || '',
                $marginalNode = null,
                isMarginal = DOM.isMarginalNode($fieldNode),
                contentNodeSelector = isMarginal ? '.marginalcontent' : '.pagecontent',
                zoomFactor = app.getView().getZoomFactor() / 100,
                tooltipTopPos = ($fieldNode.parentsUntil(contentNodeSelector).last().position().top + $fieldNode.position().top + $fieldNode.children().position().top) / zoomFactor,
                tooltipLeftPos = (event.clientX - rootNode.offset().left) / zoomFactor,
                $tooltip = $('<div>').addClass('field-tooltip-box').attr('contenteditable', 'false').text(Utils.escapeHTML(name));

            // fastload stores different data structure of attributes
            if (name.length === 0 && _.isObject($fieldNode.data('attributes')) && _.isObject($fieldNode.data('attributes').field) && $fieldNode.data('attributes').field.name) {
                $tooltip.text($fieldNode.data('attributes').field.name);
            }

            if (document.documentElement.clientHeight - event.clientY - $fieldNode.children().height() * zoomFactor < 30 * zoomFactor) {
                // tooltip goes above field
                tooltipTopPos = tooltipTopPos - $tooltip.outerHeight() - 15;
            } else {
                tooltipTopPos = tooltipTopPos + $fieldNode.children().height();
            }
            if (isMarginal) {
                $marginalNode = $fieldNode.closest('.header, .footer');
                if (DOM.isHeaderNode($marginalNode) && !DOM.isHeaderWrapper($marginalNode.parent())) {
                    tooltipTopPos += $marginalNode.position().top;
                }
                $tooltip.css('z-index', 41); // to go over marginal context menu, which has z-40
            }

            $tooltip.css({ top: tooltipTopPos, left: tooltipLeftPos });

            if (isMarginal) {
                $marginalNode.after($tooltip); // attach after header or footer node
                $marginalFieldTooltip = $tooltip;
            } else {
                if ($tooltipOverlay.length === 0) {
                    createTooltipOverlay();
                }
                $tooltipOverlay.append($tooltip);
            }
        };

        /**
         * Callback function for mouseleave event on user type fields.
         * Removes popup with name of user defined field.
         *
         */
        this.destroyTooltipForField = function () {
            $tooltipOverlay.empty();
            if ($marginalFieldTooltip) {
                $marginalFieldTooltip.remove();
                $marginalFieldTooltip = null;
            }
        };

        /**
         * Current header or footer that we are editing, needs to be updated in collection also.
         * If height of element is changed, update this value also, it is needed for correct page height calculus.
         *
         */
        this.updateEditingHeaderFooter = function () {
            Utils.takeTime('pageLayout.updateEditingHeaderFooter(): ', function () {
                var $node = ($(model.getTargetNodeForUpdate()).length && $(model.getTargetNodeForUpdate()).parent().length) ? $(model.getTargetNodeForUpdate()) : model.getHeaderFooterRootNode(), // check if not already detached
                    $replacementChildren = null,
                    nodeHeight = $node.outerHeight(true),
                    $coverOverlay = $('<div>').addClass('cover-overlay').attr('contenteditable', false),
                    collectionNodes,
                    target = model.getActiveTarget() || $node.attr('data-container-id') || '', // if at this point, target is already removed, we fetch it from node
                    obj = _.findWhere(headerFooterCollection, { id: target }),
                    type = '',
                    selectorHF = '',
                    repaintLayout = false,
                    // variables used for possible search results
                    invokeSearch = false,
                    searchSpans = null,
                    searchValue = '';

                if (model.isClipboardPasteInProgress()) {
                    // #38528 - Don't run update if paste is in progress, just deffer it.
                    self.executeDelayed(function () {
                        self.updateEditingHeaderFooter();
                    }, undefined, 'Text: pageLayout: updateEditingHeaderFooter');
                    return;
                }

                if (obj && obj.headFootType) {
                    type = obj.headFootType;
                    selectorHF = (type.indexOf('header') > -1) ? '.header' : '.footer';
                }

                if (!model.isHeaderFooterEditState()) {
                    switch (type) {
                    case 'header_first':
                        if (nodeHeight !== heightOfFirstHeader) {
                            heightOfFirstHeader = Math.max(pagePaddingTop, (odfFirstHeader.height + pagePaddingTop), nodeHeight);
                            firstPageHeight = pageHeight - heightOfFirstHeader - Math.max(pagePaddingBottom, heightOfFirstFooter);
                            repaintLayout = true;
                        }
                        break;
                    case 'header_even':
                        if (nodeHeight !== heightOfEvenHeader) {
                            heightOfEvenHeader = Math.max(pagePaddingTop, (odfEvenHeader.height + pagePaddingTop), nodeHeight);
                            evenPageHeight = pageHeight - heightOfEvenHeader - Math.max(pagePaddingBottom, heightOfEvenFooter);
                            repaintLayout = true;
                        }
                        break;
                    case 'header_default':
                        if (nodeHeight !== heightOfDefHeader) {
                            heightOfDefHeader = Math.max(pagePaddingTop, (odfDefHeader.height + pagePaddingTop), nodeHeight);
                            defPageHeight = pageHeight - heightOfDefHeader - Math.max(pagePaddingBottom, heightOfDefFooter);
                            repaintLayout = true;
                        }
                        break;
                    case 'footer_first':
                        if (nodeHeight !== heightOfFirstFooter) {
                            heightOfFirstFooter = Math.max(pagePaddingBottom, (odfFirstFooter.height + pagePaddingBottom), nodeHeight);
                            firstPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfFirstHeader) - heightOfFirstFooter;
                            repaintLayout = true;
                        }
                        break;
                    case 'footer_even':
                        if (nodeHeight !== heightOfEvenFooter) {
                            heightOfEvenFooter = Math.max(pagePaddingBottom, (odfEvenFooter.height + pagePaddingBottom), nodeHeight);
                            evenPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfEvenHeader) - heightOfEvenFooter;
                            repaintLayout = true;
                        }
                        break;
                    case 'footer_default':
                        if (nodeHeight !== heightOfDefFooter) {
                            heightOfDefFooter = Math.max(pagePaddingBottom, (odfDefFooter.height + pagePaddingBottom), nodeHeight);
                            defPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfDefHeader) -  heightOfDefFooter;
                            repaintLayout = true;
                        }
                        break;
                    default:
                        Utils.log('pagelayout.updateEditingHeaderFooter(): Unknown type of Header-Footer!');
                        return;
                    }
                }

                markMarginalNodesInsideHeadFoot($node);

                if ($node.length > 0) {
                    // find highlighted search words in header/footer, and clean them before cloning to template node
                    searchSpans = $node.find('.highlight');
                    if (searchSpans.length) {
                        invokeSearch = true;
                        searchValue = searchSpans.first().text();
                        searchSpans.removeClass('highlight');

                        model.getSearchHandler().removeSpansHighlighting(searchSpans);
                    }

                    $replacementChildren = $node.children().clone(true);
                    $headerFooterPlaceHolder.children('[data-container-id="' + target + '"]').empty().append($replacementChildren);

                    // update refrence in collection because of replaced node
                    self.headerFooterCollectionUpdate();
                }
                if (repaintLayout && !model.isHeaderFooterEditState()) {
                    model.insertPageBreaks();
                    // we need to update first header, or last footer in document, because they are not handled with insertPageBreaks function
                    if (selectorHF === '.header') {
                        collectionNodes = $firstHeaderWrapper.children('[data-container-id="' + target + '"]').not('.active-selection');
                        refreshCollectionNodes(collectionNodes, $replacementChildren, $coverOverlay);
                    } else {
                        collectionNodes = $lastFooterWrapper.children('[data-container-id="' + target + '"]').not('.active-selection');
                        refreshCollectionNodes(collectionNodes, $replacementChildren, $coverOverlay);
                    }
                } else { // just replace header-footer content
                    collectionNodes = $pageContentNode.add($firstHeaderWrapper).add($lastFooterWrapper).find(selectorHF).not('.active-selection').filter('[data-container-id="' + target + '"]');
                    refreshCollectionNodes(collectionNodes, $replacementChildren, $coverOverlay);

                    if (!fieldManager.isComplexFieldEmpty()) {
                        fieldManager.updatePageFieldsInMarginals();
                    }
                }

                if (invokeSearch) {
                    // invoke search again to refresh detached and invalid references
                    model.getSearchHandler().quickSearch(searchValue, null);
                }

                // inform listeners about modified header and footer in the case that header or footer are no longer active
                if (!model.isHeaderFooterEditState()) {
                    self.trigger('updateMarginal:leave');
                }

            });
        };

        /**
         * Method to trigger update of template header/footer node with changes in first node inside main document.
         * Also distribute changes to all other nodes of same type.
         * It is necessary to trigger this update after updateAbsoluteDrawings make changes in node.
         *
         * @param {Array|HTMLElement|jQuery} nodesCollection
         *  List of unique, first appearence header/footer nodes inside main document,
         *  that contains spacemaker for absolute positioned drawings.
         *  Currently, up to 6 elements can be in array (3 unique header, and 3 footer types).
         *
         */
        this.updateDataFromFirstNodeInDoc = function (nodesCollection) {
            Utils.takeTime('pageLayout.updateDataFromFirstNodeInDoc(): ', function () {
                var
                    $node = null,
                    $replacementChildren = null,
                    $coverOverlay = $('<div>').addClass('cover-overlay').attr('contenteditable', false),
                    collectionNodes,
                    nodeHeight = 0,
                    target = '',
                    obj = null,
                    type = '',
                    repaintLayout = false;

                if (nodesCollection && nodesCollection.length) {
                    _.each(nodesCollection, function (node) {
                        $node = $(node);
                        nodeHeight = $node.outerHeight(true);
                        target = $node.attr('data-container-id') || '';
                        // first check if node is not detached from DOM;
                        // If it is, find new one in DOM with same target
                        if (!$node.parent().length) {
                            $node = model.getRootNode(target);
                        }
                        obj = _.findWhere(headerFooterCollection, { id: target });

                        if (obj && obj.headFootType) {
                            type = obj.headFootType;
                        }

                        if (!model.isHeaderFooterEditState()) {
                            switch (type) {
                            case 'header_first':
                                if (nodeHeight !== heightOfFirstHeader) {
                                    heightOfFirstHeader = Math.max(pagePaddingTop, (odfFirstHeader.height + pagePaddingTop), nodeHeight);
                                    firstPageHeight = pageHeight - heightOfFirstHeader - Math.max(pagePaddingBottom, heightOfFirstFooter);
                                    repaintLayout = true;
                                }
                                break;
                            case 'header_even':
                                if (nodeHeight !== heightOfEvenHeader) {
                                    heightOfEvenHeader = Math.max(pagePaddingTop, (odfEvenHeader.height + pagePaddingTop), nodeHeight);
                                    evenPageHeight = pageHeight - heightOfEvenHeader - Math.max(pagePaddingBottom, heightOfEvenFooter);
                                    repaintLayout = true;
                                }
                                break;
                            case 'header_default':
                                if (nodeHeight !== heightOfDefHeader) {
                                    heightOfDefHeader = Math.max(pagePaddingTop, (odfDefHeader.height + pagePaddingTop), nodeHeight);
                                    defPageHeight = pageHeight - heightOfDefHeader - Math.max(pagePaddingBottom, heightOfDefFooter);
                                    repaintLayout = true;
                                }
                                break;
                            case 'footer_first':
                                if (nodeHeight !== heightOfFirstFooter) {
                                    heightOfFirstFooter = Math.max(pagePaddingBottom, (odfFirstFooter.height + pagePaddingBottom), nodeHeight);
                                    firstPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfFirstHeader) - heightOfFirstFooter;
                                    repaintLayout = true;
                                }
                                break;
                            case 'footer_even':
                                if (nodeHeight !== heightOfEvenFooter) {
                                    heightOfEvenFooter = Math.max(pagePaddingBottom, (odfEvenFooter.height + pagePaddingBottom), nodeHeight);
                                    evenPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfEvenHeader) - heightOfEvenFooter;
                                    repaintLayout = true;
                                }
                                break;
                            case 'footer_default':
                                if (nodeHeight !== heightOfDefFooter) {
                                    heightOfDefFooter = Math.max(pagePaddingBottom, (odfDefFooter.height + pagePaddingBottom), nodeHeight);
                                    defPageHeight = pageHeight - Math.max(pagePaddingTop, heightOfDefHeader) -  heightOfDefFooter;
                                    repaintLayout = true;
                                }
                                break;
                            default:
                                Utils.warn('pagelayout.updateDataFromFirstNodeInDoc(): Unknown type of Header-Footer!');
                                return;
                            }
                        }

                        markMarginalNodesInsideHeadFoot($node);

                        if ($node.length > 0) {
                            $replacementChildren = $node.children().clone(true);
                            $headerFooterPlaceHolder.children('[data-container-id="' + target + '"]').empty().append($replacementChildren);
                            // update refrence in collection because of replaced node
                            self.headerFooterCollectionUpdate();
                        }
                        if (!repaintLayout) {
                            collectionNodes = $pageContentNode.add($firstHeaderWrapper).add($lastFooterWrapper).find('.header, .footer').not('.active-selection').filter('[data-container-id="' + target + '"]');
                            refreshCollectionNodes(collectionNodes, $replacementChildren, $coverOverlay);
                            if (!fieldManager.isComplexFieldEmpty()) {
                                fieldManager.updatePageFieldsInMarginals();
                            }
                        }
                    });
                    if (repaintLayout && !model.isHeaderFooterEditState()) {
                        model.insertPageBreaks();
                        // we need to update first header, or last footer in document, because they are not handled with insertPageBreaks function
                        if (type.indexOf('header') > -1) {
                            collectionNodes = $firstHeaderWrapper.children('[data-container-id="' + target + '"]').not('.active-selection');
                            refreshCollectionNodes(collectionNodes, $replacementChildren, $coverOverlay);
                        } else {
                            collectionNodes = $lastFooterWrapper.children('[data-container-id="' + target + '"]').not('.active-selection');
                            refreshCollectionNodes(collectionNodes, $replacementChildren, $coverOverlay);
                        }
                    }

                } else {
                    Utils.warn('pageLayout.updateDataFromFirstNodeInDoc(): Passed empty collection of nodes as argument.');
                }
            });
        };

        /**
         * Method to jump into necessary edit mode of header/footer, from main document programatically.
         *
         * @param {jQuery} $node
         *  Header or footer root node
         *
         * @param {Object} [options]
         *  @param {Boolean} [options.calledFromCreate]
         *
         */
        this.enterHeaderFooterEditMode = function ($node, options) {
            var nodeType,
                $marginalContent;

            if (!model.getEditMode()) {
                return;
            }

            if (!$node.attr('data-container-id')) {
                if (options && options.calledFromCreate) {
                    Utils.warn('pageLayout.enterHeaderFooterEditMode(): tried to call itself recursively!');
                    return;
                }
                nodeType = getHeadFootTypeForCreateOp($node);
                Utils.info('pageLayout.enterHeaderFooterEditMode(): Node not existing yet. Creating node with type:', nodeType);
                createOperationInsertHeaderFooter(nodeType);

            } else if (model.isHeaderFooterEditState() && $node.length) {
                if (model.getHeaderFooterRootNode()[0] === $node[0]) {
                    Utils.info('pageLayout.enterHeaderFooterEditMode(): Already in edit mode with passed node!');
                    return;
                }
                self.leaveHeaderFooterEditMode();
                Utils.info('pageLayout.enterHeaderFooterEditMode(): Not the same node as passed is in edit mode!');
            }

            model.setHeaderFooterEditState(true);
            model.setActiveTarget($node.attr('data-container-id'));

            // take care of dom and css
            $node.removeClass('inactive-selection').addClass('active-selection').attr('contenteditable', true);
            insertContextMenu($node, $node.hasClass('header'), getHeadFootTypeForCreateOp($node));

            $node.find('[contenteditable="false"]').not('.inline').attr('contenteditable', true);
            $node.find('.cover-overlay').remove();
            $marginalContent = DOM.getMarginalContentNode($node);
            // node must have at least implicit paragraph to set cursor inside
            if (!$marginalContent.children().length) {
                $marginalContent.append($('<div>').addClass('p').data('implicit', true));
                model.validateParagraphElement($marginalContent.children());
            }
            model.getNode().addClass('inactive-mode').removeAttr('contenteditable');

            model.setHeaderFooterRootNode($node);

            // browser specific handling
            if (_.browser.IE) {
                _.each($node.find('span'), function (span) {
                    DOM.ensureExistingTextNode(span);
                });

                $node.parent().attr('contenteditable', false);
                $node.find('.cell').attr('contenteditable', false);
                $node.find('.textframecontent').parent().attr('contenteditable', false);
            } else if (_.browser.Firefox) {
                $node.parent().removeAttr('contenteditable');
            }

        };

        /**
         * Method to leave edit state of header/footer that was edited. Css classes are beeing cleaned up.
         *
         * @param {jQuery} [$node]
         *  Header or footer root node
         *
         * @param {jQuery} [$parent]
         *  Parent container of root node
         */
        this.leaveHeaderFooterEditMode = function ($node, $parent) {
            var selectedDrawing = null;

            if (model.isHeaderFooterEditState()) {
                if (!$node || !$node.length) {
                    var $node = model.getCurrentRootNode(),
                        $parent = $node.parent();
                }
                if (!$parent) {
                    $parent = $node.parent();
                }
                // clear possible leftover selection on drawings and textframes
                if (selection.getSelectionType() === 'drawing' || selection.isAdditionalTextframeSelection()) {
                    selectedDrawing = selection.getSelectedDrawing();

                    if (!selectedDrawing.length) {
                        selectedDrawing =  selection.getSelectedTextFrameDrawing();
                    }

                    DrawingFrame.clearSelection(selectedDrawing);
                    // removing additional tracking handler options, that are specific for OX Text (Chrome only)
                    selectedDrawing.removeData('trackingOptions');
                    selectedDrawing.removeData('tracking-options');
                    selectedDrawing.closest('.p').removeClass('visibleAnchor').find('.anchorIcon').remove();
                }
                $node.find('table.selected').removeClass('selected');

                $node.removeClass('active-selection').addClass('inactive-selection').attr('contenteditable', false);
                $node.find('[contenteditable="true"]').attr('contenteditable', false);

                if ($node.hasClass('header')) {
                    contextHeaderMenu.detach();
                    contextHeaderMenu.find('.marginal-container').hide();
                    contextHeaderMenu.find('.marginal-menu-type').removeClass('active-dropdown');
                } else {
                    contextFooterMenu.detach();
                    contextFooterMenu.find('.marginal-container').hide();
                    contextFooterMenu.find('.marginal-menu-type').removeClass('active-dropdown');
                }

                $node.prepend($('<div>').addClass('cover-overlay').attr('contenteditable', false));

                if (_.browser.IE || _.browser.Firefox) {
                    $parent.attr('contenteditable', '');
                }
                if (fieldManager.isHighlightState()) {
                    fieldManager.removeEnterHighlight();
                }
                if ($marginalFieldTooltip) {
                    $marginalFieldTooltip.remove();
                    $marginalFieldTooltip = null;
                }
                if (!fieldManager.isSimpleFieldEmpty()) {
                    updatePageCountFields();
                    updatePageNumberFields();
                }
                if (!fieldManager.isComplexFieldEmpty()) {
                    fieldManager.updatePageFieldsInMarginals();
                }

                model.getNode().removeClass('inactive-mode').attr('contenteditable', true);
                model.setActiveTarget(null);
                model.setHeaderFooterEditState(false);

            } else {
                Utils.warn('pageLayout.leaveHeaderFooterEditMode(): Not in edit mode currently!');
                return;
            }
        };

        /**
         * On leaving header/footer edit state and returning to main document, calculate where to place cursor.
         * If we are leaving header, cursor will be placed in first position on that page, bellow header.
         * If we are leaving footer, cursor will be placed in last postion on that page, above footer.
         *
         * @param {Node|jQuery} activeRootNode
         *  Header or footer node that is currently in active mode, and which we are leaving.
         *
         * @param {Node|jQuery} [parentOfRootNode]
         *  Optional node, parent node of activeRootNode. If not provided, will be fetched inside function.
         *
         */
        this.leaveHeaderFooterAndSetCursor = function (activeRootNode, parentOfRootNode) {
            var pos,
                isHeader = activeRootNode.hasClass('header');

            if (!parentOfRootNode) {
                parentOfRootNode = activeRootNode.parent();
            }

            self.leaveHeaderFooterEditMode(activeRootNode, parentOfRootNode);
            self.updateEditingHeaderFooter();

            selection.setNewRootNode(model.getNode()); // restore original rootNode

            // set selection
            if (parentOfRootNode.hasClass('header-wrapper')) {
                parentOfRootNode = parentOfRootNode.next().children().first();
            } else if (parentOfRootNode.hasClass('footer-wrapper')) {
                parentOfRootNode = parentOfRootNode.prev().children().last();
            } else if (parentOfRootNode.hasClass('page-break')) {
                if (parentOfRootNode.parent().is('td')) {
                    parentOfRootNode = parentOfRootNode.closest('tr');
                    if (isHeader) {
                        if (parentOfRootNode.next().length && !parentOfRootNode.next().hasClass('pb-row')) {
                            parentOfRootNode = parentOfRootNode.next().find('.p').first();
                        } else {
                            parentOfRootNode = parentOfRootNode.closest('table').next().find('.p').first();
                        }
                    } else {
                        if (parentOfRootNode.prev().length && !parentOfRootNode.prev().hasClass('pb-row')) {
                            parentOfRootNode = parentOfRootNode.prev().find('.p').last();
                        } else {
                            parentOfRootNode = parentOfRootNode.closest('table').prev().find('.p').last();
                        }
                    }
                } else {
                    parentOfRootNode = isHeader ? parentOfRootNode.next() : parentOfRootNode.prev();
                }
            }
            if (parentOfRootNode.length) {
                if (DOM.isParagraphNode(parentOfRootNode)) {
                    pos = isHeader ? Position.getFirstPositionInParagraph(model.getNode(), Position.getOxoPosition(model.getNode(), parentOfRootNode, 0)) : Position.getLastPositionInParagraph(model.getNode(), Position.getOxoPosition(model.getNode(), parentOfRootNode, 0));
                } else {
                    pos = Position.getOxoPosition(model.getNode(), parentOfRootNode, 0);
                }

                if (_.isArray(pos)) {
                    selection.setTextSelection(pos);
                }
            }
        };

        /**
         * Method to switch edit mode from main document to header,
         * on the page where selection is.
         *
         * @param {Number} [pageNumber]
         *  Already calculated page number where to jump - optional
         */
        this.jumpToHeaderOnCurrentPage = function (pageNumber) {
            var pageNum = pageNumber || self.getPageNumber();

            if (pageNum === 1) {
                self.enterHeaderFooterEditMode($firstHeaderWrapper.children('.header'));
            } else {
                self.enterHeaderFooterEditMode($(pageBreaksCollection[pageNum - 2]).children('.header'));
            }
            selection.resetSelection();
            selection.setTextSelection(selection.getFirstDocumentPosition());
            app.getView().scrollToChildNode(model.getCurrentRootNode(), { regardSoftkeyboard: true });
        };

        /**
         * Method to switch edit mode from main document to footer,
         * on the page where selection is.
         *
         * @param {Number} [pageNumber]
         *  Already calculated page number where to jump - optional
         */
        this.jumpToFooterOnCurrentPage = function (pageNumber) {
            var numPages = self.getNumberOfDocumentPages(),
                pageNum = pageNumber || self.getPageNumber();

            if (pageNum === numPages) {
                self.enterHeaderFooterEditMode($lastFooterWrapper.children('.footer'));
            } else {
                self.enterHeaderFooterEditMode($(pageBreaksCollection[pageNum - 1]).children('.footer'));
            }
            selection.resetSelection();
            selection.setTextSelection(selection.getFirstDocumentPosition());
            // next element from footer is pb line, because we want whole footer to be displayed
            // but only if is not last footer - which doesn't have pb line next.
            app.getView().scrollToChildNode(pageNum === numPages ? model.getCurrentRootNode() : model.getCurrentRootNode().next(), { regardSoftkeyboard: true });
        };

        /**
         * Method to programatically jump from one Header/footer to another.
         * Used for highlighting nodes in search, or changetracking.
         *
         * @param {jQuery} $node
         *  Header or footer root node
         */
        this.switchTargetNodeInHeaderMode = function ($node) {
            self.leaveHeaderFooterEditMode();
            //self.updateEditingHeaderFooter();

            self.enterHeaderFooterEditMode($node);
            selection.setNewRootNode($node); // set new selection root node
        };

        /**
         * Public method called when user clicks on checklist for header/footer different on first page.
         *
         * @param {Boolean} state
         *
         * @param {Object} [options]
         * Optional parameters:
         *  @param {Boolean} [options.removeAll=false]
         *      If set to true, this method call is precceding removing all HF in doc,
         *      and therefore there is no need to create default type before removing them all.
         */
        this.toggleFirstPageHF = function (state, options) {
            if (state === firstPage) {
                return;
            }
            var $rootNode = model.getHeaderFooterRootNode(),
                isHeader = $rootNode.hasClass('header'),
                activeNodeType = isHeader ? 'header_first' : 'footer_first',
                passiveNodeType = isHeader ? 'footer_first' : 'header_first',
                defNodeType = isHeader ? 'header_default' : 'footer_default',
                $activePlaceholderNode = $headerFooterPlaceHolder.children('.' + activeNodeType),
                $passivePlaceholderNode = $headerFooterPlaceHolder.children('.' + passiveNodeType),
                $defPlaceholderNode = $headerFooterPlaceHolder.children('.' + defNodeType),
                localGenerator = app.isODF() ? model.createOperationsGenerator() : null,
                odfRestoredNodePasive,
                odfRestoredNodeActive,
                removeAll = Utils.getBooleanOption(options, 'removeAll', false);

            if (state) {
                if (!$passivePlaceholderNode.length) {
                    if (app.isODF()) {
                        odfRestoredNodePasive = savedNodesCollection[passiveNodeType];
                        if (odfRestoredNodePasive) {
                            if (localGenerator) {
                                insertHFopODT(odfRestoredNodePasive, passiveNodeType);
                            }
                            delete savedNodesCollection[passiveNodeType];
                        } else {
                            createOperationInsertHeaderFooter(passiveNodeType);
                        }
                    } else {
                        createOperationInsertHeaderFooter(passiveNodeType);
                    }
                }
                if (!$activePlaceholderNode.length) {
                    if (app.isODF()) {
                        odfRestoredNodeActive = savedNodesCollection[activeNodeType];
                        if (odfRestoredNodeActive) {
                            if (localGenerator) {
                                insertHFopODT(odfRestoredNodeActive, activeNodeType);
                            }
                            delete savedNodesCollection[activeNodeType];
                        } else {
                            createOperationInsertHeaderFooter(activeNodeType);
                        }
                    } else {
                        createOperationInsertHeaderFooter(activeNodeType);
                    }
                }

                if (localGenerator) {
                    model.applyOperations(localGenerator);
                }

                if (odfRestoredNodePasive) {
                    $headerFooterPlaceHolder.children('.' + passiveNodeType).empty().append(odfRestoredNodePasive.children());
                    replaceAllTypesOfHeaderFooters();
                }
                if (odfRestoredNodeActive) {
                    $headerFooterPlaceHolder.children('.' + activeNodeType).empty().append(odfRestoredNodeActive.children());
                    replaceAllTypesOfHeaderFooters();
                }
            } else {
                // on de-checking type of HF, create default if not existing, before placing cursor inside
                if (!$defPlaceholderNode.length && !removeAll) {
                    createOperationInsertHeaderFooter(defNodeType);
                }
            }

            if (app.isODF()) {
                firstPage = state;
                if (!state && !removeAll) {
                    if ($activePlaceholderNode.length) {
                        // save node reference to restore it later
                        savedNodesCollection[activeNodeType] = $activePlaceholderNode;
                        localGenerator.generateOperation(Operations.DELETE_HEADER_FOOTER, { id: $activePlaceholderNode.attr('data-container-id') });
                    }
                    if ($passivePlaceholderNode.length) {
                        // save node reference to restore it later
                        savedNodesCollection[passiveNodeType] = $passivePlaceholderNode;
                        localGenerator.generateOperation(Operations.DELETE_HEADER_FOOTER, { id: $passivePlaceholderNode.attr('data-container-id') });
                    }
                    model.applyOperations(localGenerator);
                }
                self.implToggleFirstPageHF(state, { removeAll: true });
            } else {
                model.applyOperations({ name: Operations.SET_DOCUMENT_ATTRIBUTES, attrs: { page: { firstPage: state } } });
            }
            selection.setTextSelection(selection.getFirstDocumentPosition());
        };

        /**
         * Public method called when user clicks on checklist for headers/footers different on even and odd pages.
         *
         * @param {Boolean} state
         *
         * @param {Object} [options]
         * Optional parameters:
         *  @param {Boolean} [options.removeAll=false]
         *      If set to true, this method call is precceding removing all HF in doc,
         *      and therefore there is no need to create default type before removing them all.
         */
        this.toggleEvenOddHF = function (state, options) {
            if (state === evenOddPages) {
                return;
            }
            var $rootNode = model.getHeaderFooterRootNode(),
                isHeader = $rootNode.hasClass('header'),
                activeNodeType = isHeader ? 'header_even' : 'footer_even',
                passiveNodeType = isHeader ? 'footer_even' : 'header_even',
                defNodeType = isHeader ? 'header_default' : 'footer_default',
                $activePlaceholderNode = $headerFooterPlaceHolder.children('.' + activeNodeType),
                $passivePlaceholderNode = $headerFooterPlaceHolder.children('.' + passiveNodeType),
                $defPlaceholderNode = $headerFooterPlaceHolder.children('.' + defNodeType),
                localGenerator = app.isODF() ? model.createOperationsGenerator() : null,
                odfRestoredNodePasive,
                odfRestoredNodeActive,
                removeAll = Utils.getBooleanOption(options, 'removeAll', false);

            if (state) {
                if (!$passivePlaceholderNode.length) {
                    if (app.isODF()) {
                        odfRestoredNodePasive = savedNodesCollection[passiveNodeType];
                        if (odfRestoredNodePasive) {
                            if (localGenerator) {
                                insertHFopODT(odfRestoredNodePasive, passiveNodeType);
                            }
                            delete savedNodesCollection[passiveNodeType];
                        }  else {
                            createOperationInsertHeaderFooter(passiveNodeType);
                        }
                    } else {
                        createOperationInsertHeaderFooter(passiveNodeType);
                    }
                }
                if (!$activePlaceholderNode.length) {
                    if (app.isODF()) {
                        odfRestoredNodeActive = savedNodesCollection[activeNodeType];
                        if (odfRestoredNodeActive) {
                            if (localGenerator) {
                                insertHFopODT(odfRestoredNodeActive, activeNodeType);
                            }
                            delete savedNodesCollection[activeNodeType];
                        }  else {
                            createOperationInsertHeaderFooter(activeNodeType);
                        }
                    } else {
                        createOperationInsertHeaderFooter(activeNodeType);
                    }
                }

                if (localGenerator) {
                    model.applyOperations(localGenerator);
                }

                if (odfRestoredNodePasive) {
                    $headerFooterPlaceHolder.children('.' + passiveNodeType).empty().append(odfRestoredNodePasive.children());
                    replaceAllTypesOfHeaderFooters();
                }
                if (odfRestoredNodeActive) {
                    $headerFooterPlaceHolder.children('.' + activeNodeType).empty().append(odfRestoredNodeActive.children());
                    replaceAllTypesOfHeaderFooters();
                }
            } else {
                // on de-checking type of HF, create default if not existing, before placing cursor inside
                if (!$defPlaceholderNode.length && !removeAll) {
                    createOperationInsertHeaderFooter(defNodeType);
                }
            }

            if (app.isODF()) {
                evenOddPages = state;
                if (!state && !removeAll) {
                    localGenerator = model.createOperationsGenerator();
                    if ($activePlaceholderNode.length) {
                        savedNodesCollection[activeNodeType] = $activePlaceholderNode;
                        localGenerator.generateOperation(Operations.DELETE_HEADER_FOOTER, { id: $activePlaceholderNode.attr('data-container-id') });
                    }
                    if ($passivePlaceholderNode.length) {
                        savedNodesCollection[passiveNodeType] = $passivePlaceholderNode;
                        localGenerator.generateOperation(Operations.DELETE_HEADER_FOOTER, { id: $passivePlaceholderNode.attr('data-container-id') });
                    }
                    model.applyOperations(localGenerator);
                }
                self.implToggleEvenOddHF(state, { removeAll: true });
            } else {
                model.applyOperations({ name: Operations.SET_DOCUMENT_ATTRIBUTES, attrs: { page: { evenOddPages: state } } });
            }
            selection.setTextSelection(selection.getFirstDocumentPosition());
        };

        /**
         * Function triggered as implication of changed property for page atrribuge firstPage.
         *
         * @param {Boolean} state
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.removeAll=false]
         *      If set to true, this method call is precceding removing all HF in doc,
         *      and therefore there is no need to create default type before removing them all.
         */
        this.implToggleFirstPageHF = function (state, options) {
            var
                removeAll = Utils.getBooleanOption(options, 'removeAll', false);

            if (pageAttributes && pageAttributes.page) {
                if (!app.isODF()) {
                    pageAttributes.page.firstPage = state;
                }
                firstPage = state;

                replaceActiveNodeOnToggle(removeAll);
            }
        };

        /**
         * Function triggered as implication of changed property for page atrribuge evenOddPages.
         *
         * @param {Boolean} state
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.removeAll=false]
         *      If set to true, this method call is precceding removing all HF in doc,
         *      and therefore there is no need to create default type before removing them all.
         */
        this.implToggleEvenOddHF = function (state, options) {
            var
                removeAll = Utils.getBooleanOption(options, 'removeAll', false);

            if (pageAttributes && pageAttributes.page) {
                if (!app.isODF()) {
                    pageAttributes.page.evenOddPages = state;
                }
                evenOddPages = state;

                replaceActiveNodeOnToggle(removeAll);
            }
        };

        /**
         * Public accesor of replaceAllTypesOfHeaderFooters function.
         *
         */
        this.replaceAllTypesOfHeaderFooters = function () {
            replaceAllTypesOfHeaderFooters();
        };

        /**
         * Looping through current 'page content DOM' and adding heights of elements to calculate page breaks positions.
         * Starts from element which is modified and continues downwards trough DOM. Stops if no more elements are shifted to new page
         * break-above marker: used to mark element above which page break node should be inserted
         *
         */
        this.insertPageBreaks = function () {
            if (!model.isPageBreakMode() || Utils.SMALL_DEVICE || model.isDraftMode() || model.isHeaderFooterEditState()) { return; } // if page layout is disabled, dont run the function
            var
                contentHeight = 0,
                floatingDrawings,
                elementHeight = 0,
                elementHeightWithoutMB = 0,
                processingNodes = $(),
                elementSpan,
                newlySplittedSpan,
                nodeHelper,
                zoomFactor = app.getView().getZoomFactor() / 100,
                offsetPosition = [],
                modifyingElement,
                pageBreak,
                markup = '',
                elMarginBottom,
                elMarginTop,
                diffMargin = 0,
                prevElMarginBottom = 0,
                fDrawingHeight,
                fDrawingWidth = 0,
                accHeightFromPrevDrawing = 0,
                heightOfRectPartHelper = 0,
                // helper for fetching first node from where to start calculating
                procNodesFirst,
                $element,
                currentProcessingNode = model.getCurrentProcessingNode(),
                firstAndLastHFwrappers = $firstHeaderWrapper.add($lastFooterWrapper);

            if (!pageAttributes) {
                initializePageAttributes();
            }

            if (model.checkQuitFromPageBreaks()) { // if there is signal to interup page break processing, break from loop (ex. fast typing in big docs, succesive char deletion)
                model.setQuitFromPageBreaks(false);
                //quitFromPageBreak = false;
                if (pbPromise) {
                    pbPromise.abort();
                    pbPromise = null;
                }
                pbPromise = self.executeDelayed(function () {
                    model.insertPageBreaks(currentProcessingNode ? currentProcessingNode : {});
                }, PB_DELAY_TIME, 'Text: pageLayout: model.insertPageBreaks 1');
                return;
            }

            // hack for tablets with rewriten reduced css padding after document is loaded
            if (app.getView().getContentRootNode().hasClass('small-device')) {
                pagePaddingLeft = parseFloat($pageContentNode.parent().css('padding-left')); //18.898;
                pagePaddingRight = pagePaddingLeft;
                pageWidth = $pageContentNode.parent().outerWidth();
                firstAndLastHFwrappers.css({ width: pageWidth, 'margin-left': (-pagePaddingLeft) });
                firstAndLastHFwrappers.children('.header, .footer').css({ 'padding-left': pagePaddingLeft, 'padding-right': pagePaddingLeft });
                reducedPadding = true;
            } else if (reducedPadding) {
                pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 0.001);
                pagePaddingRight = Utils.convertHmmToLength(pageAttributes.page.marginRight, 'px', 0.001);
                pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 0.001);
                firstAndLastHFwrappers.css({ width: pageWidth, 'margin-left': (-pagePaddingLeft) });
                firstAndLastHFwrappers.children('.header, .footer').css({ 'padding-left': pagePaddingLeft, 'padding-right': pagePaddingLeft });
                reducedPadding = false;
            }

            function paragraphSplitWithPageBreak(element) {
                var arrBreakAboveSpans,
                    upperPartDiv,
                    totalUpperParts,
                    diffPosTopAndPrev,
                    elementLeftIndent;

                $element = $(element);
                offsetPosition = [];
                modifyingElement = currentProcessingNode[0] === element; // flag if we are modifying the element, then cached line positions are invalid
                if (modifyingElement) {
                    selection.setInsertTextPoint(null); // #32603 invalidate cached text point - reference to text node that is changed, so it returns wrong value for getFocusPositionBoundRect when scrollToSelection
                }
                if (accHeightFromPrevDrawing > 0 && fDrawingWidth > 0) {
                    offsetPosition = getOffsetPositionFromElement(element, elementHeight, contentHeight, modifyingElement, accHeightFromPrevDrawing, fDrawingWidth);
                } else {
                    offsetPosition = getOffsetPositionFromElement(element, elementHeight, contentHeight, modifyingElement);
                }
                if (offsetPosition && offsetPosition.length > 0) { // if offset position is returned
                    if (offsetPosition[0] === 0) { // if offset is 0, no need for p split, insert pb before element
                        offsetPosition.shift();
                        contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                        if (offsetPosition.length > 0) { contentHeight = 0; }
                    }
                    if (offsetPosition.length > 0) {
                        elementSpan = $element.children('span').first();
                        while (elementSpan.text().length < 1 || !(elementSpan.is('span'))) {
                            elementSpan = elementSpan.next();
                        }
                        if (elementSpan && elementSpan.length > 0) { //fix for #32563, if div.p contains only empty spans
                            _.each(offsetPosition, function (offset) {
                                var elTextLength = elementSpan.text().length;
                                while (elTextLength > 0 && elTextLength <= offset) {
                                    offset -= elTextLength;
                                    elementSpan = elementSpan.next();
                                    while (!elementSpan.is('span')) { // we might fetch some inline div nodes, like div.tab or div.linebreak
                                        elementSpan = elementSpan.next();
                                    }
                                    elTextLength = elementSpan.text().length;
                                }
                                newlySplittedSpan = DOM.splitTextSpan(elementSpan, offset, { append: true });
                                newlySplittedSpan.addClass('break-above-span');
                                $element.addClass('contains-pagebreak');
                                elementSpan = newlySplittedSpan;
                            });
                        }

                        zoomFactor = app.getView().getZoomFactor() / 100;
                        pageBreak = $();
                        arrBreakAboveSpans = $element.find('.break-above-span');
                        upperPartDiv = 0;
                        totalUpperParts = 0;
                        diffPosTopAndPrev = 0;
                        elementLeftIndent = parseInt($element.css('margin-left'), 10) + parseInt($element.css('padding-left'), 10); // for indented elements such as lists, or special paragraphs

                        _.each(arrBreakAboveSpans, function (breakAboveSpan) {
                            diffPosTopAndPrev += $(pageBreak).outerHeight(true) + upperPartDiv;
                            upperPartDiv = $(breakAboveSpan).position().top / zoomFactor - diffPosTopAndPrev;
                            totalUpperParts += upperPartDiv;
                            contentHeight += upperPartDiv;
                            nodeHelper = $element.prev();
                            while (nodeHelper.hasClass('page-break')) {
                                nodeHelper = nodeHelper.prev();
                            }
                            markup = generatePagebreakElementHtml(contentHeight, _.browser.WebKit, elementLeftIndent);
                            $(breakAboveSpan).before(markup);
                            pageBreak = $(breakAboveSpan).prev();
                            contentHeight = 0;
                            if (_.browser.Chrome && parseInt($element.css('text-indent')) < 0) { // #40776
                                pageBreak.css('display', 'inline-block');
                            }
                        });
                        contentHeight = elementHeight - totalUpperParts;
                    }
                } else {
                    contentHeight = insertPageBreaksBetweenDrawings(element, elementHeight, contentHeight, offsetPosition, zoomFactor);
                    //contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                }
            }

            function processNodeForPageBreaksInsert(element, index) {
                if (model.checkQuitFromPageBreaks()) { // if there is signal to interup page break processing, break from loop (ex. fast typing in big docs, succesive char deletion)
                    model.setQuitFromPageBreaks(false);
                    //quitFromPageBreak = false;
                    if (pbPromise) {
                        pbPromise.abort();
                        pbPromise = null;
                    }
                    pbPromise = self.executeDelayed(function () {
                        model.insertPageBreaks(currentProcessingNode ? currentProcessingNode : {});
                    }, PB_DELAY_TIME, 'Text: pageLayout: model.insertPageBreaks 2');
                    return Utils.BREAK;
                }
                $element = $(element);
                // first merge back any previous paragraph splits
                pageBreak = $element.find('.page-break');
                if (DOM.nodeContainsPageBreak($element) || pageBreak.length > 0) {
                    pageBreak.remove();
                    $element.children('.break-above-drawing').removeClass('break-above-drawing');
                    _.each($element.children('span'), function (span) {
                        if (span.classList.contains('break-above-span')) {
                            $(span).removeClass('break-above-span');
                            Utils.mergeSiblingTextSpans(span);
                        }
                    });
                    if ($element.hasClass('p')) {
                        model.validateParagraphElement(element);
                        $element.removeClass('contains-pagebreak');
                        if ($element.find('.ms-hardbreak-page').length > 0) {
                            $element.addClass('manual-page-break');
                        }
                    }
                }
                // if table is split, merge back
                if ($element.find('tr.pb-row').length > 0) {
                    $element.find('tr.pb-row').remove();
                    $element.find('tr.break-above-tr').removeClass('break-above-tr');
                    if ((_.browser.Firefox) && (DOM.isTableNode(element))) { Table.forceTableRendering(element); } // forcing Firefox to re-render table (32461)
                }
                $element.removeClass('tb-split-nb'); // return box shadow property to unsplit tables

                fDrawingHeight = 0;
                floatingDrawings = $element.find('.drawing.float');
                if (floatingDrawings.length > 0) {
                    _.each(floatingDrawings, function (drawing) {
                        if ($(drawing).outerWidth(true) !== $pageContentNode.width()) {
                            if ($(drawing).outerHeight(true) + ($(drawing).position().top / zoomFactor) > fDrawingHeight) {
                                fDrawingHeight = $(drawing).outerHeight(true) + ($(drawing).position().top / zoomFactor);
                                fDrawingWidth = $(drawing).outerWidth(true);
                            }
                        }
                    });
                }
                if (accHeightFromPrevDrawing > fDrawingHeight) {
                    fDrawingHeight = 0;
                }

                elementHeightWithoutMB = element.offsetHeight;
                elMarginBottom = Utils.getElementCssLength($element, 'margin-bottom');
                elMarginTop = Utils.getElementCssLength($element, 'margin-top');

                if (elMarginTop > prevElMarginBottom) { // el has margin top that is greater than margin bottom of prev el, include diff in page height
                    diffMargin = elMarginTop - prevElMarginBottom;
                    elementHeightWithoutMB += diffMargin;
                }
                elementHeight = elementHeightWithoutMB + elMarginBottom;
                prevElMarginBottom = elMarginBottom;

                if (DOM.isManualPageBreakNode(element)) {
                    if ($element.prev().length > 0 || $element.find('.ms-hardbreak-page').length > 0) {
                        contentHeight = manualInsertionOfPageBreak(element, contentHeight, elementHeight, { cleanup: true });
                    } else {
                        $element.removeClass('manual-page-break');
                    }
                }

                // if current content height is bellow max page height
                // and if there is no image floated inside paragraph, or if so, p fits inside page with it also
                // and if there is no floated image from one of previous paragraphs that intersects current p, or if exist, element can fit on page with it also;
                if (contentHeight + elementHeight <= pageMaxHeight && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                    if (!DOM.isManualPageBreakNode(element) || $element.find('.ms-hardbreak-page').length < 1) {
                        contentHeight += elementHeight;
                    }
                    if (accHeightFromPrevDrawing > elementHeight) {
                        accHeightFromPrevDrawing -= elementHeight;
                    } else {
                        heightOfRectPartHelper = accHeightFromPrevDrawing;
                        accHeightFromPrevDrawing = 0;
                    }
                    if (!DOM.isManualPageBreakNode(element)) {
                        prepareNodeAndRemovePageBreak(element);
                    }
                // for last element that can fit on page we need to omit margin bottom
                } else if (contentHeight + elementHeightWithoutMB <= pageMaxHeight && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                    if (!DOM.isManualPageBreakNode(element) || $element.find('.ms-hardbreak-page').length < 1) {
                        contentHeight += elementHeightWithoutMB;
                    }
                    if (accHeightFromPrevDrawing > elementHeightWithoutMB) {
                        accHeightFromPrevDrawing -= elementHeightWithoutMB;
                    } else {
                        heightOfRectPartHelper = accHeightFromPrevDrawing;
                        accHeightFromPrevDrawing = 0;
                    }
                    if (!DOM.isManualPageBreakNode(element)) {
                        prepareNodeAndRemovePageBreak(element);
                    }
                } else { //shift to new page or split
                    prevElMarginBottom = 0;
                    accHeightFromPrevDrawing = 0;
                    if (DOM.isManualPageBreakNode($element.prev()) && DOM.isPageBreakNode($element.prev().children().last())) {
                        // do nothing if previous node has manual page break and it it's is last child
                    } else {

                        nodeHelper = $element.next();
                        //if element is last in document
                        if (nodeHelper.length) {
                            // remove old page break
                            while (nodeHelper.hasClass('page-break')) {
                                nodeHelper.remove();
                                nodeHelper = $element.next();
                            }
                        }

                        if ($element.hasClass('p') && ($element.text().length > 10 || $element.find('.drawing.inline').length > 0)) { // paragraph has to be split
                            paragraphSplitWithPageBreak(element);
                            // end of paragraph split
                        } else if (DOM.isTableNode(element) && element.rows.length > 1) { // table split
                            contentHeight = insertPageBreaksInTable(element, elementHeight, contentHeight, { diffMargin: diffMargin });
                        } else { // default insertion of pb between elements
                            contentHeight = defaultInsertionOfPageBreak(element, elementHeight, contentHeight);
                        }
                    }
                }
                if (DOM.isNodeWithPageBreakAfterAttribute(element)) {
                    if ($element.next().length > 0) {
                        contentHeight = insertManualPageBreakAfter(element, contentHeight);
                    } else { // first element in document, can't have page break before - can happen on delete all where last is manual pb paragraph
                        $element.removeClass('manual-pb-after');
                    }
                }

                if (fDrawingHeight > 0) { // store the diff between floated image height and height of containing p, for next iteration
                    accHeightFromPrevDrawing = fDrawingHeight - elementHeight;
                }
                // at the last page, we add extra padding, to fill it out to the size of max page height
                if (processingNodes.length === index + 1) {
                    $pageContentNode.css({
                        'padding-bottom': (pageMaxHeight - contentHeight > 0) ? pageMaxHeight - contentHeight : '36px'
                    });
                }
            } //end of processNodeForPageBreaksInsert

            if (model.isProcessingOperations()) {
                // to prevent triggering method when not all DOM nodes are rendered (still operations in the stack, #32067)
                self.executeDelayed(function () {
                    model.insertPageBreaks(currentProcessingNode ? currentProcessingNode : {});
                }, PB_DELAY_TIME, 'Text: pageLayout: model.insertPageBreaks 3');
            } else {
                // cleanup after deleting big blocs of data at once
                while ($pageContentNode.children().first().hasClass('page-break')) {
                    $pageContentNode.children().first().remove();
                }
                if (currentProcessingNode) {
                    currentProcessingNode = $(currentProcessingNode);
                    if (currentProcessingNode.hasClass('break-above')) { // first element bellow pagebreak, we want to refresh previous page also
                        nodeHelper = currentProcessingNode.prev();
                        while (nodeHelper.hasClass('page-break')) {
                            nodeHelper = nodeHelper.prev();
                        }
                        if (nodeHelper.length > 0) {
                            currentProcessingNode = nodeHelper;
                        } else {
                            currentProcessingNode.prev('.page-break').remove();
                        }
                    }
                    if (currentProcessingNode.next().hasClass('page-break')) {
                        nodeHelper = currentProcessingNode.prev();
                        while (nodeHelper.hasClass('page-break')) {
                            nodeHelper = nodeHelper.prev();
                        }
                        if (nodeHelper.length === 0) {
                            currentProcessingNode.prevAll('.page-break').remove();
                        }
                    }
                    if (currentProcessingNode.parentsUntil('.pagecontent', 'table').length > 0) { //if we fetch valid div.p, but inside of table
                        currentProcessingNode = currentProcessingNode.parentsUntil('.pagecontent', 'table').last();
                    }
                    nodeHelper = currentProcessingNode.prev();
                    if (nodeHelper.hasClass('page-break')) {
                        while (nodeHelper.hasClass('page-break')) { // might happen when deleting multiple elements at once, some pb are left - clean-up
                            nodeHelper = nodeHelper.prev();
                        }
                        if (nodeHelper.length === 0) {
                            currentProcessingNode.prevAll('.page-break').remove();
                            processingNodes = currentProcessingNode.nextAll('.p, table').addBack();
                        } else {
                            nodeHelper = nodeHelper.prevUntil('.page-break, .contains-pagebreak').last();
                            if (DOM.nodeContainsPageBreak(nodeHelper.prev())) {
                                contentHeight = (nodeHelper.prev().height() + parseInt(nodeHelper.prev().css('margin-bottom'), 10)) - nodeHelper.prev().children('.break-above-span, .break-above-drawing').last().position().top / zoomFactor;
                            }
                            processingNodes = nodeHelper.nextAll('.p, table').addBack();
                        }
                    //} else if (DOM.isManualPageBreakNode(currentProcessingNode)) {
                      //  processingNodes = nodeHelper.nextAll('div.p, table').addBack();
                    } else if (DOM.nodeContainsPageBreak(nodeHelper)) { //if node is second bellow pagebreak - fallback
                        if (nodeHelper.find('.break-above-span').length > 0) { // special case: split p with breakabove span, above page break
                            if (DOM.isManualPageBreakNode(currentProcessingNode)) { // fix for inserting manual page break in paragraph bigger than page height
                                processingNodes = nodeHelper.prevUntil('.page-break').last().nextAll('.p, table').addBack();
                            } else {
                                processingNodes = currentProcessingNode.nextAll('.p, table').addBack();
                                contentHeight = (nodeHelper.height() + parseInt(nodeHelper.css('margin-bottom'), 10)) - nodeHelper.children('.break-above-span, .break-above-drawing').last().position().top / zoomFactor;
                            }
                        } else {
                            procNodesFirst = nodeHelper.prevUntil('.page-break, .contains-pagebreak').last();
                            nodeHelper = procNodesFirst.prev();
                            if (DOM.nodeContainsPageBreak(nodeHelper)) {
                                processingNodes = procNodesFirst.nextAll('.p, table').addBack();
                                contentHeight = (nodeHelper.height() + parseInt(nodeHelper.css('margin-bottom'), 10)) - nodeHelper.children('.break-above-span, .break-above-drawing').last().position().top / zoomFactor;
                            } else {
                                processingNodes = procNodesFirst.nextAll('.p, table').addBack();
                            }
                        }
                    } else {
                        procNodesFirst = currentProcessingNode.prevUntil('.page-break, .contains-pagebreak').last();
                        nodeHelper = procNodesFirst.prev();
                        if (DOM.nodeContainsPageBreak(nodeHelper)) {
                            processingNodes = procNodesFirst.nextAll('.p, table').addBack();
                            if (DOM.isPageBreakNode(nodeHelper.children().last())) { // #40430 - if ms page break is last in paragraph, it's a new page and content height is reset to 0
                                contentHeight = 0;
                            } else {
                                contentHeight = (nodeHelper.height() + parseInt(nodeHelper.css('margin-bottom'), 10)) - nodeHelper.children('.break-above-span, .break-above-drawing').last().position().top / zoomFactor;
                            }
                        } else {
                            processingNodes = procNodesFirst.nextAll('.p, table').addBack();
                        }
                    }
                    if (processingNodes.length === 0) { // if we are somewhere on first page, no pageBreak class at the top!
                        processingNodes = $pageContentNode.children('.p, table');
                    }
                } else if (_.isEmpty(currentProcessingNode)) { // fallback option if we don't fetch any valid node
                    processingNodes = $pageContentNode.children('.p, table');
                    currentProcessingNode = processingNodes.first();
                }

                if (processingNodes) {
                    // remove all break-above classes except the top one
                    processingNodes.removeClass('break-above').first().addClass('break-above');
                    resolvePageTypeAndHeight(processingNodes.first());

                    // process all fetched nodes
                    self.iterateArraySliced(processingNodes, processNodeForPageBreaksInsert, { delay: 50, infoString: 'Text: pageLayout: processNodeForPageBreaksInsert' })
                    .done(function () {

                        pageBreaksCollection = $pageContentNode.find('.page-break');
                        // store indexes of pagebreaks as values for page numbers
                        _.each(pageBreaksCollection, function (pageBreakInCollection, index) {
                            $(pageBreakInCollection).data('page-num', index + 1);
                        });
                        pageCount = pageBreaksCollection.length + 1;

                        populateHeaderFooterPaginationClasses(pageBreaksCollection);
                        if (self.isHeaderFooterInDocument()) {
                            replaceAllTypesOfHeaderFooters();
                        } else {
                            if (!fieldManager.isComplexFieldEmpty()) {
                                fieldManager.updatePageFieldsInMarginals();
                            }
                        }
                        if (!fieldManager.isSimpleFieldEmpty()) {
                            updatePageCountFields();
                            updatePageNumberFields();
                        }
//                        if (!fieldManager.isComplexFieldEmpty()) {
//                            fieldManager.updatePageFieldsInMarginals();
//                        }
                        _.each(cleanupElements, function (element) {
                            $(element).remove();
                        });
                        cleanupElements = [];

                        app.getView().recalculateDocumentMargin({ keepScrollPosition: !model.getEditMode() });  // Checking edit mode for 39904
                        if (repaintRemoteSelection) {
                            model.getRemoteSelection().renderCollaborativeSelections();
                            repaintRemoteSelection = false;
                        }
                        model.trigger('update:absoluteElements');   // updating comments and absolutely positioned drawings
                    });
                }
            }
        };

        /**
         * Public method triggered when some of page metric properties are changed,
         * like width, height or margins.
         *
         * @param {Object} changedPageAttributes
         *  List of changed page attributes
         */
        this.refreshPageLayout = function (changedPageAttributes) {
            // re-initialize page styles
            pageStyles = model.getStyleCollection('page');
            drawingStyles = model.getStyleCollection('drawing');
            characterStyles = model.getStyleCollection('character');
            pageAttributes = pageStyles.getElementAttributes(rootNode);

            pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 0.001);
            pagePaddingRight = Utils.convertHmmToLength(pageAttributes.page.marginRight, 'px', 0.001);
            pagePaddingTop = Utils.convertHmmToLength(pageAttributes.page.marginTop, 'px', 0.001);
            pagePaddingBottom = Utils.convertHmmToLength(pageAttributes.page.marginBottom, 'px', 0.001);
            pageHeight = Utils.convertHmmToLength(pageAttributes.page.height, 'px', 0.001);
            pageMaxHeight = pageHeight - pagePaddingTop - pagePaddingBottom;
            pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 0.001);

            if (changedPageAttributes.marginLeft) {
                rootNode.css('padding-left', changedPageAttributes.marginLeft / 100 + 'mm');
            }
            if (changedPageAttributes.marginRight) {
                rootNode.css('padding-right', changedPageAttributes.marginRight / 100 + 'mm');
            }
            if (changedPageAttributes.marginTop) {
                rootNode.css('padding-top', changedPageAttributes.marginTop / 100 + 'mm');
            }
            if (changedPageAttributes.marginBottom) {
                rootNode.css('padding-bottom', changedPageAttributes.marginBottom / 100 + 'mm');
            }
            if (changedPageAttributes.width || changedPageAttributes.marginRight || changedPageAttributes.marginLeft) {
                rootNode.css('width', (pageAttributes.page.width - pageAttributes.page.marginLeft - pageAttributes.page.marginRight) / 100 + 'mm');
            }
            if (changedPageAttributes.height) {
                rootNode.css('min-height', changedPageAttributes.height / 100 + 'mm');
            }

            // force repaint layout
            updateStartHeaderFooterStyle();
            $firstHeaderWrapper.css({ width: pageWidth, margin: ' 0px 0px 0px ' + (-pagePaddingLeft) + 'px' }).children('.header').css({
                minHeight: pagePaddingTop,
                maxWidth: pageWidth,
                padding: (self.isHeaderFooterInDocument() ? headerMargin : 0) + 'px ' + pagePaddingRight + 'px 0 ' + (pagePaddingLeft) + 'px'
            });
            $lastFooterWrapper.css({ width: pageWidth, margin: ' 0px 0px 0px ' + (-pagePaddingLeft) + 'px' }).children('.footer').css({
                minHeight: pagePaddingBottom,
                maxWidth: pageWidth,
                padding: '5px ' + pagePaddingRight + 'px ' + (self.isHeaderFooterInDocument() ? footerMargin : 0) + 'px ' + (pagePaddingLeft) + 'px'
            });

            if (model.isHeaderFooterEditState()) {
                self.leaveHeaderFooterEditMode();
                selection.setNewRootNode(model.getNode()); // restore original rootNode
                selection.setTextSelection(selection.getFirstDocumentPosition());
            }
            // invalidate cached line break positions, because of layout change
            $pageContentNode.children('.contains-pagebreak').removeData('lineBreaksData');

            repaintRemoteSelection = true;
            // update of comments, absolute drawings etc. is triggered from inside this function
            model.insertPageBreaks();
        };

        /**
         * Delete pageBreakBefore property from setAttributes in pasteInternalClipboard operations array,
         * if destination paragraph already has this property, so that the property is not overwritten by new operation.
         *
         * @param {Array} operations
         */
        this.handleManualPageBreakDuringPaste = function (operations) {
            var // setAttributes operation extracted from list of paste operations
                setAttrOperation = _.findWhere(operations, { name: 'setAttributes' }),
                // oxo position of destination paragraph where clipboard is going to be pasted
                destinationParagraphPosition,
                // dom node of destination paragraph
                destinationParagraphNode;

            if (setAttrOperation && setAttrOperation.start) {
                destinationParagraphPosition = setAttrOperation.start.slice(0, 1);
                destinationParagraphNode = Position.getParagraphElement(model.getNode(), destinationParagraphPosition);
                if (DOM.isManualPageBreakNode(destinationParagraphNode) && setAttrOperation.attrs && setAttrOperation.attrs.paragraph) {
                    if (!_.isUndefined(setAttrOperation.attrs.paragraph.pageBreakBefore)) {
                        delete setAttrOperation.attrs.paragraph.pageBreakBefore;
                    }
                }
            }
        };

        /**
         * Toggles dropdown menu for marginal context menu in header/footer.
         *
         * @param {Node|jQuery} item
         *  Child dom element of context menu type button.
         */
        this.toggleMarginalMenuDropdown = function (item) {
            var
                marginalMenuType = $(item).closest('.marginal-menu-type');

            marginalMenuType.toggleClass('active-dropdown');
            marginalMenuType.find('.marginal-container').toggle();
            marginalMenuType.find('.active-highlight').removeClass('active-highlight');
            if (firstPage) {
                if (evenOddPages) {
                    marginalMenuType.find('.marginal-first-even').addClass('active-highlight');
                } else {
                    marginalMenuType.find('.marginal-first').addClass('active-highlight');
                }
            } else if (evenOddPages) {
                marginalMenuType.find('.marginal-even').addClass('active-highlight');
            } else {
                marginalMenuType.find('.marginal-same').addClass('active-highlight');
            }
        };

        /**
         * Updating the cached page content node. This is necessary after the page
         * content node is exchanged after cancelling a long running action.
         */
        this.refreshPageContentNode = function () {
            $pageContentNode = DOM.getPageContentNode(rootNode);
        };

        /**
         * Returns all pageBreaks currently present in pagecontent.
         *
         * @returns {Array} pageBreaksCollection
         */
        this.getPageBreaksCollection = function () {
            return pageBreaksCollection;
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            selection = model.getSelection();
            fieldManager = model.getFieldManager();

            app.onImportSuccess(function () {
                if (!pageAttributes) {
                    initializePageAttributes();
                }

                if (Utils.TOUCHDEVICE) { self.insertPageBreaks(); } // one extra call required for touch devices (Bug 39954) & (Bug 40901)

                // register the listener for the event 'change:defaults'
                model.on('change:defaults', function (event, newAttributes, oldAttributes, changedAttributes) {
                    if (!app.isImportFinished()) {
                        return;
                    }
                    if (_.isObject(changedAttributes.page)) {
                        if (changedAttributes.page.firstPage !== undefined) {
                            self.implToggleFirstPageHF(changedAttributes.page.firstPage);
                        } else if (changedAttributes.page.evenOddPages !== undefined) {
                            self.implToggleEvenOddHF(changedAttributes.page.evenOddPages);
                        } else {
                            self.refreshPageLayout(changedAttributes.page);

                            // only necessary for updating horizontal comment layer.
                            // Update of comments and absolute drawings is triggered from inside insertPageBreaks, with update:absoluteElements
                            model.trigger('change:pageSettings', changedAttributes);

                            updateStartHeaderFooterStyle();
                            // ... and refreshing page layout once more
                            model.insertPageBreaks();
                        }
                    }
                });
            });
        });

    } // end of class PageLayout

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: PageLayout });

});
