/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/format/tablecellstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/text/dom'
], function (Utils, Color, Border, StyleCollection, DOM) {

    'use strict';

    var // definitions for table cell attributes
        DEFINITIONS = {

            /**
             * The number of grid columns spanned by the table cell.
             */
            gridSpan: {
                def: 1,
                scope: 'element',
                format: function (element, gridSpan) {
                    element.attr('colspan', gridSpan);
                }
            },

            /**
             * Fill color of the table cell.
             */
            fillColor: {
                def: Color.AUTO,
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                },
                preview: true // 'true' calls the own 'format' method
            },

            /**
             * Style, width and color of the left table cell border.
             */
            borderLeft: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-left', this.getCssBorder(border, { clearNone: true }));
                },
                preview: function (element, border) {
                    element.css('border-left', this.getCssBorder(border, { preview: true }));
                }
            },

            /**
             * Style, width and color of the right table cell border.
             */
            borderRight: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-right', this.getCssBorder(border, { clearNone: true }));
                },
                preview: function (element, border) {
                    element.css('border-right', this.getCssBorder(border, { preview: true }));
                }
            },

            /**
             * Style, width and color of the top table cell border.
             */
            borderTop: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-top', this.getCssBorder(border, { clearNone: true }));
                },
                preview: function (element, border) {
                    element.css('border-top', this.getCssBorder(border, { preview: true }));
                }
            },

            /**
             * Style, width and color of the bottom table cell border.
             */
            borderBottom: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border, { clearNone: true }));
                },
                preview: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border, { preview: true }));
                }
            },

            /**
             * Inner horizontal table cell borders, used in table style sheets
             * to format inner borders of specific table areas (first/last
             * column, inner vertical bands, ...).
             */
            borderInsideHor: {
                def: Border.NONE,
                scope: 'style'
            },

            /**
             * Inner vertical table cell borders, used in table style sheets to
             * format inner borders of specific table areas (first/last row,
             * inner horizontal bands, ...).
             */
            borderInsideVert: {
                def: Border.NONE,
                scope: 'style'
            },

            /**
             * Top padding of the table cell
             */
            paddingTop: {
                def: 0,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px, required for cell resizing
                    element.css('padding-top', value);
                }
            },

            /**
             * Bottom padding of the table cell
             */
            paddingBottom: {
                def: 0,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px,  required for cell resizing
                    element.css('padding-bottom', value);
                }
            },

            /**
             * Left padding of the table cell
             */
            paddingLeft: {
                def: 0,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px, required for cell resizing
                    element.css('padding-left', value);
                }
            },

            /**
             * Right padding of the table cell
             */
            paddingRight: {
                def: 0,
                format: function (element, value) {
                    value = Utils.convertHmmToLength(value, 'px', 1);
                    if (value < 2) { value = 2; }  // minimum 2px, required for cell resizing
                    element.css('padding-right', value);
                }
            }

        },

        // parent families with parent element resolver functions
        PARENT_RESOLVERS = {
            table: function (cell) { return cell.closest(DOM.TABLE_NODE_SELECTOR); }
        };

    // class TableCellStyles ==================================================

    /**
     * Contains the style sheets for table cell formatting attributes. The CSS
     * formatting will be read from and written to <th> and <td> elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function TableCellStyles(docModel) {

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'cell', {
            styleSheetSupport: false,
            families: 'paragraph character changes',
            parentResolvers: PARENT_RESOLVERS,
            formatHandler: updateTableCellFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every table cell element whose attributes have
         * been changed. Reformats the table cell according to the passed
         * attributes.
         *
         * @param {jQuery} cell
         *  The table cell element whose table attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family, containing the effective attribute values merged
         *  from style sheets and explicit attributes.
         */
        function updateTableCellFormatting(cell, mergedAttributes) {

            var // the paragraph style sheet container
                paragraphStyles = docModel.getStyleCollection('paragraph');

            // Center cell content vertically. This solution is not dynamic
            // -> changing row height or inserting text destroys center position.
            // var cellContentNode = DOM.getCellContentNode(cell);
            // cellContentNode.css('padding-top', cell.height() / 2 - cellContentNode.height() / 2);

            // update all paragraphs in the table cell
            Utils.iterateSelectedDescendantNodes(DOM.getCellContentNode(cell), DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                paragraphStyles.updateElementFormatting(paragraph, { baseAttributes: mergedAttributes });
            }, undefined, { children: true });

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(cell, mergedAttributes);

        }

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('cell', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class TableCellStyles

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: TableCellStyles });

});
