/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/text/export', [
    'io.ox/office/tk/utils',
    'io.ox/office/text/dom',
    'io.ox/office/drawinglayer/view/imageutil'
], function (Utils, DOM, Image) {

    'use strict';

    var COPY_STYLES = {
        paragraph: ['text-align', 'background-color', 'padding', 'margin', 'text-indent', 'margin', 'border-color', 'border-style', 'border-width'],
        span: ['font-family', 'font-weight', 'text-decoration', 'background-color', 'color', 'font-size', 'vertical-align', 'line-height', 'lang'],
        table: ['width', 'background-color'],
        td: ['height', 'background-color', 'border-top-style', 'border-right-style', 'border-bottom-style', 'border-left-style', 'border-top-width', 'border-right-width', 'border-bottom-width', 'border-left-width', 'border-color', 'padding'],
        img: ['width', 'height', 'border', 'margin', 'title']
    };

    var Export = {};

    /**
     * Converts the internal HTML mark-up of the current selection to
     * normalized HTML mark-up that can be exported (e.g. to the clipboard).
     *
     * @param {TextApplication} editor
     *  The text application instance.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.clipboardId]
     *      A unique id to identify clipboard operation data. Will be attached
     *      to the returned HTML mark-up.
     *  @param {Array} [options.clipboardOperations]
     *      An array with clipboard operations. Will be attached to the
     *      returned HTML mark-up.
     *
     * @returns {String}
     *  The normalized HTML mark-up for export.
     */
    Export.getHTMLFromSelection = function (editor, options) {
        var resultDiv = $('<div>'),
            selection = editor.getSelection(),
            paragraphStyles = editor.getStyleCollection('paragraph'),
            listStyles = editor.getListCollection(),
            paraStylesIds = createParagraphStyleIds(selection, paragraphStyles),
            html = $('<span>').html(selection.getHtmlFromBrowserSelection()),
            clipboardId = Utils.getStringOption(options, 'clipboardId', ''),
            clipboardOperations = Utils.getArrayOption(options, 'clipboardOperations', []),
            metaElement;

        createOptimizedStructure(html, resultDiv, paraStylesIds, listStyles);
        clearParagraphStyleIds(selection);

        // use the first child to store meta information on
        metaElement = resultDiv.children().first();
        if (metaElement) {
            metaElement.attr('id', 'ox-clipboard-data');
            if (clipboardId.length > 0) {
                metaElement.attr('data-ox-clipboard-id', clipboardId);
            }
            if (clipboardOperations.length > 0) {
                metaElement.attr('data-ox-operations', JSON.stringify(clipboardOperations));
            }
        }
        if (resultDiv.children().length === 1) {
            resultDiv.append('<span>&#8203;</span>');
        }
        return resultDiv.html();
    };

    /**
     * Creates a 'normalized' DOM structure based on the source.
     *
     * @param {jQuery} source
     *  The source html. The function assumes that this html was created by the
     *  text application.
     *
     * @returns {jQuery} result
     *  The result html.
     */
    function createOptimizedStructure(source, result, paragraphStylesMap, listStyles) {

        var outNode, tbody,
            trow, td, cellContent, colgroup, styles,
            optimized = false, listLevel = -1, listStyle, listLevels = [];

        $(source).children().each(function () {
            var optNode = $(this);

            if (DOM.isParagraphNode(optNode)) {
                styles = paragraphStylesMap[optNode.attr('styleId')];
                if (styles && styles.paragraph.listStyleId && styles.paragraph.listStyleId.length && listStyles.getList(styles.paragraph.listStyleId)) {
                    listStyle = listStyles.getList(styles.paragraph.listStyleId);
                    listLevel = styles.paragraph.listLevel;
                    listLevels = createListLevels(result, listLevels, listLevel, listStyle);
                    outNode = $('<li>');
                } else {
                    // reset list levels
                    listLevel = -1;
                    listLevels = [];
                    outNode = $('<p>');
                }
                copyStyles(optNode, outNode, 'paragraph');
                optimizeParagraphForExport(optNode, outNode);
                if (listLevel === -1) {
                    result.append(outNode);
                } else {
                    listLevels[listLevel].append(outNode);
                }
                optimized = true;
            } else if (DOM.isTableNode(optNode)) {
                optimized = true;
                // reset list levels
                listLevel = -1;
                listLevels = [];
                outNode = $('<table>');
                outNode.css('border-collapse', 'collapse');
                copyStyles(optNode, outNode, 'table');
                colgroup = optNode.find('colgroup');
                if (colgroup.lengh) {
                    outNode.append(colgroup);
                }
                tbody = $('<tbody>');
                optNode.find('tr').each(function () {
                    trow = $('<tr>');
                    $(this).find('td').each(function () {
                        td = $('<td>');
                        td.attr('colspan', this.colSpan);
                        copyStyles(this, td, 'td');
                        cellContent = $(this).find('div.cellcontent');
                        if (cellContent) {
                            createOptimizedStructure(cellContent, td, paragraphStylesMap, listStyles);
                        }
                        trow.append(td);
                    });
                    tbody.append(trow);
                });
                outNode.append(tbody);
                result.append(outNode);
            }
        });

        // if cannot optimize just append the source
        if (!optimized) {
            result.append(source);
        }
    }

    /**
     * Prepares the parentNode to receive list entries for a the specified
     * list level.
     *
     * @param {jQuery} parentNode
     *  The node which is the parent for all list entries.
     *
     * @param {Array} listLevels
     *  An array filled with jQuery nodes that can be used to add list entries
     *  for up to the specified list level.
     *
     * @param {Integer} upToLevel
     *  The max list level to be used.
     *
     * @param {Lists} listStyle
     *  The list style to be used for the upToLevel list level.
     *
     * @returns {Array}
     *  The new list level array updated to match the upToLevel&listStyle
     *  requirements.
     */
    function createListLevels(parentNode, listLevels, upToLevel, listStyle) {
        var i, node;

        if (listLevels.length <= upToLevel) {
            // Extend the list level array and DOM
            for (i = listLevels.length; i <= upToLevel; i++) {
                node = createListElement(upToLevel, listStyle);
                if (i === 0) {
                    parentNode.append(node);
                } else {
                    listLevels[i - 1].append(node);
                }
                listLevels[i] = node;
            }
        } else {
            // Reduce the list to start a new sub-list
            listLevels = listLevels.slice(0, upToLevel + 1);
        }

        return listLevels;
    }

    /**
     * Creates a jQuery element to add to the DOM for the specified
     * list level.
     *
     * @param {Integer} level
     *  The list level for which the jQuery element should to be created.
     *
     * @param {List} listStyle
     *  The list style to be used for the list element.
     *
     * @returns {jQuery}
     *  The list element as jQuery.
     */
    function createListElement(level, listStyle) {
        var format = listStyle.listLevels[level].numberFormat,
            levelText = listStyle.listLevels[level].levelText,
            node = (format === 'bullet') ? $('<ul>') : $('<ol>');

        switch (format.toLowerCase()) {
        case 'decimal':
            // do nothing;
            break;
        case 'lowerletter':
            node.attr('type', 'a');
            break;
        case 'upperletter':
            node.attr('type', 'A');
            break;
        case 'lowerroman':
            node.attr('type', 'i');
            break;
        case 'upperroman':
            node.attr('type', 'I');
            break;
        case 'bullet':
            switch (levelText) {
            case '\uf0b7':
            case '\u25CF':  // filled-circle
            case '\u2022':  // small-filled-circle
                node.css('list-style-type', 'disc');
                break;
            case '\u25CB':  // circle
            case '\u25E6':  // small-circle
                node.css('list-style-type', 'circle');
                break;
            case '\u25A0':  // filled-square
            case '\u25AA':  // small-filled-square
            case '\u25A1':  // square
            case '\u25AB':  // small-square
                node.css('list-style-type', 'square');
                break;
            case ' ':
                node.css('list-style-type', 'none');
                break;
            }
            break;
        }

        return node;
    }

    /**
     * Copy specific styles dependent on the type from
     * one element to another one.
     *
     * @param {jQuery|Node} from
     *  The node that is used as source.
     *
     * @param {jQuery|Node} to
     *  The node that receives the styles.
     *
     * @param {String} type
     *  The type of node. Currently supported types are
     *  'paragraph', 'span', 'table', 'td', 'img'.
     */
    function copyStyles(from, to, type) {
        var source = $(from), target = $(to);

        _.each(COPY_STYLES[type], function (style) {
            target.css(style, source.css(style));
        });
    }

    /**
     * Optimize a paragraph containing sub elements for html export.
     *
     * @param {jQuery} paragraph
     *  The paragraph node as jQuery.
     *
     * @param {jQuery} parentNode
     *  The node as jQuery which is the parent for the paragraph.
     */
    function optimizeParagraphForExport(paragraph, parentNode) {

        paragraph.children().each(function () {
            var node = $(this), tmpNode = null, img = null, valign = 0;

            if (DOM.isSpan(node)) {
                valign = node.css('vertical-align');
                // check for vertical-align which is used for sub/superscript
                if (valign && valign !== 'baseline') {
                    valign = parseFloat(valign);
                    tmpNode = $((valign < 0) ? '<sub>' : '<sup>');
                    node.css('vertical-align', 'baseline');
                    parentNode.append(tmpNode);
                    tmpNode.append(node);
                } else {
                    parentNode.append(node);
                }
            } else if (DOM.isImageNode(node)) {
                tmpNode = $('<img>');
                img = node.find('img');
                if (DOM.isDocumentImageNode(node)) {
                    tmpNode.attr('src', DOM.getBase64FromImageNode(node, Image.getMimeTypeFromImageUri(DOM.getUrlParamFromImageNode(node, 'get_filename'))));
                } else {
                    tmpNode.attr('src', img.attr('src'));
                }
                // Word needs the size as attributes and cannot cope with with/height in styles!
                tmpNode.attr('width', img.css('width'));
                tmpNode.attr('height', img.css('height'));
                parentNode.append(tmpNode);
            } else if (DOM.isDrawingFrame(node)) {
                parentNode.append(node);
            } else if (DOM.isTabNode(node)) {
                tmpNode = $('<span>');
                copyStyles(node, tmpNode, 'span');
                if (node[0].textContent) {
                    tmpNode.text(node[0].textContent);
                } else {
                    DOM.ensureExistingTextNode(tmpNode);
                }
                parentNode.append(tmpNode);
            } else if (node.is('br')) {
                parentNode.append(node);
            } else if (DOM.isHardBreakNode(node)) {
                parentNode.append($('<br>'));
            }
        });
    }

    /**
     * Create paragraph style map and unique ids for paragraph nodes.
     *
     * @param {Selection} selection
     *  Current selection.
     *
     * @param {ParagraphStyles} paragraphStyles
     *  The document paragraph styles.
     *
     * @returns {Object}
     *  The paragraph styles accessible via ids.
     */
    function createParagraphStyleIds(selection, paragraphStyles) {
        var result,
            paras, len, i,
            id = 10000,
            map = {};

        // visit the paragraphs and tables covered by the text selection
        result = selection.iterateContentNodes(function (contentNode) {

            // mark paragraphs with unique id and store attributes in map
            if (DOM.isParagraphNode(contentNode)) {
                map[id] = paragraphStyles.getElementAttributes(contentNode);
                $(contentNode).attr('styleId', id);
                id++;
            // entire table: find all containing paragraph, mark them and store attributes in map
            } else if (DOM.isTableNode(contentNode)) {
                paras = $(contentNode).find('div.p');
                len = paras.length;
                for (i = 0; i < len; i++) {
                    map[id] = paragraphStyles.getElementAttributes(paras[i]);
                    $(paras[i]).attr('styleId', id);
                    id++;
                }

            } else {
                return Utils.BREAK;
            }

        }, this, { shortestPath: true });

        if (result === Utils.BREAK) {
            map = {};
        }

        return map;
    }

    /**
     * Clear paragraph style ids.
     *
     * @param {Selection} selection
     *  Current selection.
     */
    function clearParagraphStyleIds(selection) {

        // visit the paragraphs and tables covered by the text selection
        selection.iterateContentNodes(function (contentNode) {

            // paragraphs may be covered partly
            if (DOM.isParagraphNode(contentNode)) {
                $(contentNode).removeAttr('styleId');
            // entire table: generate complete operations array for the table
            } else if (DOM.isTableNode(contentNode)) {
                $(contentNode).find('div.p').removeAttr('styleId');
            } else {
                return Utils.BREAK;
            }

        }, this, { shortestPath: true });
    }

    // exports ================================================================

    return Export;

});
