/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/text/components/field/fieldmanager', [
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/text/components/field/simplefield',
    'io.ox/office/text/components/field/complexField',
    'io.ox/office/text/utils/textutils',
    'io.ox/office/text/dom',
    'io.ox/office/text/position',
    'gettext!io.ox/office/text/main'
], function (TriggerObject, TimerMixin, SimpleField, ComplexField, Utils, DOM, Position, gt) {

    'use strict';

    // class FieldManager =====================================================

    /**
     * An instance of this class represents the dispatcher class
     * for all fields (simple and complex) in the edited document.
     *
     * @constructor
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function FieldManager(app) {

        var // self reference
            self = this,
            // simple field instance
            simpleField = new SimpleField(app),
            // complex field instance
            complexField = new ComplexField(app),
            // the text model object
            model = null,
            // the selection object
            selection = null,
            // range marker object
            rangeMarker = null,
            // flag to determine if cursor position is inside complex field
            isCursorHighlighted = false,
            // saved id of currently highlighted complex field during cursor traversal or click
            currentlyHighlightedId,
            // for simple fields we store node instance of highlighted field
            currentlyHighlightedNode;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Private function called after importing document.
         * It loops all fields in collection and:
         *  - marks them for highlighting,
         *  - updates date and time fields
         *  - converts page fields in header&footer to special fields
         */
        function updateDateTimeAndSpecialFields() {
            complexField.updateDateTimeAndSpecialFields();
            simpleField.updateDateTimeFieldOnLoad();
        }

        /**
         * Receiving the start and end range markers for complex fields. Using the
         * current selection, this needs to be expanded, so that a complex field
         * is always inserted completely.
         * If the current selection is a range, it might be necessary to modify only
         * the start position or only the end position or both. If the range is inside
         * one complex field, it is not expanded. The aim of this expansion is, that,
         * if the selection includes a start range marker, the end range marker must
         * also be part of the selection. And vice versa.
         * There is another type of expansion for complex fields of type placeholder.
         * In this case, the selection shall always contain the complete placeholder.
         * Therefore the expansion also happens, if the selection has no range. This
         * check needs to be enabled because of performance reasons by the option
         * 'handlePlaceHolderComplexField'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.handlePlaceHolderComplexField=false]
         *      If set to true, it is not necessary, that the current selection
         *      has a range, that will be expanded. For complex fields of type
         *      'placeholder' also a cursor selection is automatically expanded.
         *
         * @returns {Object|Null}
         *  An object containing the properties 'start' and 'end', that contain
         *  as values the range marker start node and the range marker end node.
         *  If they can be determined, they are already jQuerified. If not, this
         *  values can be null. If range markers cannot be determined, null is
         *  returned instead of an object.
         */
        function getRangeMarkersFromSelection(options) {

            var // the start node of the current selection
                startNode = null,
                // the end node of the current selection
                endNode = null,
                // whether the start of the selection is inside a complex field range
                startIsComplexField = false,
                // whether the end of the selection is inside a complex field range
                endIsComplexField = false,
                // the id of the complex field at the start position
                startId = null,
                // the id of the complex field at the end position
                endId = null,
                // the range marker start node for the complex field
                startRangeMarker = null,
                // the range marker end node for the complex field
                endRangeMarker = null,
                // whether the range of a complex field place holder node need to be set completely
                handlePlaceHolderComplexField = Utils.getBooleanOption(options, 'handlePlaceHolderComplexField', false);

            // only expand ranges or complex fields of type 'placeholder', not text cursors inside complex fields
            if (!((selection.hasRange() || handlePlaceHolderComplexField) && selection.getSelectionType() === 'text')) { return null; }

            startNode = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
            endNode = Position.getDOMPosition(selection.getRootNode(), selection.getEndPosition());

            // TODO: Ignoring nested complex fields yet

            if (startNode && startNode.node && endNode && endNode.node) {

                startNode = startNode.node;
                endNode = endNode.node;

                if (startNode.nodeType === 3) { startNode = startNode.parentNode; }
                if (endNode.nodeType === 3) { endNode = endNode.parentNode; }

                startIsComplexField = DOM.isInsideComplexFieldRange(startNode);
                endIsComplexField = DOM.isInsideComplexFieldRange(endNode);

                // no complex field involved in selection
                if (!startIsComplexField && !endIsComplexField) { return null; }

                // at least one of the two nodes is inside a complex field
                if (startIsComplexField && !endIsComplexField) {

                    // start of selection needs to be shifted to the left
                    startId = DOM.getComplexFieldMemberId(startNode);
                    startRangeMarker = rangeMarker.getStartMarker(startId);

                } else if (endIsComplexField && !startIsComplexField) {

                    // end of selection needs to be shifted to the right
                    endId = DOM.getComplexFieldMemberId(endNode);
                    endRangeMarker = rangeMarker.getEndMarker(endId);

                } else if (startIsComplexField && endIsComplexField) {

                    startId = DOM.getComplexFieldMemberId(startNode);
                    endId = DOM.getComplexFieldMemberId(endNode);

                    // both nodes are inside the same complex field -> nothing to do
                    if (startId === endId && !handlePlaceHolderComplexField) { return null; }

                    // start and end node are inside different complex fields
                    // -> the selection needs to be expanded in both directions
                    startRangeMarker = rangeMarker.getStartMarker(startId);
                    endRangeMarker = rangeMarker.getEndMarker(endId);
                }
            }

            return { start: startRangeMarker, end: endRangeMarker };
        }

        /**
         * Check, if an existing selection range needs to be expanded, so that a complex field
         * is completely part of the selection range. So it is not possible to remove only a part
         * of a selected range.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object
         *
         * @param {Function} selectionHandler
         *  The private selection handler function, that allows modification of start position or
         *  end position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.handlePlaceHolderComplexField=false]
         *      If set to true, it is not necessary, that the current selection
         *      has a range, that will be expanded. For complex fields of type
         *      'placeholder' also a cursor selection is automatically expanded.
         *      This value is used inside the function 'getRangeMarkersFromSelection'.
         */
        function checkRangeSelection(event, selectionHandler, options) {

            var // an object with the neighboring range marker nodes
                rangeMarkers = getRangeMarkersFromSelection(options);

            if (rangeMarkers !== null) {
                if (rangeMarkers.start) {
                    selectionHandler(Position.getOxoPosition(selection.getRootNode(), rangeMarkers.start), { start: true, selectionExternallyModified: true });
                }

                if (rangeMarkers.end) { // end position need to be increased by 1 for selection range
                    selectionHandler(Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), rangeMarkers.end)), { start: false, selectionExternallyModified: true });
                }
            }
        }

        /**
         * Handler for highlighting of complex fields, after selection has changed.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Function} options
         *  Passed selection options.
         *
         */
        function checkHighlighting(event, options) {
            var isSimpleTextSelection = Utils.getBooleanOption(options, 'simpleTextSelection', false),
                domPos = null,
                node = null,
                atEndOfSpan = null;

            function isBeforeCxFldStart(element) {
                return DOM.isRangeMarkerStartNode(element) && DOM.getRangeMarkerType(element) === 'field';
            }

            if ((complexField.isEmpty() && simpleField.isEmpty()) || isSimpleTextSelection || selection.hasRange()) {
                return;
            } else {
                domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
                if (domPos && domPos.node) {
                    node = domPos.node;
                    if (node.nodeType === 3) {
                        atEndOfSpan = domPos.offset === node.textContent.length;
                        node = node.parentNode;
                    }
                    if ((DOM.isComplexFieldNode(node) || DOM.isInsideComplexFieldRange(node))) {
                        self.createEnterHighlight(node);
                    } else if (atEndOfSpan && isBeforeCxFldStart(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling.nextSibling); // spec. handling for cursor positioned before cx field
                    } else if (atEndOfSpan && DOM.isFieldNode(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling, { simpleField: true });
                    } else if (self.isHighlightState()) {
                        self.removeEnterHighlight();
                    }
                }
            }
        }

        /**
         * Callback function for event listener after initial page breaks
         * rendering is finished and after document is loaded.
         *
         * Stops operation distribution to server, and updates possible date fields,
         * and convert and update special page fields in headers.
         */
        function updateFieldsCallback() {
            app.stopOperationDistribution(function () {
                updateDateTimeAndSpecialFields();
                model.getUndoManager().clearUndoActions();
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Updating field with given instruction.
         * Depending of passed instruction, different type of field is updated.
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} instruction
         *  Complex field instruction, containing type,
         *  formating and optionally style, separated by \ .
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.complex=false]
         *      If field is simple or complex type.
         */
        this.updateByInstruction = function (node, instruction, options) {
            var isComplex = Utils.getBooleanOption(options, 'complex', false);

            if (instruction) {
                if (isComplex) {
                    complexField.updateByInstruction(node, instruction);
                } else {
                    simpleField.updateByInstruction(node, instruction);
                }
            } else {
                Utils.warn('FieldManager.updateByInstruction(): missing instruction!');
            }
        };

        /**
         * Set highlighting state of complex field during cursor traversal.
         *
         * @param {Boolean} state
         *  If cursor is inside complex field or not.
         */
        this.setCursorHighlightState = function (state) {
            isCursorHighlighted = state === true ? true : false;
        };

        /**
         * If complex field highlight state is active, or not.
         *
         * @returns {Boolean}
         *  Whether the complex field highlight state is active, or not.
         */
        this.isHighlightState = function () {
            return isCursorHighlighted;
        };

        /**
         * Provides a jQuerified range start marker node.
         *
         * @returns {jQuery|null}
         *  The start marker with the specified id, or null, if no such start marker exists.
         */
        this.getComplexField = function (id) {
            return complexField.getComplexField(id);
        };

        /**
         * Check if there are complex fields in document.
         *
         * @returns {Boolean}
         */
        this.isComplexFieldEmpty = function () {
            return complexField.isEmpty();
        };

        /**
         * Gets simple field from collection by passed id.
         *
         * @param {String} id
         *  Id of queried field.
         *
         * @returns {jQuery|null}
         *  The simple field with the specified id, or null, if no such start marker exists.
         *
         */
        this.getSimpleField = function (id) {
            return simpleField.getSimpleField(id);
        };

        /**
         * Check if there are simple fields in document.
         *
         * @returns {Boolean}
         */
        this.isSimpleFieldEmpty = function () {
            return simpleField.isEmpty();
        };

        /**
         * Check if there are any type of fields in document.
         *
         * @returns {Boolean}
         */
        this.fieldsAreInDocument = function () {
            return !simpleField.isEmpty() || !complexField.isEmpty();
        };

        /**
         * Check, whether the specified target string is a valid target for a complex field.
         *
         * @param {String} target
         *  The target string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a complex field node.
         */
        this.isComplexFieldId = function (id) {
            return self.getComplexField(id) !== null;
        };

        /**
         * When pasting content, that contains complex fields, it is necessary, that the
         * complex fields get a new unique id.
         *
         * @param {Object[]} operations
         *  The collection with all paste operations.
         */
        this.handlePasteOperationTarget = function (operations) {
            complexField.handlePasteOperationTarget(operations);
        };

        /**
         * Callback function for mouseenter event trigger on complex field.
         *
         * @param {Object} event
         *  jQuery object event triggered on hover in.
         */
        this.createHoverHighlight = function (event) {
            var // id of the hovered field
                id = DOM.getComplexFieldMemberId(event.currentTarget);

            if (currentlyHighlightedId !== id) {
                rangeMarker.highLightRange(id, 'cx-field-highlight');
            }

            // fallback with jquery/css highlighting
            //model.getCurrentRootNode().find('.complex' + id).addClass('cx-field-highlight');
        };

        /**
         * Callback function for mouseleave event trigger on complex field.
         *
         * @param {Object} event
         *  jQuery object event triggered on hover out.
         */
        this.removeHoverHighlight = function (event) {
            var // id of the hovered field
                id = DOM.getComplexFieldMemberId(event.currentTarget);

            if (currentlyHighlightedId !== id) {
                rangeMarker.removeHighLightRange(id);
            }

            // fallback with jquery/css highlighting
            //model.getCurrentRootNode().find('.complex' + id).removeClass('cx-field-highlight');
        };

        /**
         * Creates highlight range on enter cursor in complex field,
         * with keyboard key pressed or mouse click.
         *
         * @param {jQuery} field
         *  Field to be highlighted.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.simpleField=false]
         */
        this.createEnterHighlight = function (field, options) {
            var // id of the highlighted field
                id = DOM.getComplexFieldMemberId(field),
                simpleField = Utils.getBooleanOption(options, 'simpleField', false);

            if (simpleField) {
                if (currentlyHighlightedId || currentlyHighlightedNode) {
                    self.removeEnterHighlight();
                }
                currentlyHighlightedNode = $(field);
                currentlyHighlightedNode.addClass('sf-highlight');
            } else {
                if (currentlyHighlightedId !== id) {
                    if (currentlyHighlightedId || currentlyHighlightedNode) {
                        self.removeEnterHighlight();
                    }
                    currentlyHighlightedId = id;
                    rangeMarker.highLightRange(id, 'cx-field-highlight');
                }
            }
            this.setCursorHighlightState(true);
        };

        /**
         * Removes highlight range on enter cursor in complex field, or in simple field,
         * with keyboard key pressed or mouse click.
         */
        this.removeEnterHighlight = function () {
            if (currentlyHighlightedNode) {
                // simple field
                currentlyHighlightedNode.removeClass('sf-highlight');
                currentlyHighlightedNode = null;
            } else {
                rangeMarker.removeHighLightRange(currentlyHighlightedId);
                currentlyHighlightedId = null;
            }
            this.setCursorHighlightState(false);
        };

        /**
         * Forces update of highlighted simple/complex field content.
         */
        this.updateHighlightedField = function () {
            var node,
                instruction,
                atEndOfSpan,
                domPos,
                node;

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
                instruction = DOM.getFieldInstruction(node);
                self.updateByInstruction(node, instruction);

                // remove highlight from deleted node
                currentlyHighlightedNode = null;
                this.setCursorHighlightState(false);

                // add highlight to newly created node
                domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
                if (domPos && domPos.node) {
                    node = domPos.node;
                    if (node.nodeType === 3) {
                        atEndOfSpan = domPos.offset === node.textContent.length;
                        node = node.parentNode;
                    }
                    if (atEndOfSpan && DOM.isFieldNode(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling, { simpleField: true });
                    }
                }
            } else if (currentlyHighlightedId) { // complex field
                node = self.getComplexField(currentlyHighlightedId);
                instruction = DOM.getComplexFieldInstruction(node);
                self.updateByInstruction(node, instruction, { complex: true });
            }
        };

        /**
         * Forces update of complex fields inside selection range.
         */
        this.updateHighlightedFieldSelection = function () {
            var instruction = null;
            if (simpleField.isEmpty() && complexField.isEmpty()) { return; }

            selection.iterateNodes(function (node) {
                if (DOM.isComplexFieldNode(node)) {
                    instruction = DOM.getComplexFieldInstruction(node);
                    self.updateByInstruction(node, instruction, { complex: true });
                } else if (DOM.isFieldNode(node)) {
                    instruction = DOM.getFieldInstruction(node);
                    self.updateByInstruction(node, instruction);
                }
            });
        };

        /**
         * Update all fields in document.
         *
         */
        this.updateAllFields = function () {
            var // the generate operations deferred
                generateDef = null;

            // helper iterator function for handling field update
            function handleFieldUpdate (entry, fieldId) {
                var field,
                    instruction;

                if (fieldId && fieldId.indexOf('s') > -1) { // simple field
                    field = self.getSimpleField(fieldId);
                    instruction = DOM.getFieldInstruction(entry);
                    if (!DOM.isSpecialField(field)) {
                        self.updateByInstruction(field, instruction);
                    }
                } else { // complex field
                    field = self.getComplexField(fieldId);
                    instruction = DOM.getComplexFieldInstruction(field);
                    if (!DOM.isSpecialField(field)) {
                        self.updateByInstruction(field, instruction, { complex: true });
                    }
                }
            }

            if (self.fieldsAreInDocument()) {
                var mergedFields = _.extend(simpleField.getAllFields(), complexField.getAllFields());

                // show a message with cancel button
                app.getView().enterBusy({
                    cancelHandler: function () {
                        if (generateDef && generateDef.abort) {
                            generateDef.abort();
                        }
                    },
                    warningLabel: gt('Sorry, updating of all fields in document will take some time.')
                });

                // iterate objects
                generateDef = self.iterateObjectSliced(mergedFields, handleFieldUpdate, { delay: 'immediate', infoString: 'Text: handleFieldUpdate' })
                .always(function () {
                    // leave busy state
                    app.getView().leaveBusy().grabFocus();
                });

                // add progress handling
                generateDef.progress(function (partialProgress) {
                    var progress = 0.2 + (partialProgress * 0.8);
                    app.getView().updateBusyProgress(progress);
                });
            }
        };

        /**
         * This method dispatch parent node to simple and complex field, for removing it.
         *
         * @param {jQuery|Node} parentNode
         *  Element that's being searched for simple and complex fields.
         */
        this.removeAllFieldsInNode = function (parentNode) {
            // checking, if the paragraph contains additional complex field nodes, that need to be removed, too.
            complexField.removeAllInsertedComplexFields(parentNode);
            // checking, if the paragraph contains additional simple field nodes, that need to be removed, too.
            simpleField.removeSimpleFieldsInNode(parentNode);
        };

        /**
         * Dispatch function to complexField class to check if search node in given start and end range
         * contains special fields that needs to be restored, like before delete operation.
         *
         * @param {Node} searchNode
         *  Node which we search for special fields.
         *
         * @param {Number} startOffset
         *  Start offset of given node.
         *
         * @param {Number} endOffset
         *  End offset of given node.
         */
        this.checkRestoringSpecialFields = function (searchNode, startOffset, endOffset) {
            complexField.checkRestoringSpecialFields(searchNode, startOffset, endOffset);
        };

        /**
         * Dispatch call to complex field to handle restoring of special fields in container node.
         *
         * @param {jQuery|Node} containerNode
         *  Node which we search for special fields.
         */
        this.checkRestoringSpecialFieldsInContainer = function (containerNode) {
            complexField.checkRestoringSpecialFieldsInContainer(containerNode);
        };

        /**
         * Dispatch function to simpleField class's method removeSimpleFieldFromCollection.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         */
        this.removeSimpleFieldFromCollection = function (node) {
            simpleField.removeSimpleFieldFromCollection(node);
        };

        /**
         * Dispatch function to complexField class's method removeFromComplexFieldCollection.
         *
         * @param {HTMLElement|jQuery} field
         *  One complex field node.
         */
        this.removeFromComplexFieldCollection = function (node) {
            complexField.removeFromComplexFieldCollection(node);
        };

        /**
         * Dispatch to simpleField for adding node to collection.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         *
         * @param {String} [target]
         *  The target, where the simple field is located.
         */
        this.addSimpleFieldToCollection = function (fieldNode, target) {
            simpleField.addSimpleFieldToCollection(fieldNode, target);
        };

        /**
         * After splitting a paragraph, it is necessary, that all complex field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         * This method dispatches to complex field class.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateComplexFieldCollector = function (paragraph) {
            complexField.updateComplexFieldCollector(paragraph);
        };

        /**
         * After splitting a paragraph, it is necessary, that all complex field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         * This method dispatches to complex field class.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateSimpleFieldCollector = function (paragraph) {
            simpleField.updateSimpleFieldCollector(paragraph);
        };

        /**
         * Dispatch to complex field's method updateCxPageFieldsInMarginals.
         */
        this.updatePageFieldsInMarginals = function () {
            complexField.updateCxPageFieldsInMarginals();
        };

        /**
         * Dispatches to handler for insertComplexField operations.
         *
         * @param {Number[]} start
         *  The logical start position for the new complex field.
         *
         * @param {String} id
         *  The unique id of the complex field.
         *
         * @param {String} instruction
         *  The instructions of the complex field.
         *
         * @param {String} target
         *  The target string corresponding to the specified start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been inserted successfully.
         */
        this.insertComplexFieldHandler = function (start, id, instruction, target) {
            return complexField.insertComplexFieldHandler(start, id, instruction, target);
        };

        /**
         * Dispatch to class complexField, to convert field to special field.
         *
         * @param {String} fieldId
         *  Id of the field
         *
         * @param {String} marginalTarget
         *  Id of the marginal target node
         *
         * @param {String} type
         *  Type of field: num pages or page number
         */
        this.convertToSpecialField = function (fieldId, marginalTarget, type) {
            complexField.convertToSpecialField(fieldId, marginalTarget, type);
        };

        /**
         * Dispatch update of date time fields on download to field classes.
         */
        this.prepareDateTimeFieldsDownload = function () {

            // helper iterator function for handling field update
            function handleFieldUpdate (entry, fieldId) {
                var field,
                    instruction;

                if (fieldId && fieldId.indexOf('s') > -1) { // simple field
                    field = self.getSimpleField(fieldId);
                    instruction = DOM.getFieldInstruction(entry);
                    simpleField.updateDateTimeField(field, instruction);
                } else { // complex field
                    field = self.getComplexField(fieldId);
                    instruction = DOM.getComplexFieldInstruction(field);
                    complexField.updateDateTimeFieldOnDownload(field, instruction);
                }
            }

            var mergedFields = _.extend(simpleField.getAllFields(), complexField.getAllFields());
            _.each(mergedFields, handleFieldUpdate);

        };

        /**
         * Calls method in simpleField class, for updating number of pages field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageCountInCurrentNode = function ($node, pageCount) {
            simpleField.updatePageCountInCurrentNode($node, pageCount);
        };

        /**
         * Calls method in simpleField class, for updating current page number field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Boolean} isHeader
         *  If node is header or not.
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageNumInCurrentNode = function ($node, isHeader, pageCount) {
            simpleField.updatePageNumInCurrentNode($node, isHeader, pageCount);
        };

        /**
         * Calls method in complexField class, for updating number of pages field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageCountInCurrentNodeCx = function ($node, pageCount) {
            complexField.updatePageCountInCurrentNodeCx($node, pageCount);
        };

        /**
         * Calls method in complexField class, for updating current page number field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Boolean} isHeader
         *  If node is header or not.
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageNumInCurrentNodeCx = function ($node, isHeader, pageCount) {
            complexField.updatePageNumInCurrentNodeCx($node, isHeader, pageCount);
        };

        this.updatePageNumInCurrentMarginalCx = function ($node, isHeader, pageNum) {
            complexField.updatePageNumInCurrentMarginalCx($node, isHeader, pageNum);
        };

        /**
         * Checks if selection contains special complex fields of type page inside header/footer.
         *
         * @returns {Boolean}
         */
        this.checkIfSpecialFieldsSelected = function () {
            var flag = false;
            if (model.isHeaderFooterEditState() && !self.isComplexFieldEmpty()) {
                selection.iterateNodes(function (node) {
                    if (DOM.isSpecialField(node)) {
                        flag = true;
                        return Utils.BREAK;
                    }
                });
            }
            return flag;
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            var d1 = $.Deferred();
            var d2 = $.Deferred();

            model = app.getModel();
            selection = model.getSelection();
            rangeMarker = model.getRangeMarker();

            app.onImportSuccess(function () {
                if (model.isLocalStorageImport() || model.isFastLoadImport()) {
                    complexField.refreshComplexFields(model.isLocalStorageImport(), model.isFastLoadImport());
                    simpleField.refreshSimpleFields();
                    // refreshing also the range marker model, if it was not done before
                    rangeMarker.refreshModel({ onlyOnce: true });
                }
                // resolve import success for update fields callback
                d2.resolve();
            });

            app.onImportFailure(function () {
                // reject promise on import failed, so that update fields doesn't hang
                d2.reject();
            });

            model.one('pageBreak:after initialPageBreaks:done', function () { return d1.resolve(); });
            // only when import is finished and page breaks are rendered call update fields
            $.when(d1, d2).then(updateFieldsCallback);

            selection.on('position:calculated', checkRangeSelection);
            selection.on('change', checkHighlighting);

            model.on('document:reloaded', function () {
                complexField.refreshComplexFields(true, false);
                simpleField.refreshSimpleFields();
            });

        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            simpleField.destroy();
            complexField.destroy();
            model = selection = rangeMarker = null;
        });

    } // class FieldManager

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: FieldManager });
});
