/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/text/components/field/basefield', [
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/text/utils/textutils'
], function (TriggerObject, Utils) {

    'use strict';

    // class BaseField ============================================

    /**
     * An instance of this class represents the base class for all fields in the
     * edited document.
     *
     * @constructor
     *
     */
    function BaseField(app) {

        var // instance of the model
            model = null,
            // instance of the number formatter
            numberFormatter = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        // methods ------------------------------------------------------------

        /**
         * Helper functions to determine type of field from instruction
         */
        this.isNumPages = function (type) {
            return (/NUMPAGES/i).test(type);
        };

        this.isPageNumber = function (type) {
            return (/PAGE/i).test(type);
        };

        this.isCurrentDate = function (type) {
            return (/^DATE/i).test(type);
        };

        this.isCurrentTime = function (type) {
            return (/^TIME/i).test(type);
        };

        /*
        function isPrintDate(type) {
            return (/PRINTDATE/i).test(type);
        }

        function isSaveDate(type) {
            return (/SAVEDATE/i).test(type);
        }
        */

        this.isFileName = function (type) {
            return (/FILE\-?NAME/i).test(type);
        };

        /**
         * Parse instruction format. Date format starts with @, and number format with *.
         *
         * @param {String} instruction
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.dateTime=false]
         *      If we want to parse instruction for date/time field or not.
         *
         * @returns {String}
         *  Parsed format from instruction.
         *
         */
        this.parseFormatInstruction = function (instruction, options) {
            var tempFormat = null,
                dateTime = Utils.getBooleanOption(options, 'dateTime', false);

            if (instruction) {
                tempFormat = instruction.replace(/\"/g, '');
                if (dateTime) {
                    if (tempFormat && tempFormat.indexOf('@') === 0) {
                        return Utils.trimString(tempFormat.replace(/\@/g, ''));
                    }
                } else {
                    if (tempFormat && tempFormat.indexOf('*') === 0) {
                        return Utils.trimString(tempFormat.replace(/\*/g, ''));
                    }
                }
            }

            return 'default';
        };

        /**
         * From raw passed instruction string, extract type,
         * and clean up remaining instruction for future formatting.
         *
         * @param {String} instruction
         *  Instruction to be cleaned up.
         *
         * @returns {Object}
         */
        this.cleanUpAndExtractType = function (instruction) {
            var type = null;

            instruction = instruction.replace(/\\\* MERGEFORMAT/g, ''); // ignoring merge format for now
            instruction = instruction.replace(/\\\./g, '.'); // strip escape characters
            instruction = instruction.split('\\');
            if (_.isArray(instruction) && instruction.length > 0) {
                type =  Utils.trimString(instruction[0]);
                instruction = instruction.length > 1 ? instruction[1] : null;
            }
            return { type: type, instruction: instruction };
        };

        /**
         * Converts number for page and numPages fields into proper format code.
         *
         * @param {Number} number
         *  Number to be inserted into field.
         *
         * @param {String} format
         *  Format of the number inserted into field.
         *
         * @param {Boolean} [ignoreDash]
         *  NumPages field doesn't have arabicDash formatting.
         *
         * @returns {String}
         *  Formatted number.
         */
        this.formatPageFieldInstruction = function (number, format, ignoreDash) {
            if (number && format) {
                if ((/ROMAN/).test(format)) {
                    number = numberFormatter.toRoman(number);
                } else if ((/roman/).test(format)) {
                    number = numberFormatter.toRoman(number).toLowerCase();
                } else if ((/ALPHABETIC/).test(format)) {
                    number = numberFormatter.toAlphabetic(number).toUpperCase();
                } else if ((/alphabetic/).test(format)) {
                    number = numberFormatter.toAlphabetic(number);
                } else if ((/ArabicDash/).test(format) && !ignoreDash) {
                    number = numberFormatter.toArabicDash(number);
                } else {
                    number = number.toString();
                }
                return number;
            }

            Utils.error('field.formatPageFieldInstruction(): missing function parameters! ', number, format);
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            numberFormatter = model.getNumberFormatter();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = numberFormatter = null;
        });

    } // class BaseField

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BaseField });
});
