/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/highlightrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, TriggerObject, TimerMixin, SheetUtils, PaneUtils, RenderUtils) {

    'use strict';

    var // convenience shortcuts
        RangeArray = SheetUtils.RangeArray;

    // class HighlightRenderer ================================================

    /**
     * Renders the highlighted cell ranges (shown e.g. during editing formulas
     * with references into the active sheet) into the DOM layers of a single
     * grid pane.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this highlighting layer renderer.
     */
    function HighlightRenderer(gridPane) {

        var // self reference
            self = this,

            // the spreadsheet model and view
            docView = gridPane.getDocView(),
            docModel = docView.getDocModel(),

            // the range highlight layer (container for the highlighted ranges of formulas)
            highlightLayerNode = gridPane.createLayerNode('highlight-layer'),

            // the active range layer (container for the active range selected in cell edit mode)
            activeRangeLayerNode = gridPane.createLayerNode('active-layer'),

            // animation timer for marching-ants effect
            animationTimer = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Renders all visible highlighted cell ranges.
         */
        function renderHighlightedRanges() {

            var // the HTML mark-up for all highlight ranges
                markup = '';

            // render all ranges that are visible in this grid pane
            if (gridPane.isVisible()) {

                var // current document edit mode
                    editMode = docModel.getEditMode(),
                    // the index of the active sheet
                    sheet = docView.getActiveSheet(),
                    // all range descriptors of the active sheet
                    highlightRanges = [],
                    // the cell range addresses of all ranges to be rendered
                    ranges = new RangeArray(),
                    // maps range keys to zero-based color indexes
                    rangeColors = {},
                    // next free zero-based color index
                    nextColor = 0;

                // collect all range descriptors of the active sheet
                docView.iterateHighlightedRanges(function (rangeDesc, rangeIndex) {
                    if (rangeDesc.range.containsSheet(sheet)) {
                        highlightRanges.push({ rangeDesc: rangeDesc, rangeIndex: rangeIndex });
                        ranges.push(rangeDesc.range.toRange()); // push simple range without sheet indexes
                    }
                });

                // render the highlighted ranges
                gridPane.iterateRangesForRendering(ranges, function (range, rectangle, index) {

                    var // the current range descriptor
                        rangeDesc = highlightRanges[index].rangeDesc,
                        // the original array index of the current range descriptor
                        rangeIndex = highlightRanges[index].rangeIndex,
                        // the unique key of the formula token containing the range (will be resolved to a scheme color)
                        tokenKey = rangeDesc.tokenKey,
                        // whether the range will be rendered with tracking handle elements
                        draggable = editMode && rangeDesc.draggable,
                        // whether the range is currently tracked (rendered with special glow effect)
                        tracking = draggable && rangeDesc.tracking;

                    // register a new range key in the 'rangeColors' map
                    if (!(tokenKey in rangeColors)) {
                        rangeColors[tokenKey] = nextColor;
                        nextColor += 1;
                    }

                    var // one-based index of the scheme color to be used for the range
                        styleId = (rangeColors[tokenKey] % Utils.SCHEME_COLOR_COUNT) + 1;

                    // adjust position of static ranges
                    if (!draggable) {
                        rectangle.left -= 1;
                        rectangle.top -= 1;
                        rectangle.width += 1;
                        rectangle.height += 1;
                    }

                    // generate the HTML mark-up for the highlighted range
                    markup += '<div class="range' + (tracking ? ' tracking-active' : '') + '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"';
                    markup += ' data-style="' + styleId + '" data-index="' + rangeIndex + '"><div class="fill"></div>';
                    if (draggable) {
                        markup += '<div class="borders">';
                        markup += _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div><div class="resizers">';
                        markup += _.map(['tl', 'tr', 'bl', 'br'], function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div>';
                    } else {
                        markup += '<div class="static-border"></div>';
                    }
                    markup += '</div>';
                });
            }

            // insert entire HTML mark-up into the container node
            highlightLayerNode[0].innerHTML = markup;
        }

        /**
         * Renders the activated cell ranges. These ranges are used for example
         * while selecting new ranges for a formula while in cell edit mode.
         */
        function renderActiveSelection() {

            var // the active ranges
                selection = docView.getSheetViewAttribute('activeSelection'),
                // the HTML mark-up for the range
                markup = '';

            // generate the HTML mark-up for the ranges
            if (gridPane.isVisible() && selection) {
                gridPane.iterateRangesForRendering(selection.ranges, function (range, rectangle) {
                    markup += '<div class="range" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"><div class="borders">';
                    markup += _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                    markup += '</div></div>';
                });
                if (!animationTimer) {
                    animationTimer = self.repeatDelayed(function (index) {
                        activeRangeLayerNode.attr('data-frame', index % 6);
                    }, 100);
                }
            } else if (animationTimer) {
                animationTimer.abort();
                animationTimer = null;
            }

            // insert entire HTML mark-up into the container node
            activeRangeLayerNode[0].innerHTML = markup;
        }

        /**
         * Renders the active selection of the sheet after it has been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {
            if ('activeSelection' in attributes) {
                renderActiveSelection();
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Changes the layers according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('HighlightRenderer.setLayerRange()', function () {
            renderHighlightedRanges();
            renderActiveSelection();
        });

        /**
         * Resets this renderer, clears the DOM layer nodes.
         */
        this.hideLayerRange = function () {
            highlightLayerNode.empty();
            activeRangeLayerNode.empty();
        };

        // initialization -----------------------------------------------------

        // update layers when document edit mode or view attributes have changed (only, if this renderer is visible)
        gridPane.listenToWhenVisible(this, docModel, 'change:editmode', renderHighlightedRanges);
        gridPane.listenToWhenVisible(this, docView, 'change:highlight:ranges', renderHighlightedRanges);
        gridPane.listenToWhenVisible(this, docView, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = gridPane = docModel = docView = null;
            highlightLayerNode = activeRangeLayerNode = animationTimer = null;
        });

    } // class HighlightRenderer

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: HighlightRenderer });

});
