/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/sheetselection', [
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SheetUtils) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Range = SheetUtils.Range,
        RangeArray = SheetUtils.RangeArray;

    // private global functions ===============================================

    /**
     * Returns whether the passed addresses are equal. Either of the parameters
     * can be null.
     */
    function equalOptAddresses(address1, address2) {
        return (!address1 && !address2) || (address1 && address2 && address1.equals(address2));
    }

    // class SheetSelection ===================================================

    /**
     * Represents a complete sheet selection (cell ranges and drawing objects).
     *
     * @constructor
     *
     * @property {RangeArray} ranges
     *  The addresses of all selected cell ranges.
     *
     * @property {Number} active
     *  Element index of the active range in the property 'ranges'.
     *
     * @property {Address} address
     *  The address of the active cell in the active range.
     *
     * @property {Address|Null} origin
     *  The exact position of the active cell. Used in case a merged range has
     *  been set as active cell using the keyboard, to keep track of the exact
     *  cell position when moving the active cell again.
     *
     * @property {Array<Array<Number>>} drawings
     *  The positions of all selected drawing frames in the sheet, as integer
     *  arrays (without leading sheet index).
     */
    function SheetSelection(ranges, active, address, drawings) {

        // initialize array of cell range addresses (default to empty array)
        this.ranges = ranges ? RangeArray.get(ranges) : new RangeArray();

        // initialize array index of active range (restrict to existing ranges, default to first range)
        this.active = Math.min(_.isNumber(active) ? Math.max(active, 0) : 0, this.ranges.length - 1);

        // initialize address (fall-back to start cell of active range)
        this.address = address ? address : this.ranges.empty() ? Address.A1.clone() : this.activeRange().start.clone();

        // support an additional address property, used to store the exact position in merged ranges
        this.origin = null;

        // initialize drawing selection (fall-back to empty array)
        this.drawings = _.isArray(drawings) ? drawings : [];

    } // class SheetSelection

    // static methods ---------------------------------------------------------

    /**
     * Creates a selection instance from a single cell address. The selection
     * will contain a single cell range covering the address, and the active
     * cell will be set to the address.
     *
     * @param {Address} address
     *  The cell address to create the sheet selection instance from.
     *
     * @returns {SheetSelection}
     *  The new sheet selection instance.
     */
    SheetSelection.createFromAddress = function (address) {
        return new SheetSelection(new Range(address.clone()));
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a deep clone of this selection instance.
     *
     * @returns {SheetSelection}
     *  A deep clone of this selection instance.
     */
    SheetSelection.prototype.clone = function () {
        var selection = new SheetSelection(this.ranges.clone(true), this.active, this.address.clone(), _.copy(this.drawings, true));
        if (this.origin) { selection.origin = this.origin.clone(); }
        return selection;
    };

    /**
     * Returns whether the passed sheet selection is equal to this sheet
     * selection.
     *
     * @param {SheetSelection} selection
     *  The other sheet selection to be compared to this instance.
     *
     * @returns {Boolean}
     *  Whether both sheet selections are equal.
     */
    SheetSelection.prototype.equals = function (selection) {
        return (this.active === selection.active) && // most simple comparison first
            this.address.equals(selection.address) &&
            equalOptAddresses(this.origin, selection.origin) &&
            this.ranges.equals(selection.ranges) &&
            _.isEqual(this.drawings, selection.drawings);
    };

    /**
     * Returns the address of the active range in this selection.
     *
     * @returns {Range|Null}
     *  The address of the active cell range; or null, if this selection is
     *  empty.
     */
    SheetSelection.prototype.activeRange = function () {
        return this.ranges[this.active] || null;
    };

    /**
     * Returns whether this selection contains the passed cell address.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether a cell range address in this selection contains the passed cell
     *  address.
     */
    SheetSelection.prototype.containsAddress = function (address) {
        return this.ranges.containsAddress(address);
    };

    /**
     * Returns whether the passed cell address is the active cell of this
     * selection.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed cell address is the active cell of this selection.
     */
    SheetSelection.prototype.isActive = function (address) {
        return !!this.address && this.address.equals(address);
    };

    // exports ================================================================

    return SheetSelection;

});
