/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/range', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/address'
], function (Utils, Interval, Address) {

    'use strict';

    var // pattern for a single cell address in a range
        CELL_PATTERN = '(' + Address.COL_PATTERN + ')(' + Address.ROW_PATTERN + ')',

        // regular expression object matching a range name
        RE_RANGE_NAME = new RegExp('^' + CELL_PATTERN + ':' + CELL_PATTERN + '$', 'i');

    // class Range ============================================================

    /**
     * Represents the address of a rectangular cell range in a sheet of a
     * spreadsheet document.
     *
     * @constructor
     *
     * @property {Address} start
     *  The address of the top-left cell in the range.
     *
     * @property {Address} end
     *  The address of the bottom-right cell in the range.
     *
     * @param {Address} start
     *  The address of the top-left cell in the range.
     *
     * @param {Address} [end]
     *  The address of the bottom-right cell in the range. Can be omitted to
     *  construct a cell range address covering a single cell.
     */
    function Range(start, end) {

        this.start = start;
        this.end = end ? end : start.clone();

    } // class Range

    // static methods ---------------------------------------------------------

    /**
     * Creates a cell range address from the passed column and row indexes. The
     * columns and rows in the resulting cell range address will be ordered
     * automatically.
     *
     * @param {Number} col1
     *  The column index of one of the borders in the cell range.
     *
     * @param {Number} row1
     *  The row index of one of the borders in the cell range.
     *
     * @param {Number} col2
     *  The column index of the other border in the cell range.
     *
     * @param {Number} row2
     *  The row index of the other border in the cell range.
     *
     * @returns {Range}
     *  The cell range address with adjusted column and row indexes.
     */
    Range.create = function (col1, row1, col2, row2) {
        var start = new Address(Math.min(col1, col2), Math.min(row1, row2)),
            end = new Address(Math.max(col1, col2), Math.max(row1, row2));
        return new Range(start, end);
    };

    /**
     * Creates a cell range address from the passed column and row interval.
     *
     * @param {Interval} colInterval
     *  The column interval of the new cell range address.
     *
     * @param {Interval} rowInterval
     *  The row interval of the new cell range address.
     *
     * @returns {Range}
     *  The cell range address created from the passed index intervals.
     */
    Range.createFromIntervals = function (colInterval, rowInterval) {
        return Range.create(colInterval.first, rowInterval.first, colInterval.last, rowInterval.last);
    };

    /**
     * Creates a cell range address from the passed column interval and row
     * indexes.
     *
     * @param {Interval} colInterval
     *  The column interval of the new cell range address.
     *
     * @param {Number} firstRow
     *  The index of the first row in the new cell range address.
     *
     * @param {Number} [lastRow]
     *  The index of the last row in the new cell range address. If omitted,
     *  the range will cover a single row only.
     *
     * @returns {Range}
     *  The cell range address created from the passed column interval and row
     *  indexes.
     */
    Range.createFromColInterval = function (colInterval, firstRow, lastRow) {
        if (typeof lastRow !== 'number') { lastRow = firstRow; }
        return Range.create(colInterval.first, firstRow, colInterval.last, lastRow);
    };

    /**
     * Creates a cell range address from the passed row interval and column
     * indexes.
     *
     * @param {Interval} rowInterval
     *  The row interval of the new cell range address. Note that this method
     *  is the one of the rare cases where row parameters precede column
     *  parameters, in order to provide an optional column parameter.
     *
     * @param {Number} firstCol
     *  The index of the first column in the new cell range address.
     *
     * @param {Number} [lastCol]
     *  The index of the last column in the new cell range address. If omitted,
     *  the range will cover a single column only.
     *
     * @returns {Range}
     *  The cell range address created from the passed row interval and column
     *  indexes.
     */
    Range.createFromRowInterval = function (rowInterval, firstCol, lastCol) {
        if (typeof lastCol !== 'number') { lastCol = firstCol; }
        return Range.create(firstCol, rowInterval.first, lastCol, rowInterval.last);
    };

    /**
     * Creates a cell range address from the passed cell addresses. The columns
     * and rows in the resulting cell range address will be ordered
     * automatically. The passed cell addresses will be deeply cloned when
     * creating the new range address.
     *
     * @param {Address} address1
     *  The first cell address used to create the new cell range address.
     *
     * @param {Address} [address2]
     *  The second cell address used to create the new cell range address. If
     *  omitted, the range will represent a single cell address.
     *
     * @returns {Range}
     *  The cell range address created from the passed cell addresses.
     */
    Range.createFromAddresses = function (address1, address2) {
        return address2 ? Range.create(address1[0], address1[1], address2[0], address2[1]) : new Range(address1.clone());
    };

    /**
     * Returns the cell range address for the passed string representation.
     *
     * @param {String} str
     *  The string representation of a cell range address, in A1 notation, with
     *  a colon as separator between start and end address.
     *
     * @returns {Range|Null}
     *  The cell range address (with adjusted column and row indexes), if the
     *  passed string is valid, otherwise null.
     */
    Range.parse = function (str) {
        var matches = RE_RANGE_NAME.exec(str);
        if (!matches) { return null; }
        var col1 = Address.parseCol(matches[1]),
            row1 = Address.parseRow(matches[2]),
            col2 = Address.parseCol(matches[3]),
            row2 = Address.parseRow(matches[4]);
        return (Math.min(col1, row1, col2, row2) >= 0) ? Range.create(col1, row1, col2, row2) : null;
    };

    /**
     * Compares the passed cell range addresses. Defines a natural order for
     * sorting with the following behavior (let A and B be cell range
     * addresses): If the start address of A is less than the start address of
     * B (using the natural ordering defined by class Address); or if both
     * start addresses are equal, and the end address of A is less than the end
     * address of B, then A is considered less than B.
     *
     * @param {Range} range1
     *  The first cell range address to be compared to the second cell range
     *  address.
     *
     * @param {Range} range2
     *  The second cell range address to be compared to the first cell range
     *  address.
     *
     * @returns {Number}
     *  - A negative number, if range1 is less than range2.
     *  - A positive number, if range1 is greater than range2.
     *  - Zero, if both cell range addresses are equal.
     */
    Range.compare = function (range1, range2) {
        return Utils.comparePairs(range1.start, range1.end, range2.start, range2.end, Address.compare);
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a deep clone of this cell range address.
     *
     * @returns {Range}
     *  A deep clone of this range address.
     */
    Range.prototype.clone = function () {
        return new Range(this.start.clone(), this.end.clone());
    };

    /**
     * Returns whether the passed cell range address is equal to this cell
     * range address.
     *
     * @param {Range} range
     *  The other cell range address to be compared to this cell range address.
     *
     * @returns {Boolean}
     *  Whether both cell range addresses contain the same start and end cell
     *  addresses.
     */
    Range.prototype.equals = function (range) {
        return this.start.equals(range.start) && this.end.equals(range.end);
    };

    /**
     * Returns whether the passed cell range address differs from this cell
     * range address.
     *
     * @param {Range} range
     *  The other cell range address to be compared to this cell range address.
     *
     * @returns {Boolean}
     *  Whether the passed cell range address differs from this cell range
     *  address.
     */
    Range.prototype.differs = function (range) {
        return this.start.differs(range.start) || this.end.differs(range.end);
    };

    /**
     * Compares this cell range address with the passed cell range address. See
     * static method Range.compare() for details.
     *
     * @param {Range} range
     *  The other cell range address to be compared to this cell range address.
     *
     * @returns {Number}
     *  - A negative number, if this cell range address is less than the passed
     *      cell range address.
     *  - A positive number, if this cell range address is greater than the
     *      passed cell range address.
     *  - Zero, if both cell range addresses are equal.
     */
    Range.prototype.compareTo = function (range) {
        return Range.compare(this, range);
    };

    /**
     * Returns the number of columns covered by this cell range address.
     *
     * @returns {Number}
     *  The number of columns covered by this cell range address.
     */
    Range.prototype.cols = function () {
        return this.end[0] - this.start[0] + 1;
    };

    /**
     * Returns the number of rows covered by this cell range address.
     *
     * @returns {Number}
     *  The number of rows covered by this cell range address.
     */
    Range.prototype.rows = function () {
        return this.end[1] - this.start[1] + 1;
    };

    /**
     * Returns the number of columns or rows covered by this cell range
     * address.
     *
     * @param {Boolean} columns
     *  Whether to return the number of columns (true), or the number of rows
     *  (false) in this cell range address.
     *
     * @returns {Number}
     *  The number of columns or rows in this cell range address.
     */
    Range.prototype.size = function (columns) {
        return columns ? this.cols() : this.rows();
    };

    /**
     * Returns the number of cells covered by this cell range address.
     *
     * @returns {Number}
     *  The number of cells covered by this cell range address.
     */
    Range.prototype.cells = function () {
        return this.cols() * this.rows();
    };

    /**
     * Returns whether this cell range address covers a single column (column
     * indexes in start address and end address are equal).
     *
     * @returns {Boolean}
     *  Whether this cell range address covers a single column.
     */
    Range.prototype.singleCol = function () {
        return this.start[0] === this.end[0];
    };

    /**
     * Returns whether this cell range address covers a single row (row indexes
     * in start address and end address are equal).
     *
     * @returns {Boolean}
     *  Whether this cell range address covers a single row.
     */
    Range.prototype.singleRow = function () {
        return this.start[1] === this.end[1];
    };

    /**
     * Returns whether this cell range address covers a single cell (start
     * address and end address are equal).
     *
     * @returns {Boolean}
     *  Whether this cell range address covers a single cell.
     */
    Range.prototype.single = function () {
        return this.start.equals(this.end);
    };

    /**
     * Returns whether this cell range address starts at the specified cell
     * address.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address starts at the specified cell address.
     */
    Range.prototype.startsAt = function (address) {
        return this.start.equals(address);
    };

    /**
     * Returns whether this cell range address ends at the specified cell
     * address.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address ends at the specified cell address.
     */
    Range.prototype.endsAt = function (address) {
        return this.end.equals(address);
    };

    /**
     * Returns whether this cell range address contains the specified column
     * index.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address contains the specified column index.
     */
    Range.prototype.containsCol = function (col) {
        return (this.start[0] <= col) && (col <= this.end[0]);
    };

    /**
     * Returns whether this cell range address contains the specified row
     * index.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address contains the specified row index.
     */
    Range.prototype.containsRow = function (row) {
        return (this.start[1] <= row) && (row <= this.end[1]);
    };

    /**
     * Returns whether this cell range address contains the specified column or
     * row index.
     *
     * @param {Number} index
     *  The zero-based column or row index to be checked.
     *
     * @param {Boolean} columns
     *  Whether to check the column index (true), or the row index (false).
     *
     * @returns {Boolean}
     *  Whether this cell range address contains the specified column or row
     *  index.
     */
    Range.prototype.containsIndex = function (index, columns) {
        return columns ? this.containsCol(index) : this.containsRow(index);
    };

    /**
     * Returns whether this cell range address contains the specified cell
     * address.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address contains the specified cell address.
     */
    Range.prototype.containsAddress = function (address) {
        return this.containsCol(address[0]) && this.containsRow(address[1]);
    };

    /**
     * Returns whether this cell range address contains the passed cell range
     * address completely.
     *
     * @param {Range} range
     *  The other cell range address to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address contains the passed cell range address.
     */
    Range.prototype.contains = function (range) {
        return this.containsAddress(range.start) && this.containsAddress(range.end);
    };

    /**
     * Returns whether this cell range address overlaps with the passed cell
     * range address by at least one cell.
     *
     * @param {Range} range
     *  The other cell range address to be checked.
     *
     * @returns {Boolean}
     *  Whether this cell range address overlaps with the passed cell range
     *  address by at least one cell.
     */
    Range.prototype.overlaps = function (range) {
        return (this.start[0] <= range.end[0]) && (range.start[0] <= this.end[0]) && (this.start[1] <= range.end[1]) && (range.start[1] <= this.end[1]);
    };

    /**
     * Returns the column interval covered by this cell range address.
     *
     * @returns {Interval}
     *  The column interval covered by this cell range address.
     */
    Range.prototype.colInterval = function () {
        return new Interval(this.start[0], this.end[0]);
    };

    /**
     * Returns the row interval covered by this cell range address.
     *
     * @returns {Interval}
     *  The row interval covered by this cell range address.
     */
    Range.prototype.rowInterval = function () {
        return new Interval(this.start[1], this.end[1]);
    };

    /**
     * Returns the column or row interval covered by this cell range address.
     *
     * @param {Boolean} columns
     *  Whether to return the column interval (true), or the row interval
     *  (false) of this cell range address.
     *
     * @returns {Interval}
     *  The column or row interval covered by this cell range address.
     */
    Range.prototype.interval = function (columns) {
        return columns ? this.colInterval() : this.rowInterval();
    };

    /**
     * Returns the range address of the top row in this cell range address.
     *
     * @returns {Range}
     *  The range address of the top row in this cell range address.
     */
    Range.prototype.header = function () {
        return Range.create(this.start[0], this.start[1], this.end[0], this.start[1]);
    };

    /**
     * Returns the range address of the bottom row in this cell range address.
     *
     * @returns {Range}
     *  The range address of the bottom row in this cell range address.
     */
    Range.prototype.footer = function () {
        return Range.create(this.start[0], this.end[1], this.end[0], this.end[1]);
    };

    /**
     * Returns the address of the bounding cell range of this cell range, and
     * the passed cell range address (the smallest range that contains both
     * ranges).
     *
     * @param {Range} range
     *  The other cell range address to create the bounding range from.
     *
     * @returns {Range}
     *  The address of the bounding cell range of this cell range, and the
     *  passed cell range address
     */
    Range.prototype.boundary = function (range) {
        return Range.create(
            Math.min(this.start[0], range.start[0]),
            Math.min(this.start[1], range.start[1]),
            Math.max(this.end[0], range.end[0]),
            Math.max(this.end[1], range.end[1])
        );
    };

    /**
     * Returns the intersecting cell range address between this cell range
     * address, and the passed cell range address.
     *
     * @param {Range} range
     *  The other cell range address.
     *
     * @returns {Range|Null}
     *  The address of the cell range covered by both cell range addresses, if
     *  existing; otherwise null.
     */
    Range.prototype.intersect = function (range) {
        return this.overlaps(range) ? Range.create(
            Math.max(this.start[0], range.start[0]),
            Math.max(this.start[1], range.start[1]),
            Math.min(this.end[0], range.end[0]),
            Math.min(this.end[1], range.end[1])
        ) : null;
    };

    /**
     * Returns the column or row index of the start cell of this cell range
     * address.
     *
     * @param {Boolean} columns
     *  Whether to return the column index (true), or the row index (false).
     *
     * @returns {Number}
     *  The column or row index of the start cell of this cell range address.
     */
    Range.prototype.getStart = function (columns) {
        return this.start.get(columns);
    };

    /**
     * Returns the column or row index of the end cell of this cell range
     * address.
     *
     * @param {Boolean} columns
     *  Whether to return the column index (true), or the row index (false).
     *
     * @returns {Number}
     *  The column or row index of the end cell of this cell range address.
     */
    Range.prototype.getEnd = function (columns) {
        return this.end.get(columns);
    };

    /**
     * Changes the column or row index of the start cell of this cell range
     * address.
     *
     * @param {Number} index
     *  The new column or row index for the start cell.
     *
     * @param {Boolean} columns
     *  Whether to change the column index (true), or the row index (false).
     *
     * @returns {Range}
     *  A reference to this instance.
     */
    Range.prototype.setStart = function (index, columns) {
        this.start.set(index, columns);
        return this;
    };

    /**
     * Changes the column or row index of the end cell of this cell range
     * address.
     *
     * @param {Number} index
     *  The new column or row index for the end cell.
     *
     * @param {Boolean} columns
     *  Whether to change the column index (true), or the row index (false).
     *
     * @returns {Range}
     *  A reference to this instance.
     */
    Range.prototype.setEnd = function (index, columns) {
        this.end.set(index, columns);
        return this;
    };

    /**
     * Changes the column or row index of the start cell, AND the end cell of
     * this cell range address.
     *
     * @param {Number} index
     *  The new column or row index for the start cell, and the end cell.
     *
     * @param {Boolean} columns
     *  Whether to change the column index (true), or the row index (false).
     *
     * @returns {Range}
     *  A reference to this instance.
     */
    Range.prototype.setBoth = function (index, columns) {
        this.start.set(index, columns);
        this.end.set(index, columns);
        return this;
    };

    /**
     * Returns a unique string key for this cell range address that can be used
     * for example as key in an associative map. This method is faster than the
     * method Range.toString().
     *
     * @returns {String}
     *  A unique string key for this cell range address.
     */
    Range.prototype.key = function () {
        return this.start.key() + ':' + this.end.key();
    };

    /**
     * Returns the string representation of this cell range address, in
     * upper-case A1 notation.
     *
     * @returns {String}
     *  The string representation of this cell range address, in upper-case A1
     *  notation.
     */
    Range.prototype.toString = function () {
        return this.start + ':' + this.end;
    };

    /**
     * Returns the JSON representation of this cell range address.
     *
     * @returns {Object}
     *  The JSON representation of this cell range address (a plain JavaScript
     *  object with the properties 'start' and 'end' being JSON representations
     *  of the start and end cell address; see method Address.toJSON() for
     *  details).
     */
    Range.prototype.toJSON = function () {
        return { start: this.start.toJSON(), end: this.end.toJSON() };
    };

    // exports ================================================================

    return _.makeExtendable(Range);

});
