/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/sheetmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/sheetselection',
    'io.ox/office/spreadsheet/model/namecollection',
    'io.ox/office/spreadsheet/model/colrowcollection',
    'io.ox/office/spreadsheet/model/mergecollection',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/model/tablecollection',
    'io.ox/office/spreadsheet/model/validationcollection',
    'io.ox/office/spreadsheet/model/condformatcollection',
    'io.ox/office/spreadsheet/model/drawing/drawingcollection',
    'io.ox/office/spreadsheet/model/viewsettingsmixin',
    'io.ox/office/spreadsheet/model/sheetoperationsgenerator'
], function (Utils, TimerMixin, Color, Border, AttributedModel, DrawingUtils, Config, Operations, SheetUtils, PaneUtils, SheetSelection, NameCollection, ColRowCollection, MergeCollection, CellCollection, TableCollection, ValidationCollection, CondFormatCollection, SheetDrawingCollection, ViewSettingsMixin, SheetOperationsGenerator) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Interval = SheetUtils.Interval,
        Range = SheetUtils.Range,
        IntervalArray = SheetUtils.IntervalArray,
        RangeArray = SheetUtils.RangeArray,

        // definitions for sheet view attributes
        SHEET_VIEW_ATTRIBUTES = {

            /**
             * Sheet selection (cells and drawing objects), as instance of the
             * class SheetSelection. The array of cell range addresses MUST NOT
             * be empty.
             */
            selection: {
                def: SheetSelection.createFromAddress(Address.A1),
                validate: function (selection) {
                    return (selection instanceof SheetSelection) ? selection : Utils.BREAK;
                },
                equals: function (selection1, selection2) {
                    return selection1.equals(selection2, true); // deep comparison
                }
            },

            /**
             * Zoom factor in the sheet, as floating-point number. The value 1
             * represents a zoom factor of 100%.
             */
            zoom: {
                def: 1,
                validate: function (zoom) {
                    return _.isNumber(zoom) ? Utils.minMax(zoom, SheetUtils.MIN_ZOOM, SheetUtils.MAX_ZOOM) : Utils.BREAK;
                }
            },

            /**
             * Whether to display the grid lines between all cells.
             */
            showGrid: {
                def: true,
                validate: function (show) { return !!show; }
            },

            /**
             * The type of the current view split. Has no effect, if both view
             * properties 'splitWidth' and 'splitHeight' are set to the value
             * zero (view is not split at all). Supports the following values:
             * - 'split': Dynamic split mode (movable split lines).
             * - 'frozen': Frozen split mode (a fixed number of columns and/or
             *      rows are frozen, split lines are not movable). When leaving
             *      frozen split mode, the split lines will be hidden.
             * - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
             *      mode). When leaving frozen split mode, the split lines
             *      remain visible and become movable (back to 'split' mode).
             */
            splitMode: {
                def: 'split',
                validate: /^(split|frozen|frozenSplit)$/
            },

            /**
             * The inner width of the left panes in view split mode. In dynamic
             * split mode, this value is interpreted in 1/100 of millimeters.
             * In frozen split mode, this value specifies the number of columns
             * shown in the frozen panes. Must not be negative. If this value
             * is zero, the view is not split into left and right panes (but
             * may be split into upper and lower panes).
             */
            splitWidth: {
                def: 0,
                validate: function (splitPos) {
                    return _.isNumber(splitPos) ? Math.max(splitPos, 0) : 0;
                }
            },

            /**
             * The inner height of the upper panes in view split mode. In
             * dynamic split mode, this value is interpreted in 1/100 of
             * millimeters. In frozen split mode, this value specifies the
             * number of rows shown in the frozen panes. Must not be negative.
             * If this value is zero, the view is not split into top and bottom
             * panes (but may be split into left and right panes).
             */
            splitHeight: {
                def: 0,
                validate: function (splitPos) {
                    return _.isNumber(splitPos) ? Math.max(splitPos, 0) : 0;
                }
            },

            /**
             * The identifier of the active pane in split mode. Must be one of
             * 'topLeft', 'bottomLeft', 'topRight', or 'bottomRight'.
             */
            activePane: {
                def: 'bottomRight',
                validate: /^(top|bottom)(Left|Right)$/
            },

            /**
             * The identifier of an entire active pane side. Must be one of the
             * values null, 'left', 'right', 'top', or 'bottom'. Used for
             * example while selecting entire columns or rows.
             */
            activePaneSide: {
                def: null,
                validate: function (paneSide) {
                    return (/^(left|right|top|bottom)$/).test(paneSide) ? paneSide : null;
                }
            },

            /**
             * The horizontal scroll position in the left view panes, as
             * floating-point number. The integral part represents the
             * zero-based index of the first column visible in the panes. The
             * fractional part represents the ratio of the width of that
             * column where the exact scroll position (the first visible pixel)
             * is located.
             */
            anchorLeft: {
                def: 0,
                validate: function (anchor) { return _.isNumber(anchor) ? anchor : null; }
            },

            /**
             * The horizontal scroll position in the right view panes, as
             * floating-point number. The integral part represents the
             * zero-based index of the first column visible in the panes. The
             * fractional part represents the ratio of the width of that
             * column where the exact scroll position (the first visible pixel)
             * is located.
             */
            anchorRight: {
                def: 0,
                validate: function (anchor) { return _.isNumber(anchor) ? anchor : null; }
            },

            /**
             * The vertical scroll position in the upper view panes, as
             * floating-point number. The integral part represents the
             * zero-based index of the first row visible in the panes. The
             * fractional part represents the ratio of the height of that
             * row where the exact scroll position (the first visible pixel) is
             * located.
             */
            anchorTop: {
                def: 0,
                validate: function (anchor) { return _.isNumber(anchor) ? anchor : null; }
            },

            /**
             * The vertical scroll position in the lower view panes, as
             * floating-point number. The integral part represents the
             * zero-based index of the first row visible in the panes. The
             * fractional part represents the ratio of the height of that
             * row where the exact scroll position (the first visible pixel) is
             * located.
             */
            anchorBottom: {
                def: 0,
                validate: function (anchor) { return _.isNumber(anchor) ? anchor : null; }
            },

            /**
             * Specific cell ranges that are in a special activated state, used
             * for example for ranges that will be inserted into a formula
             * while in formula edit mode. These ranges will be rendered in a
             * special way over the regular cell selection (with a marching
             * ants effect). Allowed values are null (no active ranges
             * available), or a sheet selection (an instance of the class
             * SheetSelection). The array of selected drawings objects MUST be
             * empty.
             */
            activeSelection: {
                def: null,
                validate: function (selection) {
                    return (selection instanceof SheetSelection) ? selection : null;
                },
                equals: function (selection1, selection2) {
                    return (!selection1 && !selection2) || (selection1 && selection2 && selection1.equals(selection2, true)); // deep comparison
                }
            },

            /**
             * Additional data for the current cell range while auto-fill
             * tracking is active. Allowed values are null (auto-fill tracking
             * not active), or an object with the following properties:
             * - {String} border
             *      Which border of the selected range will be modified by the
             *      auto-fill. Allowed values are 'left', 'right', 'top', or
             *      'bottom'.
             * - {Number} count
             *      The number of columns/rows to extend the selected range
             *      with (positive values), or to shrink the selected range
             *      (negative values).
             * This attribute will be ignored, if the current sheet selection
             * consists of more than one cell range.
             */
            autoFillData: {
                def: null,
                validate: function (data) { return _.isObject(data) ? data : null; }
            }
        };

    // global private functions ===============================================

    /**
     * Returns a rectangle descriptor for the passed column and row entry.
     */
    function getRectangle(colPosition, rowPosition) {
        return {
            leftHmm: colPosition.offsetHmm,
            topHmm: rowPosition.offsetHmm,
            widthHmm: colPosition.sizeHmm,
            heightHmm: rowPosition.sizeHmm,
            left: colPosition.offset,
            top: rowPosition.offset,
            width: colPosition.size,
            height: rowPosition.size
        };
    }

    // class SheetModel =======================================================

    /**
     * Represents a single sheet in the spreadsheet document.
     *
     * Additionally to the events triggered by the base class AttributedModel,
     * an instance of this class triggers the following events:
     * - 'change:viewattributes'
     *      After at least one view attribute has been changed. Event handlers
     *      receive an incomplete attribute map containing all changed view
     *      attributes with their new values.
     * - 'insert:columns', 'change:columns', 'delete:columns'
     *      The respective '*:entries' event forwarded from the column
     *      collection of this sheet. See class ColRowCollection for details.
     * - 'insert:rows', 'change:rows', 'delete:rows'
     *      The respective '*:entries' event forwarded from the column
     *      collection of this sheet. See class ColRowCollection for details.
     * - 'insert:merged', 'delete:merged'
     *      The respective event forwarded from the merge collection of this
     *      sheet. See class MergeCollection for details.
     * - 'insert:table', 'change:table', 'delete:table'
     *      The respective event forwarded from the table collection of this
     *      sheet. See class TableCollection for details.
     * - 'change:cells', 'change:usedarea'
     *      The respective event forwarded from the cell collection of this
     *      sheet. See class CellCollection for details.
     * - 'change:drawing', 'insert:drawing', 'delete:drawing'
     *      The respective event forwarded from the drawing collection of this
     *      sheet. See class SheetDrawingCollection for details.
     *
     * @constructor
     *
     * @extends AttributedModel
     * @extends TimerMixin
     * @extends ViewSettingsMixin
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this sheet.
     *
     * @param {String|SheetModel} sheetType
     *  The type identifier of this sheet. Supported values are:
     *  - 'worksheet': A regular worksheet with cells and formulas.
     *  - 'chartsheet': A sheet that consists entirely of a chart object.
     *  - 'macrosheet': A sheet with cells and formulas that contains sheet
     *      macros. This sheet type is a legacy feature of MS Excel. These
     *      sheets will not be shown in the user interface.
     *  - 'dialogsheet': A sheet that consists entirely of a user dialog. This
     *      sheet type is a legacy feature of MS Excel. These sheets will not
     *      be shown in the user interface.
     *  Alternatively, a source sheet model, if this constructor will be used
     *  as copy constructor.
     *
     * @param {Object} [initialAttrs]
     *  An attribute set with initial formatting attributes for the sheet.
     */
    function SheetModel(docModel, sheetType, initialAttrs) {

        var // self reference
            self = this,

            // global collections of the document model
            cellStyles = docModel.getCellStyles(),

            // the collection of defined names in this sheet
            nameCollection = null,

            // the column and row collections (size and formatting of all columns/rows)
            colCollection = null,
            rowCollection = null,

            // the collection of merged cell ranges in this sheet
            mergeCollection = null,

            // the collection of table ranges (filter, sorting)
            tableCollection = null,

            // the collection of cell ranges with validation settings
            validationCollection = null,

            // the collection of cell ranges with conditional formatting
            condFormatCollection = null,

            // the cell collection
            cellCollection = null,

            // the collection of drawing objects contained in this sheet
            drawingCollection = null,

            // handlers to adjust range addresses after inserting/deleting columns/rows
            transformationHandlers = [];

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, docModel, initialAttrs, { trigger: 'always', families: 'sheet column row' });
        TimerMixin.call(this);
        ViewSettingsMixin.call(this, SHEET_VIEW_ATTRIBUTES);

        // private methods ----------------------------------------------------

        /**
         * Expands all ranges but entire column/row ranges to the merged ranges
         * in the sheet.
         */
        function getExpandedRanges(ranges) {
            return RangeArray.map(ranges, function (range) {
                return (docModel.isColRange(range) || docModel.isRowRange(range)) ? range : mergeCollection.expandRangeToMergedRanges(range);
            });
        }

        /**
         * Builds an attribute set containing all cell formatting attributes
         * set to the null value, but without the 'styleId' property.
         */
        function buildNullAttributes() {
            var attributes = cellStyles.buildNullAttributes();
            delete attributes.styleId;
            return attributes;
        }

        /**
         * Returns whether additional columns or rows can be inserted into the
         * sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Boolean} columns
         *  True for columns, or false for rows.
         *
         * @returns {Boolean}
         *  Whether additional columns or rows can be inserted, according to
         *  the passed intervals and the number of used columns/rows in the
         *  sheet.
         */
        function intervalsInsertable(intervals, columns) {

            var // the number of inserted columns/rows
                insertCount = IntervalArray.get(intervals).merge().size(),
                // the number of used columns/rows
                usedCount = columns ? cellCollection.getUsedCols() : cellCollection.getUsedRows(),
                // the available number of columns/rows in the sheet
                maxCount = docModel.getMaxIndex(columns) + 1;

            // the number of used columns/rows, and the number of inserted columns/rows
            // must not exceed the total number of columns/rows in the sheet
            return (insertCount > 0) && ((usedCount + insertCount) <= maxCount);
        }

        /**
         * Returns whether the specified columns or rows can be deleted from
         * the sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Boolean} columns
         *  True for columns, or false for rows.
         *
         * @returns {Boolean}
         *  Whether the columns can be deleted.
         */
        function intervalsDeleteable(intervals, columns) {

            var // the number of deleted columns/rows
                deleteCount = IntervalArray.get(intervals).merge().size(),
                // the available number of columns/rows in the sheet
                maxCount = docModel.getMaxIndex(columns) + 1;

            // prevent deleting all columns/rows (entire sheet) at once
            return (deleteCount > 0) && (deleteCount < maxCount);
        }

        /**
         * Checks the passed ranges that are about to be changed by a specific
         * formatting operation, and returns a rejected promise with a specific
         * error code, if a limit has been exceeded.
         *
         * @param {RangeArray} ranges
         *  The cell range addresses to be checked.
         *
         * @returns {jQuery.Promise|Null}
         *  The value null, if all passed counts are inside the respective
         *  limits, otherwise a promise that has been rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'cols:overflow': Too many entire columns in the ranges.
         *  - 'rows:overflow': Too many entire rows in the ranges.
         *  - 'cells:overflow': Too many cells in the ranges.
         */
        function checkCellLimits(ranges) {

            var // number of changed entire columns
                cols = 0,
                // number of changed entire rows
                rows = 0,
                // number of changed cells
                cells = 0;

            // count columns, rows, and cells in all ranges
            ranges.forEach(function (range) {
                if (docModel.isColRange(range)) {
                    cols += range.cols();
                } else if (docModel.isRowRange(range)) {
                    rows += range.rows();
                } else {
                    cells += range.cells();
                }
            });

            // check all limits
            return (cells > SheetUtils.MAX_FILL_CELL_COUNT) ? SheetUtils.makeRejected('cells:overflow') :
                (rows > SheetUtils.MAX_CHANGE_ROWS_COUNT) ? SheetUtils.makeRejected('rows:overflow') :
                (cols > SheetUtils.MAX_CHANGE_COLS_COUNT) ? SheetUtils.makeRejected('cols:overflow') :
                null;
        }

        /**
         * Returns the column/row interval shown in the leading panes of the
         * split view.
         *
         * @param {Boolean} columns
         *  Whether to return the column interval of the left sheet area
         *  (true), or the row interval of the top sheet area (false).
         *
         * @returns {Interval|Null}
         *  The column/row interval shown in the leading panes. If the view is
         *  not split in the specified direction, this method returns null.
         */
        function getSplitInterval(columns) {

            var // the value of the split size view attribute (number of columns/rows in frozen split)
                splitSize = self.getViewAttribute(columns ? 'splitWidth' : 'splitHeight');

            // no split: return null
            if (splitSize === 0) { return null; }

            var // start scroll anchor in the leading sheet area
                startAnchor = self.getViewAttribute(columns ? 'anchorLeft' : 'anchorTop'),
                // the resulting column/row interval
                interval = new Interval(Math.floor(startAnchor));

            // build column/row interval in frozen split mode directly
            if (self.hasFrozenSplit()) {
                interval.last = interval.first + splitSize - 1;
                return interval;
            }

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // end scroll anchor in the leading sheet area
                endAnchor = collection.getScrollAnchorByOffset(collection.convertScrollAnchorToHmm(startAnchor) + splitSize);

            // build the resulting interval
            interval.last = Math.floor(endAnchor);

            // exclude last column/row, if visible less than half of its size
            if (!interval.single() && (endAnchor % 1 < 0.5)) {
                interval.last -= 1;
            }

            // exclude first column/row, if visible less than half of its size
            if (!interval.single() && (startAnchor % 1 > 0.5)) {
                interval.first += 1;
            }

            return interval;
        }

        /**
         * Returns the width or height of the leading area in a sheet in split
         * mode, in 1/100 of millimeters, or in pixels according to the current
         * sheet zoom factor. In frozen split mode, returns the total size of
         * the frozen columns/rows.
         *
         * @param {Boolean} columns
         *  Whether to return the width of the left sheet area (true), or the
         *  height of the top sheet area (false).
         *
         * @param {Boolean} pixel
         *  Whether to return pixels (true) or 1/100 of millimeters (false).
         *
         * @returns {Number}
         *  The width or height of the leading area if the sheet is in split
         *  mode, otherwise zero.
         */
        function getSplitSize(columns, pixel) {

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // the column interval in frozen split mode
                interval = null,
                // the size in dynamic split mode
                splitSize = 0;

            // return total size of frozen columns/rows
            if (self.hasFrozenSplit()) {
                interval = getSplitInterval(columns);
                return interval ? collection.getIntervalPosition(interval)[pixel ? 'size' : 'sizeHmm'] : 0;
            }

            // return size of dynamic split
            splitSize = self.getViewAttribute(columns ? 'splitWidth' : 'splitHeight');
            return pixel ? self.convertHmmToPixel(splitSize) : splitSize;
        }

        /**
         * Updates the frozen split attributes after columns or rows have been
         * inserted into or deleted from this sheet.
         */
        function transformFrozenSplit(interval, insert, columns) {

            var // attribute names
                LEADING_ANCHOR_NAME = columns ? 'anchorLeft' : 'anchorTop',
                TRAILING_ANCHOR_NAME = columns ? 'anchorRight' : 'anchorBottom',
                SPLIT_SIZE_NAME = columns ? 'splitWidth' : 'splitHeight',
                // the frozen column/row interval (will be null if not frozen)
                frozenInterval = self.hasFrozenSplit() ? getSplitInterval(columns) : null,
                // the number of inserted columns/rows
                insertSize = 0,
                // intersection of frozen interval and deleted interval
                intersectInterval = null,
                // the new view attributes
                attributes = {};

            // no frozen columns/rows available
            if (!frozenInterval) { return; }

            if (insert) {

                // the number of inserted columns/rows
                insertSize = interval.size();

                // inserted before frozen interval: update scroll anchor
                if (interval.first < frozenInterval.first) {
                    attributes[LEADING_ANCHOR_NAME] = frozenInterval.first + insertSize;
                    attributes[TRAILING_ANCHOR_NAME] = frozenInterval.last + 1 + insertSize;

                // inserted inside frozen interval: update size of frozen interval
                } else if (interval.first <= frozenInterval.last) {
                    attributes[SPLIT_SIZE_NAME] = frozenInterval.size() + insertSize;
                    attributes[TRAILING_ANCHOR_NAME] = frozenInterval.last + 1 + insertSize;
                }

            } else {

                // deleted inside frozen interval: update size of frozen interval
                if ((intersectInterval = interval.intersect(frozenInterval))) {
                    attributes[SPLIT_SIZE_NAME] = frozenInterval.size() - intersectInterval.size();
                }

                // deleted before frozen interval: update leading scroll anchor
                if (interval.first < frozenInterval.first) {
                    attributes[LEADING_ANCHOR_NAME] = interval.first;
                }

                // deleted before or inside frozen interval: update trailing scroll anchor
                if (interval.first <= frozenInterval.last) {
                    attributes[TRAILING_ANCHOR_NAME] = interval.first;
                }
            }

            // commit the new view attributes
            self.setViewAttributes(attributes);
        }

        /**
         * Invokes all registered range transformation handlers, after columns
         * or rows have been inserted into or deleted from this sheet, and
         * triggers an appropriate event.
         */
        function invokeTransformationHandlers(eventType, interval, insert, columns) {

            // invoke all registered transformation handlers
            _.each(transformationHandlers, function (handler) {
                handler.call(self, interval, insert, columns);
            });

            // notify all listeners
            self.trigger(eventType, interval);
        }

        /**
         * Updates the selection after a drawing object has been inserted into
         * this sheet.
         */
        function insertDrawingHandler(event, drawingModel, insertPosition) {

            var // deep copy of the current selection
                selection = self.getViewAttribute('selection'),
                // the length of the position array
                length = insertPosition.length;

            if (selection.drawings.length === 0) { return; }

            // update position of all drawings following the inserted drawing
            selection.drawings.forEach(function (position) {

                // nothing to do, if the selected drawing is a parent object
                if (position.length < length) { return; }
                // nothing to do, if the selected drawing is located in another parent object
                if (Utils.compareNumberArrays(insertPosition, position, length - 1) !== 0) { return; }

                // inserted drawing is a predecessor in the same parent
                if (insertPosition[length - 1] <= position[length - 1]) {
                    position[length - 1] += 1;
                }
            });

            self.setViewAttribute('selection', selection);
        }

        /**
         * Updates the selection after a drawing object has been deleted from
         * this sheet.
         */
        function deleteDrawingHandler(event, drawingModel, deletePosition) {

            var // deep copy of the current selection
                selection = self.getViewAttribute('selection'),
                // the length of the position array
                length = deletePosition.length;

            if (selection.drawings.length === 0) { return; }

            // update position of all drawings following the deleted drawing
            Utils.iterateArray(selection.drawings, function (position, index) {

                // nothing to do, if the selected drawing is a parent object
                if (position.length < length) { return; }
                // nothing to do, if the selected drawing is located in another parent object
                if (Utils.compareNumberArrays(deletePosition, position, length - 1) !== 0) { return; }

                // deleted drawing or any of its children has been deleted
                if (deletePosition[length - 1] === position[length - 1]) {
                    selection.drawings.splice(index, 1);
                // deleted drawing is a predecessor in the same parent
                } else if (deletePosition[length - 1] < position[length - 1]) {
                    position[length - 1] -= 1;
                }
            }, { reverse: true });

            self.setViewAttribute('selection', selection);
        }

        // operation generators -----------------------------------------------

        /**
         * Invokes the passed callback function with activated custom 'action
         * processing mode' in the document model which causes the sheet cell
         * renderers to block until all operations have been generated and
         * applied. Intended to be used to generate document operations without
         * applying them but simply sending them silently to the server. This
         * method creates a new operations generator, invokes the callback
         * function, and then sends all operations contained in the generator
         * to the server, but does not apply the operations. The passed
         * callback function must implement all necessary updates in the
         * document model by itself.
         *
         * Additionally, this method registers an attribute change listener and
         * a delete listener at the drawing collection of this sheet. All
         * changes made directly or indirectly at the drawing collection
         * (during insertion, deletion, or modification of columns or rows)
         * will be inserted as drawing operations into the operations generator
         * afterwards. After the callback function has returned, the event
         * handlers will be removed from the drawing collection.
         *
         * @param {Function} callback
         *  The callback function that will be invoked with activated action
         *  processing mode. Receives the following parameters:
         *  (1) {SheetOperatíonsGenerator} generator
         *      An empty sheet operations generator for this sheet.
         *  (2) {Number} sheet
         *      The current sheet index of this sheet in the document.
         *  May return a promise to defer the action processing mode until it
         *  will be resolved or rejected.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected according to the result
         *  of the callback function.
         */
        function createAndSendSheetOperations(callback) {

            // simulate actions processing mode (blocks the cell renderer)
            return self.createAndApplyOperations(function (generator) {

                var // the own sheet index
                    sheet = self.getIndex(),
                    // a new operations generator for the callback function (bug 39714)
                    generator2 = new SheetOperationsGenerator(self),
                    // positions of deleted drawing objects
                    deletedDrawings = [],
                    // changed drawing anchor attributes, mapped by position
                    changedDrawingAttrs = {},
                    // the promise returned by this method
                    promise = null;

                // collects deleted drawing objects (only on sheet level)
                function deleteDrawingHandler(event, drawingModel, position) {
                    if (position.length === 1) {
                        deletedDrawings.push(position);
                    }
                }

                // collects changed anchor attributes of moved drawing objects (only on sheet level)
                function changeDrawingHandler(event, drawingModel, position, type, newAttributes, oldAttributes) {
                    if ((type === 'attributes') && (position.length === 1)) {
                        var key = JSON.stringify(position);
                        if (!(key in changedDrawingAttrs)) { changedDrawingAttrs[key] = { oldValues: oldAttributes.drawing }; }
                        changedDrawingAttrs[key].newValues = _.copy(newAttributes.drawing, true);
                    }
                }

                // invoke the callback function with registered drawing event handlers
                drawingCollection.on('delete:drawing', deleteDrawingHandler);
                drawingCollection.on('change:drawing', changeDrawingHandler);

                // invoke passed callback function with the new operations generator
                promise = $.when(callback.call(self, generator2, sheet));

                // always unregister the temporary drawing event handlers
                promise.always(function () {
                    drawingCollection.off('delete:drawing', deleteDrawingHandler);
                    drawingCollection.off('change:drawing', changeDrawingHandler);
                });

                // process the changed drawings, unless the callback has returned a rejected promise
                promise.done(function () {

                    // remove entries for deleted drawings from 'changedAttributes' map
                    _.each(deletedDrawings, function (position) {
                        delete changedDrawingAttrs[JSON.stringify(position)];
                    });

                    // create 'setDrawingAttributes' operations for all changed drawings
                    _.each(changedDrawingAttrs, function (attributes, key) {
                        var position = JSON.parse(key);
                        // delete unchanged attributes
                        _.each(attributes.newValues, function (value, name) {
                            if (_.isEqual(attributes.oldValues[name], value)) { delete attributes.newValues[name]; }
                        });
                        // generate operations if some values really have been changed
                        if (!_.isEmpty(attributes.newValues)) {
                            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, {
                                scope: 'filter',
                                attrs: { drawing: attributes.newValues }
                            });
                        }
                    });

                    // create 'deleteDrawing' operations for all deleted drawings (reverse order!)
                    DrawingUtils.sortDrawingPositions(deletedDrawings);
                    Utils.iterateArray(deletedDrawings, function (position) {
                        generator.generateDrawingOperation(Operations.DELETE_DRAWING, position, { scope: 'filter' });
                    }, { reverse: true });

                    // bug 39714: append the operations generated by the callback (after all drawing operations!)
                    generator.appendOperations(generator2);
                });

                return promise;

            }, { filter: false }); // do not apply operations, only send to server
        }

        /**
         * Generates all operations to insert columns/rows into, or delete
         * columns/rows from the sheet. All operations needed to refresh
         * drawing objects will be generated too.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {String} operationName
         *  The name of the document operations to be generated.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully.
         */
        function generateIntervalOperations(generator, operationName, intervals) {

            // merge and sort the passed intervals
            intervals = intervals.merge();

            // create the operations for all intervals in reverse order (!), and silently
            // apply the operations (causes generation of the indirect drawing operations)
            return self.iterateArraySliced(intervals, function (interval) {
                generator.generateIntervalOperation(operationName, interval);
                docModel.invokeOperationHandler(generator.getLastOperation());
            }, { reverse: true });
        }

        /**
         * Returns a reduced copy of the passed column/row intervals, that will
         * contain only the formatting attributes that would change the current
         * column/row formatting.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval. The array
         *  MUST be sorted, the intervals MUST NOT overlap each other (e.g. an
         *  array as returned by the method IntervalArray.merge()). Each
         *  interval object MUST contain an additional object property 'attrs'
         *  containing the (incomplete) column/row attributes, as simple object
         *  (NOT mapped as a 'column'/'row' sub-object).
         *
         * @param {Boolean} columns
         *  Whether the passed intervals are column intervals (true), or row
         *  intervals (false).
         *
         * @returns {IntervalArray}
         *  The resulting intervals (sub set of the passed intervals) that will
         *  contain only the formatting attributes that would change the
         *  current column/row formatting.
         */
        function reduceAttributedIntervals(intervals, columns) {

            var // attribute family (column or row)
                FAMILY_NAME = columns ? 'column' : 'row',
                // the column or row collection
                collection = columns ? colCollection : rowCollection,
                // the remaining intervals with different attributes compared to current formatting
                resultIntervals = new IntervalArray();

            // remove attributes and intervals that are equal to the current formatting
            // (regardless whether set explicitly or derived from sheet default attributes)
            collection.iterateEntries(intervals, function (entry, uniqueInterval, origInterval) {

                var // the current merged column/row attributes of the visited entry
                    mergedAttrs = entry.attributes[FAMILY_NAME],
                    // the current explicit column/row attributes of the visited entry
                    explicitAttrs = entry.explicit[FAMILY_NAME],
                    // the new column/row attributes to be set at the current interval
                    newAttrs = _.copy(origInterval.attrs, true);

                // remove all attributes that are about to be changed but already have that
                // value, or that are about to be deleted but do not exist in the interval
                _.each(newAttrs, function (value, name) {
                    if (_.isNull(value) ? !(name in explicitAttrs) : _.isEqual(value, mergedAttrs[name])) {
                        delete newAttrs[name];
                    }
                });

                // ignore intervals that do not change anything (no remaining new attributes)
                if (!_.isEmpty(newAttrs)) {
                    uniqueInterval.attrs = newAttrs;
                    resultIntervals.push(uniqueInterval);
                }
            }, { hidden: 'all', unique: true });

            // merge adjacent intervals in the reduced array of intervals
            Utils.iterateArray(resultIntervals, function (interval, index) {

                var // the preceding interval
                    prevInterval = resultIntervals[index - 1];

                // merge with preceding interval, if (remaining) attributes are equal
                if (prevInterval && (prevInterval.last + 1 === interval.first) && _.isEqual(prevInterval.attrs, interval.attrs)) {
                    prevInterval.last = interval.last;
                    resultIntervals.splice(index, 1);
                }
            }, { reverse: true });

            return resultIntervals;
        }

        /**
         * Merges the passed arrays of attributed index intervals.
         *
         * @param {IntervalArray} sourceIntervals
         *  An array of column/row intervals, containing the old formatting
         *  attributes of the columns/rows in the additional property 'attrs'
         *  in each interval.
         *
         * @param {IntervalArray} targetIntervals
         *  An array of column/row intervals, containing the new formatting
         *  attributes of the columns/rows in the additional property 'attrs'
         *  in each interval.
         *
         * @returns {IntervalArray}
         *  An array of column/row intervals. The formatting attributes in the
         *  target intervals have been merged over the the attributes in the
         *  source intervals.
         */
        function mergeAttributedIntervals(sourceIntervals, targetIntervals) {

            var // the resulting intervals returned by this method
                resultIntervals = new IntervalArray();

            // returns a shallow clone of the passed attributes interval
            function cloneInterval(interval) {
                var newInterval = interval.clone();
                newInterval.attrs = interval.attrs;
                return newInterval;
            }

            // arrays will be modified in-place for simplicity, work on clones of cloned objects
            sourceIntervals = IntervalArray.map(sourceIntervals, cloneInterval);
            targetIntervals = IntervalArray.map(targetIntervals, cloneInterval);

            // merge the source intervals array over the target intervals array
            sourceIntervals.some(function (sourceInterval, sourceIndex) {

                var // index of last target interval completely preceding the current source interval
                    lastIndex = Utils.findLastIndex(targetIntervals, function (targetInterval) {
                        return targetInterval.last < sourceInterval.first;
                    }, { sorted: true });

                // move all preceding target intervals into the result
                if (lastIndex >= 0) {
                    resultIntervals.push(targetIntervals.splice(0, lastIndex + 1));
                }

                // merge the current interval over all covered target intervals
                while (true) {

                    // no more intervals available in target array: append all remaining source intervals and exit the loop
                    if (targetIntervals.length === 0) {
                        // push current interval separately (may be modified, see below)
                        resultIntervals.append(sourceInterval, sourceIntervals.slice(sourceIndex + 1));
                        return true; // exit loop
                    }

                    var // first remaining interval in the target array
                        targetInterval = targetIntervals[0],
                        // temporary interval, inserted into the result array
                        tempInterval = null;

                    // entire source interval fits into the gap before the current target interval
                    if (sourceInterval.last < targetInterval.first) {
                        resultIntervals.push(sourceInterval);
                        // continue the loop with next source interval
                        return;
                    }

                    // one interval starts before the other interval: split the leading interval
                    if (sourceInterval.first < targetInterval.first) {
                        tempInterval = cloneInterval(sourceInterval);
                        tempInterval.last = targetInterval.first - 1;
                        resultIntervals.push(tempInterval);
                        sourceInterval.first = targetInterval.first;
                    } else if (targetInterval.first < sourceInterval.first) {
                        tempInterval = cloneInterval(targetInterval);
                        tempInterval.last = sourceInterval.first - 1;
                        resultIntervals.push(tempInterval);
                        targetInterval.first = sourceInterval.first;
                    }

                    // both intervals start at the same position now: merge their shared part
                    tempInterval = new Interval(sourceInterval.first, Math.min(sourceInterval.last, targetInterval.last));
                    tempInterval.attrs = _.extend({}, sourceInterval.attrs, targetInterval.attrs);
                    resultIntervals.push(tempInterval);

                    // source and target intervals have equal size: continue with next source and target interval
                    if (sourceInterval.last === targetInterval.last) {
                        targetIntervals.shift();
                        return;
                    }

                    // source interval ends before target interval: shorten the target interval
                    if (sourceInterval.last < targetInterval.last) {
                        targetInterval.first = sourceInterval.last + 1;
                        return;
                    }

                    // otherwise: shorten source interval, continue with next target interval
                    sourceInterval.first = targetInterval.last + 1;
                    targetIntervals.shift();
                }
            });

            // append remaining target intervals
            return resultIntervals.append(targetIntervals);
        }

        /**
         * Generates all operations to modify the attributes of columns or rows
         * in the sheet. The columns/rows will be formatted, and all operations
         * needed to refresh drawing objects will be generated too.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval. The array
         *  MUST be sorted, the intervals MUST NOT overlap each other. Each
         *  interval object MUST contain an additional object property 'attrs'
         *  containing the (incomplete) column/row attributes, as simple object
         *  (NOT mapped as a 'column'/'row' sub-object).
         *
         * @param {Boolean} columns
         *  Either true to modify the column attributes, or false to modify the
         *  row attributes.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'cols:overflow': Too many columns in the passed intervals (only
         *      returned, if parameter 'columns' is true).
         *  - 'rows:overflow': Too many rows in the passed intervals (only
         *      returned, if parameter 'columns' is false).
         */
        function generateChangeIntervalOperations(generator, intervals, columns) {

            var // reduce intervals to attributes different to current formatting
                changeIntervals = reduceAttributedIntervals(intervals, columns);

            // nothing to do, if passed intervals would not change anything
            if (changeIntervals.empty()) { return $.when(); }

            var // attribute family (column or row)
                FAMILY_NAME = columns ? 'column' : 'row',
                // maximum number of columns/rows to be changed at the same time
                MAX_CHANGE_COUNT = columns ? SheetUtils.MAX_CHANGE_COLS_COUNT : SheetUtils.MAX_CHANGE_ROWS_COUNT,
                // the name of the attribute operation
                OPERATION_NAME = columns ? Operations.CHANGE_COLUMNS : Operations.CHANGE_ROWS,
                // the column or row collection
                collection = columns ? colCollection : rowCollection,
                // the null attributes that will be applied for all columns/rows when changing sheet defaults
                nullAttributes = null,
                // the intervals containing the attributes to be restored after changing the sheet defaults
                restoreIntervals = null;

            // try to find sheet default formatting for very large intervals (but not when deleting attributes)
            if (!changeIntervals.some(function (interval) { return _.some(interval.attrs, _.isNull); })) {

                var // group all remaining intervals into a map of arrays, keyed by equal attributes (to find largest intervals)
                    intervalMap = IntervalArray.group(changeIntervals, function (interval) { return JSON.stringify(interval.attrs); }),
                    // find the largest array of intervals in the attribute map (sum of all interval sizes per map element)
                    maxIntervals = _.reduce(intervalMap, function (memoIntervals, mapIntervals) {
                        mapIntervals.cachedSize = mapIntervals.size();
                        return (memoIntervals && (memoIntervals.cachedSize >= mapIntervals.cachedSize)) ? memoIntervals : mapIntervals;
                    }, null);

                // set very large intervals as sheet default attributes
                if (maxIntervals && (maxIntervals.cachedSize > collection.getMaxIndex() / 2)) {

                    var // the column/row attributes to be set as sheet default attributes
                        sheetAttrs = maxIntervals.first().attrs,
                        // the column/row intervals that are not affected by the changed sheet default attributes
                        remainingIntervals = new IntervalArray(collection.getFullInterval()).difference(maxIntervals);

                    // build the attribute set with null attributes intended to be set at all columns/rows
                    nullAttributes = Utils.mapProperties(sheetAttrs, _.constant(null));

                    // create interval array to restore current attributes for the remaining intervals
                    // that will not be affected by the changed sheet default attributes
                    restoreIntervals = new IntervalArray();
                    collection.iterateEntries(remainingIntervals, function (entry, uniqueInterval) {

                        var // the current merged column/row attributes of the visited entry
                            mergedAttrs = entry.attributes[FAMILY_NAME];

                        // insert current values of the attributes that will be changed at the sheet
                        uniqueInterval.attrs = {};
                        _.each(sheetAttrs, function (value, name) {
                            uniqueInterval.attrs[name] = mergedAttrs[name];
                        });
                        restoreIntervals.push(uniqueInterval);

                    }, { hidden: 'all', unique: true });

                    // generate a 'setSheetAttributes' operation and immediately update the sheet attributes
                    // (the column/row collection, and the drawing objects will update themselves)
                    sheetAttrs = Utils.makeSimpleObject(FAMILY_NAME, sheetAttrs);
                    generator.generateSheetOperation(Operations.CHANGE_SHEET, { attrs: sheetAttrs });
                    self.setAttributes(sheetAttrs);

                    // reduce the passed intervals again according to the new sheet defaults
                    changeIntervals = reduceAttributedIntervals(intervals, columns);
                }
            }

            // bug 34641: restrict maximum number of modified columns/rows
            if (changeIntervals.size() > MAX_CHANGE_COUNT) {
                return SheetUtils.makeRejected(columns ? 'cols:overflow' : 'rows:overflow');
            }

            // merge the intervals that will restore the original formatting after changing
            // the sheet default attributes with the new attributes passed to this method
            if (restoreIntervals) {
                changeIntervals = mergeAttributedIntervals(changeIntervals, restoreIntervals);
                changeIntervals = reduceAttributedIntervals(changeIntervals, columns);
            }

            // add an interval over all columns/rows with null attributes, if sheet defaults have been changed
            if (nullAttributes) {
                changeIntervals.unshift(collection.getFullInterval());
                changeIntervals.first().attrs = nullAttributes;
            }

            // generate the column/row operations, and simultaneously update the column/row collection
            return self.iterateArraySliced(changeIntervals, function (interval) {
                var attributes = Utils.makeSimpleObject(FAMILY_NAME, interval.attrs);
                generator.generateIntervalOperation(OPERATION_NAME, interval, { attrs: attributes });
                collection.setAttributes(interval, attributes);
            });
        }

        /**
         * Generates a 'setCellContents' operation that will insert the
         * specified cell contents (values and/or formatting) into the sheet.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operation.
         *
         * @param {Address} start
         *  The address of the first cell to be modified.
         *
         * @param {Array<Array<Object|Null>>} contents
         *  A two-dimensional array containing the values and attribute sets
         *  for the cells. Null elements specify that the respective cells will
         *  be skipped (kept unmodified).
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the public
         *  method SheetModel.setCellContents().
         */
        function generateSetCellContentsOperation(generator, start, contents, options) {

            var // additional properties for the operation
                properties = { contents: contents };

            // add the parse property (bug 32624: need to insert 'en_US' for unparsed contents)
            properties.parse = Utils.getBooleanOption(options, 'parse', false) ? Config.LOCALE : 'en_US';

            // create and apply the operation
            generator.generateCellOperation(Operations.SET_CELL_CONTENTS, start, properties);
        }

        /**
         * Generates an attribute operation for the specified cell range,
         * without changing the content values of the cells. If the range
         * covers entire columns or entire rows, generates the appropriate
         * operation automatically.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {Range} range
         *  The address of the cell range to generate the operation for.
         *
         * @param {Object} attributes
         *  The formatting attribute set to be set at the cell range.
         *
         * @param {Object} [properties]
         *  Additional properties to be inserted into the generated operation.
         */
        function generateFillRangeOperation(generator, range, attributes, properties) {

            // add the attributes to the operation
            properties = _.extend({ attrs: attributes }, properties);

            // decide whether to generate a column, row, or cell range operation
            if (docModel.isColRange(range)) {
                generator.generateIntervalOperation(Operations.CHANGE_COLUMNS, range.colInterval(), properties);
            } else if (docModel.isRowRange(range)) {
                properties.attrs.row = { customFormat: true };
                generator.generateIntervalOperation(Operations.CHANGE_ROWS, range.rowInterval(), properties);
            } else {
                generator.generateRangeOperation(Operations.FILL_CELL_RANGE, range, properties);
            }
        }

        /**
         * Generates a hyperlink operation for the specified cell range.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operation.
         *
         * @param {Range} range
         *  The address of the cell range to generate the operation for.
         *
         * @param {String} url
         *  The URL of the hyperlink to be attached to the cell range. If set
         *  to the empty string, an existing hyperlink will be removed instead.
         *
         * @param {Object} [properties]
         *  Additional properties to be inserted into the generated operation.
         */
        function generateHyperlinkOperation(generator, range, url) {
            if (url.length > 0) {
                generator.generateRangeOperation(Operations.INSERT_HYPERLINK, range, { url: url });
            } else {
                generator.generateRangeOperation(Operations.DELETE_HYPERLINK, range);
            }
        }

        /**
         * Invokes the passed callback function with activated custom 'action
         * processing mode' in the document model. This method receives the
         * name of a table range, and passes the table model to the callback
         * function, otherwise behaves similarly to the method
         * createAndSendTableOperations().
         *
         * @param {String} tableName
         *  The name of an existing table range. The empty string addresses the
         *  anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet. If the table range does not
         *  exist, this method will exit with the error code 'table:missing'.
         *
         * @param {Function} callback
         *  The callback function that will be invoked with activated action
         *  processing mode. Receives the following parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      An empty sheet operations generator.
         *  (2) {TableModel} tableModel
         *      A reference to the table model specified with the parameter
         *      'tableName' passed to this method.
         *  May return a promise to defer the action processing mode until it
         *  has been resolved or rejected.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected according to the result
         *  of the callback function. If the table name passed to this method
         *  is not valid, the promise will be rejected with the error object
         *  {cause:'table:missing'}.
         */
        function createAndSendTableOperations(tableName, callback) {

            var // the table model
                tableModel = tableCollection.getTable(tableName);

            // check existence of the table range
            if (!tableModel) { return SheetUtils.makeRejected('table:missing'); }

            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return callback.call(this, generator, tableModel);
            });
        }

        /**
         * Generates all needed operations to refresh the passed table model.
         * Evaluates the current filter settings, and updates the visibility of
         * all rows covered by the table range. Also generates all implicit
         * operations for drawing objects.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {TableModel} tableModel
         *  The model of the table range to be refreshed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the table has been refreshed
         *  successfully, or that will be rejected with an object with 'cause'
         *  property set to one of the following error codes:
         *  - 'rows:overflow': The current filter settings would modify (show
         *      or hide) too many rows at once.
         */
        function generateRefreshTableOperations(generator, tableModel) {

            var // start with the entire data range
                dataRange = tableModel.getDataRange();

            // do nothing, if no data rows are available
            if (!dataRange) { return $.when(); }

            var // the entire table range
                tableRange = tableModel.getRange(),
                // the effective table range, e.g. after expanding auto-filter data range
                effectiveRange = tableRange.boundary(dataRange);

            // update the effective table range (data range may have been expanded)
            if (tableRange.differs(effectiveRange)) {
                generator.generateTableOperation(Operations.CHANGE_TABLE, tableModel.getName(), effectiveRange.toJSON());
                docModel.invokeOperationHandler(generator.getLastOperation());
            }

            // update the visibility state of all filter rows
            return tableModel.queryFilterResult().then(function (rowIntervals) {

                var // the entire row interval of the data range
                    dataRowInterval = dataRange.rowInterval(),
                    // the new visible row intervals
                    visibleIntervals = rowIntervals.clone(true).move(dataRowInterval.first),
                    // the new hidden row intervals
                    hiddenIntervals = new IntervalArray(dataRowInterval).difference(visibleIntervals),
                    // all intervals in a single array
                    intervals = null;

                // add visibility property to the modified interval arrays, before merging and sorting the arrays
                visibleIntervals.forEach(function (interval) { interval.attrs = { visible: true, filtered: false }; });
                hiddenIntervals.forEach(function (interval) { interval.attrs = { visible: false, filtered: true }; });

                // create a single sorted array of row intervals
                intervals = visibleIntervals.concat(hiddenIntervals).sort();

                // create operations and update all affected rows
                return generateChangeIntervalOperations(generator, intervals, false);
            });
        }

        /**
         * Generates 'setCellContents' operations with calculated column header
         * labels for all blank header cells in the passed table range.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {TableModel} tableModel
         *  The model of the table range to be processed.
         */
        function generateMissingTableHeaderLabels(generator, tableModel) {

            // nothing to do for auto-filters, or tables without header row
            if (tableModel.isAutoFilter() || !tableModel.hasHeaderRow()) { return; }

            // the address of the header range of the table
            var headerRange = tableModel.getRange().header();

            // collect all existing unique header labels of the table
            var labelArray = [], labelSet = {};
            cellCollection.iterateCellsInRanges(headerRange, function (cellDesc) {
                var label = cellDesc.result, key = _.isString(label) ? label.toUpperCase() : null;
                if (key && !(key in labelSet)) {
                    labelArray.push(label);
                    labelSet[key] = true;
                } else {
                    labelArray.push(null);
                }
            }, { hidden: 'all', ordered: true });

            // Generate missing header labels (all elements in 'labelArray' that are null).
            // Collect all new labels as cell data objects in 'contentsRow' intended to be
            // used in one or more 'setCellContents' operations.
            var contentsRow = [];
            var prevLabel = SheetUtils.getTableColName();
            labelArray.forEach(function (label) {

                // a string element in the array denotes a valid header label
                if (label) {
                    contentsRow.push(null);
                    prevLabel = label;
                    return;
                }

                // Try to split the header label of the preceding column into a base name and a
                // trailing integer (fall-back to translated 'Column' as base name), and generate
                // an unused column name by increasing the training index repeatedly.
                var matches = /^(.*?)(\d+)$/.exec(prevLabel);
                var baseLabel = matches ? matches[1] : SheetUtils.getTableColName();
                var labelIndex = matches ? (parseInt(matches[2]) + 1) : 1;
                while ((baseLabel + labelIndex).toUpperCase() in labelSet) { labelIndex += 1; }

                // store the new label in the content array, in the set used to test uniqueness,
                // and in 'prevLabel' for the next iteration cycle
                prevLabel = baseLabel + labelIndex;
                contentsRow.push({ value: prevLabel });
                labelSet[prevLabel.toUpperCase()] = true;
            });

            // Generate one or more 'setCellContents' operations from the entries in 'contentsRow'.
            var startIndex = 0, endIndex = 0;
            while ((startIndex = Utils.findFirstIndex(contentsRow, _.isObject, { begin: endIndex })) >= 0) {

                // find the first null element following a sequence of generated header labels
                endIndex = Utils.findFirstIndex(contentsRow, _.isNull, { begin: startIndex + 1 });
                if (endIndex < 0) { endIndex = contentsRow.length; }

                // generate a 'setCellContents' operation, and apply it locally to insert the cells into the document
                var address = new Address(headerRange.start[0] + startIndex, headerRange.start[1]);
                var contents = [contentsRow.slice(startIndex, endIndex)];
                generateSetCellContentsOperation(generator, address, contents);
                docModel.invokeOperationHandler(generator.getLastOperation());
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Clones all settings from the passed sheet model into this sheet
         * model.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @param {SheetModel} sheetModel
         *  The source sheet model whose contents will be cloned into this
         *  sheet model.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.cloneFrom = function (sheetModel) {

            // copy all attributes
            this.setAttributes(sheetModel.getExplicitAttributes(true));
            this.setViewAttributes(sheetModel.getViewAttributes());

            // clone all collections
            nameCollection.cloneFrom(sheetModel.getNameCollection());
            colCollection.cloneFrom(sheetModel.getColCollection());
            rowCollection.cloneFrom(sheetModel.getRowCollection());
            mergeCollection.cloneFrom(sheetModel.getMergeCollection());
            tableCollection.cloneFrom(sheetModel.getTableCollection());
            validationCollection.cloneFrom(sheetModel.getValidationCollection());
            condFormatCollection.cloneFrom(sheetModel.getCondFormatCollection());
            cellCollection.cloneFrom(sheetModel.getCellCollection());
            drawingCollection.cloneFrom(sheetModel.getDrawingCollection());

            return this;
        };

        /**
         * Handler for the document operation 'setSheetAttributes'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'setSheetAttributes' document operation.
         */
        this.applySetAttributesOperation = function (context) {
            this.setAttributes(context.getObj('attrs'));
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the type of this sheet.
         *
         * @returns {String}
         *  The type identifier of this sheet.
         */
        this.getType = function () {
            return sheetType;
        };

        /**
         * Returns the current sheet index of this sheet model.
         *
         * @returns {Number}
         *  The current zero-based sheet index of this sheet model.
         */
        this.getIndex = function () {
            return docModel.getSheetIndexOfModel(this);
        };

        /**
         * Returns the name of this sheet model.
         *
         * @returns {String}
         *  The name of this sheet model.
         */
        this.getName = function () {
            return docModel.getSheetName(this.getIndex());
        };

        /**
         * Returns the collection of all defined names contained in this sheet.
         *
         * @returns {NameCollection}
         *  The collection of all defined names in this sheet.
         */
        this.getNameCollection = function () {
            return nameCollection;
        };

        /**
         * Returns the collection of all columns in this sheet, containing
         * their size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all columns in this sheet.
         */
        this.getColCollection = function () {
            return colCollection;
        };

        /**
         * Returns the collection of all rows in this sheet, containing their
         * size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all rows in this sheet.
         */
        this.getRowCollection = function () {
            return rowCollection;
        };

        /**
         * Returns the collection of all merged cell ranges.
         *
         * @returns {MergeCollection}
         *  The collection of all merged cell ranges in this sheet.
         */
        this.getMergeCollection = function () {
            return mergeCollection;
        };

        /**
         * Returns the collection of all known cells in this sheet.
         *
         * @returns {CellCollection}
         *  The collection of all known cells in this sheet.
         */
        this.getCellCollection = function () {
            return cellCollection;
        };

        /**
         * Returns the collection of all table ranges in this sheet.
         *
         * @returns {TableCollection}
         *  The collection of all table ranges in this sheet.
         */
        this.getTableCollection = function () {
            return tableCollection;
        };

        /**
         * Returns the collection of all cell ranges with validation settings.
         *
         * @returns {ValidationCollection}
         *  The collection of all cell ranges with validation settings in this
         *  sheet.
         */
        this.getValidationCollection = function () {
            return validationCollection;
        };

        /**
         * Returns the collection of all cell ranges with conditional
         * formatting settings.
         *
         * @returns {CondFormatCollection}
         *  The collection of all cell ranges with conditional formatting
         *  settings in this sheet.
         */
        this.getCondFormatCollection = function () {
            return condFormatCollection;
        };

        /**
         * Returns the drawing collection of this sheet, containing the models
         * of all drawing objects contained in this sheet.
         *
         * @returns {DrawingCollection}
         *  The collection of all drawing models in this sheet.
         */
        this.getDrawingCollection = function () {
            return drawingCollection;
        };

        /**
         * Registers a callback function that will be invoked after columns or
         * rows have been inserted into or deleted from this sheet, but before
         * this sheet model triggers the according insert/delete events.
         *
         * @param {Function} handler
         *  The callback handler function. Will be invoked in the context of
         *  this sheet model instance. Receives the following parameters:
         *  (1) {Interval} interval
         *      The column/row interval of the insert/delete operation.
         *  (2) {Boolean} insert
         *      Whether the specified column/row interval has been inserted
         *      into this sheet (true), or deleted from this sheet (false).
         *  (3) {Boolean} columns
         *      Whether the specified interval is a column interval (true), or
         *      a row interval (false).
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.registerTransformationHandler = function (handler) {
            transformationHandlers.push(handler);
            return this;
        };

        /**
         * Unregisters a callback function that has been registered with the
         * method SheetModel.registerTransformationHandler() before.
         *
         * @param {Function} handler
         *  The callback handler function to be unregistered.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.unregisterTransformationHandler = function (handler) {
            transformationHandlers = _.without(transformationHandlers, handler);
            return this;
        };

        // operations generator -----------------------------------------------

        /**
         * Creates a new operations generator associated to this sheet, invokes
         * the callback function, and then applies all operations contained in
         * the generator, and sends them to the server. This method works the
         * same as the method EditModel.createAndApplyOperations() but creates
         * a dedicated operations generator for this sheet.
         *
         * @param {Function} callback
         *  The callback function. Receives a new operations generator for this
         *  sheet (instance of the class SheetOperationsGenerator) as first
         *  parameter. May return a promise to defer applying and sending the
         *  operations until it has been resolved. Will be called in the
         *  context of this instance.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  EditModel.createAndApplyOperations() except the option 'generator'.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operations have been
         *  applied and sent successfully (with the result of that promise); or
         *  rejected, if the callback has returned a rejected promise (with the
         *  result of that promise), or if sending the operations has failed
         *  for internal reasons (with the object {cause:'operation'}).
         */
        this.createAndApplyOperations = function (callback, options) {
            options = Utils.extendOptions(options, { generator: new SheetOperationsGenerator(this) });
            return docModel.createAndApplyOperations(_.bind(callback, this), options);
        };

        // dimensions ---------------------------------------------------------

        /**
         * Converts the passed length in pixels to a length in 1/100 of
         * millimeters, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in pixels.
         *
         * @returns {Number}
         *  The converted length in 1/100 of millimeters.
         */
        this.convertPixelToHmm = function (length) {
            return Utils.convertLengthToHmm(length / this.getEffectiveZoom(), 'px');
        };

        /**
         * Converts the passed length in 1/100 of millimeters to a length in
         * pixels, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The converted length in pixels.
         */
        this.convertHmmToPixel = function (length) {
            return Utils.convertHmmToLength(length * this.getEffectiveZoom(), 'px', 1);
        };

        /**
         * Returns the absolute position of the passed cell range in the sheet,
         * in 1/100 of millimeters, as well as in pixels according to the
         * current sheet zoom factor.
         *
         * @param {Range} range
         *  The cell range address.
         *
         * @returns {Object}
         *  The absolute position and size of the range in the sheet, in the
         *  following properties:
         *  - {Number} leftHmm
         *      The left offset of the cell range, in 1/100 of millimeters.
         *  - {Number} topHmm
         *      The top offset of the cell range, in 1/100 of millimeters.
         *  - {Number} widthHmm
         *      The total width of the cell range, in 1/100 of millimeters.
         *  - {Number} heightHmm
         *      The total height of the cell range, in 1/100 of millimeters.
         *  - {Number} left
         *      The left offset of the cell range, in pixels.
         *  - {Number} top
         *      The top offset of the cell range, in pixels.
         *  - {Number} width
         *      The total width of the cell range, in pixels.
         *  - {Number} height
         *      The total height of the cell range, in pixels.
         */
        this.getRangeRectangle = function (range) {

            var // the position and size of the column interval
                colPosition = colCollection.getIntervalPosition(range.colInterval()),
                // the position and size of the row interval
                rowPosition = rowCollection.getIntervalPosition(range.rowInterval());

            return getRectangle(colPosition, rowPosition);
        };

        /**
         * Returns the absolute position of the specified cell in the sheet, in
         * 1/100 of millimeters, as well as in pixels according to the current
         * sheet zoom factor.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.expandMerged=false]
         *      If set to true, and the cell is part of a merged range, the
         *      position of the entire merged range will be returned, and the
         *      address of the merged range will be stored in the property
         *      'mergedRange' of the result object returned by this method.
         *
         * @returns {Object}
         *  The absolute position and size of the cell in the sheet, in the
         *  following properties:
         *  - {Number} leftHmm
         *      The left offset of the cell, in 1/100 of millimeters.
         *  - {Number} topHmm
         *      The top offset of the cell, in 1/100 of millimeters.
         *  - {Number} widthHmm
         *      The total width of the cell, in 1/100 of millimeters.
         *  - {Number} heightHmm
         *      The total height of the cell, in 1/100 of millimeters.
         *  - {Number} left
         *      The left offset of the cell, in pixels.
         *  - {Number} top
         *      The top offset of the cell, in pixels.
         *  - {Number} width
         *      The total width of the cell, in pixels.
         *  - {Number} height
         *      The total height of the cell, in pixels.
         *  - {Range} [mergedRange]
         *      The address of the entire merged range the specified cell is
         *      part of, if the option 'expandMerged' has been set, AND the
         *      passed address is part of a merged cell range.
         */
        this.getCellRectangle = function (address, options) {

            var // whether to expand to merged ranges
                expandMerged = Utils.getBooleanOption(options, 'expandMerged', false),
                // a merged range covering the cell position
                mergedRange = expandMerged ? mergeCollection.getMergedRange(address, 'all') : null;

            // try to extend to a merged range
            if (mergedRange) {
                return _.extend(this.getRangeRectangle(mergedRange), { mergedRange: mergedRange });
            }

            // calculate position of the single cell
            return getRectangle(colCollection.getEntry(address[0]), rowCollection.getEntry(address[1]));
        };

        /**
         * Returns the address of the cell range covering the passed absolute
         * sheet rectangle.
         *
         * @param {Object} rectangle
         *  The absolute position and size of the sheet rectangle, in pixels,
         *  or in 1/100 of millimeters, according to the 'pixel' option (see
         *  below).
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options also supported by the
         *  method ColRowCollection.getEntryByOffset(), especially the Boolean
         *  option 'pixel' that causes to interpret the rectangle in pixels.
         *
         * @returns {Range}
         *  The address of the cell range covering the passed rectangle.
         */
        this.getRangeFromRectangle = function (rectangle, options) {

            var // the column and row entries covering the start and end corner of the rectangle
                colEntry1 = colCollection.getEntryByOffset(rectangle.left, options),
                rowEntry1 = rowCollection.getEntryByOffset(rectangle.top, options),
                colEntry2 = colCollection.getEntryByOffset(rectangle.left + rectangle.width - 1, options),
                rowEntry2 = rowCollection.getEntryByOffset(rectangle.top + rectangle.height - 1, options);

            return Range.create(colEntry1.index, rowEntry1.index, colEntry2.index, rowEntry2.index);
        };

        /**
         * Returns whether the passed cell range address consists of a single
         * cell, or whether it exactly covers a single merged range.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Boolean}
         *  Whether the cell is the only (merged) cell in the range.
         */
        this.isSingleCellInRange = function (range) {
            return range.single() || mergeCollection.isMergedRange(range);
        };

        /**
         * If the passed cell range address represents a single cell, or
         * exactly covers a merged range, the surrounding content range will be
         * returned. Otherwise, the passed range will be returned.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Range}
         *  The address of the content range of a single cell, otherwise the
         *  passed cell range address.
         */
        this.getContentRangeForCell = function (range) {
            return this.isSingleCellInRange(range) ? cellCollection.findContentRange(range) : range;
        };

        /**
         * Returns the visible parts of the passed cell range addresses.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {RangeArray}
         *  The visible parts of the passed cell ranges, in the same order. If
         *  a range is split into multiple parts (hidden columns or rows inside
         *  the range), the range parts are ordered row-by-row. If the entire
         *  range list is located in hidden columns or rows, the returned array
         *  will be empty.
         */
        this.getVisibleRanges = function (ranges) {
            return RangeArray.map(ranges, function (range) {
                var colIntervals = colCollection.getVisibleIntervals(range.colInterval()),
                    rowIntervals = rowCollection.getVisibleIntervals(range.rowInterval());
                return RangeArray.createFromIntervals(colIntervals, rowIntervals);
            });
        };

        /**
         * Returns the address of a visible cell located as close as possible
         * to the passed cell address.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {String} method
         *  Specifies how to look for another cell, if the specified cell is
         *  hidden. The following lookup methods are supported:
         *  - 'exact':
         *      Does not look for other columns/rows. If the specified cell is
         *      hidden, returns null.
         *  - 'next':
         *      Looks for a visible column and/or row following the cell.
         *  - 'prev':
         *      Looks for a visible column and/or row preceding the cell.
         *  - 'nextPrev':
         *      First, looks for visible column and/or row following the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *  - 'prevNext':
         *      First, looks for visible column and/or row preceding the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *
         * @returns {Address|Null}
         *  The address of a visible cell near the passed cell address; or null
         *  if all columns or all rows are hidden.
         */
        this.getVisibleCell = function (address, method) {

            var // collection entries of visible column/row
                colEntry = colCollection.getVisibleEntry(address[0], method),
                rowEntry = rowCollection.getVisibleEntry(address[1], method);

            return (colEntry && rowEntry) ? new Address(colEntry.index, rowEntry.index) : null;
        };

        /**
         * Shrinks the passed range address to the visible area, if the leading
         * or trailing columns/rows of the range are hidden.
         *
         * @param {Range} range
         *  The address of the original cell range.
         *
         * @returns {Range|Null}
         *  The address of the visible area of the passed cell range; or null,
         *  if the range is completely hidden. The first and last column, and
         *  the first and last row of the range are visible, but some inner
         *  columns/rows in the range may be hidden.
         */
        this.shrinkRangeToVisible = function (range) {

            var // shrunken visible column interval
                colInterval = colCollection.shrinkIntervalToVisible(range.colInterval()),
                // shrunken visible row interval
                rowInterval = colInterval ? rowCollection.shrinkIntervalToVisible(range.rowInterval()) : null;

            return rowInterval ? Range.createFromIntervals(colInterval, rowInterval) : null;
        };

        /**
         * Returns the address of a range that covers the passed range, and
         * that has been expanded to all hidden columns and rows directly
         * preceding and following the passed range.
         *
         * @param {Range} range
         *  The address of a range to be expanded.
         *
         * @returns {Range}
         *  An expanded cell range address including all leading and trailing
         *  hidden columns and rows.
         */
        this.expandRangeToHidden = function (range) {

            var // expanded column interval
                colInterval = colCollection.expandIntervalToHidden(range.colInterval()),
                // expanded row interval
                rowInterval = rowCollection.expandIntervalToHidden(range.rowInterval());

            return Range.createFromIntervals(colInterval, rowInterval);
        };

        /**
         * Returns the addresses of the merged cell ranges covering the visible
         * parts of the passed ranges. First, the cell ranges will be expanded
         * to the leading and trailing hidden columns and rows. This may reduce
         * the number of ranges in the result, if there are only hidden columns
         * or rows between the passed ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {RangeArray}
         *  The addresses of the merged cell ranges, shrunken to leading and
         *  trailing visible columns and rows (but may contain inner hidden
         *  columns/rows).
         */
        this.mergeAndShrinkRanges = function (ranges) {

            // expand to hidden columns/rows (modify array in-place)
            ranges = RangeArray.map(ranges, this.expandRangeToHidden, this);

            // merge the ranges (result may be smaller due to expansion to hidden columns/rows)
            ranges = ranges.merge();

            // reduce merged ranges to visible parts (filter out ranges completely hidden)
            return RangeArray.map(ranges, this.shrinkRangeToVisible, this);
        };

        // split settings -----------------------------------------------------

        /**
         * Returns whether the view split mode is enabled for the columns in
         * this sheet, regardless of the specific split mode (dynamic or frozen
         * mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the columns in this
         *  sheet.
         */
        this.hasColSplit = function () {
            return this.getViewAttribute('splitWidth') > 0;
        };

        /**
         * Returns whether the view split mode is enabled for the rows in this
         * sheet, regardless of the specific split mode (dynamic or frozen
         * mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the rows in this sheet.
         */
        this.hasRowSplit = function () {
            return this.getViewAttribute('splitHeight') > 0;
        };

        /**
         * Returns whether the view split mode is enabled for this sheet,
         * regardless of the specific split mode (dynamic or frozen mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the sheet.
         */
        this.hasSplit = function () {
            return this.hasColSplit() || this.hasRowSplit();
        };

        /**
         * Returns the current view split mode of this sheet.
         *
         * @returns {String}
         *  The current view split mode. The empty string indicates that the
         *  sheet is not split at all. Otherwise, one of the following values
         *  will be returned:
         *  - 'split': Dynamic split mode (movable split lines).
         *  - 'frozen': Frozen split mode (a fixed number of columns and/or
         *      rows are frozen, split lines are not movable). When leaving
         *      frozen split mode, the split lines will be hidden.
         *  - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
         *      mode). When leaving frozen split mode, the split lines remain
         *      visible and become movable (dynamic 'split' mode).
         */
        this.getSplitMode = function () {
            return this.hasSplit() ? this.getViewAttribute('splitMode') : '';
        };

        /**
         * Returns whether this sheet is currently split, and dynamic split
         * mode is activated. Returns also true, if the split is currently
         * frozen, but will thaw to dynamic split mode ('frozenSplit' mode).
         *
         * @returns {Boolean}
         *  Whether dynamic split mode is activated (split modes 'split' or
         *  'frozenSplit').
         */
        this.hasDynamicSplit = function () {
            return (/^(split|frozenSplit)$/).test(this.getSplitMode());
        };

        /**
         * Returns whether this sheet is currently split, and the frozen split
         * mode is activated.
         *
         * @returns {Boolean}
         *  Whether frozen split mode is activated (split modes 'frozen' or
         *  'frozenSplit').
         */
        this.hasFrozenSplit = function () {
            return (/^(frozen|frozenSplit)$/).test(this.getSplitMode());
        };

        /**
         * Returns the width of the left area in a sheet in split mode, in
         * 1/100 of millimeters. In frozen split mode, returns the total size
         * of the frozen columns.
         *
         * @returns {Number}
         *  The width of the left area if the sheet is in split mode, otherwise
         *  zero.
         */
        this.getSplitWidthHmm = function () {
            return getSplitSize(true, false);
        };

        /**
         * Returns the height of the upper area in a sheet in split mode, in
         * 1/100 of millimeters. In frozen split mode, returns the total size
         * of the frozen rows.
         *
         * @returns {Number}
         *  The height of the upper area if the sheet is in split mode,
         *  otherwise zero.
         */
        this.getSplitHeightHmm = function () {
            return getSplitSize(false, false);
        };

        /**
         * Returns the width of the left area in a sheet in split mode, in
         * pixels according to the current sheet zoom factor. In frozen split
         * mode, returns the total size of the frozen columns.
         *
         * @returns {Number}
         *  The width of the left area if the sheet is in split mode, otherwise
         *  zero.
         */
        this.getSplitWidth = function () {
            return getSplitSize(true, true);
        };

        /**
         * Returns the height of the upper area in a sheet in split mode, in
         * pixels according to the current sheet zoom factor. In frozen split
         * mode, returns the total size of the frozen rows.
         *
         * @returns {Number}
         *  The height of the upper area if the sheet is in split mode,
         *  otherwise zero.
         */
        this.getSplitHeight = function () {
            return getSplitSize(false, true);
        };

        /**
         * Returns the column interval shown in the left area of the sheet. If
         * the first or last visible columns are visible less than half of
         * their width, they will not be included in the interval, with one
         * exception: If the resulting interval would be empty, it will contain
         * the first visible column instead, regardless how much of its width
         * is visible.
         *
         * @returns {Interval|Null}
         *  The column interval shown in the left area of the sheet if it is in
         *  split mode, otherwise null.
         */
        this.getSplitColInterval = function () {
            return getSplitInterval(true);
        };

        /**
         * Returns the row interval shown in the upper area of the sheet. If
         * the first or last visible rows are visible less than half of their
         * height, they will not be included in the interval, with one
         * exception: If the resulting interval would be empty, it will contain
         * the first visible row instead, regardless how much of its height
         * is visible.
         *
         * @returns {Interval|Null}
         *  The row interval shown in the upper area of the sheet if it is in
         *  split mode, otherwise null.
         */
        this.getSplitRowInterval = function () {
            return getSplitInterval(false);
        };

        /**
         * Sets the split size for dynamic split mode, and updates the scroll
         * anchor view attributes, if the horizontal or vertical split will be
         * enabled.
         *
         * @param {Number} splitWidth
         *  The new value for the 'splitWidth' view attribute, in 1/100 of
         *  millimeters. The value zero will disable the split between left and
         *  right sheet area.
         *
         * @param {Number} splitHeight
         *  The new value for the 'splitHeight' view attribute, in 1/100 of
         *  millimeters. The value zero will disable the split between upper
         *  and lower sheet area.
         *
         * @param {Object} [attributes]
         *  Additional view attributes to be set at this sheet model.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setDynamicSplit = function (splitWidth, splitHeight, attributes) {

            var // the old split size
                oldSplitWidth = this.getSplitWidthHmm(),
                oldSplitHeight = this.getSplitHeightHmm();

            // the new view attributes for split mode
            attributes = _.isObject(attributes) ? _.copy(attributes, true) : {};
            _.extend(attributes, { splitMode: 'split', splitWidth: splitWidth, splitHeight: splitHeight });

            // Update left/right scroll positions. Copy from left to right, if
            // split has been disabled, or copy from right to left, if split
            // has been enabled.
            if ((oldSplitWidth > 0) && (attributes.splitWidth === 0)) {
                attributes.anchorRight = this.getViewAttribute('anchorLeft');
            } else if ((oldSplitWidth === 0) && (attributes.splitWidth > 0)) {
                attributes.anchorLeft = this.getViewAttribute('anchorRight');
            }

            // Update top/bottom scroll positions. Copy from top to bottom, if
            // split has been disabled, or copy from bottom to top, if split
            // has been enabled.
            if ((oldSplitHeight > 0) && (attributes.splitHeight === 0)) {
                attributes.anchorBottom = this.getViewAttribute('anchorTop');
            } else if ((oldSplitHeight === 0) && (attributes.splitHeight > 0)) {
                attributes.anchorTop = this.getViewAttribute('anchorBottom');
            }

            // set the new attributes
            this.setViewAttributes(attributes);
            return this;
        };

        /**
         * Sets the split size for frozen split mode, and updates the scroll
         * anchor view attributes.
         *
         * @param {Interval|Null} colInterval
         *  If set to an index interval, the new column interval to be shown in
         *  frozen split mode. The value null will disable the split between
         *  left and right sheet area.
         *
         * @param {Interval|Null} rowInterval
         *  If set to an index interval, the new row interval to be shown in
         *  frozen split mode. The value null will disable the split between
         *  top and bottom sheet area.
         *
         * @param {Object} [attributes]
         *  Additional view attributes to be set at this sheet model.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setFrozenSplit = function (colInterval, rowInterval, attributes) {

            // the new view attributes for split mode
            attributes = _.isObject(attributes) ? _.copy(attributes, true) : {};
            attributes.splitMode = this.hasDynamicSplit() ? 'frozenSplit' : 'frozen';

            // set column interval if specified
            if (colInterval) {
                attributes.splitWidth = colInterval.size();
                attributes.anchorLeft = colInterval.first;
                attributes.anchorRight = colInterval.last + 1;
            } else {
                attributes.splitWidth = 0;
            }

            // set row interval if specified
            if (rowInterval) {
                attributes.splitHeight = rowInterval.size();
                attributes.anchorTop = rowInterval.first;
                attributes.anchorBottom = rowInterval.last + 1;
            } else {
                attributes.splitHeight = 0;
            }

            // set the new attributes
            this.setViewAttributes(attributes);
            return this;
        };

        /**
         * Resets all split attributes, effectively disabling any split view
         * that is currently active.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.clearSplit = function () {
            return this.setDynamicSplit(0, 0);
        };

        // sheet operations ---------------------------------------------------

        /**
         * Returns whether this sheet is visible in the user interface.
         *
         * @returns {Boolean}
         *  Whether this sheet is visible.
         */
        this.isVisible = function () {
            return this.getMergedAttributes().sheet.visible;
        };

        /**
         * Returns whether this sheet is locked. In locked sheets it is not
         * possible to insert, delete, or modify columns and rows, to edit
         * protected cells, or to manipulate drawing objects.
         *
         * @returns {Boolean}
         *  Whether this sheet is locked.
         */
        this.isLocked = function () {
            return this.getMergedAttributes().sheet.locked;
        };

        /**
         * Returns the effective zoom factor to be used for this sheet,
         * according to the value of the 'zoom' view attribute, and the type of
         * the current device. On touch devices, the effective zoom factor will
         * be increased for better usability.
         *
         * @returns {Number}
         *  The effective zoom factor for this sheet.
         */
        this.getEffectiveZoom = function () {
            var zoom = this.getViewAttribute('zoom');
            return Utils.TOUCHDEVICE ? (zoom * 1.25) : zoom;
        };

        /**
         * Returns the effective horizontal padding between cell grid lines and
         * the text contents of the cell, according to the current zoom factor
         * of this sheet.
         *
         * @returns {Number}
         *  The effective horizontal cell content padding, in pixels.
         */
        this.getEffectiveTextPadding = function () {
            // 2 pixels in 100% zoom, round up early
            return Math.floor(2 * this.getEffectiveZoom() + 0.75);
        };

        /**
         * Returns the current grid color of this sheet.
         *
         * @returns {Color}
         *  The current grid color of this sheet, as instance of the class
         *  Color.
         */
        this.getGridColor = function () {
            var gridColor = Color.parseJSON(this.getMergedAttributes().sheet.gridColor);
            return gridColor ? gridColor : new Color('auto');
        };

        /**
         * On first invocation, initializes the view attributes for this sheet
         * from the imported sheet view attributes.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.initializeViewAttributes = _.once(_.bind(function () {

            var // the merged sheet attributes
                sheetAttributes = self.getMergedAttributes().sheet,
                // all selected ranges (convert plain JSON data to RangeArray)
                selectedRanges = docModel.createRangeArray(sheetAttributes.selectedRanges),
                // the array index of the active range (bug 34549: filter may send invalid index)
                activeIndex = Utils.getIntegerOption(sheetAttributes, 'activeIndex', 0),
                // address of the active cell (convert plain JSON data to RangeArray)
                activeCell = docModel.createAddress(sheetAttributes.activeCell),
                // the array of selected drawing objects
                selectedDrawings =  Utils.getArrayOption(sheetAttributes, 'selectedDrawings', []),
                // selection, built from multiple sheet attributes, expanded to merged ranges
                selection = (selectedRanges && (selectedRanges.length > 0)) ? new SheetSelection(getExpandedRanges(selectedRanges), activeIndex, activeCell, selectedDrawings) : null,
                // the resulting view attributes
                viewAttributes = {
                    zoom: sheetAttributes.zoom,
                    showGrid: sheetAttributes.showGrid,
                    splitMode: sheetAttributes.splitMode,
                    splitWidth: sheetAttributes.splitWidth,
                    splitHeight: sheetAttributes.splitHeight,
                    activePane: sheetAttributes.activePane,
                    anchorLeft: sheetAttributes.scrollLeft,
                    anchorRight: sheetAttributes.scrollRight,
                    anchorTop: sheetAttributes.scrollTop,
                    anchorBottom: sheetAttributes.scrollBottom
                };

            // show warnings for invalid selection settings (bug 34549)
            if (!selection) {
                Utils.warn('SheetModel.initializeViewAttributes(): sheet=' + this.getIndex() + ', invalid selection');
                selection = SheetSelection.createFromAddress(Address.A1);
                selection.ranges = getExpandedRanges(selection.ranges);
            } else {
                if (selection.active !== sheetAttributes.activeIndex) {
                    Utils.warn('SheetModel.initializeViewAttributes(): sheet=' + this.getIndex() + ', invalid range index in selection');
                }
                if (!selection.activeRange().containsAddress(selection.address)) {
                    Utils.warn('SheetModel.initializeViewAttributes(): sheet=' + this.getIndex() + ', invalid active cell in selection');
                    selection.address = selection.activeRange().start.clone();
                }
            }
            viewAttributes.selection = selection;

            // Sheet view attributes in operations use the top-left grid pane as default pane
            // which is always visible without active view split. Internally, the view uses the
            // bottom-right grid pane as the default pane instead. According to the split settings,
            // the 'activePane' attribute, and the scroll attributes have to be translated.

            // no horizontal split: map all left pane settings to right pane settings
            if (sheetAttributes.splitWidth === 0) {
                viewAttributes.activePane = PaneUtils.getNextPanePos(viewAttributes.activePane, 'right');
                viewAttributes.anchorRight = viewAttributes.anchorLeft;
                delete viewAttributes.anchorLeft;
            }

            // no vertical split: map all top pane settings to bottom pane settings
            if (sheetAttributes.splitHeight === 0) {
                viewAttributes.activePane = PaneUtils.getNextPanePos(viewAttributes.activePane, 'bottom');
                viewAttributes.anchorBottom = viewAttributes.anchorTop;
                delete viewAttributes.anchorTop;
            }

            this.setViewAttributes(viewAttributes);
            return this;
        }, this));

        /**
         * Returns all changed sheet attributes as used in 'setSheetAttributes'
         * operations according to the current runtime view settings.
         *
         * @returns {Object}
         *  All changed sheet attributes matching the current view settings of
         *  this sheet.
         */
        this.getChangedViewAttributes = function () {

            var // the current merged sheet attributes
                oldSheetAttributes = this.getMergedAttributes().sheet,
                // the current view settings
                viewAttributes = this.getViewAttributes(),
                // the new sheet attributes according to the view settings
                newSheetAttributes = {
                    selectedRanges: viewAttributes.selection.ranges.toJSON(),
                    activeIndex: viewAttributes.selection.active,
                    activeCell: viewAttributes.selection.address.toJSON(),
                    selectedDrawings: viewAttributes.selection.drawings,
                    zoom: viewAttributes.zoom,
                    showGrid: viewAttributes.showGrid,
                    splitMode: viewAttributes.splitMode,
                    splitWidth: viewAttributes.splitWidth,
                    splitHeight: viewAttributes.splitHeight,
                    activePane: viewAttributes.activePane,
                    scrollLeft: getScrollIndex(viewAttributes.anchorLeft, true),
                    scrollRight: getScrollIndex(viewAttributes.anchorRight, true),
                    scrollTop: getScrollIndex(viewAttributes.anchorTop, false),
                    scrollBottom: getScrollIndex(viewAttributes.anchorBottom, false)
                };

            // converts the passed scroll anchor to a simple column/row index
            function getScrollIndex(scrollAnchor, columns) {
                return Math.min(Math.round(scrollAnchor), docModel.getMaxIndex(columns));
            }

            // Sheet view attributes in operations use the top-left grid pane as default pane
            // which is always visible without active view split. Internally, the view uses the
            // bottom-right grid pane as the default pane instead. According to the split settings,
            // the 'activePane' attribute, and the scroll attributes have to be translated.

            // no horizontal split: map all right pane settings to left pane settings
            if (newSheetAttributes.splitWidth === 0) {
                newSheetAttributes.activePane = PaneUtils.getNextPanePos(newSheetAttributes.activePane, 'left');
                newSheetAttributes.scrollLeft = newSheetAttributes.scrollRight;
                delete newSheetAttributes.scrollRight;
            }

            // no vertical split: map all bottom pane settings to top pane settings
            if (newSheetAttributes.splitHeight === 0) {
                newSheetAttributes.activePane = PaneUtils.getNextPanePos(newSheetAttributes.activePane, 'top');
                newSheetAttributes.scrollTop = newSheetAttributes.scrollBottom;
                delete newSheetAttributes.scrollBottom;
            }

            // reduce new sheet attribute to changed ones
            _.each(newSheetAttributes, function (value, name) {
                if (_.isEqual(oldSheetAttributes[name], value)) {
                    delete newSheetAttributes[name];
                }
            });

            return newSheetAttributes;
        };

        /**
         * Returns whether this sheet contains changed view attributes compared
         * to the sheet view attributes imported from the document file.
         *
         * @returns {Boolean}
         *  Whether this sheet contains changed view attributes.
         */
        this.hasChangedViewAttributes = function () {

            var // the new sheet attributes according to the view settings
                sheetAttributes = this.getChangedViewAttributes();

            // delete 'unimportant' view attributes (selection, scroll positions)
            delete sheetAttributes.selectedRanges;
            delete sheetAttributes.activeIndex;
            delete sheetAttributes.activeCell;
            delete sheetAttributes.selectedDrawings;
            delete sheetAttributes.activePane;
            delete sheetAttributes.scrollLeft;
            delete sheetAttributes.scrollRight;
            delete sheetAttributes.scrollTop;
            delete sheetAttributes.scrollBottom;

            // return whether there are changed view attributes left
            return !_.isEmpty(sheetAttributes);
        };

        // column operations --------------------------------------------------

        /**
         * Returns whether additional columns can be inserted into the sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of column intervals, or a single column interval.
         *
         * @returns {Boolean}
         *  Whether additional columns can be inserted, according to the passed
         *  column intervals and the number of used columns in the sheet.
         */
        this.canInsertColumns = function (intervals) {
            return intervalsInsertable(intervals, true);
        };

        /**
         * Generates and applies all operations to insert new columns into the
         * sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of column intervals, or a single column interval.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operations.
         */
        this.insertColumns = function (intervals) {

            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {

                // merge and sort the passed intervals
                intervals = intervals.merge();

                // Bug 39869: collect all table ranges that will be expanded to be
                // able to generate new header labels for empty columns below
                var tableModels = tableCollection.getAllTables().filter(function (tableModel) {

                    // nothing to do for auto-filters, or for tables without header row
                    if (tableModel.isAutoFilter() || !tableModel.hasHeaderRow()) { return false; }

                    // process the table, if its range will be expanded by the passed insertion intervals
                    var headerInterval = tableModel.getRange().colInterval();
                    return intervals.some(function (interval) {
                        return (headerInterval.first < interval.first) && (interval.first <= headerInterval.last);
                    });
                });

                // build an array of range arrays that can be passed to the cell collection to query the header cells
                var headerRangesArray = tableModels.map(function (tableModel) {
                    return new RangeArray(tableModel.getRange().header());
                });

                // Query all header cells from the cell collection (this will cause server requests
                // for unknown cells, and therefore MUST be done before manipulating the cell
                // collection locally via the 'insertColumns' operation). When the promise resolves,
                // all header cells will be available in the cell collection.
                var promise = cellCollection.queryCellContents(headerRangesArray, { hidden: true });

                // generate all 'insertColumns' operations with dependent drawing operations
                promise = promise.then(function () {
                    return generateIntervalOperations(generator, Operations.INSERT_COLUMNS, intervals);
                });

                // generate all missing header labels in all expanded table ranges
                promise = promise.then(function () {
                    return self.iterateArraySliced(tableModels, function (tableModel) {
                        return generateMissingTableHeaderLabels(generator, tableModel);
                    });
                });

                return promise;
            });
        };

        /**
         * Returns whether the specified columns can be deleted from the sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of column intervals, or a single column interval.
         *
         * @returns {Boolean}
         *  Whether the columns can be deleted.
         */
        this.canDeleteColumns = function (intervals) {
            return intervalsDeleteable(intervals, true);
        };

        /**
         * Generates and applies all operations to delete columns from the
         * sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of column intervals, or a single column interval.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operations.
         */
        this.deleteColumns = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateIntervalOperations(generator, Operations.DELETE_COLUMNS, intervals);
            });
        };

        /**
         * Returns the mixed formatting attributes of all columns covered by
         * the passed column intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of column intervals, or a single column interval.
         *
         * @returns {Object}
         *  The mixed attributes of all columns covered by the passed
         *  intervals, as simple object (NOT mapped as a 'column' sub-object).
         *  All attributes that could not be resolved unambiguously will be set
         *  to the value null.
         */
        this.getColumnAttributes = function (intervals) {
            return colCollection.getMixedAttributes(intervals);
        };

        /**
         * Generates and applies all operations to modify the specified column
         * attributes in the sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of column intervals, or a single column interval. Each
         *  column interval MUST contain an additional object property 'attrs'
         *  containing the (incomplete) column attributes, as simple object
         *  (NOT mapped as a 'column' sub-object).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'cols:overflow': Too many columns in the passed intervals.
         *  - 'operation': Internal error while applying the operations.
         */
        this.setColumnAttributes = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateChangeIntervalOperations(generator, intervals, true);
            });
        };

        // row operations -----------------------------------------------------

        /**
         * Returns whether additional rows can be inserted into the sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of row intervals, or a single row interval.
         *
         * @returns {Boolean}
         *  Whether additional rows can be inserted, according to the passed
         *  row intervals and the number of used rows in the sheet.
         */
        this.canInsertRows = function (intervals) {
            return intervalsInsertable(intervals, false);
        };

        /**
         * Generates and applies all operations to insert new rows into the
         * sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of row intervals, or a single row interval.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operations.
         */
        this.insertRows = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateIntervalOperations(generator, Operations.INSERT_ROWS, intervals);
            });
        };

        /**
         * Returns whether the specified rows can be deleted from the sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of row intervals, or a single row interval.
         *
         * @returns {Boolean}
         *  Whether the rows can be deleted.
         */
        this.canDeleteRows = function (intervals) {
            // bug 39869: tables want to prevent deleting their header row, and all of their data rows
            return intervalsDeleteable(intervals, false) && tableCollection.canDeleteRows(intervals);
        };

        /**
         * Generates and applies all operations to delete rows from the sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of row intervals, or a single row interval.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operations.
         */
        this.deleteRows = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateIntervalOperations(generator, Operations.DELETE_ROWS, intervals);
            });
        };

        /**
         * Returns the mixed formatting attributes of all rows covered by the
         * passed row intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of row intervals, or a single row interval.
         *
         * @returns {Object}
         *  The mixed attributes of all rows covered by the passed intervals,
         *  as simple object (NOT mapped as a 'row' sub-object). All attributes
         *  that could not be resolved unambiguously will be set to the value
         *  null.
         */
        this.getRowAttributes = function (intervals) {
            return rowCollection.getMixedAttributes(intervals);
        };

        /**
         * Generates and applies all operations to modify the specified row
         * attributes in the sheet.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of row intervals, or a single row interval. Each interval
         *  object MUST contain an additional object property 'attrs'
         *  containing the (incomplete) row attributes, as simple object (NOT
         *  mapped as a 'row' sub-object).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'rows:overflow': Too many rows in the passed intervals.
         *  - 'operation': Internal error while applying the operations.
         */
        this.setRowAttributes = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateChangeIntervalOperations(generator, intervals, false);
            });
        };

        // cell operations ----------------------------------------------------

        /**
         * Generates and applies a 'setCellContents' document operation that
         * will insert the specified cell contents (values and/or formatting)
         * into the sheet.
         *
         * @param {Address} start
         *  The address of the first cell to be modified.
         *
         * @param {Array<Array<Object|Null>>} contents
         *  A two-dimensional array containing the values and attribute sets
         *  for the cells. Null elements specify that the respective cells will
         *  be skipped (kept unmodified).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI locale.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operation.
         */
        this.setCellContents = function (start, contents, options) {
            return this.createAndApplyOperations(function (generator) {
                generateSetCellContentsOperation(generator, start, contents, options);
            });
        };

        /**
         * Generates and applies a 'setCellContents' operation that will insert
         * the specified contents (values and/or formatting) into a single cell
         * in the sheet.
         *
         * @param {Address} address
         *  The address of the cell to be modified.
         *
         * @param {Any} [value]
         *  The new value for the active cell. If omitted, only the cell
         *  formatting will be changed. If set to null or an empty string, the
         *  cell value will be deleted.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI locale.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell while applying the new attributes.
         *  @param {Any} [options.result]
         *      The result of a formula, as calculated by the local formula
         *      interpreter.
         *  @param {String} [options.url]
         *      The URL of a hyperlink to be set at the cell. If set to the
         *      empty string, an existing hyperlink will be removed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operation.
         */
        this.setSingleCellContents = function (address, value, attributes, options) {

            var // the cell contents object (value and attributes)
                contents = { attrs: attributes },
                // the URL of a hyperlink for the cell
                url = Utils.getStringOption(options, 'url', null),
                // the formula result passed in the options
                result = Utils.getOption(options, 'result');

            // add cell value, if passed (use null to delete the cell value)
            if (_.isString(value) && (value.length === 0)) {
                contents.value = null;
                options = Utils.extendOptions(options, { parse: false });
            } else if (!_.isUndefined(value)) {
                contents.value = value;
            } else {
                options = Utils.extendOptions(options, { parse: false });
            }

            // add all attributes to be cleared
            if (Utils.getBooleanOption(options, 'clear', false)) {
                contents.attrs = Utils.extendOptions(buildNullAttributes(), contents.attrs);
            }

            // remove invalid or empty attributes object
            if (!_.isObject(contents.attrs) || _.isEmpty(contents.attrs)) {
                delete contents.attrs;
            }

            // generate and apply all operations
            return this.createAndApplyOperations(function (generator) {

                // do nothing, if neither value nor attributes are present
                if (!_.isEmpty(contents)) {

                    // add the result property for formulas
                    if (_.isString(value) && !_.isUndefined(result)) {
                        contents.result = result;
                    }

                    // generate the 'setCellContents' operation (expects two-dimensional array)
                    generateSetCellContentsOperation(generator, address, [[contents]], options);
                }

                // create or remove a hyperlink for the cell
                if (_.isString(url)) {
                    generateHyperlinkOperation(generator, new Range(address), url);
                }
            });
        };

        /**
         * Generates and applies 'fillCellRange' operations that will insert
         * the specified contents (values and/or formatting) into a range of
         * cells in the sheet.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Any} [value]
         *  The new value for all cells in the passed ranges. If omitted, only
         *  the cell formatting will be changed. If set to null or an empty
         *  string, the cell values will be deleted.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cells.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI locale.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell ranges while applying the new attributes.
         *  @param {Address} [options.ref]
         *      The address of the reference cell to be inserted into the
         *      operations. If omitted, the top-left cell of the first range
         *      passed to this method will be used.
         *  @param {String} [options.url]
         *      The URL of a hyperlink to be set at the cell. If set to the
         *      empty string, an existing hyperlink will be removed.
         *  @param {String} [options.cat]
         *      The number format category the format code contained in the
         *      passed formatting attributes belongs to. Will be copied to all
         *      affected cells in the cell collection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'cols:overflow': The cell ranges contain too many entire columns.
         *  - 'rows:overflow': The cell ranges contain too many entire rows.
         *  - 'cells:overflow': The cell ranges contain too many filled cells.
         *  - 'operation': Internal error while applying the operation.
         */
        this.fillCellRanges = function (ranges, value, attributes, options) {

            var // additional properties for the 'fillCellRange' operations
                properties = {},
                // reference cell address
                ref = Utils.getObjectOption(options, 'ref'),
                // the URL of a hyperlink for the cell
                url = Utils.getStringOption(options, 'url', null),
                // the new number format category
                category = Utils.getStringOption(options, 'cat', '');

            // convert ranges to unique array
            ranges = RangeArray.get(ranges).unify();
            if (ranges.empty()) { return $.when(); }

            // use top-left cell as reference cell, if missing
            if (!ref) { ref = ranges.first().start; }

            // treat empty string as 'clear value' function
            if (_.isString(value) && (value.length === 0)) {
                value = null;
            }

            // add value to the operation options
            if (!_.isUndefined(value)) {
                properties.value = value;
            }

            // prepare shared formulas, add the parse property
            if (_.isString(value)) {
                properties.shared = 0;
                properties.ref = ref.toJSON();
                // bug 32624: need to insert 'en_US' for unparsed contents
                properties.parse = Utils.getBooleanOption(options, 'parse', false) ? Config.LOCALE : 'en_US';
            }

            // validate attributes
            if (!_.isObject(attributes) || _.isEmpty(attributes)) {
                attributes = null;
            }

            // add all attributes to be cleared
            if (Utils.getBooleanOption(options, 'clear', false)) {
                attributes = Utils.extendOptions(buildNullAttributes(), attributes);
            }

            // delete the URL explicitly, if text is cleared
            if (_.isNull(value)) {
                attributes = Utils.extendOptions({ character: { url: null } }, attributes);
            }

            // check number of modified cells (but do not limit number of cleared cells)
            if ('value' in properties) {
                if (attributes || !_.isNull(properties.value)) {
                    if (ranges.cells() > SheetUtils.MAX_FILL_CELL_COUNT) {
                        return SheetUtils.makeRejected('cells:overflow');
                    }
                }
            }

            // generate and apply all operations
            return this.createAndApplyOperations(function (generator) {

                // create operation to insert a dirty cell style into the document
                if (attributes && _.isString(attributes.styleId)) {
                    generator.generateMissingStyleSheetOperation('cell', attributes.styleId);
                }

                // always create 'fillCellRange' operations for cell values (and their attributes)
                if ('value' in properties) {

                    // add attribute to operation
                    if (attributes) {
                        properties.attrs = attributes;
                    }

                    // generate all range operations
                    ranges.forEach(function (range) {
                        generator.generateRangeOperation(Operations.FILL_CELL_RANGE, range, properties);
                    });

                // create 'fillCellRange', 'setColumnAttributes', or 'setRowAttributes' operations for attributes only
                } else if (_.isObject(attributes)) {

                    // check column/row/cell limits
                    var promise = checkCellLimits(ranges);
                    if (promise) { return promise; }

                    ranges.forEach(function (range) {
                        generateFillRangeOperation(generator, range, attributes);
                    });
                }

                // create or remove hyperlinks for the ranges
                if (_.isString(url)) {
                    ranges.forEach(function (range) {
                        generateHyperlinkOperation(generator, range, url);
                    });
                }

                // copy number format category into the cell collection (bug 34021: before applying the operation)
                if (category.length > 0) {
                    ranges.forEach(function (range) {
                        if (!docModel.isColRange(range) && !docModel.isRowRange(range)) {
                            cellCollection.setFormatCategory(range, category);
                        }
                    });
                }
            });
        };

        /**
         * Generates and applies 'fillCellRange' operations that will change
         * the visibility of the specified borders in the cell ranges. The
         * border attributes will be applied at range level (inner and outer
         * borders) instead of per-cell level.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} borderAttributes
         *  An attribute map containing all border attributes to be changed in
         *  the passed cell ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'cols:overflow': The cell ranges contain too many entire columns.
         *  - 'rows:overflow': The cell ranges contain too many entire rows.
         *  - 'cells:overflow': The cell ranges contain too many filled cells.
         *  - 'operation': Internal error while applying the operation.
         */
        this.setBorderAttributes = function (ranges, borderAttributes) {

            var // additional properties for the operations
                properties = { attrs: { cell: borderAttributes }, rangeBorders: true };

            // convert ranges to unique array
            ranges = RangeArray.get(ranges).unify();
            if (ranges.empty()) { return $.when(); }

            // reduce all ranges to a single column/row, if only a single outer line will be changed
            if (_.size(borderAttributes) === 1) {
                if ('borderLeft' in borderAttributes) {
                    ranges.forEach(function (range) { range.end[0] = range.start[0]; });
                    ranges = ranges.unify();
                } else if ('borderRight' in borderAttributes) {
                    ranges.forEach(function (range) { range.start[0] = range.end[0]; });
                    ranges = ranges.unify();
                } else if ('borderTop' in borderAttributes) {
                    ranges.forEach(function (range) { range.end[1] = range.start[1]; });
                    ranges = ranges.unify();
                } else if ('borderBottom' in borderAttributes) {
                    ranges.forEach(function (range) { range.start[1] = range.end[1]; });
                    ranges = ranges.unify();
                }
            }

            // check column/row/cell limits
            var promise = checkCellLimits(ranges);
            if (promise) { return promise; }

            // generate and apply all operations
            return this.createAndApplyOperations(function (generator) {

                // generates a 'setColumnAttributes' or 'setRowAttributes' operation for the adjacent cells of a column
                function generateAdjacentIntervalOperations(index, newBorderName, columns) {
                    var OPERATION_NAME = columns ? Operations.CHANGE_COLUMNS : Operations.CHANGE_ROWS,
                        opProperties = { attrs: { cell: Utils.makeSimpleObject(newBorderName, Border.NONE) }, rangeBorders: true };
                    // do not send borders for rows without custom format
                    if (columns || rowCollection.getEntry(index).attributes.row.customFormat) {
                        generator.generateIntervalOperation(OPERATION_NAME, new Interval(index), opProperties);
                    }
                }

                // generates a 'fillCellRange' operation for the adjacent cells of one border
                function generateAdjacentRangeOperations(firstCol, firstRow, lastCol, lastRow, newBorderName) {
                    var range = Range.create(firstCol, firstRow, lastCol, lastRow),
                        opProperties = { attrs: { cell: Utils.makeSimpleObject(newBorderName, Border.NONE) }, rangeBorders: true };
                    generator.generateRangeOperation(Operations.FILL_CELL_RANGE, range, opProperties);
                }

                // send the 'fillCellRange' operations
                ranges.forEach(function (range) {

                    // generate a 'setColumnAttributes', 'setRowAttributes', or 'fillCellRange' operation for the current range
                    if (docModel.isColRange(range)) {
                        generator.generateIntervalOperation(Operations.CHANGE_COLUMNS, range.colInterval(), properties);

                        // Generate operation for the adjacent cells of the range
                        if ((range.start[0] > 0) && ('borderLeft' in borderAttributes)) {
                            generateAdjacentIntervalOperations(range.start[0] - 1, 'borderRight', true);
                        }
                        if ((range.end[0] < docModel.getMaxCol()) && ('borderRight' in borderAttributes)) {
                            generateAdjacentIntervalOperations(range.end[0] + 1, 'borderLeft', true);
                        }
                    } else if (docModel.isRowRange(range)) {
                        properties.attrs.row = { customFormat: true };
                        generator.generateIntervalOperation(Operations.CHANGE_ROWS, range.rowInterval(), properties);

                        // Generate operation for the adjacent cells of the range
                        if ((range.start[1] > 0) && ('borderTop' in borderAttributes)) {
                            generateAdjacentIntervalOperations(range.start[1] - 1, 'borderBottom', false);
                        }
                        if ((range.end[1] < docModel.getMaxRow()) && ('borderBottom' in borderAttributes)) {
                            generateAdjacentIntervalOperations(range.end[1] + 1, 'borderTop', false);
                        }
                    } else {
                        generator.generateRangeOperation(Operations.FILL_CELL_RANGE, range, properties);

                        // Deleting all borders at neighboring cells -> first approach: delete all neighboring borders, so that
                        // there is always only one valid border between two cells.
                        // Generate operation for the adjacent cells of the range
                        if ((range.start[0] > 0) && ('borderLeft' in borderAttributes)) {
                            generateAdjacentRangeOperations(range.start[0] - 1, range.start[1], range.start[0] - 1, range.end[1], 'borderRight');
                        }
                        if ((range.end[0] < docModel.getMaxCol()) && ('borderRight' in borderAttributes)) {
                            generateAdjacentRangeOperations(range.end[0] + 1, range.start[1], range.end[0] + 1, range.end[1], 'borderLeft');
                        }
                        if ((range.start[1] > 0) && ('borderTop' in borderAttributes)) {
                            generateAdjacentRangeOperations(range.start[0], range.start[1] - 1, range.end[0], range.start[1] - 1, 'borderBottom');
                        }
                        if ((range.end[1] < docModel.getMaxRow()) && ('borderBottom' in borderAttributes)) {
                            generateAdjacentRangeOperations(range.start[0], range.end[1] + 1, range.end[0], range.end[1] + 1, 'borderTop');
                        }
                    }
                });
            });
        };

        /**
         * Generates and applies 'fillCellRange' operations that will change
         * the specified border properties (line style, line color, line width)
         * of all visible cell borders in a range of cells.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the cell range.
         *  Omitted border properties will not be changed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operation.
         */
        this.changeVisibleBorders = function (ranges, border) {

            var // additional properties for the operations
                properties = { rangeBorders: true, visibleBorders: true };

            // convert ranges to unique array
            ranges = RangeArray.get(ranges).unify();
            if (ranges.empty()) { return $.when(); }

            // check column/row/cell limits
            var promise = checkCellLimits(ranges);
            if (promise) { return promise; }

            // generate and apply all operations
            return this.createAndApplyOperations(function (generator) {

                // generates an operation for the passed range for horizontal or vertical borders
                function generateRangeOperation(range, columns) {

                    var // the passed range, that will be expanded at two borders
                        expandedRange = range.clone(),
                        // the border attributes inserted into the operation
                        borderAttributes = {};

                    // add inner border property
                    borderAttributes[columns ? 'borderInsideVert' : 'borderInsideHor'] = border;

                    // expand range at leading border, or add outer border attribute at leading sheet border
                    if (expandedRange.getStart(columns) > 0) {
                        expandedRange.start.move(-1, columns);
                    } else {
                        borderAttributes[columns ? 'borderLeft' : 'borderTop'] = border;
                    }

                    // expand range at trailing border, or add outer border attribute at trailing sheet border
                    if (expandedRange.getEnd(columns) < docModel.getMaxIndex(columns)) {
                        expandedRange.end.move(1, columns);
                    } else {
                        borderAttributes[columns ? 'borderRight' : 'borderBottom'] = border;
                    }

                    generateFillRangeOperation(generator, expandedRange, { cell: borderAttributes }, properties);
                }

                // expand the ranges and generate operations for the border attributes
                ranges.forEach(function (range) {
                    generateRangeOperation(range, true);
                    generateRangeOperation(range, false);
                });
            });
        };

        /**
         * Generates and applies 'clearCellRange' operations that will delete
         * the values and formatting from a range of cells in the sheet.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operation.
         */
        this.clearCellRanges = function (ranges) {

            // convert ranges to unique array
            ranges = RangeArray.get(ranges).merge();
            if (ranges.empty()) { return $.when(); }

            // generate and apply all operations
            return this.createAndApplyOperations(function (generator) {
                ranges.forEach(function (range) {
                    generator.generateRangeOperation(Operations.CLEAR_CELL_RANGE, range);
                });
            });
        };

        /**
         * Generates and applies an 'autoFill' operation that will copy the
         * values and formatting from a range to the surrounding cells.
         *
         * @param {Range} range
         *  The address of a single cell range.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operation.
         */
        this.autoFill = function (range, border, count) {

            var // the address of the target range
                target = null;

            // passed count must be positive
            if (count <= 0) { return SheetUtils.makeRejected('operation'); }

            // calculate target address for the 'autoFill' operation
            switch (border) {
            case 'left':
                target = new Address(range.start[0] - count, range.end[1]);
                break;
            case 'right':
                target = new Address(range.end[0] + count, range.end[1]);
                break;
            case 'top':
                target = new Address(range.end[0], range.start[1] - count);
                break;
            case 'bottom':
                target = new Address(range.end[0], range.end[1] + count);
                break;
            }

            // check validity of the target address
            if (!target || !docModel.isValidAddress(target)) {
                return SheetUtils.makeRejected('operation');
            }

            // generate and apply all operations
            return this.createAndApplyOperations(function (generator) {
                generator.generateRangeOperation(Operations.AUTO_FILL, range, { target: target.toJSON() });
            });
        };

        /**
         * Generates and applies 'mergeCells' operations for all passed ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address.
         *
         * @param {String} type
         *  The merge type. Must be one of:
         *  - 'merge': Merges the entire ranges.
         *  - 'horizontal': Merges the single rows in all ranges.
         *  - 'vertical': Merges the single columns in all ranges.
         *  - 'unmerge': Removes all merged ranges covered by the ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'merge:overlap': The passed cell ranges overlap each other.
         *  - 'merge:overflow': Tried to merge too many ranges at once.
         *  - 'operation': Internal error while applying the operation.
         */
        this.mergeRanges = function (ranges, type) {

            // convert ranges to unique array, nothing to do without ranges
            ranges = RangeArray.get(ranges).unify();
            if (ranges.empty()) { return $.when(); }

            // do not allow to merge overlapping ranges
            if ((type !== 'unmerge') && ranges.overlapsSelf()) {
                return SheetUtils.makeRejected('merge:overlap');
            }

            // generate and apply all operations
            return this.createAndApplyOperations(function (generator) {

                var // the total number of created merged ranges
                    rangeCount = 0;

                // generate the 'mergeCells' operations
                ranges.forEach(function (range) {

                    // count number of new merged ranges
                    switch (type) {
                    case 'merge':
                        rangeCount += 1;
                        break;
                    case 'horizontal':
                        // Bug 30662: reduce range to used area for entire columns
                        if (docModel.isColRange(range)) {
                            range.end[1] = Math.max(0, cellCollection.getUsedRows() - 1);
                        }
                        rangeCount += range.rows();
                        break;
                    case 'vertical':
                        // Bug 30662: reduce range to used area for entire rows
                        if (docModel.isRowRange(range)) {
                            range.end[0] = Math.max(0, cellCollection.getUsedCols() - 1);
                        }
                        rangeCount += range.cols();
                        break;
                    }

                    // generate a single 'mergeCells' operation
                    generator.generateRangeOperation(Operations.MERGE_CELLS, range, { type: type });
                });

                // overflow check
                if (rangeCount > SheetUtils.MAX_MERGED_RANGES_COUNT) {
                    return SheetUtils.makeRejected('merge:overflow');
                }
            });
        };

        // table/filter operations --------------------------------------------

        /**
         * Generates and applies a single 'insertTable' operation to insert a
         * new table range into this sheet.
         *
         * @param {String} tableName
         *  The name for the new table to be inserted. The empty string
         *  addresses the anonymous table range used to store filter settings
         *  for the standard auto filter of the sheet.
         *
         * @param {Range} range
         *  The address of the cell range covered by the new table.
         *
         * @param {Object} [attributes]
         *  The initial attribute set for the table.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'table:duplicate': The sheet already contains a table with the
         *      passed name.
         *  - 'table:overlap': The passed cell range covers an existing table.
         *  - 'operation': Internal error while applying the operation.
         */
        this.insertTable = function (tableName, range, attributes) {

            // check existence of the table range (TODO: table names must be unique in entire document)
            if (tableCollection.hasTable(tableName)) { return SheetUtils.makeRejected('table:duplicate'); }

            // check that the range does not overlap with an existing table
            if (tableCollection.findTables(range).length > 0) { return SheetUtils.makeRejected('table:overlap'); }

            // generate an 'insertTable' operation, and more dependent operations
            return this.createAndApplyOperations(function (generator) {

                // 'real' tables must not contain merged cells (neither in header, footer, nor data rows)
                if (tableName.length > 0) {
                    generator.generateRangeOperation(Operations.MERGE_CELLS, range, { type: 'unmerge' });
                }

                // create the 'insertTable' operation
                var properties = range.toJSON();
                if (_.isObject(attributes)) { properties.attrs = attributes; }
                generator.generateTableOperation(Operations.INSERT_TABLE, tableName, properties);

                // bug 36152: auto filter will hide drop-down buttons in header cells covered by
                // merged ranges (but only if the cells are merged before creating the auto filter)
                // - the drop-down button in the LAST column of a merged range remains visible
                if (tableName.length === 0) {
                    mergeCollection.getMergedRanges(range.header()).forEach(function (mergedRange) {
                        var firstCol = Math.max(mergedRange.start[0], range.start[0]),
                            lastCol = Math.min(mergedRange.end[0], range.end[0]) - 1;
                        for (var col = firstCol; col <= lastCol; col += 1) {
                            generator.generateTableOperation(Operations.CHANGE_TABLE_COLUMN, tableName, { col: col - range.start[0], attrs: { filter: { hideMergedButton: true } } });
                        }
                    });
                }
            });
        };

        /**
         * Refreshes the specified table according to its current filter
         * settings, by generating operations for showing/hiding the affected
         * rows.
         *
         * @param {String} tableName
         *  The name of the table to be refreshed. The empty string addresses
         *  the anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'table:missing': The sheet does not contain a table with the
         *      passed name.
         *  - 'rows:overflow': The current filter settings of the table would
         *      modify (show or hide) too many rows at once.
         *  - 'operation': Internal error while applying the operation.
         */
        this.refreshTable = function (tableName) {
            // send operations without applying them (done manually during creation)
            return createAndSendTableOperations(tableName, generateRefreshTableOperations);
        };

        /**
         * Generates and applies a single 'changeTable' operation to modify the
         * specified table range in this sheet.
         *
         * @param {String} tableName
         *  The name of the table to be modified. The empty string addresses
         *  the anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @param {Object} attributes
         *  The new attributes to be set for this table.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'table:missing': The sheet does not contain a table with the
         *      passed name.
         *  - 'operation': Internal error while applying the operation.
         */
        this.changeTable = function (tableName, attributes) {

            // send operations without applying them (done manually during creation)
            return createAndSendTableOperations(tableName, function (generator, tableModel) {

                // generate a single 'changeTable' operation, and immediately modify the table
                generator.generateTableOperation(Operations.CHANGE_TABLE, tableName, { attrs: attributes });
                docModel.invokeOperationHandler(generator.getLastOperation());

                // refresh the table with current filter settings
                return generateRefreshTableOperations(generator, tableModel);
            });
        };

        /**
         * Generates and applies a single 'deleteTable' operation to delete the
         * specified table range from this sheet. Additionally, if the table
         * contains active filter rules, all operations will be generated that
         * make the affected hidden rows visible again.
         *
         * @param {String} tableName
         *  The name of the table to be deleted. The empty string addresses the
         *  anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'table:missing': The sheet does not contain a table with the
         *      passed name.
         *  - 'operation': Internal error while applying the operation.
         */
        this.deleteTable = function (tableName) {

            // send operations without applying them (done manually during creation)
            return createAndSendTableOperations(tableName, function (generator, tableModel) {

                // silently deactivate the filters in the table, and refresh the
                // table (this makes all filtered rows visible)
                tableModel.setAttributes({ table: { filtered: false } }, { notify: 'never' });
                var promise = generateRefreshTableOperations(generator, tableModel);

                // ignore 'rows:overflow' error (leave rows hidden, but delete the table anyways)
                promise = promise.then(null, function (result) {
                    return (Utils.getStringOption(result, 'cause') === 'rows:overflow') ? $.when() : result;
                });

                // add the 'deleteTable' operation, and immediately delete the table
                promise.done(function () {
                    generator.generateTableOperation(Operations.DELETE_TABLE, tableName);
                    docModel.invokeOperationHandler(generator.getLastOperation());
                });

                return promise;
            });
        };

        /**
         * Generates and applies a 'changeTableColumn' operation with the
         * passed filtering and sorting attributes, and executes the new filter
         * settings by generating operations for showing/hiding the affected
         * rows.
         *
         * @param {String} tableName
         *  The name of the table to be modified. The empty string addresses
         *  the anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @param {Number} tableCol
         *  The zero-based index of the table column to be modified, relative
         *  to the cell range covered by the table.
         *
         * @param {Object} attributes
         *  The attributes to be set for the specified table column.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'table:missing': The sheet does not contain a table with the
         *      passed name.
         *  - 'rows:overflow': The passed filter settings would modify (show or
         *      hide) too many rows at once.
         *  - 'operation': Internal error while applying the operation.
         */
        this.changeTableColumn = function (tableName, tableCol, attributes) {

            // send operations without applying them (done manually during creation)
            return createAndSendTableOperations(tableName, function (generator, tableModel) {

                // store the new table column attributes
                generator.generateTableOperation(Operations.CHANGE_TABLE_COLUMN, tableName, { col: tableCol, attrs: attributes });
                docModel.invokeOperationHandler(generator.getLastOperation());

                // always refresh the filter (also if no attributes have been changed
                // actually, the cell contents may have changed in the meantime)
                return generateRefreshTableOperations(generator, tableModel);
            });
        };

        // drawing operations -------------------------------------------------

        /**
         * Generates and applies 'insertDrawing' operations that will insert
         * new drawing objects into this sheet.
         *
         * @param {Array<Number>} position
         *  The position of the first new drawing object. All drawing objects
         *  will be inserted subsequently.
         *
         * @param {Array<Object>} drawingData
         *  The type and formatting attributes of the drawing objects to be
         *  inserted. Each array element MUST be an object with the following
         *  properties:
         *  @param {String} element.type
         *      The type identifier of the drawing object.
         *  @param {Object} element.attrs
         *      Initial attributes for the drawing object, especially the
         *      anchor attributes specifying the physical position of the
         *      drawing object in the sheet.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked everytime after an
         *  'insertDrawing' operation has been generated for a drawing object.
         *  Allows to create additional operations for the drawing objects,
         *  before all these operations will be applied at once. Receives the
         *  following parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation of the current drawing).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array<Number>} position
         *      The effective document position of the current drawing object
         *      in the sheet, as inserted into the 'insertDrawing' operation.
         *  (4) {Object} data
         *      The data element from the array parameter 'drawingData' that is
         *      currently processed.
         *  The callback function may return a promise to defer processing of
         *  subsequent drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operations.
         */
        this.insertDrawings = function (position, drawingData, callback) {

            var // the own sheet index
                sheet = this.getIndex();

            // create all 'insertDrawing' operations, invoke callback function for more operations
            return this.createAndApplyOperations(function (generator) {
                return this.iterateArraySliced(drawingData, function (data) {

                    // add missing generic anchor attributes
                    var attributes = drawingCollection.addGenericAnchorAttributes(data.attrs);

                    // create the 'insertDrawing' operation at the current position
                    generator.generateDrawingOperation(Operations.INSERT_DRAWING, position, { type: data.type, attrs: attributes });

                    // invoke callback function for more operations (may return a promise)
                    var result = _.isFunction(callback) ? callback.call(this, generator, sheet, position.slice(), data) : null;

                    // adjust document position for the next drawing object
                    position = position.slice();
                    position[position.length - 1] += 1;

                    // callback may want to defer processing the following drawings
                    return result;
                });
            });
        };

        /**
         * Generates and applies one or more 'deleteDrawing' operations that
         * will delete the specified drawing objects from this sheet.
         *
         * @param {Array<Array<Number>>} positions
         *  An array with positions of the drawing objects to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operations.
         */
        this.deleteDrawings = function (positions) {

            // sort by position, remove positions of embedded drawings whose parents will be deleted too
            DrawingUtils.optimizeDrawingPositions(positions);

            // create the 'deleteDrawing' operations, visit the positions backwards to prevent
            // changed positions of following drawings after deleting a previous drawing
            return this.createAndApplyOperations(function (generator) {
                Utils.iterateArray(positions, function (position) {
                    generator.generateDrawingOperation(Operations.DELETE_DRAWING, position);
                }, { reverse: true });
            });
        };

        /**
         * Generates and applies one or more 'setDrawingAttributes' operations
         * that will change the formatting attributes of the specified drawing
         * objects.
         *
         * @param {Array<Array<Number>>} positions
         *  An array with positions of the drawing objects to be modified.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set to be applied at the drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operations.
         */
        this.setDrawingAttributes = function (positions, attributes) {
            return this.createAndApplyOperations(function (generator) {
                positions.forEach(function (position) {
                    generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: attributes });
                });
            });
        };

        // initialization -----------------------------------------------------

        // create the collection instances (constructors want to access public methods of this sheet model)
        nameCollection = new NameCollection(this);
        colCollection = new ColRowCollection(this, true);
        rowCollection = new ColRowCollection(this, false);
        mergeCollection = new MergeCollection(this);
        tableCollection = new TableCollection(this);
        validationCollection = new ValidationCollection(this);
        condFormatCollection = new CondFormatCollection(this);
        cellCollection = new CellCollection(this);
        drawingCollection = new SheetDrawingCollection(this);

        // update frozen split size and position, after inserting/deleting rows or columns
        this.registerTransformationHandler(transformFrozenSplit);

        // invoke all registered transformation handlers, after inserting/deleting
        // columns or rows, and forward all column/row collection events
        colCollection.on({
            'insert:entries': function (event, interval) { invokeTransformationHandlers('insert:columns', interval, true, true); },
            'delete:entries': function (event, interval) { invokeTransformationHandlers('delete:columns', interval, false, true); },
            'change:entries': function () { self.trigger.apply(self, ['change:columns'].concat(_.toArray(arguments).slice(1))); }
        });
        rowCollection.on({
            'insert:entries': function (event, interval) { invokeTransformationHandlers('insert:rows', interval, true, false); },
            'delete:entries': function (event, interval) { invokeTransformationHandlers('delete:rows', interval, false, false); },
            'change:entries': function () { self.trigger.apply(self, ['change:rows'].concat(_.toArray(arguments).slice(1))); }
        });

        // update cell selection after inserting or deleting merged ranges, and forward the event
        mergeCollection.on('triggered', function () {
            var selection = self.getViewAttribute('selection').clone();
            selection.ranges = getExpandedRanges(selection.ranges);
            self.setViewAttribute('selection', selection);
            self.trigger.apply(self, _.toArray(arguments).slice(1));
        });

        // forward events of the table collection
        tableCollection.on('triggered', function () { self.trigger.apply(self, _.toArray(arguments).slice(1)); });

        // forward events of the cell collection
        cellCollection.on('triggered', function () { self.trigger.apply(self, _.toArray(arguments).slice(1)); });

        // update drawing selection after inserting or deleting drawing objects, forward events of the collection
        drawingCollection.on({ 'insert:drawing': insertDrawingHandler, 'delete:drawing': deleteDrawingHandler });
        drawingCollection.on('triggered', function () { self.trigger.apply(self, _.toArray(arguments).slice(1)); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            drawingCollection.destroy();
            cellCollection.destroy();
            condFormatCollection.destroy();
            validationCollection.destroy();
            tableCollection.destroy();
            mergeCollection.destroy();
            colCollection.destroy();
            rowCollection.destroy();
            nameCollection.destroy();
            self = docModel = cellStyles = null;
            nameCollection = colCollection = rowCollection = null;
            mergeCollection = tableCollection = validationCollection = null;
            condFormatCollection = cellCollection = drawingCollection = null;
            transformationHandlers = null;
        });

    } // class SheetModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: SheetModel });

});
