/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokens', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/cellref'
], function (Utils, SheetUtils, FormulaUtils, CellRef) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Range = SheetUtils.Range,
        Range3D = SheetUtils.Range3D;

    // private global functions ===============================================

    /**
     * Returns whether the passed sheet reference may be affected by a renamed
     * sheet (always true for relative sheet references).
     */
    function isBoundSheet(sheetRef, sheet) {
        return _.isObject(sheetRef) && _.isNumber(sheetRef.sheet) && (!sheetRef.abs || (sheetRef.sheet === sheet));
    }

    // class BaseToken ========================================================

    /**
     * Base class for all formula tokens used in the tokenizer.
     *
     * @constructor
     *
     * @param {String} type
     *  The token type identifier.
     *
     * @param {Function} generator
     *  A callback function that implements converting the current value of
     *  this token to its text representation according to a specific formula
     *  grammar. Receives all parameters passed to the public method
     *  BaseToken.getText(), and must return a string. Will be invoked in the
     *  context of this token instance.
     *
     * @param {Function} debugGenerator
     *  A callback function that implements converting the current value of
     *  this token to its text representation for debug logging. Does not
     *  receive any parameter, and must return a string
     */
    var BaseToken = _.makeExtendable(function (type, generator, debugGenerator) {

        // public methods -----------------------------------------------------

        /**
         * Returns the type identifier of this token.
         *
         * @returns {String}
         *  The type identifier of this token.
         */
        this.getType = function () {
            return type;
        };

        /**
         * Returns whether the type of this token matches the passed type.
         *
         * @param {String} test
         *  A type identifier that must match the type of this token exactly.
         *
         * @returns {Boolean}
         *  Whether the type of this token matches the passed type.
         */
        this.isType = function (test) {
            return test === type;
        };

        /**
         * Returns whether the type identifier of this token matches the passed
         * regular expression.
         *
         * @param {RegExp} regExp
         *  A regular expression that will be tested against the type of this
         *  token.
         *
         * @returns {Boolean}
         *  Whether the type of this token matches the regular expression.
         */
        this.matchesType = function (regExp) {
            return regExp.test(type);
        };

        /**
         * Generates the text representation of this token, according to the
         * passed grammar configuration.
         *
         * @param {GrammarConfig} config
         *  The configuration object containing grammar-specific settings for
         *  formula operators.
         *
         * @param {Object} [options]
         *  Optional parameters for relocation of range addresses. See method
         *  ReferenceToken.getRange3D() for details.
         *
         * @returns {String}
         *  The text representation of this token.
         */
        this.getText = generator;

        /**
         * Returns a text description of the token for debugging (!) purposes.
         */
        this.toString = function () {
            return type + '[' + debugGenerator() + ']';
        };

    }); // class BaseToken

    // class FixedToken =======================================================

    /**
     * A formula token of an arbitrary type with a fixed string representation
     * independent from a formula grammar.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {String} type
     *  The token type identifier.
     *
     * @param {String} text
     *  The fixed text representation of the token.
     */
    var FixedToken = BaseToken.extend({ constructor: function (type, text) {

        // base constructor ---------------------------------------------------

        BaseToken.call(this, type, generator, generator);

        // private methods ----------------------------------------------------

        /**
         * Generates the text representation of this token.
         */
        function generator() {
            // do not use _.constant() here, as the text may change during runtime
            return text;
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the current text of this token.
         *
         * @returns {String}
         *  The current text of this token.
         */
        this.getValue = function () {
            return text;
        };

        /**
         * Replaces the text of this token.
         *
         * @param {String} newText
         *  The new text to be inserted into this token.
         *
         * @returns {Boolean}
         *  Whether the text has actually changed.
         */
        this.setValue = function (newText) {
            if (text === newText) { return false; }
            text = newText;
            return true;
        };

        /**
         * Appends more text to this token.
         *
         * @param {String} newText
         *  The new text to be appended to this token.
         *
         * @returns {Boolean}
         *  Whether the token has actually changed (the passed text was not
         *  empty).
         */
        this.appendValue = function (newText) {
            return this.setValue(text + newText);
        };

    }}); // class FixedToken

    // class OperatorToken ====================================================

    /**
     * A formula token that represents a unary or binary operator in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {String} value
     *  The grammar-independent operator identifier.
     */
    var OperatorToken = BaseToken.extend({ constructor: function (value) {

        // base constructor ---------------------------------------------------

        // the intersection operator has a variable text representation, all other operators are constant
        BaseToken.call(this, 'op', function (config) { return config.OPERATORS[value]; }, _.constant(value));

        // public methods -----------------------------------------------------

        /**
         * Returns the grammar-independent identifier of the operator.
         *
         * @returns {String}
         *  The grammar-independent identifier of the operator.
         */
        this.getValue = _.constant(value);

    }}); // class OperatorToken

    // class SeparatorToken ===================================================

    /**
     * A formula token that represents the range list operator, or a separator
     * in a function parameter list.
     *
     * @constructor
     *
     * @extends BaseToken
     */
    var SeparatorToken = BaseToken.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        BaseToken.call(this, 'sep', _.property('SEP'), _.constant(','));

    }}); // class SeparatorToken

    // class ParenthesisToken =================================================

    /**
     * A formula token that represents an opening or a closing parenthesis.
     *
     * @constructor
     *
     * @extends FixedToken
     *
     * @param {Boolean} open
     *  Whether this token represents an opening parenthesis (true), or a
     *  closing parenthesis (false).
     */
    var ParenthesisToken = FixedToken.extend({ constructor: function (open) {

        // base constructor ---------------------------------------------------

        FixedToken.call(this, open ? 'open' : 'close', open ? '(' : ')');

    }}); // class ParenthesisToken

    // class MatrixDelimiterToken =============================================

    /**
     * A formula token that represents a delimiter in a matrix literal: an
     * opening or a closing parenthesis, a row separator, or a column
     * separator.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Boolean} type
     *  The type of the matrix delimiter. Must be one of 'mat_open',
     *  'mat_close', 'mat_row', or 'mat_col'.
     */
    var MatrixDelimiterToken = FixedToken.extend({ constructor: function (type) {

        // base constructor ---------------------------------------------------

        BaseToken.call(this, type, generator, _.constant({ mat_open: '{', mat_close: '}', mat_row: '|', mat_col: ';' }[type]));

        // private methods ----------------------------------------------------

        /**
         * Generates the text representation of this matrix delimiter token.
         */
        function generator(config) {
            switch (type) {
            case 'mat_open': return '{';
            case 'mat_close': return '}';
            case 'mat_row': return config.MAT_ROW;
            case 'mat_col': return config.MAT_COL;
            }
            throw new Error('invalid matrix delimiter type');
        }

    }}); // class MatrixDelimiterToken

    // class LiteralToken =====================================================

    /**
     * A formula token that represents a literal value (a number, a string, a
     * boolean value, or an error code) in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The value for this literal token. The special value null is reserved
     *  for empty parameter in a function call, e.g. in the formula =SUM(1,,2).
     */
    var LiteralToken = BaseToken.extend({ constructor: function (value) {

        // base constructor ---------------------------------------------------

        BaseToken.call(this, 'lit', generator, debugGenerator);

        // private methods ----------------------------------------------------

        /**
         * Generates the text representation of this literal token.
         */
        function generator(config) {
            return config.generateLiteralText(value);
        }

        /**
         * Generates the text representation for debug logging.
         */
        function debugGenerator() {
            return FormulaUtils.valueToString(value);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the current value of this literal token.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  The current value of this literal token.
         */
        this.getValue = function () {
            return value;
        };

        /**
         * Changes the current value of this token.
         *
         * @param {Number|String|Boolean|ErrorCode|Null} newValue
         *  The new value for this token.
         *
         * @returns {Boolean}
         *  Whether the value of the token has actually been changed.
         */
        this.setValue = function (newValue) {
            if (value === newValue) { return false; }
            value = newValue;
            return true;
        };

    }}); // class LiteralToken

    // class MatrixToken ======================================================

    /**
     * A formula token that represents a constant matrix literal in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {Matrix} matrix
     *  The matrix literal represented by the token.
     */
    var MatrixToken = BaseToken.extend({ constructor: function (matrix) {

        // base constructor ---------------------------------------------------

        BaseToken.call(this, 'mat', generator, function () { return matrix.toString(); });

        // private methods ----------------------------------------------------

        /**
         * Generates the text representation of the matrix.
         */
        function generator(config) {
            return '{' + matrix.array.map(function (elems) {
                return elems.map(config.generateLiteralText.bind(config)).join(config.MAT_COL);
            }).join(config.MAT_ROW) + '}';
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the matrix literal contained in this token.
         *
         * @returns {Matrix}
         *  The matrix literal of this token.
         */
        this.getMatrix = function () {
            return matrix;
        };

    }}); // class MatrixToken

    // class ReferenceToken ===================================================

    /**
     * A formula token that represents a cell range reference in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this token instance.
     *
     * @param {CellRef|Null} cell1Ref
     *  The first cell reference structure (e.g. the cell address 'A1' in the
     *  formula A1:B2). If null or omitted, the token will represent a
     *  reference error (e.g. Sheet1!#REF!, after deleting the referenced
     *  column or row).
     *
     * @param {CellRef|Null} cell2Ref
     *  The second cell reference structure (e.g. the cell address 'B2' in the
     *  formula A1:B2). If null or omitted, the token will represent a single
     *  cell (e.g. Sheet1!A1).
     *
     * @param {SheetRef|Null} sheet1Ref
     *  The first sheet reference structure (e.g. 'Sheet1' in the formula
     *  Sheet1:Sheet2!A1). If null or undefined, the token represents a local
     *  cell range reference into the passed sheet (e.g. A1:B2).
     *
     * @param {SheetRef|Null} sheet2Ref
     *  The second sheet reference structure (e.g. 'Sheet2' in the formula
     *  Sheet1:Sheet2!A1). If null or undefined, the token represents a single
     *  sheet reference (e.g. Sheet1!A1).
     */
    var ReferenceToken = BaseToken.extend({ constructor: function (docModel, cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {

        // base constructor ---------------------------------------------------

        BaseToken.call(this, 'ref', generator, debugGenerator);

        // private methods ----------------------------------------------------

        /**
         * Returns the sheet range referred by this token.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.refSheet]
         *      The index of the reference sheet used to resolve references
         *      without sheets. If omitted, and this token does not contain its
         *      own sheet reference, this method will return null.
         *
         * @returns {Object|Null}
         *  The range of sheets referred by this token, in the properties
         *  'sheet1' and 'sheet2'. If the token does not contain a valid sheet
         *  range (according to passed reference sheet), null will be returned.
         */
        function resolveSheetRange(options) {
            var sheet1 = sheet1Ref ? sheet1Ref.sheet : Utils.getIntegerOption(options, 'refSheet', null),
                sheet2 = sheet2Ref ? sheet2Ref.sheet : sheet1;
            return (_.isNumber(sheet1) && (sheet1 >= 0) && _.isNumber(sheet2) && (sheet2 >= 0)) ?
                { sheet1: Math.min(sheet1, sheet2), sheet2: Math.max(sheet1, sheet2) } : null;
        }

        /**
         * Returns copies of the own cell reference objects that have been
         * relocated according to the passed settings.
         */
        function resolveCellRefs(options) {

            var // reference and target address for relocation
                refAddress = Utils.getOption(options, 'refAddress', Address.A1),
                targetAddress = Utils.getOption(options, 'targetAddress', Address.A1);

            // performance: return original cell references, if nothing to relocate
            if (refAddress.equals(targetAddress)) { return { cell1Ref: cell1Ref, cell2Ref: cell2Ref }; }

            var // the result object returned by this method: start with clones of the cell references
                result = {
                    cell1Ref: cell1Ref ? cell1Ref.clone() : null,
                    cell2Ref: cell2Ref ? cell2Ref.clone() : null
                },
                // whether to wrap the references at sheet borders
                wrapReferences = Utils.getBooleanOption(options, 'wrapReferences', false);

            // if relocating cell1Ref is not successful, remove it from the result
            if (result.cell1Ref && !result.cell1Ref.relocate(docModel, refAddress, targetAddress, wrapReferences)) {
                result.cell1Ref = null;
            }

            // if relocating cell2Ref is not successful, remove both cell references from the result
            if (result.cell1Ref && result.cell2Ref && !result.cell2Ref.relocate(docModel, refAddress, targetAddress, wrapReferences)) {
                result.cell1Ref = result.cell2Ref = null;
            }

            return result;
        }

        /**
         * Generates the text representation of the cell range reference.
         */
        function generator(config, options) {
            var result = resolveCellRefs(options);
            return config.generateReference(result.cell1Ref, result.cell2Ref, sheet1Ref, sheet2Ref);
        }

        /**
         * Returns a text description of the token for debug logging.
         */
        function debugGenerator() {
            var result = '';
            result += sheet1Ref ? sheet1Ref.toString() : '';
            result += (sheet1Ref && sheet2Ref) ? (':' + sheet2Ref.toString()) : '';
            result += sheet1Ref ? '!' : '';
            result += cell1Ref ? cell1Ref.refText() : '#REF!';
            result += (cell1Ref && cell2Ref) ? (':' + cell2Ref.refText()) : '';
            return result;
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether the sheet reference of this token is invalid (it
         * exists AND it points to a non-existing sheet).
         *
         * @returns {Boolean}
         *  Whether the sheet reference of this token is invalid.
         */
        this.hasSheetError = function () {
            return (_.isObject(sheet1Ref) && !sheet1Ref.valid()) || (_.isObject(sheet2Ref) && !sheet2Ref.valid());
        };

        /**
         * Returns the address of the cell range represented by this token,
         * including sheet indexes, or null if the range address is invalid.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.refSheet]
         *      The index of the reference sheet used to resolve missing sheet
         *      references. If omitted, and this token does not contain its own
         *      sheet reference, this method will return null.
         *  @param {Address} [options.refAddress]
         *      The address of the original reference cell this token is
         *      related to. If omitted, cell A1 will be used instead. If this
         *      address is different to the address resulting from the option
         *      'targetAddress', the returned range will be relocated according
         *      to these two addresses.
         *  @param {Address} [options.targetAddress]
         *      The address of the target cell for relocation. If omitted, cell
         *      A1 will be used instead. If this address is different to the
         *      address resulting from the option 'refAddress', the returned
         *      range will be relocated according to these two addresses.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, and if the relocated range is located outside
         *      the sheet, the range will be wrapped at the sheet borders.
         *
         * @returns {Range3D|Null}
         *  The address of the cell range referenced by this reference token;
         *  or null, if the token does not refer to a valid range.
         */
        this.getRange3D = function (options) {

            // resolve sheet references, return null on error
            var sheetRange = resolveSheetRange(options);
            if (!sheetRange) { return null; }

            // return immediately, if cell1Ref cannot be relocated
            var result = resolveCellRefs(options);
            if (!result.cell1Ref) { return null; }

            // create an intermediate range with ordered column/row indexes
            var range = Range.createFromAddresses(result.cell1Ref.address, result.cell2Ref && result.cell2Ref.address);
            return Range3D.createFromRange(range, sheetRange.sheet1, sheetRange.sheet2);
        };

        /**
         * Changes the current cell range of this token, keeps all absolute
         * flags and sheet references intact.
         *
         * @returns {Boolean}
         *  Whether the token has been changed.
         */
        this.setRange = function (range) {

            // whether the token changes (either start or end address)
            var changed = false;

            // put a clone of the start address into cell1Ref
            if (cell1Ref && cell1Ref.address.differs(range.start)) {
                cell1Ref.address = range.start.clone();
                changed = true;
            } else if (!cell1Ref) {
                cell1Ref = new CellRef(range.start.clone(), true, true);
                cell2Ref = null;
                changed = true;
            }

            // create missing cell2Ref (unless the passed range is a single cell),
            // start as a clone of cell1Ref to get the correct absolute flags
            if (!cell2Ref && !range.single()) {
                cell2Ref = cell1Ref.clone();
                changed = true;
            }

            // put a clone of the end address into cell2Ref
            if (cell2Ref && cell2Ref.address.differs(range.end)) {
                cell2Ref.address = range.end.clone();
                // reset cell2Ref if address and absolute flags are equal to cell1Ref
                if (cell1Ref.equals(cell2Ref)) { cell2Ref = null; }
                changed = true;
            }

            return changed;
        };

        /**
         * Transforms this reference token after columns or rows have been
         * inserted into or deleted from a sheet in the document.
         *
         * @returns {Boolean}
         *  Whether the token has been changed.
         */
        this.transformRange = function (sheet, interval, insert, columns, refSheet) {

            // the original range, with sheet indexes
            var oldRange = this.getRange3D({ refSheet: refSheet });

            // nothing to do, if this token represents a reference error,
            // or points to another sheet, or sheet reference is invalid
            if (!oldRange || !oldRange.isSheet(sheet)) { return false; }

            // the transformed range address (transformRange() returns a 2D range)
            var newRange = docModel.transformRange(oldRange, interval, insert, columns);

            // nothing to do, if the range does not change (compare only the columns/rows, not the sheets)
            if (newRange && newRange.equals(oldRange)) { return false; }

            // check that the range has been transformed successfully
            if (newRange) {
                this.setRange(newRange);
            } else {
                cell1Ref = cell2Ref = null;
            }
            return true;
        };

        /**
         * Relocates this reference token to a new cell position.
         *
         * @returns {Boolean}
         *  Whether the token has been changed.
         */
        this.relocateRange = function (refAddress, targetAddress, wrapReferences) {

            // do not touch tokens representing a reference error
            if (!cell1Ref) { return false; }

            // relocate first cell reference, bail out immediately on error
            if (!cell1Ref.relocate(docModel, refAddress, targetAddress, wrapReferences)) {
                cell1Ref = cell2Ref = null;
                return true;
            }

            // relocate second cell reference if existing
            if (cell2Ref && !cell2Ref.relocate(docModel, refAddress, targetAddress, wrapReferences)) {
                cell1Ref = cell2Ref = null;
            }
            return true;
        };

        /**
         * Refreshes the sheet references after a sheet has been inserted,
         * deleted, or moved in the document.
         *
         * @returns {Boolean}
         *  Whether the token has been changed.
         */
        this.transformSheet = function (toSheet, fromSheet) {
            var changed1 = _.isObject(sheet1Ref) && sheet1Ref.transform(toSheet, fromSheet),
                changed2 = _.isObject(sheet2Ref) && sheet2Ref.transform(toSheet, fromSheet);
            return changed1 || changed2;
        };

        /**
         * Changes all sheet references containing the specified old sheet
         * index to the new sheet index. Used for example when copying existing
         * sheets, and adjusting all references and defined names that point to
         * the original sheet.
         *
         * @param {Number} oldSheet
         *  The zero-based index of the original sheet.
         *
         * @param {Number} newSheet
         *  The zero-based index of the target sheet.
         *
         * @returns {Boolean}
         *  Whether the token has been changed.
         */
        this.relocateSheet = function (oldSheet, newSheet) {
            var changed1 = _.isObject(sheet1Ref) && sheet1Ref.relocate(oldSheet, newSheet),
                changed2 = _.isObject(sheet2Ref) && sheet2Ref.relocate(oldSheet, newSheet);
            return changed1 || changed2;
        };

        /**
         * Refreshes the text representation after a sheet has been renamed
         * in the document.
         *
         * @param {Number} sheet
         *  The zero-based index of the renamed sheet.
         *
         * @returns {Boolean}
         *  Whether the token has been changed.
         */
        this.renameSheet = function (sheet) {
            // the token does not contain the sheet name anymore (no need to modify something),
            // but still return true if the token is actually affected (for event handlers)
            return isBoundSheet(sheet1Ref, sheet) || isBoundSheet(sheet2Ref, sheet);
        };

    }}); // class ReferenceToken

    // class NameToken ========================================================

    /**
     * A formula token that represents a defined name in a formula.
     *
     * @constructor
     *
     * @extends BaseToken
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this token instance.
     *
     * @param {String} value
     *  The actual name of the defined name, the native (English) name of a
     *  built-in sheet function, or the name of a macro function.
     *
     * @param {SheetRef|Null} sheetRef
     *  The sheet reference structure (e.g. the particle 'Sheet1' in the
     *  formula Sheet1!my_name, or 'Module1' in the formula Module1!myFunc() of
     *  a macro call). If null or undefined, the token represents a local name
     *  in the passed sheet, a global name, a built-in function, or a globally
     *  available macro function.
     *
     * @param {Boolean} isFunc
     *  Whether the token represents a function call (true), or a defined name
     *  (false) without following opening parenthesis.
     */
    var NameToken = BaseToken.extend({ constructor: function (docModel, value, sheetRef, isFunc) {

        // base constructor ---------------------------------------------------

        BaseToken.call(this, isFunc ? 'func' : 'name', generator, debugGenerator);

        // private methods ----------------------------------------------------

        /**
         * Generates the text representation of the name.
         */
        function generator(config) {
            return config.generateSheetName(sheetRef) + ((isFunc && !sheetRef) ? config.getFunctionText(value) : value);
        }

        /**
         * Returns a text description of the token for debug logging.
         */
        function debugGenerator() {
            return (sheetRef ? (sheetRef.toString() + '!') : '') + value;
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the native name of the defined name or function, as passed
         * to the constructor.
         *
         * @returns {String}
         *  The native name of the defined name or function.
         */
        this.getValue = function () {
            return value;
        };

        /**
         * Returns whether the token contains a sheet reference structure,
         * regardless if it is valid.
         *
         * @returns {Boolean}
         *  Whether the token contains a sheet reference structure.
         */
        this.hasSheetRef = function () {
            return _.isObject(sheetRef);
        };

        /**
         * Returns whether the sheet reference of this token is invalid (it
         * exists but points to a non-existing sheet).
         *
         * @returns {Boolean}
         *  Whether the sheet reference of this token is invalid.
         */
        this.hasSheetError = function () {
            return _.isObject(sheetRef) && !sheetRef.valid();
        };

        /**
         * Returns the model of the defined name referred by this token.
         *
         * @param {Number|Null} refSheet
         *  The index of the reference sheet used to resolve sheet-local names
         *  without sheet reference. If set to null, and this token does not
         *  contain its own sheet reference, this method looks for globally
         *  defined names only.
         *
         * @returns {NameModel|Null}
         *  The model of the defined name, if existing; otherwise null. Returns
         *  always null, if this token refers to a function.
         */
        this.resolveNameModel = function (refSheet) {

            // do nothing for functions
            if (isFunc) { return null; }

            // returns the model of a defined name from the specified sheet
            function getNameFromSheet(sheet) {
                var sheetModel = _.isNumber(sheet) ? docModel.getSheetModel(sheet) : null;
                return sheetModel ? sheetModel.getNameCollection().getNameModel(value) : null;
            }

            // sheet reference exists (e.g.: Sheet2!name): resolve from specified sheet, do not try global names
            if (sheetRef) { return getNameFromSheet(sheetRef.sheet); }

            // no sheet reference: try sheet-local names first, then global names
            return getNameFromSheet(refSheet) || docModel.getNameCollection().getNameModel(value);
        };

        /**
         * Refreshes the sheet reference after a sheet has been inserted,
         * deleted, or moved in the document.
         *
         * @returns {Boolean}
         *  Whether the token has been changed.
         */
        this.transformSheet = function (toSheet, fromSheet) {
            return _.isObject(sheetRef) && sheetRef.transform(toSheet, fromSheet);
        };

        /**
         * Changes the sheet reference to the new sheet index, if it contains
         * the specified old sheet index. Used for example when copying
         * existing sheets, and adjusting all references and defined names that
         * point to the original sheet.
         *
         * @param {Number} oldSheet
         *  The zero-based index of the original sheet.
         *
         * @param {Number} newSheet
         *  The zero-based index of the target sheet.
         *
         * @returns {Boolean}
         *  Whether the token has been changed.
         */
        this.relocateSheet = function (oldSheet, newSheet) {
            return _.isObject(sheetRef) && sheetRef.relocate(oldSheet, newSheet);
        };

        /**
         * Refreshes the text representation after a sheet has been renamed in
         * the document.
         *
         * @param {Number} sheet
         *  The zero-based index of the renamed sheet.
         *
         * @returns {Boolean}
         *  Whether the token has been changed.
         */
        this.renameSheet = function (sheet) {
            // the token does not contain the sheet name anymore (no need to modify something),
            // but still return true if the token is actually affected (for event handlers)
            return isBoundSheet(sheetRef, sheet);
        };

    }}); // class NameToken

    // static class Tokens ====================================================

    var Tokens = {
        FixedToken: FixedToken,
        OperatorToken: OperatorToken,
        SeparatorToken: SeparatorToken,
        ParenthesisToken: ParenthesisToken,
        MatrixDelimiterToken: MatrixDelimiterToken,
        LiteralToken: LiteralToken,
        MatrixToken: MatrixToken,
        ReferenceToken: ReferenceToken,
        NameToken: NameToken
    };

    // exports ================================================================

    return Tokens;

});
