/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/sheetref', function () {

    'use strict';

    // class SheetRef =========================================================

    /**
     * A structure representing the sheet reference part in a range reference
     * (e.g. in Sheet1!A1:B2), or in a defined name (e.g. Sheet1!name).
     *
     * @constructor
     *
     * @property {Number|String} sheet
     *  The zero-based sheet index, if a sheet with the passed name exists in
     *  the document; or -1 for a broken sheet reference; otherwise the sheet
     *  name as string.
     *
     * @property {Boolean} abs
     *  Whether the sheet reference is marked to be absolute.
     */
    function SheetRef(sheet, abs) {

        this.sheet = sheet;
        this.abs = abs;

    } // class SheetRef

    // static methods ---------------------------------------------------------

    /**
     * Creates a sheet reference structure for the passed simple or complex
     * sheet name. At most one of the passed parameters can be a string, the
     * other parameters must be null or undefined.
     *
     * @param {SpreadsheetDocument} docModel
     *  The spreadsheet document used to look up the sheet indexes for the
     *  passed sheet names.
     *
     * @param {String|Null} simpleSheet
     *  A simple sheet name without any special characters that would cause to
     *  enclose the sheet name in apostrophes.
     *
     * @param {String|Null} complexSheet
     *  A complex sheet name with special characters that requires the sheet
     *  name to be enclosed in apostrophes.
     *
     * @param {String|Null} refError
     *  The error code of a reference error, used a sheet has been deleted,
     *  e.g. in the formula =#REF!!A1.
     *
     * @returns {SheetRef|Null}
     *  A sheet reference structure, if one of the passed parameters is a
     *  string, otherwise null.
     */
    SheetRef.create = function (docModel, simpleSheet, complexSheet, refError) {

        var sheetIndex = null;

        // try to find the passed simple sheet name in the document
        if (_.isString(simpleSheet)) {
            sheetIndex = docModel.getSheetIndex(simpleSheet);
            return new SheetRef((sheetIndex >= 0) ? sheetIndex : simpleSheet, true);
        }

        // try to find the passed complex sheet name in the document
        if (_.isString(complexSheet)) {
            complexSheet = complexSheet.replace(/''/g, '\'');
            sheetIndex = docModel.getSheetIndex(complexSheet);
            return new SheetRef((sheetIndex >= 0) ? sheetIndex : complexSheet, true);
        }

        // create a sheet reference object for the #REF! error code
        if (_.isString(refError)) {
            return new SheetRef(-1, true);
        }

        // no sheet names: sheet reference does not exist
        return null;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this sheet reference points to an existing sheet.
     *
     * @returns {Boolean}
     *  Whether this sheet reference points to an existing sheet.
     */
    SheetRef.prototype.valid = function () {
        return (typeof this.sheet === 'number') && (this.sheet >= 0);
    };

    /**
     * Returns whether this sheet reference and the passed sheet reference are
     * equal.
     *
     * @param {SheetRef} sheetRef
     *  The other sheet reference to be compared with this sheet reference.
     *
     * @returns {Boolean}
     *  Whether this sheet reference and the passed sheet reference are equal.
     */
    SheetRef.prototype.equals = function (sheetRef) {
        return (this.abs === sheetRef.abs) && (this.sheet === sheetRef.sheet);
    };

    /**
     * Transforms the sheet index in this sheet reference, after a sheet has
     * been inserted, deleted, or moved in the document.
     *
     * @param {Number|Null} toSheet
     *  The new index of the inserted or moved sheet; or null for deleted
     *  sheets.
     *
     * @param {Number|Null} fromSheet
     *  The old index of the deleted or moved sheet; or null for inserted
     *  sheets.
     *
     * @returns {Boolean}
     *  Whether the sheet index in this sheet reference has been modified.
     */
    SheetRef.prototype.transform = function (toSheet, fromSheet) {

        // invalid sheet name in reference, or nothing changes: return false
        if (!this.valid() || (toSheet === fromSheet)) {
            return false;
        }

        // sheet inserted
        if (typeof fromSheet !== 'number') {
            // referenced sheet follows inserted sheet
            if (toSheet <= this.sheet) { this.sheet += 1; return true; }

        // sheet deleted
        } else if (typeof toSheet !== 'number') {
            // referenced sheet follows deleted sheet
            if (fromSheet < this.sheet) { this.sheet -= 1; return true; }
            // referenced sheet has been deleted
            if (fromSheet === this.sheet) { this.sheet = -1; return true; }

        // sheet moved
        } else {
            // referenced sheet moved to another position
            if (fromSheet === this.sheet) { this.sheet = toSheet; return true; }
            // sheet moved backwards, referenced sheet follows
            if ((toSheet < fromSheet) && (toSheet <= this.sheet) && (this.sheet < fromSheet)) { this.sheet += 1; return true; }
            // sheet moved forwards, referenced sheet follows
            if ((fromSheet < toSheet) && (fromSheet < this.sheet) && (this.sheet <= toSheet)) { this.sheet -= 1; return true; }
        }

        return false;
    };

    /**
     * Changes the sheet index in this sheet reference to the new sheet index,
     * if it currently refers to the old sheet index.
     *
     * @param {Number} oldSheet
     *  The old index of the sheet to be relocated.
     *
     * @param {Number} newSheet
     *  The new index of the sheet to be relocated.
     *
     * @returns {Boolean}
     *  Whether the sheet index in this sheet reference has been modified.
     */
    SheetRef.prototype.relocate = function (oldSheet, newSheet) {

        // reference must be valid, or passed sheet indexes must differ, this reference must point to the old sheet
        if (this.valid() && (oldSheet !== newSheet) && (this.sheet === oldSheet)) {
            // TODO: also for absolute sheet references in ODF?
            this.sheet = newSheet;
            return true;
        }

        return false;
    };

    /**
     * Returns a text description of this cell reference for debug logging.
     */
    SheetRef.prototype.toString = function () {
        return (this.sheet === -1) ? '#REF!' : ((this.abs ? '$' : '') + this.sheet);
    };

    // exports ================================================================

    return SheetRef;

});
