/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/operators', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (SheetUtils, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all unary and binary operators used in
     * spreadsheet formulas.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of operator descriptor objects.
     *
     *************************************************************************/

    var // convenience shortcuts
        ErrorCodes = SheetUtils.ErrorCodes,
        Range3DArray = SheetUtils.Range3DArray;

    // private global functions ===============================================

    /**
     * Compares the passed literal values. The following rules apply for
     * comparison:
     * (1) Two null values are equal.
     * (2) A null value is converted to 0, if compared to a number.
     * (3) A null value is converted to '', if compared to a string.
     * (4) A null value is converted to FALSE, if compared to a Boolean.
     * (5) Numbers are always less than strings and Boolean values.
     * (6) Strings are always less than Boolean values.
     * (7) Two numbers are simply compared by value.
     * (8) Two strings are compared lexicographically and case-insensitive.
     * (9) FALSE is less than TRUE.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value1
     *  The first value to be compared.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value2
     *  The second value to be compared.
     *
     * @returns {Number}
     *  - A negative value, if value1 is less than value2.
     *  - A positive value, if value1 is greater than value2.
     *  - Zero, if both values are of the same type and are equal.
     */
    function compareValues(value1, value2) {

        // converts special value null (empty cell) according to the other value type
        function convertNull(value) {
            if (_.isNumber(value)) { return 0; }
            if (_.isString(value)) { return ''; }
            if (_.isBoolean(value)) { return false; }
            throw 'fatal';
        }

        // returns a numeric identifier according to the value type
        function getTypeId(value) {
            if (_.isNumber(value)) { return 1; }
            if (_.isString(value)) { return 2; }
            if (_.isBoolean(value)) { return 3; }
            throw 'fatal';
        }

        // two empty cells are equal
        if (_.isNull(value1) && _.isNull(value2)) { return 0; }

        // convert null values to typed values according to the other value
        if (_.isNull(value1)) { value1 = convertNull(value2); }
        if (_.isNull(value2)) { value2 = convertNull(value1); }

        // get type identifiers, compare different types
        var type1 = getTypeId(value1), type2 = getTypeId(value2);
        if (type1 !== type2) { return type1 - type2; }

        // compare values of equal types
        switch (type1) {
        case 1:
            return FormulaUtils.compareNumbers(value1, value2);
        case 2:
            return FormulaUtils.compareStrings(value1, value2); // case-insensitive
        case 3:
            return FormulaUtils.compareBooleans(value1, value2);
        }

        throw 'fatal';
    }

    // exports ================================================================

    return {

        // arithmetic operators -----------------------------------------------

        // binary addition operator, or unary plus operator
        add: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val val:num', // unary plus operator passes all values (no conversion to number)
            resolve: function (value1, value2) {
                return _.isUndefined(value2) ? value1 : (this.convertToNumber(value1) + value2);
            }
        },

        // binary subtraction operator, or unary minus operator
        sub: {
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num', // unary minus operator always converts to numbers
            resolve: function (value1, value2) {
                return _.isUndefined(value2) ? (-value1) : (value1 - value2);
            }
        },

        // binary multiplication operator
        mul: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: FormulaUtils.multiply
        },

        // binary division operator
        div: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: FormulaUtils.divide
        },

        // binary power operator
        pow: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:num val:num',
            resolve: { ooxml: FormulaUtils.power, odf: Math.pow }
        },

        // unary percent operator
        pct: {
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:num',
            resolve: function (value) { return value / 100; }
        },

        // string operators ---------------------------------------------------

        // binary string concatenation operator
        con: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:str',
            resolve: FormulaUtils.add
        },

        // comparison operators -----------------------------------------------

        // binary less-than operator
        lt: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) < 0;
            }
        },

        // binary less-than-or-equal operator
        le: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) <= 0;
            }
        },

        // binary greater-than operator
        gt: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) > 0;
            }
        },

        // binary greater-than-or-equal operator
        ge: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) >= 0;
            }
        },

        // binary equals operator
        eq: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) === 0;
            }
        },

        // binary equals-not operator
        ne: {
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val val',
            resolve: function (value1, value2) {
                return compareValues(value1, value2) !== 0;
            }
        },

        // reference operators ------------------------------------------------

        // binary list reference operator
        list: {
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref ref',
            resolve: function (ranges1, ranges2) {
                return ranges1.concat(ranges2);
            }
        },

        // binary intersection reference operator
        isect: {
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref:sheet ref:sheet',
            resolve: function (ranges1, ranges2) {

                // ranges in both operands must refer to the same sheet
                if (ranges1.first().sheet1 !== ranges2.first().sheet1) {
                    throw ErrorCodes.VALUE;
                }

                // calculate the intersection of each range pair from ranges1 and ranges2,
                // process all outer ranges separately to be able to check resulting list
                // size inside the loop, otherwise this loop may create a MAX^2 list
                var resultRanges = new Range3DArray();
                ranges1.forEach(function (range1) {
                    var ranges = ranges2.intersect(range1);
                    if (resultRanges.length + ranges.length > FormulaUtils.MAX_REF_LIST_SIZE) {
                        throw FormulaUtils.UNSUPPORTED_ERROR;
                    }
                    resultRanges.append(ranges);
                });
                return resultRanges;
            }
        },

        // binary range reference operator
        range: {
            minParams: 2,
            maxParams: 2,
            type: 'ref',
            signature: 'ref:sheet ref:sheet',
            resolve: function (ranges1, ranges2) {

                // ranges in both operands must refer to the same sheet
                if (ranges1.first().sheet1 !== ranges2.first().sheet1) {
                    throw ErrorCodes.VALUE;
                }

                // build the bounding ranges from all ranges
                return ranges1.concat(ranges2).boundary();
            }
        }
    };

});
