/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/formulautils', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, Logger, SheetUtils) {

    'use strict';

    var // shortcut for the map of error code literals
        ErrorCodes = SheetUtils.ErrorCodes;

    // static private functions ===============================================

    /**
     * Creates a special error code for internal usage in the formula engine.
     */
    function makeInternalErrorCode(code) {
        var errorCode = SheetUtils.makeErrorCode('[#' + code + ']');
        errorCode.internal = code;
        return errorCode;
    }

    // static class FormulaUtils ==============================================

    /**
     * The static class FormulaUtils provides low-level helper functions needed
     * to implement the formula engine of a spreadsheet application.
     * Additionally, the class is a console logger bound to the URL hash flag
     * 'spreadsheet:log-formulas', that logs detailed information about the
     * formula parser, compiler, and interpreter.
     */
    var FormulaUtils = {};

    // logger interface -------------------------------------------------------

    Logger.extend(FormulaUtils, { enable: 'spreadsheet:log-formulas', prefix: 'FMLA' });

    // constants --------------------------------------------------------------

    /**
     * Distance between the number 1 and the smallest number greater than 1
     * (the value of 2^-52). Provided as non-standard property Number.EPSILON
     * by Chrome and Firefox, but not supported by IE10 and IE11.
     *
     * @constant
     */
    FormulaUtils.EPSILON = Math.pow(2, -52);

    /**
     * The smallest positive normalized floating-point number (browsers may
     * support smaller numbers though, which are stored with a denormalized
     * mantissa).
     *
     * @constant
     */
    FormulaUtils.MIN_NUMBER = Math.pow(2, -1022);

    /**
     * Maximum number of function parameters (CalcEngine limitation).
     *
     * @constant
     */
    FormulaUtils.MAX_PARAM_COUNT = 254;

    /**
     * Maximum number of rows supported in matrix literals.
     *
     * @constant
     */
    FormulaUtils.MAX_MATRIX_ROW_COUNT = 16;

    /**
     * Maximum number of columns supported in matrix literals.
     *
     * @constant
     */
    FormulaUtils.MAX_MATRIX_COL_COUNT = 16;

    /**
     * Maximum number of references in a reference list literal.
     *
     * @constant
     */
    FormulaUtils.MAX_REF_LIST_SIZE = 1024;

    /**
     * Maximum number of cells in cell references visited in iterator loops.
     *
     * @constant
     */
    FormulaUtils.MAX_CELL_ITERATION_COUNT = 2000;

    /**
     * Special error code literal to transport the 'unsupported' state.
     */
    FormulaUtils.UNSUPPORTED_ERROR = makeInternalErrorCode('unsupported');

    /**
     * Special error code literal to transport the 'circular reference' result.
     */
    FormulaUtils.CIRCULAR_ERROR = makeInternalErrorCode('circular');

    // public methods ---------------------------------------------------------

    /**
     * Rounds the passed number to the next integer towards zero (positive
     * numbers will be rounded down, negative numbers will be rounded up).
     */
    FormulaUtils.trunc = function (number) {
        return (number < 0) ? Math.ceil(number) : Math.floor(number);
    };

    /**
     * Returns the sum of the passed numbers, or the concatenation of the
     * passed strings (used for callbacks).
     */
    FormulaUtils.add = function (number1, number2) {
        return number1 + number2;
    };

    /**
     * Returns the product of the passed numbers (used for callbacks).
     */
    FormulaUtils.multiply = function (number1, number2) {
        return number1 * number2;
    };

    /**
     * Returns the quotient of the passed numbers. If the divisor is zero, the
     * error code #DIV/0! will be thrown instead.
     */
    FormulaUtils.divide = function (number1, number2) {
        if (number2 === 0) { throw ErrorCodes.DIV0; }
        return number1 / number2;
    };

    /**
     * Returns the floating-point modulo of the passed numbers. If the divisor
     * is zero, the error code #DIV/0! will be thrown instead. If the passed
     * numbers have different signs, the result will be adjusted according to
     * the results of other spreadsheet applications.
     */
    FormulaUtils.modulo = function (number1, number2) {
        if (number2 === 0) { throw ErrorCodes.DIV0; }
        var result = number1 % number2;
        return ((result !== 0) && (number1 * number2 < 0)) ? (result + number2) : result;
    };

    /**
     * Returns the power of the passed numbers. If both numbers are zero, the
     * error code #NUM! will be thrown instead of returning 1.
     */
    FormulaUtils.power = function (number1, number2) {
        // Math.pow() returns 1 for 0^0; Excel returns #NUM! error
        if ((number1 === 0) && (number2 === 0)) { throw ErrorCodes.NUM; }
        if ((number1 === 0) && (number2 < 0)) { throw ErrorCodes.DIV0; }
        return Math.pow(number1, number2);
    };

    /**
     * Returns the two-parameter arc tangent of the passed coordinates. If both
     * numbers are zero, the error code #DIV/0! will be thrown instead. The
     * order of the parameters 'x' and 'y' is switched compared to the native
     * method Math.atan2, according to usage in other spreadsheet applications.
     */
    FormulaUtils.arctan2 = function (x, y) {
        if ((x === 0) && (y === 0)) { throw ErrorCodes.DIV0; }
        // Spreadsheet applications use (x,y); but Math.atan2() expects (y,x)
        return Math.atan2(y, x);
    };

    /**
     * Returns the hyperbolic sine of the passed number.
     */
    FormulaUtils.sinh = Math.sinh || function (number) {
        return (Math.exp(number) - Math.exp(-number)) / 2;
    };

    /**
     * Returns the hyperbolic cosine of the passed number.
     */
    FormulaUtils.cosh = Math.cosh || function (number) {
        return (Math.exp(number) + Math.exp(-number)) / 2;
    };

    /**
     * Returns the hyperbolic tangent of the passed number.
     */
    FormulaUtils.tanh = Math.tanh || function (number) {
        var pexp = Math.exp(number), nexp = Math.exp(-number);
        return (pexp - nexp) / (pexp + nexp);
    };

    /**
     * Returns the argument resulting in the passed hyperbolic sine.
     */
    FormulaUtils.asinh = Math.asinh || function (number) {
        return Math.log(number + Math.sqrt(number * number + 1));
    };

    /**
     * Returns the argument resulting in the passed hyperbolic cosine.
     */
    FormulaUtils.acosh = Math.acosh || function (number) {
        return Math.log(number + Math.sqrt(number * number - 1));
    };

    /**
     * Returns the argument resulting in the passed hyperbolic tangent.
     */
    FormulaUtils.atanh = Math.atanh || function (number) {
        return Math.log((1 + number) / (1 - number)) / 2;
    };

    /**
     * Returns the argument resulting in the passed hyperbolic cotangent.
     */
    FormulaUtils.acoth = Math.acoth || function (number) {
        return Math.log((number + 1) / (number - 1)) / 2;
    };

    // comparison -------------------------------------------------------------

    /**
     * Compares the passed numbers, and returns a signed indicator number.
     *
     * @param {Number} number1
     *  The first number for comparison.
     *
     * @param {Number} number2
     *  The second number for comparison.
     *
     * @returns {Number}
     *  The number -1, if number1 is less than number2; or the number 1, if
     *  number1 is greater than number2; or the number 0, if both numbers are
     *  equal. If the difference of the numbers is less than
     *  FormulaUtils.MIN_NUMBER (e.g., due to rounding errors), the numbers are
     *  also considered to be equal. If any of the numbers is NaN or infinite,
     *  the result will be NaN.
     */
    FormulaUtils.compareNumbers = function (number1, number2) {
        var diff = number1 - number2, abs = Math.abs(diff);
        return !isFinite(diff) ? Number.NaN : (abs < FormulaUtils.MIN_NUMBER) ? 0 : (diff / abs);
    };

    /**
     * Compares the passed strings lexicographically, and returns a signed
     * indicator number.
     *
     * @param {Boolean} string1
     *  The first string for comparison.
     *
     * @param {Boolean} string2
     *  The second string for comparison.
     *
     * @param {Boolean} [caseSens=false]
     *  If set to true, the strings will be compared case-sensitively. By
     *  default, the case of the strings will be ignored.
     *
     * @returns {Number}
     *  The number -1, if string1 is lexicographically less than string2; or
     *  the number 1, if string1 is lexicographically greater than string2;
     *  otherwise the number 0 (the strings are equal).
     */
    FormulaUtils.compareStrings = function (string1, string2, caseSens) {
        if (!caseSens) {
            string1 = string1.toUpperCase();
            string2 = string2.toUpperCase();
        }
        return (string1 < string2) ? -1 : (string1 > string2) ? 1 : 0;
    };

    /**
     * Compares the passed Boolean values, and returns a signed indicator
     * number.
     *
     * @param {Boolean} bool1
     *  The first Boolean value for comparison.
     *
     * @param {Boolean} bool2
     *  The second Boolean value for comparison.
     *
     * @returns {Number}
     *  The number -1, if bool1 is FALSE, and bool2 is TRUE (bool1 is 'less
     *  than' bool2); or the number 1, if bool1 is TRUE, and bool2 is FALSE
     *  (bool1 is 'greater than' bool2); otherwise the number 0 (the Boolean
     *  values are equal).
     */
    FormulaUtils.compareBooleans = function (bool1, bool2) {
        // subtraction operator converts Booleans to numbers implicitly
        return bool1 - bool2;
    };

    /**
     * Compares the internal indexes of built-in error codes, and returns a
     * signed indicator number.
     *
     * @param {ErrorCode} errorCode1
     *  The first error code for comparison.
     *
     * @param {ErrorCode} errorCode2
     *  The second error code for comparison.
     *
     * @returns {Number}
     *  The value NaN, if any of the passed error codes does not represent a
     *  built-in error (the property 'num' of the error code is not finite);
     *  the number -1, if the index of the first error code is less than the
     *  index of the second error code; the number 1, if the index of the first
     *  error code is greater than the index of the second error code, or the
     *  number 0, if both error codes have equal indexes.
     */
    FormulaUtils.compareErrorCodes = function (errorCode1, errorCode2) {
        return FormulaUtils.compareNumbers(errorCode1.num, errorCode2.num);
    };

    /**
     * Compares the passed values, and returns a signed indicator number.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value1
     *  The first value for comparison.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value2
     *  The second value for comparison.
     *
     * @param {Boolean} [caseSens=false]
     *  If set to true, strings will be compared case-sensitively. By default,
     *  the case of the strings will be ignored.
     *
     * @returns {Number}
     *  The number -1, if value1 is considered less than value2; or the number
     *  1, if value1 is considered greater than value2; otherwise the number 0
     *  (the values are equal). Blank cells (special value null) are always
     *  less than numbers, numbers are always less than strings, strings are
     *  always less than Boolean values, and Boolean values are always less
     *  than error codes. Values of the same type are compared as described for
     *  the methods FormulaUtils.compareNumbers(),
     *  FormulaUtils.compareStrings(), FormulaUtils.compareBooleans(), and
     *  FormulaUtils.compareErrorCodes().
     */
    FormulaUtils.compareValues = function (value1, value2, caseSens) {

        // returns a numeric identifier according to the value type
        function getTypeId(value) {
            return _.isNumber(value) ? 1 : _.isString(value) ? 2 : _.isBoolean(value) ? 3 : SheetUtils.isErrorCode(value) ? 4 : 0;
        }

        var type1 = getTypeId(value1),
            type2 = getTypeId(value2);

        // different types: ignore the actual values
        if (type1 !== type2) {
            return FormulaUtils.compareNumbers(type1, type2);
        }

        // compare values of equal types
        switch (type1) {
        case 1:
            return FormulaUtils.compareNumbers(value1, value2);
        case 2:
            return FormulaUtils.compareStrings(value1, value2, caseSens);
        case 3:
            return FormulaUtils.compareBooleans(value1, value2);
        case 4:
            return FormulaUtils.compareErrorCodes(value1, value2);
        }

        // both values are null
        return 0;
    };

    // numeric aggregation ----------------------------------------------------

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all numbers (and other values convertible to numbers) of all
     * function parameters to the result of the function. See method
     * FormulaContext.aggregateNumbers() for more details about the parameters.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all numbers of all its operands to the aggregation callback function.
     */
    FormulaUtils.implementNumericAggregation = function (initial, aggregate, finalize, iteratorOptions) {

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function numericAggregator() {
            return this.aggregateNumbers(arguments, initial, aggregate, finalize, iteratorOptions);
        };
    };

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all numbers (and other values convertible to numbers) of all
     * function parameters to the result of the function. Works similar to the
     * method FormulaUtils.implementNumericAggregation(), but collects all
     * visited numbers in an array, before invoking the 'finalize' callback
     * function.
     *
     * @param {Function} finalize
     *  A callback function invoked at the end to convert the collected numbers
     *  to the actual function result. Receives the following parameters:
     *  (1) {Array<Number>} numbers
     *      The visited numbers, as plain JavaScript array.
     *  (2) {Number} sum
     *      The sum of all visited numbers, for convenience.
     *  Must return the final numeric result of the spreadsheet function, or an
     *  error code object. Alternatively, may throw an error code. Will be
     *  called in the context of the resolver function (the FormulaContext
     *  instance).
     *
     * @param {Object} iteratorOptions
     *  Parameters passed to the number iterator used internally (see method
     *  FormulaContext.iterateNumbers() for details).
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor. The spreadsheet function will pass
     *  all numbers of all its operands to the aggregation callback function.
     */
    FormulaUtils.implementNumericAggregationWithArray = function (finalize, iteratorOptions) {

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function numericArrayAggregator() {

            var // all numbers collected in an array
                numbers = [];

            // stores the new number in the array, and updates the sum
            function aggregate(result, number) {
                numbers.push(number);
                return result + number;
            }

            // invokes the passed 'finalize' callback with the collected numbers
            function finalizeImpl(sum) {
                return finalize.call(this, numbers, sum);
            }

            return this.aggregateNumbers(arguments, 0, aggregate, finalizeImpl, iteratorOptions);
        };
    };

    /**
     * Creates and returns a resolver function for spreadsheet functions that
     * reduces all values matching a filter criterion to the result of the
     * function. See method FormulaContext.aggregateFiltered() for more details
     * about the parameters.
     *
     * @returns {Function}
     *  The resulting function implementation to be assigned to the 'resolve'
     *  property of a function descriptor.
     */
    FormulaUtils.implementFilterAggregation = function (initial, aggregate, finalize) {

        // create and return the resolver function to be assigned to the 'resolve' property of a function descriptor
        return function filterAggregator(sourceRef, criterion, dataRef) {
            return this.aggregateFiltered(sourceRef, criterion, dataRef, initial, aggregate, finalize);
        };
    };

    // debug ------------------------------------------------------------------

    /**
     * Returns the string representation of the passed literal value for debug
     * logging.
     *
     * @attention
     *  DO NOT use this method for GUI related code where correct locale
     *  dependent representation of the matrix elements is required.
     *
     * @param {Number|Date|String|Boolean|Complex|ErrorCode|Null}
     *  The value to be converted to the string representation.
     *
     * @returns {String}
     *  The string representation of the value.
     */
    FormulaUtils.valueToString = function (value) {
        return _.isString(value) ? ('"' + value.replace(/"/g, '""') + '"') :
            (value instanceof Date) ? value.toISOString() :
            _.isNull(value) ? 'null' :
            _.isBoolean(value) ? value.toString().toUpperCase() :
            value.toString();
    };

    /**
     * Writes all passed tokens to the browser console.
     *
     * @param {String} message
     *  A message string written before the formula tokens.
     *
     * @param {Array} tokens
     *  An array of formula tokens to be written to the browser console. The
     *  array elements can be of any type convertible to a string.
     */
    FormulaUtils.logTokens = FormulaUtils.isLoggingActive() ? function (message, tokens) {
        FormulaUtils.log(message + ': ' + tokens.join(' '));
    } : $.noop;

    // exports ================================================================

    return FormulaUtils;

});
