/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/cellref', [
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SheetUtils) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address;

    // class CellRef ==========================================================

    /**
     * A structure representing the address of a single cell in a reference
     * token (e.g. the particles '$A$1' or '$B$2' in Sheet1!$A$1:$B$2).
     *
     * @constructor
     *
     * @property {Address} address
     *  The cell address.
     *
     * @property {Boolean} absCol
     *  Whether the column reference is absolute (as in $C2).
     *
     * @property {Boolean} absRow
     *  Whether the row reference is absolute (as in C$2).
     */
    function CellRef(address, absCol, absRow) {

        this.address = address;
        this.absCol = absCol;
        this.absRow = absRow;

    } // class CellRef

    // static methods ---------------------------------------------------------

    /**
     * Creates a cell reference structure for the passed components of a cell
     * address reference, as used in a reference token.
     *
     * @param {SpreadsheetDocument} docModel
     *  The spreadsheet document used to validate the column and row indexes.
     *
     * @param {String|Boolean} absCol
     *  The absolute marker for the column. The column will be marked to be
     *  absolute, if this parameter is the string '$' or the Boolean value
     *  true.
     *
     * @param {String|Number} col
     *  The column name, or the zero-based column index.
     *
     * @param {String|Boolean} absRow
     *  The absolute marker for the row. The row will be marked to be absolute,
     *  if this parameter is the string '$' or the Boolean value true.
     *
     * @param {String|Number} row
     *  The row name, or the zero-based row index.
     *
     * @returns {CellRef|Null}
     *  A cell reference structure, if the passed column and row indexes are
     *  both valid; otherwise null.
     */
    CellRef.create = function (docModel, absCol, col, absRow, row) {

        // parse column and/or row index passed as string
        if (typeof col === 'string') { col = Address.parseCol(col); }
        if (typeof row === 'string') { row = Address.parseRow(row); }

        // validate the column and row indexes
        var address = new Address(col, row);
        if (!docModel.isValidAddress(address)) { return null; }

        // convert absolute markers to Booleans
        absCol = (absCol === true) || (absCol === '$');
        absRow = (absRow === true) || (absRow === '$');

        return new CellRef(address, absCol, absRow);
    };

    // public methods ---------------------------------------------------------

    /**
     * Creates a deep clone of this cell reference structure.
     *
     * @returns {CellRef}
     *  A deep clone of this cell reference structure.
     */
    CellRef.prototype.clone = function () {
        return new CellRef(this.address.clone(), this.absCol, this.absRow);
    };

    /**
     * Returns whether this cell reference and the passed cell reference are
     * equal.
     *
     * @param {CellRef} cellRef
     *  The other cell reference to be compared with this cell reference.
     *
     * @returns {Boolean}
     *  Whether this cell reference and the passed cell reference are equal.
     */
    CellRef.prototype.equals = function (cellRef) {
        return (this.absCol === cellRef.absCol) && (this.absRow === cellRef.absRow) && this.address.equals(cellRef.address);
    };

    /**
     * Relocates the relative column/row components in this cell reference
     * in-place.
     *
     * @param {SpreadsheetDocument} docModel
     *  The spreadsheet document used to validate the column and row indexes.
     *
     * @param {Address} refAddress
     *  The source reference address of the formula containing the token.
     *
     * @param {Address} targetAddress
     *  The target reference address for relocation.
     *
     * @param {Boolean} wrapReferences
     *  Whether to wrap the index at the sheet boundaries.
     *
     * @returns {Boolean}
     *  Whether relocation was successful, and the address of this reference
     *  is still valid.
     */
    CellRef.prototype.relocate = function (docModel, refAddress, targetAddress, wrapReferences) {

        // returns a relocated column/row index
        function relocateIndex(index, refIndex, targetIndex, maxIndex) {
            index += (targetIndex - refIndex);
            return wrapReferences ?
                ((index < 0) ? (index + maxIndex + 1) : (index > maxIndex) ? (index - maxIndex - 1) : index) :
                (((index < 0) || (index > maxIndex)) ? -1 : index);
        }

        // relocate relative column
        if (!this.absCol) {
            this.address[0] = relocateIndex(this.address[0], refAddress[0], targetAddress[0], docModel.getMaxCol());
            if (this.address[0] < 0) { return false; }
        }

        // relocate relative row
        if (!this.absRow) {
            this.address[1] = relocateIndex(this.address[1], refAddress[1], targetAddress[1], docModel.getMaxRow());
            if (this.address[1] < 0) { return false; }
        }

        return true;
    };

    /**
     * Returns the complete name of the column part of this cell reference.
     *
     * @returns {String}
     *  The complete name of the column part of this cell reference.
     */
    CellRef.prototype.colText = function () {
        return (this.absCol ? '$' : '') + Address.stringifyCol(this.address[0]);
    };

    /**
     * Returns the complete name of the row part of this cell reference.
     *
     * @returns {String}
     *  The complete name of the row part of this cell reference.
     */
    CellRef.prototype.rowText = function () {
        return (this.absRow ? '$' : '') + Address.stringifyRow(this.address[1]);
    };

    /**
     * Returns the complete name of this cell reference.
     *
     * @returns {String}
     *  The complete name of this cell reference.
     */
    CellRef.prototype.refText = function () {
        return this.colText() + this.rowText();
    };

    // exports ================================================================

    return CellRef;

});
